/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1999 by Myricom, Inc.                                   *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* This includes source for the user-level alarm API calls
   gm_initialize_alarm(), gm_set_alarm(), and gm_cancel_alarm().
   These source for all the functions is included here, because if the
   user needs one, the user needs them all.

   This code is a bit tricky, since the LANai provides only one alarm
   per port.  This alarm is used to time the first pending alarm, but
   if any new alarm is set, then the LANai alarm must be flushed to
   find out what time it is and potentially to allow the alarm to be
   rescheduled for an earlier time.  Therefore, unless no other alarm
   is set, alarms are set by adding them to an "unset alarm" queue,
   flushing any pending alarm, and then adding the alarms to the
   sorted alarm queue only once the pending LANai alarm has been
   flushed or triggered. */

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_cmp64.h"
#include "gm_internal.h"
#include "gm_send_queue.h"

#define GM_DEBUG_ALARMS 0

#if GM_DEBUG_ALARMS
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

/****************************************************************
 * LANai alarm layering
 ****************************************************************
 There is only one lanai-side alarm per port, on which we layer all GM
 alarms.  This is done by setting the LANai alarm to go off at the
 same time as the first scheduled GM alarm.  If a GM alarm arrives is
 scheduled that goes of before the first LANai alarm, then we flush
 the lanai alarm and reset it.

 Note that we are careful only to call alarm callbacks after receiving
 GM_ALARM_EVENTs, as documented in the GM API. */


/****************
 * Flush the currently scheduled lanai alarm.
 ****************/

static void
_flush_lanai_alarm (gm_port_t *port)
{
  GM_PRINT (GM_DEBUG_ALARMS, ("_flush_lanai_alarm() called.\n"));
  gm_assert (port->lanai_alarm_state == LANAI_ALARM_SET);

  /* pass a flush event to lanai in the send token queue */

  __gm_post_simple_send_event (port, GM_FLUSH_ALARM_EVENT);

  /* update state and return */

  port->lanai_alarm_state = LANAI_ALARM_FLUSHING;
  GM_PRINT (GM_DEBUG_ALARMS, ("_flush_lanai_alarm() returning.\n"));
}

/****************
 * Set a lanai alarm, knowing that no other lanai alarm is set.
 ****************/

static void
_set_lanai_alarm (gm_port_t *port, gm_u64_t deadline)
{
  struct gm_set_alarm_send_event volatile *se;
  struct gm_set_alarm_send_event batch_write;

  GM_CALLED_WITH_ARGS (("%p,"GM_U64_TMPL, port, GM_U64_ARG (deadline)));
  
  gm_assert (port->lanai_alarm_state == LANAI_ALARM_IDLE);

  /* Schedule the one and only alarm */  

  se = GM_SEND_QUEUE_SLOT (port, 0, 0, set_alarm);
  if (GM_STRUCT_WRITE_COMBINING)
    {
      batch_write.usecs = gm_hton_u64 (deadline);
      batch_write.type = gm_hton_u8 (GM_SET_ALARM_EVENT);
      GM_COPY_TO_IO_SPACE (*se, batch_write);
    }
  else
    {
      se->usecs = gm_hton_u64 (deadline);
      GM_STBAR ();
      gm_assert (!gm_ntoh_u8 (se->type));
      se->type = gm_hton_u8 (GM_SET_ALARM_EVENT);
    }
  GM_FLUSH_SEND_EVENT (port, se);

  /* update state and return */

  port->lanai_alarm_time = deadline;
  port->lanai_alarm_state = LANAI_ALARM_SET;

  GM_PRINT (GM_DEBUG_ALARMS, ("_set_lanai_alarm() returning.\n"));

  GM_RETURN_NOTHING ();
}

/************
 * Schedule a LANai GM_ALARM_EVENT.
 ************/

static void
_reschedule_lanai_alarm_if_needed (gm_port_t *port)
{
  /* verify we have something to reschedule */

  if (!port->first_alarm)
    return;
  
  /* schedule the alarm if the alarm is idle.  Otherwise, flush the
     alarm and reschedule once the alarm is idle, if needed. */

  switch (port->lanai_alarm_state)
    {
    case LANAI_ALARM_IDLE:
      _set_lanai_alarm (port, port->first_alarm->deadline);
      break;

    case LANAI_ALARM_SET:

      /* if the new first alarm is earlier than the currently
	 scheduled lanai alarm, flush the current lanai alarm so we
	 can set an earlier lanai alarm. */
	 
      if (gm_cmp64 (port->first_alarm->deadline, port->lanai_alarm_time) < 0)
	_flush_lanai_alarm (port);

      break;

    case LANAI_ALARM_FLUSHING:
      /* the alarm will be reschedule after the flush, which has
         already been requested. */
      break;
    }
}

/************
 * Handle an alarm event from the LANai.  This is called by
 * gm_unknown().
 ************/

gm_status_t
_gm_handle_alarm (gm_port_t *port)
{
  gm_alarm_t **ap, *a, *port__first_alarm;
  gm_u64_t rtc;

  GM_PRINT (GM_DEBUG_ALARMS, ("_gm_handle_alarm() called.\n"));

  /* remove all expired alarms from the alarm list.  This must be
     done before calling any alarm handler, because the alarm
     handlers may modify this list. */

  port__first_alarm = port->first_alarm;
  rtc = gm_ticks (port);
  for (ap = &port->first_alarm; *ap; ap = &(*ap)->next)
    {
      if (gm_cmp64 (rtc, (*ap)->deadline) < 0)
	break;
    }
  port->first_alarm = *ap;
  *ap = 0;

  /* perform the callbacks for all alarms that have expired */

  for (a = port__first_alarm; a; a = a->next)
    {
      a->state = GM_ALARM_FREE;
      a->next = 0;
      a->callback (a->context);
    }
  
  /* schedule next alarm check */

  port->lanai_alarm_state = LANAI_ALARM_IDLE;
  _reschedule_lanai_alarm_if_needed (port);

  GM_PRINT (GM_DEBUG_ALARMS, ("_gm_handle_alarm() returning.\n"));
  return GM_SUCCESS;
}

/************
 * Handle a flush event from the LANai.  This is called by
 * gm_unknown().
 ************/

gm_status_t
_gm_handle_flushed_alarm (gm_port_t *port)
{
  /* do not perform callbacks here, because the GM API docs state that
     callbacks are performed only after GM_ALARM_EVENTs. */

  port->lanai_alarm_state = LANAI_ALARM_IDLE;
  _reschedule_lanai_alarm_if_needed (port);

  return GM_SUCCESS;
}

/****************************************************************
 * API entry points
 ****************************************************************/

/************
 * Cancel an alarm.
 ************
 Alarms are cancelled by simply removing them from the alarm list so
 that when the next GM_ALARM_EVENT arrives from the lanai
 _gm_handle_alarm() will not find the alarm. */

GM_ENTRY_POINT
void
gm_cancel_alarm (gm_alarm_t *my_alarm)
{
  gm_alarm_t **a;

  GM_CALLED_WITH_ARGS (("%p", my_alarm));

  if (my_alarm->state == GM_ALARM_FREE)
    goto done;
  gm_assert (my_alarm->port);
  for (a = &my_alarm->port->first_alarm; *a; a = &(*a)->next)
    {
      if (*a == my_alarm)
	{
	  *a = my_alarm->next;
	  my_alarm->state = GM_ALARM_FREE;
	  goto done;
	}
    }
  GM_WARN (("Inconsistent alarm state detected.\n"));

 done:
  GM_RETURN_NOTHING ();
}

/************
 * Initialize user-allocated storage for an alarm.
 ************/

GM_ENTRY_POINT
void
gm_initialize_alarm (gm_alarm_t *my_alarm)
{
  GM_CALLED ();
  my_alarm->state = GM_ALARM_FREE;
  GM_RETURN_NOTHING ();
}

/************
 * Set an alarm, which may already be pending.  If it is pending, it
 * is rescheduled.  The user supplies MY_ALARM, a pointer to storage for
 * the alarm state, which has been initialized with gm_init_alarm_my_alarm().
 * When the alarm occurs, CALLBACK (CONTEXT) is() called.
 ************/

GM_ENTRY_POINT
void
gm_set_alarm (gm_port_t *port, gm_alarm_t *my_alarm, gm_u64_t usecs,
	      void (*callback) (void *), void *context)
{
  gm_alarm_t **a;

  GM_CALLED_WITH_ARGS (("*,*,%d,*,*", (int) usecs));
  
  /* Guarantee that the alarm is not pending */

  if (my_alarm->state != GM_ALARM_FREE)
    gm_cancel_alarm (my_alarm);
      
  /* Store the alarm state */

  my_alarm->state = GM_ALARM_SET;
  my_alarm->port = port;
  my_alarm->callback = callback;
  my_alarm->context = context;
  my_alarm->deadline = gm_ticks (port) + 2 * usecs; /* ticks are .5us each */

  /* insert the alarm in the list of alarms */

  for (a = &port->first_alarm; *a; a = &(*a)->next)
    {
      if (gm_cmp64 (my_alarm->deadline, (*a)->deadline) < 0)
	break;
    }
  my_alarm->next = *a;
  *a = my_alarm;
  
  /* arrange for a lanai alarm at an appropriate time. */

  _reschedule_lanai_alarm_if_needed (port);
  
  GM_PRINT (GM_DEBUG_ALARMS, ("gm_set_alarm() returning\n"));
  GM_RETURN_NOTHING ();
}
