/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#include "gm_compiler.h"
#include "gm_debug.h"
#include "gm_internal.h"
#include "gm_debug_mem_register.h"

void
_gm_register_memory_mutex_enter ()
{
  /* NEED_MUTEX */
  return;
}

void
_gm_register_memory_mutex_exit ()
{
  /* NEED_MUTEX */
  return;
}

GM_ENTRY_POINT gm_status_t
gm_register_memory (gm_port_t * p, void *ptr, gm_size_t length)
{
#if GM_KERNEL && GM_OS_VXWORKS && GM_CAN_REGISTER_MEMORY

#if !GM_ENABLE_VM
  
  GM_PRINT (GM_DEBUG_MEM_REGISTER,
	    ("vxworks: registering memory: GM_ENABLE_VM is not set\n"));
  return (GM_SUCCESS);

#else /* GM_ENABLE_VM */
  
  GM_PRINT (GM_DEBUG_MEM_REGISTER, ("vxworks: registering memory\n"));
  {
    char *user_vma = ptr;
    gm_port_state_t *ps = p->kernel_port_state;
    gm_page_hash_table_t *hash = &ps->instance->page_hash;
    gm_u32_t offset;
    char *end;
    gm_status_t status;

    GM_PRINT (GM_DEBUG_MEM_REGISTER, ("vxworks: gm_register_memory(0x%x \n"));

    if (!ptr)
      {
	GM_NOTE (("User did not specify input buffer.\n"));
	return (GM_FAILURE);
      }
    if (!length)
      {
	GM_NOTE (("User did not specify valid length.\n"));
	return (GM_FAILURE);
      }

    end = (char *) GM_PAGE_ROUNDUP (ptr, (char *) ptr + length);
    ptr = (void *) GM_PAGE_ROUND_DOWN (ptr, ptr);
    length = end - (char *) ptr;
    user_vma = ptr;

    ps->mem_reg_len = length;

    if (ps->id == GM_DAEMON_PORT_ID)
      {
	return (GM_PERMISSION_DENIED);
      }

    GM_PRINT (GM_DEBUG_MEM_REGISTER,
	      ("registering 0x%x bytes at user addr %p\n", ps->mem_reg_len,
	       user_vma));

    /* Verify buffer edges are page aligned. */

    if (!ps->mem_reg_len
	|| !GM_PAGE_ALIGNED (ps->mem_reg_len) || !GM_PAGE_ALIGNED (user_vma))
      {
	GM_NOTE (("User asked to register buffer of non-page size"
		  " or alignment.\n"));
	GM_PRINT (GM_DEBUG_MEM_REGISTER,
		  ("len=%d  len_align=%d  vma_align=%d\n", ps->mem_reg_len,
		   GM_PAGE_ALIGNED (ps->mem_reg_len),
		   GM_PAGE_ALIGNED (user_vma)));
	return (GM_INVALID_PARAMETER);
      }

    gm_arch_mutex_enter (&hash->sync);

    /* Register the memory one page at a time.  Extreme care must
       be taken to wrap all accesses to the user-supplied buffer
       in exception handlers in case the user has passed a bad
       buffer. */

    offset = 0;
    while (offset < ps->mem_reg_len)
      {
	/* Register the page, storing the "mdl" using the user pointer
	   field. */

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("Adding mapping.\n"));
	gm_assert (ps);
	gm_assert (ps->instance);
	GM_PRINT (GM_PRINT_LEVEL >= 4,
		  ("Calling gm_add_mapping_to_page_table (%p, %p, %d)\n",
		   ps->instance, user_vma + offset, ps->id, 0));
	status
	  = gm_add_mapping_to_page_table (ps->instance, user_vma + offset,
					  ps->id, 0);
	GM_PRINT (GM_PRINT_LEVEL >= 4,
		  ("Finished Calling add_mappings_to_page_table"
		   " (status: %d)\n", status));


	if (status != GM_SUCCESS)
	  {
	    GM_NOTE (("Failed to add mapping to page table\n"));
	    status = GM_INVALID_PARAMETER;
	    goto gm_register_loop_abort1;
	  }

	/* Prepare for next iteration */

	offset += GM_PAGE_LEN;
      }
    gm_arch_mutex_exit (&hash->sync);
    return (GM_SUCCESS);

  gm_register_loop_abort1:
    /* loic: we need to pause, the Lanai may have loaded some entries
       into its cache */
    gm_pause_lanai (ps->instance);
    while (offset)
      {
	offset -= GM_PAGE_LEN;
	gm_dereference_mapping (ps->instance, user_vma + offset, ps->id);
      }
    gm_unpause_lanai (ps->instance);
    gm_arch_mutex_exit (&hash->sync);
    return (GM_FAILURE);
  }
#endif /* GM_ENABLE_VM */
  
#elif GM_KERNEL

  GM_PARAMETER_MAY_BE_UNUSED (p);
  GM_PARAMETER_MAY_BE_UNUSED (ptr);
  GM_PARAMETER_MAY_BE_UNUSED (length);

  return GM_FAILURE;

#elif !GM_KERNEL

  char *end;
  gm_u32_t klen;

  /* feldy debug printf --Glenn */
  if (0)
    {
      gm_pid_t this_node, my_pid;
      my_pid = gm_getpid ();
      gm_get_node_id (p, &this_node);
      gm_printf ("%d %d   gm_register(%p, %p, %ld)  called node=%d\n",
		 my_pid, this_node, p, ptr, length, this_node);
    }

  if (!length)
    return GM_INVALID_PARAMETER;

  end = (char *) GM_PAGE_ROUNDUP (ptr, (char *) ptr + length);
  ptr = (void *) GM_PAGE_ROUND_DOWN (ptr, ptr);
  length = end - (char *) ptr;
  if (sizeof (length) > 4)
    {
      gm_always_assert (length < 0x100000000UL);
    }
  klen = GM_STATIC_CAST (gm_u32_t, length);

  _gm_register_memory_mutex_enter ();

  /* Tell the driver the length of the registration. */

  GM_PRINT (GM_DEBUG_MEM_REGISTER,
	    ("Registering 0x%lx bytes at address %p.\n", length, ptr));

  if (GM_OS_OSF1 || GM_OS_LINUX || GM_OS_FREEBSD || GM_OS_IRIX)
    {
      struct gm_off_len_uvma s;
      gm_status_t rv;

      s.offset = 0;
      s.len = length;
      s.uvma = (gm_up_t) ptr;

      rv = _gm_user_ioctl (p, GM_REGISTER_MEMORY_BY_STRUCT, &s, sizeof (s));
      if (rv != GM_SUCCESS)
	{
	  GM_NOTE (
		   ("Could not register memory (see kernel log) error = %d.\n",
		    rv));
	  goto abort_with_mutex;
	}
    }
  else
    {
      if (_gm_user_ioctl (p, GM_SET_REGISTER_MEMORY_LENGTH,
			  &klen, sizeof (klen)) != GM_SUCCESS)
	{
	  GM_NOTE (("Could not set registration length.\n"));
	  goto abort_with_mutex;
	}

      /* register the memory */

      if (_gm_user_ioctl (p, GM_REGISTER_MEMORY, ptr, length) != 0)
	{
	  GM_NOTE (("Could not register memory (see kernel log).\n"));
	  goto abort_with_mutex;
	}
    }


  _gm_register_memory_mutex_exit ();
  return GM_SUCCESS;

abort_with_mutex:
  _gm_register_memory_mutex_exit ();
  return GM_FAILURE;

#else
#error Do not know how to gm_register_memory().
#endif
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
