/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_compiler.h"
#include "gm_debug.h"
#include "gm_internal.h"

GM_ENTRY_POINT gm_status_t
gm_deregister_memory(gm_port_t * p, void *ptr, gm_size_t length)
{
#if GM_KERNEL
	GM_PARAMETER_MAY_BE_UNUSED (p);
	GM_PARAMETER_MAY_BE_UNUSED (ptr);
	GM_PARAMETER_MAY_BE_UNUSED (length);

#if GM_OS_VXWORKS


#if !GM_ENABLE_VM
    GM_PRINT (GM_PRINT_LEVEL >= 4,
			  ("vxworks: de_registering memory: GM_ENABLE_VM is not set\n"));
    return (GM_SUCCESS);
#else  /* GM_ENABLE_VM */

	{
		char *end;
		gm_u32_t offset;
		gm_page_hash_table_t *hash;
		char *user_vma;
		gm_port_state_t *ps = p->kernel_port_state;


		if (!ptr) {
			GM_NOTE (("User did not specify input buffer.\n"));
			return (GM_FAILURE);
		}
		if (!length) {
			GM_NOTE (("User did not specify valid length.\n"));
			return (GM_FAILURE);
		}

		end = (char *) GM_PAGE_ROUNDUP(ptr, (char *) ptr + length);
		ptr = (void *) GM_PAGE_ROUND_DOWN(ptr, ptr);
		length = end - (char *) ptr;

		GM_PRINT (GM_PRINT_LEVEL >= 4,
				  ("deregistering 0x%x bytes at user addr %p\n",
				   ps->mem_reg_len, ptr));

		hash = &ps->instance->page_hash;

		/* Verify that each of the pages is in the page table.  If
		   not, the user tried to deregister an illegitimate
		   region, and we fail. */

		gm_arch_mutex_enter(&hash->sync);

		offset = 0;
		user_vma = ptr;
		while (offset < ps->mem_reg_len) {
			if (!gm_refs_to_mapping_in_page_table(ps->instance,
											   user_vma + offset, ps->id)) {
				GM_NOTE (("User tried to deregister unregistered page.\n"));
				goto gm_deregister_abort_with_mutex;
			}
			offset += GM_PAGE_LEN;
		}

		/* Dereference each page of the buffer, and unlock the page if
		   it is no longer referenced. */

		GM_PRINT (GM_PRINT_LEVEL >= 4, ("Deregistering user buffer pages.\n"));
		gm_pause_lanai(ps->instance);
		offset = 0;
		user_vma = ptr;
		while (offset < ps->mem_reg_len) {
			gm_dereference_mapping(ps->instance, user_vma + offset, ps->id);

			/* iterate */
			offset += GM_PAGE_LEN;
		}
		gm_unpause_lanai(ps->instance);
		gm_arch_mutex_exit(&hash->sync);
		return GM_SUCCESS;

	  gm_deregister_abort_with_mutex:
		gm_arch_mutex_exit(&hash->sync);
		return (GM_FAILURE);
	}

#endif  /* GM_ENABLE_VM */
#endif	/* GM_OS_VXWORKS */


	GM_PANIC (("gm_deregister_memory() not supported in the kernel.\n"));
	return GM_FAILURE;
#else
	gm_status_t status;
	char *end;
	
	end = (char *) GM_PAGE_ROUNDUP(ptr, (char *) ptr + length);
	ptr = (void *) GM_PAGE_ROUND_DOWN(ptr, ptr);
	length = GM_STATIC_CAST(unsigned long, end - (char *) ptr);

#if 0
#warning feldy printf
	{
	  gm_pid_t this_node, my_pid;
	  my_pid = gm_getpid ();
	  gm_get_node_id (p, &this_node);
	  gm_printf ("%ld %d gm_deregister(%p, %p, %ld)  called node=%d\n",
		     (unsigned long) my_pid, this_node,p, ptr,
		     length, this_node);
	}
#endif

	if (sizeof (length) > 4)
		gm_always_assert(length <= 0xffffffffUL);
	_gm_register_memory_mutex_enter();

	/* Tell the driver how much to deregister */

#if GM_OS_OSF1 || GM_OS_LINUX || GM_OS_FREEBSD || GM_OS_IRIX
	{
	  struct gm_off_len_uvma s;
	  s.offset = 0;
	  s.len = length;
	  s.uvma = (gm_up_t)ptr;
	  
	  status = _gm_user_ioctl (p, GM_DEREGISTER_MEMORY_BY_STRUCT, &s,
							   sizeof(s));
	  if (status != GM_SUCCESS)
	    {
	      gm_pid_t this_node, my_pid;
	      my_pid = gm_getpid ();
	      gm_get_node_id (p, &this_node);
		  GM_NOTE (("%ld Could not deregister memory (%p, %ld)"
					" for gmID %d\n",
					(unsigned long) my_pid, ptr, length, this_node));
	      goto abort_with_mutex;
	    }
	}
#else
	{
	   gm_u32_t klen = GM_STATIC_CAST (gm_u32_t, length);

	   status = _gm_user_ioctl (p, GM_SET_REGISTER_MEMORY_LENGTH, &klen,
							 sizeof (klen));
	   if (status != GM_SUCCESS) {
		   GM_NOTE (("Could not set registration length.\n"));
		   goto abort_with_mutex;
       }
	}

	/* Tell the driver what to deregister. */
	status = _gm_user_ioctl (p, GM_DEREGISTER_MEMORY, ptr, length);
	if (status != GM_SUCCESS) {
		gm_pid_t this_node, my_pid;
		my_pid = gm_getpid ();
		gm_get_node_id (p, &this_node);
		GM_NOTE (("%d Could not deregister memory (%p, %ld)"
			      " for gmID %ld\n",
			      my_pid, ptr, length, this_node));
		goto abort_with_mutex;
	}

#endif /* GM_OS_OSF1 || GM_OS_LINUX || GM_OS_FREEBSD */


	_gm_register_memory_mutex_exit();
	return GM_SUCCESS;

  abort_with_mutex:
	_gm_register_memory_mutex_exit();
	return status;
#endif
}


/*
  This file uses feldy indentation:

  Local Variables:
  tab-width:4
  c-file-style:"bsd"
  End:
*/
