/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_debug.h"
#include "gm_internal.h"

#if !GM_KERNEL
#include <stdio.h>
#endif

GM_ENTRY_POINT struct gm_buf_handle *gm_bufs = 0;

GM_ENTRY_POINT char *gm_buf_status_name[] = {
  "gm_in_send",
  "gm_in_recv",
  "gm_in_app",
  "gm_invalid_status",
};

static
char *
_gm_get_buf_status_name (enum gm_buf_status s)
{
  if ((int)s < gm_in_send ||
      (int)s > gm_in_app) {
    return gm_buf_status_name[gm_invalid_status];
  }
  return gm_buf_status_name[(int)s];
}

static
struct gm_buf_handle *
_gm_find_buf (void *addr)
{
  struct gm_buf_handle *p = gm_bufs;

  while (p) {
    if (p->addr == addr) return p;
    p = p->next;
  }
  return 0;
}

GM_ENTRY_POINT
int
gm_unregister_buffer (void *addr, int size)
{
  struct gm_buf_handle **pp = &gm_bufs;
  struct gm_buf_handle *p;

  while (*pp) {
    if ((*pp)->addr == addr) {
      if (size != -1 &&
	  (*pp)->size != size) {
	GM_NOTE (("buffer %p is size %d, not %d\n", addr, (*pp)->size, size));
	gm_abort ();
      }
      gm_always_assert ((*pp)->status == gm_in_app);
      p = *pp;
      *pp = (*pp)->next;
      gm_free (p);
      return 1;
    }
    pp = &((*pp)->next);
  }
  return 0;
}

GM_ENTRY_POINT
void
gm_register_buffer (void *addr, int size)
{
  struct gm_buf_handle *b;

  gm_always_assert (addr);

  if (_gm_find_buf (addr)) {
    GM_NOTE (("%p is already registered\n", addr));
    gm_abort ();
  }

  /* check for overlap */

#if 0
  {
    struct gm_buf_handle *p;

    if (size != -1) {
      gm_always_assert (size >= GM_MIN_MESSAGE_SIZE && size <= 32);
      p = gm_bufs;
      while (p) {
	if (p->addr > addr)
	  gm_always_assert ((char *) p->addr
		     > (char *) addr + gm_max_length_for_size (size));
	if (addr > p->addr)
	  gm_always_assert ((char *) addr
		     > (char *) p->addr + gm_max_length_for_size (p->size));
	p = p->next;
      }
      /* zero the buffer and make sure it is not a bogus pointer */
      bzero (addr, gm_max_length_for_size (size));
    }
  }
#endif

  b = (struct gm_buf_handle *) gm_malloc (sizeof(struct gm_buf_handle));
  b->addr = addr;
  b->status = gm_in_app;
  b->size = size;
  b->next = gm_bufs;
  gm_bufs = b;

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("registered buffer %p, size %d.\n", addr, size));
}

GM_ENTRY_POINT
void
gm_dump_buffers (void)
{
  struct gm_buf_handle *p = gm_bufs;

  while (p) {
    (void) _gm_get_buf_status_name;
    GM_PRINT (1, ("buffer %p: size = %d, status = %s\n",
		  p, p->size, _gm_get_buf_status_name (p->status)));
    p = p->next;
  }
}
