/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_cmp64.h"
#include "gm_compiler.h"
#include "gm_internal.h"
#include "gm_debug.h"

#define GM_DEBUG_BLOCKING_RECEIVE 0

#if GM_DEBUG_BLOCKING_RECEIVE
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

static void
_never_called (void *x)
{
  GM_PARAMETER_MAY_BE_UNUSED (x);
  GM_PANIC (("should never get here\n"));
}

GM_ENTRY_POINT
gm_recv_event_t *
gm_blocking_receive (gm_port_t * p)
{
  gm_recv_event_t *ret;
  gm_alarm_t my_alarm;

  GM_CALLED_WITH_ARGS (("%p", p));

  /* Check for message already pending */

  ret = gm_receive (p);
  if (gm_ntoh_u8 (ret->recv.type) != GM_NO_RECV_EVENT)
    GM_RETURN_PTR (ret);

  /* No message is pending.  Arrange to go to sleep if no message is
     received in 1ms. */

  gm_initialize_alarm (&my_alarm);
  gm_set_alarm (p, &my_alarm, 1000, _never_called, p);
  
  /* Spin, until an event is received, and pass the event to the user.
     Make special provisions for our timer, which we want to cancel if
     and only if it doesn't get triggered before a user event is received. */

  while (1)
    {
      ret = gm_receive (p);
      switch (gm_ntoh_u8 (ret->recv.type))
	{
	case GM_NO_RECV_EVENT:
	  continue;
	  
	  /* Handle all flushed alarm events, which the user never
	     needs to know about, and which must not cause my_alarm to
	     be cancelled or reset. */

	case _GM_FLUSHED_ALARM_EVENT:
	  _gm_handle_flushed_alarm (p);
	  continue;

	  /* Intercept my_alarm. */

	case GM_ALARM_EVENT:
 	  if (gm_cmp64 (my_alarm.deadline, gm_ticks (p)) <= 0)
	    _gm_request_sleep (p);
	  goto return_to_user;
	  
	  /* Pass all other events, (including GM_SLEEP events) to the
	     user, and cancel the alarm.  Either this is an event the
	     user cares about, or it is a GM_SLEEP event, which will
	     put the user process to sleep.  Note that we may be
	     cancelling my_alarm after it has fired, but that's
	     harmless. */

	default:
	  GM_PRINT (GM_DEBUG_BLOCKING_RECEIVE,
		    ("Cancelling alarm for event of type %d.\n",
		     (int) gm_ntoh_u8 (ret->recv.type)));
	  goto return_to_user;
	}
    }

 return_to_user:
  gm_cancel_alarm (&my_alarm);
  GM_RETURN_PTR (ret);
}
