/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#include "gm_debug.h"
#include "gm_internal.h"

/* Allocate a page-aligned buffer of length ALLOC_LEN, where ALLOC_LEN
   is a multiple of GM_PAGE_LEN. */

GM_ENTRY_POINT void *
gm_alloc_pages (gm_size_t alloc_len)
{
  if (!GM_PAGE_LEN)
    return 0;

  /* special case page allocations to handle them more efficiently. */

  if (alloc_len <= GM_PAGE_LEN)
    return gm_page_alloc ();

  /* Handle larger-than-a-page case */

#if GM_KERNEL && GM_KERNEL_MALLOC_ALIGNS_PAGES
  return gm_arch_kernel_malloc (alloc_len, 0);
#elif GM_KERNEL && defined WIN32
  return ExAllocatePool (NonPagedPool, alloc_len);
#elif !GM_KERNEL && defined WIN32
  return VirtualAlloc (NULL, alloc_len, MEM_COMMIT, PAGE_READWRITE);
#else
  {
    void *tmp, *page;

    if (!GM_PAGE_LEN)
      return 0;

    tmp = gm_malloc (alloc_len + GM_PAGE_LEN + sizeof (void *));
    if (!tmp)
      return 0;

    /* reserve room for pointer to beginning of buffer. */

    page = (void *) ((char *) tmp + sizeof (void *));

    /* round page to a page boundary. */

    if (((long) page & (GM_PAGE_LEN - 1)) != 0)
      page = (void *) ((char *) page
		       + GM_PAGE_LEN - ((long) page & (GM_PAGE_LEN - 1)));

    GM_PRINT (0, ("Storing pointer at %p.\n", &((void **) page)[-1]));
    ((void **) page)[-1] = tmp;
    return page;
  }
#endif
}

GM_ENTRY_POINT void
gm_free_pages (void *addr, gm_size_t alloc_len)
{
  gm_assert (GM_PAGE_LEN);

  /* special case page allocations to handle them more efficiently. */

  if (alloc_len <= GM_PAGE_LEN)
    {
      GM_PRINT (0, ("gm_free_pages(%p, %ld) calling gm_page_free\n",
		    addr, alloc_len));
      gm_page_free (addr);
      return;
    }

  /* Handle larger-than-a-page case */

#if GM_KERNEL && GM_KERNEL_MALLOC_ALIGNS_PAGES
  gm_arch_kernel_free (addr);
#elif GM_KERNEL && defined WIN32
  ExFreePool (addr);
#elif !GM_KERNEL && defined WIN32
  VirtualFree (addr, 0, MEM_RELEASE);
#else
  GM_PRINT (0,
	    ("gm_free_pages: Retrieving pointer at %p.\n",
	     &((void **) addr)[-1]));
  gm_free (((void **) addr)[-1]);
#endif
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
