/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_internal.h"

#if GM_KERNEL
#include "gm_enable_security.h"
#include "gm_pio.h"
#endif

#define GM_DEBUG_SET_NODE_ID 0

#if GM_DEBUG_SET_NODE_ID
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

GM_ENTRY_POINT
gm_status_t
_gm_set_node_id (struct gm_port *port, unsigned int node_id)
{
#if GM_KERNEL
  unsigned int i;
  gm_port_state_t *ps;
  gm_instance_state_t *is;
  gm_status_t status;

  GM_CALLED_WITH_ARGS (("%p,%u", port, node_id));
  
  GM_PRINT (GM_DEBUG_SET_NODE_ID, ("setting node id to %u\n", node_id));
  
  ps = port->kernel_port_state;
  gm_assert (ps);
  is = ps->instance;
  gm_assert (is);
  
  if (GM_ENABLE_SECURITY && !ps->privileged)
    {
      status = GM_PERMISSION_DENIED;
      goto abort_with_status;
    }
  
  if (!is->lanai.running)
    {
      status = GM_UNATTACHED;
      goto abort_with_status;
    }
  
  if ((unsigned int) node_id > is->max_node_id)
    {
      status = GM_INVALID_PARAMETER;
      goto abort_with_status;
    }
  
  /* update the node ID in the reply packets */

  gm_write_lanai_global_u16 (is, this_node_id, (gm_u16_t) node_id);
  gm_write_lanai_global_u16 (is, mapper_state.scout_reply.packet.gm_id,
			     (gm_u16_t) node_id);
  
  /* update the unique ID in the reply packets */

  gm_write_lanai_global_byte_array
    (is, is->ethernet.addr_tab.volatile_ethernet_addr[node_id].as_bytes,
     mapper_state.scout_reply.packet.address);
  gm_write_lanai_global_byte_array
    (is, is->ethernet.addr_tab.volatile_ethernet_addr[node_id].as_bytes,
     mapper_state.config_reply.packet.address);

  /* update the host name in the reply packets */

  gm_assert (is->name_table.entry);
  gm_assert (is->name_table.entry[node_id]);
  gm_strncpy_to_lanai_globals (is, mapper_state.scout_reply.packet.hostname,
			       is->name_table.entry[node_id]);

  /* Update the state of all connections */

  GM_PRINT (GM_DEBUG_SET_NODE_ID, ("Updating %d connections.\n",
				   is->max_node_id));
  for (i = 0; i <= is->max_node_id; i++)
    {
      gm_write_lanai_global_u16 (is, connection[i].ack_packet.sender_node_id,
				 (gm_u16_t) node_id);
    }
  status = GM_SUCCESS;
  GM_RETURN_STATUS (status);
  
 abort_with_status:
  GM_RETURN_STATUS (status);
#else
  gm_status_t status;

  GM_CALLED ();
  
  gm_assert (port);

  status
    = _gm_user_ioctl (port, GM_SET_NODE_ID, &node_id, sizeof (node_id));
  if (status != GM_SUCCESS)
    {
      GM_RETURN_STATUS (status);
    }
  
  port->this_node_id = (gm_u16_t) node_id;
  GM_RETURN_STATUS (GM_SUCCESS);
#endif
}

