/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#ifndef _gm_types_h_		/* -*-c-*- */
#define _gm_types_h_

/* This file defines types used both by the LANai firmware and
   driver code. */

/************************************************************************
 *
 *   ######  #######    #    ######          #     # #######   ###
 *   #     # #         # #   #     #         ##   ## #         ###
 *   #     # #        #   #  #     #         # # # # #         ###
 *   ######  #####   #     # #     #         #  #  # #####      #
 *   #   #   #       ####### #     #         #     # #
 *   #    #  #       #     # #     #         #     # #         ###
 *   #     # ####### #     # ######          #     # #######   ###
 *
 * Because the types declared in this file are shared by the LANai and
 * the host, the structure definitions in this file must satisfy
 * the following requirements to ensure that the structures are
 * formatted identically by the LANai and host compilers:
 *
 * o All types stored in network byte order have type gm_*_n_t.  Since this
 *   file declares all host-accessible types that are stored in network byte
 *   order, this allows for compile-time endian conversion checks.
 * o unsigned integers are declared as gm_uSIZE_n_t, where SIZE is the number
 *   of bits in the integer.  Signed integers are declared as gm_sSIZE_n_t.
 * o C pointer declarations are avoided to make structure sizes independent
 *   of the pointer size of the compiler being used:
 *   - All LANai addresses have types ending in "_lp_n_t".  These types are
 *     defined as C pointers on the LANai and as gm_u32_n_t on the host.
 *   - All user virtual address have type "gm_up*_t".  These types
 *     are defined as C pointers on the host and as gm_u*_n_t in the lanai.
 *   - All DMA addresses have type "gm_dp*_t".
 * o All structure fields are aligned (using explicit padding if needed)
 *   to prevent different compilers from packing structures differently.
 * o Bitfields and enumerated types are avoided for the same reason.
 * o No zero-length arrays.
 *
 * Other conventions used in this file:
 * o All global-scope identifiers start with "_*[Gg][Mm]_" (regexp notation)
 * o All typedefs end in "_n_t"
 * o The Gnu coding standards are followed.
 ************************************************************************/

/***********************************************************************
 * Configuration defaults:
 ***********************************************************************/

#include "gm.h"
#include "gm_bitmap.h"
#include "gm_config.h"
#include "gm_cpp.h"
#include "gm_debug_counters.h"
#include "gm_debug_lanai_dma.h"
#include "gm_enable_error_counters.h"
#include "gm_enable_fast_small_send.h"
#include "gm_enable_galvantech_workaround.h"
#include "gm_enable_lmru_cache.h"
#include "gm_enable_jumbo_mtu.h"
#include "gm_enable_trace.h"
#include "gm_error_counters.h"
#include "gm_trace.h"

/* BAD: Only constants that are constant across all GM implementations
   should be here. */

/* The size of pages in host memory.  For machines without VM, it is
   convenient to pretend there are 4K pages, since the
   gm_port_protected_lanai_side structures are made to have length
   GM_PAGE_LEN. */
#if GM_ENABLE_VM
#  if !GM_PAGE_LEN && GM_MCP
#    error Must define GM_PAGE_LEN if GM_ENABLE_VM is set.
#  endif
#endif

/****************
 * Sizeof declarations
 ****************/

/* The declarations here must agree with the data types, and should be
   declared here only if compiler sizeof() directive is not
   sufficient for our purposes.  Any declaration here should be
   checked after the struct declaration with GM_CHECK_SIZEOF(). */

/* Used to define GM_NUM_RECV_TOKENS */
#define GM_SIZEOF_HOST_RECV_TOKEN		(GM_SIZEOF_UP_T == 8 ? 16 : 8)
/* Used to define GM_NUM_SEND_TOKENS */
#define GM_SIZEOF_SEND_QUEUE_SLOT		64

/* A macro used to verify the preceeding declarations after the actual
   data type definition.  It will generate a compile-time error. */

/* The number of user ports supported.  Must be <= 16 in the current
   GM version due to limitations of the connection's
   active_subport_bitmask field. */
#if GM_MIN_SUPPORTED_SRAM <= 256
#define GM_NUM_PORTS				4
#elif GM_MIN_SUPPORTED_SRAM <= 512
#define GM_NUM_PORTS				7
#else
#define GM_NUM_PORTS				8
#endif

/* The maximum number of switches that must be traversed to reach any
   remote host.  Multiple of 4. */
#define GM_MAX_NETWORK_DIAMETER 		24

				/* BAD: Can (and therefore should)
				   make these bigger on machines with
				   pages larger than 4K. */

/* Number of send tokens. */
#define GM_NUM_SEND_TOKENS (GM_NUM_SEND_QUEUE_SLOTS - 3)

/* Number of recv tokens.  2^n - 2, where 2^n of them are guaranteed
   to fit in half a page. */

#define GM_NUM_RECV_TOKENS (((GM_MIN_PAGE_LEN / 2)		\
			     / GM_SIZEOF_HOST_RECV_TOKEN)	\
                            - 2)

#define GM_NUM_DIRECTED_RECV_TOKENS		6
#define GM_NUM_ETHERNET_RECV_TOKENS 		16

/* The number of bins to be used in the hash table for storing receive
   tokens. 2^N */
#define GM_RECV_TOKEN_HASH_BINS			64

/* Twice the number of page hash table entries to be cached in the
   LANai, minus 1. 2^N-1 */

#define GM_MAX_PAGE_HASH_CACHE_INDEX (gm.page_hash.cache.max_index)

/* The maximum number of bytes for a streamlined receive (one that
   requires a host-side copy, but one less DMA). 2^N, and a multiple
   GM_RDMA_GRANULARITY. */
#define GM_MAX_FAST_RECV_BYTES			128

enum gm_size
{
  GM_SIZE_0 = 0, GM_SIZE_1 = 1, GM_SIZE_2 = 2, GM_SIZE_3 = 3,
  GM_SIZE_4 = 4, GM_SIZE_5 = 5, GM_SIZE_6 = 6, GM_SIZE_7 = 7,
  GM_SIZE_8 = 8, GM_SIZE_9 = 9, GM_SIZE_10 = 10, GM_SIZE_11 = 11,
  GM_SIZE_12 = 12, GM_SIZE_13 = 13, GM_SIZE_14 = 14, GM_SIZE_15 = 15,
  GM_SIZE_16 = 16, GM_SIZE_17 = 17, GM_SIZE_18 = 18, GM_SIZE_19 = 19,
  GM_SIZE_20 = 20, GM_SIZE_21 = 21, GM_SIZE_22 = 22, GM_SIZE_23 = 23,
  GM_SIZE_24 = 24, GM_SIZE_25 = 25, GM_SIZE_26 = 26, GM_SIZE_27 = 27,
  GM_SIZE_28 = 28, GM_SIZE_29 = 29, GM_SIZE_30 = 30, GM_SIZE_31 = 31,
  GM_RAW_TAG_SIZE = 32,		/* used to tag raw sends */
  GM_DIRECTED_TAG_SIZE = 33,	/* used to tag directed sends */
  GM_NUM_SIZES
};

#if GM_ENABLE_JUMBO_MTU
#define GM_ETHERNET_MTU (9*1024)
/* MTU to be told to the IP layer above you is GM_IP_MTU */
#define GM_IP_MTU       (9000)
#else /* !GM_ENABLE_JUMBO_MTU */
#define GM_ETHERNET_MTU (4096+64)
/* MTU to be told to the IP layer above you is GM_IP_MTU */
#define GM_IP_MTU       (3752)	/*(4096 - 64 - 256 - 14 - 2 - 8) */
#endif

/****
 * Debugging options
 ****/
#define GM_LOG_DISPATCHES 	0
#define GM_LOG_LEN		(GM_LOG_DISPATCHES ? 256 : 1)
#define GM_DISPATCH_MAX_NUM	(GM_LOG_DISPATCHES ? 256 : 1)

/***********************************************************************
 * Macros
 ***********************************************************************/

/* Misc. nonconfigurable constants. */

#define GM_NULL 		0
#define GM_TRUE 		1
#define GM_FALSE 		0
#define GM_MTU			4096
#define GM_MIN_MESSAGE_SIZE	3
/*
#define GM_MAX_DMA_CTR 		((GM_ETHERNET_MTU			\
				  > GM_MTU+2*GM_MAX_DMA_GRANULARITY)	\
				 ? GM_ETHERNET_MTU			\
				 :GM_MTU+2*GM_MAX_DMA_GRANULARITY)
*/
#define __GM_MAX(A, B) ((A) > (B) ? (A) : (B))
#define GM_MAX_DMA_CTR		__GM_MAX (GM_ETHERNET_MTU,		\
					  __GM_MAX (GM_PAGE_LEN,	\
						    GM_MTU))

#define GM_DAEMON_PORT_ID	0
#define GM_MAPPER_PORT_ID	1
#define GM_ETHERNET_PORT_ID	3

#define GM_CRC_TYPE		char
#define GM_CRC32_TYPE		unsigned int

#define GM_NO_ROUTE		((gm_u8_t) 0xFF)

				/* Implicit configuration definitions */
#define GM_NUM_HOST_PAGES	(gm_total_host_mem/GM_HOST_PAGE_LENGTH)
#define GM_NUM_SUBPORTS		(GM_NUM_PORTS * GM_NUM_SEND_TOKENS)

/* Number of USED slots in a queue.  An extra slot might be allocated
   to streamline wraparound. */
				/* 1 extra to set alarm, and *2* extra
				   to flush alarm */
#define GM_NUM_SEND_QUEUE_SLOTS ((GM_MIN_PAGE_LEN/2)			\
				 /GM_SIZEOF_SEND_QUEUE_SLOT)
/* extra for GM_SLEEP and GM_SENT*_EVENTs */
#define GM_NUM_RECV_QUEUE_SLOTS (GM_NUM_RECV_TOKENS+GM_NUM_SEND_QUEUE_SLOTS+1)
#define GM_NUM_RECV_TOKEN_QUEUE_SLOTS (GM_NUM_RECV_TOKENS)
#define GM_RECV_QUEUE_SLOTS_PER_PAGE					\
(GM_PAGE_LEN / sizeof (gm_recv_queue_slot_t))
/* Must be more than GM_NUM_SUBPORTS, even, and the smaller the better. */
#define GM_NUM_SEND_RECORDS 	(GM_NUM_SUBPORTS + 2)
/************************************************************************
 * Utility macros
 ************************************************************************/
#define GM_SLOW_POW2_ROUNDUP(x)						\
((x) <= 0x00000001 ? 0x00000001 : (x) <= 0x00000002 ? 0x00000002 :	\
 (x) <= 0x00000004 ? 0x00000004 : (x) <= 0x00000008 ? 0x00000008 :	\
 (x) <= 0x00000010 ? 0x00000010 : (x) <= 0x00000020 ? 0x00000020 :	\
 (x) <= 0x00000040 ? 0x00000040 : (x) <= 0x00000080 ? 0x00000080 :	\
 (x) <= 0x00000100 ? 0x00000100 : (x) <= 0x00000200 ? 0x00000200 :	\
 (x) <= 0x00000400 ? 0x00000400 : (x) <= 0x00000800 ? 0x00000800 :	\
 (x) <= 0x00001000 ? 0x00001000 : (x) <= 0x00002000 ? 0x00002000 :	\
 (x) <= 0x00004000 ? 0x00004000 : (x) <= 0x00008000 ? 0x00008000 :	\
 (x) <= 0x00010000 ? 0x00010000 : (x) <= 0x00020000 ? 0x00020000 :	\
 (x) <= 0x00040000 ? 0x00040000 : (x) <= 0x00080000 ? 0x00080000 :	\
 (x) <= 0x00100000 ? 0x00100000 : (x) <= 0x00200000 ? 0x00200000 :	\
 (x) <= 0x00400000 ? 0x00400000 : (x) <= 0x00800000 ? 0x00800000 :	\
 (x) <= 0x01000000 ? 0x01000000 : (x) <= 0x02000000 ? 0x02000000 :	\
 (x) <= 0x04000000 ? 0x04000000 : (x) <= 0x08000000 ? 0x08000000 :	\
 (x) <= 0x10000000 ? 0x10000000 : (x) <= 0x20000000 ? 0x20000000 :	\
 (x) <= 0x40000000 ? 0x40000000 : 0x80000000)
#define GM_POW2_PAD(x) gm_u8_n_t pad[GM_SLOW_POW2_ROUNDUP (x) - (x)]

/* return the number of elements in an array */
#define GM_NUM_ELEM(ar) (sizeof (ar) / sizeof (*ar))
#define GM_POWER_OF_TWO(n) (!((n)&((n)-1)))
/* Internal host pointer macros */
#if !GM_MCP
#  if GM_SIZEOF_VOID_P == 8
#    define _GM_ptr_ALIGN(n,m) ((void *)((gm_u64_t)(n)&~((gm_u64_t)(m)-1)))
#    define _GM_ptr_ROUNDUP(n,m) ((void *)(((gm_u64_t)(n)+(gm_u64_t)(m)-1)&~((gm_u64_t)(m)-1)))
#  elif GM_SIZEOF_VOID_P == 4
#    define _GM_ptr_ALIGN(n,m) ((void *)((gm_u32_t)(n)&~((gm_u32_t)(m)-1)))
#    define _GM_ptr_ROUNDUP(n,m) ((void *)(((gm_u32_t)(n)+(gm_u32_t)(m)-1)&~((gm_u32_t)(m)-1)))
#  endif
#  define _GM_ptr_ROUND_DOWN(n,m) _GM_ptr_ALIGN(n,m)
#endif
#if GM_SIZEOF_UP_T == 8
#  define _GM_up_ALIGN(n,m) ((gm_up_t)((gm_u64_t)(n)&~((gm_u64_t)(m)-1)))
#  define _GM_up_ROUNDUP(n,m) ((gm_up_t)(((gm_u64_t)(n)+(gm_u64_t)(m)-1)&~((gm_u64_t)(m)-1)))
#elif GM_SIZEOF_UP_T == 4
#  define _GM_up_ALIGN(n,m) ((gm_up_t)((gm_u32_t)(n)&~((gm_u32_t)(m)-1)))
#  define _GM_up_ROUNDUP(n,m) ((gm_up_t)(((gm_u32_t)(n)+(gm_u32_t)(m)-1)&~((gm_u32_t)(m)-1)))
#endif
#define _GM_up_ROUND_DOWN(n,m) _GM_up_ALIGN(n,m)
/* Internal lanai pointer macros */
#define _GM_lp_ALIGN(n,m) ((gm_lp_t)((gm_u32_t)(n)&~((gm_u32_t)(m)-1)))
#define _GM_lp_ROUNDUP(n,m) ((gm_lp_t)(((gm_u32_t)(n)+(gm_u32_t)(m)-1)&~((gm_u32_t)(m)-1)))
#define _GM_lp_ROUND_DOWN(n,m) _GM_lp_ALIGN(n,m)
/* Internal DMA pointer macros */
#define _GM_dp_ALIGN(n,m) ((gm_dp_t)((gm_dp_t)(n)&~((gm_dp_t)(m)-1)))
#define _GM_dp_ROUNDUP(n,m) ((gm_dp_t)(((gm_dp_t)(n)+(gm_dp_t)(m)-1)&~((gm_dp_t)(m)-1)))
#define _GM_dp_ROUND_DOWN(n,m) _GM_dp_ALIGN(n,m)
/* Internal u32 macros */
#define _GM_u32_ALIGN(n,m) ((gm_u32_t)(n)&~((gm_u32_t)(m)-1))
#define _GM_u32_ROUNDUP(n,m) (((gm_u32_t)(n)+(gm_u32_t)(m)-1)&~((gm_u32_t)(m)-1))
#define _GM_u32_ROUND_DOWN(n,m) _GM_u32_ALIGN(n,m)
/* Internal size macros */
#define _GM_size_ALIGN(n,m) ((gm_size_t)(n)&~((gm_size_t)(m)-1))
#define _GM_size_ROUNDUP(n,m) (((gm_size_t)(n)+(gm_size_t)(m)-1)&~((gm_size_t)(m)-1))
#define _GM_size_ROUND_DOWN(n,m) _GM_size_ALIGN(n,m)
/****
 * Generic alignment macros
 ****/
#define GM_ALIGN(type,n,m) _GM_##type##_ALIGN(n,m)
#define GM_ROUNDUP(type,n,m) _GM_##type##_ROUNDUP(n,m)
#define GM_ROUND_DOWN(type,n,m) _GM_##type##_ROUND_DOWN(n,m)
/* GM_ALIGNED is in gm.h */
/****
 * Alignment macros
 ****/
/* BAD: all GM alignment requirements will be lifted */
#define GM_PACKET_GRANULARITY   8
#define GM_DMA_ROUNDUP(t,p) 	GM_ROUNDUP (t,p, GM_DMA_GRANULARITY)
#define GM_DMA_ALIGN(t,p) 	GM_ALIGN (t,p, GM_DMA_GRANULARITY)
#if GM_MCP
#define GM_RDMA_ROUNDUP(t,p) 	GM_ROUNDUP (t,p, GM_RDMA_GRANULARITY)
#define GM_RDMA_ALIGN(t,p) 	GM_ALIGN (t,p, GM_RDMA_GRANULARITY)
#define GM_RDMA_ALIGNED(p)	GM_ALIGNED (p, GM_RDMA_GRANULARITY)
#define GM_PACKET_ROUNDUP(t,p) 	GM_ROUNDUP (t,p, GM_PACKET_GRANULARITY)
#define GM_PACKET_ALIGN(t,p) 	GM_ALIGN (t,p, GM_PACKET_GRANULARITY)
#define GM_PACKET_ALIGNED(p)	GM_ALIGNED (p, GM_PACKET_GRANULARITY)
#endif
/* Page macros */
#define GM_PAGE_ROUNDUP(t,p) 	GM_ROUNDUP (t,p, GM_PAGE_LEN)
#define GM_PAGE_ALIGN(t,p) 	GM_ALIGN (t,p, GM_PAGE_LEN)
#define GM_PAGE_ALIGNED(p)	GM_ALIGNED (p, GM_PAGE_LEN)
#define GM_PAGE_OFFSET(n)	  ((gm_u32_t)(n) & (GM_PAGE_LEN-1))
#define GM_PAGE_REMAINING(n) ((gm_u32_t)(~(gm_s32_t)((n))&(GM_PAGE_LEN-1))+1)
#define GM_PAGE_ROUND_DOWN(t,p) GM_PAGE_ALIGN(t,p)
#define GM_PAGE_SHIFT (GM_PAGE_LEN == 16384 ? 14 :			\
		       GM_PAGE_LEN == 8192 ? 13 :			\
		       12)
#define GM_DMA_PAGE_ADDR(n)	((gm_dp_t)(n) << GM_PAGE_SHIFT)
#define GM_UP_PAGE_NUM(a) 	((gm_up_t)(a) >> GM_PAGE_SHIFT)
#define GM_DP_PAGE_NUM(a) 	((gm_dp_t)(a) >> GM_PAGE_SHIFT)
#define GM_PAGE_NUM_TO_HP(page) ((void *)(((gm_up_t)page << GM_PAGE_SHIFT)))
/* Ethernet addres table macros */
/* enough to support ethernet addresses for 64K nodes & 4K pages:
   64K * 8B / 4KB */
#define GM_MAX_NUM_ADDR_TABLE_PIECES ((1<<16) * sizeof (gm_unique_id_64_t) \
				      / 4096)
/* enough to support names for 16K nodes & 4K pages:
   64K * GM_MAX_ / 4KB */
#define GM_MAX_NUM_NAME_TABLE_PIECES ((1<<16) * GM_MAX_HOST_NAME_LEN / 4096)
/* Page hash table macros */
#define GM_ENTRIES_PER_PIECE (GM_PAGE_LEN/sizeof(gm_pte_t))
#if !GM_ENABLE_VM
#define GM_PAGE_HASH_PIECE_REF_TABLE_LEN 0
#define GM_MAX_NUM_HOST_HASH_TABLE_PIECES 0
#define GM_PAGE_HASH_MAX_INDEX 0
#define GM_PAGE_HASH_MAX_INDEX 0
#else /* GM_ENABLE_VM */
#define GM_PAGE_HASH_PIECE_REF_TABLE_LEN (GM_PAGE_LEN == 4096		\
				          ? GM_PAGE_LEN * 4		\
					  : GM_PAGE_LEN)
#define GM_MAX_NUM_HOST_HASH_TABLE_PIECES 				\
(GM_PAGE_HASH_PIECE_REF_TABLE_LEN / sizeof (gm_dp_n_t))
#define GM_PAGE_HASH_MAX_INDEX ((GM_MAX_NUM_HOST_HASH_TABLE_PIECES	\
				 * GM_ENTRIES_PER_PIECE)		\
	                        - 1)
#endif /* GM_ENABLE_VM */
  /* This should never be larger than
     32bit, so OK to use
     GM_DMA_PAGE_ROUNDUP */
#define GM_RECV_QUEUE_ALLOC_LEN \
     GM_PAGE_ROUNDUP (u32, (GM_NUM_RECV_QUEUE_SLOTS \
				* sizeof (gm_recv_queue_slot_t)))
  /* Macros to combine/extract port id
     and priority into/outof a subport
     ID. */
#define GM_SUBPORT(priority,port) ((port)<<1|priority)
#define GM_SUBPORT_PRIORITY(subport_id) ((subport_id)&1)
#define GM_SUBPORT_PORT(subport_id) ((unsigned int) (subport_id)>>1U)
#define GM_MAX_SUBPORT_ID GM_SUBPORT (GM_HIGH_PRIORITY, GM_NUM_PORTS - 1)
/***********************************************************************
 * enumerations
 ***********************************************************************/
/* Aliases for priorities */
#define GM_MIN_PRIORITY	   GM_LOW_PRIORITY
#define GM_MAX_PRIORITY	   GM_HIGH_PRIORITY
/* Packet magic numbers.  These are actually the Myrinet packet
   "types" required in the first 2 bytes of every packet.  However, we
   refer to them as the packet magic number in GM source code to avoid
   confusion with the GM packet "type". */
enum gm_packet_type
{
  /* Contact help@myri.com to have one allocated for your
     project. See http://www.myri.com/scs/types.html */
#if GM_ENABLE_GALVANTECH_WORKAROUND
  GM_PACKET_TYPE = 0x0011,
#else
  GM_PACKET_TYPE = 0x0008,
#endif
  GM_MAPPING_PACKET_TYPE = 0x000f,
  GM_ETHERNET_PACKET_TYPE = 0x0020
};

/* The types of GM packets.

   NOTE: although this is an enum, we explicitly indicate the values
   to prevent accidentally changing them, since they should NEVER be
   changed, to ensure backwards compatibility. */
enum gm_packet_subtype
{
  GM_NULL_SUBTYPE = 0,
  GM_BOOT_SUBTYPE = 1,
  GM_ACK_SUBTYPE = 2,
  GM_NACK_SUBTYPE = 3,
  GM_NACK_DOWN_SUBTYPE = 4,
  GM_NACK_REJECT_SUBTYPE = 5,
  GM_RAW_HACK_SUBTYPE = 6,
  /* The following types with sizes */
  /* allow for more compact header */
  /* encoding. */

  /* Identifiers for unsegmented messages. */
  GM_RELIABLE_DATA_SUBTYPE_0 = 7,
  GM_RELIABLE_DATA_SUBTYPE_1 = 8,
  GM_RELIABLE_DATA_SUBTYPE_2 = 9,
  GM_RELIABLE_DATA_SUBTYPE_3 = 10,
  GM_RELIABLE_DATA_SUBTYPE_4 = 11,
  GM_RELIABLE_DATA_SUBTYPE_5 = 12,
  GM_RELIABLE_DATA_SUBTYPE_6 = 13,
  GM_RELIABLE_DATA_SUBTYPE_7 = 14,
  GM_RELIABLE_DATA_SUBTYPE_8 = 15,
  GM_RELIABLE_DATA_SUBTYPE_9 = 16,
  GM_RELIABLE_DATA_SUBTYPE_10 = 17,
  GM_RELIABLE_DATA_SUBTYPE_11 = 18,
  GM_RELIABLE_DATA_SUBTYPE_12 = 19,
  GM_RELIABLE_DATA_SUBTYPE_13 = 20,
  GM_RELIABLE_DATA_SUBTYPE_14 = 21,
  GM_RELIABLE_DATA_SUBTYPE_15 = 22,
  GM_RELIABLE_DATA_SUBTYPE_16 = 23,
  GM_RELIABLE_DATA_SUBTYPE_17 = 24,
  GM_RELIABLE_DATA_SUBTYPE_18 = 25,
  GM_RELIABLE_DATA_SUBTYPE_19 = 26,
  GM_RELIABLE_DATA_SUBTYPE_20 = 27,
  GM_RELIABLE_DATA_SUBTYPE_21 = 28,
  GM_RELIABLE_DATA_SUBTYPE_22 = 29,
  GM_RELIABLE_DATA_SUBTYPE_23 = 30,
  GM_RELIABLE_DATA_SUBTYPE_24 = 31,
  GM_RELIABLE_DATA_SUBTYPE_25 = 32,
  GM_RELIABLE_DATA_SUBTYPE_26 = 33,
  GM_RELIABLE_DATA_SUBTYPE_27 = 34,
  GM_RELIABLE_DATA_SUBTYPE_28 = 35,
  GM_RELIABLE_DATA_SUBTYPE_29 = 36,
  GM_RELIABLE_DATA_SUBTYPE_30 = 37,
  GM_RELIABLE_DATA_SUBTYPE_31 = 38,

  /* Identifiers for the first segment
     of a segmented message. */
  GM_RELIABLE_HEAD_DATA_SUBTYPE_13 = 39,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_14 = 40,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_15 = 41,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_16 = 42,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_17 = 43,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_18 = 44,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_19 = 45,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_20 = 46,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_21 = 47,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_22 = 48,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_23 = 49,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_24 = 50,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_25 = 51,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_26 = 52,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_27 = 53,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_28 = 54,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_29 = 55,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_30 = 56,
  GM_RELIABLE_HEAD_DATA_SUBTYPE_31 = 57,
  /* Identifiers for continuations of
     segmented messages. */
  GM_RELIABLE_BODY_DATA_SUBTYPE = 58,
  GM_RELIABLE_TAIL_DATA_SUBTYPE = 59,
  GM_NACK_OPEN_CONNECTION_SUBTYPE = 60,

  GM_DIRECTED_DATA_SUBTYPE = 61,
  GM_DIRECTED_HEAD_DATA_SUBTYPE = 62,
  GM_DIRECTED_BODY_DATA_SUBTYPE = 63,
  GM_DIRECTED_TAIL_DATA_SUBTYPE = 64,

  /* 65 */
  GM_NACK_CLOSE_CONNECTION_SUBTYPE = 66,
  GM_ETHERNET_HACK_SUBTYPE = 67,

  GM_DATAGRAM_SUBTYPE_0 = 68, GM_DATAGRAM_SUBTYPE_1 = 69,
  GM_DATAGRAM_SUBTYPE_2 = 70, GM_DATAGRAM_SUBTYPE_3 = 71,
  GM_DATAGRAM_SUBTYPE_4 = 72, GM_DATAGRAM_SUBTYPE_5 = 73,
  GM_DATAGRAM_SUBTYPE_6 = 74, GM_DATAGRAM_SUBTYPE_7 = 75,
  GM_DATAGRAM_SUBTYPE_8 = 76, GM_DATAGRAM_SUBTYPE_9 = 77,
  GM_DATAGRAM_SUBTYPE_10 = 78, GM_DATAGRAM_SUBTYPE_11 = 79,
  GM_DATAGRAM_SUBTYPE_12 = 80, GM_DATAGRAM_SUBTYPE_13 = 81,
  GM_DATAGRAM_SUBTYPE_14 = 82, GM_DATAGRAM_SUBTYPE_15 = 83,
  GM_DATAGRAM_SUBTYPE_16 = 84, GM_DATAGRAM_SUBTYPE_17 = 85,
  GM_DATAGRAM_SUBTYPE_18 = 86, GM_DATAGRAM_SUBTYPE_19 = 87,
  GM_DATAGRAM_SUBTYPE_20 = 88, GM_DATAGRAM_SUBTYPE_21 = 89,
  GM_DATAGRAM_SUBTYPE_22 = 90, GM_DATAGRAM_SUBTYPE_23 = 91,
  GM_DATAGRAM_SUBTYPE_24 = 92, GM_DATAGRAM_SUBTYPE_25 = 93,
  GM_DATAGRAM_SUBTYPE_26 = 94, GM_DATAGRAM_SUBTYPE_27 = 95,
  GM_DATAGRAM_SUBTYPE_28 = 96, GM_DATAGRAM_SUBTYPE_29 = 97,
  GM_DATAGRAM_SUBTYPE_30 = 98, GM_DATAGRAM_SUBTYPE_31 = 99,
  GM_NUM_SUBTYPES
};

enum gm_send_token_type
{
  GM_ST_RELIABLE = 0,
  GM_ST_DIRECTED = 1,
  GM_ST_RAW = 2,
  GM_ST_PROBE = 3,
  GM_ST_MAPPER_SCOUT_REPLY = 4,
  GM_ST_MAPPER_CONFIG_REPLY = 5,
  GM_ST_ETHERNET_SEND = 6,
  GM_ST_ETHERNET_BROADCAST = 7,
  GM_ST_DATAGRAM = 8,
  GM_ST_PIO_DATAGRAM = 9
};

enum gm_interrupt_type
{
  GM_NO_INTERRUPT,
  GM_PAUSE_INTERRUPT,
  GM_UNPAUSE_INTERRUPT,
  GM_WAKE_INTERRUPT,
  GM_PRINT_INTERRUPT,
  GM_FAILED_ASSERTION_INTERRUPT,
  GM_ETHERNET_RECV_INTERRUPT,
  GM_ETHERNET_SENT_INTERRUPT,
  GM_WRITE_INTERRUPT,
  GM_CLEAR_TABLES_INTERRUPT
};

/***********************************************************************
 * Types
 ***********************************************************************/

/* handy type for implying that an int acts as a boolean. */
typedef int gm_boolean_t;

#include "gm_simple_types.h"

/****************
 * Strong type checking types
 ****************/

/* The stong types used in the GM API are defined in gm.h. */

#if !GM_MCP

#if GM_STRONG_TYPES
typedef gm_u32_t gm_lp_t;
typedef struct
{
  gm_dp_t n;
}
gm_dp_n_t;

typedef struct
{
  gm_lp_t n;
}
gm_lp_n_t;
#else  /* !GM_STRONG_TYPES */
typedef gm_u32_t gm_lp_t;
typedef gm_dp_t gm_dp_n_t;
typedef gm_lp_t gm_lp_n_t;
#endif /* !GM_STRONG_TYPES */

#else /* GM_MCP */

typedef void *gm_lp_t;
typedef gm_dp_t gm_dp_n_t;
typedef gm_lp_t gm_lp_n_t;

#endif /* GM_MCP */

/************
 * LANai-side pointers
 ************/

/* Macro to allow structure pointers on the LANai to be defined as
   pointers on the LANai, but as gm_u32_n_t's on the host */

#if GM_MCP
#define GM_LP_N_T(foo) typedef foo *
#else
#define GM_LP_N_T(foo) typedef gm_lp_n_t
#endif

/* *INDENT-OFF* */
GM_LP_N_T (gm_u8_t) gm_u8_lp_n_t;
GM_LP_N_T (gm_s8_t) gm_s8_lp_n_t;
GM_LP_N_T (void *) gm_lp_n_lp_n_t;
GM_LP_N_T (gm_up_n_t) gm_up_n_lp_n_t;
/* *INDENT-ON* */

#define GM_TYPEDEF_LP_N_T(type, name) GM_LP_N_T (type name) name ## _lp_t

/* *INDENT-OFF* */
GM_TYPEDEF_LP_N_T (struct, _gm_sent_token_report);
GM_TYPEDEF_LP_N_T (struct, gm_connection);
GM_TYPEDEF_LP_N_T (struct, gm_ethernet_recv_token);
GM_TYPEDEF_LP_N_T (struct, gm_host_recv_token);
GM_TYPEDEF_LP_N_T (struct, gm_cached_pte);
GM_TYPEDEF_LP_N_T (struct, gm_port_protected_lanai_side);
GM_TYPEDEF_LP_N_T (struct, gm_port_unprotected_lanai_side);
GM_TYPEDEF_LP_N_T (struct, gm_recv_token);
GM_TYPEDEF_LP_N_T (struct, gm_send_queue_slot);
GM_TYPEDEF_LP_N_T (struct, gm_send_record);
GM_TYPEDEF_LP_N_T (struct, gm_subport);
GM_TYPEDEF_LP_N_T (union, gm_send_token);
/* *INDENT-ON* */

#if GM_MCP
typedef struct gm_handler *gm_handler_n_t;
#else
typedef gm_lp_n_t gm_handler_n_t;
#endif

/* A GM "sexno" is a structure containing both the session number
   ("sesno") and sequence number within the session ("seqno").  While
   most communication uses only sequence numbers, GM uses the session
   numbers to reliably multiplex many steams of data over a single
   connection, avoiding the connection state overhead that would be
   required if multiple connections were used. */
typedef union gm_sexno
{
  gm_s32_n_t whole;
  struct
  {
    gm_s16_n_t sesno;		/* session number */
    gm_s16_n_t seqno;		/* sequence number */
  }
  parts;
}

gm_sexno_t;

/****
 * Page hash table types
 ****/

/* A page is just GM_PAGE_LEN bytes.  This structure is useful for
   computing page numbers without casting to an integer type. */
#ifdef gm_lanai
#error
typedef struct gm_page
{
  gm_s8_n_t byte[GM_PAGE_LEN];
}

gm_page_t;
#endif

/* An page hash table entry used to map (virt_page,port)->dma_page.
   That is, the virt_page and port fields act as a key, and the
   dma_page is the value. 2^N bytes.

   WARNING: All fields are stored in network byte order.. */

typedef struct gm_pte
{
#if GM_SIZEOF_UP_T == GM_SIZEOF_DP_T
  gm_up_n_t page_port;		/* also stores port */
  gm_dp_n_t _packed_dma_addr;	/* also stores reference count */
#elif GM_SIZEOF_UP_T == 4 && GM_SIZEOF_DP_T == 8
  gm_up_n_t page_port;		/* also stores port */
  gm_u32_n_t pad;
  gm_dp_n_t _packed_dma_addr;	/* also stores reference count */
#elif GM_SIZEOF_UP_T == 8 && GM_SIZEOF_DP_T == 4
  gm_up_n_t page_port;		/* also stores port */
  gm_dp_n_t _packed_dma_addr;	/* also stores reference count */
  gm_u32_n_t pad;
#else
#error Unsupported sizes.
#endif
  /* 8 */
#if GM_ENABLE_GALVANTECH_WORKAROUND
#if (GM_SIZEOF_UP_T == GM_SIZEOF_DP_T) && (GM_SIZEOF_DP_T == 4)
  gm_u32_n_t pad_checksum1;
  gm_u32_n_t pad_checksum2;
#endif
  gm_u32_n_t pad_checksum3[3];
  gm_u32_n_t checksum;
#endif				/* GM_ENABLE_GALVANTECH_WORKAROUND */
}

gm_pte_t;

/* A cached version of a gm_pte_n_t. 2^N bytes. */
				/* 8 words */
typedef struct gm_cached_pte
{
  gm_pte_t pte;
  /* 16 */
#if GM_ENABLE_LMRU_CACHE
  gm_cached_pte_lp_t older;
  gm_cached_pte_lp_t younger;
#if GM_ENABLE_GALVANTECH_WORKAROUND
  gm_u8_n_t pad[24];
#else
#if GM_SIZEOF_UP_T != 4 || GM_SIZEOF_DP_T != 4
  gm_u64_n_t pad;		/* pad to power of 2 size. */
#endif
#endif
#endif
}

gm_cached_pte_t;

/* A cache of a host-resident page hash table.  Includes a ROOT for
   the list of recently used cache entries, an array of bins in which
   to store ENTRYs, and a CNT of the number of cached entries. */
typedef struct gm_page_hash_cache
{
#if GM_ENABLE_LMRU_CACHE
  gm_cached_pte_t root;
  /* 8 */
#endif
  gm_u32_n_t cnt;
  gm_u32_n_t _reserved_after_cnt;
  /* 8 */
  /* extra entry at end to allow wraparound optimization. */
  gm_cached_pte_lp_t entry;
  gm_u32_n_t max_index;
}

gm_page_hash_cache_t;

/* this structure is used to pass the args of a directcopy to the kernel */
typedef struct gm_directcopy
{
  /* virtual address of the source buffer in the sender process */
  gm_up_t source_addr;
  /* virtual address of the target buffer on the receiver process */
  gm_up_t target_addr;
  /* length of the buffer to copy */
  gm_size_t length;
  /* port id and board id to identify the source process (GET protocol) */
  gm_u16_t source_port_id;
  gm_u16_t source_instance_id;
}

gm_directcopy_t;

/**********************************************************************
 * Packet-related types
 **********************************************************************
 All fields of GM packets use network byte order, unless otherwise noted. */

/* The header of a GM data segment sent over the network. */
				/* 6 words */
typedef struct gm_packet_header
{
  gm_u16_n_t type;
  gm_u16_n_t subtype;

  gm_u16_n_t target_node_id;
  gm_u16_n_t sender_node_id;
  /* 8 */
  gm_sexno_t sexno;

  gm_u16_n_t length;
  gm_u8_n_t target_subport_id;
  gm_u8_n_t sender_subport_id;
  /* 8 */
#if GM_ENABLE_GALVANTECH_WORKAROUND
  gm_u32_n_t header_checksum;
  gm_u16_n_t ip_checksum;	/* used in tail packets only */
  gm_u16_n_t reserved_after_ip_checksum;
#endif
}

gm_packet_header_t;

#define GM_HEADER_PAYLOAD(p) ((gm_u8_n_t *)((gm_packet_header_t *)(p)+1))

/* A GM data segment consists of a header and a payload of GM_MTU
   bytes or less. */
typedef struct gm_packet
{
  gm_packet_header_t header;
  gm_u8_n_t payload[GM_MTU];
}

gm_packet_t;

/* A GM data segment used for a directed send.  All fields must be
   worst-case for all GM implementations for compatibility. */
typedef struct gm_directed_packet
{
  gm_packet_header_t header;
  /* 8 */
  gm_remote_ptr_n_t target_addr;
  /* 8 */
  gm_u8_n_t payload[GM_MTU + 8];
}

gm_directed_packet_t;

/* A GM ack packet */
typedef struct gm_ack_packet	/* 4 words */
{
  gm_u16_n_t type;
  gm_u16_n_t subtype;

  gm_u16_n_t target_node_id;
  gm_u16_n_t sender_node_id;

  gm_sexno_t sexno;

  gm_s32_n_t usecs_delay;	/* for other types  */
#if GM_ENABLE_GALVANTECH_WORKAROUND
  gm_u32_n_t header_checksum;
#endif
}

gm_ack_packet_t;

/************************************************************************
 * Ethernet types
 ************************************************************************/

typedef struct gm_ethernet_segment_descriptor
{
  gm_dp_n_t ptr;
  gm_u32_n_t len;
#if GM_SIZEOF_DP_T == 8
  gm_u32_n_t pad;
#endif
}

gm_ethernet_segment_descriptor_t;

#define GM_MAX_ETHERNET_SCATTER_CNT					\
  (64 / sizeof (gm_ethernet_segment_descriptor_t))

typedef struct gm_ethernet_recv_token
{
  gm_ethernet_segment_descriptor_t segment[GM_MAX_ETHERNET_SCATTER_CNT];
}

gm_ethernet_recv_token_t;

/**********************************************************************
 * Mapper packet types.
 */

enum gm_mapper_packet_subtype
{
  GM_MAPPER_SCOUT_PACKET_SUBTYPE = 0,
  GM_MAPPER_CONFIG_PACKET_SUBTYPE = 9,
  GM_MAPPER_SCOUT_REPLY_PACKET_SUBTYPE = 2,
  GM_MAPPER_CONFIG_REPLY_PACKET_SUBTYPE = 10
};

enum gm_mapper_scout_reply_options
{
  GM_MAPPER_SCOUT_REPLY_NO_OPTION = 0,
  GM_MAPPER_SCOUT_REPLY_CLOUD_OPTION = 1,
  GM_MAPPER_SCOUT_REPLY_PACKED_ROUTES_OPTION = 2,
  GM_MAPPER_SCOUT_REPLY_LONG_HOSTNAME_OPTION = 4
};


/*separate from GM_MAX_HOST_NAME_LEN because part
  of a packet type.*/

#define GM_MAPPER_SCOUT_REPLY_HOSTNAME_LENGTH 128

/* A mapper scout ack packet.  Send whenever a scout is received. */
typedef struct gm_mapper_scout_reply_packet	/* 6 words */
{
  gm_s16_n_t type;
  gm_s16_n_t subtype;
  gm_s32_n_t port;
  gm_s32_n_t phase;

  gm_u8_n_t address[6];
  gm_u8_n_t mapper_address[6];
  gm_u16_n_t gm_id;
  gm_u8_n_t mapper_open;
  gm_u8_n_t pad;
  gm_s32_n_t map_version;
  gm_u8_n_t old_hostname[32];
  gm_s32_n_t level;
  gm_s16_n_t node_type;
  gm_s16_n_t option;
  gm_u8_n_t hostname[GM_MAPPER_SCOUT_REPLY_HOSTNAME_LENGTH];
}

gm_mapper_scout_reply_packet_t;

/* A mapper config ack packet.  Send whenever a config is received. */
typedef struct gm_mapper_config_reply_packet
{
  gm_s16_n_t type;
  gm_s16_n_t subtype;
  gm_s32_n_t port;
  gm_s32_n_t phase;

  gm_u8_n_t address[6];
  gm_s16_n_t pad;
  gm_s32_n_t host_section;
}

gm_mapper_config_reply_packet_t;

/* A packet used by the mapper to scout the network */
typedef struct gm_mapper_scout_packet_n	/* 15 words */
{
  gm_s16_n_t type;
  gm_s16_n_t subtype;
  gm_s32_n_t port;
  gm_s32_n_t phase;

  gm_s32_n_t route_length;
  gm_u8_n_t route[32];
  gm_u8_n_t address[6];
  gm_s16_n_t address_padding;
  gm_s32_n_t level;
  gm_u8_n_t extended_route[1];
}

gm_mapper_scout_packet_t;

/* A packet used by the mapper to scout the network */
typedef struct gm_mapper_scout_reset_packet_n	/* 15 words */
{
  gm_s16_n_t type;
  gm_s16_n_t subtype;
  gm_s32_n_t port;
  gm_s32_n_t phase;

  gm_s32_n_t route_length;
  gm_u8_n_t route[32];
  gm_u8_n_t address[6];
  gm_s16_n_t command;
  gm_s32_n_t level;
  gm_u8_n_t extended_route[1];
}

gm_mapper_scout_reset_packet_t;

/* A mapper packet containing node configuration info */
typedef struct gm_mapper_config_packet
{
  gm_s16_n_t type;
  gm_s16_n_t subtype;
  gm_s32_n_t port;
  gm_s32_n_t phase;

  gm_s32_n_t host_section;
  gm_u8_n_t address[6];		/* Address of receiver. */
  gm_u8_n_t mapper_address[6];	/* Address of mapper. */
  gm_s16_n_t id;		/* ID of receiver. */
  gm_s16_n_t num_hosts;		/* number of hosts in network */
  gm_s32_n_t map_version;
  gm_s32_n_t num_items;		/*number of entries in byte stream */
  gm_s32_n_t num_bytes;		/*length of byte stream */
  gm_u8_n_t bytes[1];		/* variable-length */

}

gm_mapper_config_packet_t;

/* A generic mapper packet of any type. */
typedef union gm_mapper_packet
{
  struct
  {
    gm_s16_n_t type;
    gm_s16_n_t subtype;
    gm_s32_n_t port;
    gm_s32_n_t phase;
  }
  common;
  gm_mapper_config_reply_packet_t config_reply;
  gm_mapper_scout_reply_packet_t scout_reply;
  gm_mapper_config_packet_t config;
  gm_mapper_scout_packet_t scout;
  gm_mapper_scout_reset_packet_t scout_reset;
}

gm_mapper_packet_t;

/* Make sure the assumptions made by the following typedefs are legit */

#if GM_SIZEOF_UP_T > 8 || GM_MAX_DMA_GRANULARITY > 8
#error broken typedefs
#endif

#if GM_RDMA_GRANULARITY < 32
#error possibly broken typedefs
#endif


				/* Struct including the minimum data
				   that needs to be DMAd for a normal
				   receive, and having GM_RDMA_GRANULARITY. */
struct _gm_recv_event
{
  /* Pad to GM_RDMA_GRANULARITY bytes */
#if (16 + GM_SIZEOF_UP_T) % GM_RDMA_GRANULARITY
  gm_u8_n_t _reserved[GM_RDMA_GRANULARITY
		      - (16 + GM_SIZEOF_UP_T) % GM_RDMA_GRANULARITY];
#endif
  gm_up_n_t buffer;
  /* 8 */
  gm_u16_n_t ip_checksum;	/* only used for GALVANTECH_WORKAROUND */
  gm_u16_n_t reserved_after_ip_checksum;
  gm_u32_n_t length;
  /* 8 */
  gm_u16_n_t sender_node_id;
  gm_u16_n_t reserved_after_sender_node_id;
  gm_u8_n_t tag;
  gm_u8_n_t size;
  gm_u8_n_t sender_port_id;
  gm_u8_n_t type;
};

				/* Struct holding only the information
				   needed for a fast receive with NO
				   GRANULARITY PADDING. */
struct _gm_fast_recv_event
{
  gm_up_n_t message;
  gm_up_n_t buffer;
  /* 8 */
  gm_u16_n_t ip_checksum;	/* only used for GALVANTECH_WORKAROUND */
  gm_u16_n_t reserved_after_ip_checksum;
  gm_u32_n_t length;
  /* 8 */
  gm_u16_n_t sender_node_id;
  gm_u16_n_t reserved_after_sender_node_id;
  gm_u8_n_t tag;
  gm_u8_n_t size;
  gm_u8_n_t sender_port_id;
  gm_u8_n_t type;
};


				/* Struct holding only the information
				   needed to be DMAd to report an
				   ethernet recv. */
struct _gm_ethernet_recv_event
{
  gm_u8_n_t reserved[28];
  gm_u16_n_t len;
  gm_u8_n_t reserved_after_len;
  gm_u8_n_t type;
};

typedef struct gm_recv_queue_slot
{
  gm_u8_n_t _reserved[2 * GM_MAX_FAST_RECV_BYTES - sizeof (gm_recv_event_t)];
  gm_recv_event_t event;
}

gm_recv_queue_slot_t;
#define GM_SIZEOF_GM_RECV_QUEUE_SLOT_T (2 * GM_MAX_FAST_RECV_BYTES)

/* *INDENT-OFF* */
GM_TOP_LEVEL_ASSERT (sizeof (gm_recv_queue_slot_t)
		     == GM_SIZEOF_GM_RECV_QUEUE_SLOT_T);
/* *INDENT-ON* */

		     /* A chunk of memory used to stage sends */
typedef struct gm_send_chunk
{
  gm_lp_n_t smp;
  gm_lp_n_t smlt;
  gm_lp_n_t smh;
  gm_lp_n_t reserved_after_smh;
  /* 8 */
#if GM_CPU_lanai
  gm_lp_n_t send_list;
  gm_lp_n_t send_list_end;
  /* 8 */
#endif
  /* 8 */
  char route[GM_MAX_NETWORK_DIAMETER];
  /* 8 */
  union
  {
    struct
    {
      gm_u16_n_t type;
      char payload[1];
    }
    as_myrinet;
    gm_packet_t as_gm;
    gm_mapper_packet_t as_mapper_packet;
    gm_directed_packet_t as_directed_packet;
    char as_bytes[1];
    /* FIXME: remove this */
    char as_ethernet[GM_ETHERNET_MTU
		     + (GM_ETHERNET_MTU % 8 ? 8 - GM_ETHERNET_MTU % 8 : 0)];
  }
  packet;
  /* 8 */
  gm_u32_n_t cafebabe;
  gm_u8_n_t last_for_ethernet_broadcast;
  gm_u8_n_t reserved[3];
  /* 8 */
}

gm_send_chunk_t;

    /* a chunk of memory used for receives */
typedef struct gm_recv_chunk
{
  union
  {
    struct
    {
      gm_u16_n_t type;
    }
    as_myrinet;
    gm_packet_t as_gm;
    gm_ack_packet_t as_gm_ack;
    gm_mapper_packet_t as_mapper_packet;
    gm_directed_packet_t as_gm_directed;
    char as_bytes[1];
    char as_ethernet[GM_ETHERNET_MTU + ((GM_ETHERNET_MTU % 8)
					? (8 - GM_ETHERNET_MTU % 8)
					: 0)];
  }
  packet;
  /* 8 */
  /* reserve space for crc on end. */
  gm_u32_n_t _trashed0;
  gm_u32_n_t _trashed1;
  /* 8 */
  gm_u32_n_t end;		/* unused, but marks end */
  gm_u32_n_t cafebabe;
  /* 8 */
  /* Extra details about raw receives */
  gm_u32_n_t raw_type_HACK;
  gm_u32_n_t raw_length;
  /* 8 */
}

gm_recv_chunk_t;

/****
 * Token related types
 ****/

enum gm_send_event_type
{
  GM_NO_SEND_EVENT = 0,
  GM_SET_ALARM_EVENT = 1,
  GM_FLUSH_ALARM_EVENT = 2,
  GM_RAW_SEND_EVENT = 3,
  /*  xxxGM_UNRELIABLE_SEND_EVENT = 4, */
  GM_SLEEP_RQST_EVENT = 5,

  /* Special steamlined send types for
     messages less than 256 bytes
     long. */

  GM_FAST_SEND_EVENT_0 = 6, GM_FAST_SEND_EVENT_1 = 7,
  GM_FAST_SEND_EVENT_2 = 8, GM_FAST_SEND_EVENT_3 = 9,
  GM_FAST_SEND_EVENT_4 = 10, GM_FAST_SEND_EVENT_5 = 11,
  GM_FAST_SEND_EVENT_6 = 12, GM_FAST_SEND_EVENT_7 = 13,
  GM_FAST_SEND_EVENT_8 = 14, GM_FAST_SEND_EVENT_9 = 15,
  GM_FAST_SEND_EVENT_10 = 16, GM_FAST_SEND_EVENT_11 = 17,
  GM_FAST_SEND_EVENT_12 = 18, GM_FAST_SEND_EVENT_13 = 19,
  GM_FAST_SEND_EVENT_14 = 20, GM_FAST_SEND_EVENT_15 = 21,
  GM_FAST_SEND_EVENT_16 = 22, GM_FAST_SEND_EVENT_17 = 23,
  GM_FAST_SEND_EVENT_18 = 24, GM_FAST_SEND_EVENT_19 = 25,
  GM_FAST_SEND_EVENT_20 = 26, GM_FAST_SEND_EVENT_21 = 27,
  GM_FAST_SEND_EVENT_22 = 28, GM_FAST_SEND_EVENT_23 = 29,
  GM_FAST_SEND_EVENT_24 = 30, GM_FAST_SEND_EVENT_25 = 31,
  GM_FAST_SEND_EVENT_26 = 32, GM_FAST_SEND_EVENT_27 = 33,
  GM_FAST_SEND_EVENT_28 = 34, GM_FAST_SEND_EVENT_29 = 35,
  GM_FAST_SEND_EVENT_30 = 36, GM_FAST_SEND_EVENT_31 = 37,

  GM_FAST_SEND_HIGH_EVENT_0 = 38, GM_FAST_SEND_HIGH_EVENT_1 = 39,
  GM_FAST_SEND_HIGH_EVENT_2 = 40, GM_FAST_SEND_HIGH_EVENT_3 = 41,
  GM_FAST_SEND_HIGH_EVENT_4 = 42, GM_FAST_SEND_HIGH_EVENT_5 = 43,
  GM_FAST_SEND_HIGH_EVENT_6 = 44, GM_FAST_SEND_HIGH_EVENT_7 = 45,
  GM_FAST_SEND_HIGH_EVENT_8 = 46, GM_FAST_SEND_HIGH_EVENT_9 = 47,
  GM_FAST_SEND_HIGH_EVENT_10 = 48, GM_FAST_SEND_HIGH_EVENT_11 = 49,
  GM_FAST_SEND_HIGH_EVENT_12 = 50, GM_FAST_SEND_HIGH_EVENT_13 = 51,
  GM_FAST_SEND_HIGH_EVENT_14 = 52, GM_FAST_SEND_HIGH_EVENT_15 = 53,
  GM_FAST_SEND_HIGH_EVENT_16 = 54, GM_FAST_SEND_HIGH_EVENT_17 = 55,
  GM_FAST_SEND_HIGH_EVENT_18 = 56, GM_FAST_SEND_HIGH_EVENT_19 = 57,
  GM_FAST_SEND_HIGH_EVENT_20 = 58, GM_FAST_SEND_HIGH_EVENT_21 = 59,
  GM_FAST_SEND_HIGH_EVENT_22 = 60, GM_FAST_SEND_HIGH_EVENT_23 = 61,
  GM_FAST_SEND_HIGH_EVENT_24 = 62, GM_FAST_SEND_HIGH_EVENT_25 = 63,
  GM_FAST_SEND_HIGH_EVENT_26 = 64, GM_FAST_SEND_HIGH_EVENT_27 = 65,
  GM_FAST_SEND_HIGH_EVENT_28 = 66, GM_FAST_SEND_HIGH_EVENT_29 = 67,
  GM_FAST_SEND_HIGH_EVENT_30 = 68, GM_FAST_SEND_HIGH_EVENT_31 = 69,

  GM_SEND_EVENT_0 = 70, GM_SEND_EVENT_1 = 71, GM_SEND_EVENT_2 = 72,
  GM_SEND_EVENT_3 = 73, GM_SEND_EVENT_4 = 74, GM_SEND_EVENT_5 = 75,
  GM_SEND_EVENT_6 = 76, GM_SEND_EVENT_7 = 77, GM_SEND_EVENT_8 = 78,
  GM_SEND_EVENT_9 = 79, GM_SEND_EVENT_10 = 80, GM_SEND_EVENT_11 = 81,
  GM_SEND_EVENT_12 = 82, GM_SEND_EVENT_13 = 83, GM_SEND_EVENT_14 = 84,
  GM_SEND_EVENT_15 = 85, GM_SEND_EVENT_16 = 86, GM_SEND_EVENT_17 = 87,
  GM_SEND_EVENT_18 = 88, GM_SEND_EVENT_19 = 89, GM_SEND_EVENT_20 = 90,
  GM_SEND_EVENT_21 = 91, GM_SEND_EVENT_22 = 92, GM_SEND_EVENT_23 = 93,
  GM_SEND_EVENT_24 = 94, GM_SEND_EVENT_25 = 95, GM_SEND_EVENT_26 = 96,
  GM_SEND_EVENT_27 = 97, GM_SEND_EVENT_28 = 98, GM_SEND_EVENT_29 = 99,
  GM_SEND_EVENT_30 = 100, GM_SEND_EVENT_31 = 101,

  GM_DIRECTED_SEND_EVENT = 102,

  /* 103 */
  GM_RESUME_SENDING_EVENT = 104,
  GM_DROP_SENDS_EVENT = 105,
  GM_ETHERNET_SEND_EVENT = 106,
  GM_ETHERNET_BROADCAST_EVENT = 107,
  GM_ETHERNET_MARK_AND_SEND_EVENT = 108,
  GM_ETHERNET_MARK_AND_BROADCAST_EVENT = 109,
  GM_TRACE_START_EVENT = 110,
  GM_TRACE_STOP_EVENT = 111,

  GM_DATAGRAM_SEND_EVENT_0, GM_DATAGRAM_SEND_EVENT_1,
  GM_DATAGRAM_SEND_EVENT_2, GM_DATAGRAM_SEND_EVENT_3,
  GM_DATAGRAM_SEND_EVENT_4, GM_DATAGRAM_SEND_EVENT_5,
  GM_DATAGRAM_SEND_EVENT_6, GM_DATAGRAM_SEND_EVENT_7,
  GM_DATAGRAM_SEND_EVENT_8, GM_DATAGRAM_SEND_EVENT_9,
  GM_DATAGRAM_SEND_EVENT_10, GM_DATAGRAM_SEND_EVENT_11,
  GM_DATAGRAM_SEND_EVENT_12, GM_DATAGRAM_SEND_EVENT_13,
  GM_DATAGRAM_SEND_EVENT_14, GM_DATAGRAM_SEND_EVENT_15,
  GM_DATAGRAM_SEND_EVENT_16, GM_DATAGRAM_SEND_EVENT_17,
  GM_DATAGRAM_SEND_EVENT_18, GM_DATAGRAM_SEND_EVENT_19,
  GM_DATAGRAM_SEND_EVENT_20, GM_DATAGRAM_SEND_EVENT_21,
  GM_DATAGRAM_SEND_EVENT_22, GM_DATAGRAM_SEND_EVENT_23,
  GM_DATAGRAM_SEND_EVENT_24, GM_DATAGRAM_SEND_EVENT_25,
  GM_DATAGRAM_SEND_EVENT_26, GM_DATAGRAM_SEND_EVENT_27,
  GM_DATAGRAM_SEND_EVENT_28, GM_DATAGRAM_SEND_EVENT_29,
  GM_DATAGRAM_SEND_EVENT_30, GM_DATAGRAM_SEND_EVENT_31,

  GM_PIO_DATAGRAM_4_SEND_EVENT,
  GM_PIO_DATAGRAM_8_SEND_EVENT,
  GM_PIO_DATAGRAM_12_SEND_EVENT,
  GM_PIO_DATAGRAM_16_SEND_EVENT
};

#define GM_PRIORITY_SIZE(priority,size) ((priority)<<5|size)
#define GM_PRIORITY_SIZE__PRIORITY(ps) ((ps)>>5)
#define GM_PRIORITY_SIZE__SIZE(ps) ((ps)&31)

/****************************************************************
 * Send events
 ****************************************************************/

/* Special care must be taken when laying out send events to ensure
   that the following conditions are satisfied:
   o The 1-byte TYPE field must be last.
   o All fields must be aligned assuming that the END of the event
     is aligned on an 8-byte boundary.
   o The structures must have no implicit padding on the end.  This
     means if the largest field is N bytes, then the TYPE field must
     be just below a N-byte boundary. */

#if GM_SIZEOF_UP_T == 4
#define GM_UP_T(name) gm_u32_t reserved_before_ ## name ; gm_up_t name
#else
#define GM_UP_T(name) gm_up_t name
#endif

struct gm_send_send_event
{
#if GM_ENABLE_GALVANTECH_WORKAROUND
  gm_u16_n_t reserved_before_ip_checksum[GM_SIZEOF_UP_T / 2 - 1];
  gm_u16_n_t ip_checksum;
#endif
  gm_up_n_t message;
  /* 8 */
  gm_u32_n_t length;
  gm_u16_n_t target_node_id;
  gm_u8_n_t target_subport_id;
  gm_u8_n_t type;
};

struct gm_pio_datagram_send_event
{
  /* 8 */
  gm_u32_n_t data;
  gm_u16_n_t target_node_id;
  gm_u8_n_t target_subport_id;
  gm_u8_n_t type;
};

struct gm_directed_send_send_event
{
#if GM_ENABLE_GALVANTECH_WORKAROUND
  /* IP checksum useless for dirsends */
#endif
  gm_remote_ptr_n_t target_buffer;
  /* 8 */
#if GM_SIZEOF_UP_T == 4
  gm_u32_n_t reserved_before_source_buffer;
#endif
  gm_up_n_t source_buffer;
  /* 8 */
  gm_u32_n_t length;
  gm_u16_n_t target_node_id;
  gm_u8_n_t target_subport_id;
  gm_u8_n_t type;
};

struct gm_fast_send_send_event
{
#if GM_SIZEOF_UP_T == 8
  /* 8 */
  volatile gm_up_n_t message;
  /* 8 */
  gm_u16_n_t ip_checksum;	/* for GALVANTECH_WORKAROUND only */
  gm_u16_n_t reserved_after_ip_checksum;
  gm_u16_n_t target_node_id;
  gm_u8_n_t length;
  gm_u8_n_t type;
#elif GM_SIZEOF_UP_T == 4
#if GM_ENABLE_GALVANTECH_WORKAROUND
  gm_u16_n_t reserved_before_ip_checksum;
  gm_u16_n_t ip_checksum;
#endif
  /* 8 */
  volatile gm_up_n_t message;
  gm_u16_n_t target_node_id;
  gm_u8_n_t length;
  gm_u8_n_t type;
#endif
};

struct gm_raw_send_send_event
{
#if GM_ENABLE_GALVANTECH_WORKAROUND
  gm_u16_n_t reserved_before_ip_checksum[GM_SIZEOF_UP_T / 2 - 1];
  gm_u16_n_t ip_checksum;
#endif
  gm_up_n_t message;
  /* 8 */
  gm_u32_n_t total_length;
  gm_u32_n_t route_length;
  /* 8 */
  gm_u32_n_t reserved_after_route_length;
  gm_u16_n_t cleared;
  gm_u8_n_t reserved_after_cleared;
  gm_u8_n_t type;
};

struct gm_set_alarm_send_event
{
  /* 8 */
  gm_u64_n_t usecs;
  /* 8 */
  gm_u8_n_t _reserved_after_usecs[7];
  gm_u8_n_t type;
};

struct gm_simple_send_event
{
  gm_u8_n_t reserved[3];
  gm_u8_n_t type;
};

struct gm_probe_send_event
{
  gm_u16_n_t target_node_id;
  gm_u8_n_t _reserved_after_target_port_id;
  gm_u8_n_t type;
};

struct gm_resend_send_event
{
  gm_u16_n_t target_node_id;
  gm_u8_n_t target_subport_id;
  gm_u8_n_t type;
};

typedef union gm_unique_id_64_t
{
  gm_u64_t as_unswapped_u64;	/* No endian swapping checking here */
  gm_u8_n_t as_bytes[6];
}

gm_unique_id_64_t;

#if GM_SIZEOF_DP_T == 4
#define GM_MAX_ETHERNET_GATHER_CNT 7
#else
#define GM_MAX_ETHERNET_GATHER_CNT 3
#endif

struct gm_ethernet_send_event
{
  /* WARNING - if you change the layout of this struct, you need to
     change drivers/gm_ether.c (send) */
  gm_ethernet_segment_descriptor_t gather_segment[GM_MAX_ETHERNET_GATHER_CNT];
  /* 8 */
#if GM_SIZEOF_DP_T == 4
  gm_u8_n_t reserved[4];
#else
  gm_u8_n_t reserved[12];
#endif
  gm_u16_n_t target_node_id;
  gm_u8_n_t gather_cnt;
  gm_u8_n_t type;
};

/* A data structure used to pass messages from the Host to the LANai.
   All fields use LANai (network) byte order except as noted.  */
struct gm_send_queue_slot
{
  gm_u8_n_t as_bytes[GM_SIZEOF_SEND_QUEUE_SLOT - 1];
  gm_u8_n_t type;
};

/* *INDENT-OFF* */
GM_TOP_LEVEL_ASSERT (sizeof (struct gm_send_queue_slot)
		     == GM_SIZEOF_SEND_QUEUE_SLOT);
/* *INDENT-ON* */

#define GM_ST_ACKABLE_P(st) (st->common.type == GM_ST_RELIABLE		\
			     || st->common.type == GM_ST_DIRECTED	\
			     || st->common.type == GM_ST_PROBE)

#define GM_SEND_QUEUE_SLOT_EVENT(sqs, type)				\
((struct gm_ ## type ## _send_event *)(sqs+1)-1)


/* The state of a subport. */
typedef struct gm_subport
{
  gm_subport_lp_t next;
  gm_subport_lp_t prev;
  gm_connection_lp_t connection;

  gm_send_token_lp_t first_send_token;
  gm_send_token_lp_t last_send_token;

  gm_s32_n_t delay_until;
  gm_u8_n_t id;
  gm_u8_n_t disabled;
  gm_u8_n_t reserved_after_disabled[2];

  /* We only store the 16 MSbs of the
     progress time, since we only use it
     to check for fatal timeouts, which
     are huge infrequent */
  gm_s32_n_t progress_time;
}

gm_subport_t;

/* A data structure used to queue sends internally in the LANai. */
typedef union gm_send_token
{
  struct gm_st_common
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t _reserved_after_type[3];
    /* 8 */
    gm_u32_n_t sendable;	/* Must be nonzero for all types that
				   may be sent */
    gm_subport_lp_t subport;
  }
  common;
  struct gm_st_ackable
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t _reserved_after_type;
    gm_u16_n_t target_subport_id;
    /* 8 */
    gm_u32_n_t send_len;
    gm_subport_lp_t subport;
    /* 8 */
    gm_up_n_t orig_ptr;
    gm_up_n_t send_ptr;
  }
  ackable;
  struct gm_st_reliable
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t size;
    gm_u16_n_t target_subport_id;
    /* 8 */
    gm_u32_n_t send_len;
    gm_subport_lp_t subport;
    /* 8 */
    gm_up_n_t orig_ptr;
    gm_up_n_t send_ptr;
#if GM_FAST_SMALL_SEND
    gm_lp_n_t data;
#endif
#if GM_ENABLE_GALVANTECH_WORKAROUND
    /* 8 */
    gm_u16_n_t ip_checksum;
    gm_u16_n_t pad_after_ip_checksum[3];
#endif
  }
  reliable;
  struct gm_st_directed
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t size;		/* HACK */
    gm_u16_n_t target_subport_id;
    /* 8 */
    gm_u32_n_t send_len;
    gm_subport_lp_t subport;
    /* 8 */
    gm_up_n_t orig_ptr;
    gm_up_n_t send_ptr;
    /* 8 */
    gm_remote_ptr_n_t remote_ptr;
  }
  directed;
  struct gm_st_raw
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t size;		/* HACK */
    gm_u16_n_t target_subport_id;	/* HACK */
    /* 8 */
    gm_u32_n_t total_length;
    gm_subport_lp_t subport;
    /* 8 */
    gm_up_n_t orig_ptr;
    gm_up_n_t send_ptr;		/* HACK */
    gm_u32_n_t route_length;
    gm_u32_n_t reserved_after_route_length;
#if GM_ENABLE_GALVANTECH_WORKAROUND
    /* 8 */
    gm_u16_n_t ip_checksum;
    gm_u16_n_t pad_after_ip_checksum[3];
#endif
  }
  raw;
  struct gm_st_probe
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t _reserved_after_type;
    gm_u16_n_t target_subport_id;
    /* 8 */
    gm_u32_n_t send_len;
    gm_subport_lp_t subport;
  }
  probe;
  struct gm_st_ethernet
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_u8_n_t _reserved_after_type[3];
    /* 8 */
    gm_u32_n_t sendable;	/* must be nonzero */
    gm_subport_lp_t subport;
    /* NOTE: There is lots of associated state in gm.ethernet.send.state.
       This is possible because there is only one ethernet send token. */
  }
  ethernet;
  struct gm_st_datagram
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t size;
    gm_u16_n_t target_subport_id;
    /* 8 */
    gm_u32_n_t send_len;
    gm_subport_lp_t subport;
    /* 8 */
    gm_up_n_t send_ptr;
  }
  datagram;
  struct gm_st_pio_datagram
  {
    gm_send_token_lp_t next;
    gm_u8_n_t type;
    gm_s8_n_t size;
    gm_u16_n_t target_subport_id;
    /* 8 */
    gm_u32_n_t send_len;
    gm_subport_lp_t subport;
    /* 8 */
    gm_u32_n_t data[4];		/* up to 16 bytes of data */
  }
  pio_datagram;
}

gm_send_token_t;

/* Structure to keep track of unacked sends.
   There should be at least as many send records as send tokens. */
typedef struct gm_send_record
{
  gm_send_record_lp_t next;
  gm_send_token_lp_t send_token;
  /* 8 */
  gm_up_n_t before_ptr;
  gm_sexno_t sexno;
#if GM_SIZEOF_UP_T == 8
  gm_u32_n_t reserved_after_sexno;
#endif
  /* 8 */
  gm_u32_n_t before_len;
  gm_s32_n_t resend_time;
}

gm_send_record_t;

/****************************************************************
 * Host recv tokens
 ****************************************************************/

/* Structure used to pass receive tokens (which specify locations in
   which messages may be received) from the host to the LANai. */
typedef struct gm_host_recv_token
{
  gm_up_n_t volatile message;
#if GM_SIZEOF_UP_T == 8
  gm_u32_n_t _reserved1;
#endif
  gm_u8_n_t volatile tag;
  gm_u8_n_t volatile priority;
  gm_u8_n_t volatile size;
  gm_u8_n_t volatile ready;	/* must be last */
}

gm_host_recv_token_t;

/* *INDENT-OFF* */
GM_TOP_LEVEL_ASSERT (sizeof (gm_host_recv_token_t)
		     == GM_SIZEOF_HOST_RECV_TOKEN);
/* *INDENT-ON* */

typedef union gm_recv_token_key
{
  struct gm_recv_token_key_parts
  {
    gm_u16_n_t sender_node_id;
    gm_u8_n_t sender_subport_id;
    gm_u8_n_t target_subport_id;
  }
  parts;
  gm_u32_n_t whole;
}

gm_recv_token_key_t;

/* Structure stored at the end of a receive queue slot if a message is
   too big to fit directly in the recv queue slot. */
typedef struct gm_recv_token
{
  gm_recv_token_key_t key;
  gm_u8_n_t size;
  gm_u8_n_t tag;
  gm_u16_n_t reserved_after_tag;
  /* 8 */
  gm_recv_token_lp_t next;
  gm_u32_n_t _reserved_after_next;
  /* 8 */
  gm_up_n_t orig_ptr;
  gm_up_n_t recv_ptr;
  /* 8 */
  /* Special field for directed sends. */
  gm_u8_n_t start_copy_bytes[GM_MAX_DMA_GRANULARITY];
#if GM_MAX_DMA_GRANULARITY != 8
  gm_u8_n_t reserved_after_start_copy_bytes[8 - GM_MAX_DMA_GRANULARITY];
#endif
}

gm_recv_token_t;

/* State of the connection to a remote node. There is only one
   connection to each remote node, but multiple streams of data are
   multiplexed over each connection, providing reliable ordered
   delivery between subports.

   Much of the the connection state is stored directly in the
   ACK_MESSAGE.  */

#define GM_CONNECTION_FIELDS						\
    /* Fields used to form a doubly-linked list of connections with	\
       outstanding sends. */						\
    gm_connection_lp_t next_active;					\
    gm_connection_lp_t prev_active;					\
    /* 8 */								\
    gm_connection_lp_t next_to_ack;					\
    gm_u8_n_t probable_crc_error_cnt;					\
    gm_u8_n_t misrouted_packet_error_cnt;				\
    gm_u8_n_t has_route;						\
    gm_u8_n_t route_len;						\
    /* 8 */								\
    /* Ack prestaging area: ROUTE must preceed ACK_PACKET */		\
    gm_u8_n_t route[GM_MAX_NETWORK_DIAMETER];	/* 4 words */		\
    /* 8 */								\
    gm_ack_packet_t ack_packet;	/* 4 words */				\
    /* 8 */								\
    /* end of ack prestaging area */					\
    gm_sexno_t send_sexno;						\
    gm_u8_n_t ack_pending;						\
    /* routes that are not refreshed by the mapper are cleared later*/  \
    gm_s8_n_t retired;          					\
    gm_u8_n_t reserved_after_ack_pending[2];				\
    /* 8 */								\
    gm_send_record_lp_t first_send_record;				\
    gm_send_record_lp_t last_send_record;				\
    /* 8 */								\
    /* may be nonzero even if inactive */				\
    gm_subport_lp_t first_active_send_port;				\
    gm_sexno_t close_sexno;						\
    /* 8 */								\
    gm_s64_n_t open_time;						\
    /* 8 */								\
    gm_s64_n_t close_time;						\
    /* 8 */								\
    gm_s32_n_t known_alive_time;					\
    gm_u32_n_t bad_crc_cnt;						\
    /* 8 */								\
    gm_u32_n_t active_subport_bitmask;	/* 0 if inactive */		\
    gm_u32_n_t reserved_after_active_subport_bitmask

struct gm_connection_fields
{
  GM_CONNECTION_FIELDS;
};

typedef struct gm_connection
{
  GM_CONNECTION_FIELDS;
#if GM_MIN_SUPPORTED_SRAM > 256
  /* pad to 2^n bytes for fast array indexing */
  gm_u8_n_t pad[128 - sizeof (struct gm_connection_fields)];
#endif
}

gm_connection_t;

/* These macros work both from the LANai and host, since they only
   operate on bytes. */

#define GM_CONNECTION_HAS_ROUTE(c) ((c)->has_route)
#define GM_CONNECTION_ROUTE(c) (&(c)->route[GM_MAX_NETWORK_DIAMETER	\
					   - gm_ntohc ((c)->route_len)])
#define GM_CONNECTION_SET_HAS_ROUTE(c,b) ((c)->has_route = -1)
#define GM_CONNECTION_ROUTE_LEN(c) ((c)->has_route?(c)->route_len:GM_NO_ROUTE)
#define GM_CONNECTION_CLEAR_ROUTE(c) do {				\
  (c)->has_route = 0;							\
  (c)->route_len = 0;							\
  GM_STBAR ();								\
} while (0)
/* Be careful to ensure the route length is zero while setting the route. */
#define GM_CONNECTION_SET_ROUTE(c,len,ptr) do {				\
  GM_CONNECTION_CLEAR_ROUTE(c);						\
  if (len != GM_NO_ROUTE)						\
    {									\
      {									\
	char *__from, *__to, *__limit;					\
									\
	__from = (ptr);							\
	__to = (char *) &(c)->route[GM_MAX_NETWORK_DIAMETER-(len)];	\
	__limit = (char *) &(c)->route[GM_MAX_NETWORK_DIAMETER];	\
	while (__to < __limit)						\
	  *__to++ = *__from++;						\
      }									\
      GM_STBAR ();							\
      (c)->route_len = (len);						\
      (c)->has_route = -1;						\
    }									\
} while (0)

/* Structure storing all the state needed to reply to mapper scouts */
typedef struct gm_mapper_scout_reply_stage
{
  gm_u8_n_t in_send_queue;
  gm_u8_n_t route_len;
  gm_u8_n_t reserved_after_route_len[6];
  /* 8 */
  gm_u8_n_t route[GM_MAX_NETWORK_DIAMETER];
  /* 8 */
  gm_mapper_scout_reply_packet_t packet;
  /* 8 */
  gm_send_token_t send_token;
}

gm_mapper_scout_reply_stage_t;


/* Structure storing all the state needed to reply to mapper configs. */
typedef struct gm_mapper_config_reply_stage
{
  gm_u8_n_t in_send_queue;
  gm_u8_n_t route_len;
  gm_u8_n_t reserved_after_route_len[2];
  gm_u32_n_t num_routes;
  /* 8 */
  gm_u8_n_t route[GM_MAX_NETWORK_DIAMETER];
  /* 8 */
  gm_mapper_config_reply_packet_t packet;
  /* 8 */
  gm_send_token_t send_token;
}

gm_mapper_config_reply_stage_t;


#define GM_CONNECTING 1
#define GM_CONNECTED 0

/* The LANai-resident queues and other data for a user port that the
   user can modify directly.

   NOTE: This structure is mapped into user space, so no assumptions
   about the integrity of this data should be made. */

typedef struct gm_port_unprotected_lanai_side
{
  /* Host<->LANai token queues */
  struct gm_send_queue_slot send_token_queue[GM_NUM_SEND_QUEUE_SLOTS];
  /* 8 */
  /* Extra slot for wraparound. */
  gm_host_recv_token_t recv_token_queue[GM_NUM_RECV_TOKEN_QUEUE_SLOTS + 1];
#if GM_MCP
  gm_u8_n_t padding[GM_PAGE_LEN
		    - ((GM_NUM_SEND_QUEUE_SLOTS
			* sizeof (struct gm_send_queue_slot))
		       + ((GM_NUM_RECV_TOKEN_QUEUE_SLOTS + 1)
			  * sizeof (gm_host_recv_token_t)))];
#endif
}

gm_port_unprotected_lanai_side_t;

#define GM_PORT_PROTECTED_LANAI_SIDE_FIELDS				 \
    gm_send_queue_slot_lp_t send_token_queue_slot;      /*lanai addr */	 \
    gm_host_recv_token_lp_t recv_token_queue_slot;      /*lanai addr */	 \
    /* 8 */								 \
    gm_u8_n_t wake_host;						 \
    gm_u8_n_t open;							 \
    gm_u8_n_t alarm_set;						 \
    gm_u8_n_t enable_nack_down_flag;					 \
    gm_u32_n_t privileged;						 \
    /* 8 */								 \
    /* lanai internal queues */						 \
    gm_send_token_lp_t first_free_send_token;	/*lanai addr */		 \
    gm_send_token_lp_t last_free_send_token;	/*lanai addr */		 \
    /* 8 */								 \
    gm_send_token_t _send_tokens[GM_NUM_SEND_QUEUE_SLOTS /*really!*/ ];	 \
    /* 8 */								 \
    gm_recv_token_t _recv_tokens[GM_NUM_RECV_TOKENS];			 \
    /* 8 */								 \
    gm_recv_token_lp_t free_recv_token[GM_NUM_PRIORITIES][GM_NUM_SIZES]; \
    /* 8 */								 \
    /* The LANai->host recv queue */					 \
    gm_recv_token_lp_t free_recv_tokens;	/*lanai addr */		 \
    _gm_sent_token_report_lp_t sent_slot;				 \
    /* 8 */								 \
    gm_port_protected_lanai_side_lp_t next_with_alarm;			 \
    gm_u32_n_t recv_queue_slot_num;					 \
    /* 8 */								 \
    /* Assume page size is the worst-case here */			 \
    gm_dp_n_t recv_queue_slot_dma_addr [GM_NUM_RECV_QUEUE_SLOTS		 \
				     + GM_NUM_RECV_QUEUE_SLOTS%2];	 \
    /* 8 */								 \
    gm_up_n_t recv_queue_slot_host_addr [GM_NUM_RECV_QUEUE_SLOTS	 \
				      + GM_NUM_RECV_QUEUE_SLOTS%2];	 \
    /* 8 */								 \
    gm_port_protected_lanai_side_lp_t next_with_sent_packets;		 \
    gm_u32_n_t active_subport_cnt;					 \
    /* 8 */								 \
    /* Staging area for GM_SENT_EVENT events. */			 \
    gm_tokens_sent_t sent;						 \
    /* 8 */								 \
    gm_s32_n_t alarm_time;						 \
    gm_u32_n_t id;							 \
    /* 8 */								 \
    gm_u32_n_t unacceptable_recv_sizes[GM_NUM_PRIORITIES];		 \
    /* 8 */								 \
    gm_port_unprotected_lanai_side_lp_t PORT;				 \
    gm_port_protected_lanai_side_lp_t next_to_poll;

struct gm_port_protected_lanai_side_fields
{
GM_PORT_PROTECTED_LANAI_SIDE_FIELDS};

/* The protected part of of a user port.  This part of the user port
   state is NOT mapped into user space, and can, therefore, be
   trusted. */

typedef struct gm_port_protected_lanai_side
{
  GM_PORT_PROTECTED_LANAI_SIDE_FIELDS
#if GM_MIN_SUPPORTED_SRAM > 256
  GM_POW2_PAD (sizeof (struct gm_port_protected_lanai_side_fields));
#endif
}
gm_port_protected_lanai_side_t;

#if GM_MIN_SUPPORTED_SRAM > 256
GM_TOP_LEVEL_ASSERT (GM_POWER_OF_TWO (sizeof
				      (gm_port_protected_lanai_side_t)));
#endif

/****************
 * event indices.
 ****************/

/* This definition is here only to define NUM_EVENT_TYPES, which is used
   to define the gm_lanai_globals structure below. */

enum gm_event_index
{
  /* POLLing events */
  POLL_EVENT = 0,
  /* SDMA events */
  START_SDMA_EVENT,
  FINISH_SDMA_EVENT,
  /* SEND events */
  START_SEND_EVENT,
  FINISH_SEND_EVENT,
  SEND_ACK_EVENT,
  /* RECV events */
  START_RECV_PACKET_EVENT,
  FINISH_RECV_PACKET_EVENT,
  RECV_BUFFER_OVERFLOW_EVENT,
  /* RDMA events */
  START_RDMA_EVENT,
  FINISH_RDMA_EVENT,
  /* TIMER events */
  TIMER_EVENT,
  /* FAIRness events */
  FAIR_RDMA_EVENT,
  FAIR_SDMA_EVENT,
  FAIR_SDMA_RDMA_EVENT,
  /* number of events */
  NUM_EVENT_TYPES
};

/**********************************************************************/
/* A structure containing as many GM globals as possible.

   Grouping the globals into a single structure makes them easy to
   access from the host and also allows the compiler to more
   efficiently access the globals under some circumstances. */

typedef struct gm_lanai_globals
{
  gm_u32_n_t trashed_by_event_index_table_HACK;
  gm_u32_n_t magic;
  /* 8 */
  gm_u32_n_t length;
  gm_u32_n_t initialized;
  
  /****************
   * handler table
   ****************/

  /* HACK: HANDLER must be within 256 bytes of the start of this
     struct. */

  gm_handler_n_t handler[NUM_EVENT_TYPES + NUM_EVENT_TYPES % 2];
  /* 8 */

  /****************
   * small fields
   ****************/

  /* Partword fields should appear in the first 2KB of the
     globals for more efficient addressing. */

  gm_u16_n_t this_node_id;
  gm_u16_n_t max_node_id;
  gm_u16_n_t max_node_id_inuse;
  gm_u16_n_t reserved_after_max_node_id;
  /* 8 */

  /****************
   * Structs with small fields that need to be near the start of this
   * struct (in the first 2K) for efficient addressing.
   ****************/

  volatile struct gm_dma_descriptor
  {
    gm_lp_n_t next_with_flags;
    gm_u16_n_t csum[2];
    /* 8 */
    gm_u32_n_t len;
    gm_lp_n_t lanai_addr;
    /* 8 */
    gm_u32_n_t ebus_addr_high;
    gm_u32_n_t ebus_addr_low;
  }
  dma_descriptor;
  /* 8 */

  /****************
   * Frequently used fields
   ****************/

  struct
  {
    gm_port_protected_lanai_side_lp_t port;
    gm_u32_n_t reserved_after_port;
    /* 8 */
    gm_lp_n_t idle_handler;
    gm_lp_n_t active_handler;
  }
  poll;
  /* 8 */

  gm_port_unprotected_lanai_side_lp_t _PORT;
  gm_u32_n_t reserved_after__PORT;
  /* 8 */
  /* offset for converting lanai addresses to host addresses. */
  gm_port_protected_lanai_side_lp_t first_port_with_sent_packets;
  gm_subport_lp_t free_subports;
  /* 8 */
				/********************************************
				 * State machine state
				 ********************************************/
  gm_u32_n_t state;
  /* RECV state */
  gm_s32_n_t free_recv_chunk_cnt;
  /* 8 */
  gm_port_protected_lanai_side_lp_t current_rdma_port;
  gm_port_protected_lanai_side_lp_t registered_raw_recv_port;
  /* 8 */
  /* SEND state */
  gm_u32_n_t free_send_chunk_cnt;
  gm_connection_lp_t first_connection_to_ack;
  /* 8 */
  /* Send/Recv staging areas */
  gm_send_chunk_t send_chunk[2];
  /* 8 */
  gm_recv_chunk_t recv_chunk[2];
  /* 8 */
  gm_recv_token_lp_t recv_token_bin[GM_RECV_TOKEN_HASH_BINS];
  /* 8 */
  gm_connection_lp_t first_active_connection;
  gm_send_record_lp_t free_send_records;
  /* 8 */
  /* Page-crossing dma continuation state */
  gm_u32_n_t remaining_sdma_ctr;
  gm_u32_n_t remaining_rdma_ctr;
  /* 8 */
  gm_lp_n_t remaining_sdma_lar;
  gm_lp_n_t remaining_rdma_lar;
  /* 8 */
  gm_up_n_t remaining_sdma_hp;
  gm_up_n_t remaining_rdma_hp;
  /* 8 */
  gm_u32_n_t remaining_sdma_port_id;
  gm_u32_n_t remaining_rdma_port_id;
  /* 8 */
  /* Buffer for staging recv token DMAs */
  struct _gm_recv_event recv_token_dma_stage;
  /* 8 */
  gm_failed_send_event_t failed_send_event_dma_stage;
  /* 8 */
  struct
  {
    gm_u8_n_t _reserved[GM_RDMA_GRANULARITY - 1];
    gm_u8_n_t type;
  }
  report_dma_stage;
  /* 8 */
  gm_u32_n_t nack_delay;
  gm_s32_n_t rand_seed;
  /* 8 */
  gm_u32_n_t backlog_delay;
  /* the following three variables are made 32 bits for the Alpha,
     which can only perform 32-bit reads and write. */
  volatile gm_u32_n_t pause_rqst;
  /* 8 */
  gm_s32_n_t port_to_close;
  gm_s32_n_t port_to_open;
  /* 8 */
  gm_port_protected_lanai_side_lp_t finishing_rdma_for_port;
  gm_port_protected_lanai_side_lp_t first_port_with_alarm;
  /* 8 */
  volatile gm_u32_n_t resume_after_halt;
  volatile gm_s32_n_t volatile_zero;	/* BAD */
  /* 8 */
  struct
  {
    /* specify how to create DMA addrs on machines that don't have
       virtual memory. */
    gm_dp_n_t clear;
    gm_dp_n_t set;
  }
  no_VM;
  struct
  {
    gm_lp_n_t pushed_sdma_handler;
    gm_lp_n_t pushed_rdma_handler;
  }
  timer_state;

  /****************
   * infrequently used fields
   ****************/

  /* 8 */
  union
  {
    volatile gm_u32_n_t type;
    struct
    {
      volatile gm_u32_n_t type;
      volatile gm_u32_n_t port;
    }
    wake;
    struct
    {
      volatile gm_u32_n_t type;
      volatile gm_lp_n_t string;
    }
    print;
    struct
    {
      volatile gm_u32_n_t type;
      volatile gm_lp_n_t file;
      volatile gm_u32_n_t line;
      volatile gm_lp_n_t text;
    }
    failed_assertion;
    struct
    {
      volatile gm_u32_n_t type;
      volatile gm_u32_n_t len;
      volatile gm_u16_n_t checksum;	/* 0 if unknown */
      gm_u8_n_t reserved_after_checksum[6];
    }
    ethernet_recv;
    struct
    {
      volatile gm_u32_n_t type;
      gm_u32_n_t length;
      gm_lp_n_t buffer;
    }
    write;
    volatile gm_u8_n_t _reserved[16];	/* make size of union be 16 bytes */
  }
  interrupt;
  /* 8 */
  gm_s32_n_t timeout_time;
  gm_s32_n_t sram_length;
  /* 8 */
  gm_s32_n_t led;
  gm_u32_n_t reserved_after_led;
  /* 8 */
  gm_u32_n_t l2e_time[4];
  gm_u32_n_t e2l_time[4];
  /* 8 */
  gm_u32_n_t bus_width;
  gm_u32_n_t bus_rate;
  /* 8 */
  gm_u8_n_t trash[8];
  /* 8 */
  gm_u8_n_t eight_zero_bytes[8];
  /* 8 */
  struct gm_mapper_state
  {
    gm_u8_n_t unique_id[6];

    gm_mapper_scout_reply_stage_t scout_reply;
    /* 8 */
    gm_mapper_config_reply_stage_t config_reply;
    /* 8 */
  }
  mapper_state;
  /* 8 */
  gm_dp_n_t name_table_piece[GM_MAX_NUM_NAME_TABLE_PIECES];
  /* 8 */

  /****************
   * Large ethernet state
   ****************/

  /* The ethernet state should be near the end so that the GM state
     above can be addressed more efficiently using smaller
     offsets from the pointer to this structure. */

  /* 8 */
  struct
  {
    struct
    {
      /* A token reserved for ethernet sends */
      gm_send_token_t token;
      /* 8 */

      /* the staging areas for broadcasts */
      struct
      {
	/* 8 */
	gm_u8_n_t filled;
	gm_u8_n_t _reserved_after_filled[5];
	gm_u8_n_t padding[2];
	/* 8 */
	gm_u8_n_t route[GM_MAX_NETWORK_DIAMETER];
	/* 8 */
	union
	{
	  gm_u8_n_t as_bytes[1];
	  struct
	  {
	    char target_ethernet_id[6];
	    char source_ethernet_id[6];
	    char payload[GM_ETHERNET_MTU - 12 + (((GM_ETHERNET_MTU) % 8)
						 ? (8 - (GM_ETHERNET_MTU) % 8)
						 : 0)];
	  }
	  as_ethernet;
	  struct
	  {
	    gm_u16_n_t myrinet_packet_type;
	    char target_ethernet_id[6];
	    char source_ethernet_id[6];
	  }
	  as_marked_ethernet;
	}
	packet;
      }
      chunk[2];
      /* 8 */
      gm_u16_n_t target;
      gm_u8_n_t gather_cnt;
      gm_u8_n_t gather_pos;
      gm_lp_n_t next_lar;
      /* 8 */
        gm_ethernet_segment_descriptor_t
	gather_segment[GM_MAX_ETHERNET_GATHER_CNT];
      /* 8 */
      gm_u32_n_t total_len;
      gm_u8_n_t mark;
      gm_u8_n_t busy;
      gm_u16_n_t _reserved_after_busy;
      /* 8 */
      struct
      {
	gm_u32_n_t saved_word;
	gm_s8_n_t shift;
	gm_u8_n_t restore;
	gm_u16_n_t _reserved;
      }
      byte_dma_emul;
    }
    send;
    struct
    {
      gm_ethernet_recv_token_t token[GM_NUM_ETHERNET_RECV_TOKENS];
      /* 8 */
      gm_ethernet_recv_token_lp_t token_slot;
      gm_lp_n_t next_lar;
      gm_s32_n_t remaining_len;
      gm_u32_n_t total_len;
      /* 8 */
      gm_u8_n_t scatter_pos;
      gm_u8_n_t reserved_after_scatter_pos;
      gm_u16_n_t checksum;
      gm_u8_n_t reserved[4];
      /* 8 */
    }
    recv;
    /* 8 */
    gm_u8_n_t addr_stage[8];
    /* 8 */
    gm_dp_n_t addr_table_piece[GM_MAX_NUM_ADDR_TABLE_PIECES];
  }
  ethernet;
  /* 8 */

  /****************
   * Debug stuff
   ****************/

  /* Debug stuff should be close to the end since indexing to
     large offsets is more costly, and when the debug code is not
     used, we would like to save the small offsets for more
     important stuff. */

  gm_u8_lp_n_t event_index_table;
  gm_u32_n_t reserved_after_event_index_table;
  /* 8 */
  gm_s8_n_t dispatch_seen[GM_ROUNDUP (u32, GM_DISPATCH_MAX_NUM, 4)][2];
  /* 8 */
  gm_u32_n_t dispatch_cnt[GM_DISPATCH_MAX_NUM][2];
  /* 8 */
  gm_u32_n_t pause_cnt;
  gm_u32_n_t logtime_index;
  /* 8 */
  gm_s32_n_t record_log;
  gm_lp_n_lp_n_t log_slot;
  /* 8 */
  gm_lp_n_lp_n_t log_end;
  gm_s8_lp_n_t lzero;
  /* 8 */
#if GM_LOG_DISPATCHES
  gm_lp_n_t  log[GM_ROUNDUP (u32, GM_LOG_LEN, 2)];
  gm_u32_n_t logtime[GM_ROUNDUP (u32, GM_LOG_LEN, 2)];
  /* 8 */
#endif
  volatile gm_s8_lp_n_t current_handler;
  volatile gm_s8_lp_n_t current_handler_extra;
  /* 8 */
  /* debugging info */
  struct
  {
    gm_u32_n_t first_dma_page;
#if GM_SIZEOF_DP_T == 8
    gm_u32_n_t reserved;
#endif
    gm_dp_n_t min_dma_addr;
    /* 8 */
    gm_lp_t old_fp;
    gm_u32_t reserved_after_old_fp;
    /* 8 */
    gm_dma_page_bitmap_t dma_page_bitmap;
  }
  debug;
  /* 8 */
#if GM_ENABLE_TRACE
  gm_u32_n_t trace_active;
  gm_u32_n_t trace_index;
  /* 8 */
  gm_l_trace_t trace_log[GM_LANAI_NUMTRACE];
#endif

  /* Error counters */

#define GM_ERROR_CNT(name, desc) gm_u32_n_t name ## _error_cnt;
    GM_ERROR_COUNTERS
#undef GM_ERROR_CNT
    /* debug counters */
#define GM_DEBUG_CNT(name, desc) gm_u32_n_t name ## _debug_cnt;
    GM_DEBUG_COUNTERS
#undef GM_DEBUG_CNT
    /* counter padding */
    gm_u32_n_t counter_pad[2 - (GM_NUM_ERROR_COUNTERS
				+ GM_NUM_DEBUG_COUNTERS) % 2];
  /* 8 */
  gm_u32_n_t netrecv_cnt;
  gm_u32_n_t netsend_cnt;
  /* 8 */
  gm_sexno_t packet_sexno;
  gm_sexno_t ack_sexno;
  /* 8 */
  gm_u32_n_t while_waiting;
  gm_u32_n_t reserved_after_while_waiting;
  /* 8 */

  /****************
   * Arrays: Large arrays go at the end to reduce the cost of
   * referencing the smaller fields above
   ****************/

  /* Arrays used to allocate storage, but never indexed into except
     during initialization. */

  gm_send_record_t _send_record[GM_NUM_SEND_RECORDS];
  /* 8 */
  gm_subport_t _subport[GM_NUM_SUBPORTS];
  /* 8 */

  /****************
   * big arrays
   ****************/

  /* 8 */
  struct
  {
    /* Description of host-resident page hash table, which is broken
       into pieces. */
    gm_dp_n_t bogus_sdma_ptr;
    gm_dp_n_t bogus_rdma_ptr;
    /* 8 */
    /* A cache of the most recently used hash table entries. */
    gm_page_hash_cache_t cache;
    /* 8 */
  }
  page_hash;

  /* In the MCP, this should be referenced only via the "gm_port" macro
     for best performance. */

  gm_port_protected_lanai_side_t port[GM_NUM_PORTS];
  /* 8 */
  gm_s8_n_t malloc_ram[1 << 14];
  /* 8 */
  gm_u8_lp_n_t last_dispatch;
  gm_s32_n_t last_reported_second;
  /* 8 */

  /****************
   * End magic and connections
   ****************/

  gm_u32_n_t event_index_table_crc;
  gm_u32_n_t end_magic;		/* just before connection[] */
  /* 8 */
  /* Connections grow off the end of the globals, consuming as much
     memory as is available, so this should be at the end */
  gm_connection_t connection[1];
}

gm_lanai_globals_t;

/***********************************************************************
 * Inline functions (used by LANai and host)
 ***********************************************************************/

#define GM_MAX_LENGTH_FOR_SIZE(size) ((gm_u32_t)((size)<3?0:(1<<(size))-8))

/****
 * Pointer conversion macros:
 * "dp" versions convert DMA pointers (32 bits)
 * "lp" versions convert LANai pointers (32 bits)
 * "hp" versions convert host pointers (64 or 32 bits)
 ****/

/****************
 * DMA pointers
 ****************/

gm_inline static gm_dp_n_t
gm_hton_dp (gm_dp_t dp)
{
  gm_dp_n_t ret;

#if GM_SIZEOF_DP_T == 8
  GM_N (ret) = __gm_hton_u64 (dp);
#else
  GM_N (ret) = __gm_hton_u32 (dp);
#endif
  return ret;
}

gm_inline static gm_dp_t
gm_ntoh_dp (gm_dp_n_t dp)
{
#if GM_SIZEOF_DP_T == 8
  return __gm_ntoh_u64 (GM_N (dp));
#elif GM_SIZEOF_DP_T == 4
  return __gm_ntoh_u32 (GM_N (dp));
#endif
}

/****************
 * LANai pointers
 ****************/

gm_inline static gm_lp_n_t
gm_hton_lp (gm_lp_t lp)
{
  gm_lp_n_t ret;

  GM_N (ret) = (gm_lp_t) __gm_hton_u32 ((gm_u32_t) lp);
  return ret;
}

gm_inline static gm_lp_t
gm_ntoh_lp (gm_lp_n_t lp)
{
  return (gm_lp_t) __gm_ntoh_u32 ((gm_u32_t) GM_N (lp));
}

/****************
 * Remote user pointers (assumed to be 64 bits)
 ****************/

gm_inline static gm_remote_ptr_n_t
gm_hton_rp (gm_remote_ptr_t rp)
{
  gm_remote_ptr_n_t ret;

  GM_N (ret) = (gm_remote_ptr_t) __gm_hton_u64 ((gm_u64_t) rp);
  return ret;
}

gm_inline static gm_remote_ptr_t
gm_ntoh_rp (gm_remote_ptr_n_t rp)
{
  return (gm_remote_ptr_t) __gm_ntoh_u64 ((gm_u64_t) GM_N (rp));
}

/****************
 * User pointers
 ****************/

gm_inline static gm_up_n_t
gm_hton_up (gm_up_t _gm_up)
{
  gm_up_n_t ret;

#if GM_SIZEOF_UP_T == 8
  GM_N (ret) = __gm_hton_u64 (_gm_up);
#else
  GM_N (ret) = __gm_hton_u32 (_gm_up);
#endif
  return ret;
}

gm_inline static gm_up_t
gm_ntoh_up (gm_up_n_t _gm_up)
{
#if GM_SIZEOF_UP_T == 8
  return __gm_ntoh_u64 (GM_N (_gm_up));
#elif GM_SIZEOF_UP_T == 4
  return __gm_ntoh_u32 (GM_N (_gm_up));
#endif
}

#endif /* ifdef _gm_types_h_ */


/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  End:
*/
