/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file implements call-tracing hooks:
   
   GM_CALLED(): Report that a function was called.
   GM_TRACE(): Report a message.
   GM_RETURN(val): Report that a function is returning and return;
   GM_RETURN_NOTHING(): Report that a function is returning and return
   	nothing. */

#ifndef _gm_call_trace_h_
#define _gm_call_trace_h_

#include "gm_cpp.h"
#include "gm_debug.h"

/****************************************************************
 * Feature switches
 ****************************************************************/

/* Global feature switch: Turn on all tracing. */

#define GM_ENABLE_CALL_TRACE 0

/* Local feature switch:
   
   Redefine this to 1 IN OTHER FILES to enable tracing despite
   GM_ENABLE_CALL_TRACE.  For example, the following turns on tracing
   for the function FOO() only, but does not effect tracing for other
   functions in the file.

   > ...
   > #undef GM_LOCALLY_ENABLE_CALL_TRACE
   > #define GM_LOCALLY_ENABLE_CALL_TRACE 1
   > void foo (void)
   > {
   >   ...
   > }
   > #undef GM_LOCALLY_ENABLE_CALL_TRACE
   > #define GM_LOCALLY_ENABLE_CALL_TRACE 0
   > ...
*/

#define GM_LOCALLY_ENABLE_CALL_TRACE 0	/* DO NOT CHANGE HERE! */

/****************************************************************
 * Implementation
 ****************************************************************/

const char *__gm_print_called_space (void);
const char *__gm_print_return_space (void);

/* Combine the global and local feature switches. */

#define __GM_ENABLE_CALL_TRACE (GM_ENABLE_CALL_TRACE			\
				|| GM_LOCALLY_ENABLE_CALL_TRACE)

/* NOTE: Use of the "if GM_DEBUG" conditions below is predicated on the
 *       assumption that GM_PRINT has no effect unless GM_DEBUG is true.
 *       We employ this conditional in order to keep fussy compilers
 *       like MIPSpro C from complaining about unused variables etc.
 */

/* report that the current function has been called */

#if GM_DEBUG
#define GM_CALLED() do {				\
  const char *__gm_call_trace_space;			\
							\
  __gm_call_trace_space = __gm_print_called_space ();	\
  if (__GM_ENABLE_CALL_TRACE)				\
    {							\
      GM_PRINT (1, ("%s" __GM_FUNCTION__ "() called\n",	\
		    __gm_call_trace_space));		\
    }							\
} while (0)
#else
#define GM_CALLED()
#endif

#if GM_DEBUG
#define GM_CALLED_WITH_ARGS(args) do {			\
  const char *__gm_call_trace_space;			\
							\
  __gm_call_trace_space = __gm_print_called_space ();	\
  if (__GM_ENABLE_CALL_TRACE)				\
    {							\
      GM_PRINT (1, ("%s" __GM_FUNCTION__ "(",		\
		    __gm_call_trace_space));		\
      _GM_PRINT (1, args);				\
      _GM_PRINT (1, (") called\n"));			\
    }							\
} while (0)
#else
#define GM_CALLED_WITH_ARGS(args)
#endif
     
/* report that the current function is not implemented */

#define GM_NOT_IMP() GM_WARN (("not implemented\n"))

/* report that the current function is returning */

#if GM_DEBUG
#define GM_RETURN(val) do {				\
  const char *__gm_call_trace_space;			\
							\
  __gm_call_trace_space = __gm_print_return_space ();	\
  if (__GM_ENABLE_CALL_TRACE)				\
    {							\
      GM_PRINT (1, ("%s"__GM_FUNCTION__"() returning\n",	\
		    __gm_call_trace_space));		\
    }							\
  return (val);						\
} while (0)
#else
#define GM_RETURN(val) do { return (val); } while (0)
#endif

/* report that the current function is returning an int, and print
   the return value. */

#if GM_DEBUG
#define GM_RETURN_INT(val) do {					\
  int gm_return_int;						\
  const char *__gm_call_trace_space;				\
								\
  __gm_call_trace_space = __gm_print_return_space ();		\
  gm_return_int = val;						\
  if (__GM_ENABLE_CALL_TRACE)					\
    {								\
      GM_PRINT (1, ("%s"__GM_FUNCTION__"() returning %d\n",	\
		    __gm_call_trace_space,			\
		    gm_return_int));				\
    }								\
  return (gm_return_int);					\
} while (0)
#else
#define GM_RETURN_INT(val) do { return (val); } while (0)
#endif

/* report that the current function is returning a pointer, and print
   the return value. */

#if GM_DEBUG
#define GM_RETURN_PTR(val) do {					\
  const char *__gm_call_trace_space;				\
								\
  __gm_call_trace_space = __gm_print_return_space ();		\
  if (__GM_ENABLE_CALL_TRACE)					\
    {								\
      void *gm_return_ptr;					\
								\
      gm_always_assert (sizeof (val) == sizeof (void *));	\
      gm_return_ptr = (void *) val;				\
      GM_PRINT (1, ("%s"__GM_FUNCTION__"() returning %p\n",	\
		    __gm_call_trace_space,			\
		    gm_return_ptr));				\
    }								\
  return (val);							\
} while (0)
#else
#define GM_RETURN_PTR(val) do {					\
  if (__GM_ENABLE_CALL_TRACE)					\
    {								\
      gm_always_assert (sizeof (val) == sizeof (void *));	\
    }								\
  return (val);							\
} while (0)
#endif

/* report that the current function is returning a status, and print
   the return value. */

#if GM_DEBUG
#define GM_RETURN_STATUS(val) do {				\
  gm_status_t gm_return_status;					\
  const char *__gm_call_trace_space;				\
								\
  __gm_call_trace_space = __gm_print_return_space ();		\
  gm_return_status = val;					\
  if (__GM_ENABLE_CALL_TRACE)					\
    {								\
      if (__GM_ENABLE_CALL_TRACE)				\
	{							\
	  if (gm_return_status != GM_SUCCESS)			\
	    {							\
	      GM_PRINT (1, ("%s"__GM_FUNCTION__			\
			    "() returning with error %d\n",	\
			    __gm_call_trace_space,		\
			    gm_return_status));			\
	    }							\
	  else							\
	    {							\
	      GM_PRINT (1, ("%s"__GM_FUNCTION__			\
			    "() returning successfully\n",	\
			    __gm_call_trace_space));		\
	    }							\
	}							\
    }								\
  return (gm_return_status);					\
} while (0)
#else
#define GM_RETURN_STATUS(val) do { return (val); } while (0)
#endif

/* report that the current function is returning (without a
   return value). */

#if GM_DEBUG
#define GM_RETURN_NOTHING() do {				\
  const char *__gm_call_trace_space;				\
								\
  __gm_call_trace_space = __gm_print_return_space ();		\
  if (__GM_ENABLE_CALL_TRACE)					\
    {								\
      GM_PRINT (1, ("%s"__GM_FUNCTION__"() returning\n",	\
		    __gm_call_trace_space));			\
    }								\
  return;							\
} while (0)
#else
#define GM_RETURN_NOTHING() do { return; } while (0)
#endif

/* Print message M if tracing is enabled. */

#define GM_TRACE(m) GM_PRINT (__GM_ENABLE_CALL_TRACE, (": " m)


#endif /* _gm_call_trace_h_ */

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  End:
*/
