/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#ifndef _gm_h_
#define _gm_h_

#if GM_MCP || GM_BUILDING_FIRMWARE
#include "gm_config.h"
#endif

/* This file includes the minimum amount needed to support the GM API.
   It should not include any other header file or depend on any macro
   definitions that are not automatically set by the compiler. */

/************************************************************************
 * Begin Prologue
 ************************************************************************/

#ifdef	__cplusplus
extern "C"
{
#if 0
}				/* indent hack */
#endif
#endif

/************************************************************************
 * End Prologue
 ************************************************************************/

/* For htonl, etc. */

/****
 * Constants
 ****/

#define GM_MAX_HOST_NAME_LEN 128
#define GM_MAX_PORT_NAME_LEN 32
#define GM_NO_SUCH_NODE_ID 0

/************
 * Determine CPU based on what compiler is being used.
 ************/

#if GM_CPU_alpha
#  define GM_CPU_DEFINED 1
#elif GM_CPU_lanai
#  define GM_CPU_DEFINED 1
#elif GM_CPU_mips
#  define GM_CPU_DEFINED 1
#elif GM_CPU_powerpc
#  define GM_CPU_DEFINED 1
#elif GM_CPU_sparc
#  define GM_CPU_DEFINED 1
#elif GM_CPU_sparc64
#  define GM_CPU_DEFINED 1
#elif GM_CPU_x86
#  define GM_CPU_DEFINED 1
#elif GM_CPU_hppa
#  define GM_CPU_DEFINED 1
#elif GM_CPU_ia64
#  define GM_CPU_DEFINED 1
#else
#  define GM_CPU_DEFINED 0
#endif

#if !GM_CPU_DEFINED
#  if defined _MSC_VER		/* Microsoft compiler */
#    if defined _M_IX86
#      define GM_CPU_x86 1
#    elif defined _M_IA64
#      define GM_CPU_ia64 1
#    elif defined _M_ALPHA
#      define GM_CPU_alpha 1
#    else
#      error Could not determine CPU type.  You need to modify gm.h.
#    endif
#  elif defined mips
#    define GM_CPU_mips 1
#  elif defined(__GNUC__)
#    if #cpu(alpha)
#      define GM_CPU_alpha 1
#    elif #cpu(hppa)
#      define GM_CPU_hppa 1
#    elif defined lanai
#      define GM_CPU_lanai 1
#    elif defined lanai3
#      define GM_CPU_lanai 1
#    elif defined lanai7
#      define GM_CPU_lanai 1
#    elif defined(powerpc)
#      define GM_CPU_powerpc 1
#    elif #cpu(sparc64)
#      define GM_CPU_sparc64 1
#    elif defined sparc
#      define GM_CPU_sparc 1
#    elif defined i386
#      define GM_CPU_x86 1
#    elif defined(CPU)
#      if (CPU==PPC603)
#	 define GM_CPU_powerpc 1
#      elif (CPU==PPC604)
#	 define GM_CPU_powerpc 1
#      endif
#    else
#      error Could not determine CPU type.  You need to modify gm.h.
#    endif
#  elif (defined (__DECC) || defined (__DECCXX)) && defined(__alpha)
#      define GM_CPU_alpha 1
#  elif (defined (__SUNPRO_C) || defined(__SUNPRO_CC)) && (defined(sparc64) || defined(__sparcv9))
#      define GM_CPU_sparc64 1
#  elif (defined (__SUNPRO_C) || defined(__SUNPRO_CC)) && defined(sparc)
#      define GM_CPU_sparc 1
#  elif (defined (__SUNPRO_C) || defined(__SUNPRO_CC)) && defined i386
#      define GM_CPU_x86 1
#  elif defined __PGI 
#      define GM_CPU_x86 1
#  elif defined __linux__	/* Portland Group Compiler */
#    if defined __i386__
#      define GM_CPU_x86 1
#    else
#      error Could not determine CPU type.  You need to modify gm.h.
#    endif
#  elif defined(__hppa) || defined(_PA_RISC1_1)
#      define GM_CPU_hppa 1
#  else
#    error Could not determine CPU type.  You need to modify gm.h.
#  endif
#  undef GM_CPU_DEFINED
#  define GM_CPU_DEFINED 1
#endif

/************
 * Enable inlining if and only if we know it won't crash the compiler.
 ************/


#ifndef gm_inline
#  if defined _MSC_VER
#    define gm_inline __inline
#  elif defined __GNUC__
#    define gm_inline inline
#  elif GM_CPU_mips
#    define gm_inline __inline
#    define inline __inline
#  elif defined __DECC
#    define gm_inline __inline
#  elif defined __DECCXX
#    define gm_inline
#  else
#    define gm_inline
#  endif
#endif

/************
 * Define sized types
 ************/

/* gm_s64_t   64-bit signed   integer
   gm_s32_t   32-bit signed   integer
   gm_s16_t   16-bit signed   integer
   gm_s8_t     8-bit signed   integer
   gm_u64_t   64-bit unsigned integer
   gm_u32_t   32-bit unsigned integer
   gm_u16_t   16-bit unsigned integer
   gm_u8_t     8-bit unsigned integer */

#if defined(__GNUC__) || defined(__DECC) || defined(__DECCXX)
#if (GM_CPU_sparc64 && defined(__linux__)) && !GM_MCP
#if GM_KERNEL
typedef signed long gm_s64_t;
typedef unsigned long gm_u64_t;
#else
typedef signed long long gm_s64_t;
typedef unsigned long long gm_u64_t;
#endif /* GM_KERNEL */
#else
#if (GM_CPU_alpha || GM_CPU_sparc64) && !GM_MCP
typedef signed long gm_s64_t;
typedef unsigned long gm_u64_t;
#else
typedef signed long long gm_s64_t;
typedef unsigned long long gm_u64_t;
#endif
#endif

typedef signed int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned int gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#elif defined _MSC_VER
typedef signed __int64 gm_s64_t;
typedef unsigned __int64 gm_u64_t;
typedef signed __int32 gm_s32_t;
typedef signed __int16 gm_s16_t;
typedef signed __int8 gm_s8_t;
typedef unsigned __int32 gm_u32_t;
typedef unsigned __int16 gm_u16_t;
typedef unsigned __int8 gm_u8_t;
#elif GM_CPU_mips		/* see /usr/include/sgidefs.h */
#if (_MIPS_SZLONG == 64)
typedef long gm_s64_t;
typedef unsigned long gm_u64_t;
#elif defined(_LONGLONG)
typedef long long gm_s64_t;
typedef unsigned long long gm_u64_t;
#else
/* __long_long is a hidden builtin, ansi-compliant 64-bit type.  It
   should be used only here; henceforward, the gm_ types should be used. */
typedef __long_long gm_s64_t;
typedef unsigned __long_long gm_u64_t;
#endif
typedef int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#elif defined __PGI		/* Need cpu macros? */
typedef signed long long gm_s64_t;
typedef signed int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned long long gm_u64_t;
typedef unsigned int gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#elif (defined (__SUNPRO_C) || defined (__SUNPRO_CC)) && (defined(sparc64) || defined(__sparcv9))
typedef signed long gm_s64_t;
typedef signed int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned long gm_u64_t;
typedef unsigned int gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#elif (defined (__SUNPRO_C) || defined (__SUNPRO_CC)) && defined(__sparc)
typedef signed long long gm_s64_t;
typedef signed int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned long long gm_u64_t;
typedef unsigned int gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#elif (defined (__SUNPRO_C) || defined (__SUNPRO_CC)) && defined i386
typedef signed long long gm_s64_t;
typedef signed int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned long long gm_u64_t;
typedef unsigned int gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#elif defined(__hpux)
typedef signed long gm_s64_t;
typedef signed int gm_s32_t;
typedef signed short gm_s16_t;
typedef signed char gm_s8_t;
typedef unsigned long gm_u64_t;
typedef unsigned int gm_u32_t;
typedef unsigned short gm_u16_t;
typedef unsigned char gm_u8_t;
#else
#  error Could not define sized types.  You need to modify gm.h.
#endif

/************
 * LANai-compatible host-side pointer representation
 ************/

/* Define size of host pointers */

#if GM_MCP
#  define GM_SIZEOF_VOID_P 4
#else
#  if GM_CPU_sparc64 && defined(__linux__) && !GM_KERNEL
#    define GM_SIZEOF_VOID_P 4
#  elif GM_CPU_sparc64
#    define GM_SIZEOF_VOID_P 8
#  elif GM_CPU_sparc
#    define GM_SIZEOF_VOID_P 4
#  elif GM_CPU_x86
#    define GM_SIZEOF_VOID_P 4
#  elif GM_CPU_ia64
#    define GM_SIZEOF_VOID_P 8
#  elif GM_CPU_alpha
#    define GM_SIZEOF_VOID_P 8
#  elif GM_CPU_mips
#    ifdef _MIPS_SZPTR
#      define GM_SIZEOF_VOID_P (_MIPS_SZPTR / 8)
#    else
#      error Failed to define _MIPS_SZPTR
#    endif
#  elif GM_CPU_powerpc
#    define GM_SIZEOF_VOID_P 4
#  elif GM_CPU_hppa
#    define GM_SIZEOF_VOID_P 8
#  else
#    error Could not determine host pointer size.  You need to modify gm.h.
#  endif
#endif

/* GM_SIZEOF_UP_T */

#if GM_MCP

#ifndef GM_SIZEOF_UP_T
#error GM_SIZEOF_UP_T should have been defined by gm/mcp/genmak
#endif

#else				/* !GM_MCP */

#ifdef GM_SIZEOF_UP_T
#error GM_SIZEOF_UP_T should not be defined here
#endif

#if (GM_CPU_sparc64 || GM_CPU_sparc) && defined(__linux__)
#define GM_SIZEOF_UP_T 4
#else
#define GM_SIZEOF_UP_T GM_SIZEOF_VOID_P
#endif

#endif				/* !GM_MCP */

/* The "gm_up_t" is a LANai-compatible representation of a user
   virtual memory address.  If the host has 32-bit pointers, then a
   gm_up_t has 32 bits.  If the host has 64-bit pointers, but only
   32-bits of these pointers are used, then a gm_up_t still has 32
   bits (for performance reasons).  Finally, if the host has 64-bit
   pointers and more than 32 bits of the pointer is used, then a
   gm_up_t has 64-bits. */


/* Define host pointer types 

   gm_up_t   unsigned int the size of a user pointer */

#if GM_SIZEOF_UP_T == 4
typedef gm_u32_t gm_up_t;
#elif GM_SIZEOF_UP_T == 8
typedef gm_u64_t gm_up_t;
#else
#  error Host pointer size is not supported.
#endif

typedef gm_u64_t gm_remote_ptr_t;	/* assume remote pointers are large */

/****************************************************************
 * Endianess
 ****************************************************************/

/* Determine endianness */

#if GM_CPU_alpha
#  define GM_CPU_BIGENDIAN 0
#elif GM_CPU_lanai
#  define GM_CPU_BIGENDIAN 1
#elif GM_CPU_mips
#  define GM_CPU_BIGENDIAN 1
#elif GM_CPU_powerpc
#  define GM_CPU_BIGENDIAN 1
#elif GM_CPU_sparc
#  define GM_CPU_BIGENDIAN 1
#elif GM_CPU_sparc64
#  define GM_CPU_BIGENDIAN 1
#elif GM_CPU_hppa
#  define GM_CPU_BIGENDIAN 1
#elif GM_CPU_x86
#  define GM_CPU_BIGENDIAN 0
#elif GM_CPU_ia64
#  define GM_CPU_BIGENDIAN 0
#else
#  error Could not determine endianness.  You need to modify gm.h.
#endif

/****************
 * Network-order types, strongly typed on the host
 ****************/

/* The default behaviour is to turn on strong typing, except when
   using 64-bit Solaris compilers, which will pad the strong types,
   breaking this feature. */

#ifndef GM_STRONG_TYPES
#if GM_OS_SOLARIS && GM_SIZEOF_UP_T == 8
#define GM_STRONG_TYPES 0
#else
#define GM_STRONG_TYPES 1
#endif
#endif /* ndef GM_STRONG_TYPES */

/* Prevent usage of strong types on 64-bit Solaris machines.  These are
   big-endian anyway, so no one should not need it there. */

#if GM_SIZEOF_UP_T == 8 && GM_STRONG_TYPES && GM_OS_SOLARIS 
#error GM_STRONG_TYPES is incompatible with 64-bit Solaris.
#endif

#if !GM_MCP && GM_STRONG_TYPES

typedef struct
{
  gm_remote_ptr_t n;
}
gm_remote_ptr_n_t;
typedef struct
{
  gm_s16_t n;
}
gm_s16_n_t;
typedef struct
{
  gm_s32_t n;
}
gm_s32_n_t;
typedef struct
{
  gm_s64_t n;
}
gm_s64_n_t;
typedef struct
{
  gm_s8_t n;
}
gm_s8_n_t;
typedef struct
{
  gm_u16_t n;
}
gm_u16_n_t;
typedef struct
{
  gm_u32_t n;
}
gm_u32_n_t;
typedef struct
{
  gm_u64_t n;
}
gm_u64_n_t;
typedef struct
{
  gm_u8_t n;
}
gm_u8_n_t;
typedef struct
{
  gm_up_t n;
}
gm_up_n_t;
typedef gm_up_n_t gm_up_n_up_n_t;
#define GM_N(x) ((x).n)

#else /* GM_MCP || !GM_STRONG_TYPES */

typedef gm_remote_ptr_t gm_remote_ptr_n_t;
typedef gm_s16_t gm_s16_n_t;
typedef gm_s32_t gm_s32_n_t;
typedef gm_s64_t gm_s64_n_t;
typedef gm_s8_t gm_s8_n_t;
typedef gm_u16_t gm_u16_n_t;
typedef gm_u32_t gm_u32_n_t;
typedef gm_u64_t gm_u64_n_t;
typedef gm_u8_t gm_u8_n_t;
typedef gm_up_t gm_up_n_t;
typedef gm_up_t gm_up_n_up_n_t;
#define GM_N(x) (x)

#endif /* GM_MCP || !GM_STRONG_TYPES*/

/****************************************************************
 * Byte order conversion
 ****************************************************************/

/********************************
 * Unconditional conversion
 ********************************/

/* Define __gm_swap* macros, which always reverse the order of bytes. */

gm_inline static gm_u16_t
__gm_swap_u8 (gm_u8_t x)
{
  return x;
}

gm_inline static gm_u16_t
__gm_swap_u16 (gm_u16_t x)
{
  return (((x >> 8) & 0xff) | ((x & 0xff) << 8));
}

/* Architecture-independent C implementation. */

gm_inline static gm_u32_t
__gm_swap_u32_C (gm_u32_t x)
{
  return (((x >> 24) & 0xff)
	  | ((x >> 8) & 0xff00) | ((x & 0xff00) << 8) | ((x & 0xff) << 24));
}

/* Architecture-specific assembly implementations */

gm_inline static gm_u32_t
__gm_swap_u32_asm (gm_u32_t x)
{
#if defined __GNUC__ && GM_CPU_x86 && __OPTIMIZE__ && 0
__asm__ ("bswap %0": "=r" (x):"0" (x));
  return x;
#else
  return __gm_swap_u32_C (x);
#endif
}

gm_inline static gm_u32_t
__gm_swap_u32 (gm_u32_t x)
{
#if defined __GNUC__
  /* Use C implementation for constants to allow compile-time swapping.
     Otherwise, use fast assembly implementation. */
  return __builtin_constant_p (x) ? __gm_swap_u32_C (x) :
    __gm_swap_u32_asm (x);
#else
  return __gm_swap_u32_C (x);
#endif
}

/* These unconditional conversion macros are used by the conditional
   conversion macros below */

gm_inline static gm_u64_t
__gm_swap_u64 (gm_u64_t x)
{
  volatile union
  {
    gm_u64_t u64;
    gm_u32_t u32[2];
  }
  ret, old;

  old.u64 = x;
  ret.u32[0] = __gm_swap_u32 (old.u32[1]);
  ret.u32[1] = __gm_swap_u32 (old.u32[0]);
  return ret.u64;
}

/****************************************************************
 * Strongly typed conversion
 ****************************************************************/

/********************************
 * Conditional conversion
 ********************************/

#if GM_CPU_BIGENDIAN || GM_MCP
#define GM_NET_SWAP 0
#else
#define GM_NET_SWAP 1
#endif

/* These gm_ntoh* macros perform strong type checking, allowing the
   compiler to ensure that no network-ordered fields are used by host
   code without first being converted to host byte order by these
   functions.  The only way to circumvent this strong checking is
   using memory copies or complicated casts.  The compiler will catch
   simple casts. */

/****************
 * network to host
 ****************/

/* Macro to define gm_ntoh_u8(), etc. */

#define _GM_NTOH(type, size)						\
									\
/* swap using weak typing */						\
									\
gm_inline static							\
gm_##type##size##_t							\
__gm_ntoh_##type##size (gm_##type##size##_t x)				\
{									\
  if (GM_NET_SWAP)							\
    {									\
      return ((gm_##type##size##_t)					\
	      __gm_swap_u##size ((gm_u##size##_t) x));			\
    }									\
  else									\
    {									\
      return x;								\
    }									\
}									\
									\
/* swap using strong typing */						\
									\
gm_inline static							\
gm_##type##size##_t							\
_gm_ntoh_##type##size (gm_##type##size##_n_t x)				\
{									\
  return __gm_ntoh_##type##size (GM_N (x));				\
}

_GM_NTOH (u, 8);
_GM_NTOH (u, 16);
_GM_NTOH (u, 32);
_GM_NTOH (u, 64);

_GM_NTOH (s, 8);
_GM_NTOH (s, 16);
_GM_NTOH (s, 32);
_GM_NTOH (s, 64);

/****************
 * host to network
 ****************/

/* Macro to define gm_hton_u8(), etc. */

#define _GM_HTON(type, size)						\
									\
/* swap using weak typing */						\
									\
gm_inline static							\
gm_##type##size##_t							\
__gm_hton_##type##size (gm_##type##size##_t x)				\
{									\
  if (GM_NET_SWAP)							\
    {									\
      return ((gm_##type##size##_t)					\
	      __gm_swap_u##size ((gm_u##size##_t) x));			\
    }									\
  else									\
    {									\
      return x;								\
    }									\
}									\
									\
/* swap using strong typing */						\
									\
gm_inline static							\
gm_##type##size##_n_t							\
_gm_hton_##type##size (gm_##type##size##_t x)				\
{									\
  gm_##type##size##_n_t ret;						\
									\
  GM_N (ret) = __gm_hton_##type##size (x);				\
  return ret;								\
}

_GM_HTON (u, 8);
_GM_HTON (u, 16);
_GM_HTON (u, 32);
_GM_HTON (u, 64);

_GM_HTON (s, 8);
_GM_HTON (s, 16);
_GM_HTON (s, 32);
_GM_HTON (s, 64);

/********************************
 * host/network conversion macros.
 ********************************/

/* These are just like the "^_gm" versions above, but exist to give
   etags something to jump to.  The "_gm" versions above are in a macro
   to avoid code replication. */

gm_inline static gm_u64_t
gm_ntoh_u64 (gm_u64_n_t x)
{
  return _gm_ntoh_u64 (x);
}
gm_inline static gm_s64_t
gm_ntoh_s64 (gm_s64_n_t x)
{
  return _gm_ntoh_s64 (x);
}
gm_inline static gm_u32_t
gm_ntoh_u32 (gm_u32_n_t x)
{
  return _gm_ntoh_u32 (x);
}
gm_inline static gm_s32_t
gm_ntoh_s32 (gm_s32_n_t x)
{
  return _gm_ntoh_s32 (x);
}
gm_inline static gm_u16_t
gm_ntoh_u16 (gm_u16_n_t x)
{
  return _gm_ntoh_u16 (x);
}
gm_inline static gm_s16_t
gm_ntoh_s16 (gm_s16_n_t x)
{
  return _gm_ntoh_s16 (x);
}
gm_inline static gm_u8_t
gm_ntoh_u8 (gm_u8_n_t x)
{
  return _gm_ntoh_u8 (x);
}
gm_inline static gm_s8_t
gm_ntoh_s8 (gm_s8_n_t x)
{
  return _gm_ntoh_s8 (x);
}

gm_inline static gm_u64_n_t
gm_hton_u64 (gm_u64_t s)
{
  return _gm_hton_u64 (s);
}
gm_inline static gm_s64_n_t
gm_hton_s64 (gm_s64_t s)
{
  return _gm_hton_s64 (s);
}
gm_inline static gm_u32_n_t
gm_hton_u32 (gm_u32_t s)
{
  return _gm_hton_u32 (s);
}
gm_inline static gm_s32_n_t
gm_hton_s32 (gm_s32_t s)
{
  return _gm_hton_s32 (s);
}
gm_inline static gm_u16_n_t
gm_hton_u16 (gm_u16_t s)
{
  return _gm_hton_u16 (s);
}
gm_inline static gm_s16_n_t
gm_hton_s16 (gm_s16_t s)
{
  return _gm_hton_s16 (s);
}
gm_inline static gm_u8_n_t
gm_hton_u8 (gm_u8_t c)
{
  return _gm_hton_u8 (c);
}
gm_inline static gm_s8_n_t
gm_hton_s8 (gm_s8_t c)
{
  return _gm_hton_s8 (c);
}

/* Macro for accessing words in the little endian PCI config space */

gm_inline static gm_u32_t
gm_htopci_u32 (gm_u32_t x)
{
  if (GM_NET_SWAP)
    {
      return x;
    }
  else
    {
      return __gm_swap_u32 (x);
    }
}

#if !GM_MCP
gm_inline static void *
gm_ntohp (gm_up_n_t x)
{
#if GM_SIZEOF_VOID_P == 8
  return (void *) __gm_ntoh_u64 (GM_N (x));
#elif GM_SIZEOF_VOID_P == 4
  return (void *) __gm_ntoh_u32 (GM_N (x));
#endif
}
#endif

/****************
 * backwards compatibility
 ****************/

/** byte order macros: gm_ntoh* and gm_hton*, where
 ** "c" a 8-bit int (char)
 ** "s" a 16-bit int (short)
 ** "l" a 32-bit int (long)
 ** "ll" a 64-bit int (long long)
 ** "_dp" a 32-bit DMA pointer.
 ** "_lp" a 32-bit LANai pointer.
 ** "_up" a lanai-compatible representation of a user virtual memory addr. */

#if !GM_BUILDING_GM

/* These aliases are for client use only */

gm_inline static gm_u8_t
gm_ntohc (gm_u8_n_t x)
{
  return gm_ntoh_u8 (x);
}
gm_inline static gm_u16_t
gm_ntohs (gm_u16_n_t x)
{
  return gm_ntoh_u16 (x);
}
gm_inline static gm_u32_t
gm_ntohl (gm_u32_n_t x)
{
  return gm_ntoh_u32 (x);
}
gm_inline static gm_u64_t
gm_ntohll (gm_u64_n_t x)
{
  return gm_ntoh_u64 (x);
}

gm_inline static gm_u8_n_t
gm_htonc (gm_u8_t x)
{
  return gm_hton_u8 (x);
}
gm_inline static gm_u16_n_t
gm_htons (gm_u16_t x)
{
  return gm_hton_u16 (x);
}
gm_inline static gm_u32_n_t
gm_htonl (gm_u32_t x)
{
  return gm_hton_u32 (x);
}
gm_inline static gm_u64_n_t
gm_htonll (gm_u64_t x)
{
  return gm_hton_u64 (x);
}

#define gm_assert_p(a) gm_assert (a)

#endif /* !GM_BUILDING_GM */

/************
 * GM_ENTRY_POINT definition
 ************/

#ifndef GM_ENTRY_POINT		/* need to override in NT4 make-os.in */

#if defined _MSC_VER		/* microsoft compiler */
#  if GM_KERNEL
#    define GM_ENTRY_POINT
#  elif GM_BUILDING_GM_LIB
#    define GM_ENTRY_POINT __declspec (dllexport)
#  else
#    define GM_ENTRY_POINT __declspec (dllimport)
#  endif
#else
#  define GM_ENTRY_POINT
#endif

#endif /* ndef GM_ENTRY_POINT */

#if GM_DEBUG_BUFFERS
#  define gm_open    gm_open_debug_buffers
#  define gm_receive gm_receive_debug_buffers
#endif

/* Debugging */

#ifndef GM_DEBUG
#  ifndef GM_BUILDING_GM
#    define GM_DEBUG 0
#  else  /* GM_BUILDING_GM */
#    error GM_DEBUG must be defined for internal GM builds.
#  endif /* GM_BUILDING_GM */
#endif /* !GM_DEBUG */

/* A macro like "gm_assert()", only causing an error at compile-time
   rather than run time.  The expression A must be a constant
   expression.  */

#ifdef __GNUC__
#define __GM_COMPILE_TIME_ASSERT(a) do {				\
  char (*__GM_COMPILE_TIME_ASSERT)[(a) ? 1 : -1] = 0;			\
  (void) __GM_COMPILE_TIME_ASSERT; /* prevent unused var warning */	\
} while (0)
#else
#define __GM_COMPILE_TIME_ASSERT(a)
#endif

/****************
 * gm_always_assert() and gm_assert()
 ****************/

/* Macro for determining if X is a constant that can be evaluated at
   compile time.  This macro is used by gm_always_assert*() to perform as many
   compile-time checks as possible. */

#ifdef __GNUC__
#define __gm_builtin_constant_p(x) __builtin_constant_p (x)
#else
#define __gm_builtin_constant_p(x) 0
#endif

/* gm_assert() Assert iff debugging turned on. */

#if !GM_MCP
/* Don't use this in the MCP, because it requires too much memory
   to compile. */
#define gm_assert(a) do {if (GM_DEBUG) gm_always_assert (a);} while (0)
#elif GM_MCP && GM_DEBUG
#define gm_assert(a) gm_always_assert(a)
#elif GM_MCP && !GM_DEBUG
#define gm_assert(a)
#else
#error
#endif

#if GM_MCP
#define __GM_ASSERTION_FAILED(txt, line, file) do {			\
  /*LOG_DISPATCH(0,"__GM_ASSERTION_FAILED");*/				\
  _gm_assertion_failed (0, line, file);					\
} while (0)
#else
#define __GM_ASSERTION_FAILED(txt, line, file) do {			\
  _gm_assertion_failed (txt, line, file);				\
} while (0)
#endif

/* gm_always_assert(): Always assert, even if debugging is turned off,
   unless we are able to to the check at compile-time. */

#define gm_always_assert(a) do {					\
  __GM_COMPILE_TIME_ASSERT (!__gm_builtin_constant_p (a) || (a));	\
  if (!__gm_builtin_constant_p (a) && !(a))				\
    {									\
      __GM_ASSERTION_FAILED (#a, __LINE__, __FILE__);			\
    }									\
} while (0)

#define GM_ABORT() __GM_ASSERTION_FAILED ("0", __LINE__, __FILE__)

/****************
 * Compiler warning supression macros
 ****************/

#define GM_PARAMETER_MAY_BE_UNUSED(p) ((void)(p))
#define GM_VAR_MAY_BE_UNUSED(v) ((void)(v))

/****************************************************************
 * typedefs
 ****************************************************************/

typedef enum gm_status
{
  GM_SUCCESS = 0,
  GM_FAILURE = 1,
  GM_INPUT_BUFFER_TOO_SMALL = 2,
  GM_OUTPUT_BUFFER_TOO_SMALL = 3,
  GM_TRY_AGAIN = 4,
  GM_BUSY = 5,
  GM_MEMORY_FAULT = 6,
  GM_INTERRUPTED = 7,
  GM_INVALID_PARAMETER = 8,
  GM_OUT_OF_MEMORY = 9,
  GM_INVALID_COMMAND = 10,
  GM_PERMISSION_DENIED = 11,
  GM_INTERNAL_ERROR = 12,
  GM_UNATTACHED = 13,
  GM_UNSUPPORTED_DEVICE = 14,
  GM_SEND_TIMED_OUT = 15,
  GM_SEND_REJECTED = 16,
  GM_SEND_TARGET_PORT_CLOSED = 17,
  GM_SEND_TARGET_NODE_UNREACHABLE = 18,
  GM_SEND_DROPPED = 19,
  GM_SEND_PORT_CLOSED = 20,
  GM_NODE_ID_NOT_YET_SET = 21,
  GM_STILL_SHUTTING_DOWN = 22,
  GM_CLONE_BUSY = 23,
  GM_NO_SUCH_DEVICE = 24,
  /* If you add error codes, also update libgm/gm_strerror.c. --Glenn */

  GM_NUM_STATUS_CODES		/* may change in value */
#if !GM_BUILDING_GM
    /* DEPRECATED */ , GM_NUM_ERROR_CODES = GM_NUM_STATUS_CODES
#endif
    /* Do not add new codes here. */
}
gm_status_t;

enum gm_priority
{
  GM_LOW_PRIORITY = 0,
  GM_HIGH_PRIORITY = 1,
  GM_NUM_PRIORITIES = 2
};

enum gm_buf_status
{
  gm_in_send = 0,
  gm_in_recv,
  gm_in_app,
  gm_invalid_status
};

struct gm_buf_handle
{
  void *addr;
  int size;
  enum gm_buf_status status;
  struct gm_buf_handle *next;
};


typedef struct gm_alarm
{
  struct gm_alarm *next;
  enum
  {
    GM_ALARM_FREE = 0,
    GM_ALARM_SET
  }
  state;
  struct gm_port *port;
  void (*callback) (void *context);
  void *context;
  gm_u64_t deadline;
}
gm_alarm_t;

enum gm_api_version
{
  GM_API_VERSION_1_0 = 0x100,
  GM_API_VERSION_1_1 = 0x101,
  GM_API_VERSION_1_2 = 0x102,
  GM_API_VERSION_1_3 = 0x103
};

/************************************************************************
 * Macros
 ************************************************************************/

/* Determine if N is aligned on an M-byte boundary, where M is a power
   of 2 */

#define GM_ALIGNED(n,m)	(!((long)(n)&((m)-1)))

/* NOTE: all GM alignment requirements will be lifted */

#define GM_RDMA_GRANULARITY 64
#define GM_MAX_DMA_GRANULARITY 8
#if GM_RDMA_GRANULARITY % GM_MAX_DMA_GRANULARITY
#  error GM_RDMA_GRANULARITY must be a multiple of GM_MAX_DMA_GRANULARITY
#endif

#if !GM_MCP
#define GM_DMA_GRANULARITY GM_MAX_DMA_GRANULARITY
#endif

#define GM_DMA_ALIGNED(p)	GM_ALIGNED (p, GM_DMA_GRANULARITY)

/************************************************************************
 * Recv event type
 ************************************************************************/

enum gm_recv_event_type
{
#if !GM_MCP			/* don't use these in the MCP */
  GM_NO_RECV_EVENT = 0,
  GM_SENDS_FAILED_EVENT = 1,	/* deprecated */
  GM_ALARM_EVENT = 2,
  GM_SENT_EVENT = 3,
  _GM_SLEEP_EVENT = 4,
  GM_RAW_RECV_EVENT = 5,
  GM_BAD_SEND_DETECTED_EVENT = 6,
  GM_SEND_TOKEN_VIOLATION_EVENT = 7,
  GM_RECV_TOKEN_VIOLATION_EVENT = 8,
  GM_BAD_RECV_TOKEN_EVENT = 9,
  GM_ALARM_VIOLATION_EVENT = 10,
  /* normal receives */
  GM_RECV_EVENT = 11,
  GM_HIGH_RECV_EVENT = 12,
  GM_PEER_RECV_EVENT = 13,
  GM_HIGH_PEER_RECV_EVENT = 14,
  /* streamlined small message receives */
  GM_FAST_RECV_EVENT = 15,
  GM_FAST_HIGH_RECV_EVENT = 16,
  GM_FAST_PEER_RECV_EVENT = 17,
  GM_FAST_HIGH_PEER_RECV_EVENT = 18,
  /*  */
  GM_REJECTED_SEND_EVENT = 19,
  GM_ORPHANED_SEND_EVENT = 20,
  /* Types used to make a new event in
     the queue. */
  /* Directed send notification */
  _GM_DIRECTED_SEND_NOTIFICATION_EVENT = 21,
  /* GM_FREE_SEND_TOKEN_EVENT = 22, */
  /* GM_FREE_HIGH_SEND_TOKEN_EVENT = 23, */
  GM_BAD_RESEND_DETECTED_EVENT = 24,
  GM_DROPPED_SEND_EVENT = 25,
  GM_BAD_SEND_VMA_EVENT = 26,
  GM_BAD_RECV_VMA_EVENT = 27,
  _GM_FLUSHED_ALARM_EVENT = 28,
  GM_SENT_TOKENS_EVENT = 29,
  GM_IGNORE_RECV_EVENT = 30,
#endif				/* GM_MCP not defined */
  /****** Types used to make a new event in the queue. ******/
  GM_NEW_NO_RECV_EVENT = 128,
  GM_NEW_SENDS_FAILED_EVENT = 129,	/* deprecated */
  GM_NEW_ALARM_EVENT = 130,
  GM_NEW_SENT_EVENT = 131,
  _GM_NEW_SLEEP_EVENT = 132,
  GM_NEW_RAW_RECV_EVENT = 133,
  GM_NEW_BAD_SEND_DETECTED_EVENT = 134,
  GM_NEW_SEND_TOKEN_VIOLATION_EVENT = 135,
  GM_NEW_RECV_TOKEN_VIOLATION_EVENT = 136,
  GM_NEW_BAD_RECV_TOKEN_EVENT = 137,
  GM_NEW_ALARM_VIOLATION_EVENT = 138,
  /* normal receives */
  GM_NEW_RECV_EVENT = 139,
  GM_NEW_HIGH_RECV_EVENT = 140,
  GM_NEW_PEER_RECV_EVENT = 141,
  GM_NEW_HIGH_PEER_RECV_EVENT = 142,
  /* streamlined small message receives */
  GM_NEW_FAST_RECV_EVENT = 143,
  GM_NEW_FAST_HIGH_RECV_EVENT = 144,
  GM_NEW_FAST_PEER_RECV_EVENT = 145,
  GM_NEW_FAST_HIGH_PEER_RECV_EVENT = 146,
  /* */
  GM_NEW_REJECTED_SEND_EVENT = 147,
  GM_NEW_ORPHANED_SEND_EVENT = 148,
  /* Directed send notification */
  _GM_NEW_DIRECTED_SEND_NOTIFICATION_EVENT = 149,
  GM_NEW_FREE_SEND_TOKEN_EVENT = 150,
  GM_NEW_FREE_HIGH_SEND_TOKEN_EVENT = 151,
  GM_NEW_BAD_RESEND_DETECTED_EVENT = 152,
  GM_NEW_DROPPED_SEND_EVENT = 153,
  GM_NEW_BAD_SEND_VMA_EVENT = 154,
  GM_NEW_BAD_RECV_VMA_EVENT = 155,
  _GM_NEW_FLUSHED_ALARM_EVENT = 156,
  GM_NEW_SENT_TOKENS_EVENT = 157,
  GM_NEW_IGNORE_RECV_EVENT = 158,
  /* Add new types here. */

  GM_NUM_RECV_EVENT_TYPES
    /* DO NOT add new types here. */
};

/****************
 * recv
 ****************/

typedef struct gm_recv
{
  /* Pad to GM_RDMA_GRANULARITY bytes */
#if (32 % GM_RDMA_GRANULARITY)
  gm_u8_n_t _rdma_padding[GM_RDMA_GRANULARITY - (32 % GM_RDMA_GRANULARITY)];
#endif
  /* Pad to 32 bytes */
#if (16 + 2 * GM_SIZEOF_UP_T) % 32
  gm_u8_n_t _reserved[32 - (16 + 2 * GM_SIZEOF_UP_T) % 32];
#endif
  gm_up_n_t message;
  gm_up_n_t buffer;
  /* 8 */
  gm_u16_n_t ip_checksum;	/* only used for GALVANTECH_WORKAROUND */
  gm_u16_n_t reserved_after_ip_checksum;
  gm_u32_n_t length;
  /* 8 */
  gm_u16_n_t sender_node_id;
  gm_u16_n_t reserved_after_sender_node_id;
  gm_u8_n_t tag;
  gm_u8_n_t size;
  gm_u8_n_t sender_port_id;
  gm_u8_n_t type;
}
gm_recv_t;

/****************
 * sent token report
 ****************/

struct _gm_sent_token_report
{
  gm_u8_n_t token;
  gm_u8_n_t status;
};

#define GM_MAX_SENT_TOKEN_REPORTS_PER_EVENT ((GM_RDMA_GRANULARITY-2) / 2)

/****************
 * tokens sent
 ****************/

typedef struct gm_tokens_sent
{
  struct _gm_sent_token_report report[31];
  gm_u8_n_t reserved_before_type;
  gm_u8_n_t type;
}
gm_tokens_sent_t;

/****************
 * sent
 ****************/

#define GM_TOKENS_SENT_EVENT__FIRST (ts) ((gm_u8_t *) (ts + 1) - 2)

typedef struct gm_sent
{
  /* Pad to GM_RDMA_GRANULARITY bytes */
#if (32 % GM_RDMA_GRANULARITY)
  gm_u8_n_t _rdma_padding[GM_RDMA_GRANULARITY - (32 % GM_RDMA_GRANULARITY)];
#endif
  /* Pad to 32 bytes */
#if (2 * GM_SIZEOF_UP_T) % 32
  gm_u8_n_t _reserved[32 - (2 * GM_SIZEOF_UP_T) % 32];
#endif
  gm_up_n_up_n_t message_list;
  gm_u8_n_t _reserved_after_message_list[GM_SIZEOF_UP_T - 1];
  gm_u8_n_t type;
}
gm_sent_t;

/****************
 * failed send
 ****************/

				/* Struct of GM_RDMA_GRANULARITY for
				   reporting rejected sends. */
typedef struct gm_failed_send_event
{
  /* Pad to GM_RDMA_GRANULARITY bytes */
#if (32 % GM_RDMA_GRANULARITY)
  gm_u8_n_t _rdma_padding[GM_RDMA_GRANULARITY - (32 % GM_RDMA_GRANULARITY)];
#endif
  /* Pad to 32 bytes */
#if (8 + GM_SIZEOF_UP_T) % 32
  gm_u8_n_t _reserved[32 - (8 + GM_SIZEOF_UP_T) % 32];
#endif
  gm_up_n_t message;
  /* 8 */
  gm_u32_n_t reserved_after_message;
  gm_u16_n_t target_node_id;
  gm_u8_n_t target_subport_id;
  gm_u8_n_t type;
}
gm_failed_send_event_t;

/****************
 * directed send notification
 ****************/

typedef struct gm_directed_send_notification
{
  /* Pad to GM_RDMA_GRANULARITY bytes */
#if (32 % GM_RDMA_GRANULARITY)
  gm_u8_n_t _rdma_padding[GM_RDMA_GRANULARITY - (32 % GM_RDMA_GRANULARITY)];
#endif
  /* Pad to 32 bytes */
#if (8 + 2 * GM_MAX_DMA_GRANULARITY + GM_SIZEOF_UP_T) % 32
  gm_u8_n_t _reserved[32 - ((8 + 2 * GM_MAX_DMA_GRANULARITY
			     + GM_SIZEOF_UP_T) % 32)];
#endif
  gm_up_n_t buffer;
  /* 8 */
#if GM_MAX_DMA_GRANULARITY != 8
#error
#endif
  gm_u8_n_t sender_port_id;
  gm_u8_n_t start_copy_bytes[GM_MAX_DMA_GRANULARITY - 1];
  /* 8 */
  gm_u8_n_t end_copy_bytes[GM_MAX_DMA_GRANULARITY];
  /* 8 */
  gm_u32_n_t length;
  gm_u16_n_t sender_node_id;
  gm_u8_n_t size;
  gm_u8_n_t type;
}
gm_directed_send_notification_t;


/****************
 * flushed alarm
 ****************/

typedef struct gm_flushed_alarm
{
  /* Pad to GM_RDMA_GRANULARITY bytes */
#if (32 % GM_RDMA_GRANULARITY)
  gm_u8_n_t _rdma_padding[GM_RDMA_GRANULARITY - (32 % GM_RDMA_GRANULARITY)];
#endif
  /* Pad to 32 bytes */
  gm_u8_n_t _reserved[32 - 8];
  /* 8 */
  gm_u32_n_t usecs_remaining;
  gm_u8_n_t reserved_after_usecs_remaining[3];
  gm_u8_n_t type;
}
gm_flushed_alarm_t;

/****************
 * recv event
 ****************/

typedef union gm_recv_event
{
  /* Each of these must be the same size so the TYPE field at the
     end of each is aligned. */
  gm_tokens_sent_t tokens_sent;
  gm_recv_t recv;
  gm_sent_t sent;
  gm_failed_send_event_t failed_send;
  gm_directed_send_notification_t directed_send_notification;
  gm_flushed_alarm_t flushed_alarm;
}
gm_recv_event_t;

#define GM_RECV_EVENT_TYPE(e) (gm_ntoh_u8 ((e)->recv.type))

/* GM equivalent of ANSI "offsetof()" macro, since WindowsNT botches
   (or used to botch) it somewhere in the DDK header files. */

#define GM_OFFSETOF(type,field) ((gm_offset_t) &((type *)0)->field)


struct gm_port;
typedef void (*gm_send_completion_callback_t) (struct gm_port * p,
					       void *context,
					       gm_status_t status);

/************************************************************************
 * Globals (undefined until gm_init() or gm_open() called).
 ************************************************************************/

#if !GM_PAGE_LEN
extern GM_ENTRY_POINT unsigned long GM_PAGE_LEN;
#endif

/****************************************************************
 * Data type abstraction typedefs
 ****************************************************************/

#if defined _M_IA64
typedef gm_u64_t gm_size_t;
typedef gm_s64_t gm_offset_t;
#else
typedef unsigned long gm_size_t;
typedef signed long gm_offset_t;
#endif

/****************************************************************
 * Function prototypes
 ****************************************************************/

GM_ENTRY_POINT void _gm_assertion_failed (const char *, int, const char *);

GM_ENTRY_POINT void gm_abort (void);
GM_ENTRY_POINT int gm_alloc_send_token (struct gm_port *p,
					unsigned int priority);
GM_ENTRY_POINT gm_status_t gm_allow_remote_memory_access (struct gm_port *p);
GM_ENTRY_POINT void gm_bcopy (void *from, void *to, gm_size_t len);
GM_ENTRY_POINT union gm_recv_event *gm_blocking_receive (struct gm_port *p);
GM_ENTRY_POINT union
  gm_recv_event *gm_blocking_receive_no_spin (struct gm_port *p);
GM_ENTRY_POINT void gm_bzero (void *ptr, gm_size_t len);
GM_ENTRY_POINT void *gm_calloc (gm_size_t len, gm_size_t cnt);
struct gm_alarm;
GM_ENTRY_POINT void gm_cancel_alarm (struct gm_alarm *gm_alarm);
GM_ENTRY_POINT void gm_close (struct gm_port *p);
GM_ENTRY_POINT void gm_datagram_send (struct gm_port *p, void *message,
				      unsigned int size, gm_size_t len,
				      unsigned int priority,
				      unsigned int target_node_id,
				      unsigned int target_port_id,
				      gm_send_completion_callback_t callback,
				      void *context);
GM_ENTRY_POINT void gm_datagram_send_4 (struct gm_port *p, gm_u32_t message,
					unsigned int size, gm_size_t len,
					unsigned int priority,
					unsigned int target_node_id,
					unsigned int target_port_id,
					gm_send_completion_callback_t
					callback, void *context);
GM_ENTRY_POINT gm_status_t gm_deregister_memory (struct gm_port *p, void *ptr,
						 gm_size_t length);
GM_ENTRY_POINT void gm_directed_send (struct gm_port *p, void *source_buffer,
				      gm_remote_ptr_t target_buffer,
				      gm_size_t len,
				      enum gm_priority priority,
				      unsigned int target_node_id,
				      unsigned int target_port_id);
GM_ENTRY_POINT void gm_directed_send_with_callback (struct gm_port *p,
						    void *source_buffer,
						    gm_remote_ptr_t
						    target_buffer,
						    gm_size_t len,
						    enum gm_priority priority,
						    unsigned int
						    target_node_id,
						    unsigned int
						    target_port_id,
						    gm_send_completion_callback_t
						    callback, void *context);
GM_ENTRY_POINT void *gm_dma_calloc (struct gm_port *p, gm_size_t count,
				    gm_size_t length);
GM_ENTRY_POINT void gm_dma_free (struct gm_port *p, void *addr);
GM_ENTRY_POINT void *gm_dma_malloc (struct gm_port *p, gm_size_t length);
GM_ENTRY_POINT void gm_flush_alarm (struct gm_port *p);
GM_ENTRY_POINT void gm_free (void *ptr);
GM_ENTRY_POINT void gm_free_send_token (struct gm_port *p,
					unsigned int priority);
GM_ENTRY_POINT void gm_free_send_tokens (struct gm_port *p,
					 unsigned int priority,
					 unsigned int count);
GM_ENTRY_POINT gm_status_t gm_get_host_name (struct gm_port *port,
					     char name[GM_MAX_HOST_NAME_LEN]);
GM_ENTRY_POINT gm_status_t gm_get_node_type (struct gm_port *port,
					     int *node_type);
GM_ENTRY_POINT gm_status_t gm_get_node_id (struct gm_port *port,
					   unsigned int *n);
GM_ENTRY_POINT gm_status_t gm_get_unique_board_id (struct gm_port *port,
						   char unique[6]);
GM_ENTRY_POINT gm_status_t gm_get_mapper_unique_id (struct gm_port *port,
						    char unique[6]);
GM_ENTRY_POINT void gm_hex_dump (const void *ptr, gm_size_t len);
GM_ENTRY_POINT unsigned int gm_host_name_to_node_id (struct gm_port *port,
						     char *_host_name);
GM_ENTRY_POINT void gm_initialize_alarm (struct gm_alarm *my_alarm);
GM_ENTRY_POINT int gm_isprint (int c);
GM_ENTRY_POINT void *gm_malloc (gm_size_t len);
GM_ENTRY_POINT void *gm_page_alloc (void);
GM_ENTRY_POINT void gm_page_free (void *addr);
GM_ENTRY_POINT void *gm_alloc_pages (gm_size_t len);
GM_ENTRY_POINT void gm_free_pages (void *addr, gm_size_t len);
GM_ENTRY_POINT gm_size_t gm_max_length_for_size (unsigned int size);
GM_ENTRY_POINT gm_status_t gm_max_node_id (struct gm_port *port,
					   unsigned int *n);
GM_ENTRY_POINT gm_status_t gm_max_node_id_inuse (struct gm_port *port,
						 unsigned int *n);
GM_ENTRY_POINT int gm_memcmp (const void *a, const void *b, gm_size_t len);
GM_ENTRY_POINT void *gm_memorize_message (void *message, void *buffer,
					  unsigned int len);
GM_ENTRY_POINT unsigned int gm_min_message_size (struct gm_port *port);
GM_ENTRY_POINT unsigned int gm_min_size_for_length (gm_size_t length);
GM_ENTRY_POINT unsigned int gm_mtu (struct gm_port *port);
GM_ENTRY_POINT char *gm_node_id_to_host_name (struct gm_port *port,
					      unsigned int node_id);
GM_ENTRY_POINT gm_status_t gm_node_id_to_unique_id (struct gm_port *port,
						    unsigned int n,
						    char unique[6]);
GM_ENTRY_POINT unsigned int gm_num_ports (struct gm_port *p);
GM_ENTRY_POINT unsigned int gm_num_send_tokens (struct gm_port *p);
GM_ENTRY_POINT unsigned int gm_num_receive_tokens (struct gm_port *p);
GM_ENTRY_POINT gm_status_t gm_open (struct gm_port **p, unsigned int unit,
				    unsigned int port, const char *port_name,
				    enum gm_api_version version);
GM_ENTRY_POINT void gm_provide_receive_buffer (struct gm_port *p, void *ptr,
					       unsigned int size,
					       unsigned int priority);
GM_ENTRY_POINT void gm_provide_receive_buffer_with_tag (struct gm_port *p,
							void *ptr,
							unsigned int size,
							unsigned int priority,
							unsigned int tag);
GM_ENTRY_POINT int gm_receive_pending (struct gm_port *p);
GM_ENTRY_POINT int gm_next_event_peek (struct gm_port *p, gm_u16_t *sender);
GM_ENTRY_POINT union gm_recv_event *gm_receive (struct gm_port *p);
GM_ENTRY_POINT gm_status_t gm_register_memory (struct gm_port *p,
					       void *ptr, gm_size_t length);
GM_ENTRY_POINT int gm_send_token_available (struct gm_port *p,
					    unsigned int priority);
GM_ENTRY_POINT void gm_send (struct gm_port *p, void *message,
			     unsigned int size, gm_size_t len,
			     unsigned int priority,
			     unsigned int target_node_id,
			     unsigned int target_port_id);
GM_ENTRY_POINT
  void gm_send_with_callback (struct gm_port *p, void *message,
			      unsigned int size, gm_size_t len,
			      unsigned int priority,
			      unsigned int target_node_id,
			      unsigned int target_port_id,
			      gm_send_completion_callback_t callback,
			      void *context);
GM_ENTRY_POINT void gm_send_to_peer (struct gm_port *p, void *message,
				     unsigned int size, gm_size_t len,
				     unsigned int priority,
				     unsigned int target_node_id);
GM_ENTRY_POINT
  void gm_send_to_peer_with_callback (struct gm_port *p, void *message,
				      unsigned int size, gm_size_t len,
				      unsigned int priority,
				      unsigned int target_node_id,
				      gm_send_completion_callback_t callback,
				      void *context);
GM_ENTRY_POINT gm_status_t gm_set_acceptable_sizes (struct gm_port *p,
						    enum gm_priority
						    priority, gm_size_t mask);
GM_ENTRY_POINT void gm_set_alarm (struct gm_port *p,
				  struct gm_alarm *my_alarm,
				  gm_u64_t usecs,
				  void (*callback) (void *), void *context);
GM_ENTRY_POINT gm_size_t gm_strlen (const char *cptr);
GM_ENTRY_POINT char *gm_strncpy (char *to, const char *from, int len);
GM_ENTRY_POINT int gm_strcmp (const char *a, const char *b);
GM_ENTRY_POINT int gm_strncmp (const char *a, const char *b, int len);
GM_ENTRY_POINT int gm_strncasecmp (const char *a, const char *b, int len);
GM_ENTRY_POINT gm_u64_t gm_ticks (struct gm_port *port);
GM_ENTRY_POINT gm_status_t gm_unique_id (struct gm_port *port,
					 char unique[6]);
GM_ENTRY_POINT gm_status_t gm_unique_id_to_node_id (struct gm_port *port,
						    char unique[6],
						    unsigned int *node_id);
GM_ENTRY_POINT void gm_unknown (struct gm_port *p, union gm_recv_event *e);
GM_ENTRY_POINT gm_status_t _gm_get_route (struct gm_port *p,
					  unsigned int node_id, char *route,
					  unsigned int *len);

/****************
 * buffer debugging
 ****************/

GM_ENTRY_POINT void gm_dump_buffers (void);
GM_ENTRY_POINT void gm_register_buffer (void *addr, int size);
GM_ENTRY_POINT int gm_unregister_buffer (void *addr, int size);

/************
 * Lookaside table entry points
 ************/

GM_ENTRY_POINT struct gm_lookaside *gm_create_lookaside (gm_size_t entry_len,
							 gm_size_t
							 min_entry_cnt);
GM_ENTRY_POINT void gm_destroy_lookaside (struct gm_lookaside *l);
GM_ENTRY_POINT void *gm_lookaside_alloc (struct gm_lookaside *l);
GM_ENTRY_POINT void *gm_lookaside_zalloc (struct gm_lookaside *l);
GM_ENTRY_POINT void gm_lookaside_free (void *ptr);

/************
 * Hash table entry points
 ************/

#define GM_HASH_SMOOTH 1	/* not yet supported */

GM_ENTRY_POINT struct gm_hash
  *gm_create_hash (long (*gm_user_compare) (void *key1, void *key2),
		   unsigned long (*gm_user_hash) (void *key1),
		   gm_size_t key_len, gm_size_t data_len,
		   gm_size_t gm_min_entries, int flags);
GM_ENTRY_POINT void gm_destroy_hash (struct gm_hash *h);
GM_ENTRY_POINT void *gm_hash_remove (struct gm_hash *hash, void *key);
GM_ENTRY_POINT void *gm_hash_find (struct gm_hash *hash, void *key);
GM_ENTRY_POINT gm_status_t gm_hash_insert (struct gm_hash *hash, void *key,
					   void *datum);
GM_ENTRY_POINT void gm_hash_rekey (struct gm_hash *hash, void *old_key,
				   void *new_key);

GM_ENTRY_POINT long gm_hash_compare_strings (void *key1, void *key2);
GM_ENTRY_POINT unsigned long gm_hash_hash_string (void *key);
GM_ENTRY_POINT long gm_hash_compare_longs (void *key1, void *key2);
GM_ENTRY_POINT unsigned long gm_hash_hash_long (void *key);
GM_ENTRY_POINT long gm_hash_compare_ints (void *key1, void *key2);
GM_ENTRY_POINT unsigned long gm_hash_hash_int (void *key);
GM_ENTRY_POINT long gm_hash_compare_ptrs (void *key1, void *key2);
GM_ENTRY_POINT unsigned long gm_hash_hash_ptr (void *key);

/************
 * crc
 ************/

GM_ENTRY_POINT unsigned long gm_crc (void *ptr, gm_size_t len);
GM_ENTRY_POINT unsigned long gm_crc_str (const char *ptr);

/************
 * random number generation
 ************/

GM_ENTRY_POINT int gm_rand (void);
GM_ENTRY_POINT void gm_srand (int seed);
GM_ENTRY_POINT unsigned int gm_rand_mod (unsigned int modulus);

/************
 * init/finalize
 ************/

GM_ENTRY_POINT gm_status_t gm_init (void);
GM_ENTRY_POINT void gm_finalize (void);

/************
 * base 2 logarithm computation
 ************/

GM_ENTRY_POINT extern const unsigned char gm_log2_roundup_table[257];
GM_ENTRY_POINT unsigned long gm_log2_roundup (unsigned long n);

/************
 * GM mutex's (currently open-coded for user-mode)
 ************/

GM_ENTRY_POINT struct gm_mutex *gm_create_mutex (void);
GM_ENTRY_POINT void gm_destroy_mutex (struct gm_mutex *mu);
GM_ENTRY_POINT void gm_mutex_enter (struct gm_mutex *mu);
GM_ENTRY_POINT void gm_mutex_exit (struct gm_mutex *mu);

/************
 * GM zone
 ************/

GM_ENTRY_POINT struct gm_zone *gm_zone_create_zone (void *base,
						    gm_size_t len);
GM_ENTRY_POINT void gm_zone_destroy_zone (struct gm_zone *zone);
GM_ENTRY_POINT void *gm_zone_free (struct gm_zone *zone, void *a);
GM_ENTRY_POINT void *gm_zone_malloc (struct gm_zone *zone, gm_size_t length);
GM_ENTRY_POINT void *gm_zone_calloc (struct gm_zone *zone, gm_size_t count,
				     gm_size_t length);
GM_ENTRY_POINT int gm_zone_addr_in_zone (struct gm_zone *zone, void *p);

/****************
 * Default event handlers.
 *
 * gm_unknown() will call these for you, but you can call them directly
 * to enhance performance.
 ****************/

GM_ENTRY_POINT void
gm_handle_directed_send_notification (struct gm_port *p, gm_recv_event_t * e);
GM_ENTRY_POINT void gm_handle_sent_tokens (struct gm_port *p,
					   gm_recv_event_t * e);

/****************
 * fault tolerance
 ****************/

GM_ENTRY_POINT void gm_resume_sending (struct gm_port *p,
				       unsigned int priority,
				       unsigned int target_node_id,
				       unsigned int target_port_id,
				       gm_send_completion_callback_t callback,
				       void *context);

GM_ENTRY_POINT void gm_drop_sends (struct gm_port *port,
				   unsigned int priority,
				   unsigned int target_node_id,
				   unsigned int target_port_id,
				   gm_send_completion_callback_t callback,
				   void *context);

/****************
 * PID support
 ****************/

typedef gm_u32_t gm_pid_t;
GM_ENTRY_POINT gm_pid_t gm_getpid (void);

/****************
 * Direct copy support
 ****************/

GM_ENTRY_POINT gm_status_t gm_directcopy_get (struct gm_port *p,
					      void *source_addr,
					      void *target_addr,
					      gm_size_t length,
					      unsigned int source_instance_id,
					      unsigned int source_port_id);

/****************
 * Misc
 ****************/

GM_ENTRY_POINT void gm_perror (const char *message, gm_status_t error);
GM_ENTRY_POINT int gm_sleep (unsigned seconds);
GM_ENTRY_POINT void gm_exit (gm_status_t status);
GM_ENTRY_POINT int
gm_printf (const char *format, ...)
#if __GNUC__
  __attribute__ ((format (printf, 1, 2)))
#endif
  ;

/****************************************************************
 * GM-1.4 additions
 ****************************************************************/

GM_ENTRY_POINT char *gm_strerror (gm_status_t error);
GM_ENTRY_POINT gm_status_t gm_set_enable_nack_down(struct gm_port *port,
						   int flag);

/****************
 * GM mark support
 ****************/

typedef gm_size_t gm_mark_t;
struct gm_mark_set;

GM_ENTRY_POINT gm_status_t gm_mark (struct gm_mark_set *set, gm_mark_t *m);
GM_ENTRY_POINT int gm_mark_is_valid (struct gm_mark_set *set, gm_mark_t *m);
GM_ENTRY_POINT gm_status_t gm_create_mark_set (struct gm_mark_set **set,
					       unsigned long init_count);
GM_ENTRY_POINT void gm_destroy_mark_set (struct gm_mark_set *set);
GM_ENTRY_POINT void gm_unmark (struct gm_mark_set *set, gm_mark_t *m);
GM_ENTRY_POINT void gm_unmark_all (struct gm_mark_set *set);

/****************
 * gm_on_exit
 ****************/

typedef void (*gm_on_exit_callback_t)(gm_status_t status, void *arg);
GM_ENTRY_POINT gm_status_t gm_on_exit (gm_on_exit_callback_t, void *arg);

/************************************************************************
 * Epilogue
 ************************************************************************/

#ifdef __cplusplus
#if 0
{				/* indent hack */
#endif
}
#endif

#endif /* ifndef _gm_h_ */

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  End:
*/
