/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

#include "am.h"

am_mesg_t *_am_cur_buf;
int _am_cur_buf_used;

static am_chainable_mesg_t *next_to_handle = 0;
static am_chainable_mesg_t *last_to_handle = 0;

static char *msg_type_names[] =
{
	"NO_RECV_TYPE",
	"REQ0_TYPE",
	"REQ1_TYPE",
	"REQ2_TYPE",
	"REQ3_TYPE",
	"REQ4_TYPE",
	"REQDF_TYPE",
	"STORE_TYPE",
	"STORE_DONE_TYPE",
	"STORE_ASYNC_TYPE",
	"GET_TYPE",
	"REP0_TYPE",
	"REP1_TYPE",
	"REP2_TYPE",
	"REP3_TYPE",
	"REP4_TYPE",
	"REPDF_TYPE",
	"GET_REP_TYPE",
};

/*
 * call handlers - _am_cur_buf must 
 * point to an available buffer
 */
static inline
void
handle_message(am_mesg_t * p,
			   int sender)
{
	int reply = 0;

	if (p->i4.type >= REP0_TYPE)
		reply = 1;

	_am_cur_buf_used = 0;

	if (p->i4.handler)
		switch (p->i4.type) {

		 case REQ0_TYPE:
		 case REP0_TYPE:
			 (p->i4.handler) (am_from_gm_node_map(sender));
			 break;

		 case REQ1_TYPE:
		 case REP1_TYPE:
			 (p->i4.handler) (am_from_gm_node_map(sender),
							  p->i4.arg1);
			 break;

		 case REQ2_TYPE:
		 case REP2_TYPE:
			 (p->i4.handler) (am_from_gm_node_map(sender),
							  p->i4.arg1, p->i4.arg2);
			 break;

		 case REQ3_TYPE:
		 case REP3_TYPE:
			 (p->i4.handler) (am_from_gm_node_map(sender),
							  p->i4.arg1, p->i4.arg2,
							  p->i4.arg3);
			 break;

		 case REQ4_TYPE:
		 case REP4_TYPE:
			 (p->i4.handler) (am_from_gm_node_map(sender),
							  p->i4.arg1, p->i4.arg2,
							  p->i4.arg3, p->i4.arg4);
			 break;

		 case REQDF_TYPE:
		 case REPDF_TYPE:
			 (p->df.handler) (am_from_gm_node_map(sender),
							  p->df.arg1, p->df.arg2,
							  p->df.arg3);
			 break;

		 case STORE_TYPE:
		 case STORE_ASYNC_TYPE:
			 (p->store.handler) (am_from_gm_node_map(sender),
								 p->store.rva, p->store.nbytes,
								 p->store.handler_arg);
			 break;

		 case GET_TYPE:
			 if (p != _am_cur_buf)
				 gm_memorize_message(p, _am_cur_buf, STORE_TYPE_MESG_LEN);

			 _am_cur_buf_used = 1;
			 _am_cur_buf->store.type = GET_REP_TYPE;

			 _am_get_high_token();
			 gm_directed_send(port,
							  p->store.rva,
							  p->store.lva,
							  p->store.nbytes,
							  GM_HIGH_PRIORITY,
							  sender,
							  port_id);

			 _am_get_high_token();
			 gm_send_to_peer(port,
							 _am_cur_buf,
							 RR_MESG_SIZE,
							 STORE_TYPE_MESG_LEN,
							 GM_HIGH_PRIORITY,
							 sender);
			 break;

		 case GET_REP_TYPE:
			 (p->store.handler) (am_from_gm_node_map(sender),
								 p->store.lva, p->store.nbytes,
								 p->store.handler_arg);
			 break;

		 default:
			 GAM_PRINT(0, ("error: bad type in handle\n"));
			 gm_always_assert(0);
		}

	if (reply) {
		/*
		 * reply handlers are not allowed to send
		 */
		gm_assert(!_am_cur_buf_used);

		GAM_PRINT(3, ("providing a high priority recv buffer\n"));
		gm_provide_receive_buffer(port,
								  _am_cur_buf,
								  RR_MESG_SIZE,
								  GM_HIGH_PRIORITY);
	}
	else {
		if (!_am_cur_buf_used) {
			GAM_PRINT(3, ("providing a low priority recv buffer\n"));
			gm_provide_receive_buffer(port,
									  _am_cur_buf,
									  RR_MESG_SIZE,
									  GM_LOW_PRIORITY);
		}
	}
}

inline
void
handle_queued_messages(void)
{
	am_chainable_mesg_t *cp;

	while (next_to_handle) {
		cp = next_to_handle;
		next_to_handle = cp->next;
		_am_cur_buf = &cp->message;

		/*
		   printf ("handling a %s from the queue\n",
		   msg_type_names[cp->message.i4.type]);
		 */

		handle_message(&cp->message, cp->sender);
	}
	gm_assert(!next_to_handle);
}

static inline
void
am_free_sent_message(am_mesg_t * p)
{
	am_chainable_mesg_t *cp;

	cp = am_chainable_parent(p);

	cp->next = free_buffers[RR_MESG_SIZE];
	free_buffers[RR_MESG_SIZE] = cp;
}

void 
_am_poll(int handle)
{
	gm_recv_event_t *e;
	am_mesg_t *p, *b;
	int len;
	am_chainable_mesg_t *cp;

#if GM_DEBUG
	static int recurse = 0;
	/*
	 * _am_poll can be called recursively by a handler,
	 * in which case we must not call any more handlers
	 */
	recurse++;
	gm_always_assert(recurse == 1 ||
			  (recurse == 2 && !handle));
#endif

	if (handle)
		handle_queued_messages();

	/*
	 * drain the network
	 */
	while (1) {
		e = gm_receive(port);
		switch (e->recv.type) {

		 case GM_NO_RECV_EVENT:
#if GM_DEBUG
			 recurse--;
#endif
			 return;

		 case GM_FAST_PEER_RECV_EVENT:
		 case GM_PEER_RECV_EVENT:
		 case GM_FAST_HIGH_PEER_RECV_EVENT:
		 case GM_HIGH_PEER_RECV_EVENT:
			 b = (am_mesg_t *) gm_ntoh_hp(e->recv.buffer);
			 gm_assert(b);

			 if (e->recv.type == GM_FAST_PEER_RECV_EVENT ||
					 e->recv.type == GM_FAST_HIGH_PEER_RECV_EVENT) {
				 p = (am_mesg_t *) gm_ntoh_hp(e->recv.message);
				 gm_assert(p);
			 }
			 else {
				 p = b;
			 }

			 len = gm_ntohl(e->recv.length);

#if GM_DEBUG
			 switch (p->i4.type) {
			  case REQ0_TYPE:
			  case REP0_TYPE:
				  gm_always_assert(len == I0_MESG_LEN);
				  break;
			  case REQ1_TYPE:
			  case REP1_TYPE:
				  gm_always_assert(len == I1_MESG_LEN);
				  break;
			  case REQ2_TYPE:
			  case REP2_TYPE:
				  gm_always_assert(len == I2_MESG_LEN);
				  break;
			  case REQ3_TYPE:
			  case REP3_TYPE:
				  gm_always_assert(len == I3_MESG_LEN);
				  break;
			  case REQ4_TYPE:
			  case REP4_TYPE:
				  gm_always_assert(len == I4_MESG_LEN);
				  break;
			  case REQDF_TYPE:
			  case REPDF_TYPE:
				  gm_always_assert(len == DF_MESG_LEN);
				  break;
			  case STORE_TYPE:
			  case STORE_ASYNC_TYPE:
			  case GET_TYPE:
			  case GET_REP_TYPE:
				  gm_always_assert(len == STORE_TYPE_MESG_LEN);
				  break;
			  default:
				  GAM_PRINT(0, ("unknown type!\n"));
				  gm_always_assert(0);
			 }
#endif

			 if (handle) {
				 handle_queued_messages();

				 _am_cur_buf = b;

				 /*
				    printf ("handling a %s\n", msg_type_names[p->i4.type]);
				  */

				 handle_message(p, gm_ntohs(e->recv.sender_node_id));
			 }
			 else {
				 /*
				    printf ("queueing a %s\n", msg_type_names[p->i4.type]);
				  */

				 if (p != b)
					 gm_memorize_message(p, b, len);
				 cp = am_chainable_parent(b);
				 cp->sender = gm_ntohs(e->recv.sender_node_id);
				 cp->next = 0;

				 if (next_to_handle) {
					 last_to_handle
						 = last_to_handle->next
						 = cp;
				 }
				 else {
					 next_to_handle
						 = last_to_handle
						 = cp;
				 }
			 }
			 break;

		 case GM_SENT_EVENT:
			 {
				 void **pp;

				 pp = (void **) gm_ntohl((long int) e->sent.message_list);
				 do {
					 p = (am_mesg_t *) gm_ntohl((long) *pp);

					 GAM_PRINT(3, ("freeing a sent message %p\n", p));

					 switch (p->i4.type) {

					  case REQ0_TYPE:
					  case REQ1_TYPE:
					  case REQ2_TYPE:
					  case REQ3_TYPE:
					  case REQ4_TYPE:
					  case REQDF_TYPE:
					  case GET_TYPE:
						  gm_free_send_token(port, GM_LOW_PRIORITY);
						  am_free_sent_message(p);
						  break;

					  case STORE_TYPE:
						  p->i4.type = STORE_DONE_TYPE;
						  gm_free_send_tokens(port, GM_LOW_PRIORITY, 2);
						  am_free_sent_message(p);
						  break;

					  case STORE_ASYNC_TYPE:
						  (p->store.endfunc) (p->store.dest, p->store.lva,
											  p->store.nbytes,
											  p->store.endfunc_arg);
						  gm_free_send_tokens(port, GM_LOW_PRIORITY, 2);
						  am_free_sent_message(p);
						  break;

					  case GET_REP_TYPE:
						  gm_free_send_tokens(port, GM_HIGH_PRIORITY, 2);
						  gm_provide_receive_buffer(port,
													p,
													RR_MESG_SIZE,
													GM_LOW_PRIORITY);
						  break;

					  case REP0_TYPE:
					  case REP1_TYPE:
					  case REP2_TYPE:
					  case REP3_TYPE:
					  case REP4_TYPE:
					  case REPDF_TYPE:
						  gm_free_send_token(port, GM_HIGH_PRIORITY);
						  gm_provide_receive_buffer(port,
													p,
													RR_MESG_SIZE,
													GM_LOW_PRIORITY);
						  break;

					  default:
						  GAM_PRINT(0, ("bad message type in sent event!\n"));
						  gm_always_assert(0);
					 }
					 pp++;
				 }
				 while (/*gm_ntoh_hp*/(*pp));
			 }
			 break;

		 default:
			 GAM_PRINT(3, ("got an unknown EVENT %d\n", e->recv.type));
			 gm_assert(e->recv.type != _GM_SLEEP_EVENT);
			 gm_unknown(port, e);

		}
	}

#if GM_DEBUG
	recurse--;
#endif
}
