/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/************************************************************************
 * This file includes the OS-specific driver code for Solaris.
 ************************************************************************/

/* this is here to avoid gcc-2.8.1 and up problems with varargs */
#ifdef __GNUC__
#define _SYS_VARARGS_H
#include <stdarg.h>
#endif



#include "gm_arch.h"
#include "gm_call_trace.h"
#include "gm_lanai.h"
#include "gm_page_hash.h"
#include "gm_pio.h"

#if GM_SUPPORT_PCI
/* #warning Compiled with PCI support */
#include <sys/pci.h>
#endif

#include <sys/mkdev.h>
#include <sys/kmem.h>


#ifndef __GNUC__
#include <sys/varargs.h>
#endif

#include <sys/cmn_err.h>
#include <sys/ddi.h>
#include <sys/sunddi.h>

/************************************************************************
 * Forward declarations (used to build tables)
 ************************************************************************/

static int gm_solaris_open (dev_t * devp, int flags, int otyp,
			    cred_t * credp);
static int gm_solaris_close (dev_t dev, int flags, int otyp, cred_t * credp);
#if GM_BC_DRIVER
static int gm_solaris_read (dev_t dev, struct uio *uio_ptr,
			    cred_t * cred_ptr);
#endif
static int gm_solaris_mmap (dev_t dev, off_t off, int prot);
#if GM_OS_SOLARIS7
static int gm_solaris_ioctl (dev_t dev, int cmd, intptr_t arg,
			     int mode, cred_t * cred_p, int *rval_p);
#else
static int gm_solaris_ioctl (dev_t dev, int cmd, int arg,
			     int mode, cred_t * cred_p, int *rval_p);
#endif
static int gm_solaris_segmap (dev_t dev, off_t off, struct as *asp,
			      caddr_t * addrp, off_t len, unsigned int prot,
			      unsigned int maxprot, unsigned int flags,
			      cred_t * cred_p);

static int gm_solaris_identify (dev_info_t * dip);
static int gm_solaris_attach (dev_info_t * dip, ddi_attach_cmd_t cmd);
static int gm_solaris_detach (dev_info_t * dip, ddi_detach_cmd_t cmd);
static int gm_solaris_getinfo (dev_info_t * dip, ddi_info_cmd_t infocmd,
			       void *arg, void **result);
#if defined GM_CAN_REGISTER_MEMORY && defined GM_REGISTER_BY_WRITE
static int gm_solaris_aread (dev_t dev, struct aio_req *aio_reqp,
			     cret_t * cred_p);
static int gm_solaris_awrite (dev_t dev, struct aio_req *aio_reqp,
			      cret_t * cred_p);
#endif

/***********************************************************************
 * Globals
 ***********************************************************************/

/************
 * Loadable module required data structures
 ************/

static struct cb_ops gm_solaris_cb_ops = {
  gm_solaris_open,		/* open */
  gm_solaris_close,		/* close */
  nodev,			/* not a block driver */
  nodev,			/* no print routine */
  nodev,			/* no dump routine */
#if GM_BC_DRIVER
  gm_solaris_read,		/* read --- for compatibility only */
#else
  nodev,			/* no read routine */
#endif
  nodev,			/* no write routine */
  gm_solaris_ioctl,		/* ioctl */
  nodev,			/* no devmap routine */
  gm_solaris_mmap,		/* mmap */
  gm_solaris_segmap,		/* segmap -- using default */
  nochpoll,			/* no chpoll routine */
  ddi_prop_op,			/* prop_op -- using default */
  NULL,				/* cb_str not a STREAMS driver */
  D_NEW | D_MP,			/* safe for multi-thread/multi-processor */
  CB_REV,			/* revision */
#if GM_CAN_REGISTER_MEMORY && GM_REGISTER_BY_WRITE
  gm_solaris_aread,		/* aread */
  gm_solaris_awrite,		/* awrite */
#else
  nodev,			/* no aread routine */
  nodev				/* no awrite routine */
#endif
};

static struct dev_ops gm_solaris_ops = {
  DEVO_REV,			/* DEVO_REV indicated by manual  */
  0,				/* device reference count */
  gm_solaris_getinfo,		/* getinfo */
  gm_solaris_identify,		/* identify (contrary to docs, IS needed) */
  nulldev,			/* probe (nulldev self-identifying devices) */
  gm_solaris_attach,		/* attach */
  gm_solaris_detach,		/* detatch */
  nodev,			/* device reset routine -- set to nodev */
  &gm_solaris_cb_ops,
  (struct bus_ops *) NULL,	/* bus operations */
};

static struct modldrv modldrv = {
  &mod_driverops,
  "Myrinet GM Driver ($Id: gm_arch.c,v 1.103 2000/10/11 00:33:17 maxstern Exp $) ",
  &gm_solaris_ops,
};

static struct modlinkage modlinkage = {
  MODREV_1,
  {&modldrv, NULL},
};

/************
 * Other globals
 ************/

void *gm_instancep;		/* For per-device structures */

ddi_dma_attr_t gm_dma_attr = {
#ifdef ORIG_ONE
  DMA_ATTR_V0,			/* version */
  0x0,				/* addr_lo */
  0xFFFFFFFF,			/* addr_hi */
  0x7FFFFFFF,			/* count_max */
  0 /* set before used */ ,	/* align (for DMA resources) */
  0x7F,				/* burstsizes */
  1,				/* minxfer */
  0x7FFFFFFF,			/* maxxfer */
  0x7FFFFFFF,			/* seg */
  -1,				/* sgllen */
  1,				/* dma granularity */
  0				/* flags -- must be 0 */
#elif defined BOBS
  DMA_ATTR_V0,			/* version */
  (unsigned long long) 0x0,	/* addr lower */
  (unsigned long long) 0xFFFFFFFF,	/* addr upper */
  (unsigned long long) 0x00FFFFFF,	/* counter */
  (unsigned long long) 0x04,	/* 4-byte word alignment */
  0x7F,				/* burst size */
  4,				/* min-xfer size */
  (unsigned long long) 0x00FFFFFF,	/* max_xfer size */
  (unsigned long long) 0x00FFFFFF,	/* max_seg size */
  20,				/* Gather count */
  4,				/* dma granularity */
  0				/* flags -- must be 0 */
#else
  DMA_ATTR_V0,			/* version */
  0x0,				/* addr_lo */
  0xFFFFFFFF,			/* addr_hi */
  0x7FFFFFFF,			/* count_max */
  0 /* set before used */ ,	/* align (for DMA resources) */
  0x7F,				/* burstsizes */
  4,				/* minxfer */
  0x7FFFFFFF,			/* maxxfer */
  0x7FFFFFFF,			/* seg */
  255,				/* sgllen */
  0,				/* set before used *//* dma granularity */
  0				/* flags -- must be 0 */
#endif
};

struct ddi_device_acc_attr gm_dev_access_attr = {
  DDI_DEVICE_ATTR_V0,		/* version */
  DDI_NEVERSWAP_ACC,		/* endian flash */
  DDI_STRICTORDER_ACC		/* data order */
};

unsigned int gm_arch_solaris_instance_ok[32] = { 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0
};

/**********************************************************************
 * GM OS-abstraction required functions.
 **********************************************************************/

#if GM_SUPPORT_PCI

/************
 * PCI configuration space access functions.
 ************/

gm_status_t
gm_arch_read_pci_config_32 (gm_instance_state_t * is,
			    gm_offset_t offset, gm_u32_t * value)
{
  *value = pci_config_get32 (is->arch.pci_acc_handle, offset);
  return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_32 (gm_instance_state_t * is,
			     gm_offset_t offset, gm_u32_t value)
{
  pci_config_put32 (is->arch.pci_acc_handle, offset, value);
  return GM_SUCCESS;
}

/* 16-bit accesses */

gm_status_t
gm_arch_read_pci_config_16 (gm_instance_state_t * is,
			    gm_offset_t offset, gm_u16_t * value)
{
  *value = pci_config_get16 (is->arch.pci_acc_handle, offset);
  return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_16 (gm_instance_state_t * is,
			     gm_offset_t offset, gm_u16_t value)
{
  pci_config_put16 (is->arch.pci_acc_handle, offset, value);
  return GM_SUCCESS;
}

/* 8-bit accesses */

gm_status_t
gm_arch_read_pci_config_8 (gm_instance_state_t * is,
			   gm_offset_t offset, gm_u8_t * value)
{
  *value = pci_config_get8 (is->arch.pci_acc_handle, offset);
  return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_8 (gm_instance_state_t * is,
			    gm_offset_t offset, gm_u8_t value)
{
  pci_config_put8 (is->arch.pci_acc_handle, offset, value);
  return GM_SUCCESS;
}
#endif

/************
 * GM synchronization functions
 ************/

/* For solaris, we must be careful to pass an appropriate iblock
   cookie to gm_mutex_init if the mutex is ever user by the interrupt
   handler. */

void
gm_arch_sync_init (gm_arch_sync_t * s, gm_instance_state_t * is)
{
  ddi_iblock_cookie_t _mu_cookie;

#if 1
  if (s == &is->pause_sync)
    {
      _mu_cookie = is->arch.iblock_cookie;
      gm_assert (is->arch.iblock_cookie);
    }
  else
    {
      _mu_cookie = 0;
    }
#elif 0
  _mu_cookie = is ? is->arch.iblock_cookie : NULL;
  /* gm_assert (_mu_cookie || !is); */
#else
  _mu_cookie = NULL;
#endif

  gm_bzero (s, sizeof (*s));
  mutex_init (&s->mu, "GM mutex", MUTEX_DRIVER, 0);
  mutex_init (&s->_mu, "GM condvar mutex", MUTEX_DRIVER, _mu_cookie);
  cv_init (&s->cv, "GM condvar", CV_DRIVER, NULL);
}

void
gm_arch_sync_reset (gm_arch_sync_t * s)
{
  s->wake_cnt = 0;
}

void
gm_arch_sync_destroy (gm_arch_sync_t * s)
{
  GM_CALLED ();
  cv_destroy (&s->cv);
  mutex_destroy (&s->_mu);
  mutex_destroy (&s->mu);
  GM_RETURN_NOTHING ();
}

void
gm_arch_mutex_enter (gm_arch_sync_t * s)
{
  mutex_enter (&s->mu);
}

void
gm_arch_mutex_exit (gm_arch_sync_t * s)
{
  mutex_exit (&s->mu);
}

/* Sleep functions. Return 0 on wake, -1 on timeout, and 1 on signal. */

/* sleep until awakened or get a signal */

gm_arch_sleep_status_t gm_arch_signal_sleep (gm_arch_sync_t * s)
{
  gm_arch_sleep_status_t ret = GM_SLEEP_WOKE;

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_arch_signal_sleep() called.\n"));

  mutex_enter (&s->_mu);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_arch_signal_sleep() entered mutex.\n"));
  while (s->wake_cnt <= 0)
    {
      if (cv_wait_sig (&s->cv, &s->_mu) == 0)
	{
	  GM_NOTE (("gm_arch_signal_sleep returning GM_SLEEP_INTERRUPTED\n"));
	  ret = GM_SLEEP_INTERRUPTED;
	  break;
	}
    }
  s->wake_cnt--;
  mutex_exit (&s->_mu);

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_arch_signal_sleep() returning.\n"));

  return ret;
}

/* utility function to do either type of timed sleep */

gm_arch_sleep_status_t gm_arch_timed_sleep (gm_arch_sync_t * s, int seconds)
{
  long timeout_time;
  gm_arch_sleep_status_t ret = GM_SLEEP_WOKE;

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_arch_timed_sleep() called.\n"));

  drv_getparm (LBOLT, &timeout_time);
  timeout_time += seconds * drv_usectohz (1000000);

  mutex_enter (&s->_mu);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_arch_timed_sleep() entered mutex.\n"));
  while (s->wake_cnt <= 0)
    {
      if (cv_timedwait (&s->cv, &s->_mu, timeout_time) == -1)
	{
	  GM_NOTE (("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT\n"));
	  ret = GM_SLEEP_TIMED_OUT;
	  break;
	}
    }
  s->wake_cnt--;
  mutex_exit (&s->_mu);

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_arch_timed_sleep() returning.\n"));

  return ret;
}

/* Wake the thread sleeping on the synchronization variable. */

void
gm_arch_wake (gm_arch_sync_t * s)
{
  GM_PRINT (GM_PRINT_LEVEL >= 10, ("gm_arch_wake() called.\n"));

  mutex_enter (&s->_mu);
  GM_PRINT (GM_PRINT_LEVEL >= 10, ("gm_arch_wake() entered mutex.\n"));
  s->wake_cnt++;
  cv_signal (&s->cv);
  mutex_exit (&s->_mu);

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("gm_arch_wake() returning.\n"));
}

/* Sleep for the indicated duration */

void
gm_arch_spin (gm_instance_state_t * is, gm_u32_t usecs)
{
#if GM_SOLARIS_COMPLICATED_SPIN
  long timeout_time;
  drv_getparm (LBOLT, &timeout_time);
  timeout_time += drv_usectohz (usecs);

  mutex_enter (&is->arch.spin_sync._mu);
  cv_timedwait_sig (&is->arch.spin_sync.cv, &is->arch.spin_sync._mu,
		    timeout_time);
  mutex_exit (&is->arch.spin_sync._mu);
#else
  drv_usecwait (usecs);
#endif
}

gm_status_t gm_arch_gethostname (char *ptr, int len)
{
  /* not implemented yet */
  return (GM_FAILURE);
}

/************
 * Page locking
 ************/

#if GM_CAN_REGISTER_MEMORY
typedef struct gm_solaris_page_lock
{
  struct buf *buf;
  ddi_dma_handle_t dma_handle;
}
gm_solaris_page_lock_t;

typedef struct gm_solaris_page_key
{
  unsigned int port;
  gm_up_t user_vma;
}
gm_solaris_page_lock_key_t;

long
gm_solaris_page_lock_compare (void *a, void *b)
{
  return gm_memcmp (a, b, sizeof (gm_solaris_page_lock_key_t));
}

unsigned long
gm_solaris_page_lock_hash (void *a)
{
  return gm_crc (a, sizeof (gm_solaris_page_lock_key_t));
}

void
gm_solaris_print_buf (struct buf *b)
{
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("buf %p = {\n"));
#define GM_FOO(a,c) GM_PRINT (GM_PRINT_LEVEL >= 5,		\
			      ("  " #a " = " c ",\n", b->a))
  GM_FOO (b_flags, "0x%x");
  GM_FOO (av_forw, "%p");
  GM_FOO (av_back, "%p");
  GM_FOO (b_bcount, "0x%x");
  GM_FOO (b_un.b_addr, "%p");
  GM_FOO (b_blkno, "%ld");
  GM_FOO (b_resid, "0x%x");
  GM_FOO (b_bufsize, "0x%x");
  GM_FOO (b_iodone, "%p");
  GM_FOO (b_error, "0x%x");
  GM_FOO (b_private, "%p");
  GM_FOO (b_edev, "0x%lx");
#undef GM_FOO
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("}\n"));
}

int
gm_solaris_iodone (struct buf *b)
{
  return 0;
}

gm_status_t
gm_arch_lock_user_buffer_page (gm_instance_state_t * is, void *in,
			       gm_dp_t * dma_addr, gm_arch_page_lock_t * lock)
{
#if GM_REGISTER_BY_WRITE
  /* The page is already locked by aphysio.  We just need to fetch
     and return the DMA address stored in is->arch.page_lock_hash by
     gm_solaris_strategy(). */

  gm_solaris_page_lock_t *lock;
  gm_solaris_page_lock_key_t key;

  key.user_addr = in;
  key.port = port;

  lock = gm_hash_find (is->arch.page_lock_hash, &key);
  if (!lock)
    {
      GM_WARN (("Internal error in gm_arch_lock_user_buffer_page.\n"));
      return GM_INTERNAL_ERROR;
    }
  *dma_addr = lock->dma_addr;
  return GM_SUCCESS;
#else
  struct buf *b;
  uint_t cnt;
  ddi_dma_cookie_t dma_cookie;
  ddi_dma_handle_t dma_handle;
  gm_solaris_page_lock_t lock;

  GM_CALLED ();

  gm_assert (GM_PAGE_LEN);
  gm_assert (GM_PAGE_ALIGNED (in));

  /* Map the pages into the kernel by hacking with bufs. */

  b = getrbuf (KM_NOSLEEP);
  if (!b)
    {
      GM_NOTE (("Could not allocate buf to lock page.\n"));
      goto abort_with_nothing;
    }
  b->b_bcount = b->b_bufsize = GM_PAGE_LEN;
  b->b_flags = B_PHYS;		/* indicate address is a user VMA */
  b->b_un.b_addr = in;

  b->b_edev = is->arch.dev;
  b->b_iodone = gm_solaris_iodone;
  b->av_forw = b->av_back = b;
  gm_solaris_print_buf (b);
  bp_mapin (b);
  gm_solaris_print_buf (b);
  if (!b->b_un.b_addr)
    {
      GM_NOTE (("Could not lock down user memory.\n"));
      goto abort_with_rbuf;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("mapped user VMA %p to kernel VMA %p.\n", in, b->b_un.b_addr));

  /* Get the DMA addr, by hacking with DMA cookies. */

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Allocating DMA handle.\n"));
  if (ddi_dma_alloc_handle (is->arch.dip, &gm_dma_attr, 0, 0,
			    &dma_handle) != DDI_SUCCESS)
    {
      GM_NOTE (("Could not allocate dma handle.\n"));
      goto abort_with_mapping;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Binding DMA handle.\n"));
  if (ddi_dma_addr_bind_handle (dma_handle,
#if 0
				/* map kernel VMA */
				0, b->b_un.b_addr,
#else
				/* map user VMA */
				threadp ()->t_procp->p_as, in,
#endif
				GM_PAGE_LEN,
				DDI_DMA_RDWR | DDI_DMA_CONSISTENT,
				DDI_DMA_DONTWAIT, NULL,
				&dma_cookie, &cnt) != DDI_DMA_MAPPED)
    {
      GM_NOTE (("Could not bind DMA handle.\n"));
      goto abort_with_dma_handle;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Performing sanity checks.\n"));
  if (cnt != 1)
    {
      GM_NOTE (("internal error: too many cookies.\n"));
      goto abort_with_binding;
    }
  if (dma_cookie.dmac_size != GM_PAGE_LEN)
    {
      GM_NOTE (("internal error: unexpected cookie size.\n"));
      goto abort_with_binding;
    }

  /* Store the lock where we can find it when we want to unlock. */

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Storing lock in hash table.\n"));
  {
    gm_solaris_page_lock_key_t key;

    lock.buf = b;
    lock.dma_handle = dma_handle;
    key.user_vma = in;
    key.port = port;

    if (gm_hash_insert (is->arch.page_lock_hash, &key, &lock) != GM_SUCCESS)
      {
	GM_NOTE (("Could not store locking information.\n"));
	goto abort_with_binding;
      }
  }

  /* Report success */

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Reporting success.\n"));
  if (sizeof (gm_dp_t) == 4)
    *dma_addr = (gm_dp_t) dma_cookie.dmac_address;
  else
    *dma_addr = (gm_dp_t) dma_cookie.dmac_laddress;
  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_binding:
  ddi_dma_unbind_handle (lock.dma_handle);
abort_with_dma_handle:
  ddi_dma_free_handle (&lock.dma_handle);
abort_with_mapping:
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("mapping out\n"));
  bp_mapout (b);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("freeing buf\n"));
abort_with_rbuf:
  freerbuf (b);
abort_with_nothing:
  GM_RETURN_STATUS (GM_FAILURE);
#endif
}

void
gm_arch_unlock_user_buffer_page (gm_arch_page_lock_t * lock)
{
#if GM_REGISTER_BY_WRITE
  /* We don't need to unlock the page, because that will be done by
     gm_solaris_strategy(). */
  return;
#elif 0
  ddi_dma_unbind_handle (lock->dma_handle);
  ddi_dma_free_handle (&lock->dma_handle);
  gm_bzero (lock, sizeof (*lock));
#else
  gm_solaris_page_lock_key_t key;
  gm_solaris_page_lock_t *lock;

  key.user_vma = user_vma;
  key.port = port;
  lock = gm_hash_find (is->arch.page_lock_hash, &key);
  gm_assert (lock);
  ddi_dma_unbind_handle (lock->dma_handle);
  ddi_dma_free_handle (&lock->dma_handle);
  bp_mapout (lock->buf);
  freerbuf (lock->buf);
  gm_hash_remove (is->arch.page_lock_hash, &key);
#endif
}

#endif /* GM_CAN_REGISTER_MEMORY */

/************************************************************************
 * Architecture-specific required functions
 ************************************************************************/

void gm_arch_abort (void)
{
	cmn_err (CE_CONT, "***** don't panic, but gm_arch_abort () was called.****\n");
}


/* Return the value of GM_PAGE_LEN set during driver initialization. */

gm_status_t gm_arch_get_page_len (unsigned long *page_len)
{
  if (!GM_PAGE_LEN)
    {
      GM_NOTE (("gm_solaris internal error: GM_PAGE_LEN unknown in call\n"
		"to gm_arch_get_page_len().\n"));
      return GM_FAILURE;
    }
  *page_len = GM_PAGE_LEN;
  return GM_SUCCESS;
}

/************
 * gm_port_state initialization 
 ************/

/* This is called just after the port state is created (in gm_minor.c)
   to perform architecture-specific initialization. */

gm_status_t gm_arch_port_state_init (gm_port_state_t * ps)
{
  return GM_SUCCESS;
}

/* This is called just before the port state is destroyed (in
   gm_minor.c) to perform architecture-specific finalization. */

void
gm_arch_port_state_fini (gm_port_state_t * ps)
{
  return;
}


/* This is called just after the port state is initialized (in gm_minor.c)
   to perform architecture-specific initialization. */

gm_status_t gm_arch_port_state_open (gm_port_state_t * ps)
{
  return GM_SUCCESS;
}

/* This is called just before the port state is finalized (in
   gm_minor.c) to perform architecture-specific finalization. */

void
gm_arch_port_state_close (gm_port_state_t * ps)
{
  return;
}

/************
 * gm_ioctl support
 ************/

/* Copy user data to a kernel buffer. */

gm_status_t
gm_arch_copyin (gm_port_state_t * ps,
		void *what, void *where, gm_size_t amount)
{
  int errno;

  errno = ddi_copyin ((caddr_t) what, (caddr_t) where, amount,
		      ps->arch.ddi_copyxx_context);
  return errno ? GM_FAILURE : GM_SUCCESS;
}

/* Copy kernel data to a user buffer */

gm_status_t
gm_arch_copyout (gm_port_state_t * ps,
		 void *what, void *where, gm_size_t amount)
{
  int errno;

  errno = ddi_copyout ((caddr_t) what, (caddr_t) where, amount,
		       ps->arch.ddi_copyxx_context);
  return errno ? GM_FAILURE : GM_SUCCESS;
}

/************
 * Kernel Memory Allocation
 ************/

/* malloc() analogue for the driver */

void *
gm_arch_kernel_malloc (unsigned long len, int flags)
{
  gm_u32_t *p;

  /* Allocate extra space at beginning to store length for kmem_free call, and
     return an 8-byte aligned buffer. */
  gm_assert (sizeof (long) <= 8);
  p = (gm_u32_t *) kmem_alloc (8 + len, KM_NOSLEEP);
  if (!p)
    return p;
  p[0] = 8 + len;
  return &p[2];
}

/* free() analogue for the driver */

void
gm_arch_kernel_free (void *ptr)
{
  gm_u32_t *p;

  p = ptr;
  kmem_free (&p[-2], p[-2]);
}

/************
 * DMA region manipulation
 ************/

/************
 * ??? Required for non cache-coherent I/O architectures, which we
 * do not yet support.
 ************/

gm_status_t gm_arch_dma_region_sync (gm_arch_dma_region_t * r, int command)
{
  int errno;

  errno = ddi_dma_sync (r->dma_handle, 0, 0, command);
  return errno ? GM_FAILURE : GM_SUCCESS;
}

/* Allocate LEN bytes of DMA memory that is contiguous in kernal space
   by possibly segmented in DMA space.

   If r->register_function is non-null, call r->register_page_function
   (r, dma_addr) for each page. */

gm_status_t
gm_arch_dma_region_alloc (gm_instance_state_t * is, gm_arch_dma_region_t * r,
			  gm_size_t len, gm_u32_t flags,
			  gm_status_t (*register_page_func) (void *,
							     gm_dp_t,
							     gm_u32_t),
			  void *arg)
{
  int error;
  int ddi_flags;
  size_t trash;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("in get_dma_region_alloc; len = %d\n", len));

  if (!GM_PAGE_LEN)
    {
      error = GM_INTERNAL_ERROR;
      goto abort_with_nothing;
    }

  gm_dma_attr.dma_attr_align = GM_PAGE_LEN;
  gm_dma_attr.dma_attr_granular = GM_PAGE_LEN;
  error = ddi_dma_alloc_handle (is->arch.dip, &gm_dma_attr, DDI_DMA_DONTWAIT,
				NULL, &r->dma_handle);
  if (error != DDI_SUCCESS)
    {
      GM_NOTE (("Could not allocate DMA handle; error = %d.", error));
      if (error == DDI_DMA_BADATTR)
	GM_NOTE (("Bad DMA attributes."));
      goto abort_with_nothing;
    }

  ddi_flags = 0;
  if (flags & GM_ARCH_DMA_READ)
    {
      ddi_flags |= DDI_DMA_READ;
      flags &= ~GM_ARCH_DMA_READ;
    }
  if (flags & GM_ARCH_DMA_WRITE)
    {
      ddi_flags |= DDI_DMA_WRITE;
      flags &= ~GM_ARCH_DMA_WRITE;
    }
  if (flags & GM_ARCH_DMA_CONSISTENT)
    {
      ddi_flags |= DDI_DMA_CONSISTENT;
      flags &= ~GM_ARCH_DMA_CONSISTENT;
    }
  if (flags & GM_ARCH_DMA_STREAMING)
    {
      ddi_flags |= DDI_DMA_STREAMING;
      flags &= ~GM_ARCH_DMA_STREAMING;
    }
  if (flags != 0)
    {
      GM_NOTE (("ddi_dma_mem_alloc: warning - got an unexpected flag\n"));
      error = GM_INTERNAL_ERROR;
      goto abort_with_nothing;
    }
  ddi_flags |= DDI_DMA_CONSISTENT;

  error = ddi_dma_mem_alloc (r->dma_handle,
			     GM_PAGE_ROUNDUP (u32, len) + GM_PAGE_LEN,
			     &gm_dev_access_attr,
			     ddi_flags & ~(DDI_DMA_READ | DDI_DMA_WRITE),
			     DDI_DMA_DONTWAIT, NULL, (caddr_t *) & r->addr,
			     &trash, &r->acc_handle);
  if (error != DDI_SUCCESS)
    {
      GM_NOTE (("Could not allocate DMAable memory.\n"));
      goto abort_with_dma_handle;
    }

  r->addr = GM_PAGE_ROUNDUP (ptr, r->addr);

  error = ddi_dma_addr_bind_handle (r->dma_handle, NULL, (caddr_t) r->addr,
				    GM_PAGE_ROUNDUP (u32, len),
				    ddi_flags,
				    DDI_DMA_DONTWAIT, NULL,
				    &r->cookie, &r->cookie_cnt);
  if (error != DDI_SUCCESS)
    {
      GM_NOTE (("Could not bind dma handle."));
      goto abort_with_dma_mem;
    }

  /* Compute the DMA status bits for the dma region. */

  r->sts = ddi_dma_burstsizes (r->dma_handle) >> 3 & 0x1ff;

#if GM_SUPPORT_PCI
  /* Verify computed PCI burstsize is legit. */

  if ((is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI) && (r->sts != 0xf))
    {
      GM_NOTE (("PCI burst size not computed correctly.\n"));
      error = GM_INTERNAL_ERROR;
      goto abort_with_bound_handle;
    }
#endif

  /* Call the page registration functions for each page in the DMA
     region. */

  if (register_page_func != NULL)
    {
      unsigned cookie_cnt, page_num = 0;
      ddi_dma_cookie_t cookie;

      cookie_cnt = r->cookie_cnt;
      cookie = r->cookie;
      while (cookie_cnt--)
	{
	  ASSERT (GM_PAGE_ALIGNED (cookie.dmac_address));
	  ASSERT (GM_PAGE_ALIGNED (cookie.dmac_size));
	  while (cookie.dmac_size)
	    {
	      if (register_page_func (arg, cookie.dmac_address, page_num++)
		  != GM_SUCCESS)
		{
		  error = GM_OUT_OF_MEMORY;	/* out of contiguous mem */
		  goto abort_with_bound_handle;
		}
	      cookie.dmac_address += GM_PAGE_LEN;
	      cookie.dmac_size -= GM_PAGE_LEN;
	    }
	  if (cookie_cnt)
	    ddi_dma_nextcookie (r->dma_handle, &cookie);
	}
    }

  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_bound_handle:
  ddi_dma_unbind_handle (r->dma_handle);
abort_with_dma_mem:
  ddi_dma_mem_free (&r->acc_handle);
abort_with_dma_handle:
  ddi_dma_free_handle (&r->dma_handle);
abort_with_nothing:
  GM_RETURN_STATUS (error);
}

void
gm_arch_dma_region_free (gm_arch_dma_region_t * r)
{
  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("unbinding\n"));
  ddi_dma_unbind_handle (r->dma_handle);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("mem freeing\n"));
  ddi_dma_mem_free (&r->acc_handle);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("handle freeing\n"));
  ddi_dma_free_handle (&r->dma_handle);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("bzeroing\n"));
  bzero ((void *) r, sizeof (*r));

  GM_RETURN_NOTHING ();
}

/* Return the DMA address of the next page in the DMA region. */

static gm_dp_t
_gm_dma_region_dma_addr (gm_arch_dma_region_t * r, int advance)
{
  gm_dp_t ret;

  GM_PRINT (GM_PRINT_LEVEL >= 7, ("Using cookie: (0x%x, 0x%x, 0x%x)\n",
				  r->cookie.dmac_address, r->cookie.dmac_size,
				  r->cookie.dmac_type));

  ASSERT (r->cookie.dmac_size);
  ASSERT (GM_PAGE_ALIGNED (r->cookie.dmac_size));
  ASSERT (GM_PAGE_ALIGNED (r->cookie.dmac_address));

  ret = (gm_dp_t) r->cookie.dmac_address;

  if (!advance)
    return ret;

  if (!GM_PAGE_LEN)
    return GM_INTERNAL_ERROR;

  r->cookie.dmac_size -= GM_PAGE_LEN;
  r->cookie.dmac_address =
    (unsigned int) ((char *) (r->cookie.dmac_address) + GM_PAGE_LEN);

  if (!r->cookie.dmac_size && (r->cookie_cnt > 1))
    {
      ddi_dma_nextcookie (r->dma_handle, &r->cookie);
      r->cookie_cnt--;
    }
  return ret;
}

gm_dp_t gm_arch_dma_region_dma_addr (gm_arch_dma_region_t * r)
{
  return _gm_dma_region_dma_addr (r, 0);
}

gm_dp_t gm_arch_dma_region_dma_addr_advance (gm_arch_dma_region_t * r)
{
  return _gm_dma_region_dma_addr (r, 1);
}

/* Return the kernal virtual address for a DMA region */

void *
gm_arch_dma_region_kernel_addr (gm_arch_dma_region_t * r)
{
  return r->addr;
}

/* Return the STS bits for a DMA region. */

gm_s32_t gm_arch_dma_region_status (gm_arch_dma_region_t * r)
{
  return r->sts;
}

/************
 * I/O space mapping
 ************/

#if GM_SUPPORT_PCI
gm_status_t gm_arch_enable_PCI_bus_master_DMA (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Enabling PCI bus master DMA.\n"));

  if (!is->arch.pci_acc_handle)
    {
      GM_NOTE (("internal error: is->arch.pci_acc_handle not set.\n"));
      return GM_FAILURE;
    }

  is->ifc.pci.config.Command
    =
    pci_config_get16 (is->arch.pci_acc_handle,
		      GM_OFFSETOF (gm_pci_config_t, Command));
  is->ifc.pci.config.Command |= GM_PCI_COMMAND_MASTER;
  pci_config_put16 (is->arch.pci_acc_handle,
		    GM_OFFSETOF (gm_pci_config_t, Command),
		    is->ifc.pci.config.Command);

  /* Pause for at least 10ms */
  gm_arch_spin (is, 15000);

  return GM_SUCCESS;
}

gm_status_t gm_arch_disable_PCI_bus_master_DMA (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Disabling PCI bus master DMA.\n"));

  if (!is->arch.pci_acc_handle)
    {
      GM_NOTE (("internal error: is->arch.pci_acc_handle not set.\n"));
      return GM_FAILURE;
    }

  is->ifc.pci.config.Command
    =
    pci_config_get16 (is->arch.pci_acc_handle,
		      GM_OFFSETOF (gm_pci_config_t, Command));
  is->ifc.pci.config.Command &= ~GM_PCI_COMMAND_MASTER;
  pci_config_put16 (is->arch.pci_acc_handle,
		    GM_OFFSETOF (gm_pci_config_t, Command),
		    is->ifc.pci.config.Command);

  /* Pause for at least 10ms */
  gm_arch_spin (is, 15000);

  return GM_SUCCESS;
}
#endif /* GM_SUPPORT_PCI */

/* utility function for the following functions.  Returns the Solaris
   register set to use for the board */

gm_status_t gm_solaris_reg_set (gm_instance_state_t * is, int *reg_set)
{
  /* return cached value, if any. */
  if (is->arch.reg_set_cached)
    {
      *reg_set = is->arch.reg_set;
      return GM_SUCCESS;
    }

#if GM_SUPPORT_PCI
#  if GM_SUPPORT_64_BIT_PCI
#    error not yet supported
#  endif
  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      /* Determine the register set containing the PCI resource we
         want to map (the memory-mappable part of the interface). We do
         this by scanning the DDI "reg" property of the interface,
         which is an array of gm_ddi_reg_set structures.  */

#define REGISTER_NUMBER(ip) (ip[0] >>  0 & 0xff)
#define FUNCTION_NUMBER(ip) (ip[0] >>  8 & 0x07)
#define DEVICE_NUMBER(ip)   (ip[0] >> 11 & 0x1f)
#define BUS_NUMBER(ip)      (ip[0] >> 16 & 0xff)
#define ADDRESS_SPACE(ip)   (ip[0] >> 24 & 0x03)
#define PCI_ADDR_HIGH(ip)   (ip[1])
#define PCI_ADDR_LOW(ip)    (ip[2])
#define PCI_SPAN_HIGH(ip)   (ip[3])
#define PCI_SPAN_LOW(ip)    (ip[4])

#define GM_DDI_REG_SET_32_BIT_MEMORY_SPACE 2
#define GM_DDI_REG_SET_64_BIT_MEMORY_SPACE 3

      int *data, i, *rs;
      u_int nelementsp;
      char *address_space_name[] = { "Configuration Space",
	"I/O Space",
	"32-bit Memory Space",
	"64-bit Memory Space"
      };


      if (ddi_prop_lookup_int_array (DDI_DEV_T_ANY, is->arch.dip,
				     DDI_PROP_DONTPASS, "reg", &data,
				     &nelementsp) != DDI_SUCCESS)
	{
	  GM_NOTE (("Could not determine register set.\n"));
	  return GM_FAILURE;
	}

      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("There are %d register sets.\n", nelementsp / 5));
      if (!nelementsp)
	{
	  GM_NOTE (("Didn't find any \"reg\" properties.\n"));
	  ddi_prop_free (data);
	  return GM_FAILURE;
	}

      /* Scan for the register number. */

      GM_PRINT (GM_PRINT_LEVEL >= 1, ("*** Scanning for register number.\n"));
      for (i = 0; i < nelementsp / 5; i++)
	{
	  rs = &data[5 * i];

	  GM_PRINT (GM_PRINT_LEVEL >= 2, ("Examining register set %d:\n", i));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  Register number = %d.\n", REGISTER_NUMBER (rs)));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  Function number = %d.\n", FUNCTION_NUMBER (rs)));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  Device number   = %d.\n", DEVICE_NUMBER (rs)));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  Bus number      = %d.\n", BUS_NUMBER (rs)));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  Address space   = %d (%s ).\n", ADDRESS_SPACE (rs),
		     address_space_name[ADDRESS_SPACE (rs)]));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  pci address 0x%08x %08x\n", PCI_ADDR_HIGH (rs),
		     PCI_ADDR_LOW (rs)));
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("  pci span 0x%08x %08x\n", PCI_SPAN_HIGH (rs),
		     PCI_SPAN_LOW (rs)));

	  /* We are looking for a memory property. */

	  if (ADDRESS_SPACE (rs) == GM_DDI_REG_SET_64_BIT_MEMORY_SPACE
	      || ADDRESS_SPACE (rs) == GM_DDI_REG_SET_32_BIT_MEMORY_SPACE)
	    {
	      *reg_set = i;
	      is->arch.reg_set = i;
	      is->arch.reg_set_cached = 1;
	      GM_PRINT (GM_PRINT_LEVEL >= 5,
			("Memory uses register set %d.\n", *reg_set));

	      /* Round board span up to a power of two to fix a bug */
	      /* in the Ultrasparc version of Solaris 2.6. */
	      is->board_span = 1 << gm_log2_roundup (PCI_SPAN_LOW (rs));

	      is->arch.board_span_cached = 1;
	      GM_PRINT (GM_PRINT_LEVEL >= 5,
			("Board span is 0x%x\n", (int) is->board_span));
	      break;
	    }
	}

      ddi_prop_free (data);

      /* If no match, fail. */
      if (i >= nelementsp / 5)
	{
	  GM_PRINT (GM_PRINT_LEVEL >= 2,
		    ("gm_solaris_reg_set returning GM_FAILURE\n"));
	  return GM_FAILURE;
	}

      GM_PRINT (GM_PRINT_LEVEL >= 2,
		("gm_solaris_reg_set returning GM_SUCCESS\n"));
      return GM_SUCCESS;
    }
#endif

#if GM_SUPPORT_SBUS
  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      *reg_set = is->arch.reg_set = 0;
      is->arch.reg_set_cached = 1;
      return GM_SUCCESS;
    }
#endif

  return GM_FAILURE;
}

gm_status_t
gm_arch_map_io_space (gm_instance_state_t * is, gm_u32_t offset, gm_u32_t len,
		      void **kaddr)
{
  int reg_set;

  if (gm_solaris_reg_set (is, &reg_set) != GM_SUCCESS)
    return GM_FAILURE;

  /* Map the entire board into memory. */

#if 0
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("gm_arch_map_io_space: mapping reg_set=%d at offset 0x%x, len 0x%x.\n",
	     reg_set, offset, len));
  if (ddi_map_regs (is->arch.dip, reg_set, (caddr_t *) kaddr, offset, len) !=
      DDI_SUCCESS)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 2,
		("gm_arch_map_io_space: ddi_map_regs failed\n"));
      return GM_FAILURE;
    }
#else
  if (ddi_regs_map_setup (is->arch.dip, reg_set,
			  (caddr_t *) kaddr,
			  offset, len, &gm_dev_access_attr,
			  &is->arch.acc_handle) != DDI_SUCCESS)
    return GM_FAILURE;
#endif

  return GM_SUCCESS;
}

void
gm_arch_unmap_io_space (gm_instance_state_t * is, gm_u32_t offset,
			gm_u32_t len, void **kaddr)
{
  int reg_set;

  if (gm_solaris_reg_set (is, &reg_set) != GM_SUCCESS)
    return;

  GM_PRINT (GM_PRINT_LEVEL >= 2,
	    ("Unmapping reg_set = %d  at kaddr 0x%x offset 0x%x, len 0x%x.\n",
	     reg_set, (unsigned int) *kaddr, offset, len));

#if 0
  ddi_unmap_regs (is->arch.dip, reg_set, (caddr_t *) kaddr, offset, len);
#else
  ddi_regs_map_free (&is->arch.acc_handle);
#endif
}

#if GM_SUPPORT_SBUS

/************
 * SBUS eeprom copying
 ************/

/* Platform-dependent code for copying the eeprom information off of a
   SBUS board.  Note that the information stored in the eeprom on SBUS
   boards is accessed as individual SBUS properties in older boards
   and as a byte array in newer cards. */

gm_status_t gm_arch_sbus_copy_eeprom (gm_instance_state_t * is)
{
  int unit;
  u_int eeprom_len;
  dev_info_t *dip;
  gm_myrinet_eeprom_t *eeprom_ptr = NULL;

  GM_CALLED ();

  gm_assert (is);
  gm_assert (is->arch.dip);
  gm_assert (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS);

  unit = is->id;
  dip = is->arch.dip;

  /* Try to read the EEPROM as a byte array.  This works for newer boards. */

  if (ddi_prop_lookup_byte_array (DDI_DEV_T_ANY, dip,
				  DDI_PROP_DONTPASS, "myrinet-eeprom-info",
				  (u_char **) & eeprom_ptr, &eeprom_len)
      == DDI_PROP_SUCCESS)
    {
      is->lanai.eeprom = *eeprom_ptr;
      ddi_prop_free (eeprom_ptr);
    }
  else
    {
      /* Could not read the EEPROM as a byte array, so try to
         construct an eeprom from the board properties.  This works
         for older boards. */

      unsigned char *board_id = 0;
      int addrlen = 10;

      GM_INFO (("[%d]: looking for older EEPROM\n", unit));

      is->lanai.eeprom.board_type = 0xffff;
      is->lanai.eeprom.bus_type = GM_MYRINET_BUS_SBUS;

      is->lanai.eeprom.lanai_cpu_version =
	ddi_prop_get_int (DDI_DEV_T_ANY, dip, DDI_PROP_DONTPASS,
			  "cpu_version", 0);
      if (is->lanai.eeprom.lanai_cpu_version == 0)
	{
	  GM_NOTE (("[%d]: cpu_version property not found\n", unit));
	  goto abort_with_nothing;
	}

      is->lanai.eeprom.lanai_clockval =
	ddi_prop_get_int (DDI_DEV_T_ANY, dip, DDI_PROP_DONTPASS,
			  "clock_value", 0);
      if (is->lanai.eeprom.lanai_clockval == 0)
	{
	  GM_NOTE (("[%d]: clockval property not found\n", unit));
	  goto abort_with_nothing;
	}

      is->lanai.eeprom.lanai_sram_size =
	ddi_getprop (DDI_DEV_T_ANY, dip, DDI_PROP_DONTPASS, "sram_size", 0);
      if (is->lanai.eeprom.lanai_sram_size == 0)
	{
	  GM_NOTE (("[%d]: sram_size property not found\n", unit));
	  goto abort_with_nothing;
	}

      if (ddi_prop_lookup_byte_array (DDI_DEV_T_ANY, dip, DDI_PROP_DONTPASS,
				      "myrinet-board-id",
				      &board_id,
				      (u_int *) & addrlen)
	  != DDI_PROP_SUCCESS)
	{
	  GM_NOTE (("[%d]: board_id property not found\n", unit));
	  goto abort_with_nothing;
	}

      bcopy (&board_id[0], &is->lanai.eeprom.lanai_board_id[0], 6);
      ddi_prop_free (board_id);
    }

  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_nothing:
  GM_RETURN_STATUS (GM_FAILURE);
}
#endif

/************************************************************************
 * Solaris-specific functions (not required by GM OS-abstraction)
 ************************************************************************/

#if 0
/* Get an iblock cookie just like ddi_get_iblock_cookie, only
   compatible with earlier versions of Solaris. */

gm_status_t
gm_solaris_get_iblock_cookie (dev_info_t * dip, u_int inumber,
			      ddi_iblock_cookie_t * iblock_cookie)
{
// #define GM_SOLARIS_NEW_STYLE_INTERRUPTS
#if GM_SOLARIS_NEW_STYLE_INTERRUPTS
  return ddi_get_iblock_cookie (dip, inumber, iblock_cookie);
#else
  ddi_idevice_cookie_t foo;

  if (ddi_add_ntr (dip, 0, iblock_cookie, &foo,
		   (u_int (*)(caddr_t)) nulldev, NULL) != DDI_SUCCESS)
    {
      GM_NOTE (("could not add null interrupt handler."));
      goto abort_with_nothing;
    }
  return GM_SUCCESS;

abort_with_nothing:
  return GM_FAILURE;
#endif
}
#endif

/************
 * Utility functions for generic error printing
 ************/

/* initialized in _init. */
kmutex_t gm_solaris_print_mu;

void
gm_solaris_print (char *format, ...)
{
  va_list ap;
  /* mutex_enter (&gm_solaris_print_mu); */
  va_start (ap, format);
  vcmn_err (CE_CONT, format, ap);
  va_end (ap);

  /* mutex_exit (&gm_solaris_print_mu); */
/*   drv_usecwait (100000); */
}

void
gm_solaris_info (char *format, ...)
{
  va_list ap;
  /* mutex_enter (&gm_solaris_print_mu); */
  va_start (ap, format);
  vcmn_err (CE_CONT, format, ap);
  va_end (ap);

  /* mutex_exit (&gm_solaris_print_mu); */
/*   drv_usecwait (100000); */
}

void
gm_solaris_warn (char *format, ...)
{
  va_list ap;
  /* mutex_enter (&gm_solaris_print_mu); */
  va_start (ap, format);
  vcmn_err (CE_WARN, format, ap);
  va_end (ap);
  /* mutex_exit (&gm_solaris_print_mu); */
/*   drv_usecwait (100000); */
}

void
gm_solaris_note (char *format, ...)
{
  va_list ap;
  /* mutex_enter (&gm_solaris_print_mu); */
  va_start (ap, format);
  vcmn_err (CE_NOTE, format, ap);
  va_end (ap);
  /* mutex_exit (&gm_solaris_print_mu); */
/*   drv_usecwait (100000); */
}

void
gm_solaris_panic (char *format, ...)
{
  va_list ap;
/*   mutex_enter (&gm_solaris_print_mu); */
  va_start (ap, format);
  vcmn_err (CE_CONT, format, ap);
  va_end (ap);
  cmn_err (CE_PANIC, "(gm panic)");
/*   mutex_exit (&gm_solaris_print_mu); */
/*   drv_usecwait (100000); */
}


/************************************************************************
 * Module entry points    (called via dev_ops structure)
 ************************************************************************/

static int
gm_solaris_attach (dev_info_t * dip, ddi_attach_cmd_t cmd)
{
  gm_instance_state_t *is;
  int instance;
  enum gm_bus_type bus_type;

  GM_CALLED ();

/*OK HERE*/

  GM_NOTE (("Myrinet board detected\n"));
  /* Verify command. */

  if (cmd != DDI_ATTACH)
    goto abort_with_nothing;

  /* set the page length ASAP, as it is used allover. */

  if (!GM_PAGE_LEN)
    for (GM_PAGE_LEN = 4096; !ddi_btop (dip, GM_PAGE_LEN); GM_PAGE_LEN <<= 1);
  GM_INFO (("GM_PAGE_LEN (page size) is %d\n", GM_PAGE_LEN));

  instance = ddi_get_instance (dip);

  /* Verify that board can be DMA master. */

  if (ddi_slaveonly (dip) == DDI_SUCCESS)
    {
      GM_NOTE (("Myrinet board #%d in slave-only slot cannot be used.",
		instance));
      goto abort_with_nothing;
    }

  /* Allocate the "instance state" for the interface (as Solaris soft state) */

  if (ddi_soft_state_zalloc (gm_instancep, instance) != DDI_SUCCESS)
    {
      GM_NOTE (("myrinet myri[%d]: soft state alloc failed", instance));
      goto abort_with_nothing;
    }
  is = ddi_get_soft_state (gm_instancep, instance);

  /* Add the interrupt hander now that we know what IS is.  This will
     set is->arch.iblock_cookie so that we can properly initialize any
     mutex that depends on it.

     HACK: The interrupt handler will crash the machine if it called
     before the instance state is initialized.  We install it here
     anyway to get the iblock cookie, and then enable interrupts only
     after the instance is initialized. */


  /* Verify that high-level interrupts are supported */

  if (ddi_intr_hilevel (dip, 0) != 0)
    {
      GM_NOTE (("High level interrupts not supported"));
      goto abort_with_instance_state;
    }

  ddi_get_iblock_cookie (dip, 0, &is->arch.iblock_cookie);

  is->id = instance;
  is->arch.dip = dip;
#if GM_SOLARIS_COMPLICATED_SPIN
  gm_arch_sync_init (&is->arch.spin_sync, is);
#endif

#if GM_CAN_REGISTER_MEMORY
  is->arch.page_lock_hash
    = gm_create_hash (gm_solaris_page_lock_compare,
		      gm_solaris_page_lock_hash,
		      sizeof (gm_solaris_page_lock_key_t),
		      sizeof (gm_solaris_page_lock_t), 0, 0);
  if (is->arch.page_lock_hash == 0)
    {
      GM_NOTE (("Could not allocate page lock hash table.\n"));
      goto abort_with_spin_sync;
    }
#endif

  /* Determine the ifc type. */

  do
    {
#if GM_SUPPORT_PCI
      if ((gm_strcmp (ddi_get_name (dip), "pci10e8,8043") == 0)
	  || (gm_strcmp (ddi_get_name (dip), "pci14c1,8043") == 0)
	  || (gm_strcmp (ddi_get_name (dip), "pci103c,106b") == 0)
	  || (gm_strcmp (ddi_get_name (dip), "pci14c1,0") == 0))
	{
	  bus_type = GM_MYRINET_BUS_PCI;
	  break;
	}
#endif /* GM_SUPPORT_PCI */

#if GM_SUPPORT_SBUS
      if (gm_strcmp (ddi_get_name (dip), "MYRICOM,mlanai") == 0)
	{
	  bus_type = GM_MYRINET_BUS_SBUS;
	  break;
	}
#endif

      GM_NOTE (("[%d]: name NOT matched (%s)", instance, ddi_get_name (dip)));
      goto abort_with_page_lock_hash;
    }
  while (0);
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("[%d]: name matched (%s)\n", instance, ddi_get_name (dip)));

#if GM_SUPPORT_PCI
  /* Setup access to PCI configurations space before calling
     gm_instance_init(). */

  if (bus_type == GM_MYRINET_BUS_PCI)
    {
      /* Enable access to PCI configuration space */
      if (pci_config_setup (dip, &is->arch.pci_acc_handle) != DDI_SUCCESS)
	{
	  GM_NOTE (("Could not setup access to PCI config space.\n"));
	  goto abort_with_page_lock_hash;
	}
    }
#endif

  /* Initialize the interface, including mapping the board regions,
     copying eeprom, and starting the LANai control program (with
     interrupts disabled.) */


  if (gm_instance_init (is, instance, bus_type) != GM_SUCCESS)
    {
      GM_NOTE (("[%d]: Could not initialize instance.\n", instance));
      goto abort_with_initialized_config_state;
    }
  GM_INFO (("gm_instance_init instance=%d completed successfully\n",
	     instance));
  gm_arch_solaris_instance_ok[instance]++;

  /*** Set up interrupt handler for the device. ***/

#if GM_SUPPORT_SBUS

  /* Set the SBUS ifc interrupt level */

  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      int int_level;

      int_level = ddi_getprop (DDI_DEV_T_ANY, dip, DDI_PROP_DONTPASS,
			       "interrupts", 0);
      if (int_level == 0)
	int_level = 4;

      if (is->lanai.eeprom.lanai_cpu_version >= 0x400)
	{
	  /* set the board interrupt level from the PROM value */

	  GM_INFO (("[%d]: Setting LANai board[%d] to "
		    "interrupt level = %d  0x%x\n",
		    instance, instance, int_level, 1 << int_level));

#if GM_NO_PARTWORD_PIO_WRITE
#error SBUS card and GM_NO_PARTWORD_PIO_WRITE are incompatible.
#endif
	  ((unsigned short *) is->lanai.control_regs)[1]
	    = GM_N(gm_hton_u16(1 << int_level));

	  GM_PRINT (GM_PRINT_LEVEL >= 1,
		    ("[%d]: LANAI_CONTROL[instance][1] = (short)0x%x\n",
		     instance,
		     ((unsigned short *) is->lanai.control_regs)[1]));

	  GM_PRINT (GM_PRINT_LEVEL >= 1,
		    ("[%d]: Setting LANai board[%d] to interrupt "
		     "level = %d  0x%x AGAIN \n", instance, instance,
		     int_level, 1 << int_level));

#if GM_NO_PARTWORD_PIO_WRITE
#error SBUS card and GM_NO_PARTWORD_PIO_WRITE are incompatible.
#endif
	  ((unsigned short *) is->lanai.control_regs)[1]
	    = GM_N(gm_hton_u16(1 << int_level));

	  GM_INFO (("[%d]: LANAI_CONTROL[instance] = (int)0x%x \n", instance,
		    *(unsigned int *) &((unsigned short *) is->lanai.
					control_regs)[0]));
	}
    }
#endif /* GM_SUPPORT_SBUS */

  /* enable interrupts.  HACK: This must be done only after it is safe
     to call the interrupt handler for this device, which we installed
     earlier. */

  if (ddi_add_intr (dip, 0, &is->arch.iblock_cookie, 0,
		    (u_int (*)(caddr_t)) gm_intr, (void *) is) != GM_SUCCESS)
    {
      GM_NOTE (("cannot add interrupt handler."));
      goto abort_with_initialized_instance;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("Enabling interrupts for instance %d.\n", is->id));
  if (gm_enable_interrupts (is) != GM_SUCCESS)
    {
      GM_NOTE (("Cannot enable interrupts.\n"));
      goto abort_with_initialized_instance;
    }

#if 0
  GM_PRINT (1, ("debug spinning\n"));
  gm_arch_spin (is, 10000000);
  GM_PRINT (1, ("debug pausing\n"));
  gm_pause_lanai (is);
  GM_PRINT (1, ("debug unpausing\n"));
  gm_unpause_lanai (is);
  GM_PRINT (1, ("debug done (un)pausing\n"));
#endif

  /************
   * Create minor devices.  /dev/gm%d is the clone device for normal users,
   * and /dev/gmp%d is the clone device for privileged users.
   ************/

  /* Determine minor node number to use for /dev/gm? . */

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor number.\n"));
  if (gm_minor_alloc (is, &is->clone_minor) != GM_SUCCESS)
    {
      GM_NOTE (("Failed to allocate minor number.\n"));
      goto abort_with_enabled_interrupts;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("using minor number %ld.\n", (long) is->clone_minor));
  gm_minor_get_port_state (is->clone_minor)->instance = is;
  gm_minor_get_port_state (is->clone_minor)->privileged = 0;

  /* Determine minor node number to use for /dev/gmp? . */

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor number.\n"));
  if (gm_minor_alloc (is, &is->privileged_clone_minor) != GM_SUCCESS)
    {
      GM_NOTE (("Failed to allocate minor number.\n"));
      goto abort_with_minor;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("using minor number %ld.\n",
				  (long) is->privileged_clone_minor));
  gm_minor_get_port_state (is->privileged_clone_minor)->instance = is;
  gm_minor_get_port_state (is->privileged_clone_minor)->privileged = 1;

  /* Create the minor node (/dev/gm?) */
  {
    char name[20];

    /* Create /devices/gm* clone device. */

    sprintf (name, "gm%d", instance);
    if (ddi_create_minor_node (dip, name, S_IFCHR, is->clone_minor,
			       DDI_PSEUDO, 0) != DDI_SUCCESS)
      {
	GM_NOTE (("ddi_create_minor_node_failed for unit %d.", instance));
	goto abort_with_minor2;
      }

    /* Create /devices/gmp* privileged clone device. */

    sprintf (name, "gmp%d", instance);
    if (ddi_create_minor_node (dip, name, S_IFCHR, is->privileged_clone_minor,
			       DDI_PSEUDO, 0) != DDI_SUCCESS)
      {
	GM_NOTE (("ddi_create_minor_node_failed for unit %d.", instance));
	goto abort_with_clone_minor_node;
      }

#if GM_BC_DRIVER
#error broken minor number details
    /* Create /devices/mlanai* for backwards-compatibility. */

    sprintf (name, "mlanai%x", instance);
    if (ddi_create_minor_node
	(dip, name, S_IFCHR,
	 GM_INSTANCE_TO_COMPATIBILITY_DEVICE_MINOR (instance),
	 DDI_PSEUDO, 0) != DDI_SUCCESS)
      {
	GM_NOTE (("ddi_create_minor_node_failed for unit %d.", instance));
	goto abort_with_privileged_minor_node;
      }
#endif
  }

  GM_PRINT (GM_PRINT_LEVEL >= 3, ("created minor nodes\n"));

  /* Put pointer to private info into dip. */

  ddi_set_driver_private (dip, (caddr_t) is);

  /* Report attachment */

  ddi_report_dev (dip);

  GM_PRINT (GM_PRINT_LEVEL >= 3, ("Reported attachment.  Done.\n"));

  GM_RETURN_INT (DDI_SUCCESS);

  /********************* Error handling */

abort_with_privileged_clone_minor_node:
abort_with_clone_minor_node:
  ddi_remove_minor_node (dip, NULL);
abort_with_minor2:
  gm_minor_free (is->privileged_clone_minor);
abort_with_minor:
  gm_minor_free (is->clone_minor);
abort_with_enabled_interrupts:
  gm_disable_interrupts (is);
  ddi_remove_intr (dip, 0, is->arch.iblock_cookie);
abort_with_initialized_instance:
  gm_instance_finalize (is);
abort_with_initialized_config_state:
  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    pci_config_teardown (&is->arch.pci_acc_handle);
abort_with_page_lock_hash:
#if GM_CAN_REGISTER_MEMORY
  gm_destroy_hash (is->arch.page_lock_hash);
abort_with_iblock_cookie:
#endif
abort_with_spin_sync:
#if GM_SOLARIS_COMPLICATED_SPIN
  gm_arch_sync_destroy (&is->arch.spin_sync);
#endif
abort_with_instance_state:
  ddi_soft_state_free (gm_instancep, instance);
abort_with_nothing:
  GM_RETURN_INT (DDI_FAILURE);
}

/* Unattach everything in reverse order */

static int
gm_solaris_detach (dev_info_t * dip, ddi_detach_cmd_t cmd)
{
  gm_instance_state_t *is;
  int instance;

  GM_CALLED ();
  if (cmd != DDI_DETACH)
    GM_RETURN_INT (DDI_FAILURE);

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling ddi_get_instance()\n"));
  instance = ddi_get_instance (dip);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called ddi_get_instance()\n"));
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling ddi_get_soft_state()\n"));
  is = ddi_get_soft_state (gm_instancep, instance);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called ddi_get_soft_state()\n"));

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling ddi_remove_minor_node\n"));
  ddi_remove_minor_node (dip, NULL);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called ddi_remove_minor_node\n"));
  gm_minor_free (is->privileged_clone_minor);
  gm_minor_free (is->clone_minor);
  gm_disable_interrupts (is);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling ddi_remove_intr()\n"));
  ddi_remove_intr (dip, 0, is->arch.iblock_cookie);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called ddi_remove_intr()\n"));
  gm_instance_finalize (is);
  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling pci_config_teardown()\n"));
      pci_config_teardown (&is->arch.pci_acc_handle);
      GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called pci_config_teardown()\n"));
    }
#if GM_CAN_REGISTER_MEMORY
  gm_destroy_hash (is->arch.page_lock_hash);
#endif
#if GM_SOLARIS_COMPLICATED_SPIN
  gm_arch_sync_destroy (&is->arch.spin_sync);
#endif
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling ddi_soft_state_free()\n"));
  ddi_soft_state_free (gm_instancep, instance);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called ddi_soft_state_free()\n"));
  gm_arch_solaris_instance_ok[instance] = 0;

  GM_RETURN_INT (DDI_SUCCESS);
}

/* Perform minor device number to device instance mapping for the
   kernel. */

static int
gm_solaris_getinfo (dev_info_t * dip, ddi_info_cmd_t infocmd, void *arg,
		    void **result)
{
  gm_instance_state_t *is;
  gm_port_state_t *ps;
  dev_t dev;

  GM_CALLED ();

  switch (infocmd)
    {
    case DDI_INFO_DEVT2INSTANCE:
      dev = (dev_t) arg;
      ps = gm_minor_get_port_state (getminor (dev));
      if (!ps)
	GM_RETURN (DDI_FAILURE);
      is = ps->instance;
      if (!is)
	GM_RETURN_INT (DDI_FAILURE);
      *result = (void *) is->id;
      GM_RETURN_INT (DDI_SUCCESS);

    case DDI_INFO_DEVT2DEVINFO:
      dev = (dev_t) arg;
      ps = gm_minor_get_port_state (getminor (dev));
      if (!ps)
	GM_RETURN (DDI_FAILURE);
      is = ps->instance;
      if (!is)
	GM_RETURN_INT (DDI_FAILURE);
      *result = (void *) is->arch.dip;
      GM_RETURN_INT (DDI_SUCCESS);
    }
  GM_RETURN_INT (DDI_FAILURE);
}

static int
gm_solaris_localize_status (gm_status_t status)
{
#define CASE(from,to) case from : return to
  switch (status)
    {
      CASE (GM_SUCCESS, 0);
      CASE (GM_INPUT_BUFFER_TOO_SMALL, EFAULT);
      CASE (GM_OUTPUT_BUFFER_TOO_SMALL, EFAULT);
      CASE (GM_TRY_AGAIN, EAGAIN);
      CASE (GM_BUSY, EBUSY);
      CASE (GM_MEMORY_FAULT, EFAULT);
      CASE (GM_INTERRUPTED, EINTR);
      CASE (GM_INVALID_PARAMETER, EINVAL);
      CASE (GM_OUT_OF_MEMORY, ENOMEM);
      CASE (GM_INVALID_COMMAND, EINVAL);
      CASE (GM_PERMISSION_DENIED, EPERM);
      CASE (GM_INTERNAL_ERROR, EPROTO);
      CASE (GM_UNATTACHED, EUNATCH);
      CASE (GM_UNSUPPORTED_DEVICE, ENXIO);
    default:
      return gm_solaris_localize_status (GM_INTERNAL_ERROR);
    }
}

/************
 * Character device driver entry points.
 ************/

/* open a device.  For GM, this is a clone device, and we don't
   do much initialization here, other than to allocate a new minor number
   and uninitialized gm_port_state.  Later, a GM_SET_PORT_NUMBER ioctl
   will allow the gm_port_state to be fully initialized. */

static int
gm_solaris_open (dev_t * devp, int flags, int otyp, cred_t * credp)
{
  gm_status_t status;
  minor_t minor;
  dev_t dev;
  gm_port_state_t *ps, *clone_ps;

  GM_CALLED ();

  if (otyp != OTYP_CHR)
    {
      GM_NOTE (("gm_solaris_open() called with bad otyp.\n"));
      GM_RETURN_INT (GM_INVALID_PARAMETER);
    }

  /* Determine the port (device) to clone. */

  clone_ps = gm_minor_get_port_state (getminor (*devp));

  /* Determine an unused minor number for the opened device. */

  if (gm_minor_alloc (clone_ps->instance, &minor) != GM_SUCCESS)
    {
      GM_NOTE (("Could not create new minor number.\n"));
      status = GM_TRY_AGAIN;
      goto abort_with_nothing;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Using minor number %d.\n", minor));
  ps = gm_minor_get_port_state (minor);
  gm_assert (ps);

  /* Record the instance associated with this minor number. */

  ps->instance = clone_ps->instance;
  gm_assert (ps->instance);

  /* Record the privileges from the clone device */

  ps->privileged = clone_ps->privileged;

  /* Signal OS to make new device with the same major number and a
     new minor number. */

  dev = ps->instance->arch.dev = makedevice (getmajor (*devp), minor);
  if (dev == NODEV)
    {
      GM_NOTE (("Could not make device number.\n"));
      status = GM_OUT_OF_MEMORY;
      goto abort_with_minor;
    }
  *devp = dev;

  /* success, so commit */

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("User opened device with minor number 0x%x.\n", getminor (dev)));
  GM_RETURN_INT (0);

abort_with_minor:
  gm_minor_free (minor);
abort_with_nothing:
  GM_RETURN_INT (gm_solaris_localize_status (status));
}

static int
gm_solaris_close (dev_t dev, int flags, int otyp, cred_t * credp)
{
  gm_port_state_t *ps;

  GM_CALLED ();

  /* Get a pointer to the port state for the port being closed. */

  ps = gm_minor_get_port_state (getminor (dev));
  gm_assert (ps);
  if (ps->opened)
    gm_port_state_close (ps);
  gm_minor_free (getminor (dev));

  GM_RETURN_INT (0);
}

/* Map interface memory into the process.  This can be used to map the
   LANai control register, special registers, SRAM, or copy block into
   the user space.  The offsets and sizes of these various regions can
   be obtained using the GM_GET_MAPPING_SPECS ioctl, which returns
   a description of the mappings in a "gm_mapping_specs"
   structure.

   The pages of the copy block must be mapped in order and without
   skipping any offsets.

   gm_segmap ensures that all offsets passed to gm_mmap are
   legitimate, so there is no need to range check offsets here.
   However, checks must be performed here for "promiscuous" mappings
   of the device registers an SRAM areas other than the "send queue"
   and "activation" pages. */

static int
gm_solaris_mmap (dev_t dev, off_t off, int prot)
{
  void *result;
  gm_port_state_t *ps;
  int ret;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("Mapping offset 0x%x for minor device %d\n", off,
	     getminor (dev)));

  ps = gm_minor_get_port_state (getminor (dev));
  gm_assert (ps);

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling gm_mmap().\n"));
  if (gm_mmap (ps, off, &result) != GM_SUCCESS)
    goto abort_with_nothing;
  gm_assert (result);

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("calling hat_getkpfnum(%p).\n", result));
  ret = hat_getkpfnum ((caddr_t) result);

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("returning 0x%x.\n", ret));
  GM_RETURN_INT (ret);

abort_with_nothing:
  GM_RETURN_INT (-1);
}

/* Simply call gm_ioctl. */

static int
#if GM_OS_SOLARIS7
gm_solaris_ioctl (dev_t dev, int cmd, intptr_t argp,
		  int mode, cred_t * cred_p, int *rval_p)
#else
gm_solaris_ioctl (dev_t dev, int cmd, int arg, int mode, cred_t * cred_p,
		  int *rval_p)
#endif
{
  gm_port_state_t *ps;
  gm_status_t gm_status;
#if GM_OS_SOLARIS7
  int *arg = (int *) argp;
#endif

  /* Make sure ioctl is not being called by the kernel, since we cannot
     handle that yet. */
  ASSERT (!(mode & FKIOCTL));

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("gm_solaris_ioctl() called for dev with minor number 0x%x.\n",
	     getminor (dev)));

  ps = gm_minor_get_port_state (getminor (dev));
  gm_assert (ps);
  gm_arch_mutex_enter (&ps->sync);

  /* Call the portable IOCTL. */
  ps->arch.ddi_copyxx_context = mode;
  gm_status
    = gm_ioctl (ps, cmd, (void *) arg, INT_MAX, (void *) arg, INT_MAX, 0);
  gm_arch_mutex_exit (&ps->sync);
  return (gm_solaris_localize_status (gm_status));
}

/* Perform segment mapping.

   We basically use the default ddi_segmap routine, but dynamically
   allocate copy block memory by calling gm_prepare_for_mmap before
   calling ddi_segmap so gm_mmap( ) (which is called by
   ddi_segmap( ) via gm_mmap( )) can simply assume assume the copy
   block is fully allocated.

   Also, after calling ddi_segmap successfully, the address of the
   mappings is passed to gm_finish_mmap( ), which uses this info to
   add the mapped pages to the page hash table for the device, which
   records the mappings between user virtual addresses and dma
   addresses.  This must be done here because this is the only place
   the user virtual addresses are available.

   Additionally, this segmap( ) implementation ensures that all mapping
   requests are properly page-aligned, and that mappings of the copy
   block memory are performed in order without any gaps. */

static int
gm_solaris_segmap (dev_t dev,
		   off_t off,
		   struct as *asp,
		   caddr_t * addrp,
		   off_t len,
		   unsigned int prot,
		   unsigned int maxprot, unsigned int flags, cred_t * cred_p)
{
  gm_port_state_t *ps;
  int error;
  gm_status_t gm_status;

  GM_CALLED ();

  ps = gm_minor_get_port_state (getminor (dev));
  gm_assert (ps);
  gm_arch_mutex_enter (&ps->sync);

  /* Verify the mapping is legal and ensure all resources are
     available, allocating them if needed. */

  gm_status = gm_prepare_to_mmap (ps, off, len,
				  ((prot & PROT_WRITE ? GM_MAP_WRITE : 0)
				   | (prot & PROT_READ ? GM_MAP_READ : 0)));
  if (gm_status != GM_SUCCESS)
    {
      error = gm_solaris_localize_status (gm_status);
      GM_NOTE (("Error preparing to mmap.\n"));
      goto abort_with_mutex;
    }

  /* Perform the mapping using the default segmap implementation,
     which calls mmap to map the individual pages. */

  error =
    ddi_segmap (dev, off, asp, addrp, len, prot, maxprot, flags, cred_p);
  if (error != DDI_SUCCESS)
    {
      GM_WARN (("Unanticipated memory mapping error %d", error));
      goto abort_with_mutex;
    }
  ASSERT (GM_PAGE_ALIGNED (*addrp));

  /* Now that the pages have been mapped, allow GM to know where
     everything was mapped. */

  gm_status = gm_finish_mmap (ps, off, len, (gm_up_t) * addrp);
  if (gm_status != GM_SUCCESS)
    {
      error = gm_solaris_localize_status (gm_status);
      GM_WARN (("Unexpected error finishing mmap.\n"));
      goto abort_with_mutex;
    }

  gm_arch_mutex_exit (&ps->sync);
  GM_RETURN_INT (GM_SUCCESS);

abort_with_mutex:
  gm_arch_mutex_exit (&ps->sync);
  GM_RETURN_INT (error);
}

#if GM_BC_DRIVER
#error broken minor number details

/* Implement gm_read so we can return EEPROM when emulating mlanai
   driver. */

static int
gm_solaris_read (dev_t dev, struct uio *uio, cred_t * credp)
{
  gm_instance_state_t *is;

  GM_CALLED ();

  if (!GM_COMPATIBILITY_DEVICE_MINOR_P (getminor (dev)))
    return EACCES;

  is = ddi_get_soft_state (gm_instancep,
			   GM_MINOR_TO_INSTANCE (getminor (dev)));

  GM_RETURN_INT (uiomove ((caddr_t) & is->lanai.eeprom,
			  min (sizeof (gm_myrinet_eeprom_t),
			       uio->uio_iov->iov_len), UIO_READ, uio));
}
#endif

static int
gm_solaris_identify (dev_info_t * dip)
{
  char *name;

  GM_CALLED ();

  name = ddi_get_name (dip);

  GM_PRINT (GM_PRINT_LEVEL >= 3, ("identifying name \"%s\"\n", name));

  if (gm_strcmp (name, "MYRICOM,mlanai") == 0
      || gm_strcmp (name, "pci10e8,8043") == 0
      || gm_strcmp (name, "pci14c1,8043") == 0
      || gm_strcmp (name, "pci14c1,0") == 0)
    GM_RETURN_INT (DDI_IDENTIFIED);
  else
    GM_RETURN_INT (DDI_NOT_IDENTIFIED);
}

#if GM_CAN_REGISTER_MEMORY && GM_REGISTER_BY_WRITE
static int
gm_solaris_strategy (struct buf *b)
{
  gm_solaris_page_lock_key_t key;
  gm_solaris_page_lock_t lock;

  GM_CALLED ();
  gm_solaris_print_buf (b);

  biodone (b);
  GM_RETURN_INT (0);
}

static int
gm_solaris_aread (dev_t dev, struct aio_req *aio, cret_t * cred)
{
  gm_port_state_t *ps;
  gm_instance_state_t *is;

  ps = gm_minor_get_port_state (getminor (dev));
  gm_assert (ps);
  if (!ps->opened)
    return (ENXIO);
  is = ps->instance;
  gm_arch_mutex_enter (&is->arch.page_lock.sync);
  ps->arch.page_lock_key_template.port = ps->id;
#error
  ps->arch.page_lock_
    error =
    aphysio (gm_solaris_strategy, anocancel, dev, B_READ, minphys, aio);
  gm_arch_mutex_exit (&is->arch.page_lock.sync);
  return error;
}

static int
gm_solaris_awrite (dev_t dev, struct aio_req *aio_req, cret_t * cred)
{
  ps = gm_minor_get_port_state (getminor (dev));
  gm_assert (ps);
  if (!ps->opened)
    return (ENXIO);

  return aphysio (gm_solaris_strategy, anocancel, dev, B_WRITE, minphys, aio);
}
#endif

/************************************************************************
 * Dynamic loading entry points.
 ************************************************************************/

int
_init (void)
{
  int error;

  mutex_init (&gm_solaris_print_mu, "GM print mu", MUTEX_DRIVER, NULL);

  GM_CALLED ();

  /* preallocate state for 1 Myrinet card */

  error
    = ddi_soft_state_init (&gm_instancep, sizeof (gm_instance_state_t), 1);
  if (error != 0)
    goto abort_with_nothing;

  error = mod_install (&modlinkage);
  if (error != 0)
    goto abort_with_device_state;

  GM_RETURN_INT (DDI_SUCCESS);

abort_with_device_state:
  ddi_soft_state_fini (&gm_instancep);
abort_with_nothing:
  GM_NOTE (("gm device driver initialization failed."));
  GM_RETURN_INT (error);
}

int
_fini (void)
{
  int error;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling mod_remove()\n"));
  error = mod_remove (&modlinkage);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called mod_remove()\n"));
  if (error)
    {
      GM_NOTE (("error calling mod_remove.\n"));
      GM_RETURN_INT (error);
    }
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Calling ddi_soft_state_fini()\n"));
  gm_assert (gm_instancep);
  ddi_soft_state_fini (&gm_instancep);
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called ddi_soft_state_fini()\n"));

  /* Don't use "RETURN" macro, since it might print something after the
     print mutex has been destroyed. */
#ifdef __GNUC__
  GM_PRINT (GM_PRINT_LEVEL >= 5, (__FUNCTION__ "() returning\n"));
#else
  GM_PRINT (GM_PRINT_LEVEL >= 5, ("_fini  returning\n"));
#endif

  mutex_destroy (&gm_solaris_print_mu);
  return 0;
}

int
_info (struct modinfo *modinfop)
{
  GM_CALLED ();
  GM_RETURN_INT (mod_info (&modlinkage, modinfop));
}



/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
