/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

#include    <sys/types.h>
#include    <sys/errno.h>
#include    <sys/debug.h>
#include    <sys/stropts.h> 
#include    <sys/stream.h>
#include    <sys/strlog.h>
#include    <sys/cmn_err.h>
#include    <sys/kmem.h>
#include    <sys/conf.h> 
#include    <sys/ksynch.h>
#include    <sys/stat.h>
#include    <sys/dlpi.h>
#include    <sys/modctl.h>
#ifdef  KSTAT
#include        <sys/kstat.h>
#endif
#include    <sys/time.h>
#include    <sys/ddi.h>
#include    <sys/sunddi.h>

#include "gm_internal.h"
#include "gm_ether.h"
#include "myri.h"

static char *myri_compile_flags = MYRI_CFLAGS;

/* this needs to be defined for host_ifc.c */
extern struct myri *myridev;
extern int myridebug;

extern struct timeval time_start[32], time_stop[32];

#ifdef sparc_solaris
void
usleep(unsigned usec)
{
  unsigned i;
  for (i = 0; i < usec; i++) {
  }
}
#endif

/* increment (x) by (inc) and modulo (max) */
gm_inline unsigned
MODINC(int x, int inc, int max)
{

  register int tmp;
  register int j;

  tmp = x + inc;
  if ((j = (tmp - max)) >= 0) {
    return j;
  }
  return (tmp);
}

gm_inline int gm_get_receive_unused(struct myri *myrip)
{
  return (NUM_RECV_BUFF1 - myrip->recv_buffer_count);
}



/***********************  myri_tolanai()  *****************************/

int
myri_tolanai(queue_t * wq, mblk_t * mp, struct myri *myrip)
{
  int rv = 1;
  gm_u8_t *myri_addr = (gm_u8_t *)mp->b_rptr;

  CMN_ERR(3,  (CE_CONT, "        myri_tolanai() called\n"));

  MUTEX_ENTER(&myrip->myri_xmitlock);
  myrip->xmit_count++;

#ifdef MYRI_DEBUG
  ASSERT(myrip->xmit_count == 1);
#endif

  /* send the one message pointed to by mp */

  /* send using DMA */
  {
    register int index;
    register gm_u32_t len;

    /* start with the next one */
    index = CYCLE(myrip->last_send_index, NUM_SEND_BUFF);

    if (myrip->sendinfo[index].status == BUF_UNUSED) {
      /* found a buffer */

      myrip->sendinfo[index].status = BUF_LANAI;
      myrip->last_send_index = index;
      myrip->send_buffer_count++;
	    
      /* copy the chain to the buffer */
      len = copy_mblk_to_buffer((char *) mp,
				(char *) myrip->sendinfo[index].kptr);

      CMN_ERR(3, (CE_CONT, "tolanai: packet before handing to LANai\n"));
       
      /* gm_hex_dump((char *) myrip->sendinfo[index].kptr, 16); */ 
      
      /* hand it to the LANai */
	    
      {			    
	register gm_ethernet_segment_descriptor_t *gather;

	gather = &myrip->sendinfo[index].ether_pkt;
	gather->len = gm_hton_u32(len);
		
	if (ddi_dma_sync(myrip->sendinfo[index].handle, 0,
			 len, DDI_DMA_SYNC_FORDEV) != DDI_SUCCESS) {
	  CMN_ERR_PRINT (0, (CE_CONT, "DDI_DMA_SYNC_FORDEV FAILED index = %d len = %d\n",
			     index, len));
	}

	CMN_ERR(0,(CE_CONT,"ethernet_send idx=%d ptr=0x%lx len=%d\n",
		index, gather->ptr,gm_hton_u32(gather->len)));

	if (myri_addr[0] != 0xff) {
	  gm_ethernet_send(myrip->gm_port, 1, gather, myri_addr, 0);
	} else {		  
	  gm_ethernet_broadcast(myrip->gm_port, 1, gather, 0);
	}
	
	myrip->myri_opackets++;

	/* free chain */
	freemsg(mp);
	mp = NULL;

	myrip->xmit_count--;
	MUTEX_EXIT(&myrip->myri_xmitlock);

	/* check for stuff that got blocked */
	if (myrip->myri_wantw)
	    myriwenable(myrip);
	return (rv);

      }					/* hand to lanai */
    } else {
      /* no send buffers free! */
      CMN_ERR_PRINT (0, (CE_CONT, "myri_tolanai: no send buffers free!\n"));
      myrip->myri_oerrors++;
      /* free chain */
      freemsg(mp);
      mp = NULL;

      myrip->xmit_count--;
      MUTEX_EXIT(&myrip->myri_xmitlock);
	    
      return (0);
    }
  }							/* send using DMA */

send_error:
  myrip->myri_oerrors++;
  if (mp) {
    freemsg(mp);
  }
  mp = NULL;

  myrip->xmit_count--;
  MUTEX_EXIT(&myrip->myri_xmitlock);

  return (0);
}


/******************   myri_mutex_enter()   *********************/
/*
 * Grab all the semaphores.
 *
 */
gm_inline void  
myri_mutex_enter(struct myri *myrip)
{
  MUTEX_ENTER(0);
  MUTEX_ENTER(&myrip->myri_xmitlock);
  ++myrip->xmit_count;
#ifdef MYRI_DEBUG
  ASSERT(myrip->xmit_count == 1);
#endif
  MUTEX_ENTER(&myrip->myri_recvlock);
  ++myrip->recv_count;
#ifdef MYRI_DEBUG
  ASSERT(myrip->recv_count == 1);
#endif
  /* MUTEX_ENTER(&myrip->myri_bufflock); */
}


/******************   myri_mutex_exit()   *********************/
/*
 * Release all the semaphores.
 *
 */
gm_inline void
myri_mutex_exit(struct myri *myrip)
{
  /* MUTEX_EXIT(&myrip->myri_bufflock); */
  myrip->recv_count--;
  MUTEX_EXIT(&myrip->myri_recvlock);
  myrip->xmit_count--;
  MUTEX_EXIT(&myrip->myri_xmitlock);
  MUTEX_EXIT(0);
}

/***********************  myri_dma_error()   *****************************/
static char *dma_error_names[16] = {"DDI_DMA_MAPPED",
				    "DDI_DMA_NORESOURCES",
				    "DDI_DMA_NOMAPPING",
				    "DDI_DMA_TOOBIG",
				    "DDI_DMA_TOOSMALL",
				    "DDI_DMA_LOCKED",
				    "DDI_DMA_BADLIMITS",
				    "DDI_DMA_STALE",
				    "DDI_DMA_BADATTR",
				    "DDI_DMA_INUSE"};

void
myri_dma_error(int rv)
{
  if ((rv<=0) && (rv>=-15)){
    rv = -rv;
    cmn_err(CE_CONT, "dma_setup %s (%d)\n",
	    dma_error_names[rv],-rv);
  }
  else {
    cmn_err(CE_CONT, "dma_setup rv=%d not understood!\n", rv);
  }
}
  
/*****************   myri_handle_lanai_intr()  ************/
/*
 * Called from myriintr.
 *
 */

void
myri_recv_intr(void *context, unsigned int len, gm_u16_t csum)
{
  struct myri *myrip = (struct myri *) context;
  mblk_t *mp;

  MUTEX_ENTER(&myrip->myri_recvlock);
  myrip->recv_count++;

#ifdef MYRI_DEBUG
  ASSERT(myrip->recv_count == 1);
#endif
  mp = myri_do_recv(myrip, len);
  
  myrip->recv_count--;
  MUTEX_EXIT(&myrip->myri_recvlock);
  
#ifdef LATER
  if (had_to_copy || gm_get_receive_unused(myrip) <= ( NUM_RECV_BUFF1 / 4)) {
    /* now try to find more bufs */
    struct buff_info binfo;
    
    binfo.myrip = myrip;
    binfo.index = -1;
    binfo.queue = RECV1_Q;
    myri_add_buff(&binfo);
    
    had_to_copy = 0;
  }
#endif

#ifdef IPQ
  {
    queue_t *ipq = myrip->myri_ipq;
    struct medium_header *ehp;
    
    ehp = (struct medium_header *)mp->b_rptr;
    
    if ((get_ether_type(ehp) == ETHERTYPE_IP) &&
	((ehp->med_dhost.medaddr[0] & 01) == 0) &&
	(ipq) &&
	canput(ipq->q_next)) {
      CMN_ERR (3, (CE_CONT, "receive fast path\n"));
      mp->b_rptr += sizeof (struct medium_header);
      (void) putnext(ipq, mp);
    } else
      myrisendup(myrip, mp);
  }						       
#else
  myrisendup(myrip, mp);
#endif
  CMN_ERR(10, (CE_CONT, "myriINTERRUPT exiting\n"));
}

void
myri_sent_intr(void *context)
{
    struct myri *myrip = (struct myri *) context;

    CMN_ERR(10, (CE_CONT, "myri_recv_intr called \n"));
    CMN_ERR(3, (CE_CONT, "SEND_EVENT \n"));
    MUTEX_ENTER(&myrip->myri_xmitlock);
    myrip->sendinfo[myrip->head_send_index].status = BUF_UNUSED;
    myrip->head_send_index = CYCLE(myrip->head_send_index, NUM_SEND_BUFF);
    myrip->send_buffer_count--;
    MUTEX_EXIT(&myrip->myri_xmitlock);
#ifdef SET_WANTW
    /* check for pending send messages */
    if (myrip->myri_wantw) {
      myriwenable(myrip);
    }
#endif	/* SET_WANTW */
}

/***************** myri_do_recv()  **********************/
/*
 * Check the recv_ack queue.
 * called while already holding the recv_lock mutex
 * ASSERT(mutex_owned(&myrip->recvlock));
 *
 */

mblk_t *
myri_do_recv(struct myri * myrip, unsigned int length)
{
  mblk_t *m = NULL;
  int index;
  struct buff_info *infop;

#ifdef MYRI_DEBUG
  ASSERT(mutex_owned(&myrip->myri_recvlock));  
#endif

  CMN_ERR (3, (CE_CONT, "recv'ed message len %d\n", length));

  if (length > 0) {
    index = myrip->head_recv_index ;
    infop = (struct buff_info *)myrip->recv_buf_index[index].infop;
    myrip->head_recv_index = CYCLE(index, NUM_RECV_BUFF1);
    myrip->recv_buffer_count--;

#ifdef MYRI_DEBUG    
    ASSERT (index >= 0);
    ASSERT (index < NUM_RECV_BUFF1);
#endif

    if (infop->status == BUF_LANAI) {
      /* FIX - SYNC?? */
      if (ddi_dma_sync(infop->handle, 0, length, 
		       DDI_DMA_SYNC_FORKERNEL) != DDI_SUCCESS) {
	CMN_ERR_PRINT(0, (CE_CONT, "DDI_DMA_SYNC_FORKERNEL failed index = %d len = %d\n",
			  infop->index, length));
      }
      
      /* loan it out if possible */
    
    
      if ((length < 128) || 
	  (gm_get_receive_unused(myrip) <= MYRI_LBUF_MIN)) {
      
	/*
	 * not enough buffers at the LANai, will copy
	 * msg
	 */
	m = allocb(length, BPRI_HI);
	if (m) {	  
	  bcopy((void *) infop->kptr, (void *) m->b_rptr, length);
	  m->b_wptr = (u_char *) ((u_char *) m->b_rptr + length);   
	  /*  m->b_datap->db_type = M_DATA; */
	  
	  infop->status = BUF_UNUSED;
	  myri_add_buff(infop);
	} else {
	  CMN_ERR_PRINT(0, (CE_CONT, "do_ecv: failed allocb\n"));
	  infop->status = BUF_UNUSED;
	  myri_add_buff(infop);
	  goto recv_error;
	}
      } else { /* Loan buffer */
	/*
	 * lanai has enough buffers, will loan this
	 * one to higher layers
	 */
	
	
	CMN_ERR (BUFF_PRINT, (CE_CONT, "myri: NO copy (%d>%d)\n",
			     gm_get_receive_unused(myrip),
			     MYRI_LBUF_MIN));

	
	/* this avoids a copy */
	if ((m = esballoc((unsigned char *) infop->kptr, length,
			  BPRI_HI,
			  &infop->free_func))) {

	  m->b_wptr = (u_char *) ((u_char *) m->b_rptr + length);
	  /* m->b_datap->db_type = M_DATA; */
	  infop->status = BUF_LOANED;
	} else {
	  CMN_ERR_PRINT(0, (CE_CONT, "do_recv: failed esballoc\n"));
	  infop->status = BUF_UNUSED;
	  myri_add_buff(infop);
	  goto recv_error;
	}
      }					/* else lanai count OK */
    } else {
      CMN_ERR_PRINT (0, (CE_CONT, "do_recv: bad INDEX  index = %d \n", index));

#if (BUFF_PRINT<=GM_PRINT_LEVEL)
      myri_dump_buff_info(&myrip->recv1info[0]);
#endif							/* BUFF_PRINT */
      goto recv_error;
    }
    
    m->b_cont = NULL;

  } else if ((length == 0) || (!m)) {
    return (NULL);
  }
  
  myrip->myri_ipackets++;

  /* remove the initial padding */
  {
    u_char id_byte1= *(m->b_rptr);
    u_char id_byte2= *(m->b_rptr+1);
    if (id_byte1 == 0x0 && id_byte2 == 0x20) {
      m->b_rptr += 2;
    } else {
      CMN_ERR_PRINT (0, (CE_CONT, "Bad myri_pad value = 0x%x 0x%x\n",
			id_byte1, id_byte2));
      freemsg(m);
      m = NULL;
      return (NULL);
    }
  }
  /* m->b_rptr should now be pointing at the ether header */
  
  return (m);


recv_error:
  myrip->myri_ierrors++;
  
  if (m) {
    freemsg(m);
  }

  CMN_ERR (1, (CE_CONT, "Got to recv_error.\n"));
  return (NULL);
}



void
myri_dump_msg(mblk_t * mp)
{
  if (mp) {
    int len = 0, cnt = 0;
    mblk_t *mp_tmp;
    
    len = msgdsize(mp);

    CMN_ERR(0, (CE_CONT, "   MSG: len = %d bytes\n", len));
    
    mp_tmp = mp;
    while (mp_tmp) {
      CMN_ERR(0, (CE_CONT, "  %d: rptr = 0x%x  len = %d   type = %d\n",
		  cnt++, mp_tmp->b_rptr,
		  (mp_tmp->b_wptr - mp_tmp->b_rptr),
		  mp_tmp->b_datap->db_type));
      if (GM_PRINT_LEVEL >= 0)
	gm_hex_dump(mp_tmp->b_rptr, (mp_tmp->b_wptr - mp_tmp->b_rptr));
      mp_tmp = mp_tmp->b_cont;
    }						/* while */
  }
}



/******************  copy_mblk_to_buffer()  ***************************/
/*
 * Note: length is the exact length of the packet after padding with zeros
 *       The true length of the packet before padding comes from the mblk itself
 *       as we copy it.
 *
 */
gm_u32_t
copy_mblk_to_buffer(char *vptr, 
		    char *lptr2)
{
  mblk_t *m = (mblk_t *) vptr;
  unsigned char *ptr;
  unsigned short *lptr = (unsigned short *) lptr2;
  int buf_len;
  int add;
  gm_u32_t len = 2;

  /* put padding on the front */

  lptr[0] = __gm_hton_u16(GM_ETHERNET_PACKET_TYPE); 
  ptr = (unsigned char *) &lptr[1];

  while (m) {
    buf_len = (m->b_wptr - m->b_rptr);
    bcopy( (void *)m->b_rptr, (void *)ptr, buf_len);
    ptr += buf_len;
    len += buf_len;
    /* 
     * gm_hex_dump((caddr_t) m->b_rptr, buf_len); 
     */
    m = m->b_cont;
  }

#ifdef CLASSIC  
  while (!GM_DMA_ALIGNED(ptr)) {
    /*
     * odd length message - increase length and zero out the
     * bytes.  Since we used a send buffer we know we have space
     * at the end.
     */

    *ptr++ = 0;
    len++;
  }                   
#else
       
  add = ((unsigned) ptr) & 0x7;
      
  switch (add) {
  case 1:
    *ptr++ = 0;
  case 2:
    *ptr++ = 0;
  case 3:
    *ptr++ = 0;
  case 4:
    *ptr++ = 0;
  case 5:
    *ptr++ = 0;
  case 6:
    *ptr++ = 0;
  case 7:
    *ptr = 0;
  default:
    break;
  }
#endif
  len = GM_DMA_ROUNDUP(u32, len);
  
#ifdef DO_CKSUM
  if (!verify_cksum((unsigned char *) &lptr[0]), m) {
    CMN_ERR(0, (CE_CONT, "TCP checksum not verified! len = %d\n", length));
  }
#endif							/* DO_CKSUM */

  return len;
}
  
/* end of myri.c */







