/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

static char *myricopyright = "\tCopyright (c) 1999 by Myricom, Inc.";

/*
 *  SunOS Multithreaded STREAMS DLPI Driver
 *
 * This file is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify this file without charge, but are not authorized to
 * license or distribute it to anyone else except as part of a product
 * or program developed by the user.
 *
 * THIS FILE IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * This file is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY THIS FILE
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even
 * if Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */


/*
 * Includes, Declarations and Local Data
 */

/* this is here to avoid gcc-2.8.1 and up problems with varargs */
#ifdef __GNUC__
#define _SYS_VARARGS_H
#include <stdarg.h>
#endif

#include	<sys/types.h>
#include	<sys/errno.h>
#include	<sys/debug.h>
#include	<sys/stropts.h>
#include	<sys/stream.h>
#include	<sys/strlog.h>
#include	<sys/cmn_err.h>

#ifndef __GNUC__
#include <sys/varargs.h>
#endif

#include	<sys/kmem.h>
#include	<sys/conf.h>
#include	<sys/ksynch.h>
#include	<sys/stat.h>
#include	<sys/dlpi.h>
#include 	<sys/modctl.h>
#ifdef  KSTAT
#include        <sys/kstat.h>
#endif
#include    <sys/time.h>
/*
   #include <string.h>
 */

#include 	<sys/pci.h>
#include	<sys/ddi.h>
#include	<sys/sunddi.h>

#include "gm_internal.h"
#include "gm_ether.h"
#include "myri.h"

/* needed for linking to GM functions */
char _depends_on[] = "drv/gm";

#define MYRIVERSION	"Myrinet GM DLPI Driver"

static void pserror(queue_t *, mblk_t *, int);
static void psiocack(queue_t *, mblk_t *, int, int);
static void psiocnak(queue_t *, mblk_t *, int, int);
static mblk_t *psexchange(queue_t *, mblk_t *, int, int, long);
static void psdlbindack(queue_t *, mblk_t *, u_long, caddr_t, int, int, u_long);
static void psdlokack(queue_t *, mblk_t *, u_long);
static void psdlerrorack(queue_t *, mblk_t *, u_long, u_long, u_long);
static void psdluderrorind(queue_t *, mblk_t *, u_char *, u_long, u_long, u_long);
static void psdlphysaddrack(queue_t *, mblk_t *, caddr_t, int);          

static int myriidentify(dev_info_t *);
static int myriprobe(dev_info_t *);
static int myriattach(dev_info_t *, ddi_attach_cmd_t);
static int myridetach(dev_info_t *, ddi_detach_cmd_t);
static int myriinfo(dev_info_t *, ddi_info_cmd_t, void *, void **);
static int myriopen(queue_t *, dev_t *, int, int, cred_t *);
static int myriclose(queue_t *, int, cred_t *);
static int myriwput(queue_t *, mblk_t *);
static int myriwsrv(queue_t *);
#ifdef FLOW_CONTROL
static int myrirsrv(queue_t *);
#endif							/* FLOW_CONTROL */
static void myriproto(queue_t *, mblk_t *);
static void myriioctl(queue_t *, mblk_t *);
static void myriareq(queue_t *, mblk_t *);
static void myridreq(queue_t *, mblk_t *);
static void myridodetach(struct myristr *);
static void myribreq(queue_t *, mblk_t *);
static void myriubreq(queue_t *, mblk_t *);
static void myriireq(queue_t *, mblk_t *);
static void myriponreq(queue_t *, mblk_t *);
static void myripoffreq(queue_t *, mblk_t *);
static void myriemreq(queue_t *, mblk_t *);
static void myridmreq(queue_t *, mblk_t *);
static void myripareq(queue_t *, mblk_t *);
static void myrispareq(queue_t *, mblk_t *);
static void myriudreq(queue_t *, mblk_t *);
extern void myri_recv_intr(void *context, unsigned int len, gm_u16_t csum);
extern void myri_sent_intr(void *context);
static void myrisetipq(struct myri *);
static void myri_dl_ioc_hdr_info(queue_t *wq, mblk_t *mp);
/*
   static struct myristr *
   myripaccept(struct myristr *, struct myri *, int,
   struct medium_addr *);
   static void myrireclaim(struct myri *);
   static void myriread(struct myri *, caddr_t);
 */

static int myrimcmatch(struct myristr *, struct medium_addr *);
static int myriinit(struct myri *);
static void myriuninit(struct myri *myrip);
void myrierror(dev_info_t *, const char *fmt,...);
mblk_t * myriaddudind(struct myri *, mblk_t *, struct medium_addr *,
		      struct medium_addr *, int, ulong);

struct timeval time_start[32], time_stop[32];


gm_inline void
bcopy_addr(caddr_t src, caddr_t dst)
{
  register u_char *srcp = (u_char *) (src);
  register u_char *dstp = (u_char *) (dst);
  
  *dstp++ = *srcp++;
  *dstp++ = *srcp++;
  *dstp++ = *srcp++;
  *dstp++ = *srcp++;
  *dstp++ = *srcp++;
  *dstp++ = *srcp++;
}

gm_inline int
bcmp_addr(caddr_t src, caddr_t dst)
{
  register u_char *srcp = (u_char *) (src);
  register u_char *dstp = (u_char *) (dst);

  if ((*dstp++ == *srcp++) &&
      (*dstp++ == *srcp++) &&
      (*dstp++ == *srcp++) &&
      (*dstp++ == *srcp++) &&
      (*dstp++ == *srcp++) &&
      (*dstp++ == *srcp++)) {
    return (0);
  } else {
    return (1);
  }
}



/* Local Static def's */

/*
 * Physical Medium broadcast address definition.
 */
static struct medium_addr medbroadcastaddr =
{
    {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff
    }
};

/*
 * Linked list of "myri" structures - one per device.
 */
struct myri *myridev = NULL;

/*
 * Linked list of active (inuse) driver Streams.
 */
static struct myristr *myristrup = NULL;
static krwlock_t myristruplock;


/*
 * Our DL_INFO_ACK template.
 */
static dl_info_ack_t myriinfoack =
{
    DL_INFO_ACK,				/* dl_primitive */
    MEDIUM_MTU,					/* dl_max_sdu */
    0,						/* dl_min_sdu */
    DEVICE_ADDRL,				/* dl_addr_length */
    DL_ETHER,					/* dl_mac_type */
    0,						/* dl_reserved */
    0,						/* dl_current_state */
    -2,						/* dl_sap_length */
    DL_CLDLS,					/* dl_service_mode */
    0,						/* dl_qos_length */
    0,						/* dl_qos_offset */
    0,						/* dl_range_length */
    0,	   				/* dl_range_offset */
    DL_STYLE2,					/* dl_provider_style */
    sizeof(dl_info_ack_t),		/* dl_addr_offset */
    DL_VERSION_2,				/* dl_version */
    MEDIUM_ADDRL,				/* dl_brdcst_addr_length */
    sizeof(dl_info_ack_t) + DEVICE_ADDRL,	/* dl_brdcst_addr_offset */
    0					       /* dl_growth */
};

/*
 * Allocate and zero-out "number" structures
 * each of type "structure" in kernel memory.
 */
#define	GETSTRUCT(structure, number) ((structure *) kmem_zalloc(\
                                                 (u_int) (sizeof (structure) * (number)), KM_NOSLEEP))

#define	MYRISAPMATCH(sap, type, flags) ((sap == type)? 1 : \
					((flags & SLALLSAP)? 1 : 0))

/*
 * Defining DEBUG on the compile line (-DDEBUG) will enable debugging
 * statements in this driver.
 * Error message control:
 * 0 = no messages.
 * 1 = Error messages.
 * Can set the value of the debug flag using adb or in /etc/system file
 * with "set myri:myridebug=<value>
 */
     
int myridebug = 1;
void *myri_state;				/* opaque handle for state structs */

/* Standard Streams declarations */

static struct module_info myriminfo =
{
  MYRIIDNUM,					/* mi_idnum */
  MYRINAME,					/* mi_idname */
  MYRIMINPSZ,					/* mi_minpsz */
  MYRIMAXPSZ,					/* mi_maxpsz */
  MYRIHIWAT,					/* mi_hiwat */
  MYRILOWAT,					/* mi_lowat */
};

static struct qinit myririnit =
{
    NULL,						/* qi_putp */
#ifdef FLOW_CONTROL
    myrirsrv,					/* qi_srvp */
#else
    NULL,						/* qi_srvp */
#endif
    (int (*)()) myriopen,		/* qi_qopen */
    (int (*)()) myriclose,		/* qi_qclose */
    NULL,						/* qi_qadmin */
    &myriminfo,					/* qi_minfo */
    NULL,						/* qi_mstat */
};

static struct qinit myriwinit =
{
    (int (*)()) myriwput,		/* qi_putp */
    (int (*)()) myriwsrv,		/* qi_srvp */
    NULL,						/* qi_qopen */
    NULL,						/* qi_qclose */
    NULL,						/* qi_qadmin */
    &myriminfo,					/* qi_minfo */
    NULL,						/* qi_mstat */
};

static struct streamtab myri_info =
{
    &myririnit,					/* st_rdinit */
    &myriwinit,					/* st_wrinit */
    NULL,						/* st_muxrinit */
    NULL,						/* st_muxwrinit */
};


/* Module Loading/Unloading and Autoconfiguration declarations */

/*
 * cb_ops contains the driver entry points and is roughly equivalent
 * to the cdevsw and bdevsw  structures in previous releases.
 *
 * dev_ops contains, in addition to the pointer to cb_ops, the routines
 * that support loading and unloading our driver.
 *
 * Unsupported entry points are set to nodev, except for the poll
 * routine , which is set to nochpoll(), a routine that returns ENXIO.
 */

static struct cb_ops myri_cb_ops =
{
  (int (*)())nodev,		/* cb_open */
  nodev,			/* cb_close */
  nodev,			/* cb_strategy */
  nodev,			/* cb_print */
  nodev,			/* cb_dump */
  nodev,			/* cb_read */
  nodev,			/* cb_write */
  nodev,			/* cb_ioctl */
  nodev,			/* cb_devmap */
  nodev,			/* cb_mmap */
  nodev,			/* cb_segmap */
  nochpoll,			/* cb_chpoll */
  ddi_prop_op,			/* cb_prop_op */
  &myri_info,			/* cb_stream */
  D_MP | D_NEW,			/* cb_flag */
};

static struct dev_ops myri_ops =
{
    DEVO_REV,					/* devo_rev */
    0,							/* devo_refcnt */
    myriinfo,					/* devo_getinfo */
    myriidentify,				/* devo_identify */
    nulldev,					/* devo_probe */
    myriattach,					/* devo_attach */
    myridetach,					/* devo_detach */
    (int (*)(dev_info_t *, ddi_reset_cmd_t)) nodev,		/* devo_reset */
    &myri_cb_ops,				/* devo_cb_ops */
    (struct bus_ops *) NULL,	/* devo_bus_ops */
};


/*
 * Module linkage information for the kernel.
 */
static struct modldrv modldrv =
{
    &mod_driverops,				/* Type of module.  This one is a driver */
    "Myrinet GM-based dlpi driver",		/* Description */
    &myri_ops,					/* driver ops */
};

static struct modlinkage modlinkage =
{
    MODREV_1,
    {&modldrv, NULL}
};


/*
 * Module Loading and Installation Routines .
 */

/*
 * Module Installation
 * Install the driver and "pre-allocate" space for instances of the driver
 */

int
_init(void)
{
    int status;

    CMN_ERR(3, (CE_CONT, "myri: _init called\n"));

    if ((status = ddi_soft_state_init(&myri_state,
				      sizeof(struct myri), 4)) != 0) {
      CMN_ERR(3, (CE_CONT, "myri:_init - soft_state_init failed: 0x%x\n",
			status));
      return (status);
    }

    /* initialize global locks here */
    rw_init(&myristruplock, "myri streams linked list lock",
	    RW_DRIVER, (void *) NULL);

    CMN_ERR (3, (CE_CONT, "myri: _init calling mod_install\n"));
    status = mod_install(&modlinkage);
    CMN_ERR (3, (CE_CONT, "myri: _init returning - status = %d\n", status));

    return (status);
}

/*
 * Module Removal
 */

int
_fini(void)
{
    CMN_ERR(3, (CE_CONT, "myri: _fini called\n"));

    {
	int status;
	/*
	 * At this point, make sure you have cancelled all call backs
	 * and de-allocated all memory
	 */


	if ((status = mod_remove(&modlinkage)) != 0) {
	  CMN_ERR(0, (CE_CONT,
		      "myri:_fini- mod_remove failed: 0x%x\n", status));
	  return (status);
	}
	ddi_soft_state_fini(&myri_state);

	return (status);
    }

}

/*
 * Return Module Info.
 */

int
_info(struct modinfo *modinfop)
{
  CMN_ERR(3, (CE_CONT, "myri: _info called\n"));
  return (mod_info(&modlinkage, modinfop));
}



/*
 * Exchange one msg for another.  Free old msg and allocate
 * a new one if either (1) mp is NULL, (2), requested size
 * is larger than current size, or (3) reference count of
 * the current msg is greater than one.
 * Set db_type and b_rptr/b_wptr appropriately.
 * Set the first longword of the msg to 'primtype' if
 * 'primtype' is not -1.
 *
 * On allocb() failure, return NULL after sending an
 * M_ERROR msg w/ENOSR error upstream.
 */

static mblk_t *
psexchange(queue_t * wq, mblk_t * mp, int size,
	   int type, long primtype)
{
  CMN_ERR(3, (CE_CONT, "psexchange(0x%lx, 0x%lx, 0x%x, 0x%x, 0x%lx)\n",
	wq,mp,size,type,primtype));

  if ((MBLKSIZE(mp) < size) || (DB_REF(mp) > 1)) {
    freemsg((mp));
    if ((mp = allocb(size, BPRI_HI)) == NULL) {
      if ((mp = allocb(1, BPRI_HI))) {
	  pserror(wq, mp, ENOSR);
      }
      CMN_ERR(0, (CE_CONT, "psexchange failed to allocate a buffer\n"));
      return (NULL);
    }
  }
  mp->b_datap->db_type = type;
  mp->b_rptr = mp->b_datap->db_base;
  mp->b_wptr = mp->b_rptr + size;
  if (primtype >= 0) {
    dl_info_ack_t *ackptr = (dl_info_ack_t *)mp->b_rptr;
	CMN_ERR(3, (CE_CONT, "psexchange - setting type = 0x%lx\n",primtype));
    	ackptr->dl_primitive = primtype;
  }

  {
     unsigned char *cptr = (unsigned char *)mp->b_rptr;
     CMN_ERR(3, (CE_CONT, "psexchg ptr=0x%lx header is 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",
			mp->b_rptr, cptr[0],cptr[1],cptr[2],cptr[3],
			cptr[4],cptr[5],cptr[6],cptr[7]));
  }

  return (mp);
}


/*
 *  Some STREAMS routines.
 *
 */

static void
pserror(queue_t * wq, mblk_t * mp, int error)
{
    if ((mp = (mblk_t *) psexchange(wq, mp, 1, M_ERROR, -1)) == NULL)
	return;
    CMN_ERR(0,(CE_CONT,"*** pserror called - has buggy assignment??\n"));
    *mp->b_rptr = (unsigned char) error;
    qreply(wq, mp);
}

/*
 * Convert an M_IOCTL into an M_IOCACK.
 * Assumption:  mp points to an M_IOCTL msg.
 */

static void
psiocack(queue_t * wq, mblk_t * mp, int count, int error)
{
    struct iocblk *iocp;

    mp->b_datap->db_type = M_IOCACK;
    iocp = (struct iocblk *) mp->b_rptr;
    iocp->ioc_count = count;
    iocp->ioc_error = error;
    qreply(wq, mp);
}

/*
 * Convert an M_IOCTL into an M_IOCNAK.
 * Assumption:  mp points to an M_IOCTL msg.
 */

static void
psiocnak(queue_t * wq, mblk_t * mp, int count, int error)
{
    struct iocblk *iocp;

    mp->b_datap->db_type = M_IOCNAK;
    iocp = (struct iocblk *) mp->b_rptr;
    iocp->ioc_count = count;
    iocp->ioc_error = error;
    qreply(wq, mp);
}



/*
 *  Common DLPI routines.
 */


static void
psdlbindack(queue_t * wq, mblk_t * mp, u_long sap, caddr_t addrp,
	    int addrlen, int maxconind, u_long xidtest)
{
    union DL_primitives *dlp;
    int size;

    size = sizeof(dl_bind_ack_t) + addrlen;
    if ((mp = psexchange(wq, mp, size, M_PCPROTO, DL_BIND_ACK))
	== NULL)
	return;

    dlp = (union DL_primitives *) mp->b_rptr;
    dlp->bind_ack.dl_sap = sap;
    dlp->bind_ack.dl_addr_length = addrlen;
    dlp->bind_ack.dl_addr_offset = sizeof(dl_bind_ack_t);
    dlp->bind_ack.dl_max_conind = maxconind;
    dlp->bind_ack.dl_xidtest_flg = xidtest;
    bcopy((caddr_t) addrp,
	  (caddr_t) (mp->b_rptr + sizeof(dl_bind_ack_t)),
	  addrlen);

    qreply(wq, mp);
}

static void
psdlokack(queue_t * wq, mblk_t * mp, u_long correct_primitive)
{
  union DL_primitives *dlp;
  
  CMN_ERR(3, (CE_CONT, "psdlokack(0x%lx, 0x%lx, 0x%lx\n",
	wq, mp, correct_primitive));

  if ((mp = psexchange(wq, mp, (int)sizeof(dl_ok_ack_t), (int)M_PCPROTO,
		       (long)DL_OK_ACK)) == NULL) {
    CMN_ERR(0, (CE_CONT, "psdlokack psexchange failed\n"));
    return;
  }

  dlp = (union DL_primitives *) mp->b_rptr;
  dlp->ok_ack.dl_correct_primitive = correct_primitive;
  qreply(wq, mp);
}

static void
psdlerrorack(queue_t * wq, mblk_t * mp, u_long error_primitive,
	     u_long errno, u_long unix_errno)
{
    register union DL_primitives *dlp;

    if ((mp = psexchange(wq, mp, sizeof(dl_error_ack_t), M_PCPROTO,
			 DL_ERROR_ACK)) == NULL)
	return;
    dlp = (union DL_primitives *) mp->b_rptr;
    dlp->error_ack.dl_error_primitive = error_primitive;
    dlp->error_ack.dl_errno = errno;
    dlp->error_ack.dl_unix_errno = unix_errno;
    qreply(wq, mp);
}

static void
psdluderrorind(queue_t * wq, mblk_t * mp, u_char * addrp,
	       u_long addrlen, u_long errno, u_long unix_errno)
{
    union DL_primitives *dlp;
    char buf[DLADDRL];
    int size;

    if (addrlen > DLADDRL)
	addrlen = DLADDRL;

    bcopy((caddr_t) addrp, (caddr_t) buf, addrlen);

    size = sizeof(dl_uderror_ind_t) + addrlen;

    if ((mp = psexchange(wq, mp, size, M_PCPROTO, DL_UDERROR_IND)) == NULL)
	return;

    dlp = (union DL_primitives *) mp->b_rptr;
    dlp->uderror_ind.dl_dest_addr_length = addrlen;
    dlp->uderror_ind.dl_dest_addr_offset = sizeof(dl_uderror_ind_t);
    dlp->uderror_ind.dl_unix_errno = unix_errno;
    dlp->uderror_ind.dl_errno = errno;
    bcopy((caddr_t) buf,
	  (caddr_t) (mp->b_rptr + sizeof(dl_uderror_ind_t)),
	  addrlen);
    qreply(wq, mp);
}

static void
psdlphysaddrack(queue_t * wq, mblk_t * mp, caddr_t addrp, int len)
{
    union DL_primitives *dlp;
    int size;

    size = sizeof(dl_phys_addr_ack_t) + len;
    if ((mp = psexchange(wq, mp, size, M_PCPROTO, DL_PHYS_ADDR_ACK)) ==
	NULL)
	return;
    dlp = (union DL_primitives *) mp->b_rptr;
    dlp->physaddr_ack.dl_addr_length = len;
    dlp->physaddr_ack.dl_addr_offset = sizeof(dl_phys_addr_ack_t);
    bcopy(addrp, (caddr_t) (mp->b_rptr + sizeof(dl_phys_addr_ack_t)), len);
    qreply(wq, mp);
}


/*
 * Set or clear the device ipq pointer.
 * XXX Assumes IP is SLFAST.
 */
static void
myrisetipq(struct myri *myrip)
{
  struct myristr *slp;
  int	ok = 1;
  queue_t *ipq = NULL;

  rw_enter(&myristruplock, RW_READER);
  
  for (slp = myristrup; slp; slp = slp->sl_nextp)
    if (slp->sl_myrip == myrip) {
      if (slp->sl_flags & (SLALLPHYS|SLALLSAP))
	ok = 0;
      if (slp->sl_sap == ETHERTYPE_IP) {
	if (ipq == NULL) {
	  ipq = slp->sl_rq;
	}
	else {
	  ok = 0;
	}
      }
    }

  rw_exit(&myristruplock);
  
  if (ok)
    myrip->myri_ipq = ipq;
  else
    myrip->myri_ipq = NULL;
}

/*********************   do_putnext()  ***********************/
/*
 * Called from myri_sendup and myrirsrv. "first_time" indicates
 * whether this is a fresh message or one that came off of the
 * read queue (previously blocked).
 */

gm_inline int
do_putnext(struct myristr *slp, mblk_t * mp)
{
  struct myri *myrip;
  register queue_t *rq;

  rq = slp->sl_rq;
  myrip = slp->sl_myrip;
  
  if (canputnext(rq)) {
    putnext(rq, mp);
    CMN_ERR(3, (CE_CONT, "do_putnext: queueing qsize = %d   next = %d \n",
		  qsize(rq),
		(rq->q_next) ? qsize(rq->q_next) : -1));
    return (1);
  } else {
    CMN_ERR(1, (CE_CONT, "do_putnext: DROPPING  qsize = %d   next = %d\n",
		qsize(rq), (rq->q_next) ? qsize(rq->q_next) : -1));
    
    freemsg(mp);
    myrip->myri_ierrors++;
    return (0);
  }
}

/**************   myrifastpath()  *******************/
gm_inline int
myrifastpath(queue_t * wq, mblk_t * mp)
{
/* this routine is called if
 * case M_PROTO:
 * case M_PCPROTO:
 */

    union DL_primitives *dlp;
    struct myristr *slp;
    u_long prim;

    slp = (struct myristr *) wq->q_ptr;
    dlp = (union DL_primitives *) mp->b_rptr;
    prim = dlp->dl_primitive;

    if (prim == DL_UNITDATA_REQ) {

      register struct myri *myrip;
      register dl_unitdata_req_t *dludp;
      register mblk_t *nmp, *nmp2;
      register struct myridladdr *dlap;
      register struct medium_header *headerp;
      int off, len, rv;
      
      myrip = slp->sl_myrip;

      if ((NUM_SEND_BUFF - myrip->send_buffer_count) <= 0) {
	CMN_ERR(SEND_BLOCKED, (CE_CONT, "myri fastpath fails, too many sends pending"));

	return (0);
      }
	
      rv = mutex_tryenter(&slp->sl_lock);
      if (!rv) {
	CMN_ERR(3, (CE_CONT, "myriFASTPATH FAILED mutex already held\n"));
	return (0);
      }


      dludp = (dl_unitdata_req_t *) mp->b_rptr;
      
      off = dludp->dl_dest_addr_offset;
      len = dludp->dl_dest_addr_length;
      
      /*
       * Validate destination address format.
       */
      if (!MBLKIN(mp, off, len) || (len != DEVICE_ADDRL)) {
	CMN_ERR(3, (CE_CONT, "myriFASTPATH FAILED bad addr format\n"));
	mutex_exit(&slp->sl_lock);
	return (0);
      }
      
      /*
       * Error if no M_DATA follows.
       */
      nmp = mp->b_cont;
      if (nmp == NULL) {
	mutex_exit(&slp->sl_lock);
	CMN_ERR(3, (CE_CONT, "myriFASTPATH FAILED no m_data follows\n"));
	return (0);
      }

      dlap = (struct myridladdr *) (mp->b_rptr + off);
      
      /*
       * Create medium header by either prepending it onto the next
       * mblk if possible, or reusing the M_PROTO block if not.
       */
#if 0
      if ((DB_REF(nmp) == 1) &&
	  (MBLKHEAD(nmp) >= sizeof(struct medium_header)) &&
	  (((ulong) nmp->b_rptr & 0x1) == 0)) {
	nmp->b_rptr -= sizeof(struct medium_header);
	headerp = (struct medium_header *) nmp->b_rptr;
	bcopy_addr((caddr_t) & dlap->dl_phys, (caddr_t) & headerp->med_dhost);
	bcopy_addr((caddr_t) & myrip->myri_ouraddr, (caddr_t) & headerp->med_shost);
	/*      headerp->med_type = dlap->dl_sap; */
	freeb(mp);
	mp = nmp;
      } else
#endif 
	{					/* FIX Elliot */
	  /* XXX FIX BOB check it out */
	  if (DB_REF(mp) == 1)
	    nmp2 = mp;
	  else {
	    nmp2 = copymsg(mp);
	    if (nmp2 == NULL) {
	      mutex_exit(&slp->sl_lock);
	      CMN_ERR(3, (CE_CONT, "cannot CopyMsg in FastPath\n"));
	      return 0;
	    }
	    freemsg(mp);
	  }
	  
	  DB_TYPE(nmp2) = M_DATA;
	  headerp = (struct medium_header *) nmp2->b_rptr;
	  nmp2->b_wptr = nmp2->b_rptr + sizeof(struct medium_header);
	  bcopy_addr((caddr_t) & dlap->dl_phys,
		     (caddr_t) & headerp->med_dhost);
	  bcopy_addr((caddr_t) & myrip->myri_ouraddr,
		     (caddr_t) & headerp->med_shost);
	    
	  /*      headerp->med_type = dlap->dl_sap; */
	  mp = nmp2;
	}

      put_ether_type(headerp, slp->sl_sap);
      mutex_exit(&slp->sl_lock);
      /*
       * Transmit it.
       */

      rv = myri_tolanai(wq, mp, myrip);


      CMN_ERR(3, (CE_CONT, "myriFASTPATH suceeded\n"));
      return (1);
    } else {
      /* not fastpath */
      CMN_ERR(3, (CE_CONT, "myriFASTPATH FAILED not DL_UNITDATA_REQ\n"));
      return (0);
    }
}


/*
 * ------------------------------------------------------
 *  Autoconfiguration Routines
 */

/*
 * identify(9e) - are we the right driver.
 */
static int
myriidentify(dev_info_t * dip)
{
    CMN_ERR(3, (CE_CONT, "myriidentify called... str = '%s'\n",
		ddi_get_name(dip)));

    if (strcmp(ddi_get_name(dip), MYRINAME) == 0) {
	CMN_ERR(3, (CE_CONT, "     IDENTIFIED\n"));
	return (DDI_IDENTIFIED);
    }
    else {
	CMN_ERR_PRINT (0, (CE_CONT, "'%s'  NOT identified\n"));
	return (DDI_NOT_IDENTIFIED);
    }
}

/*
 * Interface exists: make available by filling in network interface
 * record.  System will initialize the interface when it is ready
 * to accept packets.
 */

extern unsigned int gm_arch_solaris_instance_ok[32];

static int
myriattach(dev_info_t * dip, ddi_attach_cmd_t cmd)
{

    struct myri *myrip;
    int instance;

    if (cmd != DDI_ATTACH) {
	myrierror(dip, "myriattach ! DDI_ATTACH : failed\n");
	return (DDI_FAILURE);
    }

    instance = ddi_get_instance(dip);

    CMN_ERR(0, (CE_CONT, "myriattach called... name = '%s' instance = %d   depends = '%s'\n",
		ddi_get_name(dip), instance, _depends_on));

    CMN_ERR(0, (CE_CONT, "myriattach: '%s' %d 0x%x 0x%x 0x%x\n",
		ddi_get_name(dip),
		instance,
		(unsigned) ddi_get_nodeid(dip),
		(unsigned) ddi_get_driver(dip),
		(unsigned) ddi_get_driver_private(dip)));



     /* this array is initialized to zero, then each instance is set to 1 */
     /* after gm_instance_init successfully returns */
     if (gm_arch_solaris_instance_ok[instance] == 0) {
	int rv;
	CMN_ERR_PRINT (0, (CE_CONT, "myri%d: NOT initialized yet? GM driver not properly loaded?\n"
				, instance));
	CMN_ERR_PRINT (0, (CE_CONT, "myri%d: Try running 'gm_board_info' to load GM\n"
				, instance));
#ifdef DELETE
	rv = open("/dev/gm0",O_RDWR);
	if (rv!=-1) {
		CMN_ERR_PRINT (0, (CE_CONT, "myri%d: opened /dev/gm0\n",instance));
		close(rv);
	}
	else {
		CMN_ERR_PRINT (0, (CE_CONT, "myri%d: couldn't open /dev/gm0\n",instance));
	}
#endif DELETE
	return (DDI_FAILURE);
     }
     else {
	CMN_ERR_PRINT (0, (CE_CONT, "myri%d: GM driver was properly loaded\n"
				, instance));
     }

    /*
     * Allocate soft data structure
     */

    if (ddi_soft_state_zalloc(myri_state, instance) != DDI_SUCCESS) {
	CMN_ERR_PRINT (0, (CE_CONT, "myri%d: soft state alloc failed\n", instance));
	return (DDI_FAILURE);
    }

    CMN_ERR(1, (CE_CONT, "allocated the soft state\n"));

    if (!(myrip = (struct myri *) ddi_get_soft_state(myri_state, instance))) {
	CMN_ERR_PRINT (0, (CE_CONT, "myri%d: myriattach: failed ddi_get_soft_state(%d)\n",
		    instance, instance));
	/* FIX - free the zalloc_soft_state */
	return (DDI_FAILURE);
    }

    myrip->unit = instance;

    if (gm_open(&myrip->gm_port, 
		instance, 
		GM_ETHERNET_PORT_ID, 
		"ethernet", 
		GM_API_VERSION_1_0) != GM_SUCCESS) {
      CMN_ERR_PRINT (0, (CE_CONT, "myri[%d]: failed to open GM port\n", instance));
      
      /* FIX - free the zalloc_soft_state */
      return (DDI_FAILURE);
    } 
    
    bcopy_addr((caddr_t) & myrip->gm_port->kernel_port_state->instance->lanai.eeprom.lanai_board_id[0],
	       (caddr_t) & myrip->myri_ouraddr.medaddr[0]);

    CMN_ERR(0, (CE_CONT, "myri%d: id = %x:%x:%x:%x:%x:%x\n",
		instance,
		myrip->myri_ouraddr.medaddr[0],
		myrip->myri_ouraddr.medaddr[1],
		myrip->myri_ouraddr.medaddr[2],
		myrip->myri_ouraddr.medaddr[3],
		myrip->myri_ouraddr.medaddr[4],
		myrip->myri_ouraddr.medaddr[5]));

    /*
     * Initialize mutex's for this device.
     */
    /* if used in interrupt routine, arg4 = iblock_cookie, else NULL */

    mutex_init(&myrip->myri_xmitlock, "myri xmit lock", MUTEX_DRIVER, 0);
    /* (void *)myrip->gm_port->kernel_port_state->instance->arch.iblock_cookie); */
    mutex_init(&myrip->myri_recvlock, "myri recv lock", MUTEX_DRIVER,
	       (void *)myrip->gm_port->kernel_port_state->instance->arch.iblock_cookie); 
    mutex_init(&myrip->myri_bufflock, "myri buff lock", MUTEX_DRIVER, 
	       (void *)myrip->gm_port->kernel_port_state->instance->arch.iblock_cookie);
    

    CMN_ERR(0, (CE_CONT, "initialized mutex myri_xmitlock = 0x%lx\n",
		&myrip->myri_xmitlock));
    CMN_ERR(0, (CE_CONT, "initialized mutex myri_recvlock = 0x%lx\n",
		&myrip->myri_recvlock));
    CMN_ERR(0, (CE_CONT, "initialized mutex myri_bufflock = 0x%lx\n",
		&myrip->myri_bufflock));

    /*
     * Stuff private info into dip.
     */

    gm_ethernet_set_recv_intr_callback
      (myrip->gm_port, myri_recv_intr, myrip);
    gm_ethernet_set_sent_intr_callback
      (myrip->gm_port, myri_sent_intr, myrip);

    ddi_set_driver_private(dip, (caddr_t) myrip);
    myrip->myri_dip = dip;


    /*
     * Link this per-device structure in with the rest.
     */

    myrip->myri_nextp = myridev;
    myridev = myrip;


    /*
     * Create the filesystem device node.
     */
    if (ddi_create_minor_node(dip, MYRINAME, S_IFCHR,
			      instance, DDI_NT_NET, CLONE_DEV) == DDI_FAILURE) {
	if (myridebug) {
	    myrierror(dip, "ddi_create_minor_node failed");
	}
	ddi_remove_minor_node(dip, NULL);
	goto bad;
    }

    CMN_ERR(3, (CE_CONT, "created device  myri%d\n", instance));


#ifdef  KSTAT
    myristatinit(myrip);
#endif							/* KSTAT */

    if (!myri_alloc_buffers(myrip)) {
	CMN_ERR_PRINT (0, (CE_CONT, "myri_alloc_buff  FAILED!!!!! **********  \n"));
	goto bad;
    }
    
    myri_free_buff_all(myrip);
    myri_reset_recv_buff(myrip);

    ddi_report_dev(dip);
    CMN_ERR_PRINT (0, (CE_CONT, MYRIVERSION));
    CMN_ERR_PRINT (0, (CE_CONT, "%s\n", myricopyright));


    return (DDI_SUCCESS);    

bad:
    ddi_remove_minor_node(dip, NULL);

    /* destroy all mutex's */
    mutex_destroy(&myrip->myri_recvlock);
    mutex_destroy(&myrip->myri_xmitlock);
    mutex_destroy(&myrip->myri_bufflock);

    gm_close(myrip->gm_port);

    if (myridebug) {
	myrierror(myrip->myri_dip, "attach failed");
    }


    if (myrip) {
	ddi_soft_state_free(myri_state, instance);
    }

    return (DDI_FAILURE);

}


/*
 * Detach - Free resources allocated in attach
 */

/*ARGSUSED */
static int
myridetach(dev_info_t * dip, ddi_detach_cmd_t cmd)
{
    int instance;
    struct myri *myrip;

    if (cmd != DDI_DETACH) {
	return (DDI_FAILURE);
    }

    instance = ddi_get_instance(dip);

    CMN_ERR (0, (CE_CONT,"myri[%d]: myridetach called\n",instance));

    myrip = (struct myri *) ddi_get_soft_state(myri_state, instance);
    if (myrip == NULL) {
	return (DDI_FAILURE);
    }

	gm_ethernet_set_recv_intr_callback(myrip->gm_port, 0, 0);
	gm_ethernet_set_sent_intr_callback(myrip->gm_port, 0, 0);

    gm_close(myrip->gm_port); 

    /* FIX  - be careful if buffers have been loaned to higher layers */
    /* FIX  - probably need to NOT free and set a flag to test in callback function */
    CMN_ERR(0, (CE_CONT,"myri[%d]: detach calling myri_Unalloc_buffers\n",instance));
    myri_unalloc_buffers(myrip);

    kstat_delete(myrip->ksp);

    /* destroy all mutex's */
    mutex_destroy(&myrip->myri_recvlock);
    mutex_destroy(&myrip->myri_xmitlock);
    mutex_destroy(&myrip->myri_bufflock);

    /* release soft state */
    ddi_soft_state_free(myri_state, instance);

    /* release minor node */
    ddi_remove_minor_node(dip, NULL);

    return (DDI_SUCCESS);
}



/*
 * Translate "dev_t" to a pointer to the associated "dev_info_t".
 */
/* ARGSUSED */
static int
myriinfo(dev_info_t * dip, ddi_info_cmd_t infocmd, void *arg,
	 void **result)
{
    dev_t dev;
    int instance, rc;
    struct myri *myrip;

    CMN_ERR(3, (CE_CONT, "myriinfo called\n"));

    switch (infocmd) {
    case DDI_INFO_DEVT2DEVINFO:
	dev = (dev_t) arg;
	instance = (int) getminor(dev);
	myrip = (struct myri *) ddi_get_soft_state(myri_state, instance);
	CMN_ERR(3, (CE_CONT, "DDI_INFO_DEVT2DEVINFO: minor=%d inst=%d\n",
		    getminor(dev), instance));
	if (myrip == NULL) {
	    *result = NULL;
	    rc = DDI_FAILURE;
	}
	else {
	    *result = myrip->myri_dip;
	    rc = DDI_SUCCESS;
	}
	break;

    case DDI_INFO_DEVT2INSTANCE:
	dev = (dev_t) arg;
	instance = (int) getminor(dev);
	*result = (void *) instance;
	rc = DDI_SUCCESS;
	CMN_ERR(3,  (CE_CONT, "DDI_INFO_DEVT2INSTANCE: minor = %d inst = 0x%lx\n",
		    getminor(dev), instance));
	break;

    default:
	rc = DDI_FAILURE;
	break;
    }

    CMN_ERR(3, (CE_CONT, "returning %s\n",
		(rc == DDI_SUCCESS) ? "SUCCESS" : "FAILURE"));
    return (rc);
}

#ifdef  KSTAT

static int
myristat_kstat_update(kstat_t * ksp, int rw)
{
    struct myri *myrip;
    struct myristat *myrisp;

    if (rw == KSTAT_WRITE)
	return (EACCES);
    myrip = (struct myri *) ksp->ks_private;
    myrisp = (struct myristat *) ksp->ks_data;
    myrisp->les_ipackets.value.ul = myrip->myri_ipackets;
    myrisp->les_ierrors.value.ul = myrip->myri_ierrors;
    myrisp->les_opackets.value.ul = myrip->myri_opackets;
    myrisp->les_oerrors.value.ul = myrip->myri_oerrors;
    myrisp->les_collisions.value.ul = myrip->myri_collisions;
    return (0);
}

static void
myristatinit(struct myri *myrip)
{
    struct kstat *ksp;
    struct myristat *myrisp;

    if ((ksp = kstat_create(MYRINAME, ddi_get_instance(myrip->myri_dip),
			    NULL, "net", KSTAT_TYPE_NAMED,
			    sizeof(struct myristat) / sizeof(kstat_named_t), 0)) == NULL) {
	myrierror(myrip->myri_dip, "kstat_create failed");
	return;
    }
    myrip->ksp = ksp;
    myrisp = (struct myristat *) (ksp->ks_data);
    kstat_named_init(&myrisp->les_ipackets, "ipackets",
		     KSTAT_DATA_ULONG);
    kstat_named_init(&myrisp->les_ierrors, "ierrors",
		     KSTAT_DATA_ULONG);
    kstat_named_init(&myrisp->les_opackets, "opackets",
		     KSTAT_DATA_ULONG);
    kstat_named_init(&myrisp->les_oerrors, "oerrors",
		     KSTAT_DATA_ULONG);
    kstat_named_init(&myrisp->les_collisions, "collisions",
		     KSTAT_DATA_ULONG);
    ksp->ks_update = myristat_kstat_update;
    ksp->ks_private = (void *) myrip;
    kstat_install(ksp);

}

#endif							/* KSTAT */

/*ARGSUSED */
static
int
myriopen(queue_t * rq, dev_t * devp, int flag, int sflag, cred_t * credp)
{
    struct myristr *slp;
    struct myristr **prevslp;
    minor_t minordev;
    int rc = 0;

    CMN_ERR(3, (CE_CONT, "myriopen called  rq = 0x%lx\n", rq));

    ASSERT(sflag != MODOPEN);

    /*
     * Serialize all driver open and closes.
     */
    rw_enter(&myristruplock, RW_WRITER);

    /*
     * Determine minor device number.
     */
    prevslp = &myristrup;
    if (sflag == CLONEOPEN) {
	minordev = 0;
	for (; (slp = *prevslp); prevslp = &slp->sl_nextp) {
	    if (minordev < slp->sl_minor)
		break;
	    minordev++;
	}
	*devp = makedevice(getmajor(*devp), minordev);
    }
    else {
	minordev = getminor(*devp);
    }

    CMN_ERR(3, (CE_CONT, "myriopen minordev = %d\n", minordev));
    if (rq->q_ptr)
	goto done;

    if ((slp = GETSTRUCT(struct myristr, 1)) == NULL) {
	rc = ENOMEM;
	rw_exit(&myristruplock);
	return (rc);
    }

    slp->sl_minor = minordev;
    slp->sl_rq = rq;
    slp->sl_state = DL_UNATTACHED;
    slp->sl_sap = 0;
    slp->sl_flags = 0;
    slp->sl_myrip = NULL;
    slp->sl_mccount = 0;
    slp->sl_mctab = NULL;
    mutex_init(&slp->sl_lock, "myri stream lock", MUTEX_DRIVER, (void *) 0);

    /*
     * Link new entry into the list of active entries.
     */
    slp->sl_nextp = *prevslp;
    *prevslp = slp;

    rq->q_ptr = WR(rq)->q_ptr = slp;

    /*
     * Disable our write-side  & read-side service procedures.
     */
    noenable(WR(rq));
    noenable(rq);

done:
    rw_exit(&myristruplock);
    qprocson(rq);
    return (rc);
}

static int
myriclose(queue_t * rq, int flag, cred_t * credp)
{
    struct myristr *slp;
    struct myristr **prevslp;

    CMN_ERR(3, (CE_CONT, "myriclose called  rq = 0x%lx\n", rq));

    ASSERT(rq);
    ASSERT(rq->q_ptr);


    slp = (struct myristr *) rq->q_ptr;

    /*
     * Implicit detach Stream from interface.
     */
    if (slp->sl_myrip)
	myridodetach(slp);

    qprocsoff(rq);

    rw_enter(&myristruplock, RW_WRITER);

    /*
     * Unlink the per-Stream entry from the active list and free it.
     */
    for (prevslp = &myristrup; (slp = *prevslp); prevslp = &slp->sl_nextp)
	if (slp == (struct myristr *) rq->q_ptr)
	    break;
    ASSERT(slp);
    *prevslp = slp->sl_nextp;

    mutex_destroy(&slp->sl_lock);
    kmem_free(slp, sizeof(struct myristr));

    rq->q_ptr = WR(rq)->q_ptr = NULL;

    rw_exit(&myristruplock);
    return (0);
}

/*
 * Start transmission.
 * Return zero on success,
 * otherwise put msg on wq, set 'want' flag and return nonzero.
 */
gm_inline int
myristart(queue_t * wq, mblk_t * mp, struct myri *myrip)
{
  int rv;
#ifdef SET_WANTW
  if ((NUM_SEND_BUFF - myrip->send_buffer_count) <= 0) {
    CMN_ERR(SEND_BLOCKED, (CE_CONT,
			   "myristart: put msg on wq, set 'want' flag and return nonzero. (%d) \n", rv));
    putq(wq, mp);
    myrip->myri_wantw = 1;
    return (1);
  }
#endif	/* SET_WANTW */

  rv = myri_tolanai(wq, mp, myrip);
  return 1 - rv;
}

static int
myriwput(queue_t * wq, mblk_t * mp)
{
  struct myristr *slp = (struct myristr *) wq->q_ptr;
  
  CMN_ERR(3, (CE_CONT, "myriwput called\n"));
  
  switch (DB_TYPE(mp)) {
  case M_DATA:
    {
      struct myri *myrip = slp->sl_myrip;
      
      CMN_ERR(3, (CE_CONT, "    type = M_DATA\n"));
      if (((slp->sl_flags & (SLFAST | SLRAW)) == 0) ||
	  (slp->sl_state != DL_IDLE) ||
	  (myrip == NULL)) {
	CMN_ERR(3, (CE_CONT, "   calling pserror\n"));
	pserror(wq, mp, EPROTO);
	break;
      }
      
      /*
       * If any msgs already enqueued or the interface will
       * loop back up the message (due to MYRIPROMISC),
       * then enqueue the msg.  Otherwise just xmit it
       * directly.
       */
      if (wq->q_first) {
	CMN_ERR(3, (CE_CONT, "   enqueueing the message\n"));
	putq(wq, mp);
	myrip->myri_wantw = 1;
      } else if (myrip->myri_flags & MYRIPROMISC) {
	CMN_ERR(3, (CE_CONT, "   enqueueing the message\n"));
	putq(wq, mp);
	qenable(wq);
      } else {
	CMN_ERR(3, (CE_CONT, "   calling myristart\n"));
	(void) myristart(wq, mp, myrip);
      }
    }
    break;
    
  case M_PROTO:
  case M_PCPROTO:
    {
      int rv = -1;
      CMN_ERR(3, (CE_CONT, "    type = M_PROTO or M_PCPROTO\n"));
      
      if (!wq->q_first){     
	rv = myrifastpath(wq, mp);
      }
      
      if (rv != 1) {
	CMN_ERR(3, (CE_CONT, "   no fastpath, putq the PROTO msg rv = %d\n",
		    rv));
	putq(wq, mp);
	qenable(wq);
      }
    }
    break;

  case M_IOCTL:
    CMN_ERR(3, (CE_CONT, "    type = M_IOCTL\n"));
    myriioctl(wq, mp);
    break;

  case M_FLUSH:
    CMN_ERR(3, (CE_CONT, "    type = M_FLUSH\n"));
    if (*mp->b_rptr & FLUSHW) {
      flushq(wq, FLUSHALL);
      *mp->b_rptr &= ~FLUSHW;
    }
    if (*mp->b_rptr & FLUSHR)
      qreply(wq, mp);
    else
      freemsg(mp);
	break;
	
  default:
    CMN_ERR(3, (CE_CONT, "    type = default = %d\n",
		DB_TYPE(mp)));
    freemsg(mp);
    break;
  }
  return (0);
}

/*
 * Enqueue M_PROTO/M_PCPROTO (always) and M_DATA (sometimes) on the wq.
 *
 * Processing of some of the M_PROTO/M_PCPROTO msgs involves acquiring
 * internal locks that are held across upstream putnext calls.
 * Specifically there's the problem of myriintr() holding myri_intrlock
 * and myristruplock when it calls putnext() and that thread looping
 * back around to call myriwput and, eventually, myriinit() to create a
 * recursive lock panic.  There are two obvious ways of solving this
 * problem: (1) have myriintr() do putq instead of putnext which provides
 * the loopback "cutout" right at the rq, or (2) allow myriintr() to putnext
 * and put the loopback "cutout" around myriproto().  We choose the latter
 * for performance reasons.
 *
 * M_DATA messages are enqueued on the wq *only* when the xmit side
 * is out of buffer space.  Once the xmit resource is available again,
 * wsrv() is enabled and tries to xmit all the messages on the wq.
 */
static
int
myriwsrv(queue_t * wq)
{
  register mblk_t *mp;
  struct myristr *slp;
  register struct myri *myrip;
  
  CMN_ERR(3, (CE_CONT, "myriWSRV called\n"));
  
  slp = (struct myristr *) wq->q_ptr;
  myrip = slp->sl_myrip;
  
  while ((mp = getq(wq))) {
    switch (DB_TYPE(mp)) {
    case M_DATA:
      if (myrip) {
	if (myristart(wq, mp, myrip))
	  return 0;
      }  else {
	CMN_ERR(3, (CE_CONT, "   myrip = NULL, freeing the message\n"));
	freemsg(mp);
      }
      break;
      
    case M_PROTO:
    case M_PCPROTO:
      myriproto(wq, mp);
      break;
      
    default:				/* nothing is working at ths point */
      ASSERT(0);
      break;
    }
  }
  return (0);
}


static void
myriproto(queue_t * wq, mblk_t * mp)
{
    union DL_primitives *dlp;
    struct myristr *slp;
    u_long prim;

    slp = (struct myristr *) wq->q_ptr;
    dlp = (union DL_primitives *) mp->b_rptr;
    prim = dlp->dl_primitive;

    CMN_ERR(3, (CE_CONT, "myriproto called primitive = %d (0x%x)\n",
		prim,prim));


    switch (prim) {
    case DL_UNITDATA_REQ:
      myriudreq(wq, mp);
      break;

    case DL_ATTACH_REQ:
      mutex_enter(&slp->sl_lock);
      myriareq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_DETACH_REQ:
      mutex_enter(&slp->sl_lock);
      myridreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_BIND_REQ:
      mutex_enter(&slp->sl_lock);
      myribreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_UNBIND_REQ:
      mutex_enter(&slp->sl_lock);
      myriubreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_INFO_REQ:
      mutex_enter(&slp->sl_lock);
      myriireq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_PROMISCON_REQ:
      mutex_enter(&slp->sl_lock);
      myriponreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_PROMISCOFF_REQ:
      mutex_enter(&slp->sl_lock);
      myripoffreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_ENABMULTI_REQ:
      mutex_enter(&slp->sl_lock);
      myriemreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_DISABMULTI_REQ:
      mutex_enter(&slp->sl_lock);
      myridmreq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_PHYS_ADDR_REQ:
      mutex_enter(&slp->sl_lock);
      myripareq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    case DL_SET_PHYS_ADDR_REQ:
      mutex_enter(&slp->sl_lock);
      myrispareq(wq, mp);
      mutex_exit(&slp->sl_lock);
      break;

    default:
      psdlerrorack(wq, mp, prim, DL_UNSUPPORTED, 0);
      break;
    }
}

static void
myriioctl(queue_t * wq, mblk_t * mp)
{
    struct iocblk *iocp = (struct iocblk *) mp->b_rptr;
    struct myristr *slp = (struct myristr *) wq->q_ptr;

    CMN_ERR(3, (CE_CONT, "myriioctl called\n"));

    switch (iocp->ioc_cmd) {
    case DLIOCRAW:			/* raw M_DATA mode */
      slp->sl_flags |= SLRAW;
      psiocack(wq, mp, 0, 0);
      break;
#ifdef IPQ
    case DL_IOC_HDR_INFO:	/* M_DATA "fastpath" info request */
      myri_dl_ioc_hdr_info(wq, mp);
      break;
#endif
    default:
      psiocnak(wq, mp, 0, EINVAL);
      break;
    }
}
/*
 * M_DATA "fastpath" info request.
 * Following the M_IOCTL mblk should come a DL_UNITDATA_REQ mblk.
 * We ack with an M_IOCACK pointing to the original DL_UNITDATA_REQ mblk
 * followed by an mblk containing the raw ethernet header corresponding
 * to the destination address.  Subsequently, we may receive M_DATA
 * msgs which start with this header and may send up
 * up M_DATA msgs with b_rptr pointing to a (uint32_t) group address
 * indicator followed by the network-layer data (IP packet header).
 * This is all selectable on a per-Stream basis.
 */
static void
myri_dl_ioc_hdr_info(queue_t *wq, mblk_t *mp)
{
  mblk_t	*nmp;
  struct myristr	*slp;
  struct myridladdr	*dlap;
  dl_unitdata_req_t	*dludp;
  struct medium_header	*headerp;
  struct myri	*myrip;
  int	off, len;
  int	minsize;

  CMN_ERR(3, (CE_CONT, "myri_dl_ioc called\n"));  
  slp = (struct myristr *)wq->q_ptr;
  minsize = sizeof (dl_unitdata_req_t) + DEVICE_ADDRL;
  
  /*
   * Sanity check the request.
   */
  if ((mp->b_cont == NULL) ||
      (MBLKL(mp->b_cont) < minsize) ||
      (*((uint32_t *)mp->b_cont->b_rptr) != DL_UNITDATA_REQ) ||
      ((myrip = slp->sl_myrip) == NULL)) {
    psiocnak(wq, mp, 0, EINVAL);
    return;
  }
  
  /*
   * Sanity check the DL_UNITDATA_REQ destination address
   * offset and length values.
   */
  dludp = (dl_unitdata_req_t *)mp->b_cont->b_rptr;
  off = dludp->dl_dest_addr_offset;
  len = dludp->dl_dest_addr_length;
  if (!MBLKIN(mp->b_cont, off, len) || (len != DEVICE_ADDRL)) {
    psiocnak(wq, mp, 0, EINVAL);
    return;
  }
  
  dlap = (struct myridladdr *)(mp->b_cont->b_rptr + off);
  
  /*
   * Allocate a new mblk to hold the ether header.
   */
  if ((nmp = allocb(sizeof (struct medium_header), BPRI_MED)) == NULL) {
    psiocnak(wq, mp, 0, ENOMEM);
    return;
  }
  nmp->b_wptr += sizeof (struct medium_header);
  
  /*
   * Fill in the ether header.
   */
  headerp = (struct medium_header *)nmp->b_rptr;
  bcopy_addr((caddr_t)&dlap->dl_phys, (caddr_t)&headerp->med_dhost);
  bcopy_addr((caddr_t)&myrip->myri_ouraddr, (caddr_t)&headerp->med_shost);
  put_ether_type(headerp, slp->sl_sap);
  
  /*
   * Link new mblk in after the "request" mblks.
   */
  linkb(mp, nmp);
  
  slp->sl_flags |= SLFAST;
  
  /*
   * XXX Don't bother calling myrisetipq() here.
   */
  
  psiocack(wq, mp, myri_msgsize(mp->b_cont), 0);
}

int
myri_msgsize(register mblk_t *mp)
{
  register int n = 0;
  
  for (; mp; mp = mp->b_cont)
    n += MBLKL(mp);
  
  return (n);
}

/* Generic DLPI interface routines */

static void
myriareq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    union DL_primitives *dlp;
    struct myri *myrip;
    int ppa;

    CMN_ERR(3, (CE_CONT, "myriareq called\n"));

    slp = (struct myristr *) wq->q_ptr;
    dlp = (union DL_primitives *) mp->b_rptr;

    if (MBLKL(mp) < DL_ATTACH_REQ_SIZE) {
	CMN_ERR(3, (CE_CONT, "myriareq error < DL_ATTACH_REQ_SIZE\n"));
	psdlerrorack(wq, mp, DL_ATTACH_REQ, DL_BADPRIM, 0);
	return;
    }

    if (slp->sl_state != DL_UNATTACHED) {
	CMN_ERR(3, (CE_CONT, "myriareq error != DL_UNATTACHED \n"));
	psdlerrorack(wq, mp, DL_ATTACH_REQ, DL_OUTSTATE, 0);
	return;
    }

    ppa = dlp->attach_req.dl_ppa;

    /*
     * Valid ppa?
     */
    for (myrip = myridev; myrip; myrip = myrip->myri_nextp)
	if (ppa == ddi_get_instance(myrip->myri_dip))
	    break;
    if (myrip == NULL) {
	CMN_ERR(3, (CE_CONT, "myriareq error myrip is NULL\n"));
	psdlerrorack(wq, mp, dlp->dl_primitive, DL_BADPPA, 0);
	return;
    }

    /*
     * Has device been initialized?  Do so if necessary.
     */
    if ((myrip->myri_flags & MYRIRUNNING) == 0) {
	if (myriinit(myrip)) {
	    CMN_ERR(3, (CE_CONT, "myriareq error INITFAILED\n"));
	    psdlerrorack(wq, mp, dlp->dl_primitive,
			 DL_INITFAILED, 0);
	    return;
	}
    }

    /*
     * Set link to device and update our state.
     */
    slp->sl_myrip = myrip;
    slp->sl_state = DL_UNBOUND;

    CMN_ERR(3, (CE_CONT, "myriareq completed OK\n"));
    psdlokack(wq, mp, DL_ATTACH_REQ);
}

static void
myridreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;

    CMN_ERR(3, (CE_CONT, "myridreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_DETACH_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_DETACH_REQ, DL_BADPRIM, 0);
	return;
    }

    if (slp->sl_state != DL_UNBOUND) {
	psdlerrorack(wq, mp, DL_DETACH_REQ, DL_OUTSTATE, 0);
	return;
    }

    myridodetach(slp);
    psdlokack(wq, mp, DL_DETACH_REQ);
}

/*
 * Detach a Stream from an interface.
 */
static void
myridodetach(struct myristr *slp)
{
    struct myristr *tslp;
    struct myri *myrip;
    int reinit = 0;

    CMN_ERR(3, (CE_CONT, "myridodetach called\n"));

    ASSERT(slp->sl_myrip);

    myrip = slp->sl_myrip;
    slp->sl_myrip = NULL;

    /*
     * Disable promiscuous mode if on.
     */
    if (slp->sl_flags & SLALLPHYS) {
	slp->sl_flags &= ~SLALLPHYS;
	reinit = 1;
    }

    /*
     * Disable ALLMULTI mode if on.
     */
    if (slp->sl_flags & SLALLMULTI) {
	slp->sl_flags &= ~SLALLMULTI;
	reinit = 1;
    }

    /*
     * Disable any Multicast Addresses.
     */
    slp->sl_mccount = 0;
    if (slp->sl_mctab) {
	kmem_free(slp->sl_mctab, DEVMCALLOC);
	slp->sl_mctab = NULL;
	reinit = 1;
    }

    /*
     * Detach from device structure. Uninit the device when no other
     * streams are attached to it.
     */
    for (tslp = myristrup; tslp; tslp = tslp->sl_nextp)
	if (tslp->sl_myrip == myrip)
	    break;
    if (tslp == NULL)
	myriuninit(myrip);
    else if (reinit)
	(void) myriinit(myrip);

    slp->sl_state = DL_UNATTACHED;

#ifdef IPQ
    myrisetipq(myrip);
#endif
}

static void
myribreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    union DL_primitives *dlp;
    struct myri *myrip;
    struct myridladdr myriaddr;
    u_long sap;
    int xidtest;

    CMN_ERR(3, (CE_CONT, "myribreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_BIND_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_BIND_REQ, DL_BADPRIM, 0);
	return;
    }

    if (slp->sl_state != DL_UNBOUND) {
	psdlerrorack(wq, mp, DL_BIND_REQ, DL_OUTSTATE, 0);
	return;
    }

    dlp = (union DL_primitives *) mp->b_rptr;
    myrip = slp->sl_myrip;
    sap = dlp->bind_req.dl_sap;
    xidtest = dlp->bind_req.dl_xidtest_flg;

    if (xidtest) {
	psdlerrorack(wq, mp, DL_BIND_REQ, DL_NOAUTO, 0);
	return;
    }

    if (sap > MEDIUMTYPE_MAX) {
	psdlerrorack(wq, mp, dlp->dl_primitive, DL_BADSAP, 0);
	return;
    }

    /*
     * Save SAP value for this Stream and change state.
     */
    slp->sl_sap = sap;
    slp->sl_state = DL_IDLE;

    CMN_ERR(3, (CE_CONT, "sap %ld\n", sap));
    myriaddr.dl_sap = sap;
    bcopy_addr((caddr_t) & myrip->myri_ouraddr,
	       (caddr_t) & myriaddr.dl_phys);

    CMN_ERR(3, (CE_CONT, "binding a stream sap = %x  wq = %x  rq = %x\n",
		sap, wq, RD(wq)));

    CMN_ERR(3, (CE_CONT, "slp->sl_rq = %x  RD(wq) = %x\n", slp->sl_rq, RD(wq)));

    if (slp->sl_rq != RD(wq)) {
	CMN_ERR(3, (CE_CONT, "UPDATING the rq pointer\n"));
	slp->sl_rq = RD(wq);
    }

    psdlbindack(wq, mp, sap, (caddr_t) & myriaddr, DEVICE_ADDRL, 0, 0);

#ifdef IPQ
    myrisetipq(myrip); 
#endif

}

static void
myriubreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;

    CMN_ERR(3, (CE_CONT, "myriubreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_UNBIND_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_UNBIND_REQ, DL_BADPRIM, 0);
	return;
    }

    if (slp->sl_state != DL_IDLE) {
	psdlerrorack(wq, mp, DL_UNBIND_REQ, DL_OUTSTATE, 0);
	return;
    }

    CMN_ERR(3, (CE_CONT, "UNbinding sap = %x\n", slp->sl_sap));

    slp->sl_state = DL_UNBOUND;

    putnextctl1(RD(wq), M_FLUSH, FLUSHRW);

    psdlokack(wq, mp, DL_UNBIND_REQ);

#ifdef IPQ
    myrisetipq(slp->sl_myrip);
#endif
}

static void
myriireq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    dl_info_ack_t *dlip;
    struct myridladdr *dlap;
    struct medium_addr *ep;
    int size;

    CMN_ERR(3, (CE_CONT, "myriireq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_INFO_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_INFO_REQ, DL_BADPRIM, 0);
	return;
    }

    /*
     * Exchange current msg for a DL_INFO_ACK.
     */
    size = sizeof(dl_info_ack_t) + DEVICE_ADDRL + MEDIUM_ADDRL;
    if ((mp = psexchange(wq, mp, size, M_PCPROTO, DL_INFO_ACK)) == NULL)
	return;

    /*
     * Fill in the DL_INFO_ACK fields and reply.
     */
    dlip = (dl_info_ack_t *) mp->b_rptr;

    bcopy((caddr_t) & myriinfoack, (caddr_t) dlip, sizeof(myriinfoack));

    dlip->dl_current_state = slp->sl_state;
    dlap = (struct myridladdr *) (mp->b_rptr + dlip->dl_addr_offset);
    dlap->dl_sap = (u_short) slp->sl_sap;
    if (slp->sl_myrip) {
      bcopy_addr((caddr_t) & slp->sl_myrip->myri_ouraddr,  (caddr_t) & dlap->dl_phys);
    } else {
	bzero((caddr_t) & dlap->dl_phys, MEDIUM_ADDRL);
    }
    ep = (struct medium_addr *) (mp->b_rptr + dlip->dl_brdcst_addr_offset);
    bcopy_addr((caddr_t) & medbroadcastaddr, (caddr_t) ep);
    qreply(wq, mp);
}

static void
myriponreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;

    CMN_ERR(3, (CE_CONT, "myriponreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_PROMISCON_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_PROMISCON_REQ, DL_BADPRIM, 0);
	return;
    }

    switch (((dl_promiscon_req_t *) mp->b_rptr)->dl_level) {
    case DL_PROMISC_PHYS:
	slp->sl_flags |= SLALLPHYS;
	break;

    case DL_PROMISC_SAP:
	slp->sl_flags |= SLALLSAP;
	break;

    case DL_PROMISC_MULTI:
	slp->sl_flags |= SLALLMULTI;
	break;

    default:
	psdlerrorack(wq, mp, DL_PROMISCON_REQ,
		     DL_NOTSUPPORTED, 0);
	return;
    }

    if (slp->sl_myrip) {
      (void) myriinit(slp->sl_myrip);
    }

    if (slp->sl_myrip) {
#ifdef IPQ
      myrisetipq(slp->sl_myrip); 
#endif
    }
    

    psdlokack(wq, mp, DL_PROMISCON_REQ);
}

static void
myripoffreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    int flag;

    CMN_ERR(3, (CE_CONT, "myripoffreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_PROMISCOFF_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_PROMISCOFF_REQ, DL_BADPRIM, 0);
	return;
    }

    switch (((dl_promiscoff_req_t *) mp->b_rptr)->dl_level) {
    case DL_PROMISC_PHYS:
	flag = SLALLPHYS;
	break;

    case DL_PROMISC_SAP:
	flag = SLALLSAP;
	break;

    case DL_PROMISC_MULTI:
	flag = SLALLMULTI;
	break;

    default:
	psdlerrorack(wq, mp, DL_PROMISCOFF_REQ,
		     DL_NOTSUPPORTED, 0);
	return;
    }

    if ((slp->sl_flags & flag) == 0) {
	psdlerrorack(wq, mp, DL_PROMISCOFF_REQ, DL_NOTENAB, 0);
	return;
    }

    slp->sl_flags &= ~flag;
    if (slp->sl_myrip)
	(void) myriinit(slp->sl_myrip);

    if (slp->sl_myrip){
#ifdef IPQ
      myrisetipq(slp->sl_myrip);
#endif
    }

    psdlokack(wq, mp, DL_PROMISCOFF_REQ);
}

static void
myriemreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    union DL_primitives *dlp;
    struct medium_addr *addrp;
    int off;
    int len;

    CMN_ERR(3, (CE_CONT, "myriemreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_ENABMULTI_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_ENABMULTI_REQ, DL_BADPRIM, 0);
	return;
    }

    if (slp->sl_state == DL_UNATTACHED) {
	psdlerrorack(wq, mp, DL_ENABMULTI_REQ, DL_OUTSTATE, 0);
	return;
    }

    dlp = (union DL_primitives *) mp->b_rptr;
    len = dlp->enabmulti_req.dl_addr_length;
    off = dlp->enabmulti_req.dl_addr_offset;
    addrp = (struct medium_addr *) (mp->b_rptr + off);

    if ((len != MEDIUM_ADDRL) ||
	!MBLKIN(mp, off, len)) {
	psdlerrorack(wq, mp, DL_ENABMULTI_REQ, DL_BADADDR, 0);
	return;
    }

    if ((slp->sl_mccount + 1) >= DEVMAXMC) {
	psdlerrorack(wq, mp, DL_ENABMULTI_REQ, DL_TOOMANY, 0);
	return;
    }

    /*
     * Allocate table on first request.
     */
    if (slp->sl_mctab == NULL)
	if ((slp->sl_mctab = (struct medium_addr *)
	     kmem_alloc(DEVMCALLOC, KM_NOSLEEP)) == NULL) {
	    psdlerrorack(wq, mp, DL_ENABMULTI_REQ,
			 DL_SYSERR, ENOMEM);
	    return;
	}

    slp->sl_mctab[slp->sl_mccount++] = *addrp;

    (void) myriinit(slp->sl_myrip);
    psdlokack(wq, mp, DL_ENABMULTI_REQ);
}

static void
myridmreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    union DL_primitives *dlp;
    struct medium_addr *addrp;
    int off;
    int len;
    int i;

    CMN_ERR(3,  (CE_CONT, "myridmreq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_DISABMULTI_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_DISABMULTI_REQ, DL_BADPRIM, 0);
	return;
    }

    if (slp->sl_state == DL_UNATTACHED) {
	psdlerrorack(wq, mp, DL_DISABMULTI_REQ, DL_OUTSTATE, 0);
	return;
    }

    dlp = (union DL_primitives *) mp->b_rptr;
    len = dlp->disabmulti_req.dl_addr_length;
    off = dlp->disabmulti_req.dl_addr_offset;
    addrp = (struct medium_addr *) (mp->b_rptr + off);

    if ((len != MEDIUM_ADDRL) || !MBLKIN(mp, off, len)) {
	psdlerrorack(wq, mp, DL_DISABMULTI_REQ, DL_BADADDR, 0);
	return;
    }

    /*
     * Find the address in the multicast table for this Stream and delete
     * it by shifting all subsequent multicast table entries over one.
     */
    for (i = 0; i < slp->sl_mccount; i++) {
	/* all this is architecture specific */
	if (bcmp_addr((caddr_t) addrp, (caddr_t) & slp->sl_mctab[i]) == 0) {
	    bcopy((caddr_t) & slp->sl_mctab[i + 1],
		  (caddr_t) & slp->sl_mctab[i],
		  ((slp->sl_mccount - i) *
		   sizeof(struct medium_addr)));
	    slp->sl_mccount--;
	    (void) myriinit(slp->sl_myrip);
	    psdlokack(wq, mp, DL_DISABMULTI_REQ);
	    break;
	}
    }
}

static void
myripareq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    union DL_primitives *dlp;
    int type;
    struct myri *myrip;
    struct medium_addr addr;

    CMN_ERR(3, (CE_CONT, "myripareq called\n"));

    slp = (struct myristr *) wq->q_ptr;

    if (MBLKL(mp) < DL_PHYS_ADDR_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_PHYS_ADDR_REQ, DL_BADPRIM, 0);
	return;
    }

    dlp = (union DL_primitives *) mp->b_rptr;
    type = dlp->physaddr_req.dl_addr_type;
    myrip = slp->sl_myrip;

    if (myrip == NULL) {
	psdlerrorack(wq, mp, DL_PHYS_ADDR_REQ, DL_OUTSTATE, 0);
	return;
    }

    switch (type) {
    case DL_FACT_PHYS_ADDR:
	CMN_ERR(3, (CE_CONT, "        DL_FACT_PHYS_ADDR:\n"));
	bcopy_addr((caddr_t) & addr, (caddr_t) & myrip->myri_ouraddr);
	break;

    case DL_CURR_PHYS_ADDR:
	CMN_ERR(3, (CE_CONT, "        DL_CURR_PHYS_ADDR:\n"));
	bcopy_addr((caddr_t) & myrip->myri_ouraddr, (caddr_t) & addr);
	break;

    default:
	CMN_ERR(3, (CE_CONT, "        default:\n"));
	psdlerrorack(wq, mp, DL_PHYS_ADDR_REQ,
		     DL_NOTSUPPORTED, 0);
	return;
    }

    psdlphysaddrack(wq, mp, (caddr_t) & addr, MEDIUM_ADDRL);
}


static void
myrispareq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    union DL_primitives *dlp;
    int off;
    int len;
    struct medium_addr *addrp;
    struct myri *myrip;

    CMN_ERR(3, (CE_CONT, "myrispareq called\n"));


    slp = (struct myristr *) wq->q_ptr;



    if (MBLKL(mp) < DL_SET_PHYS_ADDR_REQ_SIZE) {
	psdlerrorack(wq, mp, DL_SET_PHYS_ADDR_REQ, DL_BADPRIM, 0);
	return;
    }

    dlp = (union DL_primitives *) mp->b_rptr;
    len = dlp->set_physaddr_req.dl_addr_length;
    off = dlp->set_physaddr_req.dl_addr_offset;

    if (!MBLKIN(mp, off, len)) {
	psdlerrorack(wq, mp, DL_SET_PHYS_ADDR_REQ, DL_BADPRIM, 0);
	return;
    }

    addrp = (struct medium_addr *) (mp->b_rptr + off);

    /*
     * If length of address isn't right or the address specified is a
     * multicast or broadcast address. then return error.
     */
    if ((addrp->medaddr[0] & 0x1) ||
	(!((len == MEDIUM_ADDRL) || (len == DEVICE_ADDRL)))) {
	/* multicast or broadcast address */
	/* or length is bad */
	psdlerrorack(wq, mp, DL_SET_PHYS_ADDR_REQ, DL_BADADDR, 0);
	return;
    }

    /*
     * Error if this stream is not attached to a device.
     */
    if ((myrip = slp->sl_myrip) == NULL) {
	psdlerrorack(wq, mp, DL_SET_PHYS_ADDR_REQ, DL_OUTSTATE, 0);
	return;
    }

    /*
     * Set new interface local address and re-init device. This is
     * destructive to any other streams attached to this device.
     */
    bcopy_addr((caddr_t) addrp, (caddr_t) & myrip->myri_ouraddr);

    CMN_ERR_PRINT (0, (CE_CONT, "myrispareq: got new PHYSICAL addr %x:%x:%x:%x:%x:%x\n",
		      myrip->myri_ouraddr.medaddr[0],
		      myrip->myri_ouraddr.medaddr[1],
		      myrip->myri_ouraddr.medaddr[2],
		      myrip->myri_ouraddr.medaddr[3],
		      myrip->myri_ouraddr.medaddr[4],
		      myrip->myri_ouraddr.medaddr[5]));

    /* GM_update addresses for ARP?? */


    (void) myriinit(slp->sl_myrip);

    psdlokack(wq, mp, DL_SET_PHYS_ADDR_REQ);
}

static void
myriudreq(queue_t * wq, mblk_t * mp)
{
    struct myristr *slp;
    register struct myri *myrip;
    register dl_unitdata_req_t *dludp;
    register mblk_t *nmp, *nmp2;
    struct myridladdr *dlap;
    struct medium_header *headerp;
    int off, len;
    int header_len = sizeof(struct medium_header);

    CMN_ERR(3, (CE_CONT, "myriudreq called\n"));

    slp = (struct myristr *) wq->q_ptr;
    myrip = slp->sl_myrip;

    dludp = (dl_unitdata_req_t *) mp->b_rptr;

    off = dludp->dl_dest_addr_offset;
    len = dludp->dl_dest_addr_length;

    CMN_ERR(3, (CE_CONT, "   off = %d   len = %d   MED_ADDRL = %d DEVICE_ADRL = %d\n",
		off, len, MEDIUM_ADDRL, DEVICE_ADDRL));

    /*
     * Validate destination address format.
     */
    if (!MBLKIN(mp, off, len) || (len != DEVICE_ADDRL)) {
	psdluderrorind(wq, mp, (u_char *) (mp->b_rptr + off), len,
		       DL_BADADDR, 0);
	CMN_ERR(0, (CE_CONT, "  something wrong with the addrlen  MBLKIN = 0x%x\n",
		    MBLKIN(mp, off, len)));
	return;
    }

    /*
     * Error if no M_DATA follows.
     */
    nmp = mp->b_cont;
    if (nmp == NULL) {
	psdluderrorind(wq, mp, (u_char *) (mp->b_rptr + off), len,
		       DL_BADDATA, 0);
	CMN_ERR(0, (CE_CONT, "  nmp == NULL?\n"));
	return;
    }

    dlap = (struct myridladdr *) (mp->b_rptr + off);

    /*
     * Create medium header by either prepending it onto the next mblk if
     * possible, or reusing the M_PROTO block if not.
     */

    /* BUG HERE FIX - what if DB_REF(nmp) > 1!! */
#if 0
    if ((DB_REF(nmp) == 1) && 
	(MBLKHEAD(nmp) >= header_len) &&
	(((ulong) nmp->b_rptr & 0x1) == 0)) {
	nmp->b_rptr -= header_len;
	headerp = (struct medium_header *) nmp->b_rptr;
	bcopy_addr((caddr_t) & dlap->dl_phys, (caddr_t) & headerp->med_dhost);
	bcopy_addr((caddr_t) & myrip->myri_ouraddr, (caddr_t) & headerp->med_shost);
	/*  headerp->med_type = dlap->dl_sap; */
	freeb(mp);
	mp = nmp;
    } else 
#endif

      {
	/* XXX FIX BOB check it out */
	if (DB_REF(mp) == 1)
	  nmp2 = mp;
	else {
	  nmp2 = copymsg(mp);
	  if (nmp2 == NULL) {
	    CMN_ERR(3, (CE_CONT, "cannot CopyMsg in FastPath\n"));
	    return;
	  }
	  freemsg(mp);
	}
	
	DB_TYPE(nmp2) = M_DATA;
	headerp = (struct medium_header *) nmp2->b_rptr;
	nmp2->b_wptr = nmp2->b_rptr + header_len;
	bcopy_addr((caddr_t) & dlap->dl_phys, (caddr_t) & headerp->med_dhost);
	bcopy_addr((caddr_t) & myrip->myri_ouraddr, (caddr_t) & headerp->med_shost);
	/*  headerp->med_type = dlap->dl_sap; */
	mp = nmp2;
    }

    mutex_enter(&slp->sl_lock);
    put_ether_type(headerp, slp->sl_sap);
    mutex_exit(&slp->sl_lock);
    /*
     * Transmit it.
     */

    (void) myristart(wq, mp, myrip);

}

/*
 * Start xmit on any msgs previously enqueued on any write queues.
 */
void
myriwenable(struct myri *myrip)
{
    struct myristr *slp = myristrup;
    queue_t *wq;

    CMN_ERR(3, (CE_CONT, "myriwenable called\n"));

    /*
     * Order of wantw accesses is important.
     */
    do {
	myrip->myri_wantw = 0;
	for (; slp; slp = slp->sl_nextp)
	    if ((wq = WR(slp->sl_rq))->q_first)
		qenable(wq);
    } while (myrip->myri_wantw);
}

/*
 * XXX Test upstream destination sap and address match.
 */
struct myristr *
myriaccept(struct myristr *slp, struct myri *myrip,
	   int type, struct medium_addr *addrp)
{
    int sap;
    int flags;

    CMN_ERR(3, (CE_CONT, "myriaccept called\n"));

    for (; slp; slp = slp->sl_nextp) {
	sap = slp->sl_sap;
	flags = slp->sl_flags;
	CMN_ERR(3, (CE_CONT, "   sap = %x  type = %x  flags = %x\n",
		    sap, type, flags));

	if ((slp->sl_myrip == myrip) && MYRISAPMATCH(sap, type, flags)) {
	    /* bcmp and family are architecture specific */
	    if ((bcmp_addr((caddr_t) addrp, (caddr_t) & myrip->myri_ouraddr) == 0) ||
		(bcmp_addr((caddr_t) addrp, (caddr_t) & medbroadcastaddr) == 0) ||
		(flags & SLALLPHYS) || 
		myrimcmatch((struct myristr *) slp, (struct medium_addr *) addrp)) {
		CMN_ERR(3, (CE_CONT, "myriaccept: found a match slp = %x sap = %x\n",
			    slp, sap));
		return (slp);
	    }
	}
    }

    return (NULL);
}

/*
 * XXX Send packet upstream.
 * Assume mp->b_rptr points to medium_head`er.
 */
void
myrisendup(struct myri *myrip, mblk_t * mp)
{
    int type;
    struct medium_addr *dhostp, *shostp;
    struct myristr *slp, *next_slp;
    mblk_t *nmp;
    ulong isgroupaddr = 0;

    CMN_ERR(3, (CE_CONT, "myrisendup called\n"));

    dhostp = &((struct medium_header *) mp->b_rptr)->med_dhost;
    shostp = &((struct medium_header *) mp->b_rptr)->med_shost;
    type = get_ether_type((struct medium_header *) mp->b_rptr);

    /* if multicast or broadcast, set groupaddr = TRUE */
    if (dhostp->medaddr[0] & 0x1) {
	isgroupaddr = 1;
	CMN_ERR(3, (CE_CONT, "      is GROUP address %x:%x:%x:%x:%x:%x\n",
		    dhostp->medaddr[0],
		    dhostp->medaddr[1],
		    dhostp->medaddr[2],
		    dhostp->medaddr[3],
		    dhostp->medaddr[4],
		    dhostp->medaddr[5]));
    }

    /*
     * While holding a reader lock on the linked list of streams
     * structures, attempt to match the address criteria for each stream
     * and pass up the  DL_UNITDATA_IND.
     */

    rw_enter(&myristruplock, RW_READER);

    if ((slp = myriaccept(myristrup, myrip, type, dhostp)) == NULL) {
	rw_exit(&myristruplock);
	freemsg(mp);
	CMN_ERR(3, (CE_CONT, "    dropping message slp = NULL\n"));
	return;
    }


    /*
     * Loop on matching open streams until (*acceptfunc)() returns NULL.
     */
#if 0
    do {
      CMN_ERR(3, (CE_CONT, "inside loop slp = %x  sap = %x\n",
		  slp, slp->sl_sap));

      next_slp = 0;
      if (slp->sl_nextp) 
	next_slp = myriaccept(slp->sl_nextp, myrip, type, dhostp);

      if (next_slp) {
	if (!(nmp = dupmsg(mp))) {
	  CMN_ERR_PRINT (0, (CE_CONT, "    CAN'T dup message slp = %x  sap = %x\n",
			     slp, slp->sl_sap));
	  myrip->myri_ierrors++;
	  continue;
	}
      } else{
	nmp = mp;	
      }

      if (slp->sl_flags & SLRAW) {
	CMN_ERR(3, (CE_CONT, "    calling putq(%x) SLRAW1 sap = %x\n",
		    slp->sl_rq, slp->sl_sap));
	do_putnext(slp, nmp);
      } else {
	if ((nmp = myriaddudind(myrip, nmp, shostp,
				dhostp, type, isgroupaddr))) {
	  CMN_ERR(3, (CE_CONT,
		      "    calling putq(%x) after myriaddudind1 sap = %x\n",
		      slp->sl_rq, slp->sl_sap));
	  do_putnext(slp, nmp);
	} else {
	  CMN_ERR_PRINT (0, (CE_CONT, "myrisendup **** myriaddudind failed\n"));
	  /*
	   * message was freed in myriaddudind
	   */
	  myrip->myri_ierrors++;
	}
      }
      slp = next_slp;
    } while (slp);
#endif

	/*
	 * Loop on matching open streams until (*acceptfunc)() returns NULL.
	 */
    for (;(next_slp = myriaccept(slp->sl_nextp, myrip, type, dhostp)); slp = next_slp)
      if (canput(slp->sl_rq->q_next)) {
	if ((nmp = dupmsg(mp))) {
	  if ((slp->sl_flags & SLFAST) && !isgroupaddr) {
	    nmp->b_rptr += sizeof (struct medium_header);
	    (void) putnext(slp->sl_rq, nmp);
	  } else if (slp->sl_flags & SLRAW)
	    (void) putnext(slp->sl_rq, nmp);
	  else if ((nmp = myriaddudind(myrip, nmp, shostp,
				       dhostp, type, isgroupaddr)))
	    (void) putnext(slp->sl_rq, nmp);
	} else {
	  CMN_ERR_PRINT (0, (CE_CONT, "    CAN'T dup message slp = %x  sap = %x\n",
			     slp, slp->sl_sap));
	  myrip->myri_ierrors++;
	}
      } 

    /*
     * Do the last one.
     */
    if (canput(slp->sl_rq->q_next)) {
      if ((slp->sl_flags & SLFAST) && !isgroupaddr) {
	mp->b_rptr += sizeof (struct medium_header);
	(void) putnext(slp->sl_rq, mp);
      } else if (slp->sl_flags & SLRAW)
	(void) putnext(slp->sl_rq, mp);
      else if ((mp = myriaddudind(myrip, mp, shostp, dhostp,
				  type, isgroupaddr)))
	(void) putnext(slp->sl_rq, mp);
    } else {
      freemsg(mp);
      myrip->myri_ierrors++;
    }

    rw_exit(&myristruplock);
}



#ifdef FLOW_CONTROL

/**************************   myrirsrv()   ************************/
static int
myrirsrv(queue_t * rq)
{
    mblk_t *mp = NULL;
    struct myristr *slp;
    struct myri *myrip;
    queue_t *qptr;
    int counter = 0;

    CMN_ERR(3) (CE_CONT, "myriRSRV(0x%x)\n", rq);

    rw_enter(&myristruplock, RW_READER);

    slp = (struct myristr *) rq->q_ptr;
    if (!slp) {
	goto rsrv_cleanup;
    }
    myrip = slp->sl_myrip;
    if (!myrip) {
	goto rsrv_cleanup;
    }

    /*
     * messages that are queued have destination queue at the front
     */
    while ((mp = getq(rq))) {

	qptr = *((queue_t **) mp->b_rptr);
	if (!qptr->q_ptr) {
	    goto rsrv_cleanup;
	}

	mp->b_rptr += sizeof(queue_t *);

	if (!do_putnext(qptr, mp)) {
	    /* message was queued again */
	    CMN_ERR_PRINT (0, (CE_CONT, "myriRSRV queued again \n"));
	    CMN_ERR_PRINT (0, (CE_CONT, "\t q = 0x%x  msg = 0x%x q_siz=%d qn_siz=%d, cnt=%d\n",
			qptr, mp, qsize(rq), qsize(rq->q_next), ++counter));

	    rw_exit(&myristruplock);
	    return (0);
	}
	else {
	    CMN_ERR_PRINT (0, (CE_CONT, "myriRSRV clr(%d) msg q = 0x%x q_siz=%d qn_siz=%d\n",
			       qptr, qsize(rq), qsize(rq->q_next), ++counter));
	}
    }

    rw_exit(&myristruplock);

    return (0);

rsrv_cleanup:
    CMN_ERR_PRINT (0, (CE_CONT, "myriRSRV flush msg queue= 0x%x & disable\n", rq));

    if (mp) {
	freemsg(mp);
    }
    flushq(rq, FLUSHALL);
    noenable(rq);

    rw_exit(&myristruplock);
    return (0);
}

#endif							/* FLOW_CONTROL */

/*
 * Prefix msg with a DL_UNITDATA_IND mblk and return the new msg.
 */
mblk_t *
myriaddudind(struct myri * myrip, mblk_t * mp,
	     struct medium_addr * shostp, struct medium_addr * dhostp,
	     int type, ulong isgroupaddr)
{
  dl_unitdata_ind_t *dludindp;
  struct myridladdr *dlap;
  mblk_t *nmp;
  int size;
  int dl_len = sizeof(dl_unitdata_ind_t);
    
#define TMP_OFFSET	(2 * sizeof(ulong))

  CMN_ERR(3, (CE_CONT, "myriaddudind called\n"));
  
  /* strip off the Ethernet header */
  adjmsg(mp, sizeof(struct medium_header));
  
  /*
   * Allocate an M_PROTO mblk for the DL_UNITDATA_IND.
   */
  size = dl_len + TMP_OFFSET + 2 * DEVICE_ADDRL;
  nmp = allocb(MYRIHEADROOM + size, BPRI_HI);
  if (nmp == NULL) {
    myrip->myri_ierrors++;
    CMN_ERR_PRINT (0, (CE_CONT, "myriaddudind: allocb failed"));
    freemsg(mp);
    return (NULL);
  }
  DB_TYPE(nmp) = M_PROTO;
  nmp->b_wptr = nmp->b_datap->db_lim;
  nmp->b_rptr = nmp->b_wptr - size;
  
  /*
   * Construct a DL_UNITDATA_IND primitive.
   */
  dludindp = (dl_unitdata_ind_t *) nmp->b_rptr;
  dludindp->dl_primitive = DL_UNITDATA_IND;
  dludindp->dl_dest_addr_length = DEVICE_ADDRL;
  dludindp->dl_dest_addr_offset = dl_len + TMP_OFFSET;
  dludindp->dl_src_addr_length = DEVICE_ADDRL;
  dludindp->dl_src_addr_offset = dl_len + TMP_OFFSET + DEVICE_ADDRL;
  dludindp->dl_group_address = isgroupaddr;
  
  dlap = (struct myridladdr *) (nmp->b_rptr + dl_len + TMP_OFFSET);
  bcopy_addr((caddr_t) dhostp, (caddr_t) & dlap->dl_phys);
  dlap->dl_sap = (u_short) type;
  
  dlap = (struct myridladdr *) (nmp->b_rptr + dl_len + TMP_OFFSET + DEVICE_ADDRL);
  bcopy_addr((caddr_t) shostp, (caddr_t) & dlap->dl_phys);
  dlap->dl_sap = (u_short) type;

  /*
   * Link the M_PROTO and M_DATA together.
   */
  nmp->b_cont = mp;
  return (nmp);
}

/*
 * Return TRUE if the given multicast address is one
 * of those that this particular Stream is interested in.
 */
static
int
myrimcmatch(struct myristr *slp, struct medium_addr *addrp)
{
    struct medium_addr *mctab;
    int mccount;
    int i;

    CMN_ERR(3, (CE_CONT, "myrimcmatch called\n"));

    /*
     * XXX Return FALSE if not a multicast address.
     */

    if (!(addrp->medaddr[0] & 0x1)) {
	return (0);
    }

    /*
     * Check if all multicasts have been enable for this Stream
     */
    if (slp->sl_flags & SLALLMULTI) {
	return (1);
    }

    /*
     * Return FALSE if no multicast addresses enable for this Stream.
     */
    if (slp->sl_mccount == 0) {
	return (0);
    }

    /*
     * Otherwise, find it in the table.
     */

    mccount = slp->sl_mccount;
    mctab = slp->sl_mctab;

    for (i = 0; i < mccount; i++)
	if (!bcmp_addr((caddr_t) addrp, (caddr_t) & mctab[i]))
	    return (1);

    return (0);
}

/*
 * XXX Initialize chip and driver.
 * Return 0 on success, nonzero on error.
 */
static int
myriinit(struct myri *myrip)
{

    CMN_ERR(3, (CE_CONT, "myriinit called\n"));

    rw_enter(&myristruplock, RW_WRITER);

    myrip->myri_flags = 0;
    myrip->myri_wantw = 0;
    myrip->myri_flags |= MYRIRUNNING;

    myriwenable(myrip);

/* done: */
    rw_exit(&myristruplock);
    CMN_ERR(3, (CE_CONT, "myriinit returning %d\n",
		!(myrip->myri_flags & MYRIRUNNING)));
    return (!(myrip->myri_flags & MYRIRUNNING));
}

/*
 * XXX Un-initialize (STOP) DEVICE
 */
static void
myriuninit(struct myri *myrip)
{
    CMN_ERR(3,  (CE_CONT, "myriUNinit called\n"));

    /*
     * XXX Allow time  for pending xmit's to complete.
     */

    /*
     * Grab mutex's
     */

    myri_mutex_enter(myrip);

    myrip->myri_flags &= ~MYRIRUNNING;

    /*
     * XXX Stop the chip.
     */

    /* turn interrupts off and hold the lanai in reset */

    /* GM_stop (close??) */

    /*
     * Release mutex's
     */

    myri_mutex_exit(myrip);

    return;
}

/*ARGSUSED */
void
myrierror(dev_info_t * dip, const char *fmt,...)
{

    char name[16];
    char buf[256];
    va_list ap;

    if (dip) {
	(void) sprintf(name, "%s [%d]", ddi_get_name(dip),
		       ddi_get_instance(dip));
    }
    else {
	(void) sprintf(name, MYRINAME);
    }

    va_start(ap, fmt);
    (void) vsprintf(buf, fmt, ap);
    va_end(ap);

    CMN_ERR_PRINT (0, (CE_CONT, "%s:\t%s", name, buf));

}






/* end of dlpi.c */






