/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997, 1998, 1999 by Myricom, Inc.                 *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/


/***************************************************
 * Common driver header files                      *
 ***************************************************/
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/map.h>
#include <sys/buf.h>
#include <sys/vm.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/kernel.h>
#include <sys/devio.h>
#include <sys/malloc.h>
#include <hal/cpuconf.h>
#include <sys/exec.h>
#include <io/common/devdriver.h>
#include <sys/sysconfig.h>
#include <kern/kalloc.h>
#include <sys/mman.h>
#include <kern/thread.h>
/*****************************
  network driver includes
 ******************************/
#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/if_ether.h>
#include <net/ether_driver.h>
#include <sys/mbuf.h>
#include <io/dec/pci/pci.h>
#include <io/dec/netif/lan_common.h>
#include <machine/cpu.h>

#include "gm_internal.h"
#include "gm_arch.h"
#include "gm_call_trace.h"
#include "gm_page_hash.h"
#include "gm_lanai.h"

#define GM_USE_IPL 1
#define GM_USE_THREADS 1	/* 0 or 1 seems to busy-wait and eat CPU time */
#define GM_DO_SIMPLE_LOCK	1

extern int gd_attach(int unit);
static int register_configuration();
static int register_major_number();

#ifdef USE_ZLIB
#include "zlib.h"
#endif

#ifndef NULL
#define NULL (void *)0
#endif

#if GM_DEBUG
int gm_print_level = GM_PRINT_LEVEL;
#endif

int num_gm = 0;

#define MAX_NMYRI 4
#define MAX_MINOR_SIZE 64

#define GET_MINOR(minor_id) (minor_id - (MAX_NMYRI*2 - num_gm*2))
#define EXTERN_MINOR(minor_id) (minor_id + (MAX_NMYRI*2 - num_gm*2))
#define INVALID_MINOR(minor_id) ((minor_id +(MAX_NMYRI*2 - num_gm*2)) >= MAX_MINOR_SIZE)


#define GM_ARCH_DEBUG_SLEEP_WAKE	0

/***************************************************
 *        !!* BUS-SPECIFIC CHANGE *!!              *
 ***************************************************
 * If you are writing a device driver to           *
 * operate on more than one bus, you must include  *
 * the correct bus-specific header file.  The      *
 * following list shows the correct header files   *
 * for the EISA, ISA, PCI, and TURBOchannel buses: *
 *                                                 *
 *      #include <io/dec/eisa/eisa.h>              *
 *      #include <io/dec/eisa/isa.h>               *
 *      #include <io/dec/pci/pci.h>                *
 *      #include <io/dec/tc/tc.h>                  *
 *                                                 *
 * Because the fictitious MYRI device controller   *
 * is connected to a TURBOchannel bus, the tc.h    *
 * file is included in the /dev/gm driver for    *
 * illustrative purposes.  If your driver operates *
 * on all of these buses, you would include all of *
 * these bus-specific header files.                *
 ***************************************************/

/***************************************************
 * The /dev/gm driver's register_major_number    *
 * interface uses the following #defines when      *
 * obtaining a major number.                       *
 ***************************************************/




#define NO_DEV -1
#define MAJOR_INSTANCE 1




/***************************************************
 *      Data structure sizing approach             *
 ***************************************************
 *                                                 *
 * The following #define will be used to allocate  *
 * data structures needed by the /dev/gm driver. *
 * There can be at most 8 instances of the MYRI    *
 * device controller on the system.  This means    *
 * that MAX_NMYRI is the maximum number of         *
 * controllers that the /dev/gm driver can       *
 * support.  This is a small number of instances   *
 * of the driver and the data structures           *
 * themselves are not large, so it is acceptable   *
 * to allocate for the maximum configuration.      *
 *                                                 *
 ***************************************************/




/***************************************************
 * Declarations Section                            *
 ***************************************************/




/***************************************************
 *        Bits used in read/write operations       * 
 ***************************************************/




#define DN_RESET 0001			/* Device ready for data transfer */
#define DN_ERROR 0002			/* Indicate error */

/***************************************************
 *       Defines for softc structure               * 
 ***************************************************/




#define DN_OPEN  1				/* Device open bit */
#define DN_CLOSE 0				/* Device close bit */

/***************************************************
 *     Forward declarations of driver interfaces   *
 ***************************************************/



/* Function prototypes and externs here  */

static int gm_osf1_open(dev_t dev, int flag, int format);
static int gm_osf1_intr(gm_instance_state_t * is);
static int gm_osf1_close(dev_t dev, int flag, int format);
static int gm_osf1_ioctl(dev_t dev, unsigned int cmd, caddr_t data, int flag);
int gm_osf1_if_ioctl(register struct ifnet *ifp, unsigned int cmd, caddr_t data);
static int gm_osf1_mmap(dev_t dev, off_t off, int prot);
static int gm_osf1_probe(struct pci_config_hdr *pci_cfg_hdr, struct controller *ctlr);
static int gm_osf1_cattach(struct controller *);
int gm_osf1_dattach(struct controller *);

static int gm_osf1_ctlr_unattach(struct bus *bus, struct controller *ctlr);
struct controller *gm_osf1_get_controller(int unit);

static void callback_register_configuration(int point, int order,
									 ulong argument, ulong event_arg);
static void callback_register_major_number(int point, int order,
									ulong argument, ulong event_arg);

/***************************************************
 *      controller and driver Structures           * 
 ***************************************************/


static gm_instance_state_t gm_softc[MAX_NMYRI];
static gm_instance_state_t *dev_head;

/***************************************************
 * Declare an array of pointers to controller      *
 * structures                                      *
 ***************************************************/

struct controller *gminfo[MAX_NMYRI];

/***************************************************
 * Declare and initialize driver structure         *
 ***************************************************/

/*static struct driver  - should this really be static??? drew */
struct driver gmdriver =
{
	gm_osf1_probe,				/* probe */
	0,							/* slave */
	gm_osf1_cattach,			/* cattach */
	0,							/* dattach */
	0,							/* go */
	0,							/* addr_list */
	0,							/* dev_name */
	0,							/* dev_list */
	"gm",						/* ctlr_name */
	gminfo,						/* ctlr_list */
	0,							/* xclu */
	0,							/* addr1_size */
	0,							/* addr1_atype */
	0,							/* addr2_size */
	0,							/* addr2_atype */
	gm_osf1_ctlr_unattach,		/* ctlr_unattach */
	0							/* dev_unattach */
};



/***************************************************
 * The handler_add interface is used to register   *
 * the interrupt handler for the /dev/gm driver. *
 * The gm_id_t array contains IDs that are used  *
 * to enable, disable, and deregister the          *
 * interrupt handlers.                             *
 ***************************************************/



ihandler_id_t *gm_id_t[MAX_NMYRI];


/***************************************************
 * The following list describes the use of the     *
 * gm_is_dynamic, callback_return_status, and    *
 * num_gm variables:                             *
 *                                                 *
 *  o gm_is_dynamic                              *
 *                                                 *
 *    The gm_is_dynamic variable will be used to *
 *    control any differences in tasks performed   *
 *    by the statically or dynamically configured  *
 *    /dev/gm driver.  Implementing a device     *
 *    driver to handle a static or dynamic         *
 *    configuration request gives customers        *
 *    maximum flexibility in how they configure    *
 *    the driver into the kernel.                  *
 *                                                 *
 *  o callback_return_status                       *
 *                                                 *
 *    The callback_return_status variable is used  *
 *    by the callback_register_major_number        *
 *    interface to determine if a previous failure *
 *    has occurred in statically configuring the   *
 *    /dev/gm driver.                            *
 *                                                 *
 *  o num_gm                                     *
 *                                                 *
 *    The num_gm variable is used to count the   *
 *    number of controllers probed.  The gmprobe *
 *    interface increments this variable each time *
 *    it probes a MYRI controller.                 *
 ***************************************************/



static int gm_is_dynamic = 0;

static int callback_return_status = ESUCCESS;
int gm_intr_always_serviced = 0;

static int myri_gm_config = FALSE;	/* State flags indicating driver configured */
static int gm_devno = NO_DEV;	/* No major number assigned yet. */


/***************************************************
 * External function references.  These are needed *
 * for the devsw declaration.                      *
 ***************************************************/


extern int nodev(), nulldev();




/**************************************************
*        !!* BUS SPECIFIC CHANGE *!!              *
***************************************************
* Bus-Specific Option Data                        *
***************************************************
 * You specify bus-specific option data for the   *
 * EISA, ISA, PCI, TURBOchannel, and VME buses    *
 * in the sysconfigtab database.  You supply the  *
 * bus-specific option data for each bus that     *
 * your driver operates on.                       *
 *                                                *
 * Typically, third-party driver writers specify  *
 * the bus-specific option data in a              *
 * sysconfigtab file fragment.  See               *
 * Writing Device Drivers: Tutorial and the       *
 * bus-specific books for information on the      *
 * syntax associated with the following           *
 * bus-specific option  data attribute fields:    *
 *                                                *
 *  o EISA_Option                                 *
 *  o ISA_Option                                  *
 *  o PCI_Option                                  *
 *  o TC_Option                                   *
 *  o VBA_Option                                  *
 ***************************************************/



/**************************************************
 * The following code sets the members of the     *
 * the dsent structure to appropriate  values     *
 * for the /dev/gm driver.  The address of this *
 * structure is passed to the devsw_add           *
 * interface, which registers the I/O services    *
 * interfaces in the dsent table and reserves a   *
 * major number for the /dev/gm driver.         * 
 **************************************************/




static
struct dsent gm_devsw_entry =
{
	gm_osf1_open,				/* d_open */
	gm_osf1_close,				/* d_close */
	nodev,						/* d_strategy */
	nodev,						/* d_read */
	nodev,						/* d_write */
	gm_osf1_ioctl,				/* d_ioctl */
	nodev,						/* d_dump */
	nodev,						/* d_psize */
	nodev,						/* d_stop */
	nodev,						/* d_reset */
	nodev,						/* d_select */
	gm_osf1_mmap,				/* d_mmap */
	0,							/* d_segmap */
	NULL,						/* d_ttys */
	DEV_FUNNEL,					/* d_funnel */
	0,							/* d_bflags */
	0							/* d_cflags */
};

/***************************************************
 *        !!* BUS-SPECIFIC CHANGE *!!              *
 ***************************************************/

/* #define DN_BUSNAME1    "pci" */
#define DN_BUSNAME1    DRIVER_WILDNAME


/***************************************************
 * The following code declares variables that are  *
 * required by the cfgmgr framework.  You use      *
 * these variables as fields in the driver's       *
 * attribute table.  The attribute table for the   *
 * /dev/gm driver is called gm_attributes.         * 
 ***************************************************/
static int majnum = NO_DEV;
static int gmversion = 0;
#if (GM_PRINT_LEVEL >= 1)
static int debug = 1;
static int verbose = 1;
#else
static int debug = 2;
static int verbose = 0;
#endif

#ifdef USE_PCIDBG
extern int pcidbg_flag;
#endif							/* USE_PCIDBG */


static int init_clock = 0;
static unsigned long clock_offset = 0;
/*
 * Proper value for 4.1 card at Colorado, your milage may vary
 */
static unsigned int clock_value = 0x40414041;

static int use_sparse = 0;
static char mcfgname[CFG_ATTR_NAME_SZ] = {0};
static unsigned char unused[300] = {0};
static unsigned char cma_dd[120] = {0};
static unsigned char pci_optiondata[500] = {0};

/***********************************************************************
 * Globals
 ***********************************************************************/

/*
 * Myrinet attribute table
 */

cfg_subsys_attr_t
gm_attributes[] =
{

	/* Fields used in this driver. */
	{
		"Module_Config_Name", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) mcfgname, 2, CFG_ATTR_NAME_SZ, 0
	}
	,
	{
		"CMA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) cma_dd, 0, 300, 0
	}
	,

  /***************************************************
   *        NOTE                                     *
   *                                                 *
   * There are a number of fields (attributes) that  *
   * appear in the sysconfigtab file fragment that   *
   * are not needed by the /dev/gm driver.         *
   * The cfgmgr framework consumes these fields      *
   * (attributes) to perform kernel configuration    *
   * work and to make device special files.  To      *
   * avoid warning messages from the cfgmgr          *
   * framework, you should add these attributes to   *
   * the driver's attribute table.                   *
   *                                                 *
   * The following device driver-related attributes  *
   * are not used by the /dev/gm driver and are    *
   * included in the attributes table to avoid       *
   * the warning messages.                           * 
   ***************************************************/

	{
		"TC_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Module_Path", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Dir", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Subdir", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Block_Subdir", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Char_Subdir", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Major_Req", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Block_Major", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Block_Minor", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Block_Files", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Char_Major", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Char_Minor", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Char_Files", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_User", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Group", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"Device_Mode", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"EISA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"ISA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,
	{
		"VBA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) unused, 0, 300, 0
	}
	,

	/*
	 * Dirk added this..
	 */

	{
		"PCI_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
			(caddr_t) pci_optiondata, 0, 500, 0
	}
	,

  /**********************************************
   * The /dev/gm device driver modifies the   *
   * following attributes during a configure    *
   * or unconfigure operation.  The cfgmgr      *
   * framework uses these attributes to provide *
   * the device special files that device       *
   * drivers need.                              *
   **********************************************/

	{
		"majnum", CFG_ATTR_INTTYPE, CFG_OP_QUERY,
			(caddr_t) & majnum, 0, 512, 0
	}
	,

	{
		"version", CFG_ATTR_INTTYPE, CFG_OP_QUERY,
			(caddr_t) & gmversion, 0, 9999999, 0
	}
	,

	{
		"debug", CFG_ATTR_INTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & debug, 0, 2, 0
	}
	,

	{
		"verbose", CFG_ATTR_INTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & verbose, 0, 2, 0
	}
	,

#ifdef USE_PCIDBG
	{
		"pcidebug", CFG_ATTR_UINTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & pcidbg_flag, 0, 0x0fffffff, 0
	}
	,
#endif							/* USE_PCIDBG */


	/*
	 * options to automatically initialize the clock, should it need it...
	 */
	{
		"init_clock", CFG_ATTR_INTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & init_clock, 0, 1, 0
	}
	,

	{
		"clock_offset", CFG_ATTR_ULONGTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & clock_offset, 0, 1, 0
	}
	,

	{
		"clock_value", CFG_ATTR_UINTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & clock_value, 0, 0xffffffff, 0
	}
	,

	/*
	 * Ideally, we're punting on sparse space stuff
	 */
	{
		"use_sparse", CFG_ATTR_INTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & use_sparse, 0, 1, 0
	}
	,

	{
		"shared_irq_always_ret_serviced", CFG_ATTR_INTTYPE,
			CFG_OP_QUERY | CFG_OP_RECONFIGURE | CFG_OP_CONFIGURE,
			(caddr_t) & (gm_intr_always_serviced), 0, 2, 0
	}
	,

	{
		0, 0, 0, 0, 0, 0, 0
	}
};

/* For the life of me, I couldn't find an interface like perror that
 * takes an error number, so...  */

static char *
perrormsg(int err)
{
	switch (err) {

	 case ESUCCESS:
		 return "ESUCCESS";
	 case EPERM:
		 return "EPERM";
	 case ENOENT:
		 return "ENOENT";
	 case ESRCH:
		 return "ESRCH";
	 case EINTR:
		 return "EINTR";
	 case EIO:
		 return "EIO";
	 case ENXIO:
		 return "ENXIO";
	 case E2BIG:
		 return "E2BIG";
	 case ENOEXEC:
		 return "ENOEXEC";
	 case EBADF:
		 return "EBADF";
	 case ECHILD:
		 return "ECHILD";
	 case EDEADLK:
		 return "EDEADLK";
	 case ENOMEM:
		 return "ENOMEM";
	 case EACCES:
		 return "EACCES";
	 case EFAULT:
		 return "EFAULT";
	 case ENOTBLK:
		 return "ENOTBLK";
	 case EBUSY:
		 return "EBUSY";
	 case EEXIST:
		 return "EEXIST";
	 case EXDEV:
		 return "EXDEV";
	 case ENODEV:
		 return "ENODEV";
	 case ENOTDIR:
		 return "ENOTDIR";
	 case EISDIR:
		 return "EISDIR";
	 case EINVAL:
		 return "EINVAL";
	 case ENFILE:
		 return "ENFILE";
	 case EMFILE:
		 return "EMFILE";
	 case ENOTTY:
		 return "ENOTTY";
	 case ETXTBSY:
		 return "ETXTBSY";
	 case EFBIG:
		 return "EFBIG";
	 case ENOSPC:
		 return "ENOSPC";
	 case ESPIPE:
		 return "ESPIPE";
	 case EROFS:
		 return "EROFS";
	 case EMLINK:
		 return "EMLINK";
	 case EPIPE:
		 return "EPIPE";
	 case EDOM:
		 return "EDOM";
	 case ERANGE:
		 return "ERANGE";
	 case EWOULDBLOCK:
		 return "EWOULDBLOCK";
	 case EINPROGRESS:
		 return "EINPROGRESS";
	 case EALREADY:
		 return "EALREADY";
	 case ENOTSOCK:
		 return "ENOTSOCK";
	 case EDESTADDRREQ:
		 return "EDESTADDRREQ";
	 case EMSGSIZE:
		 return "EMSGSIZE";
	 case EPROTOTYPE:
		 return "EPROTOTYPE";
	 case ENOPROTOOPT:
		 return "ENOPROTOOPT";
	 case EPROTONOSUPPORT:
		 return "EPROTONOSUPPORT";
	 case ESOCKTNOSUPPORT:
		 return "ESOCKTNOSUPPORT";
	 case EOPNOTSUPP:
		 return "EOPNOTSUPP";
	 case EPFNOSUPPORT:
		 return "EPFNOSUPPORT";
	 case EAFNOSUPPORT:
		 return "EAFNOSUPPORT";
	 case EADDRINUSE:
		 return "EADDRINUSE";
	 case EADDRNOTAVAIL:
		 return "EADDRNOTAVAIL";
	 case ENETDOWN:
		 return "ENETDOWN";
	 case ENETUNREACH:
		 return "ENETUNREACH";
	 case ENETRESET:
		 return "ENETRESET";
	 case ECONNABORTED:
		 return "ECONNABORTED";
	 case ECONNRESET:
		 return "ECONNRESET";
	 case ENOBUFS:
		 return "ENOBUFS";
	 case EISCONN:
		 return "EISCONN";
	 case ENOTCONN:
		 return "ENOTCONN";
	 case ESHUTDOWN:
		 return "ESHUTDOWN";
	 case ETOOMANYREFS:
		 return "ETOOMANYREFS";
	 case ETIMEDOUT:
		 return "ETIMEDOUT";
	 case ECONNREFUSED:
		 return "ECONNREFUSED";
	 case ELOOP:
		 return "ELOOP";
	 case ENAMETOOLONG:
		 return "ENAMETOOLONG";
	 case EHOSTDOWN:
		 return "EHOSTDOWN";
	 case EHOSTUNREACH:
		 return "EHOSTUNREACH";
	 case ENOTEMPTY:
		 return "ENOTEMPTY";
	 case EPROCLIM:
		 return "EPROCLIM";
	 case EUSERS:
		 return "EUSERS";
	 case EDQUOT:
		 return "EDQUOT";
	 case ESTALE:
		 return "ESTALE";
	 case EREMOTE:
		 return "EREMOTE";
	 case EBADRPC:
		 return "EBADRPC";
	 case ERPCMISMATCH:
		 return "ERPCMISMATCH";
	 case EPROGUNAVAIL:
		 return "EPROGUNAVAIL";
	 case EPROGMISMATCH:
		 return "EPROGMISMATCH";
	 case EPROCUNAVAIL:
		 return "EPROCUNAVAIL";
	 case ENOLCK:
		 return "ENOLCK";
	 case ENOSYS:
		 return "ENOSYS";
	 case EFTYPE:
		 return "EFTYPE";
	 case ENOMSG:
		 return "ENOMSG";
	 case EIDRM:
		 return "EIDRM";
	 case ENOSR:
		 return "ENOSR";
	 case ETIME:
		 return "ETIME";
	 case EBADMSG:
		 return "EBADMSG";
	 case EPROTO:
		 return "EPROTO";
	 case ENODATA:
		 return "ENODATA";
	 case ENOSTR:
		 return "ENOSTR";
	 case ECLONEME:
		 return "ECLONEME";
	 case EDIRTY:
		 return "EDIRTY";
	 case EDUPPKG:
		 return "EDUPPKG";
	 case EVERSION:
		 return "EVERSION";
	 case ENOPKG:
		 return "ENOPKG";
	 case ENOSYM:
		 return "ENOSYM";
	 case ECANCELED:
		 return "ECANCELED";
	 case EFAIL:
		 return "EFAIL";
	 case EINPROG:
		 return "EINPROG";
	 case EMTIMERS:
		 return "EMTIMERS";
	 case ENOTSUP:
		 return "ENOTSUP";
	 case EAIO:
		 return "EAIO";
	 default:
		 {
			 static char buffer[128];
			 sprintf(buffer, "errno #%d", err);
			 return buffer;
		 }
	}
}

int
gm_configure(cfg_op_t op,
			 cfg_attr_t * indata, size_t indatalen,
			 cfg_attr_t * outdata, size_t outdatalen)
{
	int retval, retval2;
	int i;
	int driver_cfg_state;

  /***************************************************
   *                                                 *
   * MAX_DEVICE_CFG_ENTRIES represents the maximum   *
   * number of attributes that the cfgmgr framework  *
   * configures on behalf of the device driver.      *
   ***************************************************/

#define MAX_DEVICE_CFG_ENTRIES 36

  /***************************************************
   * The cfg_attr_t list passed into the             *
   * gm_configure interface's  indata argument     *
   * contains the strings that are stored in the     *
   * sysconfigtab database for this subsystem (the   *
   * /dev/gm driver).                              *
   ***************************************************/

	cfg_attr_t cfg_buf[MAX_DEVICE_CFG_ENTRIES];


	if (debug) {
		printf("gm_configure: called op = %d (0x%x)\n",op,op);
	}
	switch (op) {

	/***************************************************
     *            Configure (load) the driver.         *
     *                                                 *
     ***************************************************/

	 case CFG_OP_CONFIGURE:

		 if (debug) {
			 printf("gm: gm_configure: CFG_OP_CONFIGURE.\n");
		 }


	/***************************************************
     *      Pass Attributes Verification               *
     *                                                 *
     * Attributes passed through the cfg_attr_t        *
     * structure are not known to be valid until the   *
     * /dev/gm driver can check their status in the  *
     * indata argument.  A status other than           *
     * CFG_FRAME_SUCCESS is an error condition.  You   *
     * can use this code to:                           *
     *                                                 *
     *  o Check cfgmgr loading problems with the       *
     *    gm_attributes (struct type                 *
     *    cfg_subsys_attr_t)                           *
     *                                                 *
     *  o Display the contents and status of the       *
     *    attributes in the sysconfigtab database for  *
     *    the /dev/gm driver                         *
     *                                                 *
     *  o Report cfgmgr's status that indicates a      *
     *    failure to load any of the attribute fields  *
     *    into the gm_attributes structure           *
     ***************************************************/


		 bcopy(indata, cfg_buf[0].name,
			   indatalen * (sizeof(cfg_attr_t)));

		 for (i = 0; i < indatalen; i++) {
#if (GM_PRINT_LEVEL > 2)
			 printf("gm: scan %s\n", cfg_buf[i].name);
			 if (cfg_buf[i].type == CFG_ATTR_STRTYPE) {
				 printf("gm: ...which is string:%s\n", cfg_buf[i].attr.str.val);
			 }
#endif

			 switch (cfg_buf[i].type) {
			  case CFG_ATTR_STRTYPE:
				  break;
			  default:
				  switch (cfg_buf[i].status) {
				   case CFG_FRAME_SUCCESS:
					   break;
				   default:
					   printf("gm: %s:", cfg_buf[i].name);
					   switch (cfg_buf[i].status) {
						case CFG_ATTR_EEXISTS:
							printf("gm: Attribute does not exist\n");
							break;
						case CFG_ATTR_EOP:
							printf("gm: Attribute does not support operation\n");
							break;
						case CFG_ATTR_ESUBSYS:
							printf("gm: Subsystem Failure\n");
							break;
						case CFG_ATTR_ESMALL:
							printf("gm: Attribute size/value too small\n");
							break;
						case CFG_ATTR_ELARGE:
							printf("gm: Attribute size/value too large\n");
							break;
						case CFG_ATTR_ETYPE:
							printf("gm: Attribute invalid type\n");
							break;
						case CFG_ATTR_EINDEX:
							printf("gm: Attribute invalid index\n");
							break;
						case CFG_ATTR_EMEM:
							printf("gm: Attribute memory allocation error\n");
							break;
						default:
							printf("gm: **Unknown attribute: ");
							printf("gm: %x\n", cfg_buf[i].status);
					   }
					   /*        return(EINVAL); */
				  }
				  break;
			 }					/* switch */
		 }						/* for */

		 if (debug) {
			 printf("\ngm: Done dealing with configuration switches\n");
		 }

		 /*
		  * If this device driver has already been configured,
		  * either statically or dynamically, then return this
		  * gm_configure call indicating an error when the
		  * cfgmgr framework calls the CFG_OP_CONFIGURE entry
		  * point of the driver's configure interface.
		  */
		 if (myri_gm_config == TRUE) {
			 if (debug) {
				 printf("gm: gm already configured, bailing\n");
			 }
			 return (EINVAL);
		 }

		 if (strcmp(mcfgname, "") == 0) {
			 strcpy(mcfgname, "gm");
		 }
		 else {
			 retval = cfgmgr_get_state(mcfgname, &driver_cfg_state);

			 if (debug) {
				 printf("gm: cfgmgr_get_state returns %s\n", perrormsg(retval));
			 }

			 if (retval != ESUCCESS) {
				 if (debug) {
					 printf("gm: failing due to cfg_mgr_get_state\n");
				 }
				 return (EINVAL);
			 }

			 if (driver_cfg_state == SUBSYSTEM_STATICALLY_CONFIGURED) {
				 if (debug) {
					 printf("gm: Myrinet is statically configured\n");
				 }

				 callback_return_status = ESUCCESS;

				 gm_is_dynamic = SUBSYSTEM_STATICALLY_CONFIGURED;

				 register_callback(callback_register_configuration,
								   CFG_PT_PRECONFIG,
								   CFG_ORD_NOMINAL, (long) 0L);

				 register_callback(callback_register_major_number,
								   CFG_PT_POSTCONFIG,
								   CFG_ORD_NOMINAL, (long) 0L);
			 }
			 else {
				 if (debug) {
					 printf("gm: Dynamically configuring gm subsystem\n");
				 }

				 gm_is_dynamic = SUBSYSTEM_DYNAMICALLY_CONFIGURED;

				 retval = register_configuration();
				 if (debug) {
					 printf("gm: register_configuration returned %s\n",
							perrormsg(retval));
				 }
				 if (retval != ESUCCESS) {
					 return (retval);
				 }

				 retval = configure_driver(mcfgname,
										   DRIVER_WILDNUM,
										   DRIVER_WILDNAME,
										   &gmdriver);
				 if (debug) {
					 printf("gm: configure_driver returned %s\n",
							perrormsg(retval));
				 }

				 if (retval != ESUCCESS) {
					 return (retval);
				 }

				 retval = register_major_number();
				 if (retval != ESUCCESS) {
					 printf("gm: **** register_major_number fails, returns %s (%d) ***** \n",
							perrormsg(retval), retval);
					 if ((retval2 = unconfigure_driver(DN_BUSNAME1,
												  DRIVER_WILDNUM, &gmdriver,
									   DN_BUSNAME1, DRIVER_WILDNUM)) != 0) {
						 printf("gm: unconfigure_driver failed.  %s (%d)\n",
								perrormsg(retval2), retval2);
					 }
					 else {
						 printf("gm: unconfigure_driver succeeded\n");
					 }
					 return (retval);
				 }

			 }

			 printf("gm: Apparently, all systems are go?\n");
			 break;


	  /***************************************************
       * Unconfigure (unload) the driver.                *
       ***************************************************/

	 case CFG_OP_UNCONFIGURE:

			 if (debug) {
				 printf("gm: gm_configure: CFG_OP_UNCONFIGURE.\n");
			 }
			 if (gm_is_dynamic == SUBSYSTEM_STATICALLY_CONFIGURED) {
				 return (ESUCCESS);
			 }

			 for (i = 0; i < num_gm; i++) {
				 if (gm_softc[i].arch.sc_openf != 0) {
					 if (debug) {
						printf("gm: gm_configure: gm_softc[%d].arch.sc_openf != 0 - EBUSY\n",i);
					 }
					 return (EBUSY);
				 }
			 }


	  /***************************************************
       *        Call cdevsw_del to remove the driver     *
       *        entry points from the in-memory resident *
       *        cdevsw table.  This is done prior to     *
       *        deleting the loadable configuration      *
       *        and handlers to prevent users from       *
       *        accessing the device in the middle of    *
       *        deconfigure operation.                   *
       ***************************************************/

			 retval = devsw_del(mcfgname, MAJOR_INSTANCE);
			 if (retval == NO_DEV) {
				if (debug) {
					printf("gm: gm_configure: devsw_del returns NO_DEV - ESRCH\n");
				}
				 return (ESRCH);
			 }

			 if (unconfigure_driver(DN_BUSNAME1,
									DRIVER_WILDNUM, &gmdriver,
									DN_BUSNAME1, DRIVER_WILDNUM) != 0) {
				 if (debug) {
					 printf("gm: gm_configure:unconfigure_driver failed.\n");
				 }
				 return (ESRCH);
			 }
		 }

		 gm_is_dynamic = 0;
		 myri_gm_config = FALSE;
		 break;

	 case CFG_OP_QUERY:
		if (debug) {
			printf("gm: gm_configure: CFG_OP_QUERY\n");
		}
		 break;

	 case CFG_OP_RECONFIGURE:
		if (debug) {
			printf("gm: gm_configure: CFG_OP_RECONFIGURE\n");
		}
		 break;

	 default:					/* Unknown operation type */
		if (debug) {
			printf("gm: gm_configure: default - unknown operation\n");
		}
		 return (ENOTSUP);
	}

	return (ESUCCESS);
}



/***************************************************
 *           register_configuration()              *
 ***************************************************
 * Name:        register_configuration             *
 *                                                 *
 * Arguments:   Myri                               *
 *                                                 *
 * Kernel support interface calls:                 *
 *                                                 *
 *      o create_controller_struct                 *
 *      o create_device_struct                     *
 *                                                 *
 * Called by:                                      *
 *                                                 *
 *      o During dynamic configuration             *
 *          - gm_configure                       *
 *                                                 *
 *      o During static Configuration              *
 *          - cfgmgr framework callback            *
 *            Specifically, the /dev/gm driver   *
 *            registers an interface called        *
 *            callback_register_configuration.     *
 *            The cfgmgr framework calls           *
 *            callback_register_configuration,     *
 *            which contains a call to the         *
 *            register_configuration interface.    *
 *                                                 *
 * Return Codes:                                   *
 *                                                 *
 *      Failure:        ENOMEM                     *
 *                                                 *
 * Description:                                    *
 *                                                 *
 * The register_configuration interface is         *
 * responsible for registering the controller and  *
 * device information for the /dev/gm driver.    *
 * The dynamically configured /dev/gm driver     *
 * directly calls the register_configuration       *
 * interface.  The statically configured /dev/gm *
 * driver calls register_configuration through a   *
 * register callback called                        *
 * callback_register_configuration.  (The          *
 * gm_configure interface calls the              *
 * register_callback interface to register the     *
 * register_configuration interface so that it can *
 * be called at some later time.                   *
 *                                                 *
 * Writing an interface similar to                 *
 * register_configuration is the method for        *
 * specifying the controller and device            *
 * information associated with a device driver.    *
 *                                                 *
 * The register_configuration interface causes the *
 * appropriate controller and device structures    *
 * to be created and integrated into the system    *
 * (hardware) configuration tree for statically    *
 * and dynamically configured drivers.             * 
 *                                                 *
 * This method requires device driver writers to   *
 * determine how much operator control should be   *
 * allowed over the system's topology creation     *
 * support code.  If appropriate, driver writers   *
 * can define sysconfigtab database fields that    *
 * will allow  the operator to control what        *
 * hardware-related information that this          *
 * interface passes to the appropriate structures. *
 ***************************************************/

static int
register_configuration()
{


  /***************************************************
   * Declare controller_config and device_config     *
   * structures and structure pointers.  These       *
   * structures are the storage mechanism for        *
   * populating the controller and device            * 
   * associated with a specific device driver.       *
   ***************************************************/

	struct controller_config ctlr_register;
	struct controller_config *ctlr_register_ptr = &ctlr_register;
	struct device_config device_register;
	struct device_config *device_register_ptr = &device_register;
	int status;

  /***************************************************
   * Register a controller structure for the         *
   * /dev/gm driver.  The system manager can       *
   * specify attribute fields for the gm: entry in *
   * etc/sysconfigtab database that determine the    *
   * number of controller structures to be created.  *
   * It is here that the user-configurable attribute *
   * NMYRI would be used to regulate the number of   *
   * controller structures to be created up to the   *
   * MAX_NMYRI value defined in the driver.          *
   ***************************************************/

	if (debug) {
		printf("gm: Entering register_configuration\n");
		printf("gm: subsystem_name = '%s'\n", mcfgname);
		printf("gm: bus_name = '%s'\n", DN_BUSNAME1);
	}

	ctlr_register_ptr->revision = CTLR_CONFIG_REVISION;
	strcpy(ctlr_register_ptr->subsystem_name, mcfgname);
	strcpy(ctlr_register_ptr->bus_name, DN_BUSNAME1);
	ctlr_register_ptr->devdriver = &gmdriver;

  /***************************************************
   * Call the create_controller_struct to create the *
   * controller structure associated with the        *
   * /dev/gm driver.                               * 
   ***************************************************/

	status = create_controller_struct(ctlr_register_ptr);
	if (debug) {
		printf("gm: create_controller_struct returns %s\n", perrormsg(status));
	}
	if (status != ESUCCESS) {
		return (status);
	}

  /***************************************************
   * Register a device structure for the /dev/gm   *
   * driver.                                         * 
   ***************************************************/

	device_register_ptr->revision = DEVICE_CONFIG_REVISION;
	/*
	 * According to the documentation, the device_type field isn't really used
	 */
	strcpy(device_register_ptr->device_type, "disk");
	strcpy(device_register_ptr->device_name, "gm");
	strcpy(device_register_ptr->controller_name, "gm");
	device_register_ptr->phys_unit_num = 0;
	device_register_ptr->logical_unit_number = 0;
	device_register_ptr->controller_num = 0;

  /***************************************************
   * Call the create_device_struct to create the     *
   * device structure associated with the /dev/gm  *
   * driver.                                         * 
   ***************************************************/


	status = create_device_struct(device_register_ptr);
	if (debug) {
		printf("gm: create_device_struct returns %s\n", perrormsg(status));
	}

	if (status != ESUCCESS) {
		return (status);
	}

	if (debug) {
		printf("gm: returning from register_configuration, everything worked\n", status);
	}

	return (ESUCCESS);
}

/***************************************************
 *        register_major_number()                  *
 ***************************************************
 * Name:  register_major_number                    *
 *                                                 *
 * Arguments:                                      *
 *                                                 *
 *                                                 *
 * Kernel support interface calls:                 *
 *                                                 *
 *      o devsw_add                                *
 *                                                 *
 * Called by:                                      *
 *                                                 *
 *          o cfgmgr framework                     *
 *                                                 *
 * Return Codes:                                   *
 *                                                 *
 *      Success:        ESUCCESS                   *
 *                                                 *
 *      Failure:        ENODEV                     *
 *                                                 *
 * Description:                                    *
 *                                                 *
 * The register_major_number interface registers   *
 * a driver's I/O services interfaces (and other   *
 * information) and reserves a major number.  It   *
 * accomplishes these tasks by calling the         *
 * devsw_add interface.                            *
 ***************************************************/

static int
register_major_number()
{

  /***************************************************
   * If there are no devices present in the system   *
   * after driver configuration completes, set the   *
   * cfgmgr framework status to unconfigured and     *
   * exit the callback.                              *
   ***************************************************/


	if (debug) {
		printf("gm: enter register_major_number\n");
	}


	if (num_gm == 0) {
		if (debug) {
			printf("gm: num_gm == 0? No boards found? Returning ENODEV \n");
		}

		return (ENODEV);
	}

  /***************************************************
   * Call the devsw_add interface to register the    *
   * driver's I/O services interfaces (and other     *
   * information) and to reserve a major number in   *
   * the device switch table.  Each allocation       *
   * provides a block and character device entry.    *
   ***************************************************/

	majnum = devsw_add(mcfgname, MAJOR_INSTANCE,
					   majnum, &gm_devsw_entry);


  /***************************************************
   * The call to devsw_add could fail if the driver  *
   * requests a specific major number and that major *
   * number is currently in use.  The call could     *
   * also fail if the device switch table is         *
   * currently full.                                 *
   ***************************************************/

	if (majnum == NO_DEV) {
		if (debug) {
			printf("gm: majnum == NO_DEV? failing\n");
		}
		return (ENODEV);
	}

  /***************************************************
   * Store the major number returned by the          *
   * devsw_add interface so that it can be used      *
   * later to unconfigure the device.                *
   ***************************************************/

	GM_INFO (("major device number assigned = %d\n", majnum));

	gm_devno = majnum;





  /***************************************************
   *          Set this state field to indicate that  *
   *          the driver has successfully            *
   *          configured.                            *
   ***************************************************/



	myri_gm_config = TRUE;

	if (debug) {
		printf("gm: returning from register_major_number with success\n");
	}

	return (ESUCCESS);
}

/***************************************************
 *       callback_register_configuration()         *
 ***************************************************
 * Name:   callback_register_configuration()       *
 *                                                 *
 * Arguments:                                      *
 *                                                 *
 *    int    point  - Driver callback point        *
 *    int    order  - Priority at callback         *
 *    ulong  argument  - User argument             *
 *    ulong  event_arg  - Argument passed by       *
 *                        kernel dispatcher        *
 *                                                 *                        
 *                                                 *
 * Kernel support interface calls:                 *
 *                                                 *
 *      o cfgmgr_set_status                        *
 *                                                 *
 * Driver interface calls:                         *
 *                                                 *
 *      o register_configuration                   *
 *                                                 *
 * Called by:                                      *
 *                                                 *
 *          o cfgmgr framework through a call to   *
 *            the register_callback interface      *
 *                                                 *
 * Return Codes:                                   *
 *                                                 *
 *      Success:        ESUCCESS                   *
 *                                                 *
 *      Failure:        EBUSY                      *
 *                      EINVAL                     *
 *                      ESRCH                      *
 *                                                 *
 * Description:                                    *
 *                                                 *
 * This is the static cfgmgr jacket used to        *
 * do configuration registration in the boot path. *
 * This jacket interface handles the static        *
 * callback specific error handling but utilizes   *
 * the same configuration interface the driver     *
 * uses when dynamically configured.               *
 ***************************************************/


static void
callback_register_configuration(int point, int order, ulong argument,
								ulong event_arg)
{
	int status;

  /***************************************************
   * Detect if a previous failure has occurred in    *
   * statically configuring this driver. If so, exit *
   * the callback without doing any configuration    *
   * work.
   ***************************************************/


	if (debug) {
		printf("gm: enter callback_register_configuration\n");
	}

	if (callback_return_status != ESUCCESS) {
		if (debug) {
			printf("gm: callback_return_status == %d, failing\n", callback_return_status);
		}
		return;
	}

  /***************************************************
   *  Register this driver's configuration with the  *
   * kernel's topology.                              *
   ***************************************************/


	status = register_configuration();

  /***************************************************
   * If configuration registration is successful,    * 
   * return from this callback.  Otherwise, signal   *
   * the cfgmgr framework that this driver has       *
   * failed to configure and set the variable        *
   * callback_return_status to the failure           * 
   * condition.                                      *
   *                                                 *
   ************* NOTE ********************************
   *                                                 *
   * A side effect of calling the cfgmgr_set_status  *
   * interface is that the configure interface or    *
   * this driver is called with the unconfigure op   *
   * passed.  The unconfigure interface must be      *
   * prepared for this event.                        *
   ***************************************************/


	if (status != ESUCCESS) {

		if (debug) {
			printf("gm: register_configuration failed because %d\n", status);
		}


		cfgmgr_set_status(mcfgname);
		callback_return_status = status;
		return;
	}
}

/***************************************************
 *     callback_register_major_number()            *
 ***************************************************
 * Name:  callback_register_major_number           *
 *                                                 *
 * Arguments:                                      *
 *                                                 *
 *    int    point  - driver callback point        *
 *    int    order  - Priority at callback         *
 *    ulong  argument - User argument              *
 *    ulong  event_arg  - Argument passed by       *
 *                        kernel dispatcher        *
 *                                                 *                        
 *                                                 *
 * Kernel support interface calls:                 *
 *                                                 *
 *      o cfgmgr_set_status                        *
 *                                                 *
 * Driver interface calls:                         *
 *                                                 *
 *      o register_major_number                    *
 *                                                 *
 * Called by:                                      *
 *                                                 *
 *          o cfgmgr framework as static callback  *
 *                                                 *
 * Return Codes:                                   *
 *                                                 *
 *      Success:        ESUCCESS                   *
 *                                                 *
 *      Failure:        EBUSY                      *
 *                      EINVAL                     *
 *                      ESRCH                      *
 *                                                 *
 * Description:                                    *
 *                                                 *
 * This is the Static cfgmgr jacket used to        *
 * do major number registration in the boot path.  *
 * This jacket routine handles the static callback *
 * specific error handling but utilizes the same   *
 * configuration routine used by the driver when   *
 * dynamically configuring.                        *
 *                                                 *
 ***************************************************/

static void
callback_register_major_number(int point, int order, ulong argument, ulong event_arg)
{
	int status;

  /***************************************************
   * Detect if a previous failure has occurred in    *
   * statically configuring this driver. If so, exit *
   * the callback without doing any configuration    *
   * work.
   ***************************************************/


	if (callback_return_status != ESUCCESS) {
		if (debug) {
			printf("gm: callback_return_success != ESCUCESS\n");
		}
		return;
	}

  /***************************************************
   * Call the driver-specific register_major_number  *
   * interface to register the driver's I/O services *
   * interfaces (and other information) and reserve  *
   * a major number.                                 *
   ***************************************************/


	status = register_major_number();

  /***************************************************
   * If configuration registration is successful,    * 
   * return from this callback.  Otherwise, signal   *
   * the cfgmgr framework that this driver has       *
   * failed to configure and set the variable        *
   * callback_return_status to the failure           * 
   * condition.                                      *
   *                                                 *
   ************* NOTE ********************************
   *                                                 *
   * A side effect of calling the cfgmgr_set_status  *
   * interface is that the configure interface or    *
   * this driver is called with the unconfigure op   *
   * passed.  The unconfigure interface must be      *
   * prepared for this event.                        *
   ***************************************************/


	if (status != ESUCCESS) {
		if (debug) {
			printf("gm: register_major_number returned %s, fails\n",
				   perrormsg(status));
		}
		cfgmgr_set_status(mcfgname);
		callback_return_status = status;
		return;
	}
}

static void
print_pci_cfg(struct pci_config_hdr *p)
{
	/* 64 byte predefined header of 256 byte config space for every device    */
	/* First 16 bytes defined the same for all types of devices.      */
	printf("gm: vid %x, did %x, command %x, status %x, rev_id %x,\n",
		   p->vendor_id,		/* Vendor id             */
		   p->device_id,		/* Device id             */
		   p->command,			/* Command           */
		   p->status,			/* Status            */
		   p->rev_id			/* Device revision ID        */
		);
	printf("gm: cc pio %x, cc subc %x, cc base %x\n",
		   p->class_code.pio_int,	/* Class code            */
		   p->class_code.sub_class,		/* Class code          */
		   p->class_code.base	/* Class code                */
		);

	printf("gm: cachels %x, latency %x, hdr_type %x, bist %x\n",
		   p->cache_line_size,	/* System cache-size, 32b words */
		   p->latency_timer,	/* Bus latency timer         */
		   p->hdr_type,			/* Defines type for 10h-3Fh      */
		   p->bist				/* Built in self test        */
		);

	printf("gm: bar0-5 %x %x %x %x %x %x\n",
	/* Next 48 bytes have diff. layouts based on device's base function.   */
	/* Note: For bar(Base Address Register) values --
	 *    Upper (32-)bits used to store per-platform mapping info;
	 *  Lower (32-)bits native PCI address (lower 4-bits 0'd).
	 */
		   p-> /* 1 */ bar0,	/* Base address reg 0    */
		   p-> /* 2 */ bar1,	/* Base address reg 1    */
		   p-> /* 3 */ bar2,	/* Base address reg 2    */
		   p-> /* 4 */ bar3,	/* Base address reg 3    */
		   p-> /* 5 */ bar4,	/* Base address reg 4    */
		   p-> /* 6 */ bar5		/* Base address reg 5    */
		);

	printf("gm: rsvd0-1 %x %x \n",
		   p-> /* 7 */ hdrtype_u.type0.rsvd0,	/* Reserved      */
		   p-> /* 8 */ hdrtype_u.type0.rsvd1	/* Reserved      */
		);

	printf("gm: exp_rom %x cis_ptr %x, sub_vid %x, sub_dev %x, rsvd3 %x\n",
		   p-> /* 9 */ exp_rom_bar,		/* Expansion ROM address   */
		   p-> /* 10 */ cis_ptr,	/* Cardbus CIS Pointer */
		   p-> /* 11 */ sub_vendor_id,	/* Subsystem Vendor ID   */
		   p->sub_device_id,	/* Subsystem Device ID   */
		   p->hdrtype_u.type0.rsvd3		/* reserved; align qw    */
		);

	printf("gm: intrline %x, min_gnt %x, max_lat %x, rsvd4 %x\n",
		   p-> /* 12 */ intr_line,	/* Interrupt line            */
		   p->intr_pin,			/* Interrupt pin         */
		   p->min_gnt,			/* Minimum grant         */
		   p->max_lat,			/* Maximum latency   */
		   p->hdrtype_u.type0.rsvd4		/* Align to next qw bound    */
		);

	printf("gm: config_base %x\n",
	/* Device specific regs. start here, location 64, through 255 */
		   p->config_base		/* slot's config space base */
		);
}

static void
print_struct_controller(struct controller *ctlr)
{
	printf("gm: Controller address is 0x%p\n", ctlr);
	if (ctlr == NULL)
		return;
	printf("gm: ctlr_type= %d\n", ctlr->ctlr_type);
	printf("gm: ctlr_name= %s\n", ctlr->ctlr_name);
	printf("gm: ctlr_num = %d\n", ctlr->ctlr_num);
	printf("gm: bus_name = %s\n", ctlr->bus_name);
	printf("gm: bus_num  = %d\n", ctlr->bus_num);
	printf("gm: rctlr    = %d\n", ctlr->rctlr);
	printf("gm: slot     = %d\n", ctlr->slot);
	printf("gm: ..done\n");
}


static int
gm_osf1_intr(gm_instance_state_t * is)
{
	unsigned int s;

	s = splimp();
	gm_intr(is);
	splx(s);
	return (0);
}

static int
gm_osf1_probe(struct pci_config_hdr *pci_cfg_hdr,
			  struct controller *ctlr)
{
	gm_instance_state_t *sc;
	struct handler_intr_info gm_intr_info;
	ihandler_t gm_ihandle;
	int unit;
	vm_offset_t paddr;
	int i;


	if (debug) {
		printf("gm: gm_osf1_probe\n");
		/* print_pci_cfg(pci_cfg_hdr); */
		print_struct_controller(ctlr);
	}

	/* determine whether device has been configured by console poweron */

	/*
	 * if the board is not recognized as a bus master and its memory space is
	 * not recognized, then the POST system did not configure the board
	 * correctly and its CSR's are not available for access. 
	 */

	unit = ctlr->ctlr_num;

	if (!(pci_cfg_hdr->command & CMD_MEM_SPACE) ||
			!(pci_cfg_hdr->command & CMD_BUS_MASTER)) {
		printf("gm: gm%d  at pci%d not configured by PCI POST code.\n",
			   unit, ctlr->bus_num);
		return (0);
	}

	if (unit < 0 || unit >= MAX_NMYRI) {
		printf("gm: can't configure gm unit = %d, increase MAX_NMYRI\n",unit);
		return 0;
	}

	if (debug) {
		printf("gm_osf1_probe: using unit = %d\n",unit);;
	}
	sc = &gm_softc[unit];
	/* save controller structure in softc struct */
	sc->arch.ctlr = ctlr;
	sc->arch.is_open = 0;
	sc->arch.is_dense = 0;
	sc->arch.revision = pci_cfg_hdr->rev_id;
	sc->id = unit;

	/* keep pci_cfg_hdr around in case we need it later for resets, etc */
	ctlr->private[0] = pci_cfg_hdr;
	sc->arch.pci_cfg_hdr = pci_cfg_hdr;

	/*
	 * the address of the gmnet PCI config registers are stored in base
	 * address register 0. 
	 */

	sc->arch.csr_base = (io_handle_t) pci_cfg_hdr->bar0;

	if (debug) {
		printf("gm: BASE %p\n", sc->arch.csr_base);
	}


	paddr = iohandle_to_phys(sc->arch.csr_base,
							 HANDLE_LINEAR_SPACE | HANDLE_LONGWORD);
	num_gm++;

	if (debug) {
		GM_PRINT (GM_PRINT_LEVEL >= 2, ("paddr %p \n", paddr));
	}

	/*
	 * Register the interrupt handler
	 */

	gm_intr_info.configuration_st = (caddr_t) ctlr;
	gm_intr_info.intr = gm_osf1_intr;
	gm_intr_info.param = (caddr_t) sc;
	gm_intr_info.config_type = (CONTROLLER_CONFIG_TYPE | SHARED_INTR_CAPABLE);

	gm_ihandle.ih_bus = ctlr->bus_hd;
	gm_ihandle.ih_bus_info = (char *) &gm_intr_info;

	sc->arch.hid = handler_add(&gm_ihandle);
	if (sc->arch.hid == (ihandler_id_t *) (NULL)) {
		printf("gm%d: failed to add interrupt handler\n", unit);
		return (0);
	}
	else {
		if (debug) {
			printf("gm%d: added interrupt handler\n", unit);
		}
		/*    handler_enable(sc->hid); */
	}

	if (debug>=3) {
		printf("gm: base: %p\n",
			   sc->arch.csr_base);
		printf("gm: base dense: %p\n",
			   iohandle_to_phys(sc->arch.csr_base, HANDLE_DENSE_SPACE));
		printf("gm: base word dense: %p\n",
			   iohandle_to_phys(sc->arch.csr_base, HANDLE_DENSE_SPACE
								| HANDLE_LONGWORD));
		printf("gm: base byte sparse: %p\n",
			   iohandle_to_phys(sc->arch.csr_base, HANDLE_SPARSE_SPACE
								| HANDLE_BYTE));
		printf("gm: base word sparse: %p\n",
			   iohandle_to_phys(sc->arch.csr_base, HANDLE_SPARSE_SPACE
								| HANDLE_WORD));
	}

    /*
     * Try to allocate another controller to support an additional unit.
     */
    if (register_configuration() != ESUCCESS) {
        printf("gm%d: WARNING: create_controller failed for next unit\n", unit);
    }
    else {
		if (debug) {
			printf("gm%d: Allocated a new controller_struct for next device\n",unit);
		}
	}


	return 1;
}


/**********************************************************************
 * GM OS-abstraction required functions.
 **********************************************************************/

void gm_arch_abort (void)
{
	printf("GM: *** gm_arch_abort was called - yikes ***\n");
}

#if GM_SUPPORT_PCI

/************
 * PCI configuration space access functions.
 ************/

gm_status_t
gm_arch_read_pci_config_32(gm_instance_state_t * is,
			   gm_offset_t offset,
			   gm_u32_t * valueptr)
{
	*valueptr = READ_BUS_D32(((io_handle_t) is->arch.pci_cfg_hdr->config_base) + offset);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_32(gm_instance_state_t * is,
			    gm_offset_t offset,
			    gm_u32_t value)
{
	WRITE_BUS_D32(((io_handle_t) is->arch.pci_cfg_hdr->config_base) + offset, value);

	return GM_SUCCESS;
}

/* 16-bit accesses */

gm_status_t
gm_arch_read_pci_config_16(gm_instance_state_t * is,
			   gm_offset_t offset,
			   gm_u16_t * valueptr)
{
	*valueptr = READ_BUS_D16(((io_handle_t) is->arch.pci_cfg_hdr->config_base) + offset);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_16(gm_instance_state_t * is,
			    gm_offset_t offset,
			    gm_u16_t value)
{
	WRITE_BUS_D16(((io_handle_t) is->arch.pci_cfg_hdr->config_base) + offset, value);
	return GM_SUCCESS;
}

/* 8-bit accesses */

gm_status_t
gm_arch_read_pci_config_8(gm_instance_state_t * is,
			  gm_offset_t offset,
			  gm_u8_t * valueptr)
{
	*valueptr = READ_BUS_D8(((io_handle_t) is->arch.pci_cfg_hdr->config_base) + offset);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_8(gm_instance_state_t * is,
			   gm_offset_t offset,
			   gm_u8_t value)
{
	WRITE_BUS_D8(((io_handle_t) is->arch.pci_cfg_hdr->config_base) + offset, value);
	return GM_SUCCESS;
}
#endif

/************
 * GM synchronization functions
 ************/

/* For dec, we must be careful to pass an appropriate iblock
   cookie to gm_mutex_init if the mutex is ever user by the interrupt
   handler. */

void
gm_arch_sync_init(gm_arch_sync_t * s, gm_instance_state_t * is)
{
	gm_bzero(s, sizeof(*s));
	lock_init(&s->mu, TRUE);
	s->count = 0;
	simple_lock_init(&s->_mu);
}

void
gm_arch_sync_reset(gm_arch_sync_t * s)
{
	s->wake_cnt = 0;
}

void
gm_arch_sync_destroy(gm_arch_sync_t * s)
{
	GM_CALLED ();
	simple_lock_terminate(&s->_mu);
	lock_terminate(&s->mu);
	GM_RETURN_NOTHING ();
}

void
gm_arch_mutex_enter(gm_arch_sync_t * s)
{
	/*   s->s = splimp(); */
	lock_write(&s->mu);
	s->count++;
	if (s->count > 1)
		GM_PRINT (1, ("enter mutex %p %d\n", &s->mu, s->count));
}

void
gm_arch_mutex_exit(gm_arch_sync_t * s)
{
	lock_done(&s->mu);
	s->count--;
	if (s->count >= 1)
		GM_PRINT (1, ("exit mutex %p %d\n", &s->mu, s->count));


	/*   splx(s->s); */
}

/* Sleep functions. Return 0 on wake, -1 on timeout, and 1 on signal. */

/* sleep until awakened or get a signal */

#define GM_DO_SIMPLE_LOCK	1

gm_arch_sleep_status_t
gm_arch_signal_sleep(gm_arch_sync_t * s)
{
#if GM_USE_IPL
	int ipl;
#endif

	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_signal_sleep() called.\n"));
	printf("gm_arch_signal_sleep() called.\n");

#if GM_USE_IPL
	ipl = splimp();
#endif

#if 1

	if (GM_DO_SIMPLE_LOCK)
		simple_lock(&s->_mu);

	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_signal_sleep() entered mutex.\n"));
	printf("gm_arch_signal_sleep() entered mutex.\n");
	while (s->wake_cnt <= 0) {
#if GM_USE_THREADS
		assert_wait_mesg((vm_offset_t) & s->cv, TRUE, "gm_signal_sleep");

		if (GM_DO_SIMPLE_LOCK)
			simple_unlock(&s->_mu);

#if GM_USE_IPL
		splx(ipl);
#endif
		thread_set_timeout(10 * hz);
		thread_block();
		switch (current_thread()->wait_result) {
		 case THREAD_AWAKENED:
			 GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_signal_sleep() returning GM_SLEEP_WOKE\n"));
			 printf("gm_arch_signal_sleep() returning GM_SLEEP_WOKE\n");
			 s->wake_cnt--;
			 return GM_SLEEP_WOKE;
			 break;

		 case THREAD_TIMED_OUT:
			 GM_NOTE (("gm_arch_signal_sleep returning GM_SLEEP_TIMED_OUT\n"));
			 return GM_SLEEP_TIMED_OUT;
			 break;

		 case THREAD_INTERRUPTED:
			 GM_NOTE (("gm_arch_signal_sleep returning GM_SLEEP_TIMED_OUT (THREAD_INTERRUPTED)\n"));
			 return GM_SLEEP_TIMED_OUT;
			 break;

		 default:
			 GM_NOTE (("gm_arch_signal_sleep returning GM_SLEEP_TIMED_OUT (default %d)\n",
						 current_thread()->wait_result));
			 return GM_SLEEP_TIMED_OUT;
			 break;
		}

#else							/* !THREADS */

		if (mpsleep((vm_offset_t) & s->cv, PCATCH, "gm_sig_sleep", 0,
					&s->_mu, MS_LOCK_SIMPLE) != 0) {

			GM_NOTE (("gm_arch_signal_sleep returning GM_SLEEP_INTERRUPTED\n"));

			if (GM_DO_SIMPLE_LOCK)
				simple_unlock(&s->_mu);

#if GM_USE_IPL
			splx(ipl);
#endif
			return GM_SLEEP_INTERRUPTED;
			break;
		}
#endif							/* GM_USE_THREADS */
	}
	s->wake_cnt--;
	if (GM_DO_SIMPLE_LOCK)
		simple_unlock(&s->_mu);

#if GM_USE_IPL
	splx(ipl);
#endif

#endif

	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_signal_sleep() returning.\n"));
	printf("gm_arch_signal_sleep() returning.\n");

	return GM_SLEEP_WOKE;
}

/* utility function to do either type of timed sleep */

gm_arch_sleep_status_t
gm_arch_timed_sleep(gm_arch_sync_t * s, int seconds)
{
#if GM_USE_IPL
	int ipl;
#endif
	long timeout_time;

#if GM_USE_IPL
	ipl = splimp();
#endif
	if (GM_DO_SIMPLE_LOCK)
		simple_lock(&s->_mu);
	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_timed_sleep() called.\n"));

	while (s->wake_cnt <= 0) {
#if GM_USE_THREADS
		assert_wait_mesg((vm_offset_t) & s->cv, FALSE, "gm_timed_sleep");

		if (GM_DO_SIMPLE_LOCK)
			simple_unlock(&s->_mu);
#if GM_USE_IPL
		splx(ipl);
#endif
		thread_set_timeout(seconds * hz);
		thread_block();
		switch (current_thread()->wait_result) {
		 case THREAD_AWAKENED:
			 GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_timed_sleep() returning GM_SLEEP_WOKE\n"));
			 s->wake_cnt--;
			 return GM_SLEEP_WOKE;
			 break;

		 case THREAD_TIMED_OUT:
			 GM_NOTE (("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT\n"));
			 return GM_SLEEP_TIMED_OUT;
			 break;

		 case THREAD_INTERRUPTED:
			 GM_NOTE (("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT (THREAD_INTERRUPTED)\n"));
			 return GM_SLEEP_TIMED_OUT;
			 break;

		 default:
			 GM_NOTE (("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT (default %d)\n",
						 current_thread()->wait_result));
			 return GM_SLEEP_TIMED_OUT;
			 break;
		}

#else							/* !THREADS */

		if (mpsleep((vm_offset_t) & s->cv, 0, "gm_sig_sleep", seconds * hz,
					&s->_mu, MS_LOCK_SIMPLE) != 0) {
			GM_NOTE (("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT\n"));
			if (GM_DO_SIMPLE_LOCK)
				simple_unlock(&s->_mu);
#if GM_USE_IPL
			splx(ipl);
#endif
			return GM_SLEEP_TIMED_OUT;
			break;
		}
#endif							/* GM_USE_THREADS */
	}
	s->wake_cnt--;
	if (GM_DO_SIMPLE_LOCK)
		simple_unlock(&s->_mu);
#if GM_USE_IPL
	splx(ipl);
#endif

	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_timed_sleep() returning.\n"));

	return GM_SLEEP_WOKE;
}

/* Wake the thread sleeping on the synchronization variable. */

void
gm_arch_wake(gm_arch_sync_t * s)
{
#if GM_USE_IPL
	int ipl;
#endif

	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_wake() called.\n"));
#if GM_USE_IPL
	ipl = splimp();
#endif

	simple_lock(&s->_mu);

	/*  if (s->wake_cnt <= 0) */

#if GM_USE_THREADS
	thread_wakeup((vm_offset_t) & s->cv);
#else
	wakeup((vm_offset_t) & s->cv);
#endif
	s->wake_cnt++;
	simple_unlock(&s->_mu);

#if GM_USE_IPL
	splx(ipl);
#endif
	GM_PRINT (GM_ARCH_DEBUG_SLEEP_WAKE, ("gm_arch_wake() returning.\n"));
}

/* Sleep for the indicated duration */

void
gm_arch_spin(gm_instance_state_t * is, gm_u32_t usecs)
{
    DELAY(usecs);

/*
	unsigned int mytime;
	short event;
	mytime =  (usecs * hz)/1000000;
	if (mytime==0) {
		mytime=1;
	}
	mpsleep((vm_offset_t) & event, 0, "gm_arch_spin", mytime, 0, 0);
*/
}

gm_status_t
gm_arch_gethostname(char *ptr, int len)
{
    GM_PRINT (GM_PRINT_LEVEL >= 4,("gm_arch_gethostname(%p,%d) called %p '%s' %d\n",
		ptr,len,hostname,hostname,hostnamelen));
    bzero(ptr, len);
	if (hostnamelen <=0) {
		return GM_FAILURE;
	}
    strncpy(ptr, hostname, hostnamelen);
    ptr[len-1]=0;
    GM_PRINT (GM_PRINT_LEVEL >= 4,("gm_arch_gethostname(%p,%d) ==> '%s' len = %d\n",
		ptr,len,ptr,hostnamelen));
    return GM_SUCCESS;
}


/************
 * Page locking
 ************/

#if GM_CAN_REGISTER_MEMORY

gm_status_t
gm_arch_lock_user_buffer_page(gm_instance_state_t * is, gm_up_t in,
				  gm_dp_t * dma_addr, gm_arch_page_lock_t * lock)
{
	int unit,size;
	struct controller *controller_p;
	struct proc *proc_p;
	sg_entry_t entry;

	GM_PRINT (GM_PRINT_LEVEL >= 4,("gm_arch_lock_user_buffer_page(%p, %p, %p, %p) called\n",
								   is,in,dma_addr,lock));

	unit = is->id;
	controller_p = gm_softc[unit].arch.ctlr;
	if (!controller_p) {
		GM_NOTE (("gm_arch_lock_user_buffer_page fails - no controller info?\n"));
		return(GM_FAILURE);
	}
	
	proc_p = task_to_proc(current_task());   
	if (!proc_p) {
		GM_NOTE (("gm: gm_arch_lock_user_buffer_page fails - no process pointer?\n"));
		return(GM_FAILURE);
	}
	
	size = dma_map_alloc(GM_PAGE_LEN, controller_p, &lock->dma_handle,
						 (DMA_IN | DMA_OUT | DMA_CONTIG));
	if (size == 0) {
		GM_NOTE (("gm: gm_arch_lock_user_buffer_page: dma_map_alloc failed\n"));
		return(GM_FAILURE);
	}

	size = dma_map_load (GM_PAGE_LEN, (unsigned long)in, proc_p, controller_p, 
					&lock->dma_handle, 0, 
					(DMA_IN | DMA_OUT | DMA_CONTIG));

	if (!size) {
		GM_NOTE ((__GM_FUNCTION__ "fails\n"
				  "    dma_map_load(%d, 0x%lx, %p, %p, %p, %d, 0x%x)"
				  " returns %d\n",
				  GM_PAGE_LEN, (unsigned long)in, proc_p, controller_p,
				  &lock->dma_handle, 0, (DMA_IN | DMA_OUT | DMA_CONTIG),
				  size));
		dma_map_dealloc(lock->dma_handle);
		return(GM_FAILURE);
	}

	entry = dma_get_curr_sgentry(lock->dma_handle);
	if (!entry) {
		GM_NOTE (("gm_arch_lock_user_buffer_page: dma_get_curr_sgentry failed\n"));
		dma_map_unload(0,lock->dma_handle);
		dma_map_dealloc(lock->dma_handle);
		return(GM_FAILURE);
	}
	if (entry->bc != GM_PAGE_LEN) {
		GM_NOTE (("gm_arch_lock_user_buffer_page: byte_count wrong %d != %d\n",
				  entry->bc, GM_PAGE_LEN));
		dma_map_unload(0,lock->dma_handle);
		dma_map_dealloc(lock->dma_handle);
		return(GM_FAILURE);
	}
	*dma_addr = (gm_dp_t)entry->ba;
	GM_PRINT (GM_PRINT_LEVEL >= 4,("gm_arch_lock_user_buffer_page(%p, %p, %p, %p) OK dma = %p\n",
			is,in,dma_addr,lock, (void *)*dma_addr));
	
	return(GM_SUCCESS);
}

void
gm_arch_unlock_user_buffer_page(gm_arch_page_lock_t * lock)
{
	int rv;

	GM_PRINT (GM_PRINT_LEVEL >= 4,("gm_arch_unlock_user_buffer_page(%p) called \n",lock));
	rv = dma_map_unload(0,lock->dma_handle);
	if (!rv) {
		GM_WARN (("gm_arch_unlock_user_buffer_page(%p): dma_map_unload failed\n",
				  lock));
		return;
	}
	rv = dma_map_dealloc(lock->dma_handle);
	if (!rv) {
		GM_WARN (("gm_arch_unlock_user_buffer_page(%p): dma_map_dealloc failed\n",
				  lock));
		return;
	}
	GM_PRINT (GM_PRINT_LEVEL >= 4,("gm_arch_unlock_user_buffer_page(%p): succeeds\n", lock));
}


#endif							/* GM_CAN_REGISTER_MEMORY */

/************************************************************************
 * Architecture-specific required functions
 ************************************************************************/

/* Return the value of GM_PAGE_LEN set during driver initialization. */

gm_status_t
gm_arch_get_page_len(unsigned long *page_len)
{
	if (!GM_PAGE_LEN) {
		GM_NOTE (("gm_dec internal error: GM_PAGE_LEN unknown in call\n"
					"to gm_arch_get_page_len().\n"));
		return GM_FAILURE;
	}
	*page_len = GM_PAGE_LEN;
	return GM_SUCCESS;
}

/************
 * gm_port_state initialization 
 ************/

/* This is called just after the port state is created (in gm_minor.c)
   to perform architecture-specific initialization. */

gm_status_t
gm_arch_port_state_init(gm_port_state_t * ps)
{
	return GM_SUCCESS;
}

/* This is called just before the port state is destroyed (in
   gm_minor.c) to perform architecture-specific finalization. */

void
gm_arch_port_state_fini(gm_port_state_t * ps)
{
	return;
}


/* This is called just after the port state is initialized (in gm_minor.c)
   to perform architecture-specific initialization. */

gm_status_t
gm_arch_port_state_open(gm_port_state_t * ps)
{
	return GM_SUCCESS;
}

/* This is called just before the port state is finalized (in
   gm_minor.c) to perform architecture-specific finalization. */

void
gm_arch_port_state_close(gm_port_state_t * ps)
{
	return;
}

/************
 * gm_ioctl support
 ************/

/* These are required so the gm_ioctl( ) can copy its arguments
   and results. */

gm_status_t
gm_arch_copyin(gm_port_state_t * ps,
	       void *what,
	       void *where,
	       gm_size_t amount)
{
	GM_PRINT (GM_PRINT_LEVEL >= 7, ("gm_arch_copyIN: 0x%lx -> 0x%lx  (%d bytes)\n",
				 what, where, amount));
#if 0
	{
		int rv;
		rv = copyin(what,where,amount);
		if (!rv) {
			GM_NOTE (("gm_arch_copyIN: 0x%lx -> 0x%lx  (%d bytes) FAILED\n",
					  what, where, amount));
			return GM_FAILURE;
		}
	}
#else
	bcopy((caddr_t) what, (caddr_t) where, amount);
#endif

	return GM_SUCCESS;
}

gm_status_t
gm_arch_copyout(gm_port_state_t * ps,
		void *what,
		void *where,
		gm_size_t amount)
{
	int rv;

	GM_PRINT (GM_PRINT_LEVEL >= 7, ("gm_arch_copyOUT: 0x%lx -> 0x%lx  (%d bytes)\n",
				 what, where, amount));

#if 0
	{
		int rv;
		rv = copyout(what,where,amount);
	
		if (!rv) {
			GM_NOTE (("gm_arch_copyOUT: 0x%lx -> 0x%lx  (%d bytes) FAILED\n",
					  what, where, amount));
			return GM_FAILURE;
		}
	}
#else
	if (amount > (unsigned int) (1 << 12)) {
		GM_PRINT (GM_PRINT_LEVEL >= 1, ("copyout - reduce the length to (1<<12)\n"));
		amount = (1 << 12);
	}
	bcopy((caddr_t) what, (caddr_t) where, amount);
#endif
	return GM_SUCCESS;
}

/************
 * Kernel Memory Allocation
 ************/

/* malloc() analogue for the driver */

void *
gm_arch_kernel_malloc(unsigned long len, int flags)
{
	gm_u32_t *p;

	/* Allocate extra space at beginning to store length for kmem_free call, and
	   return an 8-byte aligned buffer. */
	gm_assert(sizeof(long) <= 8);
	MALLOC(p, gm_u32_t *, len, M_DEVBUF, M_WAITOK);

	if (!p) {
		return p;
	}
	return &p[0];
}

/* free() analogue for the driver */

void
gm_arch_kernel_free(void *ptr)
{
	FREE(ptr, M_DEVBUF);
}

/************
 * DMA region manipulation
 ************/

/************
 * ??? Required for non cache-coherent I/O architectures, which we
 * do not yet support.
 ************/

gm_status_t
gm_arch_dma_region_sync(gm_arch_dma_region_t * r, int command)
{
	return GM_SUCCESS;
}

/* Allocate LEN bytes of DMA memory that is contiguous in kernel space
   by possibly segmented in DMA space.

   If r->register_function is non-null, call r->register_page_function
   (r, dma_addr) for each page. */

gm_status_t
gm_arch_dma_region_alloc(gm_instance_state_t * is, gm_arch_dma_region_t * r,
						 gm_size_t len, gm_u32_t flags,
						 gm_register_page_function_t register_page_func,
						 void *arg)
{
	int error, trash;
	int ddi_flags;
	int dma_size;
	int rlen;

	GM_CALLED ();

	GM_PRINT (GM_PRINT_LEVEL >= 2, ("gm_arch_dma_region_alloc: len = 0x%x flags =0x%x\n", len, flags));


	if (len <= 0) {
		GM_WARN (("** gm_arch_dma_region_alloc called with bad rlen = %d\n ** ",
					 len));
		len = 128;

	}

	if (!GM_PAGE_LEN) {
		error = GM_INTERNAL_ERROR;
		goto abort_with_nothing;
	}

	ddi_flags = 0;
	if (flags & GM_ARCH_DMA_READ) {
		ddi_flags |= DMA_IN;
		flags &= ~GM_ARCH_DMA_READ;
	}
	if (flags & GM_ARCH_DMA_WRITE) {
		ddi_flags |= DMA_OUT;
		flags &= ~GM_ARCH_DMA_WRITE;
	}
	if (flags & GM_ARCH_DMA_CONSISTENT) {
		flags &= ~GM_ARCH_DMA_CONSISTENT;
	}
	if (flags & GM_ARCH_DMA_STREAMING) {
		flags &= ~GM_ARCH_DMA_STREAMING;
	}

	GM_PRINT (GM_PRINT_LEVEL >= 2, ("in get_dma_region_alloc; len = %d flags = 0x%x\n", len, ddi_flags));
	if (flags != 0) {
		GM_NOTE (("gm_arch_dma_region_alloc: warning - got an unexpected flag\n"));
		error = GM_INTERNAL_ERROR;
		goto abort_with_nothing;
	}

	ddi_flags |= DMA_ALL;
	ddi_flags |= DMA_CONTIG;

	rlen = GM_PAGE_ROUNDUP(u32, len);
	/*   r->addr = (void *)contig_malloc (rlen, PAGE_SIZE, 
	   0, M_DEVBUF, M_WAITOK); */

	MALLOC(r->alloc_addr, void *, rlen + GM_PAGE_LEN, M_DEVBUF, M_WAITOK);

	r->addr = GM_PAGE_ROUNDUP(ptr, r->alloc_addr);

	if (r->addr == 0) {
		GM_NOTE (("gm: WARNING - can't contig_malloc dma mem %d\n", rlen));
		goto abort_with_nothing;
	}
	GM_PRINT (GM_PRINT_LEVEL >= 5, ("addr %p, %d\n", r->addr, rlen));

	r->len = rlen;
	r->rlen = rlen + GM_PAGE_LEN;

	dma_size = dma_map_alloc(rlen, is->arch.ctlr, &r->dma_handle, ddi_flags);
	if (dma_size == 0) {
		GM_NOTE (("gm: dma_map_alloc failed len = %d\n", rlen));
		goto abort_with_dma_mem;
	}

	gm_assert(GM_PAGE_ALIGNED(r->addr));


	dma_size = dma_map_load(rlen, (vm_offset_t) r->addr, 0, is->arch.ctlr,
							&r->dma_handle, 0, ddi_flags);


	if (dma_size == 0) {
		GM_NOTE (("gm: dma_map_load failed  vma = 0x%p len = %dbytes\n",
					r->addr, rlen));
		goto abort_with_bound_handle;
	}

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("dma_size %d\n", dma_size));

	r->sts = 0xf;

	r->sge = 0;
	r->sge_cnt = 0;


	/* Call the page registration functions for each page in the DMA
	   region. */
	if (register_page_func) {
		gm_u32_t page_num = 0;
		GM_PRINT (GM_PRINT_LEVEL >= 7, ("registering the pages\n"));
		while (page_num * GM_PAGE_LEN < r->len) {
			gm_dp_t bus_addr = gm_arch_dma_region_dma_addr_advance(r);
			register_page_func(arg, bus_addr, page_num);
			page_num++;
		}
	}

	GM_RETURN_STATUS (GM_SUCCESS);
  abort_with_load:
	dma_map_unload(0, r->dma_handle);
  abort_with_bound_handle:
	dma_map_dealloc(r->dma_handle);
  abort_with_dma_mem:
	/*   contig_free(r->addr, rlen, M_DEVBUF); */
	FREE(r->alloc_addr, M_DEVBUF);
  abort_with_nothing:
	GM_RETURN_STATUS (error);
}

void
gm_arch_dma_region_free(gm_arch_dma_region_t * r)
{
	GM_CALLED ();

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("unloading DMA\n"));
	dma_map_unload(0, r->dma_handle);
	GM_PRINT (GM_PRINT_LEVEL >= 4, ("unbinding\n"));
	dma_map_dealloc(r->dma_handle);
	GM_PRINT (GM_PRINT_LEVEL >= 4, ("handle freeing\n"));
	/*   contig_free(r->addr, GM_PAGE_ROUNDUP (u32, r->len), M_DEVBUF); */
	FREE(r->alloc_addr, M_DEVBUF);
	GM_PRINT (GM_PRINT_LEVEL >= 4, ("bzeroing\n"));
	bzero((void *) r, sizeof(*r));

	GM_RETURN_NOTHING ();
}

/* Return the DMA address of the next page in the DMA region. */

static
    gm_dp_t
_gm_dma_region_dma_addr(gm_arch_dma_region_t * r, int advance)
{
	gm_dp_t ret;
	sg_entry_t sge;
	vm_size_t sge_addr;
	gm_s32_t sge_count;

	GM_PRINT (GM_PRINT_LEVEL >= 7, ("_gm_dma_region_dma_addr(0x%lx, %d)\n", (unsigned long) r, advance));

	if (r->sge == 0 || r->sge_cnt >= r->sge->bc) {
		gm_assert(!r->sge || r->sge_cnt == r->sge->bc);
		r->sge = dma_get_next_sgentry(r->dma_handle);
		if (!r->sge) {
			GM_NOTE (("The dma list has been exhausted, the caller is probably wrong\n"));
			return 0;
		}
		gm_assert(r->sge && r->sge->bc);
		gm_assert(GM_PAGE_ALIGNED(r->sge->bc));
		gm_assert(GM_PAGE_ALIGNED(r->sge->ba));
		GM_PRINT (GM_PRINT_LEVEL >= 7, ("Using cookie: (0x%x, 0x%x)\n", r->sge->ba, r->sge->bc));
		r->sge_cnt = 0;
	}

	ret = (gm_dp_t) r->sge->ba + r->sge_cnt;

	if (advance)
		r->sge_cnt += GM_PAGE_LEN;

	return ret;
}

gm_dp_t
gm_arch_dma_region_dma_addr(gm_arch_dma_region_t * r)
{
	return _gm_dma_region_dma_addr(r, 0);
}

gm_dp_t
gm_arch_dma_region_dma_addr_advance(gm_arch_dma_region_t * r)
{
	return _gm_dma_region_dma_addr(r, 1);
}

/* Return the kernel virtual address for a DMA region */

void *
gm_arch_dma_region_kernel_addr(gm_arch_dma_region_t * r)
{
	return r->addr;
}

/* Return the STS bits for a DMA region. */


gm_s32_t
gm_arch_dma_region_status(gm_arch_dma_region_t * r)
{
	return 0xf;
}

gm_status_t
_gm_kernel_recv_queue_update(gm_port_t * p)
{
	return GM_SUCCESS;
}



/************
 * I/O space mapping
 ************/


#if GM_SUPPORT_PCI
gm_status_t
gm_arch_enable_PCI_bus_master_DMA(gm_instance_state_t * is)
{
	GM_PRINT (GM_PRINT_LEVEL >= 4, ("Enabling PCI bus master DMA.\n"));

	if (!is->arch.pci_cfg_hdr) {
		GM_NOTE (("internal error: is->arch.pci_cfg_hdr not set.\n"));
		return GM_FAILURE;
	}

	is->ifc.pci.config.Command
		= READ_BUS_D16(((io_handle_t) is->arch.pci_cfg_hdr->config_base) |
					   GM_OFFSETOF(gm_pci_config_t, Command));
	is->ifc.pci.config.Command |= GM_PCI_COMMAND_MASTER;
	WRITE_BUS_D16(((io_handle_t) is->arch.pci_cfg_hdr->config_base) |
				  GM_OFFSETOF(gm_pci_config_t, Command),
				  is->ifc.pci.config.Command);

	/* Pause for at least 10ms */
	/*   gm_arch_spin (is, 15000); */
	mb();

	return GM_SUCCESS;
}

gm_status_t
gm_arch_disable_PCI_bus_master_DMA(gm_instance_state_t * is)
{
	GM_PRINT (GM_PRINT_LEVEL >= 4, ("Disabling PCI bus master DMA.\n"));

	if (!is->arch.pci_cfg_hdr) {
		GM_NOTE (("internal error: is->arch.pci_cfg_hdr not set.\n"));
		return GM_FAILURE;
	}

	is->ifc.pci.config.Command
		= READ_BUS_D16(((io_handle_t) is->arch.pci_cfg_hdr->config_base) |
					   GM_OFFSETOF(gm_pci_config_t, Command));
	is->ifc.pci.config.Command &= ~GM_PCI_COMMAND_MASTER;
	WRITE_BUS_D16(((io_handle_t) is->arch.pci_cfg_hdr->config_base) |
				  GM_OFFSETOF(gm_pci_config_t, Command),
				  is->ifc.pci.config.Command);

	/* Pause for at least 10ms */
	/*   gm_arch_spin (is, 15000); */
	mb();

	return GM_SUCCESS;
}
#endif							/* GM_SUPPORT_PCI */

gm_status_t
gm_arch_map_io_space(gm_instance_state_t * is, gm_u32_t offset, gm_u32_t len,
					 void **kaddr)
{
	int reg_set;
	vm_offset_t paddr;

#if 0
	if (gm_reg_set(is, &reg_set) != GM_SUCCESS)
		return GM_FAILURE;
#endif

	paddr = iohandle_to_phys(is->arch.csr_base,
							 HANDLE_LINEAR_SPACE | HANDLE_LONGWORD);

	if (!paddr) {
		paddr = iohandle_to_phys(is->arch.csr_base,
								 HANDLE_DENSE_SPACE | HANDLE_LONGWORD);
	}

	if (!paddr) {
		GM_WARN
			(("iohandle_to_phys failed for LINEAR and DENSE   base = 0x%p\n",
			  is->arch.csr_base));
	}


	*kaddr = (void *) PHYS_TO_KSEG(paddr);

	if (debug) {
		GM_PRINT (GM_PRINT_LEVEL >= 2, ("paddr %p kaddr %p\n", paddr, *kaddr));
	}
	return GM_SUCCESS;
}

void
gm_arch_unmap_io_space(gm_instance_state_t * is, gm_u32_t offset, gm_u32_t len,
					   void **kaddr)
{
	return;
}

/***********************************************************************
 * Utility functions
 ***********************************************************************/


/*********************************************************************
 * other stuff
 *********************************************************************/

/************************************************************************
 * Module entry points    (called via dev_ops structure)
 ************************************************************************/
/* Unattach everything in reverse order */

/************
 * Character device driver entry points.
 ************/

/* open a device.  For GM, this is a clone device, and we don't
   do much initialization here, other than to allocate a new minor number
   and uninitialized gm_port_state.  Later, a GM_SET_PORT_NUMBER ioctl
   will allow the gm_port_state to be fully initialized. */

static int
gm_osf1_open(dev_t dev, int flags, int format)
{
	gm_status_t status;
	gm_port_state_t *ps;
	gm_instance_state_t *is;
	int unit;
	u_int minor_id;

	GM_CALLED ();

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_osf1_open: open called gm dev %d\n", minor(dev)));

	minor_id = minor(dev);
	if (minor_id >= (MAX_NMYRI * 2)) {
		GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_osf1_open: open again\n"));
	}
	else {
		if (minor_id < 0)
			return NODEV;

		unit = minor_id / 2;
		is = &gm_softc[unit];

		if (!is || !is->arch.ctlr) {
			return ENODEV;
		}
	}

	/* success, so commit */

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("User opened device with minor number 0x%x.\n",
				 minor(dev)));
	GM_RETURN_INT (ESUCCESS);
}

static int
gm_osf1_close(dev_t dev, int flags, int format)
{
	gm_port_state_t *ps;
	gm_instance_state_t *is;
	int unit;

	GM_CALLED ();

	GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_osf1_close called\n"));

	if (minor(dev) >= (MAX_NMYRI * 2)) {

		ps = gm_minor_get_port_state(GET_MINOR(minor(dev)));
		gm_assert(ps);

		if (ps->opened)
			gm_port_state_close(ps);
		gm_minor_free(GET_MINOR(minor(dev)));
	}
	else {
		unit = minor(dev) / 2;

		is = &gm_softc[unit];

		if (!is || !is->arch.ctlr) {
			return ENODEV;
		}
	}

	GM_RETURN_INT (0);
}

/* Map interface memory into the process.  This can be used to map the
   LANai control register, special registers, SRAM, or copy block into
   the user space.  The offsets and sizes of these various regions can
   be obtained using the GM_GET_MAPPING_SPECS ioctl, which returns
   a description of the mappings in a "gm_mapping_specs"
   structure.

   The pages of the copy block must be mapped in order and without
   skipping any offsets.

   gm_segmap ensures that all offsets passed to gm_mmap are
   legitimate, so there is no need to range check offsets here.
   However, checks must be performed here for "promiscuous" mappings
   of the device registers an SRAM areas other than the "send queue"
   and "activation" pages. */

static int
gm_osf1_mmap(dev_t dev, off_t off, int prot)
{
	void *result;
	gm_port_state_t *ps;

	int unit = GET_MINOR(minor(dev));
	gm_instance_state_t *is;
	u_long phys;
	long pfn = 0;
	off_t memory_offset = 0;
	int error;
	gm_status_t status;

	GM_CALLED ();

	GM_PRINT (GM_PRINT_LEVEL >= 2, ("gm_osf1_mmap(0x%x, 0x%x, 0x%x) called\n", dev, off, prot));

	ps = gm_minor_get_port_state(unit);
	gm_assert(ps);
	gm_arch_mutex_enter(&ps->sync);

	/* Verify the mapping is legal and ensure all resources are
	   available, allocating them if needed. */

	error = gm_prepare_to_mmap(ps, off, GM_PAGE_LEN,
							   (prot & PROT_WRITE ? GM_MAP_WRITE : 0
								| prot & PROT_READ ? GM_MAP_READ : 0));
	if (error != GM_SUCCESS) {
		GM_NOTE (("Error preparing to mmap.\n"));
		goto abort_with_mutex;
	}

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("Mapping offset 0x%x for minor device %d\n", off,
				 minor(dev)));

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_osf1_mmap: Calling gm_mmap().\n"));
	status = gm_mmap(ps, off, &result);
	if (status != GM_SUCCESS) {
		GM_NOTE (("gm_osf1_mmap: gm_mmap failed\n"));
		goto abort_with_nothing;
	}
	gm_assert(result);


#define MAPPING_OF_TYPE(type, off, permissions)                         \
  (ps->mappings.type.offset <= off                                      \
   && off < (ps->mappings.type.offset + ps->mappings.type.len))



	if ((MAPPING_OF_TYPE(copy_block, off, permissions) ||
		 MAPPING_OF_TYPE(recv_queue, off, permissions))) {
		phys = vtop(NULL, result);
	}
	else {
		phys = KSEG_TO_PHYS(result);
	}

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_osf1_mmap: phys = 0x%lx\n", phys));

	pfn = pmap_phys_to_frame(phys);

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_osf1_mmap: kseg 0x%lx frame 0x%lx\n", result, pfn));
  abort_with_mutex:
	gm_arch_mutex_exit(&ps->sync);
	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_osf1_mmap: returning 0x%lx.\n", pfn));
	GM_RETURN_INT ((int) pfn);


  abort_with_nothing:
	GM_RETURN_INT (0);
}

static int
gm_osf1_localize_status(gm_status_t status)
{
#define CASE(from,to) case from : return to
	switch (status) {
		 CASE(GM_SUCCESS, 0);
		 CASE(GM_INPUT_BUFFER_TOO_SMALL, EFAULT);
		 CASE(GM_OUTPUT_BUFFER_TOO_SMALL, EFAULT);
		 CASE(GM_TRY_AGAIN, EAGAIN);
		 CASE(GM_BUSY, EBUSY);
		 CASE(GM_MEMORY_FAULT, EFAULT);
		 CASE(GM_INTERRUPTED, EINTR);
		 CASE(GM_INVALID_PARAMETER, EINVAL);
		 CASE(GM_OUT_OF_MEMORY, ENOMEM);
		 CASE(GM_INVALID_COMMAND, EINVAL);
		 CASE(GM_PERMISSION_DENIED, EPERM);
		 CASE(GM_INTERNAL_ERROR, EPROTO);
		 CASE(GM_UNATTACHED, -1);	/* ??? */
		 CASE(GM_UNSUPPORTED_DEVICE, ENXIO);
	 default:
		 return gm_osf1_localize_status(GM_INTERNAL_ERROR);
	}
}


static
    gm_status_t
gm_osf1_ioctl(dev_t dev, u_int cmd, caddr_t arg, int mode)
{
	gm_port_state_t *ps;
	int error,rv;
	unsigned copied_out;


	/* Make sure ioctl is not being called by the kernel, since we cannot
	   handle that yet. */
#if 0
	gm_assert(!(mode & FKIOCTL));
#endif

	GM_PRINT (GM_PRINT_LEVEL >= 7, ("gm_osf1_ioctl: called for dev with minor number 0x%x.\n",
				  minor(dev)));
	GM_PRINT (GM_PRINT_LEVEL >= 7, ("gm_osf1_ioctl: cmd = 0x%x  arg = 0x%lx  mode = 0x%x\n",
				  cmd, arg, mode));

	if (cmd == GM_GET_DEV) {
		gm_instance_state_t *is;
		gm_port_state_t *clone_ps;
		gm_status_t status = ESUCCESS;
		u_int minor_id = minor(dev);

		GM_PRINT (GM_PRINT_LEVEL >= 10, ("gm_osf1_ioctl: command is GM_GET_DEV\n"));

		if (minor_id >= (MAX_NMYRI * 2)) {
			GM_NOTE (("gm_osf1_ioctl() WRONG %d\n", minor_id));
			return ENODEV;
		}

		is = &gm_softc[(u_int) minor_id / 2];

		if (!is || !is->arch.ctlr) {
			return ENODEV;
		}

		clone_ps = gm_minor_get_port_state(minor(dev));

		GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_osf1_ioctl: Allocating minor number %d %d.\n", 
			minor(dev), minor_id));
		/* Determine an unused minor number for the opened device. */

		if (gm_minor_alloc(clone_ps->instance, &minor_id) != GM_SUCCESS) {
			GM_NOTE (("gm_osf1_ioctl: Could not create new minor number.\n"));
			error = EINVAL;
			goto abort_with_nothing;
		}

		if (INVALID_MINOR(minor_id)) {
			gm_minor_free(minor_id);
			goto abort_with_nothing;
		}

		GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_osf1_ioctl: Using minor number %d.\n", minor_id));
		ps = gm_minor_get_port_state(minor_id);
		gm_assert(ps);

		/* Record the instance associated with this minor number. */

		ps->instance = clone_ps->instance;
		gm_assert(ps->instance);

		/* Record the privileges from the clone device */

		ps->privileged = clone_ps->privileged;

		GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_osf1_ioctl: Copyout %d to arg = 0x%lx\n", minor_id, arg));
		{
			int *a = (int *) arg;
			a[0] = EXTERN_MINOR(minor_id);
			GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_osf1_ioctl: new minor?? = 0x%x\n", a[0]));
		}
		return ESUCCESS;
	}
	else {
#if 0
		if (((unsigned int) cmd) < 0xFFFFFFFF) {
			GM_NOTE (("gm_osf1_ioctl: bad cmd for ioctl?? = 0x%x\n", cmd));
			return ENODEV;
		}
#endif
		ps = gm_minor_get_port_state(GET_MINOR(minor(dev)));
		gm_assert(ps);
		gm_arch_mutex_enter(&ps->sync);
		/* Call the portable IOCTL. */
		ps->arch.ddi_copyxx_context = mode;
		rv = gm_ioctl(ps, cmd, (void *) arg, INT_MAX,
			 (void *) arg, INT_MAX, &copied_out);
		error = gm_osf1_localize_status(rv);
		if (error == 0) {
			error = ESUCCESS;
		}
		GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_osf1_ioctl(%s): rv=%d err=%d copied %d bytes\n", 
			_gm_ioctl_cmd_name(cmd), rv,error, copied_out));

		gm_arch_mutex_exit(&ps->sync);
	}
  abort_with_nothing:
	return (error);

}



#if GM_BC_DRIVER
#error broken minor number details

/* Implement gm_read so we can return EEPROM when emulating mlanai
   driver. */

static int
gm_read(dev_t dev, struct uio *uio, cred_t * credp)
{
	gm_instance_state_t *is;

	GM_CALLED ();

	if (!GM_COMPATIBILITY_DEVICE_MINOR_P(getminor(dev)))
		return EACCES;

	is = ddi_get_soft_state(gm_instancep,
							GM_MINOR_TO_INSTANCE(getminor(dev)));

	RETURN(uiomove((caddr_t) & is->lanai.eeprom,
				   min(sizeof(gm_gmnet_eeprom_t), uio->uio_iov->iov_len),
				   UIO_READ, uio));
}
#endif


static int
gm_osf1_cattach(struct controller *ctlr)
{
	struct handler_intr_info gm_intr_info;
	ihandler_t gm_ihandle;
	int unit;
	vm_offset_t paddr;
	int i;
	gm_instance_state_t *is;
	int instance;
	enum gm_bus_type bus_type;
	int ipok = 0;

	if (debug) {
		printf("gm: gm_osf1_cattach(%p) entered\n", ctlr);
	}

	unit = ctlr->ctlr_num;
	is = &gm_softc[unit];

	GM_PAGE_LEN = PAGE_SIZE;


	/* Determine the ifc type. */

	bus_type = GM_MYRINET_BUS_PCI;	/* PCI only */


	/* Initialize the interface, including mapping the board regions,
	   copying eeprom, and starting the LANai control program (with
	   interrupts disabled.) */

	printf("\n");

    GM_INFO (("GM driver version %s build %s\n", _gm_version, _gm_build_id));

	if (gm_instance_init(is, unit, bus_type) != GM_SUCCESS) {
		GM_NOTE (("[%d]: Could not initialize instance.\n", unit));
		goto abort_with_initialized_config_state;
	}

	/* enable interrupts.  HACK: This must be done only after it is safe
	   to call the interrupt handler for this device, which we installed
	   earlier. */

	handler_enable(is->arch.hid);


	GM_PRINT (GM_PRINT_LEVEL >= 1, ("Enabling interrupts for instance %d.\n", is->id));
	if (gm_enable_interrupts(is) != GM_SUCCESS) {
		GM_NOTE (("Cannot enable interrupts.\n"));
		goto abort_with_initialized_instance;
	}


  /************
   * Create minor devices.  /dev/gm%d is the clone device for normal users,
   * and /dev/gmp%d is the clone device for privileged users.
   ************/

	/* Determine minor node number to use for /dev/gm? . */

	GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor number.\n"));
	if (gm_minor_alloc(is, &is->clone_minor) != GM_SUCCESS) {
		GM_NOTE (("Failed to allocate minor number.\n"));
		goto abort_with_enabled_interrupts;
	}
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("using minor number %ld.\n", (long) is->clone_minor));
	gm_minor_get_port_state(is->clone_minor)->instance = is;
	gm_minor_get_port_state(is->clone_minor)->privileged = 0;

	/* Determine minor node number to use for /dev/gmp? . */

	GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor number.\n"));
	if (gm_minor_alloc(is, &is->privileged_clone_minor) != GM_SUCCESS) {
		GM_NOTE (("Failed to allocate minor number.\n"));
		goto abort_with_minor;
	}
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("using minor number %ld.\n",
				 (long) is->privileged_clone_minor));
	gm_minor_get_port_state(is->privileged_clone_minor)->instance = is;
	gm_minor_get_port_state(is->privileged_clone_minor)->privileged = 1;


#if 1
	GM_INFO (("Attaching the IP interface\n"));
	ipok = gd_attach(unit);
	if (!ipok) {
		GM_NOTE (("IP interface attach FAILED\n"));
	}
	else {
		GM_INFO (("IP interface attach ok\n"));
	}
#else
	GM_INFO (("NOT attaching the IP interface.\n"));
#endif

	if (ipok) {
		return 1;
	}

/********************* Error handling */

  abort_with_minor:
	gm_minor_free(is->clone_minor);
  abort_with_enabled_interrupts:
	gm_disable_interrupts(is);

	if (handler_disable(is->arch.hid) != 0)
		return (0);

	if (handler_del(is->arch.hid) != 0)
		return (0);

  abort_with_initialized_instance:
	gm_instance_finalize(is);
  abort_with_initialized_config_state:
  abort_with_page_lock_hash:
  abort_with_instance_state:
  abort_with_nothing:
	is->arch.ctlr = NULL;
	GM_RETURN_INT (0);

	return ESUCCESS;
}

static int
gm_dattach(struct controller *ctlr)
{
	/* Code for gmcattach interface goes here. */


	if (debug) {
		printf("gm: dattach\n");
	}
	return ESUCCESS;
}

static int
gm_osf1_ctlr_unattach(struct bus *bus, struct controller *ctlr)
{
	gm_instance_state_t *is;
	int instance = ctlr->ctlr_num;
	/*
	 * Not clear what this is going to do for the Myrinet
	 */
	if (debug) {
		printf("gm: unattach\n");
	}


	GM_CALLED ();
	is = &gm_softc[instance];

	if (is && is->arch.ctlr) {
		GM_PRINT (GM_PRINT_LEVEL >= 5, ("Called gm_osf1_ctlr_unattach()\n"));
		gm_minor_free(is->privileged_clone_minor);
		gm_minor_free(is->clone_minor);
		gm_disable_interrupts(is);

		if (handler_disable(is->arch.hid) != 0)
			return (ENODEV);

		if (handler_del(is->arch.hid) != 0)
			return (ENODEV);

		gm_instance_finalize(is);

		is->arch.ctlr = NULL;
	}

	return (ESUCCESS);			/* Return success status */
}

void
zmemcpy(dest, source, len)
unsigned char *dest;
unsigned char *source;
u_int len;
{
	if (len == 0)
		return;
	do {
		*dest++ = *source++;	/* ??? to be unrolled */
	} while (--len != 0);
}

struct controller *
gm_osf1_get_controller(int unit)
{
	return gm_softc[unit].arch.ctlr;
}


/*
  This file uses feldy indentation:

  Local Variables:
  tab-width:4
  c-file-style:"bsd"
  End:
*/
