#include <ndis.h>
#include "gmm.h"

/*
     6 bytes       6 bytes       2 bytes      Up to 1500 bytes
   +-------------+-------------+-------------+-------------------------+
   | Destination | Source      | E-type      | Network Protocol Packet |
   | MAC Address | MAC Address |(IPX: 8137)  |                         |
   +-------------+-------------+-------------+-------------------------+
*/


int gmm_tryToGatherBuffer (gmm_Context*context,
			   NDIS_PACKET*packet,
			   gmm_Buffer*buffer,
			   char ethernetAddress[6],
			   gm_ethernet_segment_descriptor_t*segments)
{

  unsigned int nBuffers, nSegments;
  UINT nAddresses;
  unsigned int i, j;
  unsigned int totalLength, length;
  ULONG low, high;
  NDIS_BUFFER*ndisBuffer;
  void*a;
  NDIS_PHYSICAL_ADDRESS_UNIT addresses [GM_MAX_ETHERNET_GATHER_CNT];

  insist (segments && packet);
  
  insist (packet && buffer && ethernetAddress);

  GMM_PRINT (("gmm_tryToGatherBuffer: buffer = %x\n", buffer));
  
  NdisQueryPacket (packet, NULL, &nBuffers, &ndisBuffer, &totalLength);

  insist (totalLength <= GMM_MTU - 8);
  insist (ndisBuffer);
  insist (nBuffers > 0);

  if (totalLength < GMM_SMALLEST_GATHER)
  {

    GMM_PRINT (("buffer is too small to be worth gathering.\n"));
    return 0;
  }
    
  GMM_PRINT (("NDIS Buffer is in %d pieces \n", nBuffers));

  GMM_PRINT (("gmm_tryToGatherBuffer: totalLength is %d\n", totalLength));
  
  if (nBuffers > GM_MAX_ETHERNET_GATHER_CNT)
    return 0;

#if !GMM_EMULATE_BYTE_DMAS
  if (totalLength != GM_DMA_ROUNDUP (u32, totalLength))
    return 0;
#endif

  
  /*get dest ethernet address*/
  NdisQueryBuffer (ndisBuffer, &a, &length);

  insist (a);
  
  NdisMoveMemory (ethernetAddress, a, 6);

  nSegments = buffer->numNtBuffers = 0;
  buffer->length = 0;
  
  for (i = 0; i < nBuffers; i++)
  {
    insist (ndisBuffer);

    GMM_PRINT (("gmm_tryToGatherBuffer: calling NdisQueryBuffer\n"));
  
    NdisQueryBuffer (ndisBuffer, &a, &length);
#if !GMM_EMULATE_BYTE_DMAS
    if (length != GM_DMA_ROUNDUP (u32, length))
      return 0;
#endif

    GMM_PRINT (("gmm_tryToGatherBuffer: calling NdisGetBufferPhysicalArraySize\n"));
  
    NdisGetBufferPhysicalArraySize (ndisBuffer, &nAddresses);

    GMM_PRINT (("gmm_tryToGatherBuffer: NdisGetBufferPhysicalArraySize returned %d\n", nAddresses));

    if (nAddresses + nSegments > GM_MAX_ETHERNET_GATHER_CNT)
      goto abortWithMapRegisters;

    insist (buffer->mapRegister + buffer->numNtBuffers >= 0);
    insist (buffer->mapRegister + buffer->numNtBuffers < GMM_NUM_MAP_REGISTERS);

      
    GMM_PRINT (("gmm_tryToGatherBuffer: calling NdisMStartBufferPhysicalMapping\n"));
    GMM_PRINT (("buffer = %x\n", buffer));
    GMM_PRINT (("context = %x\n", context));
    GMM_PRINT (("context->miniportAdapterHandle = %x\n", context->miniportAdapterHandle));

    GMM_PRINT (("ndisBuffer = %x\n", ndisBuffer));
    GMM_PRINT (("buffer->mapRegister + buffer->numNtBuffers = %x\n", buffer->mapRegister + buffer->numNtBuffers));
    GMM_PRINT (("TRUE = %x\n", TRUE));
    GMM_PRINT (("addresses = %x\n", addresses));
    GMM_PRINT (("&nAddresses = %x\n", &nAddresses));

    NdisMStartBufferPhysicalMapping
    (
     context->miniportAdapterHandle,
     ndisBuffer,
     buffer->mapRegister + buffer->numNtBuffers,
     TRUE,
     addresses,
     &nAddresses);
    
    GMM_PRINT (("gmm_tryToGatherBuffer: nAddresses is %d\n", nAddresses));
  
    if (nAddresses == 0)
      goto abortWithMapRegisters;
    
    buffer->ntBuffers [buffer->numNtBuffers++] = ndisBuffer;

    if (nAddresses + nSegments > GM_MAX_ETHERNET_GATHER_CNT)
      goto abortWithMapRegisters;

    for (j = 0; j < nAddresses; j++)
    {
      length = addresses[j].Length;
      insist (length > 0 && length + buffer->length <= totalLength);

#if !GMM_EMULATE_BYTE_DMAS
      if (length != GM_DMA_ROUNDUP (u32, length))
	goto abortWithMapRegisters;
#endif

      low = NdisGetPhysicalAddressLow (addresses[j].PhysicalAddress);
      high = NdisGetPhysicalAddressHigh (addresses[j].PhysicalAddress);

      insist (high == 0);
      
      if (low == 0
#if !GMM_EMULATE_BYTE_DMAS
	  || !GM_DMA_ALIGNED (low)
#endif
	  )
	goto abortWithMapRegisters;
      
      segments[nSegments].ptr = gm_hton_dp (low);
      segments[nSegments].len = gm_htonl (length);

      GMM_PRINT (("gmm_tryToGatherBuffer: ptr is %x, length is %d \n", low, length));
      nSegments++;

      buffer->length += length;  
    }
    NdisFlushBuffer (ndisBuffer, TRUE);
    NdisGetNextBuffer (ndisBuffer, &ndisBuffer);
  }

  
  insist (buffer->length == totalLength);
  buffer->ntPacket = packet;
  GMM_PRINT (("gmm_tryToGatherBuffer: buffer->ntPacket = %x\n", buffer->ntPacket));
  GMM_PRINT (("gmm_tryToGatherBuffer: returning %d \n", nSegments));
  
  return nSegments;

  abortWithMapRegisters:
  for (i = 0; i < buffer->numNtBuffers; i++)
    NdisMCompleteBufferPhysicalMapping (context->miniportAdapterHandle, buffer->ntBuffers [i], i);

  exception:
  return 0;				
}

int gmm_copyPacketToBuffer (NDIS_PACKET*packet, gmm_Buffer*buffer, char ethernetAddress[6],
			    gm_ethernet_segment_descriptor_t*segments)
{
  UINT i, nBuffers, totalLength, length;
  NDIS_BUFFER*ndisBuffer;
  char*p, *a;
  gm_u32_t seglen;

  insist (sizeof (short) == 2);
  insist (packet && buffer && ethernetAddress);

  GMM_PRINT (("gmm_copyPacketToBuffer: buffer = %x\n", buffer));
  
  NdisQueryPacket (packet, NULL, &nBuffers, &ndisBuffer, &totalLength);
  
  insist (totalLength <= GMM_MTU);
  insist (ndisBuffer);
  insist (nBuffers > 0);

  p = buffer->virtualAddress;
  insist (p);
  
  buffer->length = 0;
  buffer->numNtBuffers = 0;
  
  /*get dest ethernet address*/
  NdisQueryBuffer (ndisBuffer, &a, &length);

  insist (a);
  
  NdisMoveMemory (ethernetAddress, a, 6);

  GMM_PRINT (("NDIS Buffer is in %d pieces \n", nBuffers));
 
  for (i = 0; i < nBuffers; i++)
  {
    insist (ndisBuffer);
    
    NdisQueryBuffer (ndisBuffer, &a, &length);
    insist (length > 0 && length + buffer->length <= totalLength + sizeof (short));

    GMM_PRINT (("Copying NDIS buffer piece ptr = %x, length = %d\n", a, length));
 
    NdisMoveMemory (p, a, length);
    p+= length;
    buffer->length += length;  
    NdisGetNextBuffer (ndisBuffer, &ndisBuffer);
  }

  insist (buffer->length == totalLength);
  buffer->ntPacket = packet;
  GMM_PRINT (("gmm_copyPacketToBuffer: buffer->ntPacket = %x\n", buffer->ntPacket));
  
  NdisFlushBuffer (buffer->ndisBuffer, TRUE);
  
#if GMM_EMULATE_BYTE_DMAS
  seglen = buffer->length;
#else
  seglen = GM_DMA_ROUNDUP (u32, buffer->length);
#endif
  insist (seglen <= GMM_MTU - 8);
    
  segments[0].ptr = gm_hton_dp (NdisGetPhysicalAddressLow (buffer->physicalAddress));
  segments[0].len = gm_htonl (seglen);

  GMM_PRINT (("ptr = %x, len = %d\n", segments [0].ptr, gm_htonl (segments [0].len)));
  
  return 1;
  
  exception: return 0;
}



void gmm_dumpPacket (char*p, int length)
{
  int i;
  
  DbgPrint ("dumping packet, length %d\n", length);

  if (length > 16)
    length = 16;
  
  for (i = 0; i < length; i++)
    DbgPrint ("%02x%c",0xff & p[i], i % 8 == 7 ? '\n' : ' ');
  DbgPrint ("\n");
}


void gmm_dumpPieces (NDIS_PACKET*packet)
{
  UINT i, nBuffers, totalLength, length;
  NDIS_BUFFER*ndisBuffer;
  char*a;

  insist (sizeof (short) == 2);
  insist (packet);

  NdisQueryPacket (packet, NULL, &nBuffers, &ndisBuffer, &totalLength);
  
  insist (totalLength <= GMM_MTU);
  insist (ndisBuffer);
  insist (nBuffers > 0);
  
  /*get dest ethernet address*/
  NdisQueryBuffer (ndisBuffer, &a, &length);

  insist (a);

  GMM_PRINT (("NDIS Buffer is in %d pieces \n", nBuffers));
 
  for (i = 0; i < nBuffers; i++)
  {
    insist (ndisBuffer);
    
    NdisQueryBuffer (ndisBuffer, &a, &length);

    GMM_PRINT (("dumping NDIS buffer piece ptr = %x, length = %d\n", a, length));

    gmm_dumpPacket (a, length);
    NdisGetNextBuffer (ndisBuffer, &ndisBuffer);
  }
  
  exception: return;
}

BOOLEAN gmm_SendPacket (gmm_Context *context,
			PNDIS_PACKET packet,
			gmm_Buffer *b)
{
  UINT length;
  UINT broad;
  UINT nSegments;
  unsigned char ethernetAddress [6];
  gm_ethernet_segment_descriptor_t segments[GM_MAX_ETHERNET_GATHER_CNT];
  void*address;

#if GMM_EMULATE_BYTE_DMAS
    /*avoid copy if possible*/

    nSegments = gmm_tryToGatherBuffer (context,
				     packet,
				       b,
				       ethernetAddress,
				       segments);
#endif

#if GMM_EMULATE_BYTE_DMAS
    if (nSegments == 0)
    {
#endif
      /*copy packet into buffer*/
      GMM_PRINT (("gmm_SendPackets: calling gmm_copyPacketToBuffer\n"));
  
      if (!gmm_copyPacketToBuffer (packet, b, ethernetAddress, segments))
      {
	gmm_queuePut (&context->sendFreeQueue, b);
	NdisReleaseSpinLock (&context->receiveQueueSpinLock);
	  return FALSE;
      }
    
      NdisQueryBuffer (b->ndisBuffer, &address, &length);
      insist (address == b->virtualAddress && length == b->originalLength);

      nSegments = 1;
      dumpPacket (address, gm_htonl (segments[0].len));

#if GMM_EMULATE_BYTE_DMAS
    }
    else
    {
      dumpPieces (packet);
      
      GMM_PRINT (("gmm_SendPackets: actually avoided copying %d segments!\n",
		  nSegments));
    }
#endif    
    insist (nSegments > 0);
    insist (packets[i] == b->ntPacket);
    
    GMM_PRINT (("Sending %d segments to ether: %x.%x.%x.%x.%x.%x.\n", nSegments,ethernetAddress[0],ethernetAddress[1],ethernetAddress[2],ethernetAddress[3],ethernetAddress[4],ethernetAddress[5]));

    broad = ethernetAddress[0] & 0x01;

  NDIS_SET_PACKET_STATUS (packet, NDIS_STATUS_PENDING);
    gmm_queuePut (&context->sendPendQueue, b);
    
    //context->sends++;
    context->gmm_numGoodSends++;

    if (broad)
      gm_ethernet_broadcast (context->port, nSegments, segments, 1);
    else        
      gm_ethernet_send (context->port, nSegments, segments, ethernetAddress, 1);
  return TRUE;
 exception:
  return FALSE;
}

void gmm_SendPackets (NDIS_HANDLE miniportAdapterContext,
		      PPNDIS_PACKET packets,
		      UINT numPackets)
{
  UINT i;
  gmm_Buffer*b;
  gmm_Context*context = (gmm_Context*)miniportAdapterContext;
  
  GMM_PRINT (("gmm_SendPackets called. numPackets = %d.\n", numPackets));
  NdisAcquireSpinLock (&context->receiveQueueSpinLock);
  insist (context);
  
  /*first mark all packets as dropped.*/
  for (i = 0; i < numPackets; i++)
    NDIS_SET_PACKET_STATUS (packets[i], NDIS_STATUS_RESOURCES);
  
  /*now go through and send each packet*/
  
  for (i = 0; i < numPackets; i++)
  {
    /*get a new send buffer to (possibly) copy packet into*/
    
    if (!(b = gmm_queueGet (&context->sendFreeQueue)))
    {
      GMM_PRINT (("no free send packets at all.\n"));
      for (; i < numPackets; ++i)
	{
	  ENQUEUE_PACKET (context->firstTxQueue, context->lastTxQueue,
			  packets[i]);
	  NDIS_SET_PACKET_STATUS (packets[i], NDIS_STATUS_PENDING);
	  context->numPacketsQueued++;
	}
      NdisReleaseSpinLock (&context->receiveQueueSpinLock);
      return;
    }
    
    GMM_PRINT (("Got Buffer = %x\n", b));
    gmm_SendPacket (context, packets[i], b);
  
  }
  GMM_PRINT (("gmm_SendPackets: returning\n"));
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  return;
 exception:
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  DbgPrint ("gmm_SendPackets exception\n");
  DbgBreakPoint ();
}













