#include <ndis.h>
#include "gmm.h"

static int gmm_receiveFeedBuffer (gmm_Context*context, gmm_Buffer*buffer)
{
  gm_ethernet_segment_descriptor_t segment;

  insist (context && buffer);

  insist (NdisGetPhysicalAddressLow (buffer->physicalAddress));
  insist (buffer->originalLength >= GMM_MTU); /* big enough for receive */
  insist (GM_DMA_ALIGNED (buffer->originalLength)); /* GM compatible */

  segment.ptr = gm_hton_dp (NdisGetPhysicalAddressLow (buffer->physicalAddress));
  segment.len = gm_htonl (buffer->originalLength);
       
  _gm_provide_ethernet_scatter_list (context->port, 1, &segment);
  context->receives++;
  
  gmm_queuePut (&context->receivePendQueue, buffer);
  
  return 1;
  exception:return 0;
}


int gmm_receiveFeedBuffers (gmm_Context*context)
{
  gmm_Buffer*b;
  
  insist (context);

  while (b = gmm_queueGet (&context->receiveFreeQueue))
    gmm_receiveFeedBuffer (context, b);

  return 1;
  exception: return 0;
}


static int gmm_receiveFormPacket (gmm_Context*context, gmm_Buffer*buffer)
{
  insist (context && buffer);

  GMM_PRINT (("gmm_receive. length = %d\n", buffer->length));

  insist (buffer->length > 0 && buffer->length <= GMM_MTU);

  if (buffer->length <= GMM_HEADER_SIZE)
  {
    GMM_ALWAYS_PRINT (("dropping received packet\n"));
    context->gmm_numNoBuffers++;
    gmm_receiveFeedBuffer (context, buffer);
    return 0;
  }

  NdisFlushBuffer (buffer->ndisBuffer, FALSE);
  NdisMUpdateSharedMemory (context->miniportAdapterHandle,
			   buffer->length,
			   buffer->virtualAddress,
			   buffer->physicalAddress);
  
  NdisAdjustBufferLength (buffer->ndisBuffer, buffer->length - GMM_HEADER_SIZE);
#if 0 /* feldy - was 1 */
  NDIS_SET_PACKET_STATUS (buffer->myPacket, NDIS_STATUS_RESOURCES);
#else
  if (context->numNdisReceives >= GMM_NUM_RECEIVE_BUFFERS_THRESHOLD)
  {
    NDIS_SET_PACKET_STATUS (buffer->myPacket, NDIS_STATUS_RESOURCES);
  }
  else
  {
    NDIS_SET_PACKET_STATUS (buffer->myPacket, NDIS_STATUS_SUCCESS);
    context->numNdisReceives++;
  }
#endif
  NDIS_SET_PACKET_HEADER_SIZE (buffer->myPacket, GMM_ETHERNET_HEADER_SIZE);

  // dumpPacket ((char*)buffer->virtualAddress + 2, buffer->length - 2);
  
  return 1;
  
  exception: 0;
}


VOID gmm_ReturnPacket (NDIS_HANDLE miniportAdapterContext, PNDIS_PACKET packet)
{
  gmm_Context*context = (gmm_Context*)miniportAdapterContext;
  gmm_Buffer*b;

  NdisAcquireSpinLock (&context->receiveQueueSpinLock);
  insist (context && packet);

  b = *((gmm_Buffer**)packet->MiniportReserved);
  insist (b);

  context->numNdisReceives--;
  insist (context->numNdisReceives >= 0);
  gmm_receiveFeedBuffer (context, b);
  
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  return;
  exception:;
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  DbgPrint ("gmm_ReturnPacket execption\n");
  DbgBreakPoint ();
}

void gmm_flushReceiveDoneQueue (gmm_Context*context)
{      
  gmm_Buffer*buffers [GMM_NUM_RECEIVE_BUFFERS];
  NDIS_PACKET*ndisPackets [GMM_NUM_RECEIVE_BUFFERS];
  int nReceives, i;

  insist (context);
  
  for (nReceives = 0; nReceives < GMM_NUM_RECEIVE_BUFFERS;)
  {
    /*NdisDprAcquireSpinLock (&context->receiveQueueSpinLock);*/
    buffers [nReceives] = gmm_queueGet (&context->receiveDoneQueue);
    /*NdisDprReleaseSpinLock (&context->receiveQueueSpinLock);*/

    if (!buffers [nReceives])
      break;
    
    if (!gmm_receiveFormPacket (context, buffers [nReceives]))
      continue;
    
    ndisPackets[nReceives] = buffers [nReceives]->myPacket;
    nReceives++;
    
  }

  if (nReceives)
  {
    GMM_PRINT (("gmm_flushReceiveDoneQueue: %d receives\n", nReceives));
  }
  
  #if 1
  if (nReceives > 0)
  {  
    NdisReleaseSpinLock (&context->receiveQueueSpinLock);
    NdisMIndicateReceivePacket (context->miniportAdapterHandle, ndisPackets, nReceives);
    NdisAcquireSpinLock (&context->receiveQueueSpinLock);
    
    context->gmm_numGoodReceives += nReceives;

    for (i = 0; i < nReceives; i++)
    {
      insist (ndisPackets [i] == buffers [i]->myPacket);
      switch (NDIS_GET_PACKET_STATUS (ndisPackets[i]))
      {
      case NDIS_STATUS_SUCCESS:
	context->numNdisReceives--;
	insist (context->numNdisReceives >= 0);
	/*fallthrough*/
      case NDIS_STATUS_RESOURCES:
	gmm_receiveFeedBuffer (context, buffers[i]);
	break;
      }
    }
  }
  #else

  for (i = 0; i < nReceives; i++)
  {
    GMM_PRINT (("received %d\n", buffers[i]->length));
    
    NdisMEthIndicateReceive (context->miniportAdapterHandle,
			     &buffers [i],
			     buffers[i]->virtualAddress + GMM_HEADER_SIZE,
			     GMM_ETHERNET_HEADER_SIZE,
			     buffers[i]->virtualAddress +
			     GMM_HEADER_SIZE + GMM_ETHERNET_HEADER_SIZE,
			     buffers[i]->length - GMM_HEADER_SIZE - GMM_ETHERNET_HEADER_SIZE,
			     buffers[i]->length - GMM_HEADER_SIZE - GMM_ETHERNET_HEADER_SIZE);
    gmm_receiveFeedBuffer (context, buffers [i]);
    
  }
  #endif
  exception:;
}

/*interrupts must be disabled while calling this*/

void gmm_flushSendDoneQueue (gmm_Context*context)
{
  unsigned int i;
  gmm_Buffer*b;
  
  insist (context);
  
  for (;;)
  {
    /*NdisDprAcquireSpinLock (&context->receiveQueueSpinLock);*/
    b = gmm_queueGet (&context->sendDoneQueue);
    /*NdisDprReleaseSpinLock (&context->receiveQueueSpinLock);*/

    if (!b)
      break;
    
    for (i = 0; i < b->numNtBuffers; i++)
    {
      GMM_PRINT (("calling NdisMCompleteBufferPhysicalMapping\n"));
      GMM_PRINT (("b->mapRegister + i = %x\n", b->mapRegister + i));
      GMM_PRINT (("b->ntPacket = %x\n", b->ntPacket));
      
      NdisMCompleteBufferPhysicalMapping (context->miniportAdapterHandle,
					  b->ntBuffers[i],
					  b->mapRegister + i);
    }
    
    GMM_PRINT (("calling NdisMSendComplete\n"));
  
    NdisReleaseSpinLock (&context->receiveQueueSpinLock);
    NdisMSendComplete (context->miniportAdapterHandle,
		       b->ntPacket,
		       NDIS_STATUS_SUCCESS);
    NdisAcquireSpinLock (&context->receiveQueueSpinLock);

    GMM_PRINT (("NdisMSendComplete returned\n"));
    
    gmm_queuePut (&context->sendFreeQueue, b);
  }

  if (context->flushingSendQueue && gmm_queueEmpty (&context->sendPendQueue))
  {
    GMM_ALWAYS_PRINT (("calling NdisMResetComplete\n"));
    context->flushingSendQueue = 0;
    NdisMResetComplete (context->miniportAdapterHandle, NDIS_STATUS_SUCCESS, TRUE);
  }

  exception:;
}


static void gmm_flushCallback (NDIS_HANDLE miniportAdapterContext, PVOID callbackContext)
{
  gmm_Context*context = (gmm_Context*)miniportAdapterContext;

  insist (context);

  gmm_flushReceiveDoneQueue (context);
  gmm_flushSendDoneQueue (context);

  exception:;
}

void gmm_flushSend (NDIS_HANDLE miniportAdapterContext, PVOID callbackContext)
{
  gmm_Context *context = (gmm_Context*) callbackContext;
  gmm_Buffer*b;

  NdisAcquireSpinLock (&context->receiveQueueSpinLock);

  b = gmm_queueGet (&context->sendPendQueue);
  if (b == NULL)
    DbgBreakPoint ();
  insist (b);
  gmm_queuePut (&context->sendDoneQueue, b);

  gmm_flushSendDoneQueue (context);
  if (context->firstTxQueue != NULL)
    {
      PNDIS_PACKET packet;
      GMM_PRINT (("immediately send a queued item\n"));
      b = gmm_queueGet (&context->sendFreeQueue);
      insist (b);
      packet = context->firstTxQueue;
      DEQUEUE_PACKET (context->firstTxQueue, context->lastTxQueue);
      insist (packet);
      context->numPacketsQueued--;
      gmm_SendPacket (context, packet, b);
    }
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  return ;
 exception:
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  DbgPrint ("gmm_flushSend exception\n");
  DbgBreakPoint ();
}

void gmm_sendInterrupt (gmm_Context*context)
{
#if 1
  gmm_flushSend (context->miniportAdapterHandle, context);
#else
  NDIS_HANDLE handle;
  NDIS_STATUS status;

  insist (context);

  GMM_PRINT (("gmm_sendInterrupt: got a send interrupt."));
  //context->sends--;
  //GMM_PRINT (("%d sends now outstanding\n", context->sends));
  
  
  if (NdisIMSwitchToMiniport (context->miniportAdapterHandle, &handle))
  {
    gmm_flushSend (context->miniportAdapterHandle, context);
    NdisIMRevertBack (context->miniportAdapterHandle, handle); 
  }
  else
  {
    status = NdisIMQueueMiniportCallback (context->miniportAdapterHandle,
					  gmm_flushSend, context);

    if (status == NDIS_STATUS_FAILURE)
    {
      GMM_ALWAYS_PRINT (("NdisImQueueMiniportCallback failed\n"));
    }
  }
  exception:;
#endif
}

void gmm_receiveInterrupt (gmm_Context*context, unsigned length, gm_u16_t csum)
{
#if 1
  gmm_Buffer *b;
  NdisAcquireSpinLock (&context->receiveQueueSpinLock);
  
  insist (context);
  insist (length > 0 && length <= GMM_MTU);

  GMM_PRINT (("gmm_receiveInterrupt: got a receive done\n"));

  b = gmm_queueGet (&context->receivePendQueue);
  insist (b);
  b->length = length;

  GMM_PRINT (("gmm_checkReceiveQueue: length received was %d\n", b->length));
  gmm_queuePut (&context->receiveDoneQueue, b);

  gmm_flushReceiveDoneQueue (context);
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  return;
 exception:
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  DbgPrint ("gmm_receiveInterrupt execption\n");
  DbgBreakPoint ();
#else
  NDIS_HANDLE handle;
  NDIS_STATUS status;
  gmm_Buffer*b;
  
  insist (context);
  insist (length > 0 && length <= GMM_MTU);
  
  
  GMM_PRINT (("gmm_receiveInterrupt: got a receive done"));
  //context->receives--;
  //GMM_PRINT ((%d receives now left in the LANai.\n", context->receives));
  b = gmm_queueGet (&context->receivePendQueue);
  insist (b);
  b->length = length;
  
  GMM_PRINT (("gmm_checkReceiveQueue: length received was %d\n", b->length));
  gmm_queuePut (&context->receiveDoneQueue, b);

  if (NdisIMSwitchToMiniport (context->miniportAdapterHandle, &handle))
  {
    gmm_flushReceiveDoneQueue (context);
    NdisIMRevertBack (context->miniportAdapterHandle, handle); 
  }
  else
  {
    status = NdisIMQueueMiniportCallback (context->miniportAdapterHandle, gmm_flushCallback, context);

    if (status == NDIS_STATUS_FAILURE)
    {
      GMM_ALWAYS_PRINT (("NdisImQueueMiniportCallback failed\n"));
    }
  }
  
  exception:;
#endif
}


NDIS_STATUS gmm_TransferData (OUT PNDIS_PACKET packet,
		  OUT PUINT bytesTransferred,
		  IN NDIS_HANDLE miniportAdapterContext,
		  IN NDIS_HANDLE miniportReceiveContext,
		  IN UINT byteOffset,
		  IN UINT bytesToTransfer)
{
  insist (0);
  exception:return NDIS_STATUS_FAILURE;
}
