/*
  miniport specific code for the gm nt driver
*/

#include <ndis.h>
#include "gmm.h"

/* BAD: These declarations are only needed because Windows 2000 requires that
   device creation happen from within MiniportInitialize. */
#ifdef GM_OS_WIN2K
extern PDRIVER_OBJECT the_one_and_only_driver_object;
extern UNICODE_STRING the_one_and_only_registry_path;
NTSTATUS
gm_nt_find_myrinet_interfaces (IN PDRIVER_OBJECT DriverObject,
			       IN PUNICODE_STRING RegistryPath,
			       gm_u64_t NdisAdapterHandleAsU64);
#endif

int gmm_queueEmpty (gmm_Queue*queue)
{
  return queue->head == 0;
}

void gmm_queuePut (gmm_Queue*queue, gmm_Buffer*b)
{
  insist (queue && b);

  b->next = 0;

  if (!queue->tail)
  {
    insist (!queue->head);
    queue->tail = queue->head = b;  
  }
  else
  {
    insist (queue->head);
    queue->tail->next = b;
    queue->tail = b;
  }
  
  exception:;
}


gmm_Buffer*gmm_queueGet (gmm_Queue*queue)
{
  gmm_Buffer*b;

  insist (queue);
  
  if (!queue->head)
  {
    insist (!queue->tail);
    return 0;
  }
  
  b = queue->head;

  queue->head = (queue->head)->next;
  if (!queue->head)
  {
    insist (b == queue->tail);
    queue->tail = 0;
  }

  return b;
  
  exception: return 0;
}


int gmm_allocateQueue (gmm_Context*context, gmm_Queue*queue, int nBuffers, int length, int bufferPadding)
{
  int i;
  gmm_Buffer*p = 0;
  ULONG low, high;
  UINT physicalArraySize;
  NDIS_STATUS status;

  // NDIS_PHYSICAL_ADDRESS_UNIT physicalAddressUnit [100];
  
  insist (context);  
  insist (nBuffers > 0 && length > 0 && bufferPadding >= 0 && bufferPadding <= 4);

  queue->head = queue->tail = 0;
  
  /*allocate structures to hold buffer information, and the buffers themselves*/
  for (i = 0; i < nBuffers; i++)
  {
    p = gmm_calloc (sizeof (gmm_Buffer));

    if (!p)
    {
      GMM_ALWAYS_PRINT (("gmm_allocatePackets: gmm_calloc failed.\n"));
      goto abortWithChain;
    }

    p->port = context->port;
    p->mapRegister = i * GM_MAX_ETHERNET_GATHER_CNT;
    insist (p->mapRegister >= 0 && p->mapRegister < GMM_NUM_MAP_REGISTERS);
    
    p->originalLength = length;
    
    gmm_allocate_shared_memory (p->port, p->originalLength, &p->virtualAddress,
				&low, &high);
    
    /*
    NdisMAllocateSharedMemory (context->miniportAdapterHandle, p->originalLength, TRUE, &p->virtualAddress,
			       &p->physicalAddress);
    */
    if (!p->virtualAddress)
    {
      GMM_ALWAYS_PRINT (("gmm_allocateQueue: allocateSharedMemory failed (%d bytes).\n", p->originalLength));
      goto abortWithChain;
    }
    
    NdisSetPhysicalAddressLow (p->physicalAddress, low);
    NdisSetPhysicalAddressHigh (p->physicalAddress, high);
    
    NdisAllocateBuffer (&status,
			&p->ndisBuffer,
			context->bufferPoolHandle,
			p->virtualAddress + bufferPadding,
			p->originalLength - bufferPadding);

    if (status != NDIS_STATUS_SUCCESS)
    {
      GMM_ALWAYS_PRINT (("gmm_allocateQueue: NdisAllocateBuffer failed.\n"));
      goto abortWithChain;
    }
    
    NdisGetBufferPhysicalArraySize (p->ndisBuffer, &physicalArraySize);
    insist (physicalArraySize == 1);


    NdisAllocatePacket (&status,
			&p->myPacket,
			context->packetPoolHandle);
    
    if (status != NDIS_STATUS_SUCCESS)
    {
      GMM_ALWAYS_PRINT (("gmm_allocateQueue: NdisAllocatePacket failed.\n"));
      goto abortWithChain;
    }
    
    NdisChainBufferAtFront (p->myPacket, p->ndisBuffer);

    /* set up reverse path from myPacket to p so when gmm_ReturnPackets is
       called we can find the gmm_Buffer that owns the packet */
    *(gmm_Buffer**)&p->myPacket->MiniportReserved = p;

    gmm_queuePut (queue, p);       
  }

  return 1;
  
  abortWithChain:
  gmm_freeQueue (queue);
  
  return 0;
  
  exception: return 0;
}

/*Frees a chain of buffers allocated by gmm_allocateBufferQueue */
void gmm_freeQueue (gmm_Queue*queue)
{
  gmm_Buffer*p;

  insist (queue);
  
  while (p = gmm_queueGet (queue))
  {
    gmm_free_shared_memory (p->port, p->virtualAddress); 

    NdisFreePacket (p->myPacket);
    NdisFreeBuffer (p->ndisBuffer);
    gmm_free (p, sizeof (gmm_Buffer));
  }
  exception:;
}


/*Allocates and clears a block of non-shared memory.
  Should be freed with gmm_free.*/
void*gmm_calloc (UINT size)
{
  void*p;
  NDIS_PHYSICAL_ADDRESS minusOne = NDIS_PHYSICAL_ADDRESS_CONST (-1, -1);
  
  NdisAllocateMemory (&p, size, 0, minusOne);

  if (p)
    NdisZeroMemory (p, size);

  return p;
}


/*Frees memory allocated with gmm_calloc */
void gmm_free (void*p, UINT size)
{
  NdisFreeMemory (p, size, 0);
}


NDIS_STATUS gmm_Initialize (PNDIS_STATUS openErrorStatus,
			    PUINT selectedMediumIndex,
			    PNDIS_MEDIUM mediumArray,
			    UINT mediumArraySize,
			    NDIS_HANDLE miniportAdapterHandle,
			    NDIS_HANDLE configurationHandle)
{
  UINT i;
  gmm_Context*context;
  NDIS_STATUS status = NDIS_STATUS_SUCCESS;
  NDIS_HANDLE ConfigHandle;
  //UINT numMapRegisters;
  // void*p;
  // NDIS_RESOURCE_LIST*list;
  
  insist (selectedMediumIndex && mediumArray);
  insist (mediumArraySize > 0);
  insist (miniportAdapterHandle && configurationHandle);

  GMM_PRINT (("gmm_Initialize\n"));

  /*search for ethernet type*/
  for (i = 0; i < mediumArraySize; i++)
    if (mediumArray[i] == NdisMedium802_3)
      break;

  /*ethernet not found*/
  if (i == mediumArraySize)
    return NDIS_STATUS_UNSUPPORTED_MEDIA;

  /*select ethernet*/
  *selectedMediumIndex = i;
  
  /*allocate context*/
  if (!(context = gmm_calloc (sizeof (*context))))
  {
    GMM_ALWAYS_PRINT (("gmm_Initialize: couldn't allocate context structure.\n"));
    return NDIS_STATUS_RESOURCES;
    
  }
  
  context->miniportAdapterHandle = miniportAdapterHandle;
  NdisOpenConfiguration (&status, &ConfigHandle, configurationHandle);
  if (status != NDIS_STATUS_SUCCESS)
    {
      GMM_ALWAYS_PRINT (("gmm_Initialize: couldn't open configuration handle\n"));
      return status;
    }
  context->cacheFillSize = NdisGetCacheFillSize ();

  NdisAllocateSpinLock (&context->receiveQueueSpinLock);

  NdisMSetAttributesEx (miniportAdapterHandle,
			context,
			GMM_WATCHDOG_SECONDS,
			NDIS_ATTRIBUTE_BUS_MASTER |
			NDIS_ATTRIBUTE_DESERIALIZE,
			NdisInterfacePci
			);

#if GM_OS_WIN2K
  status = gm_nt_find_myrinet_interfaces (the_one_and_only_driver_object,
					  &the_one_and_only_registry_path,
					  GM_STATIC_CAST(gm_u64_t, miniportAdapterHandle));
  if (status != STATUS_SUCCESS)
    goto abortWithContext;
#endif

  NdisCloseConfiguration (ConfigHandle);

  GMM_ALWAYS_PRINT (("NdisGetCacheFillSize = %d\n",
		     (int) NdisGetCacheFillSize ()));

  status = NdisMAllocateMapRegisters (miniportAdapterHandle,
				      0,
				      TRUE,
				      GMM_NUM_MAP_REGISTERS,
				      GMM_MAX_BYTES_PER_DMA);

  if (status != NDIS_STATUS_SUCCESS)
  {
    GMM_ALWAYS_PRINT (("NdisMAllocateMapRegisters failed\n"));
    goto abortWithContext;
  }
  
  {
    
    PNDIS_MINIPORT_BLOCK  miniport = (PNDIS_MINIPORT_BLOCK) (miniportAdapterHandle);

    if (!miniport->MapRegisters)
    {
        GMM_PRINT (("NdisMAllocateMapRegisters succeeded, but still did not allocate map registers\n"));
	GMM_PRINT (("allocating them ourselves."));
	
	if (!(miniport->MapRegisters = gmm_calloc (sizeof (MAP_REGISTER_ENTRY) * GMM_NUM_MAP_REGISTERS)))
	{
	  GMM_ALWAYS_PRINT (("couldn't alloc map registers."));
	  goto abortWithContext;
	}
    }
  }

  GMM_PRINT (("allocating packet pool\n"));
  
  NdisAllocatePacketPool (&status,
			  &context->packetPoolHandle,
			  GMM_MAX_NDIS_PACKETS,
			  0);

  if (status != NDIS_STATUS_SUCCESS)
  {
    GMM_ALWAYS_PRINT (("NdisAllocatePacketPool failed\n"));
    goto abortWithMapRegsisters;
  }

  GMM_PRINT (("allocating buffer pool\n"));
  
  NdisAllocateBufferPool (&status, &context->bufferPoolHandle, GMM_MAX_NDIS_BUFFERS);

  if (status != NDIS_STATUS_SUCCESS)
    goto abortWithPacketPool;

  GMM_PRINT (("gmm_Initialize: calling gm_open\n"));
  
  if (GM_SUCCESS != gm_open (&context->port, 0, GM_ETHERNET_PORT_ID,
			     GMM_PORT_NAME, GM_API_VERSION_1_0))
  {
    GMM_ALWAYS_PRINT (("gmm_Initialize: gm_open failed.\n"));
    status = NDIS_STATUS_FAILURE;
    goto abortWithBufferPool;
  }

  GMM_PRINT (("gmm_Initialize: gm_open returned\n"));
  
  insist (GMM_NUM_SEND_BUFFERS <= GM_NUM_SEND_TOKENS);
  insist (GMM_NUM_RECEIVE_BUFFERS <= GM_NUM_ETHERNET_RECV_TOKENS);
  
  if (!gmm_allocateQueue (context, &context->sendFreeQueue, GMM_NUM_SEND_BUFFERS, GMM_MTU, 0))
  {
    GMM_ALWAYS_PRINT (("gmm_Initialize: gmm_allocateQueue failed.\n"));
    status = NDIS_STATUS_RESOURCES;
    goto abortWithOpen;
  }

  if (!gmm_allocateQueue (context, &context->receiveFreeQueue, GMM_NUM_RECEIVE_BUFFERS, GMM_MTU, GMM_HEADER_SIZE))
  {
    GMM_ALWAYS_PRINT (("gmm_Initialize: gmm_allocateQueue failed.\n"));
    status = NDIS_STATUS_RESOURCES;
    goto abortWithSendQueue;
  }

  GMM_PRINT (("gmm_Initialize: Buffers allocated.\n"));

  /* Set interrupt handler before posting recieves, which can cause
     interrupts, which we don't want to miss. --Glenn */
  
  /*register interrupt callback*/

  gm_ethernet_set_recv_intr_callback (context->port, gmm_receiveInterrupt, context);
  gm_ethernet_set_sent_intr_callback (context->port, gmm_sendInterrupt, context);
  
  /*feed gm receive buffers*/
  gmm_receiveFeedBuffers (context);

  return NDIS_STATUS_SUCCESS;
  
// abortWithReceiveQueue:
  gmm_freeQueue (&context->receiveFreeQueue);
  

  abortWithSendQueue:
  gmm_freeQueue (&context->sendFreeQueue);
  
  abortWithOpen:
  gm_close (context->port);
   
  abortWithBufferPool:
  NdisFreeBufferPool (context->bufferPoolHandle);
  
  abortWithPacketPool:
  NdisFreePacketPool (context->packetPoolHandle);

  abortWithMapRegsisters:
  NdisMFreeMapRegisters (miniportAdapterHandle);
  
  abortWithContext:
  gmm_free (context, sizeof (*context));

  GMM_PRINT (("failing from gmm_Initialize ()\n"));
  exception: return status;
}


VOID gmm_Halt (NDIS_HANDLE miniportAdapterContext)
{
  gmm_Context*context = (gmm_Context*) miniportAdapterContext;
  
  GMM_ALWAYS_PRINT (("gmm_Halt called.\n"));
  
  gmm_disable_interrupts(context->port);

  gmm_freeQueue (&context->sendFreeQueue);
  gmm_freeQueue (&context->sendPendQueue);
  gmm_freeQueue (&context->sendDoneQueue);

  gmm_freeQueue (&context->receiveFreeQueue);
  gmm_freeQueue (&context->receivePendQueue);
  gmm_freeQueue (&context->receiveDoneQueue);

  NdisFreeBufferPool (context->bufferPoolHandle);
  NdisFreePacketPool (context->packetPoolHandle);
  NdisMFreeMapRegisters (context->miniportAdapterHandle);
  NdisFreeSpinLock (&context->receiveQueueSpinLock);
  
  gm_ethernet_set_recv_intr_callback (context->port, 0, 0);
  gm_ethernet_set_sent_intr_callback (context->port, 0, 0);
  
  gmm_enable_interrupts(context->port);
  gm_close (context->port);
  gmm_free (context, sizeof (gmm_Context));
}


NDIS_STATUS gmm_Reset (PBOOLEAN addressingReset,
		       NDIS_HANDLE miniportAdapterContext)
{
  gmm_Context*context = (gmm_Context*) miniportAdapterContext;
  
  DbgPrint ("gmm_Reset\n");
  GMM_ALWAYS_PRINT (("gmm_Reset called.\n"));

  NdisAcquireSpinLock (&context->receiveQueueSpinLock);

  *addressingReset = TRUE;
  
  if (gmm_queueEmpty (&context->sendPendQueue))
    {
      NdisReleaseSpinLock (&context->receiveQueueSpinLock);
    return NDIS_STATUS_SUCCESS;
    }
  
  GMM_ALWAYS_PRINT (("gmm_Reset to be completed later\n"));
  
  context->flushingSendQueue = 1;
  
  NdisReleaseSpinLock (&context->receiveQueueSpinLock);
  return NDIS_STATUS_PENDING;
}

BOOLEAN gmm_CheckForHang (IN NDIS_HANDLE miniportAdapterContext)
{
  return FALSE;
}


NTSTATUS
gmm_DriverEntry
(PDRIVER_OBJECT DriverObject,
 PUNICODE_STRING RegistryPath)
{
  NTSTATUS status;
  NDIS_MINIPORT_CHARACTERISTICS portInfo;
  NDIS_HANDLE wrapperHandle;

  GMM_PRINT (("GMM Driver Entry called\n"));

  insist (GMM_MTU <= GM_ETHERNET_MTU);
  
  NdisMInitializeWrapper
    (&wrapperHandle,
     DriverObject,
     RegistryPath,
     NULL);

  GMM_PRINT (("NdisMInitializeWrapper returned\n"));

  NdisZeroMemory (&portInfo, sizeof(NDIS40_MINIPORT_CHARACTERISTICS));
    
  portInfo.MajorNdisVersion = GM_MINIPORT_MAJOR_VERSION;
  portInfo.MinorNdisVersion = GM_MINIPORT_MINOR_VERSION;
  
  portInfo.HaltHandler = gmm_Halt;
  portInfo.InitializeHandler = gmm_Initialize;
  portInfo.QueryInformationHandler = gmm_QueryInformation;
  portInfo.ResetHandler = gmm_Reset;
  portInfo.SendPacketsHandler = gmm_SendPackets;
  portInfo.SetInformationHandler = gmm_SetInformation;
  portInfo.ReturnPacketHandler = gmm_ReturnPacket;
  /*portInfo.TransferDataHandler = gmm_TransferData;*/
  //  portInfo.CheckForHangHandler = gmm_CheckForHang;


  
  status = NdisMRegisterMiniport (wrapperHandle,
				  &portInfo,
   				  sizeof(NDIS40_MINIPORT_CHARACTERISTICS));

  GMM_PRINT (("NdisMRegisterMiniport returned\n"));

  if (status != NDIS_STATUS_SUCCESS)
  {
    GMM_ALWAYS_PRINT (("ndisMRegisterMiniport failed with %x\n", status));
    goto abortWithWrapper;
  }

  GMM_ALWAYS_PRINT (("GM miniport driver loaded.\n"));
  return STATUS_SUCCESS;

  abortWithWrapper:
  NdisTerminateWrapper (wrapperHandle, NULL);

  exception:
  GMM_ALWAYS_PRINT (("GM miniport driver failed to load.\n"));
  return STATUS_UNSUCCESSFUL;
  
}
