/*
  gm_arch.c
  architecture specific code for hpux 11.0 GM driver
  finucane@myri.com (David Finucane)
*/


#include "gm_internal.h"
#include "gm_page_hash.h"
#include "gm_lanai.h"
#include "gm_instance.h"
#include "gm_impl.h"

#include <_types.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/malloc.h>
#include <sys/dma.h>
#include <mod_conf.h>
#include <sys/moddefs.h>
#include <sys/io.h>
#include <sys/pci.h>
#include <wsio/pci.h>
#include <limits.h>
#include <wsio/wsio.h>	/* this must be the last #include */

#ifndef PAGESIZE
#define PAGESIZE 4096
#endif

#include "finucane.h"

extern struct mod_operations gio_mod_ops;

int gm_hpux_load (void*driver_info_arg);
int gm_hpux_unload (void*driver_info_arg);
int gm_hpux_open (dev_t dev, int flag);
int gm_hpux_close (dev_t dev);
int gm_hpux_ioctl (dev_t dev, int cmd, caddr_t*arg, int flag);

static gm_arch_sync_t gm_hpux_debug_sync;

#define MOD_WSIO_PCI 4

static struct mod_type_data gm_hpux_driver_link =
{
  "gm",
  0
};

static struct modlink gm_hpux_mod_link [2] =
{
  { &gio_mod_ops,
    &gm_hpux_driver_link
  },
  {
    0, 0
  }
};

static struct mod_conf_data gm_hpux_mc =
{
  MCD_VERSION, 0
};

struct modwrapper gm_wrapper = 
{
  MODREV,
  gm_hpux_load,
  gm_hpux_unload,
  0,
  &gm_hpux_mc,
  gm_hpux_mod_link
};

static drv_info_t gm_hpux_driver_info =
{
  "gm",
  "ext_bus",
  DRV_MP_SAFE,
  -1,
  -1,
  0,
  0,
  0
};

static drv_ops_t gm_hpux_driver_ops =
{
  gm_hpux_open,
  gm_hpux_close,
  0,
  0,
  0,
  0,
  0,
  0,
  gm_hpux_ioctl,
  0,
  0,
  0,
  0,
  0
};

static wsio_drv_data_t gm_hpux_data =
{
  "myrinet_card",
  T_INTERFACE,
  DRV_CONVERGED,
  0,
  0
};

static wsio_drv_info_t gm_hpux_wsio_driver_info =
{
  &gm_hpux_driver_info,
  &gm_hpux_driver_ops,
  &gm_hpux_data
};

static struct gfsw gm_hpux_gfsw;

static int gm_hpux_dev_init ();
static int (*gm_hpux_saved_dev_init)(void);
static int (*gm_hpux_saved_attach)(uint32_t, struct isc_table_type*);

static gm_instance_state_t* gm_hpux_instances [GM_ARCH_MAX_INSTANCE];
static int gm_hpux_num_instances;

#define GM_HPUX_IS_STATIC_MINOR(n)((n) < gm_hpux_num_instances*2)
#define GM_HPUX_STATIC_MINOR_TO_UNIT(n)((n)/2)

static int gm_hpux_init (struct isc_table_type*isc)
{
  gm_instance_state_t*is; 
  unsigned char val;
  
  if (gm_hpux_num_instances > GM_ARCH_MAX_INSTANCE)
  {
    gm_hpux_printf (("gm_hpux: too many units\n"));
    return 0;
    
  }
  gm_hpux_printf (("1\n"));
  
  if (!(is = gm_arch_kernel_malloc (sizeof (gm_instance_state_t), 0)))
  {
    gm_hpux_printf (("gm_hpux_init: failed to alloc instance state\n"));
    return 0;
  }

  gm_hpux_printf (("2\n"));
  
  gm_hpux_instances [gm_hpux_num_instances] = is;
  is->arch.unit = gm_hpux_num_instances;
  gm_hpux_num_instances++;
  
  is->arch.isc = isc;
  
  gm_hpux_printf (("3\n"));
  
  if (isc->if_reg_ptr)
    is->arch.physical_base_address = isc->if_reg_ptr;
  else
    pci_read_cfg_uint32_isc (isc, 0x10, &is->arch.physical_base_address);
  
  gm_hpux_printf (("3\n"));

  is->arch.physical_base_address &= ~0xf;
  isc->if_reg_ptr = is->arch.physical_base_address;

  gm_hpux_printf (("4\n"));

  if (gm_instance_init (is, is->arch.unit, GM_MYRINET_BUS_PCI) != GM_SUCCESS)
  {
    gm_hpux_printf (("gm_hpux_init: gm_instance_init failed\n"));
    goto abort_with_malloc;
  }  

  gm_hpux_printf (("5\n"));

  if (gm_minor_alloc(is, &is->clone_minor) != GM_SUCCESS) 
  {
    gm_hpux_printf (("gm_minor_alloc failed.\n"));
    goto abort_with_is;
  }

  gm_hpux_printf (("6\n"));

  gm_minor_get_port_state(is->clone_minor)->instance = is;
  gm_minor_get_port_state(is->clone_minor)->privileged = 0;

  gm_hpux_printf (("7\n"));

  if (gm_minor_alloc(is, &is->privileged_clone_minor) != GM_SUCCESS) 
  {
    gm_hpux_printf (("gm_minor_alloc failed.\n"));
    goto abort_with_clone_minor;
  }
  gm_hpux_printf (("8\n"));

  gm_minor_get_port_state (is->privileged_clone_minor)->instance = is;
  gm_minor_get_port_state (is->privileged_clone_minor)->privileged = 1;

  gm_hpux_printf (("9\n"));

  gm_always_assert (GM_HPUX_IS_STATIC_MINOR (is->privileged_clone_minor));
  gm_always_assert (GM_HPUX_STATIC_MINOR_TO_UNIT(is->privileged_clone_minor) == is->arch.unit);
  gm_always_assert (GM_HPUX_IS_STATIC_MINOR (is->clone_minor));
  gm_always_assert (GM_HPUX_STATIC_MINOR_TO_UNIT(is->clone_minor) == is->arch.unit);
  
  return 1;
  abort_with_clone_minor:
  gm_minor_free (is->clone_minor);

  abort_with_is:
  gm_instance_finalize (is);

  abort_with_malloc:
  gm_arch_kernel_free (is);
  gm_hpux_instances [gm_hpux_num_instances] = 0;
  gm_hpux_num_instances--;

  return 0;
}

static int gm_hpux_detach (gm_instance_state_t*is)
{
  if (is->privileged_clone_minor)
  {
    gm_minor_free (is->privileged_clone_minor);
    is->privileged_clone_minor = 0;
  }
  if (is->clone_minor)
  {
    gm_minor_free (is->clone_minor);
    is->clone_minor = 0;
  }
  gm_instance_finalize (is);
  gm_always_assert (is->arch.unit >= 0 && is->arch.unit < gm_hpux_num_instances);
  gm_always_assert (is == gm_hpux_instances [is->arch.unit]);
  gm_arch_kernel_free (is);
  gm_hpux_instances [is->arch.unit] = 0;
}

static int gm_hpux_attach (uint32_t product_id, struct isc_table_type*isc)
{
  PCI_ID*id = (PCI_ID*) &product_id;
  
  printf ("gm_hpux_attach \n");
  if ((id->vendor_id == GM_PCI_VENDOR_MYRICOM && id->device_id == GM_PCI_DEVICE_MYRINET || 
       id->vendor_id == GM_PCI_VENDOR_MYRICOM2 && id->device_id == GM_PCI_DEVICE_MYRINET))
  {
    printf ("found a card\n");

    gm_hpux_printf (("testing debug print\n"));
    gm_hpux_printf (("debug print worked\n"));
    
    isc->gfsw = &gm_hpux_gfsw;
    isc->gfsw->init = gm_hpux_init;
    isc->gfsw->diag = 0;
    isc->if_id = (int)(id->device_id & 0x0000ffffU);
    isc_claim (isc, &gm_hpux_wsio_driver_info);
  }
  
  if (gm_hpux_saved_attach)
    return (*gm_hpux_saved_attach)(product_id, isc);
  else
    return 1;
}

static int gm_hpux_load (void*driver_info_arg)
{
  int i;
  
  gm_assert (driver_info_arg);
  gm_hpux_wsio_driver_info.drv_info = (struct drv_info*) driver_info_arg;

  printf ("hello. I was compiled on %s\n",__DATE__);

  if (!wsio_install_driver (&gm_hpux_wsio_driver_info))
  {
    printf ("wsio_install_driver failed\n");
    return ENXIO;
  }

  if (mod_wsio_attach_list_add (MOD_WSIO_PCI, &gm_hpux_attach))
  {
    printf ("gm_hpux_load: mod_wsio_attach_list_add failed\n");
  }
  
  gm_arch_sync_init (&gm_hpux_debug_sync, 0);
  gm_arch_page_len (&GM_PAGE_LEN);

  gm_hpux_dev_init ();
  gm_init ();
  return 0;
}

static int gm_hpux_unload (void*driver_info_arg)
{
  int i;
  
  gm_hpux_printf (("gm_hpux_unload.\n"));
  if (mod_wsio_attach_list_remove (MOD_WSIO_PCI, &gm_hpux_attach))
  {
    gm_hpux_printf (("gm_hpux_attach: mod_wsio_attach_list_remove failed\n"));
    /*return ENXIO;*/
  }
  
  gm_hpux_wsio_driver_info.drv_info = (struct drv_info*) driver_info_arg;
  wsio_uninstall_driver (&gm_hpux_wsio_driver_info);

  for (i = 0; i < gm_hpux_num_instances; i++)
  {
    gm_always_assert (gm_hpux_instances [i]);
    gm_hpux_detach (gm_hpux_instances [i]);
  }

  return 0;
}

static int gm_hpux_dev_init ()
{
  if (gm_hpux_saved_dev_init)
    return (*gm_hpux_saved_dev_init)();
  else
    return 1;
}


/*called if statically configured*/
int gm_install ()
{
  gm_hpux_saved_attach = pci_attach;
  pci_attach = gm_hpux_attach;
  
  gm_hpux_saved_dev_init = dev_init;
  dev_init = gm_hpux_dev_init;
  
  return wsio_install_driver (&gm_hpux_wsio_driver_info);  
}


static int gm_hpux_open (dev_t dev, int flag)
{
  int minor_id;
  minor_id  = minor (dev);

  gm_hpux_printf (("gm_hpux_open with minor number %d\n", minor_id));

  if (GM_HPUX_IS_STATIC_MINOR (minor_id))
    return gm_hpux_instances [GM_HPUX_STATIC_MINOR_TO_UNIT (minor_id)] ? 0 : ENODEV;
  else if (minor_id > GM_ARCH_MAX_MINOR)
    return ENODEV;

  return 0;
}

static int gm_hpux_close (dev_t dev)
{
  int minor_id;
  gm_instance_state_t*is;
  gm_port_state_t*ps;
  
  minor_id  = minor (dev);

  gm_hpux_printf (("gm_hpux_open with minor number %d\n", minor_id));

  if (GM_HPUX_IS_STATIC_MINOR (minor_id))
  {
    if (!(is = gm_hpux_instances [GM_HPUX_STATIC_MINOR_TO_UNIT (minor_id)]))
      return ENODEV;
  }
  else
  {
    if (minor_id > GM_ARCH_MAX_MINOR)
      return ENODEV;
    
    if (!(ps = gm_minor_get_port_state (minor_id)))
    {
      gm_hpux_printf (("gm_hpux_close: couldn't get port state for minor %d\n", minor_id));
      return ENODEV;
    }
    if (ps->opened)
      gm_port_state_close (ps);
    gm_minor_free (minor_id);
  }
  return 0;
}

static int gm_hpux_localize_status (gm_status_t status)
{
#define CASE(from,to) case from : return to
  switch (status) {
    CASE(GM_SUCCESS, 0);
    CASE(GM_INPUT_BUFFER_TOO_SMALL, EFAULT);
    CASE(GM_OUTPUT_BUFFER_TOO_SMALL, EFAULT);
    CASE(GM_TRY_AGAIN, EAGAIN);
    CASE(GM_BUSY, EBUSY);
    CASE(GM_MEMORY_FAULT, EFAULT);
    CASE(GM_INTERRUPTED, EINTR);
    CASE(GM_INVALID_PARAMETER, EINVAL);
    CASE(GM_OUT_OF_MEMORY, ENOMEM);
    CASE(GM_INVALID_COMMAND, EINVAL);
    CASE(GM_PERMISSION_DENIED, EPERM);
    default:
      CASE(GM_INTERNAL_ERROR, EPFNOSUPPORT);
      CASE(GM_UNATTACHED, -1);	/* ??? */
      CASE(GM_UNSUPPORTED_DEVICE, ENXIO);
  }
}

static int gm_hpux_ioctl (dev_t dev, int cmd, caddr_t*arg, int flag)
{
  int r, minor_id;
  gm_instance_state_t*is;
  gm_port_state_t*cps, *ps;
  void*p;

  minor_id = minor (dev);
  
  if (cmd == GM_GET_DEV)
  {
    if (!GM_HPUX_IS_STATIC_MINOR (minor_id))
    {
      gm_hpux_printf (("gm_hpux_ioctl: bad minor number %d for GM_GET_DEV\n", minor_id));
      return ENODEV;
    }
  
    if (!(is = gm_hpux_instances [GM_HPUX_STATIC_MINOR_TO_UNIT (minor_id)]))
      return ENODEV;

    if (!(cps = gm_minor_get_port_state (minor_id)))
    {
      gm_hpux_printf (("gm_hpux_ioctl: gm_minor_get_port_state failed\n"));
      return ENODEV;
    }
  
    if (gm_minor_alloc (cps->instance, &minor_id) != GM_SUCCESS)
    {
      gm_hpux_printf (("gm_hpux_ioctl: gm_minor_alloc failed\n"));
      return EINVAL;
    }
  
    if (!(ps = gm_minor_get_port_state (minor_id)))
    {
      gm_hpux_printf (("gm_hpux_ioctl: gm_minor_get_port_state failed for new device\n"));
      return ENODEV;
    }
    ps->instance = cps->instance;
    ps->privileged = cps->privileged;
  
    p = *((void**) arg);
    gm_arch_copyout (cps, &minor_id, (void*) p, sizeof (int));
    return 0;
  }
  else
  {
    if (!(ps = gm_minor_get_port_state (minor_id)))
      return ENXIO;
    
    gm_arch_mutex_enter (&ps->sync);
    p = *((void**) arg);
    r = gm_ioctl (ps, cmd, p, INT_MAX, p, INT_MAX, 0);
    gm_arch_mutex_exit (&ps->sync);
    return gm_hpux_localize_status (r);
  }
}

void *gm_arch_kernel_malloc (unsigned long len, int flags)
{
  void*p;
  
  gm_hpux_printf (("gm_arch_kernel_malloc called with length %d\n", len));
  p = kmalloc (len, M_IOSYS, M_NOWAIT);
  gm_hpux_printf (("result was %p\n", p));
  
  return p;
}

void gm_arch_kernel_free (void *ptr)
{
  kfree (ptr, M_IOSYS);
}

void gm_arch_sync_init (gm_arch_sync_t *s, gm_instance_state_t *is)
{
  b_initsema (&s->semaphore, 1, REAL_DRV_SEMA_ORDER, "gm_arch_sync");
  return;
}

void gm_arch_mutex_enter (gm_arch_sync_t *s)
{
  b_psema (&s->semaphore);
}

void gm_arch_mutex_exit (gm_arch_sync_t *s)
{
  b_vsema (&s->semaphore);
}

void gm_arch_sync_destroy (gm_arch_sync_t *s)
{
}

void gm_arch_sync_reset (gm_arch_sync_t *s)
{
  gm_arch_sync_init (s, 0);
  return;
}

void gm_arch_wake (gm_arch_sync_t * s)
{
  wakeup (s);
}

gm_arch_sleep_status_t gm_arch_signal_sleep (gm_arch_sync_t *s)
{
  get_sleep_lock (s);
  return sleep (s, PZERO | PCATCH) == 0 ? GM_SLEEP_WOKE : GM_SLEEP_INTERRUPTED;
}

static int gm_hpux_timeout (gm_arch_sync_t*s)
{
  wakeup (s);
}

static gm_arch_sleep_status_t gm_hpux_timed_sleep (gm_arch_sync_t *s, int milliseconds)
{
  get_sleep_lock (s); 
  timeout (gm_hpux_timeout, s, milliseconds * HZ / 1000);
  if (sleep (s, PZERO))
  {
    untimeout (gm_hpux_timeout, s);
    return GM_SLEEP_INTERRUPTED;
  }
  return GM_SLEEP_WOKE;
}

gm_arch_sleep_status_t gm_arch_timed_sleep (gm_arch_sync_t *s, int seconds)
{
  return gm_hpux_timed_sleep (s, seconds * 1000);
}

void gm_hpux_debug_sleep ()
{
  gm_hpux_timed_sleep (&gm_hpux_debug_sync, 100);
}

gm_status_t gm_arch_dma_region_alloc (gm_instance_state_t * is, gm_arch_dma_region_t * r,gm_size_t len, gm_u32_t flags,gm_register_page_function_t register_page_function, void *user_arg)
{
  int i, num_pages;

  if (wsio_allocate_shared_memory (is->arch.isc, len, &r->iova, &r->va, 0) != SHMEM_OK)
    return GM_FAILURE;

  r->iova_advance = r->iova;
  
  num_pages = (GM_PAGE_ROUNDUP (u32, len) / GM_PAGE_LEN);
  
  gm_always_assert (num_pages >= 1);
  
  if (register_page_function)
  {
    for (i = 0; i < num_pages; i++)
    {
      gm_dp_t p = (gm_dp_t) (r->iova + PAGESIZE * i);
      register_page_function (user_arg, p, i);
    }
  }
  return GM_SUCCESS;
}

void gm_arch_dma_region_free (gm_arch_dma_region_t * r)
{
  wsio_free_shared_memory (r->ai->isc, r->length, r->iova, r->va, 0);
}

gm_dp_t gm_arch_dma_region_dma_addr (gm_arch_dma_region_t * r)
{
  gm_dp_t p;
  p = (gm_dp_t) r->iova;
  return p;
}

gm_dp_t gm_arch_dma_region_dma_addr_advance (gm_arch_dma_region_t * r)
{
  gm_dp_t p;
  p = (gm_dp_t) r->iova_advance;
  r->iova_advance += PAGESIZE;
  return p;
}

void * gm_arch_dma_region_kernel_addr (gm_arch_dma_region_t * r)
{
  return r->va;
}

gm_status_t gm_arch_dma_region_sync (gm_arch_dma_region_t * r, int command)
{
  dma_sync_IO (KERNELSPACE, r->va, r->length, 0);
  return GM_SUCCESS;
}

gm_status_t gm_arch_map_io_space (gm_instance_state_t *is, gm_u32_t offset, gm_u32_t len, void **kaddr)
{
  *kaddr = map_mem_to_host (is->arch.isc, is->arch.physical_base_address + offset, len);
  return *kaddr ? GM_SUCCESS : GM_FAILURE;
}

void gm_arch_unmap_io_space (gm_instance_state_t *is, gm_u32_t offset, gm_u32_t len, void **kaddr)
{
  unmap_mem_from_host (is->arch.isc, *kaddr, len);
}


gm_status_t gm_arch_gethostname (char *ptr, int len)
{
  return GM_FAILURE;
}

void gm_arch_spin (gm_instance_state_t *is, gm_u32_t usecs)
{
  busywait (usecs);
}

gm_status_t gm_arch_get_page_len (unsigned long *page_len)
{
  return GM_FAILURE;
}

gm_status_t gm_arch_port_state_init (gm_port_state_t *ps)
{
  return GM_FAILURE;
}

gm_status_t gm_arch_port_state_open (gm_port_state_t *ps)
{
  return GM_FAILURE;
}

void gm_arch_port_state_close (gm_port_state_t *ps)
{
  return;
}

void gm_arch_port_state_fini (gm_port_state_t *ps)
{
  return;
}

gm_status_t gm_arch_read_pci_config_32 (gm_instance_state_t *is, gm_offset_t offset, gm_u32_t *value)
{
  pci_read_cfg_uint32_isc (is->arch.isc, offset, value);
  return GM_SUCCESS;
}

gm_status_t gm_arch_write_pci_config_32 (gm_instance_state_t *is, gm_offset_t offset, gm_u32_t value)
{
  pci_write_cfg_uint32_isc (is->arch.isc, offset, value);
  return GM_SUCCESS;
}

gm_status_t gm_arch_read_pci_config_16 (gm_instance_state_t *is, gm_offset_t offset, gm_u16_t *value)
{
  pci_read_cfg_uint16_isc (is->arch.isc, offset, value);
  return GM_SUCCESS;
}

gm_status_t gm_arch_write_pci_config_16 (gm_instance_state_t *is, gm_offset_t offset, gm_u16_t value)
{
  pci_write_cfg_uint16_isc (is->arch.isc, offset, value);
  return GM_SUCCESS;
}

gm_status_t gm_arch_read_pci_config_8 (gm_instance_state_t *is, gm_offset_t offset, gm_u8_t *value)
{
  pci_read_cfg_uint8_isc (is->arch.isc, offset, value);
  return GM_SUCCESS;
}

gm_status_t gm_arch_write_pci_config_8 (gm_instance_state_t *is, gm_offset_t offset, gm_u8_t value)
{
  pci_write_cfg_uint8_isc (is->arch.isc, offset, value);
  return GM_SUCCESS;
}

gm_port_state_t *gm_arch_port_for_id (gm_instance_state_t * is, unsigned port)
{
  return GM_FAILURE;
}

gm_status_t gm_arch_copyout (gm_port_state_t *ps, void *from, void *to, gm_size_t len)
{
  return copyout (from, to, len) == 0 ? GM_SUCCESS : GM_FAILURE;
}

gm_status_t gm_arch_copyin (gm_port_state_t *ps, void *from, void *to, gm_size_t len)
{
  return copyin (from, to, len) == 0 ? GM_SUCCESS : GM_FAILURE;
}

void gm_arch_abort (void)
{
  return;
}

void memcpy (char*a, char*b, int len)
{
  bcopy (b, a, len);
}

gm_status_t gm_arch_page_len (unsigned long*r)
{
  *r = PAGESIZE;
  return GM_SUCCESS;
}

/*don't care*/
gm_status_t gm_arch_mmap_contiguous_segment (gm_port_state_t *ps, void * kaddr, unsigned long blockSize, gm_up_t *vaddr)
{
  return GM_FAILURE;
}
void gm_arch_munmap_contiguous_segments (gm_port_state_t *ps)
{
  return;
}
gm_s32_t gm_arch_dma_region_status (gm_arch_dma_region_t *r)
{
  return 0; 
}
