/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#include "gm_debug.h"
#include "gm_internal.h"
#include "gm_pio.h"

#define GM_DEBUG_PIO 0

#ifdef __GNUC__
#define MY_START() GM_PRINT (GM_DEBUG_PIO, ("%s called\n", __FUNCTION__))
#define MY_FINISH() GM_PRINT (GM_DEBUG_PIO, ("%s returning\n", __FUNCTION__))
#else
#define MY_START()
#define MY_FINISH()
#endif

void
__gm_kton_bcopy (struct gm_instance_state *is,
		 void *_from, volatile gm_u8_n_t * _to, gm_size_t len)
{
  MY_START ();
  if (GM_NO_PARTWORD_PIO_WRITE)
    {
      gm_u8_n_t *from, *to;

      from = _from;
      to = (gm_u8_n_t *) _to;

      /* start partword access */
      
      __gm_pio_pause_lanai (is);

      /* write up to the first doubleword boundary. */
      
      while (len && !GM_ALIGNED (to, sizeof (gm_u64_t)))
	{
	  ___gm_pio_write_u8 (to++, *from++);
	  len--;
	}
      
      /* write down to the last doubleword boundary. */

      while (!GM_ALIGNED (len, sizeof (gm_u64_t)))
	{
	  --len;
	  ___gm_pio_write_u8 (&to[len], from[len]);
	}

      /* done with partword access */

      gm_unpause_lanai (is);

      /* Copy interior doublewords */
      
      while (len)
	{
	  gm_assert (len >= sizeof (gm_u64_t));
	  __gm_pio_write_u64 (is, (gm_u64_n_t *) to,
			      *(gm_u64_n_t *) from);
	  to += sizeof (gm_u64_t);
	  from += sizeof (gm_u64_t);
	  len -= sizeof (gm_u64_t);
	}
    }
  else
    {
      gm_bcopy (_from, (void *) _to, len);
    }
  MY_FINISH ();
}

void
__gm_ntok_bcopy (struct gm_instance_state *is,
		 volatile gm_u8_n_t * from, void *_to, gm_size_t len)
{
  MY_START ();
  if (GM_NO_PARTWORD_PIO_READ)
    {
      char *to = (char *) _to;

      while (len--)
	{
	  *to++ = gm_ntoh_u8 (__gm_pio_read_u8 (is, from++));
	}
    }
  else
    {
      gm_bcopy ((void *) from, _to, len);
    }
  MY_FINISH ();
}

void
__gm_nton_bcopy (struct gm_instance_state *is,
		 volatile gm_u8_n_t * from, volatile gm_u8_n_t * to,
		 gm_size_t len)
{
  /* NOTE: this function uses __gm_pio_write instead of _gm_pio_write
     because in all the places it is used, the lanai has already been
     paused ( or not yet started ) */

  MY_START ();

  if (GM_NO_PARTWORD_PIO_READ)
    {
      gm_u8_n_t tmp;

      while (len--)
	{
	  tmp = __gm_pio_read_u8 (is, from++);
	  ___gm_pio_write_u8 (to++, tmp);
	}
    }
  else
    {
      gm_bcopy ((void *) from, (void *) to, len);
    }
  MY_FINISH ();
}


gm_u8_n_t *
__gm_kton_strncpy (struct gm_instance_state *is, volatile gm_u8_n_t * _to,
		   char *from, gm_size_t len)
{
  MY_START ();
  if (GM_NO_PARTWORD_PIO_WRITE)
    {
      gm_u8_n_t *to;

      to = (gm_u8_n_t *) _to;
      __gm_pio_pause_lanai (is);
      while (len--)
	{
	  ___gm_pio_write_u8 (to++, gm_hton_u8 (*from));
	  if (*from)
	    from++;
	}
      gm_unpause_lanai (is);
    }
  else
    {
      gm_strncpy ((char *) _to, from, (int)len);
    }
  MY_FINISH ();
  return (gm_u8_n_t *) _to;
}

char *
__gm_ntok_strncpy (struct gm_instance_state *is, char *_to,
		   volatile gm_u8_n_t * from, gm_size_t len)
{
  MY_START ();
  if (GM_NO_PARTWORD_PIO_READ)
    {
      char *to;

      to = _to;
      __gm_pio_pause_lanai (is);
      while (len--)
	{
	  *to = gm_ntoh_u8 (__gm_pio_read_u8 (is, from));
	  if (*to++)
	    from++;
	}
      gm_unpause_lanai (is);
    }
  else
    {
      gm_strncpy (_to, (char *) from, (int)len);
    }
  MY_FINISH ();
  return _to;
}

void
__gm_kton_bzero (struct gm_instance_state *is, volatile gm_u8_n_t *_ptr,
		 gm_size_t len)
{
  MY_START ();
  if (GM_NO_PARTWORD_PIO_WRITE)
    {
      gm_u8_n_t *to;

      to = (gm_u8_n_t *) _ptr;
      
      /* pause the lanai for partword PIO */
	  
      __gm_pio_pause_lanai (is);
	  
      /* Write up to the next word boundary */
	  
      while (len && !GM_ALIGNED (to, sizeof (gm_u64_t)))
	{
	  ___gm_pio_write_u8 (to++, gm_hton_u8 (0));
	  len--;
	}
	  
      /* Write down to the last word boundary */
	  
      while (!GM_ALIGNED (len, sizeof (gm_u64_t)))
	{
	  --len;
	  ___gm_pio_write_u8 (&to[len], gm_hton_u8 (0));
	}

      /* Done with partword PIO */

      gm_unpause_lanai (is);
	  
      /* Clear the interior of the packet one word at a time. */

      if (len)
	{
	  gm_assert (GM_ALIGNED (to, sizeof (gm_u64_t)));
	  gm_assert (GM_ALIGNED (len, sizeof (gm_u64_t)));
	  gm_safer_bzero ((void *) to, (int)len);
	}
    }
  else
    {
      gm_bzero ((void *) _ptr, len);
    }
  MY_FINISH ();
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
