/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file includes support for dynamic minor instance number management.
   The functions auto-initialize and auto-finalize. */

#include "gm_call_trace.h"
#include "gm_impl.h"		/* Needed for gm_port_state_t  - Max */
#include "gm_internal.h"
#include "gm_arch_types.h"	/* Needed for gm_arch_minor_t  - Max */
#include "gm_debug_port_state.h"

/************************************************************************
 * Instance minor number management
 ************************************************************************/

static struct gm_hash *gm_minor_hash;
static struct gm_lookaside *gm_minor_lookaside;
static gm_arch_minor_t gm_minor_cnt;
static gm_arch_minor_t gm_minor_limit;
static int gm_minor_initialized;

/************************************************************************
 * Internal functions
 ************************************************************************/

/************
 * Initialization
 ************/

static long
gm_minor_hash_compare (void *key1, void *key2)
{
  return gm_memcmp (key1, key2, sizeof (gm_arch_minor_t));
}

static unsigned long
gm_minor_hash_hash (void *key)
{
  return gm_crc (key, sizeof (gm_arch_minor_t));
}

static gm_status_t
gm_init_minor_support (void)
{
  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_minor_cnt = %d, gm_minor_limit = %d\n",
                                  gm_minor_cnt, gm_minor_limit));

  if (gm_minor_initialized != 0)
    return GM_SUCCESS;

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Initializing GM minor support.\n"));

  gm_minor_lookaside = gm_create_lookaside (sizeof (gm_port_state_t), 3);
  if (!gm_minor_lookaside)
    {
      GM_NOTE (("Could not create minor lookaside list.\n"));
      goto abort_with_nothing;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 2, ("GM minor: create hash table .\n"));
  gm_minor_hash = gm_create_hash (gm_minor_hash_compare, gm_minor_hash_hash,
				  sizeof (gm_arch_minor_t), 0, 3, 0);
  if (!gm_minor_hash)
    {
      GM_NOTE (("Could not create minor hash table.\n"));
      goto abort_with_lookaside;
    }
  gm_minor_initialized = 1;
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("GM minor support successful.\n"));
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_minor_cnt = %d, gm_minor_limit = %d\n",
                                  gm_minor_cnt, gm_minor_limit));

  return GM_SUCCESS;

abort_with_lookaside:
  gm_destroy_lookaside (gm_minor_lookaside);
  gm_minor_lookaside = 0;
abort_with_nothing:
  GM_PANIC (("GM minor support successful FAILED.\n"));
  GM_RETURN_STATUS (GM_FAILURE);
}

/************
 * Finalization
 ************/

static void
gm_finalize_minor_support (void)
{
  GM_CALLED ();

  /* Reduce gm_minor_limit if needed */

  while (gm_minor_limit && gm_minor_get_port_state (gm_minor_limit - 1) == 0)
    gm_minor_limit--;

  /* If there is no allocated minor number, really finalize the
     gm_minor support */

  if (!gm_minor_cnt && gm_minor_initialized)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1, ("Finalizing GM minor support.\n"));
      gm_assert (gm_minor_cnt == 0);
      gm_destroy_hash (gm_minor_hash);
      gm_minor_hash = 0;
      gm_destroy_lookaside (gm_minor_lookaside);
      gm_minor_lookaside = 0;
      gm_minor_initialized = 0;
    }

  GM_RETURN_NOTHING ();
}

/************************************************************************
 * Entry points
 ************************************************************************/

/************
 * get port state for minor number
 ************/

gm_port_state_t *
gm_minor_get_port_state (gm_arch_minor_t minor)
{
  gm_port_state_t *ret;

  /* initialize this package, if needed */

  ret = (gm_port_state_t *) gm_hash_find (gm_minor_hash, &minor);
  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("got port state %p for minor %d.\n", ret, minor));
  return ret;
}

/************
 * Reserve a minor number.
 ************/

/* This function reserves a specific unused minor number.  If the
   caller wants a minor number to be picked automatically, gm_minor_alloc()
   should be used instead */

gm_status_t gm_minor_reserve (gm_instance_state_t * is, gm_arch_minor_t minor)
{
  gm_port_state_t *ps;

  GM_CALLED ();
  gm_assert (is);

  GM_PRINT (GM_PRINT_LEVEL >= 1,
            ("gm_minor_reserve called for unit %d, minor %d\n",
	     is->id, minor));

  /* initialize this package, if needed */

  if (gm_init_minor_support () != GM_SUCCESS)
    {
      GM_NOTE (("Failed to initialize minor number support.\n"));
      goto abort_with_nothing;
    }

#if GM_ARCH_MAX_MINOR
  if (minor >= GM_ARCH_MAX_MINOR)
    {
      GM_NOTE (("Tried to reserve overlarge minor number.\n"));
      goto abort_with_init;
    }
#endif

  /* Verify the minor number is not already reserved. */

  if (gm_minor_get_port_state (minor))
    {
      GM_NOTE (
	       ("Tried to reserve minor number %d it is already reserved.\n",
		minor));
      goto abort_with_init;
    }

  /* Allocate and initialize a new port state.  Must be initialized
     before inserting. */

  ps = (gm_port_state_t *) gm_lookaside_zalloc (gm_minor_lookaside);
  if (!ps)
    {
      GM_NOTE (
	       ("Failed to allocate a new port state from lookaside list.\n"));
      goto abort_with_init;
    }
  gm_arch_sync_init (&ps->sync, is);

  ps->instance = is;
  ps->minor = minor;

  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("gm_minor_reserve: setting ps->instance = %p  ps->minor = %d\n",
	     is, minor));

  /* Do architecture-specific initialization. */

  if (gm_arch_port_state_init (ps) != GM_SUCCESS)
    {
      GM_NOTE (("Failed to initialize architecture-specific port state.\n"));
      goto abort_with_port_state;
    }

  /* Insert the initialized port state in the hash table */

  GM_PRINT (GM_DEBUG_PORT_STATE,
	    ("Inserting port state %p for minor %d.\n", ps, minor));
  if (gm_hash_insert (gm_minor_hash, &minor, ps) != GM_SUCCESS)
    {
      GM_NOTE (("Failed to insert port state in minor hash table.\n"));
      goto abort_with_arch_port_state_init;
    }

  /* Update minor numbers */

  if (minor >= gm_minor_limit)
    gm_minor_limit = minor + 1;
  gm_minor_cnt++;
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_minor_cnt = %d, gm_minor_limit = %d\n",
                                  gm_minor_cnt, gm_minor_limit));

  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_arch_port_state_init:
  gm_arch_port_state_fini (ps);
abort_with_port_state:
  gm_lookaside_free (ps);
abort_with_init:
  gm_finalize_minor_support ();
abort_with_nothing:
  GM_RETURN_STATUS (GM_FAILURE);
}

/************
 * Allocate a minor number
 ************/

/* Return an unused minor number in O(1) time, on average.  Also,
   attempt to return small minor node numbers.  If GM_ARCH_MAX_MINOR
   is defined, never return a larger minor number.  If
   GM_ARCH_MAX_MINOR is not defined, presumably the driver has some
   other way of detecting minor number overflow. */

gm_status_t
gm_minor_alloc (gm_instance_state_t * is, gm_arch_minor_t * minor_p)
{
  gm_arch_minor_t minor;
  gm_status_t status;

  GM_CALLED ();
  gm_assert (is);
  gm_assert (minor_p);


  /* initialize this package, if needed */

  status = gm_init_minor_support ();
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not initialized GM minor number support.\n"));
      goto abort_with_nothing;
    }

  /* pick an unused minor number to reserve, attempting to pick a
     small one. */

  if (gm_minor_cnt + (gm_minor_cnt >> 4) + 1 >= gm_minor_limit)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1, ("Using a new minor number.\n"));
      /* minor usage is dense, so expand range of minor numbers */

      minor = gm_minor_limit++;
    }
  else
    {
      GM_NOTE (("Using an unused minor number.\n"));
      /* minor usage is sparse, so pick random unused minor number in
         the range */

      do
      {
        minor = gm_rand_mod (gm_minor_limit);
	GM_PRINT (GM_PRINT_LEVEL >= 1,
                  ("gm_rand_mod(%d) returned %d\n", gm_minor_limit, minor));
      }
      while (gm_minor_get_port_state (minor));
    }

  /* Reserve the picked minor number */

  status = gm_minor_reserve (is, minor);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not reserve higher minor number.\n"));
      goto abort_with_init;
    }

  /* report success */

  *minor_p = minor;
  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_init:
  gm_finalize_minor_support ();
abort_with_nothing:
  GM_RETURN_STATUS (status);
}

/************
 * free a previously allocated minor number.
 ************/

void
gm_minor_free (gm_arch_minor_t minor)
{
  gm_port_state_t *ps;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Freeing minor number %d.\n", minor));

  /* free the minor number by removing the associated port state from
     the hash table. */

  gm_assert (gm_minor_hash);
  ps = (gm_port_state_t *) gm_hash_remove (gm_minor_hash, &minor);
  gm_assert (ps);
  gm_assert (ps->minor == minor);
  gm_port_state_close (ps);
  gm_assert (!ps->opened);
  gm_arch_port_state_fini (ps);
  ps->instance = 0;
  gm_arch_sync_destroy (&ps->sync);
  gm_lookaside_free (ps);

  /* reduce the minor number range if the largest minor number was
     just freed. */

  gm_minor_cnt--;
  gm_finalize_minor_support ();

  GM_RETURN_NOTHING ();
}

/************
 * Locks
 ************/

void
gm_minor_mutex_enter (gm_arch_minor_t minor)
{
  gm_arch_mutex_enter (&gm_minor_get_port_state (minor)->sync);
}

void
gm_minor_mutex_exit (gm_arch_minor_t minor)
{
  gm_arch_mutex_exit (&gm_minor_get_port_state (minor)->sync);
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
