/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* This file implements platform-dependent support for LANai
   interfaces.  This file relies heavily on the platform-dependent
   functions declared in gm_impl.h, and implemented in the
   architecture's gm_arch.c file. */


#if !GM_KERNEL
#error GM_KERNEL not defined in gm_instance.c
#endif

#include "gm_call_trace.h"
#include "gm_cpp.h"
#include "gm_debug.h"
#include "gm_debug_host_dma.h"
#include "gm_debug_instance_state.h"
#include "gm_debug_lanai_dma.h"
#include "gm_enable_force_64bit_mode.h"
#include "gm_enable_prototype_nic_support.h"
#include "gm_instance.h"
#include "gm_internal.h"
#include "gm_pio.h"
#include "zlib.h"


/****************************************************************
 * Forward declarations
 ****************************************************************/
#if GM_SUPPORT_PCI_REV_3
static gm_status_t gm_update_clockval_rev3 (gm_instance_state_t * is);
#endif /* GM_SUPPORT_PCI_REF_3 */
	  
/************************************************************************
 * Package initialization/finalization
 ************************************************************************/

/* convert CSPI_MAP26xx to use 0/1 instead of def/undef */
#ifdef CSPI_MAP26xx
#  undef CSPI_MAP26xx
#  define CSPI_MAP26xx 1
#else
#  define CSPI_MAP26xx 0
#endif

#if CSPI_MAP26xx
#if GM_OS_LINUX
#include <asm/cspi.h>
#elif GM_OS_VXWORKS
#include "boardCsr.h" /* SAB */
#else
#error GM does not support this OS for CSPI 26xx/27xx boards
#endif /* GM_OS... */
#endif /* CSPI_MAP26xx */

static unsigned long gm_init_instance_cnt;
static struct gm_hash *gm_instance_hash;

static gm_status_t
gm_init_instance_support (void)
{
  GM_CALLED ();
#if !GM_ETHERBOOT
  if (!gm_init_instance_cnt)
    {
      gm_instance_hash = gm_create_hash (gm_hash_compare_ints,
					 gm_hash_hash_int,
					 sizeof (unsigned int), 0, 1, 0);
      if (!gm_instance_hash)
	return GM_FAILURE;
    }
#endif
  gm_init_instance_cnt++;
  GM_RETURN_STATUS (GM_SUCCESS);
}

static void
gm_finalize_instance_support (void)
{
  GM_CALLED ();
  gm_assert (gm_init_instance_cnt > 0);
  if (!--gm_init_instance_cnt)
    {
#if !GM_ETHERBOOT
      gm_destroy_hash (gm_instance_hash);
#endif
    }
  GM_RETURN_NOTHING ();
}

/**********************************************************************
 * Interface register abstraction
 **********************************************************************/

/* default do-nothing handler */

static void
gm_oo_set_abort (gm_instance_state_t * is, gm_u32_t val)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);
  GM_PARAMETER_MAY_BE_UNUSED (val);

  GM_WARN (
	   ("Internal driver error: Uninitialized dynamic handler called.\n"));
  gm_always_assert (0);
}

static gm_u32_t
gm_oo_get_abort (gm_instance_state_t * is)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);

  GM_WARN (
	   ("Internal driver error: Uninitialized dynamic handler called.\n"));
  gm_always_assert (0);
  return (0);
}

/****************
 * Version checking functions
 ****************/

/* These functions test for recognized version numbers.  The should
   only accept version numbers that are known to work. */

static int
gm_is_l4 (gm_instance_state_t * is)
{
  switch (is->lanai.eeprom.lanai_cpu_version)
    {
    case 0x400:
      return GM_ENABLE_PROTOTYPE_NIC_SUPPORT;

    case 0x401:
    case 0x402:
    case 0x403:
      return 1;

    default:
      break;			/* for some really picky compilers */
    }
  return 0;
}

static int
gm_is_l5 (gm_instance_state_t * is)
{
  switch (is->lanai.eeprom.lanai_cpu_version)
    {
    case 0x500:
    case 0x501:
    case 0x502:
    case 0x503:
    case 0x504:
    case 0x505:
    case 0x506:
      return 1;

    default:
      break;			/* for some really picky compilers */
    }
  return 0;
}

static int
gm_is_l6 (gm_instance_state_t * is)
{
  switch (is->lanai.eeprom.lanai_cpu_version)
    {
    case 0x601:
    case 0x602:
      return 1;

    default:
      break;			/* for some really picky compilers */
    }
  return 0;
}

static int
gm_is_l7 (gm_instance_state_t * is)
{
  switch (is->lanai.eeprom.lanai_cpu_version)
    {
    case 0x700:		/* This was missing ???  - Max 2000/06/07 */
    case 0x701:
    case 0x702:
    case 0x703:
    case 0x704:
    case 0x705:
      return 1;

    default:
      break;			/* for some really picky compilers */
    }
  return 0;
}

static int
gm_is_l8 (gm_instance_state_t * is)
{
  switch (is->lanai.eeprom.lanai_cpu_version)
    {
    case 0x800:
    case 0x801:
      return GM_ENABLE_PROTOTYPE_NIC_SUPPORT;

    case 0x802:
      return 1;

    default:
      break;			/* for some really picky compilers */
    }
  return 0;
}

static int
gm_is_l9 (gm_instance_state_t * is)
{
  switch (is->lanai.eeprom.lanai_cpu_version)
    {
    case 0x900:
    case 0x901:
    case 0x902:
    case 0x903:
    case 0x904:
    case 0x905:
      return 1;

    default:
      break;			/* for some really picky compilers */
    }
  return 0;
}

/********************
 * LANai-version-dependent functions
 ********************/

/****
 * LANai4
 ****/

static void
gm_set_EIMR_l4 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l4.EIMR, val);
}

static void
gm_set_ISR_l4 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l4.ISR, val);
}

static gm_u32_t
gm_get_EIMR_l4 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l4.EIMR);
}

static gm_u32_t
gm_get_ISR_l4 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l4.ISR);
}

/****
 * LANai5
 ****/

static void
gm_set_EIMR_l5 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l5.EIMR, val);
}

static void
gm_set_ISR_l5 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l5.ISR, val);
}

static gm_u32_t
gm_get_EIMR_l5 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l5.EIMR);
}

static gm_u32_t
gm_get_ISR_l5 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l5.ISR);
}

/****
 * LANai6
 ****/

static void
gm_set_EIMR_l6 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l6.EIMR, val);
}

static void
gm_set_ISR_l6 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l6.ISR, val);
}

static gm_u32_t
gm_get_EIMR_l6 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l6.EIMR);
}

static gm_u32_t
gm_get_ISR_l6 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l6.ISR);
}

/****
 * LANai7
 ****/

static void
gm_set_EIMR_l7 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l7.EIMR, val);
}

static void
gm_set_ISR_l7 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l7.ISR, val);
}

static gm_u32_t
gm_get_EIMR_l7 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l7.EIMR);
}

static gm_u32_t
gm_get_ISR_l7 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l7.ISR);
}

/****
 * LANai8
 ****/

static void
gm_set_EIMR_l8 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l8.EIMR, val);
}

static void
gm_set_ISR_l8 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l8.ISR, val);
}

static gm_u32_t
gm_get_EIMR_l8 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l8.EIMR);
}

static gm_u32_t
gm_get_ISR_l8 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l8.ISR);
}

/****
 * LANai9
 ****/

static void
gm_set_EIMR_l9 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l9.EIMR, val);
}

static void
gm_set_ISR_l9 (gm_instance_state_t * is, gm_u32_t val)
{
  gm_write_lanai_special_reg_u32 (is, l9.ISR, val);
}

static gm_u32_t
gm_get_EIMR_l9 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l9.EIMR);
}

static gm_u32_t
gm_get_ISR_l9 (gm_instance_state_t * is)
{
  return gm_read_lanai_special_reg_u32 (is, l9.ISR);
}

/****************
 * 
 ****************/

void
gm_set_EIMR (gm_instance_state_t * is, gm_u32_t val)
{
  is->_set_EIMR (is, val);
  /* is->pausible = val;  moved this functionality to gm_enable/disable interrupts */
}

#if GM_SUPPORT_PCI
/**********************************************************************
 * PCI register access.
 **********************************************************************/

static gm_status_t
gm_read_pci_cache_line_size (gm_instance_state_t * is, gm_u8_t * val)
{
  return gm_arch_read_pci_config_8 (is,
				    GM_OFFSETOF (gm_pci_config_t,
						 Cache_Line_Size), val);
}

static gm_status_t
gm_read_pci_config_command (gm_instance_state_t * is, gm_u16_t * command)
{
  return gm_arch_read_pci_config_16 (is,
				     GM_OFFSETOF (gm_pci_config_t,
						  Command), command);
}

static gm_status_t
gm_write_pci_config_command (gm_instance_state_t * is, gm_u16_t value)
{
  return gm_arch_write_pci_config_16 (is,
				      GM_OFFSETOF (gm_pci_config_t, Command),
				      value);
}

static gm_status_t
gm_read_pci_config_revision_id (gm_instance_state_t * is, gm_u8_t * value)
{
  return gm_arch_read_pci_config_8 (is,
				    GM_OFFSETOF (gm_pci_config_t,
						 Revision_ID), value);
}
#endif /* GM_SUPPORT_PCI */
/**********************************************************************
 * LANai register setting functions  (not performance critical)
 **********************************************************************/

#define GM_HACKED_STS 0

static gm_status_t
gm_setup_DMA_engine (gm_instance_state_t * is)
{
  /* Verify that this function was not called out-of-order */

  if (!is->lanai.globals)
    return GM_FAILURE;

  if (CSPI_MAP26xx)
    {
      /* not sure we need to do anything */
      /* Set burst size to 32 bytes (the cache line size) and PCI
         compatibility. */
      gm_write_lanai_special_reg_u32 (is, l5.BURST, 0x5);
      return GM_SUCCESS;
    }

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      if (gm_is_l4 (is))
	{
	  if (GM_HACKED_STS)
	    {
	      gm_write_lanai_special_reg_u32 (is, l4.DMA_STS, 0);
	      GM_PRINT (GM_PRINT_LEVEL >= 4,
			("gm_setup_DMA_engine: GM_HACKED_STS 'sts'"
			 " set to 0 would have been 0x%x.\n",
			 gm_arch_dma_region_status (&is->page_hash.
						    segment[0].dma_region)));
	    }
	  else
	    {
	      gm_write_lanai_special_reg_u32
		(is, l4.DMA_STS,
		 gm_arch_dma_region_status (&is->page_hash.
					    segment[0].dma_region));
	      GM_PRINT (GM_PRINT_LEVEL >= 4,
			("gm_setup_DMA_engine: DMA 'sts' set to 0x%x.\n",
			 gm_arch_dma_region_status (&is->page_hash.
						    segment[0].dma_region)));
	    }
	  return GM_SUCCESS;
	}
    }
  
#if GM_SUPPORT_PCI
  
  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  if (gm_is_l4 (is))
	    {
	      /* DMA_STS is always 0xf for L4 PCI */
	      gm_write_lanai_special_reg_u32 (is, l4.DMA_STS, 0xf);
	      return GM_SUCCESS;
	    }
	}
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  if (gm_is_l5 (is))
	    {
	      gm_lp_t *descriptors;
	      gm_lp_t pointer;

	      /* This sets the burst size to 32 bytes (the
	         cache line size) and PCI compatibility. */
	      gm_write_lanai_special_reg_u32 (is, l5.BURST, 0x5);

	      /* Tell the LANai where to find the DMA descriptor for
	         "channel 0" */

	      descriptors = (gm_lp_t *) ((gm_u8_t *) is->board_base
					 + 0x00800100);
	      pointer = (gm_lp_t) ((gm_u8_t *) is->lanai.globals
				   + GM_OFFSETOF (gm_lanai_globals_t,
						  dma_descriptor) -
				   (gm_u8_t *) is->lanai.sram);
	      descriptors[0] = gm_htopci_u32 (pointer);
	      GM_PRINT (GM_PRINT_LEVEL >= 4,
			("DMA descriptor at LANai address 0x%x.\n", pointer));
	      return GM_SUCCESS;
	    }
	  if (gm_is_l6 (is))
	    {
	      gm_lp_t *descriptors;
	      gm_lp_t pointer;

	      /* Tell the LANai where to find the DMA descriptor for
	         "channel 0" */

	      descriptors = (gm_lp_t *) ((gm_u8_t *) is->board_base
					 + 0x00800100);
	      pointer = (gm_lp_t) ((gm_u8_t *) is->lanai.globals
				   + GM_OFFSETOF (gm_lanai_globals_t,
						  dma_descriptor) -
				   (gm_u8_t *) is->lanai.sram);
	      descriptors[0] = gm_htopci_u32 (pointer);
	      GM_PRINT (GM_PRINT_LEVEL >= 4,
			("DMA descriptor at LANai address 0x%x.\n", pointer));
	      return GM_SUCCESS;
	    }
	  if (gm_is_l7 (is) || gm_is_l8 (is) || gm_is_l9 (is))
	    {
	      gm_lp_t *descriptors;
	      gm_lp_t pointer;

	      /* Tell the LANai where to find the DMA descriptor for
	         "channel 0" */

	      /* The "magic number" in this assignment is the byte
	         offset of the bank of pointers for DMA and Doorbell
	         as described in the M2M-PCI64X specification: */

	      descriptors = (gm_lp_t *) ((gm_u8_t *) is->board_base
					 + 0x00800100);
	      pointer = (gm_lp_t) ((gm_u8_t *) is->lanai.globals
				   + GM_OFFSETOF (gm_lanai_globals_t,
						  dma_descriptor) -
				   (gm_u8_t *) is->lanai.sram);
	      descriptors[0] = gm_htopci_u32 (pointer);
	      GM_PRINT (GM_PRINT_LEVEL >= 4,
			("DMA descriptor at LANai address 0x%x.\n", pointer));
	      return GM_SUCCESS;
	    }
	}
    }
#endif /* GM_SUPPORT_PCI */
  return GM_FAILURE;
}

static gm_status_t
gm_set_clock_val (gm_instance_state_t * is)
{

#if GM_SUPPORT_PCI
  if (GM_SUPPORT_PCI_REV_3 && is->ifc.pci.config.Revision_ID == 3)
    {
      unsigned int clockval = 0;

      /* use a 1x clockval initially */
      clockval = is->lanai.eeprom.lanai_clockval;
      clockval = clockval & (unsigned int) 0xFFFFFF8F;
      GM_INFO (("gm_set_clock_val: PCI rev3 board - using a 1x clock"
		" clockval = 0x%x\n", clockval));

      gm_write_lanai_special_reg_u32 (is, l7.CLOCK, clockval);
      goto done;
    }
#endif /* GM_SUPPORT_PCI */
  if (gm_is_l4 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l4.clock_val,
				      is->lanai.eeprom.lanai_clockval);
      goto done;
    }

  if (gm_is_l5 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l5.clock_val,
				      is->lanai.eeprom.lanai_clockval);
      goto done;
    }

  if (gm_is_l6 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l6.CLOCK,
				      is->lanai.eeprom.lanai_clockval);
      goto done;
    }

  if (gm_is_l7 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l7.CLOCK,
				      is->lanai.eeprom.lanai_clockval);
      goto done;
    }

  if (gm_is_l8 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l8.CLOCK,
				      is->lanai.eeprom.lanai_clockval);
      goto done;
    }

  if (gm_is_l9 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l9.CLOCK,
				      is->lanai.eeprom.lanai_clockval);
      goto done;
    }

  return GM_FAILURE;

done:
  gm_arch_spin (is, 100000);	/* seems to need long delay. */
  return GM_SUCCESS;
}


/* Stop the runaway phase lock loop circuitry in revision 2 boards.  This
   should be done after taking the board out of reset, but before reading
   the EEPROM. */

static gm_status_t
gm_stop_runaway_PLL (gm_instance_state_t * is)
{
#if GM_SUPPORT_PCI
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Stopping runaway PLL, if needed.\n"));

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  /* HACK: Must assume that the clock_val register is at the same
	     location as for a LANai 5. */
	  gm_write_lanai_special_reg_u32 (is, l5.clock_val, 0x00000080);
	  /* Pause for at least 10ms to allow LANai PLL to synchronize */
	  gm_arch_spin (is, 20000);
	  return GM_SUCCESS;
	}
    }

  GM_WARN (("Could not stop runaway PLL.\n"));
  return GM_FAILURE;
#else /* !GM_SUPPORT_PCI */
  return GM_SUCCESS;
#endif /* !GM_SUPPORT_PCI */
}

static gm_status_t
gm_set_SMP (gm_instance_state_t * is, gm_lp_t val)
{
  if (gm_is_l4 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l4.SMP, val);
      return GM_SUCCESS;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_set_SMP\n"));	/* SAB */
  if (gm_is_l5 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l5.SMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l6.SMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l7 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l7.SMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l8 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l8.SMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l9 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l9.SMP, val);
      return GM_SUCCESS;
    }

  return GM_FAILURE;
}

static gm_status_t
gm_set_RMP (gm_instance_state_t * is, gm_lp_t val)
{
  if (gm_is_l4 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l4.RMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l5 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l5.RMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l6.RMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l7 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l7.RMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l8 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l8.RMP, val);
      return GM_SUCCESS;
    }

  if (gm_is_l9 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l9.RMP, val);
      return GM_SUCCESS;
    }

  return GM_FAILURE;
}

static gm_status_t
gm_set_VERSION (gm_instance_state_t * is)
{
  if (gm_is_l4 (is))
    {
      switch (is->lanai.eeprom.lanai_cpu_version)
	{
	case 0x400:
	  gm_write_lanai_special_reg_u32 (is, l4.VERSION, 0);
	  goto done;
	case 0x401:
	case 0x402:
	case 0x403:
	  gm_write_lanai_special_reg_u32 (is, l4.VERSION, 3);
	  goto done;
	}
    }

  GM_NOTE (("Don't know VERSION value to use for LANai %x.%x\n",
	    is->lanai.eeprom.lanai_cpu_version >> 8,
	    is->lanai.eeprom.lanai_cpu_version & 0xff));
  return GM_FAILURE;

done:
  gm_arch_spin (is, 20000);
  return GM_SUCCESS;
}

static gm_status_t
gm_set_WINDOW (gm_instance_state_t * is)
{
  if (gm_is_l5 (is))
    {
      gm_write_lanai_special_reg_u32 (is, l5.WINDOW, 3);
      gm_arch_spin (is, 20000);
      return GM_SUCCESS;
    }

  GM_NOTE (("Don't know WINDOW value to use for LANai 0x%x.%x\n",
	    is->lanai.eeprom.lanai_cpu_version >> 8,
	    is->lanai.eeprom.lanai_cpu_version & 0xff));
  return GM_FAILURE;
}

/**********************************************************************
 * LANai register readout functions  (not performance critical)
 **********************************************************************/

static gm_status_t
gm_get_SMP (gm_instance_state_t * is, gm_u32_t * val)
{
  if (gm_is_l4 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l4.SMP);
      return GM_SUCCESS;
    }

  if (gm_is_l5 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l5.SMP);
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l6.SMP);
      return GM_SUCCESS;
    }

  if (gm_is_l7 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l7.SMP);
      return GM_SUCCESS;
    }

  if (gm_is_l8 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l8.SMP);
      return GM_SUCCESS;
    }

  if (gm_is_l9 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l9.SMP);
      return GM_SUCCESS;
    }

  return GM_FAILURE;
}

static gm_status_t
gm_get_RMP (gm_instance_state_t * is, gm_u32_t * val)
{
  if (gm_is_l4 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l4.RMP);
      return GM_SUCCESS;
    }

  if (gm_is_l5 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l5.RMP);
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l6.RMP);
      return GM_SUCCESS;
    }

  if (gm_is_l7 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l7.RMP);
      return GM_SUCCESS;
    }

  if (gm_is_l8 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l8.RMP);
      return GM_SUCCESS;
    }

  if (gm_is_l9 (is))
    {
      *val = gm_read_lanai_special_reg_u32 (is, l9.RMP);
      return GM_SUCCESS;
    }

  return GM_FAILURE;
}

static gm_status_t
gm_get_RTC (gm_instance_state_t * is, gm_s32_t * val)
{
  if (gm_is_l4 (is))
    {
      *val = gm_read_lanai_special_reg_s32 (is, l4.RTC);
      return GM_SUCCESS;
    }

  if (gm_is_l5 (is))
    {
      *val = gm_read_lanai_special_reg_s32 (is, l5.RTC);
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
      *val = gm_read_lanai_special_reg_s32 (is, l6.RTC);
      return GM_SUCCESS;
    }

  if (gm_is_l7 (is))
    {
      *val = gm_read_lanai_special_reg_s32 (is, l7.RTC);
      return GM_SUCCESS;
    }

  if (gm_is_l8 (is))
    {
      *val = gm_read_lanai_special_reg_s32 (is, l8.RTC);
      return GM_SUCCESS;
    }

  if (gm_is_l9 (is))
    {
      *val = gm_read_lanai_special_reg_s32 (is, l9.RTC);
      return GM_SUCCESS;
    }

  return GM_FAILURE;
}

/**********************************************************************
 * LANai microcontroller functions
 **********************************************************************/

static gm_status_t
gm_disable_beats_l6_pci2 (gm_instance_state_t * is)
{
  int base_addr;

  /* Get the address of the status/control array. */

  if (__gm_cop_wakeup (is))
    goto l6_abort;
  if (__gm_cop_send (is, 0x1f))
    goto l6_abort;
  if (__gm_cop_send (is, 0x00))
    goto l6_abort;
  if ((base_addr = __gm_cop_receive (is)) < 0)
    goto l6_abort;
  if (__gm_cop_end (is))
    goto l6_abort;

  /* write 0x0c to DModeReg (offset 0x10) */

  if (__gm_cop_wakeup (is))
    goto l6_abort;
  if (__gm_cop_send (is, 0x20))
    goto l6_abort;
  if (__gm_cop_send (is, base_addr + 0x10))
    goto l6_abort;
  if (__gm_cop_send (is, 0x0c))
    goto l6_abort;
  if (__gm_cop_end (is))
    goto l6_abort;

  return GM_SUCCESS;

l6_abort:
  GM_WARN (("Could not disable beats.\n"));
  return GM_FAILURE;
}

static gm_status_t
gm_init_uc (gm_instance_state_t * is)
{
  if (gm_is_l4 (is))
    {
      return GM_SUCCESS;
    }

  if (gm_is_l5 (is))
    {
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
#if GM_SUPPORT_PCI
      if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
	{
	  if (GM_SUPPORT_PCI_REV_2
	      && ((is->ifc.pci.config.Revision_ID == 2) ||
		  (is->ifc.pci.config.Revision_ID == 3)))
	    {
	      return gm_disable_beats_l6_pci2 (is);
	    }
	}
#endif /* GM_SUPPORT_PCI */
    }

  if (gm_is_l7 (is) || gm_is_l8 (is) || gm_is_l9 (is))
    {
      return GM_SUCCESS;
    }

  return GM_FAILURE;
}

/* If we have a cspi board, use the functions specific to that board
   instead of the PCIDMA chip based ones */
#if CSPI_MAP26xx
#include "cspi-26xx.c"
#else

/**********************************************************************
 * Control register access.
 **********************************************************************/

/****
 * LANai control register debugging
 ****/

static gm_status_t
gm_check_control_register_value (gm_instance_state_t * is)
{
#if 1				/* disabled by feldy --Glenn */
  GM_PARAMETER_MAY_BE_UNUSED (is);
  return GM_SUCCESS;
#else
  gm_u32_t crv;			/* control register value */
  GM_PARAMETER_MAY_BE_UNUSED (is);

  if (!is)
    {
      GM_NOTE (("internal error: is is NULL\n"));
      return GM_FAILURE;
    }

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      crv = gm_ntoh_u16 (*(gm_u16_n_t *) is->lanai.control_regs);
      if ((crv >> 15 & 1) == (crv >> 14 & 1))
	goto error;
      if ((crv >> 13 & 1) == (crv >> 12 & 1))
	goto error;
      return GM_SUCCESS;
    }

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      crv = gm_ntoh_u32 (is->lanai.control_regs[0]);
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  if ((crv >> 31 & 1) == (crv >> 30 & 1))
	    goto error;
	  if ((crv >> 29 & 1) == (crv >> 28 & 1))
	    goto error;
	  return GM_SUCCESS;
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;

error:
  GM_NOTE (("Bogus control register value detected: 0x%x\n", crv));
  return GM_FAILURE;
#endif
}



/****
 * LANai reset control
 ****/

gm_status_t
gm_lanai_reset_off (gm_instance_state_t * is)
{
  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1, ("Turning off reset.\n"));
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("control_reg = 0x%04x before deasserting reset\n",
		 (int) *(gm_u16_t *) is->lanai.control_regs));
      gm_always_assert (!GM_NO_PARTWORD_PIO_WRITE);
      *(gm_u16_n_t *) is->lanai.control_regs = gm_hton_u16 (1 << 15);
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("control_reg = 0x%04x after deasserting reset.\n",
		 (int) *(gm_u16_t *) is->lanai.control_regs));
      GM_STBAR ();
      goto done;
    }

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 7);
	  GM_STBAR ();
	  goto done;
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 7);
	  GM_STBAR ();
	  goto done;
	}
    }

  return GM_FAILURE;

done:
  /* Allow the LANai time to come out of reset before any of its
     registers is accessed. */
  gm_arch_spin (is, 20000);
  return GM_SUCCESS;
}

gm_status_t
gm_lanai_reset_on (gm_instance_state_t * is)
{
  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      gm_always_assert (!GM_NO_PARTWORD_PIO_WRITE);
      *(gm_u16_n_t *) is->lanai.control_regs = gm_hton_u16 (1 << 14);
      GM_STBAR ();
      return GM_SUCCESS;
    }


  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 6);
	  GM_STBAR ();
	  return GM_SUCCESS;
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 6);
	  GM_STBAR ();
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}

/****
 * Ebus reset control
 ****/

/* Return GM_SUCCESS if operation was performed, or operation is not
   applicable. */

gm_status_t
gm_ebus_reset_off (gm_instance_state_t * is)
{
  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 5);
	  GM_STBAR ();
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}

gm_status_t
gm_ebus_reset_on (gm_instance_state_t * is)
{
  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 4);
	  GM_STBAR ();
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}

/****
 * board reset control
 ****/

/* These functions must function correctly before the EEPROM is read,
   so they don't depend on any part of the EEPROM, except the "bus_type"
   field, which is set by the caller.

   Return GM_SUCCESS of operation was performed, or operation is not
   applicable. */

gm_status_t
gm_board_reset_off (gm_instance_state_t * is)
{
  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 3);
	  GM_STBAR ();
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}

gm_status_t
gm_board_reset_on (gm_instance_state_t * is)
{
  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 2);
	  GM_STBAR ();
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}


/* we've discovered some slots don't initialize correctly, force
   64-bit mode or 32-bit mode */

static gm_status_t
gm_board_force_64or32bit_mode (gm_instance_state_t * is, int which, int *readback)
{
  gm_u32_t bit_shift=0;
  *readback = 0;

  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  /* bit 07 force 64-bit mode */
	  /* bit 06 force 64-bit mode */
	  if (which==64)
	     bit_shift = 31;
	  if (which==32)
	     bit_shift = 30;
          if (!bit_shift) 
	     return GM_FAILURE;

	  GM_PRINT(1,("Forcing %d-bit PCI bus width\n",which));

	  is->lanai.control_regs[0]
	    	= gm_hton_u32 ((gm_u32_t) (((gm_u32_t) 1) << bit_shift));
	  GM_STBAR ();
	  if (gm_ntoh_u32 (is->lanai.control_regs[0])
	      & (gm_u32_t) (((gm_u32_t) 1) << bit_shift))
	    {
	      *readback = 1;
	    }
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}

/* we've discovered some slots don't initialize correctly check it */
/* This function is exported for debugging use in arch-specific code */
gm_status_t
gm_read_correct_sequence_bit (gm_instance_state_t * is, int *correct)
{
  *correct = 0;

  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if ((is->ifc.pci.config.Revision_ID == 2) ||
	  (is->ifc.pci.config.Revision_ID == 3))
	{
	  /* bit 14 correct RST/REQ64 sequence */
	  if (gm_ntoh_u32 (is->lanai.control_regs[0]) & (1 << 22))
	    {
	      *correct = 1;
	      return GM_SUCCESS;
	    }
	  return GM_SUCCESS;
	}
    }

  return GM_FAILURE;
}

/****
 * Interrupt generation control
 ****/

gm_status_t
gm_enable_interrupts (gm_instance_state_t * is)
{
  GM_CALLED ();

  if (gm_check_control_register_value (is) != GM_SUCCESS)
    {
      GM_WARN (("gm_check_control_register_value failed.\n"));
      return GM_FAILURE;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 9, ("gm_enable_interrupts called\n"));

  /* Make sure the host only ever gets host interrupts. */

  gm_set_EIMR (is, GM_HOST_SIG_BIT);
  GM_STBAR ();
  gm_arch_spin (is, 20000);

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      gm_assert (!GM_NO_PARTWORD_PIO_WRITE);
      *(gm_u16_n_t *) is->lanai.control_regs = gm_hton_u16 (1 << 13);
      GM_STBAR ();
      is->pausible = 1;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 5);
	  GM_STBAR ();
	  is->pausible = 1;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  /* is->lanai.control_regs[0]=1<<25; */
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 1);
	  GM_STBAR ();
	  is->pausible = 1;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
    }

  GM_RETURN_STATUS (GM_FAILURE);
}

gm_status_t
gm_disable_interrupts (gm_instance_state_t * is)
{
  GM_CALLED ();

  if (gm_check_control_register_value (is) != GM_SUCCESS)
    return GM_FAILURE;

  gm_set_EIMR (is, 0);

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      gm_assert (!GM_NO_PARTWORD_PIO_WRITE);
      *(gm_u16_n_t *) is->lanai.control_regs = gm_hton_u16 (1 << 12);
      GM_STBAR ();
      is->pausible = 0;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 4);
	  GM_STBAR ();
	  is->pausible = 0;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  is->lanai.control_regs[0] = gm_hton_u32 (1 << 0);
	  GM_STBAR ();
	  is->pausible = 0;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
    }

  GM_RETURN_STATUS (GM_FAILURE);
}

/****
 * Reading the control reg for printing/debugging purposes.  Note
 * that host byte order is used.
 ****/

static gm_u32_t
gm_get_control_reg (gm_instance_state_t * is)
{
  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    return 0;

  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  return gm_ntoh_u32 (is->lanai.control_regs[0]);
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  return gm_ntoh_u32 (is->lanai.control_regs[0]);
	}
    }
  return GM_FAILURE;
}

#endif /* !CSPI_MAP26xx */

/**********************************************************************/

static gm_status_t
gm_instance_init_functions (gm_instance_state_t * is)
{
  /* By default, initialize handlers to abort if called.  The function
     pointers will be replaced with appropriate functions as enough
     information becomes available to do so. */

  is->_set_EIMR = gm_oo_set_abort;
  is->set_ISR = gm_oo_set_abort;
  is->get_EIMR = gm_oo_get_abort;
  is->get_ISR = gm_oo_get_abort;

  return GM_SUCCESS;
}

#if GM_SUPPORT_PCI
/* Set flag if board and host are both capable of 64-bit DMA addressing.
 *
 * NOTE:  This is the CAPABILITY flag.  Whether the driver will actually
 *        USE 64-bit DMA addressing (aka A64 DMA) will be decided by the
 *        platform-specific code.
 * NOTE:  For the PCI case, this test is currently based on the fact that the
 *        board is L7 or greater, and that GM_SIZEOF_DP_T is 8.  The
 *        implication is that GM_SIZEOF_DP_T will represent the capability of
 *        the host.  Be sure either to preserve this association, or change
 *        the design.
 *     
 *        As of today, only IRIX hosts support 8-byte (also called A64) DMA.
 *                                                             - Max 00/03/21
 */
static gm_status_t
gm_set_64_bit_flag(gm_instance_state_t * is)
{
   if (GM_SUPPORT_PCI && (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI))
   {
      if (is->ifc.pci.config.Revision_ID >= 2)
      {
         if (GM_SIZEOF_DP_T == 8)
	 {
            is->flags |= GM_INSTANCE_64BIT_DMA_ADDR_OK;
            GM_PRINT(1, ("Unit %d is 64-bit-DMA-addr-capable\n", is->id));
	 }
         else
            GM_WARN(("Unit %d is 64-bit-DMA-addr-capable, but "
                     "GM_SIZEOF_DP_T = %d\n", is->id, GM_SIZEOF_DP_T));
      }
      else
	 GM_PRINT(1, ("Using A32 DMA for unit %d\n", is->id));
   }
   return GM_SUCCESS;
}

#endif /* GM_SUPPORT_PCI */

/* Initialize the board-specific functions for the interface.  Returns
   GM_SUCCESS on success. This should be called as soon as the EEPROM
   contents have been copied into the LANai. */

static gm_status_t
gm_init_eeprom_dependent_functions (gm_instance_state_t * is)
{
  gm_assert (is);

  if (gm_is_l4 (is))
    {
      is->_set_EIMR = gm_set_EIMR_l4;
      is->set_ISR = gm_set_ISR_l4;
      is->get_EIMR = gm_get_EIMR_l4;
      is->get_ISR = gm_get_ISR_l4;
      return GM_SUCCESS;
    }

  if (gm_is_l5 (is))
    {
      is->_set_EIMR = gm_set_EIMR_l5;
      is->set_ISR = gm_set_ISR_l5;
      is->get_EIMR = gm_get_EIMR_l5;
      is->get_ISR = gm_get_ISR_l5;
      return GM_SUCCESS;
    }

  if (gm_is_l6 (is))
    {
      is->_set_EIMR = gm_set_EIMR_l6;
      is->set_ISR = gm_set_ISR_l6;
      is->get_EIMR = gm_get_EIMR_l6;
      is->get_ISR = gm_get_ISR_l6;
      return GM_SUCCESS;
    }

  if (gm_is_l7 (is))
    {
      is->_set_EIMR = gm_set_EIMR_l7;
      is->set_ISR = gm_set_ISR_l7;
      is->get_EIMR = gm_get_EIMR_l7;
      is->get_ISR = gm_get_ISR_l7;
      return GM_SUCCESS;
    }

  if (gm_is_l8 (is))
    {
      is->_set_EIMR = gm_set_EIMR_l8;
      is->set_ISR = gm_set_ISR_l8;
      is->get_EIMR = gm_get_EIMR_l8;
      is->get_ISR = gm_get_ISR_l8;
      return GM_SUCCESS;
    }

  if (gm_is_l9 (is))
    {
      is->_set_EIMR = gm_set_EIMR_l9;
      is->set_ISR = gm_set_ISR_l9;
      is->get_EIMR = gm_get_EIMR_l9;
      is->get_ISR = gm_get_ISR_l9;
      return GM_SUCCESS;
    }

  /* No match, so fail */

  GM_NOTE (("gm_init_eeprom_dependent_functions: lanai 0x%x not supported\n",
	    is->lanai.eeprom.lanai_cpu_version));

  return GM_FAILURE;
}

/**********************************************************************
 * Memory-mapped resource location interface
 **********************************************************************/

/* gm_resource_location (IS, TYPE, RL) stores the resouce location of
   the resouce of type TYPE for the instance IS at RL.

   The implementation is inefficient in order to allow a tabular code
   structure that obviously yields correct memory layouts described
   below. This slightly inefficient but easily maintained
   implementation was chosen to simplify code maintenance in the
   future when there are even more board layouts.

   The Revision 1 PCI board layout is as follows:

   +-------------------+
   | memory            | 512K (used if memory <= 512K)
   +-------------------+ (0x00080000)
   | eeprom            | 256K
   +-------------------+ (0x000c0000)
   | special registers | 128K
   +-------------------+ (0x000e0000)
   | control registers | 128K
   +-------------------+ (0x00100000)
   | memory 2          | 1M (used if memory > 512K)
   +-------------------+


   The Revision 2 PCI board layout is as follows:

   +-------------------+ (0)
   | memory            | 
   +-------------------+ (0x00800040)
   | control registers | 
   +-------------------+ (0x00804000)
   | special registers | 
   +-------------------+ (0x00A00000)
   | eeprom            | 
   +-------------------+


   Sbus layout is as follows: (EEPROM is not memory-mapped)

 Byte offset   What                    Length
 -----------   -------------           ------
  0x00000000   Control Space (FPGA)    256*1024 Bytes ( 256KBytes)
  0x00040000   LANai Special Regs      256*1024 Bytes ( 256KBytes)
  0x00080000   LANai SRAM Memory      1024*1024 Bytes (1024KBytes)
  0x00180000   End of Board Memory            TOTAL   (2048KBytes)



CSPI MAP-26xx
 Byte offset   What                    Length
 -----------   -------------           ------
  0x40000000   LANai SRAM Memory      8*1024*1024 Bytes (8MBytes)
  0x48000000   LANai Special Regs        256*1024 Bytes ( 256KBytes)
  0xf0000000   PPCLAN Board Regs              256 Bytes (  256Bytes)
  0xffffff00   EEPROM                         128 Bytes (  128Bytes)



 */

static gm_status_t
gm_resource_location (gm_instance_state_t * is, enum gm_resource_type type,
		      gm_resource_location_t * rl)
{
  gm_resource_location_t eeprom;
  gm_resource_location_t memory;
  gm_resource_location_t control;
  gm_resource_location_t specials;

  gm_assert (rl);
#if !GM_ETHERBOOT
  if (!GM_PAGE_LEN)
    return GM_INTERNAL_ERROR;
#endif

  /* Compute all locations.  It's OK to build illegitimate location
     information if IS is not sufficiently initialized, because the
     caller should never actually request that such descriptors be
     returned to it. */

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      control.offset = 0x00000000;
      specials.offset = 0x00040000;
      memory.offset = 0x00080000;

      control.length = 256 * 1024;
      specials.length = 256 * 10324;
      memory.length = is->lanai.eeprom.lanai_sram_size;
      goto match;
    }
#if GM_SUPPORT_PCI
  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  memory.offset = 1024 * (0);
	  eeprom.offset = 1024 * (512);
	  specials.offset = 1024 * (512 + 256);
	  control.offset = 1024 * (512 + 256 + 128);
	  if (is->lanai.eeprom.lanai_sram_size >= 1024 * 512)
	    memory.offset = 1024 * (512 + 256 + 128 + 128);

	  memory.length = is->lanai.eeprom.lanai_sram_size;
	  eeprom.length = 256 * 1024;
	  specials.length = 128 * 1024;
	  control.length = 128 * 1024;
	  goto match;
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  memory.offset = 0;
	  control.offset = 0x00800040;
	  specials.offset = 0x00804000;
	  eeprom.offset = 0x00A00000;

	  memory.length = is->lanai.eeprom.lanai_sram_size;
	  control.length = 0x4000;
	  specials.length = 1024 * 16;
	  eeprom.length = 1024 * 1024 * 4;
	  goto match;
	}
    }
#endif /* GM_SUPPORT_PCI */
  
#if CSPI_MAP26xx
  {
    /* should probably replace these with the defines from
       <asm/cspi.h> (linux) or boardCsr.h (vxWorks) */
    memory.offset   = 0x40000000;
    control.offset  = 0xf0000000;
    specials.offset = 0x48000000;
    eeprom.offset   = 0xffffff00;

    memory.length   = is->lanai.eeprom.lanai_sram_size;
    control.length  = 0x00004000;
    specials.length = 0x00004000;
    eeprom.length   = 0x00000100;
    goto match;
  }
#endif

  return GM_FAILURE;

match:

  /* Return the requested location in the user-supplied buffer. */

  switch (type)
    {
    case GM_MEMORY_REGION:
      *rl = memory;
      break;
    case GM_EEPROM_REGION:
      *rl = eeprom;
      break;
    case GM_SPECIAL_REGS_REGION:
      *rl = specials;
      break;
    case GM_CONTROL_REGS_REGION:
      *rl = control;
      break;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 3, ("Found resource at 0x%x of length 0x%x.\n",
				  rl->offset, rl->length));

  /* Check for bogus returns, which should never happen if the
     instance state passed to this function was sufficiently
     initialized by the caller. */

  if (rl->length == 0)
    return GM_FAILURE;

  return GM_SUCCESS;
}

/* Map part of the LANai PCI interface into system space.  This
   generic function is used below to map the SRAM, special registers,
   and control registers, among other things.

   Returns pointer to mapped region, or NULL on failure.
 */

static void *
gm_map_board_region (gm_instance_state_t * is, enum gm_resource_type type)
{
  gm_resource_location_t rl;

  if (CSPI_MAP26xx)
    is->board_base = 0;
  else
    gm_assert (is->board_base);
  
  /* Determine the location of the resource to map */
  if (gm_resource_location (is, type, &rl) != GM_SUCCESS)
    return 0;

#if CSPI_MAP26xx && GM_OS_LINUX
  /* For most systems, we map one large space once, and we use
     this function to differentiate the different regions in
     that one large space.  But unfortunately, CSPI boards have
     their LANAI chip soldered on, and have their resources
     scattered throughout the address space.  Therefore we have to
     map the individual resources here */
  /* Addendum, the ppclan and eeprom areas are pre-mapped to the
     same area as the physical address by the linux kernel so we
     don't do it here*/
  if (type != GM_EEPROM_REGION && type != GM_CONTROL_REGS_REGION)
    {
      void *mapped_region;
      gm_arch_map_io_space( is, rl.offset, rl.length, &mapped_region );
      rl.offset = (gm_u32_t) mapped_region;
    }
#endif

  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("Mapping board resource at 0x%x of length 0x%x.\n", rl.offset,
	     rl.length));

  /* Verify the board has been mapped. */

  if (!CSPI_MAP26xx && !is->board_base)
    {
      GM_WARN (("internal error: board base not set.\n"));
      return 0;
    }

  /* Return pointer to the requested resource. */

  return (void *) ((char *) is->board_base + rl.offset);
}

/* Unmap a region mapped above */

static void
gm_unmap_board_region (gm_instance_state_t * is, enum gm_resource_type type,
		       void *ptr)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);
  GM_PARAMETER_MAY_BE_UNUSED (type);
  GM_PARAMETER_MAY_BE_UNUSED (ptr);

#if CSPI_MAP26xx && GM_OS_LINUX
{  gm_resource_location_t rl;

  if( type != GM_EEPROM_REGION && type != GM_CONTROL_REGS_REGION )
    {
      
      if (gm_resource_location (is, type, &rl) != GM_SUCCESS)
	return;
      gm_arch_unmap_io_space( is, rl.offset, rl.length, ptr );
    }
} 
#endif /* CSPI_MAP26xx */
}

#define GM_PRINT_EEPROM 1

static void
gm_print_eeprom (gm_instance_state_t * is)
{
  gm_myrinet_eeprom_t *ee;
  char board_label[sizeof (ee->board_label) + 1];

  ee = &is->lanai.eeprom;

  /* Print EEPROM contents */

  GM_PRINT (GM_PRINT_EEPROM,
            ("Contents of Myrinet EEPROM:\n"));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     lanai_clockval = 0x%08x\n", ee->lanai_clockval));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     lanai_cpu_version = 0x%04x\n", ee->lanai_cpu_version));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     lanai_board_id = %02x:%02x:%02x:%02x:%02x:%02x\n",
  		 ee->lanai_board_id[0], ee->lanai_board_id[1],
  		 ee->lanai_board_id[2], ee->lanai_board_id[3],
  		 ee->lanai_board_id[4], ee->lanai_board_id[5]));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     lanai_sram_size = 0x%08x (%dK)\n",
  		 ee->lanai_sram_size,
  		 (ee->lanai_sram_size >> 10) & 0x003fffff));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     fpga_version = \"%s\"\n", ee->fpga_version));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     more_version = \"%s\"\n", ee->more_version));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     delay_line_value = 0x%04x\n", ee->delay_line_value));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     max_lanai_speed = 0x%04x\n", ee->max_lanai_speed));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     board_type = 0x%04x\n", ee->board_type));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     bus_type = 0x%04x\n", ee->bus_type));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     product_code = 0x%04x\n", ee->product_code));
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     serial_number = 0x%08x\n", ee->serial_number));

  gm_strncpy (board_label, (char *) ee->board_label,
	      sizeof (ee->board_label));
  board_label[sizeof (ee->board_label)] = 0;
  _GM_PRINT (GM_PRINT_EEPROM,
             ("     board_label = \"%s\"\n", board_label));
}

/* Convert the eeprom from network byte order to host byte order,
   and fix buggy PCI eeproms */

static void
gm_fixup_eeprom (gm_instance_state_t * is)
{
  gm_myrinet_eeprom_t *ee;
  gm_myrinet_eeprom_n_t *from;

  ee = &is->lanai.eeprom;
  from = (gm_myrinet_eeprom_n_t *) & is->lanai.eeprom;

  /* fix the byte ordering of the eeprom to be in host order. */

  ee->lanai_clockval = gm_ntoh_u32 (from->lanai_clockval);
  ee->lanai_cpu_version = gm_ntoh_u16 (from->lanai_cpu_version);
  ee->lanai_sram_size = gm_ntoh_u32 (from->lanai_sram_size);
  ee->board_type = gm_ntoh_u16 (from->board_type);
  ee->bus_type = gm_ntoh_u16 (from->bus_type);
  ee->product_code = gm_ntoh_u16 (from->product_code);
  ee->serial_number = gm_ntoh_u32 (from->serial_number);
  ee->max_lanai_speed = gm_ntoh_u16 (from->max_lanai_speed);

  /* Null terminate board label in old eeproms that don't have a board
     label. */

  if (!gm_isprint (ee->board_label[0]))
    {
      ee->board_label[0] = 0;
    }

#if 0 && glenn
#warning temporary debug code
  
  /* fix prototype L9 prototype eeproms */

  if (GM_ENABLE_PROTOTYPE_NIC_SUPPORT && gm_is_l9 (is))
    {
      gm_u32_t override_clockval;

      override_clockval = 0x052082e0;	/* 4x */
      override_clockval = 0x052082c0;	/* 3x */
      override_clockval = 0x052082b0;	/* 2.5x */
      override_clockval = 0x05208290;	/* 1.5x */
      override_clockval = 0x052082a0;	/* 2x */
      override_clockval = 0x05208280;	/* 1x */

#if __GNUC__
#warning this uncommented hack has been disabled.  --Glenn
#endif
#if defined __SMP__ && 0
      ee->lanai_board_id[0] = 0xff;
#endif

      if (ee->lanai_clockval != override_clockval)
	{
	  GM_NOTE (("Changing prototype L9 clockval from 0x%x to 0x%x\n",
		    ee->lanai_clockval, override_clockval));
	  ee->lanai_clockval = override_clockval;
	}

      ee->max_lanai_speed = 134;
      ee->max_lanai_speed = 145;
    }
#endif

  /* fix buggy eeproms */

  if (!ee->lanai_clockval)
    ee->lanai_clockval = 0x11371137;

  if (GM_ENABLE_PROTOTYPE_NIC_SUPPORT && gm_is_l5 (is))
    {
      /* HACK for playing with prototype LANai5 PCI interfaces. */
      GM_WARN (("Overriding EEPROM for EXPERIMENTAL reasons.\n"));
      ee->lanai_clockval = 0x04278280;	/* 30MHz, single clocked */
      ee->lanai_clockval = 0x04278290;	/* 45MHz, single clocked */
      ee->lanai_clockval = 0x04278291;	/* 45MHz, double clocked */
      ee->lanai_clockval = 0x07098691;	/* 45MHz, double clocked, custom */
      ee->lanai_clockval = 0x04278281;	/* 30MHz, double clocked */
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("Using clock value 0x%08x.\n", ee->lanai_clockval));
      ee->lanai_sram_size = 1024 * 1024;
    }
}

/* Function to copy the eeprom from any type of interface into the
   instance state */

gm_status_t
gm_copy_eeprom (gm_instance_state_t * is)
{
  enum gm_bus_type known_type;
  gm_myrinet_eeprom_t *ee;	/* mapped eeprom */


  known_type = (enum gm_bus_type) is->lanai.eeprom.bus_type;
  gm_assert (known_type != GM_MYRINET_BUS_UNKNOWN);

#if GM_SUPPORT_SBUS
  /* Call special code for sbus interfaces, since eeprom mapping is
     not supported for them */
  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      if (gm_arch_sbus_copy_eeprom (is) != GM_SUCCESS)
	goto abort_with_known_type;
      goto copied_eeprom;
    }
#endif

#if CSPI_MAP26xx
  {
    unsigned long i;
    /* map, copy, unmap the eerpm */
    ee = (gm_myrinet_eeprom_t *) gm_map_board_region(is, GM_EEPROM_REGION);
    
    if (ee)
      {	  
	for (i = 0; i < sizeof (gm_myrinet_eeprom_t) / 4; i++)
	  ((gm_u32_t *) & is->lanai.eeprom)[i] = ((gm_u32_t *) ee)[i];
      }
    
    
    is->lanai.eeprom.bus_type = GM_MYRINET_BUS_MAP26xx;
    is->lanai.eeprom.board_type = GM_MYRINET_BOARD_TYPE_MAP26xx;
    gm_unmap_board_region( is, GM_EEPROM_REGION, ee );
  }
  goto copied_eeprom;
#endif
  
  /****************
   * PCI
   ****************/

#if GM_SUPPORT_PCI
  
  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      /* map, copy, and upmap the eeprom. */
      GM_PRINT (GM_PRINT_LEVEL >= 4, ("Mapping eeprom.\n"));
      ee = (gm_myrinet_eeprom_t *) gm_map_board_region (is, GM_EEPROM_REGION);
      if (ee)
	{
	  /****
	   * Rev 1
	   ****/

	  if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	    {
	      unsigned int ui;

	      GM_PRINT (GM_PRINT_LEVEL >= 4, ("Reading PCI Rev 1 eeprom.\n"));

	      /* Copy eeprom as 32-bit ints */
	      for (ui = 0; ui < sizeof (gm_myrinet_eeprom_t) / 4; ui++)
		((gm_u32_t *) & is->lanai.eeprom)[ui] = ((gm_u32_t *) ee)[ui];
	    }

	  /****
	   * Revs 2, 3
	   ****/

	  if (GM_SUPPORT_PCI_REV_2
	      && ((is->ifc.pci.config.Revision_ID == 2) ||
		  (is->ifc.pci.config.Revision_ID == 3)))
	    {
	      unsigned int i, quicklogic = 1;
	      gm_u32_n_t *ee_word_ptr;

	      ee_word_ptr = (gm_u32_n_t *) ee;
	      for (i = 0; i < 16; i++)
		{
		  if ((gm_ntoh_u32 (ee_word_ptr[i]) >> 24 & 0xFF) != i)
		    {
		      quicklogic = 0;
		      break;
		    }
		}

	      if (!quicklogic)
		{
		  unsigned int ui;

		  GM_PRINT (GM_PRINT_LEVEL >= 4,
			    ("Reading normal-format eeprom.\n"));

		  /* Copy eeprom as 32-bit ints */
		  for (ui = 0; ui < sizeof (gm_myrinet_eeprom_t) / 4; ui++)
		    {
		      ((gm_u32_t *) & is->lanai.eeprom)[ui]
			= ((gm_u32_t *) ee)[ui];
		    }
		}
	      else
		{
		  unsigned int ui;

		  GM_INFO (("Reading QuickLogic-format eeprom.\n"));

		  /* Copy eeprom in hacked-up quicklogic fashion (one byte
		     per word) */
		  for (ui = 0; ui < sizeof (gm_myrinet_eeprom_t); ui++)
		    {
		      GM_PRINT (GM_PRINT_LEVEL >= 4,
				("raw word[%d] = 0x%08x\n", ui,
				 ((gm_u32_t *) ee)[ui]));
		      ((gm_u8_t *) & is->lanai.eeprom)[ui] =
			((gm_u8_t *) ee)[4 * ui + 3];
		    }
		  if ((is->lanai.eeprom.lanai_cpu_version < 0x0500) ||
		      (is->lanai.eeprom.lanai_cpu_version > 0x0FFF))
		    {
		      ee_word_ptr = (gm_u32_n_t *) & is->lanai.eeprom;
		      GM_WARN (("EEPROM is corrupt.\n"
				"    %x %x %x %x\n",
				gm_ntoh_u32 (ee_word_ptr[0]),
				gm_ntoh_u32 (ee_word_ptr[1]),
				gm_ntoh_u32 (ee_word_ptr[2]),
				gm_ntoh_u32 (ee_word_ptr[3])));
		      goto abort_with_mapping;
		    }
		}
	    }

	  gm_unmap_board_region (is, GM_EEPROM_REGION, ee);

	  gm_fixup_eeprom (is);
	  goto copied_eeprom;
	}
    }
#endif /* GM_ENABLE_PCI */

  return GM_FAILURE;

copied_eeprom:
  gm_print_eeprom (is);

  /* Handle interfaces that report their bus type incorrectly. (sigh) */
  is->lanai.eeprom.bus_type = known_type;

  /* Now that we have the contents of the interface EEPROM, initialize
     the dynamic functions for the instance, which depend on the
     interface type. */

  if (gm_init_eeprom_dependent_functions (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not init dynamic instance functions.\n"));
      return GM_FAILURE;
    }
  return GM_SUCCESS;

abort_with_mapping:
  gm_unmap_board_region (is, GM_EEPROM_REGION, ee);
#if GM_SUPPORT_SBUS
abort_with_known_type:
#endif
  is->lanai.eeprom.bus_type = known_type;	/* restore */
  return GM_FAILURE;
}

/************************************************************************
 * Board mapping/unmapping
 ************************************************************************/

/************
 * utility functions to assist mapping.
 ************/

#if GM_SUPPORT_PCI

static gm_status_t
gm_enable_PCI_config_command_bit (gm_instance_state_t * is, gm_u16_t value)
{
  gm_u16_t command;
  gm_status_t status;

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Enabling PCI command bit 0x%x\n", value));
  status = gm_read_pci_config_command (is, &command);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not read PCI command register.\n"));
      return status;
    }

  command |= value;
  status = gm_write_pci_config_command (is, command);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not write PCI command register.\n"));
      return status;
    }

  /* Pause for at least 10ms */
  gm_arch_spin (is, 15000);

  status = gm_read_pci_config_command (is, &command);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not read back PCI command register.\n"));
      return status;
    }

  if ((command & value) != value)
    {
      GM_NOTE (
	       ("gm_enable_PCI_config_command_bit couldn't set bit = 0x%x\n",
		value));
    }

  return GM_SUCCESS;
}

static gm_status_t
gm_disable_PCI_config_command_bit (gm_instance_state_t * is, gm_u16_t value)
{
  gm_u16_t command;


/* This function should be using gm_read_pci_config_command() and
   gm_write_pci_config_command()                  - Max 2000/0606  */
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Disable PCI command bit 0x%x\n", value));
  if (gm_arch_read_pci_config_16 (is, GM_OFFSETOF (gm_pci_config_t, Command),
				  &command) != GM_SUCCESS)
    {
      GM_NOTE (("Could not read PCI command register.\n"));
      return GM_FAILURE;
    }
  command &= ~(value);
  if (gm_arch_write_pci_config_16 (is, GM_OFFSETOF (gm_pci_config_t, Command),
				   command) != GM_SUCCESS)
    {
      GM_NOTE (("Could not write PCI command register.\n"));
      return GM_FAILURE;
    }
  /* Pause for at least 10ms */
  gm_arch_spin (is, 15000);
  return GM_SUCCESS;
}

#endif /* GM_SUPPORT_PCI */

/************
 * Utility function to perform tests on mappings.
 ************/

/* test a region of RAM to make sure it is readable, writable, and
   is comprised of unique memory locations. */

static gm_status_t
gm_test_and_clear_sram (gm_instance_state_t * is,
			volatile gm_u8_n_t * orig_ptr, unsigned int orig_len)
{
  unsigned int offset, len = orig_len / 4;
  volatile gm_u32_n_t *ptr = (gm_u32_n_t *) orig_ptr;
  gm_u32_t tmp;
  
  if (((long) orig_ptr & 3) != 0)
    return GM_FAILURE;

  /* test for stuck bits... */

  /* set the even bits and clear the odd bits. */

  gm_always_assert (((long) orig_ptr & 3) == 0);
  gm_always_assert ((orig_len & 3) == 0);
  for (offset = 0; offset < len; offset++)
    __gm_kton_u32 (is, &ptr[offset], 0xaaaaaaaa);
  for (offset = 0; offset < len; offset++)
    {
      tmp = __gm_ntok_u32 (is, &ptr[offset]);
      if (tmp != 0xaaaaaaaa)
	{
	  GM_NOTE (("SRAM test failed at 0x%08x.\n",
		    offset * sizeof (ptr[0])));
	  _GM_NOTE (("Got 0x%08x, re-try 0x%08x, expected 0x%08x.\n", tmp,
		     __gm_ntok_u32 (is, &ptr[offset]), 0xaaaaaaaa));
	  return GM_FAILURE;
	}
    }

  /* ... and vice versa */

  for (offset = 0; offset < len; offset++)
    __gm_kton_u32 (is, &ptr[offset], 0x55555555);
  for (offset = 0; offset < len; offset++)
    {
      tmp = __gm_ntok_u32 (is, &ptr[offset]);
      if ( tmp != 0x55555555)
	{
	  GM_NOTE (("SRAM test failed at 0x%08x.\n",
		    offset * sizeof (ptr[0])));
	  _GM_NOTE (("Got 0x%08x, re-try 0x%08x, expected 0x%08x.\n", tmp,
		    __gm_ntok_u32(is, &ptr[offset]), 0x55555555));
	  return GM_FAILURE;
	}
    }

  /* test for mirroring by writing unique identifiers into each 32-bit
     word. */

#if 0
#error Does not use PIO functions --Glenn
  /* align the bottom of the buffer on a 32-bit boundary. */
  offset = (unsigned long) ptr & 3;
  if (offset)
    {
      ptr = (volatile unsigned char *) ((char *) ptr - offset + 4);
      len = len + offset - 4;
    }

  /* align the top of the buffer on a 32-bit boundary. */
  offset = len & 3;
  if (offset)
    len -= offset;
#endif

  /* Write unique IDs */
  for (offset = 0; offset < len; offset++)
    __gm_kton_u32 (is, &ptr[offset], offset);

  /* Verify unique IDs */
  for (offset = 0; offset < len; offset++)
    {
      if (__gm_ntok_u32 (is, &ptr[offset]) != offset)
	{
	  GM_NOTE
	    (("SRAM test failed at 0x%08x read: 0x%08x expected: 0x%08x.\n",
	      offset * sizeof (ptr[0]), __gm_ntok_u32(is, &ptr[offset]),
	      offset ));
	  return GM_FAILURE;
	}
    }

  /* All tests passed.  Clear memory and return. */

  __gm_kton_bzero (is, orig_ptr, orig_len);
  return GM_SUCCESS;
}

static gm_status_t
gm_test_RMP (gm_instance_state_t * is)
{
  gm_u32_t val;

  /* Test writing and reading RMP */

  if (gm_set_RMP (is, 0xcafebab8) != GM_SUCCESS)
    {
      GM_WARN (("Could not set RMP to cafebabe.\n"));
      goto abort_with_nothing;
    }
  if (gm_get_RMP (is, &val) != GM_SUCCESS)
    {
      GM_WARN (("Could not read RMP (cafebab8).\n"));
      goto abort_with_nothing;
    }
  if (val != 0xcafebab8)
    {
      GM_WARN (("RMP = 0x%x (!= 0xcafebab8)\n", val));
      goto abort_with_nothing;
    }

  /* Clear RMP */

  if (gm_set_RMP (is, 0) != GM_SUCCESS)
    {
      GM_WARN (("Could not set RMP to 0.\n"));
      goto abort_with_nothing;
    }
  if (gm_get_RMP (is, &val) != GM_SUCCESS)
    {
      GM_WARN (("Could not read RMP (0).\n"));
      goto abort_with_nothing;
    }
  if (val != 0)
    {
      GM_WARN (("RMP = 0x%x (!= 0)\n", val));
      goto abort_with_nothing;
    }

  return GM_SUCCESS;

abort_with_nothing:
  return GM_FAILURE;
}

/* Put the interface into a known state - requires control regs to be
   mapped. */
static void
gm_reset_everything (gm_instance_state_t * is)
{
  gm_lanai_reset_on (is);
  gm_ebus_reset_on (is);
  gm_board_reset_on (is);
  gm_arch_spin (is, 20000);
}

/****************************************************************
 * Map the various parts of the LANai into system memory.
 ****************************************************************/

/* Due to hardware variations, the steps needed to detect and map
   various boards varies significantly from board revision to board
   revision.  Therefore, we implement one version of
   gm_map_io_spaces() for each board revision. */

/****************
 * sbus
 ****************/

static gm_status_t
gm_map_io_spaces_sbus (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4,
            ("Mapping LANai[%d] io spaces.\n", is->id));

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("mapping SBUS board.\n"));

  /* Map the entire board */

  is->board_span = 0x00180000;
  if (gm_arch_map_io_space (is, 0, is->board_span, &is->board_base)
      != GM_SUCCESS)
    goto abort_with_nothing;

  /* Map the control regs. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Mapping control registers.\n"));
  is->lanai.control_regs
    = (gm_u32_n_t *) gm_map_board_region (is, GM_CONTROL_REGS_REGION);
  if (!is->lanai.control_regs)
    {
      GM_WARN (("Could not map control_regs\n"));
      goto abort_with_mapped_board;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("Control regs at %p.\n", is->lanai.control_regs));
#define GM_PRINT_CONTROL_REG() GM_PRINT (GM_PRINT_LEVEL >= 4, ("is->lanai.control_regs[0] = 0x%04x.\n", ((gm_u16_t *)is->lanai.control_regs)[0]))
  GM_PRINT_CONTROL_REG ();

  /* Force the interface into reset, since the previously loaded
     driver may have left it in a messy state. */
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Resetting LANai.\n"));
  gm_reset_everything (is);

  GM_PRINT_CONTROL_REG ();

  /* Copy the EEPROM from the LANai, mapping it temporarily to do so.
     This must be done before mapping the SRAM, since the SRAM region
     location depends on the contents of the EEPROM.

     Also, gm_copy_eeprom() intializes all the eeprom-dependent
     instance function pointers, so the EEPROM should be copied before
     any of these functions is invoked (other than
     is->board_reset_off() above. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Copying board EEPROM\n"));
  if (gm_copy_eeprom (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not copy EEPROM.\n"));
      goto abort_with_control_regs;
    }

  /* We don't set the delay_line value, so GM won't work on the very
     very earliest LANai4 boards.  If we set it, this would be the
     place. */

  /* Map the special registers */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping special registers.\n"));
  is->lanai.special_regs
    = ((gm_lanai_special_registers_t *)
       gm_map_board_region (is, GM_SPECIAL_REGS_REGION));
  if (!is->lanai.special_regs)
    {
      GM_WARN (("Could not map special_regs\n"));
      goto abort_with_eeprom;
    }

  /* After mapping the special regs, set the clock value */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Setting LANai clock_val to 0x%08x.\n",
				  is->lanai.eeprom.lanai_clockval));
  if (gm_set_clock_val (is) != GM_SUCCESS)
    goto abort_with_special_regs;

  /* Map and clear the SRAM. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping SRAM (0x%x bytes).\n",
				  is->lanai.eeprom.lanai_sram_size));
  is->lanai.sram = gm_map_board_region (is, GM_MEMORY_REGION);
  if (is->lanai.sram == NULL)
    {
      GM_WARN (("Could not map SRAM\n"));
      goto abort_with_special_regs;
    }

  return GM_SUCCESS;

abort_with_special_regs:
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION, is->lanai.special_regs);
abort_with_eeprom:
abort_with_control_regs:
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (gm_u32_t *) is->lanai.control_regs);
abort_with_mapped_board:
abort_with_nothing:
  return GM_FAILURE;
}

static void
gm_unmap_io_spaces_sbus (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Unmapping IO spaces.\n"));
  gm_unmap_board_region (is, GM_MEMORY_REGION, (char *) is->lanai.sram);
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION, is->lanai.special_regs);
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (gm_u32_t *) is->lanai.control_regs);
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
}

/****************
 * map26xx
 ****************/

static gm_status_t
gm_map_io_spaces_map26xx (gm_instance_state_t * is)
{
  GM_INFO (("Mapping CSPI MAP 26xx board.\n"));

  /* Determine the board span */

  /* CSPI 26xx dont use pci 'boards' so set board_base to 0 */
  is->board_base = 0;
  
  /* Map the control regs first, since we must take the board out of reset
     before accessing the EEPROM to prevent bus errors. */

  GM_INFO (("Mapping control registers.\n"));
  is->lanai.control_regs
    = (gm_u32_n_t *) gm_map_board_region (is, GM_CONTROL_REGS_REGION);
  if (!is->lanai.control_regs)
    {
      GM_WARN (("Could not map control_regs\n"));
      goto abort_with_mapped_board;
    }
  GM_INFO (("Mapped control registers at %p.\n",
	    is->lanai.control_regs));
  
  /* Force the interface into reset, since the previously loaded
     driver may have left it in a messy state. */
  gm_reset_everything (is);

  /* Copy the EEPROM from the LANai, mapping it temporarily to do so.
     This must be done before mapping the SRAM, since the SRAM region
     location depends on the contents of the EEPROM.

     Also, gm_copy_eeprom() intializes all the eeprom-dependent
     instance function pointers, so the EEPROM should be copied before
     any of these functions is invoked (other than
     is->board_reset_off() above. */

  if (gm_copy_eeprom (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not copy EEPROM.\n"));
      goto abort_with_control_regs;
    }

  /* We don't set the delay_line value, so GM won't work on the very
     very earliest LANai4 boards.  If we set it, this would be the
     place. */

  /* Map the special registers */

  GM_INFO (("Mapping special registers.\n"));
  is->lanai.special_regs
    = ((gm_lanai_special_registers_t *)
       gm_map_board_region (is, GM_SPECIAL_REGS_REGION));
  if (!is->lanai.special_regs)
    {
      GM_WARN (("Could not map special_regs\n"));
      goto abort_with_eeprom;
    }
  GM_INFO (("Mapped special registers at %p.\n",
	    is->lanai.special_regs));
  
  /* After mapping the special regs, set the clock value */

  GM_INFO (("Setting LANai clock_val to 0x%08x.\n",
	    is->lanai.eeprom.lanai_clockval));
  if (gm_set_clock_val (is) != GM_SUCCESS)
    goto abort_with_special_regs;
  
  (void) gm_get_control_reg;
  GM_INFO (("control reg value is 0x%x.\n",
	    gm_get_control_reg (is)));

#define SAB
#ifdef SAB
  /* Now that we have mapped the special regs, but before we access
     any memory accessed through the LANai EBUS, we need to take the
     ebus out of reset. We also make sure the LANai is in reset before
     we clear its memory below.  */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Taking EBUS out of reset. (0x%08x)\n",
				  gm_get_control_reg (is)));
  if (gm_ebus_reset_off (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not take EBUS out of reset.\n"));
      goto abort_with_eeprom;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("control reg value is 0x%x.\n", gm_get_control_reg (is)));
#endif /* SAB */
#undef SAB

  /* Map and clear the SRAM. */

  GM_INFO (("Mapping SRAM (0x%x bytes).\n",
	    is->lanai.eeprom.lanai_sram_size));
  is->lanai.sram = gm_map_board_region (is, GM_MEMORY_REGION);
  if (is->lanai.sram == NULL)
    {
      GM_WARN (("Could not map SRAM\n"));
      goto abort_with_special_regs;
    }
  GM_INFO (("Mapped SRAM at %p.\n", is->lanai.sram));
  GM_INFO (("Clearing LANai SRAM from %p to %p.\n",
	    is->lanai.sram,
	    (char *) is->lanai.sram + is->lanai.eeprom.lanai_sram_size));
  __gm_kton_bzero (is, is->lanai.sram, is->lanai.eeprom.lanai_sram_size);
  GM_INFO (("Cleared LANai[%d] SRAM\n", is->id));

  return GM_SUCCESS;

abort_with_special_regs:
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION, is->lanai.special_regs);
abort_with_eeprom:
abort_with_control_regs:
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (void *) is->lanai.control_regs);
abort_with_mapped_board:
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
abort_with_nothing:
  return GM_FAILURE;
}

static void
gm_unmap_io_spaces_map26xx (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Unmapping IO spaces (CSPI MAP26xx).\n"));
  gm_unmap_board_region (is, GM_MEMORY_REGION, (void *) is->lanai.sram);
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION,
			 (void *) is->lanai.special_regs);
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (void *) is->lanai.control_regs);
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
}

#if GM_SUPPORT_PCI

/****************
 * PCI
 ****************/

/* Determine the PCI board span */

gm_u32_t
gm_pci_board_span (gm_instance_state_t * is)
{
  gm_u32_t bits;

  if (is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      gm_u32_t save;

      if (gm_arch_read_pci_config_32
	  (is, GM_OFFSETOF (gm_pci_config_t,
			    Base_Addresses_Registers[0]), &save)
	  != GM_SUCCESS)
	{
	  return 0;
	}
      if (gm_arch_write_pci_config_32
	  (is, GM_OFFSETOF (gm_pci_config_t, Base_Addresses_Registers[0]),
	   0xffffffff) != GM_SUCCESS)
	{
	  return 0;
	}
      if (gm_arch_read_pci_config_32
	  (is, GM_OFFSETOF (gm_pci_config_t,
			    Base_Addresses_Registers[0]), &bits)
	  != GM_SUCCESS)
	{
	  return 0;
	}
      if (gm_arch_write_pci_config_32
	  (is, GM_OFFSETOF (gm_pci_config_t,
			    Base_Addresses_Registers[0]), save) != GM_SUCCESS)
	{
	  return 0;
	}
      return ~(bits & 0xfffffff0) + 1;
    }

  return 0;
}

static gm_status_t
gm_map_io_spaces_pci_rev_1 (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4,
            ("Mapping LANai[%d] io spaces.\n", is->id));

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("mapping PCI Rev 1 board.\n"));

  /* Determine the board span */

  is->board_span = gm_pci_board_span (is);
  if (!is->board_span)
    {
      GM_NOTE (("Could not determine board span.\n"));
      goto abort_with_nothing;
    }

  /* Map the entire board */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping entire board.\n"));
  if (gm_arch_map_io_space (is, 0, is->board_span, &is->board_base)
      != GM_SUCCESS)
    {
      GM_NOTE (("Could not map entire LANai interface into kernel.\n"));
      goto abort_with_nothing;
    }

  /* Map the control regs first, since we must take the board out of reset
     before accessing the EEPROM to prevent bus errors. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping control registers.\n"));
  is->lanai.control_regs
    = (gm_u32_n_t *) gm_map_board_region (is, GM_CONTROL_REGS_REGION);
  if (!is->lanai.control_regs)
    {
      GM_NOTE (("Could not map control_regs\n"));
      goto abort_with_mapped_board;
    }

  /* Force the interface into reset, since the previously loaded
     driver may have left it in a messy state. */
  gm_reset_everything (is);

  /* Copy the EEPROM from the LANai, mapping it temporarily to do so.
     This must be done before mapping the SRAM, since the SRAM region
     location depends on the contents of the EEPROM.

     Also, gm_copy_eeprom() intializes all the eeprom-dependent
     instance function pointers, so the EEPROM should be copied before
     any of these functions is invoked (other than
     is->board_reset_off() above. */

  if (gm_copy_eeprom (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not copy EEPROM.\n"));
      goto abort_with_control_regs;
    }

  /* We don't set the delay_line value, so GM won't work on the very
     very earliest LANai4 boards.  If we set it, this would be the
     place. */

  /* Map the special registers */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping special registers.\n"));
  is->lanai.special_regs
    = ((gm_lanai_special_registers_t *)
       gm_map_board_region (is, GM_SPECIAL_REGS_REGION));
  if (!is->lanai.special_regs)
    {
      GM_NOTE (("Could not map special_regs\n"));
      goto abort_with_eeprom;
    }

  /* After mapping the special regs, set the clock value */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Setting LANai clock_val to 0x%08x.\n",
				  is->lanai.eeprom.lanai_clockval));
  if (gm_set_clock_val (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not set lanai clock\n"));
      goto abort_with_special_regs;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("control reg value is 0x%x.\n", gm_get_control_reg (is)));

  /* Map and clear the SRAM. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping SRAM (0x%x bytes).\n",
				  is->lanai.eeprom.lanai_sram_size));
  is->lanai.sram = gm_map_board_region (is, GM_MEMORY_REGION);
  if (is->lanai.sram == NULL)
    {
      GM_WARN (("Could not map SRAM\n"));
      goto abort_with_special_regs;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Clearing LANai SRAM from %p to %p.\n",
				  is->lanai.sram,
				  (char *) is->lanai.sram +
				  is->lanai.eeprom.lanai_sram_size));
  __gm_kton_bzero (is, is->lanai.sram, is->lanai.eeprom.lanai_sram_size);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Cleared LANai[%d] SRAM\n", is->id));

  return GM_SUCCESS;

abort_with_special_regs:
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION, is->lanai.special_regs);
abort_with_eeprom:
abort_with_control_regs:
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (void *) is->lanai.control_regs);
abort_with_mapped_board:
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
abort_with_nothing:
  return GM_FAILURE;
}

static void
gm_unmap_io_spaces_pci_rev_1 (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Unmapping IO spaces.\n"));
  gm_unmap_board_region (is, GM_MEMORY_REGION, (void *) is->lanai.sram);
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION,
			 (void *) is->lanai.special_regs);
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (void *) is->lanai.control_regs);
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
}


static gm_status_t
gm_map_io_spaces_pci_rev_2 (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4,
            ("Mapping LANai[%d] io spaces.\n", is->id));

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("mapping PCI Rev 2 or 3 board.\n"));

  /* Determine the board span */

  is->board_span = gm_pci_board_span (is);
  if (!is->board_span)
    {
      GM_NOTE (("Could not determine board span.\n"));
      goto abort_with_nothing;
    }

  /* Map the entire board */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping entire board.\n"));
  if (gm_arch_map_io_space (is, 0, is->board_span, &is->board_base)
      != GM_SUCCESS)
    {
      GM_NOTE (("pci_rev2: Could NOT map board into kernel (span = 0x%x)\n",
		is->board_span));
      goto abort_with_nothing;
    }

  /* Disable busmaster DMA until the board is taken out of ebus
     reset because these boards can cause bus glitches */

  if (gm_disable_PCI_config_command_bit (is, GM_PCI_COMMAND_MASTER) !=
      GM_SUCCESS)
    {
      GM_NOTE (("Could not disable PCI bus master DMA.\n"));
      goto abort_with_mapped_board;
    }

  /* Map the control regs first, since we must take the board out of reset
     before accessing the EEPROM to prevent bus errors. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping control registers.\n"));
  is->lanai.control_regs
    = (gm_u32_n_t *) gm_map_board_region (is, GM_CONTROL_REGS_REGION);
  if (!is->lanai.control_regs)
    {
      GM_WARN (("Could not map control_regs\n"));
      goto abort_with_mapped_board;
    }

  /* Force the interface into reset, since the previously loaded
     driver may have left it in a messy state. */
  gm_reset_everything (is);

  /* Take the board out of reset, so we can get at the eeprom. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Taking board out of reset.\n"));
  if (gm_board_reset_off (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not take interface out of reset.\n"));
      goto abort_with_control_regs;
    }

  /* Map the special regs before reading the EEPROM so we can stop the
     runaway PLL (via the clock_val register) before reading the
     eeprom */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping special registers.\n"));
  is->lanai.special_regs
    = ((gm_lanai_special_registers_t *)
       gm_map_board_region (is, GM_SPECIAL_REGS_REGION));
  if (!is->lanai.special_regs)
    {
      GM_WARN (("Could not map special_regs\n"));
      goto abort_with_board_reset_off;
    }

#if 0
  /* EXPERIMENTAL: Try to read the EEPROM 100 times. */
  {
    int attempt;
    for (attempt = 0; attempt < 100; attempt++)
      {
	if (gm_stop_runaway_PLL (is) != GM_SUCCESS)
	  continue;
	if (gm_copy_eeprom (is) != GM_SUCCESS)
	  continue;
	else
	  break;
      }
  }
#else
  /* Make sure the PLL on the board is not running away, draining too much
     power to read the EEPROM. */

  if (gm_stop_runaway_PLL (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not stop runaway PLL\n"));
      goto abort_with_special_regs;
    }
#endif

  /* Copy the EEPROM from the LANai, mapping it temporarily to do so.
     This must be done before mapping the SRAM, since the SRAM region
     location depends on the contents of the EEPROM.

     Also, gm_copy_eeprom() intializes all the eeprom-dependent
     instance function pointers, so the EEPROM should be copied before
     any of these functions is invoked (other than
     is->board_reset_off() above. */

  if (gm_copy_eeprom (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not copy EEPROM.\n"));
      goto abort_with_special_regs;
    }

  /* Set the clock value */
  if (gm_set_clock_val (is) != GM_SUCCESS)
    goto abort_with_eeprom;

  /* Reenable PCI bus master DMA */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Reenabling PCI bus master DMA.\n"));
  if (gm_enable_PCI_config_command_bit (is, GM_PCI_COMMAND_MASTER) !=
      GM_SUCCESS)
    goto abort_with_eeprom;

  {
    int mode64 = 0;
    gm_read_correct_sequence_bit (is, &mode64);
    if (mode64)
      {
	GM_INFO (("64-bit slot initialized correctly.\n"));
        is->bus_width = 64;

	if (GM_ENABLE_FORCE_32BIT_MODE)
	  {
	    int readback = 0;
	    gm_board_force_64or32bit_mode (is, 32, &readback);
	    if (readback)
	      {
		GM_INFO (("PCI slot FORCED to 32-bit mode\n"));
	      }
	    else
	      {
		GM_INFO (("unable to FORCE PCI slot to 32-bit mode"
			  " - 64-bit mode enabled\n"));
	      }
	  }
      }
    else
      {
	GM_INFO (("32-bit slot detected.\n"));
	is->bus_width = 32;

	if (GM_ENABLE_FORCE_64BIT_MODE)
	  {
	    int readback = 0;
	    gm_board_force_64or32bit_mode (is, 64, &readback);
	    if (readback)
	      {
		GM_INFO (("PCI slot FORCED to 64-bit mode\n"));
	      }
	    else
	      {
		GM_INFO (("unable to FORCE PCI slot to 64-bit mode"
			  " - 32-bit mode enabled\n"));
	      }
	  }
      }
  }

  /* Now that we have mapped the special regs, but before we access
     any memory accessed through the LANai EBUS, we need to take the
     ebus out of reset. We also make sure the LANai is in reset before
     we clear its memory below.  */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Taking EBUS out of reset. (0x%08x)\n",
				  gm_get_control_reg (is)));
  if (gm_ebus_reset_off (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not take EBUS out of reset.\n"));
      goto abort_with_eeprom;
    }

  gm_arch_spin (is, 20000);	/* BAD: Should not be needed. */

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("control reg value is 0x%x.\n", gm_get_control_reg (is)));

  /* Initialize the L6 microcontroller, if any. */

  if (gm_init_uc (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not initialize microcontroller.\n"));
      goto abort_with_ebus_reset_off;
    }

  /* Map and clear the SRAM. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("  Mapping SRAM (0x%x bytes).\n",
				  is->lanai.eeprom.lanai_sram_size));
  is->lanai.sram = gm_map_board_region (is, GM_MEMORY_REGION);
  if (is->lanai.sram == NULL)
    {
      GM_WARN (("Could not map SRAM\n"));
      goto abort_with_ebus_reset_off;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Clearing LANai SRAM from %p to %p.\n",
				  is->lanai.sram,
				  (char *) is->lanai.sram +
				  is->lanai.eeprom.lanai_sram_size));
  __gm_kton_bzero (is, (void *) is->lanai.sram,
		   is->lanai.eeprom.lanai_sram_size);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Cleared LANai[%d] SRAM\n", is->id));

  return GM_SUCCESS;

abort_with_ebus_reset_off:
  gm_ebus_reset_on (is);
abort_with_eeprom:
abort_with_special_regs:
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION, is->lanai.special_regs);
abort_with_board_reset_off:
  gm_board_reset_on (is);
abort_with_control_regs:
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (void *) is->lanai.control_regs);
abort_with_mapped_board:
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
abort_with_nothing:
  return GM_FAILURE;
}

static void
gm_unmap_io_spaces_pci_rev_2 (gm_instance_state_t * is)
{
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Unmapping IO spaces.\n"));
  gm_unmap_board_region (is, GM_MEMORY_REGION, (void *) is->lanai.sram);
  gm_ebus_reset_on (is);
  gm_unmap_board_region (is, GM_SPECIAL_REGS_REGION, is->lanai.special_regs);
  gm_board_reset_on (is);
  gm_unmap_board_region (is, GM_CONTROL_REGS_REGION,
			 (void *) is->lanai.control_regs);
  gm_arch_unmap_io_space (is, 0, is->board_span, &is->board_base);
}

static void
gm_verify_pci_command_bit (gm_instance_state_t * is, gm_u16_t bit,
			   const char *s)
{
  gm_u16_t command;

  gm_read_pci_config_command (is, &command);
  if (!(command & bit))
    {
      GM_INFO (("PCI config %s NOT set by BIOS\n", s));
      if (gm_enable_PCI_config_command_bit (is, bit) != GM_SUCCESS)
	{
	  GM_WARN (("PCI config  could NOT be set\n"));
	}
      else
	{
	  gm_read_pci_config_command (is, &command);
	  if (command & bit)
	    GM_INFO (("PCI config %s now turned ON\n", s));
	  else
	    GM_WARN (("PCI config %s didn't stay on\n", s));
	}
    }
  else
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1, ("PCI config %s is already set\n", s));
    }
}

#endif /* GM_SUPPORT_PCI */

/****************************************************************
 * Generic board mapping/unmapping.
 ****************************************************************/

gm_status_t
gm_map_io_spaces (gm_instance_state_t * is)
{
  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      return gm_map_io_spaces_sbus (is);
    }

#if GM_SUPPORT_PCI
  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("PCI_REV %d\n", is->ifc.pci.config.Revision_ID));

      gm_verify_pci_command_bit (is, GM_PCI_COMMAND_INVALIDATE,
				 "memory-write-and-invalidate");
      gm_verify_pci_command_bit (is, GM_PCI_COMMAND_MASTER, "master");
      gm_verify_pci_command_bit (is, GM_PCI_COMMAND_MEMORY, "memory");

      if (GM_SUPPORT_PCI_REV_1 && is->ifc.pci.config.Revision_ID == 1)
	{
	  return gm_map_io_spaces_pci_rev_1 (is);
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  return gm_map_io_spaces_pci_rev_2 (is);
	}
    }
#endif /* GM_SUPPORT_PCI */
  
  if (CSPI_MAP26xx && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_MAP26xx)
    {
      return gm_map_io_spaces_map26xx (is);
    }

  return GM_FAILURE;
}

void
gm_unmap_io_spaces (gm_instance_state_t * is)
{
  GM_CALLED ();

  if (GM_SUPPORT_SBUS && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_SBUS)
    {
      gm_unmap_io_spaces_sbus (is);
      return;
    }

#if GM_SUPPORT_PCI
  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (is->ifc.pci.config.Revision_ID == 1)
	{
	  gm_unmap_io_spaces_pci_rev_1 (is);
	  return;
	}

      if (GM_SUPPORT_PCI_REV_2
	  && ((is->ifc.pci.config.Revision_ID == 2) ||
	      (is->ifc.pci.config.Revision_ID == 3)))
	{
	  gm_unmap_io_spaces_pci_rev_2 (is);
	  return;
	}
    }
#endif /*GM_SUPPORT_PCI*/
  
  if (CSPI_MAP26xx && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_MAP26xx)
    {
      gm_unmap_io_spaces_map26xx (is);
      return;
    }

  GM_RETURN_NOTHING ();
}

/********************
 * MCP loading
 ********************/

/* Functions required to uncompress the MCP */

static voidpf
gm_zlib_calloc (voidpf opaque, uInt items, uInt size)
{
  gm_u32_t len;
  void *ret;

  GM_PARAMETER_MAY_BE_UNUSED (opaque);

  size = GM_ROUNDUP (u32, size, 8);
  len = items * size;
  ret = gm_arch_kernel_malloc (len, 0);
  gm_bzero (ret, len);
  return ret;
}

static void
gm_zlib_free (voidpf opaque, voidpf addr)
{
  GM_PARAMETER_MAY_BE_UNUSED (opaque);
  gm_arch_kernel_free (addr);
}


/* declare all the MCPs supported by this driver */

#define GM_MCP_DECL(type)						\
extern const char gm_control_program ## type [];			\
extern const unsigned gm_control_program ## type ## _length;		\
extern const unsigned int gm_control_program ## type ## _uncompressed_length

#if GM_SUPPORT_64_BIT_USER_PTRS && GM_SUPPORT_32_BIT_USER_PTRS
#error only one user ptr bit size can be supported by a driver
#endif

#if GM_SUPPORT_64_BIT_USER_PTRS

GM_MCP_DECL (_64b_l4_0k);
GM_MCP_DECL (_64b_l4_0k_compact);
GM_MCP_DECL (_64b_l4_16k);
GM_MCP_DECL (_64b_l4_4k);
GM_MCP_DECL (_64b_l4_4k_compact);
GM_MCP_DECL (_64b_l4_4k_compact512);
GM_MCP_DECL (_64b_l4_8k);
GM_MCP_DECL (_64b_l4_8k_compact);
GM_MCP_DECL (_64b_l4_8k_compact512);
GM_MCP_DECL (_64b_l5_0k);
GM_MCP_DECL (_64b_l5_16k);
GM_MCP_DECL (_64b_l5_4k);
GM_MCP_DECL (_64b_l5_8k);
GM_MCP_DECL (_64b_l6_0k);
GM_MCP_DECL (_64b_l6_16k);
GM_MCP_DECL (_64b_l6_4k);
GM_MCP_DECL (_64b_l6_8k);
GM_MCP_DECL (_64b_l7_0k);
GM_MCP_DECL (_64b_l7_0k_newfeatures);
GM_MCP_DECL (_64b_l7_16k);
GM_MCP_DECL (_64b_l7_16k_newfeatures);
GM_MCP_DECL (_64b_l7_4k);
GM_MCP_DECL (_64b_l7_4k_newfeatures);
GM_MCP_DECL (_64b_l7_8k);
GM_MCP_DECL (_64b_l7_8k_newfeatures);
GM_MCP_DECL (_64b_l8_0k);
GM_MCP_DECL (_64b_l8_16k);
GM_MCP_DECL (_64b_l8_4k);
GM_MCP_DECL (_64b_l8_8k);
GM_MCP_DECL (_64b_l9_0k);
GM_MCP_DECL (_64b_l9_0k_newfeatures);
GM_MCP_DECL (_64b_l9_16k);
GM_MCP_DECL (_64b_l9_16k_newfeatures);
GM_MCP_DECL (_64b_l9_4k);
GM_MCP_DECL (_64b_l9_4k_newfeatures);
GM_MCP_DECL (_64b_l9_8k);
GM_MCP_DECL (_64b_l9_8k_newfeatures);

#define GM_MCP_PARAM(type)						\
  &gm_control_program_64b_##type [0],					\
  &gm_control_program_64b_##type##_length,				\
  &gm_control_program_64b_##type##_uncompressed_length

#elif GM_SUPPORT_32_BIT_USER_PTRS

GM_MCP_DECL (_32b_l4_0k);
GM_MCP_DECL (_32b_l4_0k_compact);
GM_MCP_DECL (_32b_l4_16k);
GM_MCP_DECL (_32b_l4_4k);
GM_MCP_DECL (_32b_l4_4k_compact);
GM_MCP_DECL (_32b_l4_8k);
GM_MCP_DECL (_32b_l4_8k_compact);
GM_MCP_DECL (_32b_l4_8k_compact512);
GM_MCP_DECL (_32b_l5_0k);
GM_MCP_DECL (_32b_l5_16k);
GM_MCP_DECL (_32b_l5_4k);
GM_MCP_DECL (_32b_l5_8k);
GM_MCP_DECL (_32b_l6_0k);
GM_MCP_DECL (_32b_l6_16k);
GM_MCP_DECL (_32b_l6_4k);
GM_MCP_DECL (_32b_l6_8k);
GM_MCP_DECL (_32b_l7_0k);
GM_MCP_DECL (_32b_l7_0k_newfeatures);
GM_MCP_DECL (_32b_l7_16k);
GM_MCP_DECL (_32b_l7_16k_newfeatures);
GM_MCP_DECL (_32b_l7_4k);
GM_MCP_DECL (_32b_l7_4k_newfeatures);
GM_MCP_DECL (_32b_l7_8k);
GM_MCP_DECL (_32b_l7_8k_newfeatures);
GM_MCP_DECL (_32b_l8_0k);
GM_MCP_DECL (_32b_l8_16k);
GM_MCP_DECL (_32b_l8_4k);
GM_MCP_DECL (_32b_l8_8k);
GM_MCP_DECL (_32b_l9_0k);
GM_MCP_DECL (_32b_l9_16k);
GM_MCP_DECL (_32b_l9_4k);
GM_MCP_DECL (_32b_l9_8k);
GM_MCP_DECL (_32b_l9_0k_newfeatures);
GM_MCP_DECL (_32b_l9_16k_newfeatures);
GM_MCP_DECL (_32b_l9_4k_newfeatures);
GM_MCP_DECL (_32b_l9_8k_newfeatures);

#define GM_MCP_PARAM(type)						\
  &gm_control_program_32b_##type [0],					\
  &gm_control_program_32b_##type##_length,				\
  &gm_control_program_32b_##type##_uncompressed_length

#else /* GM_SUPPORT_32_BIT_USER_PTRS */
#error None of GM_SUPPORT_*_BIT_USER_PTRS is set
#endif

#if GM_SUPPORT_L7 | GM_SUPPORT_L8 | GM_SUPPORT_L9
GM_MCP_DECL (_lanai_rate);
#endif

/* List of control programs, in order of preference. */

static struct gm_control_program_descriptor
{
  const char *program;
  const unsigned *zlength;
  const unsigned *mcp_length;
  const gm_u32_t required_mem_size;
  const gm_u16_t min_lanai_version;
  const gm_u16_t max_lanai_version;
  const gm_u32_t page_len;
  const gm_u32_t cache_line_size;
  const char *program_name;
}
gm_mcp_descriptor[] =
{
#if GM_SUPPORT_L4 && GM_SUPPORT_0K_PAGES
  {
    GM_MCP_PARAM (l4_0k),
      512 * 1024, 0x401, 0x403, 4096, GM_RDMA_GRANULARITY, "L4 0K",}
  ,
#if GM_MIN_SUPPORTED_SRAM == 256
  {				/* compact MCP for 256K boards */
    GM_MCP_PARAM (l4_0k_compact),
      256 * 1024, 0x401, 0x403, 4096, GM_RDMA_GRANULARITY, "L4 0K compact",}
  ,
#endif
#endif
#if GM_SUPPORT_L4 && GM_SUPPORT_4K_PAGES
#if GM_MIN_SUPPORTED_SRAM == 256
  {				/* compact MCP for 256K boards */
    GM_MCP_PARAM (l4_4k_compact),
      256 * 1024, 0x401, 0x403, 4096, GM_RDMA_GRANULARITY, "L4 4K compact"}
  ,
#endif
  {
    GM_MCP_PARAM (l4_4k),
      512 * 1024, 0x401, 0x403, 4096, GM_RDMA_GRANULARITY, "L4 4K",}
  ,
#endif
#if GM_SUPPORT_L4 && GM_SUPPORT_8K_PAGES
#if GM_MIN_SUPPORTED_SRAM == 256
  {				/* compact MCP for 256K boards */
    GM_MCP_PARAM (l4_8k_compact),
      256 * 1024, 0x401, 0x403, 8192, GM_RDMA_GRANULARITY, "L4 8K compact"}
  ,
#endif
#if GM_MIN_SUPPORTED_SRAM == 512
  {				/* compact MCP for 512K boards */
    GM_MCP_PARAM (l4_8k_compact512),
      512 * 1024, 0x401, 0x403, 8192, GM_RDMA_GRANULARITY, "L4 8K compact"}
  ,
#endif
  {
    GM_MCP_PARAM (l4_8k),
      512 * 1024, 0x401, 0x403, 8192, GM_RDMA_GRANULARITY, "L4 8K",}
  ,
#endif
#if GM_SUPPORT_L4 && GM_SUPPORT_16K_PAGES
#if GM_MIN_SUPPORTED_SRAM == 256
  {				/* compact MCP for 256K boards */
    GM_MCP_PARAM (l4_16k_compact),
      256 * 1024, 0x401, 0x403, 16384, GM_RDMA_GRANULARITY, "L4 16K compact"}
  ,
#endif
  {
    GM_MCP_PARAM (l4_16k),
      512 * 1024, 0x401, 0x403, 16384, GM_RDMA_GRANULARITY, "L4 16K",}
  ,
#endif
    /* L5 */
#if GM_SUPPORT_L5 && GM_SUPPORT_0K_PAGES
  {
    GM_MCP_PARAM (l5_0k),
      512 * 1024, 0x500, 0x506, 4096, GM_RDMA_GRANULARITY, "L5 0K",}
  ,
#endif
#if GM_SUPPORT_L5 && GM_SUPPORT_4K_PAGES
  {
    GM_MCP_PARAM (l5_4k),
      512 * 1024, 0x500, 0x506, 4096, GM_RDMA_GRANULARITY, "L5 4K",}
  ,
#endif
#if GM_SUPPORT_L5 && GM_SUPPORT_8K_PAGES
  {
    GM_MCP_PARAM (l5_8k),
      512 * 1024, 0x500, 0x506, 8192, GM_RDMA_GRANULARITY, "L5 8K",}
  ,
#endif
#if GM_SUPPORT_L5 && GM_SUPPORT_16K_PAGES
  {
    GM_MCP_PARAM (l5_16k),
      512 * 1024, 0x500, 0x506, 16384, GM_RDMA_GRANULARITY, "L5 16K",}
  ,
#endif
    /* L6 */
#if GM_SUPPORT_L6 && GM_SUPPORT_4K_PAGES
  {
    GM_MCP_PARAM (l6_4k),
      512 * 1024, 0x600, 0x601, 4096, GM_RDMA_GRANULARITY, "L6 4K",}
  ,
#endif
#if GM_SUPPORT_L6 && GM_SUPPORT_8K_PAGES
  {
    GM_MCP_PARAM (l6_8k),
      512 * 1024, 0x600, 0x601, 8192, GM_RDMA_GRANULARITY, "L6 8K",}
  ,
#endif
    /* L7 */
#if GM_SUPPORT_L7 && GM_SUPPORT_0K_PAGES
  {
    GM_MCP_PARAM (l7_0k),
      512 * 1024, 0x700, 0x705, 4096, GM_RDMA_GRANULARITY, "L7 0K",}
  ,
#endif
#if GM_SUPPORT_L7 && GM_SUPPORT_4K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l7_4k_newfeatures),
      1024 * 1024, 0x700, 0x705, 4096, GM_RDMA_GRANULARITY,
      "L7 4K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l7_4k),
      1024 * 1024, 0x700, 0x705, 4096, GM_RDMA_GRANULARITY, "L7 4K",}
  ,
#endif
#if GM_SUPPORT_L7 && GM_SUPPORT_8K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l7_8k_newfeatures),
      1024 * 1024, 0x700, 0x705, 8192, GM_RDMA_GRANULARITY,
      "L7 8K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l7_8k),
      1024 * 1024, 0x700, 0x705, 8192, GM_RDMA_GRANULARITY, "L7 8K",}
  ,
#endif
#if GM_SUPPORT_L7 && GM_SUPPORT_16K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l7_16k_newfeatures),
      1024 * 1024, 0x700, 0x705, 16384, GM_RDMA_GRANULARITY,
      "L7 16K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l7_16k),
      1024 * 1024, 0x700, 0x705, 16384, GM_RDMA_GRANULARITY, "L7 16K",}
  ,
#endif
#if GM_SUPPORT_L8 && GM_SUPPORT_4K_PAGES
  {
    GM_MCP_PARAM (l8_4k),
      512 * 1024, 0x800, 0x802, 4096, GM_RDMA_GRANULARITY, "L8 4K",}
  ,
#endif
#if GM_SUPPORT_L8 && GM_SUPPORT_8K_PAGES
  {
    GM_MCP_PARAM (l8_8k),
      512 * 1024, 0x800, 0x802, 8192, GM_RDMA_GRANULARITY, "L8 8K",}
  ,
#endif
#if GM_SUPPORT_L8 && GM_SUPPORT_16K_PAGES
  {
    GM_MCP_PARAM (l8_16k),
      512 * 1024, 0x800, 0x802, 16384, GM_RDMA_GRANULARITY, "L8 16K",}
  ,
#endif
#if GM_SUPPORT_L9 && GM_SUPPORT_0K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l9_0k_newfeatures),
      1024 * 1024, 0x900, 0x904, 4096, GM_RDMA_GRANULARITY,
      "L9 0K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l9_0k),
      512 * 1024, 0x900, 0x904, 4096, GM_RDMA_GRANULARITY, "L9 0K",}
  ,
#endif
#if GM_SUPPORT_L9 && GM_SUPPORT_4K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l9_4k_newfeatures),
      1024 * 1024, 0x900, 0x904, 4096, GM_RDMA_GRANULARITY,
      "L9 4K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l9_4k),
      512 * 1024, 0x900, 0x904, 4096, GM_RDMA_GRANULARITY, "L9 4K",}
  ,
#endif
#if GM_SUPPORT_L9 && GM_SUPPORT_8K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l9_8k_newfeatures),
      1024 * 1024, 0x900, 0x904, 8192, GM_RDMA_GRANULARITY,
      "L9 8K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l9_8k),
      512 * 1024, 0x900, 0x904, 8192, GM_RDMA_GRANULARITY, "L9 8K",}
  ,
#endif
#if GM_SUPPORT_L9 && GM_SUPPORT_16K_PAGES
#if GM_ENABLE_NEW_FEATURES
  {
    GM_MCP_PARAM (l9_16k_newfeatures),
      1024 * 1024, 0x900, 0x904, 16384, GM_RDMA_GRANULARITY,
      "L9 16K (new features)",}
  ,
#endif
  {
    GM_MCP_PARAM (l9_16k),
      512 * 1024, 0x900, 0x904, 16384, GM_RDMA_GRANULARITY, "L9 16K",}
  ,
#endif
  {
    0, 0, 0, 0, 0, 0, 0, 0, 0	/* null final array element */
  }
};




/************************************************************************
 * Ethernet address translation table
 ************************************************************************/

/* This driver caches a table of ethernet IDs for each GM ID.  In
   addition, it provides another LANai-writable table of ethernet IDs.
   stale entries are detected by comparing the cached entry with the
   entry in the LANai-writable table before returning values to the
   user.  The cache is updated whenever stale entries are detected. */

static gm_status_t
gm_register_addr_table_piece (void *arg, gm_dp_t dma_addr, gm_u32_t page_num)
{
  gm_instance_state_t *is = arg;

  gm_assert (page_num < GM_MAX_NUM_ADDR_TABLE_PIECES);
  gm_assert (is);
  gm_assert (is->lanai.globals);

  if (((page_num % 16) == 0) || (page_num == GM_MAX_NUM_ADDR_TABLE_PIECES - 1))
     GM_PRINT (GM_PRINT_LEVEL >= 1,
               ("Recording DMA addr 0x%8x for address table page %d.\n",
               dma_addr, page_num));
  else
     GM_PRINT (GM_PRINT_LEVEL >= 7,
               ("Recording DMA addr 0x%8x for address table page %d.\n",
               dma_addr, page_num));
  gm_write_lanai_global_dp (is, ethernet.addr_table_piece[page_num],
                            dma_addr);
  return GM_SUCCESS;
}

static gm_status_t
gm_register_name_table_piece (void *arg, gm_dp_t dma_addr, gm_u32_t page_num)
{
  gm_instance_state_t *is = arg;

  gm_assert (page_num < GM_MAX_NUM_NAME_TABLE_PIECES);
  gm_assert (is);
  gm_assert (is->lanai.globals);

  if (((page_num % 16) == 0) || (page_num == GM_MAX_NUM_NAME_TABLE_PIECES - 1))
     GM_PRINT (GM_PRINT_LEVEL >= 1,
               ("Recording DMA addr 0x%8x for name table page %d.\n",
               dma_addr, page_num));
  else
     GM_PRINT (GM_PRINT_LEVEL >= 7,
               ("Recording DMA addr 0x%8x for name table page %d.\n",
               dma_addr, page_num));
  gm_write_lanai_global_dp (is, name_table_piece[page_num], dma_addr);
  return GM_SUCCESS;
}

static long
compare_unique_ids (void *key1, void *key2)
{
  return gm_memcmp (key1, key2, sizeof (gm_unique_id_64_t));
}

static unsigned long
hash_unique_ids (void *key)
{
  return gm_crc (key, sizeof (gm_unique_id_64_t));
}

#define GM_DEBUG_ALLOC_ADDR_TRANSLATION_TABLE 0
static gm_status_t
gm_alloc_addr_translation_table (gm_instance_state_t * is)
{
  gm_status_t status;
  gm_size_t e_addr_table_size;
  gm_size_t e_name_table_size;

  /* Allocate the LANai-maintained addr table (one entry per node) */
  if (!is->max_node_id)
    {
      status = GM_INTERNAL_ERROR;
      goto abort_with_nothing;
    }
  e_addr_table_size = (is->max_node_id + 1) * sizeof (gm_unique_id_64_t);
  GM_PRINT (GM_DEBUG_HOST_DMA,
	    ("Allocating the ethernet address translation table\n"));
  status = gm_arch_dma_region_alloc (is,
				     &is->ethernet.addr_tab.dma_region,
				     e_addr_table_size,
				     GM_ARCH_DMA_RDWR |
				     GM_ARCH_DMA_CONSISTENT,
				     gm_register_addr_table_piece, is);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not alloc address translation table DMAable mem."));
      goto abort_with_nothing;
    }
  GM_PRINT (GM_DEBUG_ALLOC_ADDR_TRANSLATION_TABLE,
	    ("Allocated DMAable address translation table.\n"));
  is->ethernet.addr_tab.volatile_ethernet_addr
    = gm_arch_dma_region_kernel_addr (&is->ethernet.addr_tab.dma_region);
  if (!is->ethernet.addr_tab.volatile_ethernet_addr) {
    GM_NOTE (("Couldn't set the volatile_ethernet_addr\n"));
    goto abort_with_dmaable_table;
  }
  
  gm_bzero (is->ethernet.addr_tab.volatile_ethernet_addr, e_addr_table_size);


  /* Allocate the ethernet_id->gm_id hash table (one entry per node) */

  is->ethernet.addr_tab.cache_hash
    = gm_create_hash (compare_unique_ids, hash_unique_ids,
		      sizeof (gm_unique_id_64_t), sizeof (unsigned int),
		      is->max_node_id, 0);
  if (!is->ethernet.addr_tab.cache_hash)
    {
      GM_NOTE (("Could not alloc ethernet address table cache hash\n"));
      goto abort_with_dmaable_table;
    }
  GM_PRINT (GM_DEBUG_ALLOC_ADDR_TRANSLATION_TABLE,
	    ("Allocated ethernet address table cache hash.\n"));

  /* Allocate the LANai-maintained name table (one entry per node) */
  if (!is->max_node_id)
    {
      status = GM_INTERNAL_ERROR;
      goto abort_with_cache_hash;
    }
  e_name_table_size = (is->max_node_id + 1) * GM_MAX_HOST_NAME_LEN;
  GM_PRINT (GM_DEBUG_HOST_DMA,
	    ("Allocating the ethernet name translation table\n"));
  status = gm_arch_dma_region_alloc (is,
				     &is->name_table.dma_region,
				     e_name_table_size,
				     GM_ARCH_DMA_RDWR
				     | GM_ARCH_DMA_CONSISTENT,
				     gm_register_name_table_piece, is);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not alloc name translation table DMAable mem.\n"));
      goto abort_with_cache_hash;
    }
  GM_PRINT (GM_DEBUG_ALLOC_ADDR_TRANSLATION_TABLE,
	    ("Allocated DMAable name translation table\n"));
  is->name_table.entry
    = gm_arch_dma_region_kernel_addr (&is->name_table.dma_region);
  if (!is->name_table.entry)
    {
      GM_WARN (("Internal error: could nt get kernel addr for name table\n"));
      goto abort_with_name_dma_region;
    }
  GM_PRINT (GM_DEBUG_ALLOC_ADDR_TRANSLATION_TABLE,
	    ("*** name translation table from %p to %p\n",
	     is->name_table.entry,
	     is->name_table.entry[is->max_node_id + 1]));
  gm_bzero (is->name_table.entry, e_name_table_size);
  gm_assert (GM_MAX_HOST_NAME_LEN == ((char *) is->name_table.entry[1]
				      - (char *) is->name_table.entry[0]));

  return GM_SUCCESS;

abort_with_name_dma_region:
  /* prevent bogus DMAs using stale table values */
  gm_clear_lanai_global (is, name_table_piece);
  gm_arch_dma_region_free (&is->name_table.dma_region);
abort_with_cache_hash:
  gm_destroy_hash (is->ethernet.addr_tab.cache_hash);
abort_with_dmaable_table:
  /* prevent bogus DMAs using stale table values */
  gm_clear_lanai_global (is, ethernet.addr_table_piece);
  gm_arch_dma_region_free (&is->ethernet.addr_tab.dma_region);
abort_with_nothing:
  return GM_FAILURE;
}

static void
gm_free_addr_translation_table (gm_instance_state_t * is)
{
  /* prevent bogus DMAs using stale table values */
  gm_clear_lanai_global (is, name_table_piece);
  gm_arch_dma_region_free (&is->name_table.dma_region);
  gm_destroy_hash (is->ethernet.addr_tab.cache_hash);
  /* prevent bogus DMAs using stale table values */
  gm_clear_lanai_global (is, ethernet.addr_table_piece);
  gm_arch_dma_region_free (&is->ethernet.addr_tab.dma_region);
}

/************************************************************************
 * Load the control program
 ************************************************************************/

/* Load the Myrinet Control Program (mcp) into the board's sram
 * and take the board out of reset. 
 *
 * NOTE: Some or all SGI (IRIX) platforms do not set the Cache Line Size
 *       register of the PCI Config space.  In this case, the value
 *       is->cache_line_size will always be 0, and the MCP-selection logic
 *       below will not be influenced by this parameter.  As far as we can
 *       tell, this does not cause us any problems.         - Max 99/04/27
 */

#define GM_DEBUG_MCP_LOADING 1

static gm_status_t
gm_instance_load_mcp (gm_instance_state_t * is)
{
  struct gm_control_program_descriptor *d;
  gm_status_t status;

#if !GM_ETHERBOOT
  if (!GM_PAGE_LEN)
    return GM_INTERNAL_ERROR;
#endif

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Loading the MCP.\n"));

  /* find the biggest control program that will fit in the LANai SRAM. */

  GM_PRINT (GM_DEBUG_MCP_LOADING,
            ("Finding best control program for this LANai[%d].\n",
            is->id));

  for (d = &gm_mcp_descriptor[0]; d->program; d++)
    {
      GM_PRINT (GM_DEBUG_MCP_LOADING,
                ("check MCP: '%s'  len= %u (0x%x), req= %u (0x%x),\n"
                 "    min_ver= %x, max_ver= %x, page_len = %d, caln= %d\n",
                d->program_name,
		*d->mcp_length,       *d->mcp_length,
	        d->required_mem_size, d->required_mem_size,
	        d->min_lanai_version, d->max_lanai_version,
	        d->page_len,          d->cache_line_size));
      if (d->required_mem_size <= is->lanai.eeprom.lanai_sram_size
	  && *d->mcp_length <= is->lanai.eeprom.lanai_sram_size
	  && d->min_lanai_version <= is->lanai.eeprom.lanai_cpu_version
	  && d->max_lanai_version >= is->lanai.eeprom.lanai_cpu_version
#if GM_ETHERBOOT
	  && d->page_len == 0
#else
	  && d->page_len == GM_PAGE_LEN
#endif
#if !GM_OS_VXWORKS
#if !GM_CPU_x86 && !GM_CPU_powerpc
	  && d->cache_line_size >= is->cache_line_size
#endif
#endif
	)

	break;
    }
  if (!d->program)
    {
      GM_WARN (("Could not find matching control program for unit %d.\n",
                is->id));
      _GM_WARN (("LANai = 0x%x  sram_size = 0x%x Bytes\n",
		 is->lanai.eeprom.lanai_cpu_version,
		 is->lanai.eeprom.lanai_sram_size));
      _GM_WARN (("Supported MCPs:\n"));
      for (d = &gm_mcp_descriptor[0]; d->program; d++)
	{
	  GM_INFO (("MCP: '%s' m:%d l:%d v:%x-%x pl:%d cl:%d\n",
		    d->program_name, d->required_mem_size, *d->mcp_length,
		    d->min_lanai_version, d->max_lanai_version, d->page_len,
		    d->cache_line_size));
	}
      status = GM_UNSUPPORTED_DEVICE;
      goto abort_with_nothing;
    }

  GM_INFO (("Using MCP for unit %d: '%s' m:%d l:%d v:%x-%x pl:%d cl:%d\n",
	     is->id,
	     d->program_name, d->required_mem_size,*d->mcp_length,
	     d->min_lanai_version, d->max_lanai_version,
	     d->page_len, d->cache_line_size));
#if GM_SUPPORT_PCI
  if (GM_SUPPORT_PCI_REV_3 && is->ifc.pci.config.Revision_ID == 3)
    {
       if (gm_update_clockval_rev3(is) != GM_SUCCESS)
         {
           GM_NOTE (("Couldn't update the LANai clockval for PCI rev3 board.\n"));
	   goto abort_with_nothing;

         }
       GM_INFO(("Updated the clockval for pci rev 3 board\n"));
    }
#endif /* GM_SUPPORT_PCI */
  GM_INFO (("Inflating control program.\n"));
  /* Copy the control program into the LANai. */

  {
    /* Load a compressed MCP - see zlib.h for
       details */

    z_stream zs;
    int save_sram_size, rv;
    void *inflate_buffer;

    zs.zalloc = gm_zlib_calloc;
    zs.zfree = gm_zlib_free;
    zs.opaque = 0;

    save_sram_size = is->lanai.eeprom.lanai_sram_size;
    if (GM_NO_PARTWORD_PIO_WRITE)
      {
	/* inflate into kernel memory buffer to avoid partword PIO */

	inflate_buffer = gm_malloc (save_sram_size);
	if (!inflate_buffer)
	  {
	    GM_NOTE (("Could not allocate inflate buffer\n"));
	    goto abort_with_nothing;
	  }
	__gm_kton_bzero (is, inflate_buffer, save_sram_size);
      }
    else
      {
	/* inflate directly into LANai SRAM */

	inflate_buffer = (void *) is->lanai.sram;
      }

    rv = inflateInit (&zs);
    if (rv != Z_OK)
      {
	GM_NOTE (("fatal error in inflateInit(%d)\n", rv));
	goto abort_with_inflate_buffer;
      }

    zs.next_in = (unsigned char *) d->program;
    zs.avail_in = *d->zlength;

    save_sram_size = is->lanai.eeprom.lanai_sram_size;

    zs.next_out = (unsigned char *) inflate_buffer;
    zs.avail_out = save_sram_size;

    GM_PRINT (GM_PRINT_LEVEL >= 10, ("next_in %x %x %x %x %x\n", *zs.next_in,
				     *(zs.next_in + 1), *(zs.next_in + 2),
				     *(zs.next_in + 3), *(zs.next_in + 4)));


    GM_PRINT (GM_PRINT_LEVEL >= 10, ("start to inflate mcp\n"));
    _GM_PRINT (GM_PRINT_LEVEL >= 10,
	       ("   lanai sram space originally    = %d\n", save_sram_size));
    _GM_PRINT (GM_PRINT_LEVEL >= 10,
	       ("   bytes remaining to uncompress  = %d\n", zs.avail_in));
    _GM_PRINT (GM_PRINT_LEVEL >= 10,
	       ("   Available space for decompress = %d\n", zs.avail_out));


    {
      int err;

      struct internal_state
      {

	/* mode */
	enum
	{
	  METHOD,		/* waiting for method byte */
	  FLAG,			/* waiting for flag byte */
	  DICT4,		/* four dictionary check bytes to go */
	  DICT3,		/* three dictionary check bytes to go */
	  DICT2,		/* two dictionary check bytes to go */
	  DICT1,		/* one dictionary check byte to go */
	  DICT0,		/* waiting for inflateSetDictionary */
	  BLOCKS,		/* decompressing blocks */
	  CHECK4,		/* four check bytes to go */
	  CHECK3,		/* three check bytes to go */
	  CHECK2,		/* two check bytes to go */
	  CHECK1,		/* one check byte to go */
	  DONE,			/* finished check, done */
	  BAD
	}			/* got an error--stay here */
	mode;			/* current inflate mode */

	/* mode dependent information */
	union
	{
	  uInt method;		/* if FLAGS, method byte */
	  struct
	  {
	    uLong was;		/* computed check value */
	    uLong need;		/* stream check value */
	  }
	  check;		/* if CHECK, check values to compare */
	  uInt marker;		/* if BAD, inflateSync's marker bytes count */
	}
	sub;			/* submode */

	/* mode independent information */
	int nowrap;		/* flag for no wrapper */
	uInt wbits;		/* log2(window size)  (8..15, default 15) */
	int *blocks;		/* current inflate_blocks state */

      };

      if ((err = inflate (&zs, Z_FINISH)) != Z_STREAM_END
	  || zs.avail_in != 0 || zs.avail_out == 0)
	{
	  GM_NOTE (("fatal error in inflate - mcp too big   err %d?\n", err));
	  _GM_NOTE (("   lanai sram space originally    = %d\n",
		     save_sram_size));
	  _GM_NOTE (("   bytes remaining to uncompress  = %d\n",
		     zs.avail_in));
	  _GM_NOTE (("   Available space for decompress = %d\n",
		     zs.avail_out));
	  _GM_NOTE (("   inflate error msg %s\n", zs.msg));
	  _GM_NOTE (("   inflate error char %x\n",
		     *(unsigned char *) zs.next_in));
	  _GM_NOTE (("   inflate error char %x\n",
		     *(unsigned char *) (zs.next_in - 1)));
	  _GM_NOTE (("   inflate error char %x\n",
		     *(unsigned char *) (zs.next_in + 1)));
	  _GM_NOTE (("   inflate error total %ld\n", zs.total_in));
	  goto abort_with_inflate_init;
	}
    }

    if (inflateEnd (&zs) != Z_OK)
      {
	GM_NOTE (("fatal error in inflateEnd\n"));
	goto abort_with_inflate_buffer;
      }


    if (GM_NO_PARTWORD_PIO_WRITE)
      {

	__gm_kton_bcopy (is, inflate_buffer, is->lanai.sram, zs.total_out);
	gm_free (inflate_buffer);
      }

    GM_PRINT (GM_DEBUG_MCP_LOADING, ("Inflated mcp from %ld to %ld bytes.\n",
				     *d->zlength, zs.total_out));

    /* error handling */

    if (0)
      {
      abort_with_inflate_init:
	inflateEnd (&zs);
      abort_with_inflate_buffer:
	gm_free (inflate_buffer);
	goto abort_with_nothing;
      }
  }


#ifdef DELETE_FELDY
  /* The special registers are not writeable when the LANai is in reset.
     They are cleared to 0 during reset, so this might have worked assuming
     the registers are cleared when reset goes high rather than low.
     For LANai7, this might be a real problem given the separation of
     LANai reset and ebus_reset, let's just skip it - feldy
   */

  /* Make sure SMP is really cleared before taking the interface out
     of reset */

  if (gm_set_SMP (is, 0) != GM_SUCCESS)
    {
      GM_WARN (("Could not clear SMP.\n"));
      goto abort_with_nothing;
    }
  {
    gm_u32_t smp;
    if (gm_get_SMP (is, &smp) != GM_SUCCESS || smp != 0)
      {
	GM_WARN (("SMP was not cleared.\n"));
	goto abort_with_nothing;
      }
  }
#endif /* DELETE_FELDY */

  /* Take the LANai out of reset. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Taking the LANai out of reset\n"));
  if (gm_lanai_reset_off (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not take LANai out of reset.\n"));
      goto abort_with_set_SMP;
    }
  is->lanai.running = 1;
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("LANai now out of reset\n"));

  /* Now that the LANai is out of reset, test host access to LANai
     registers. */

  if (gm_test_RMP (is) != GM_SUCCESS)
    {
      GM_WARN (("RMP test FAILED!\n"));
      goto abort_running;
    }

  /* Wait for the MCP to initialize itself */

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("Waiting for the LANai MCP to initialize itself.\n"));
  {
    gm_u32_t smp;
    gm_s32_t start_time, current_time;

    if (gm_get_RTC (is, &start_time) != GM_SUCCESS)
      goto abort_running;

    do
      {
	if (gm_get_SMP (is, &smp) != GM_SUCCESS)
	  goto abort_running;
	if (gm_get_RTC (is, &current_time) != GM_SUCCESS)
	  goto abort_running;
	/* Test for timeout.  This simple comparison works because RTC
	   is initialized to zero during reset */
	if (current_time - start_time > 20 * 1000 * 1000)
	  {
	    gm_lp_t prt_str_ofst;
	    char    *s_print;
	    char    *s_assert;

	    GM_NOTE (("[%d]: lanai did not initialize itself in 10 seconds"
		      " (0x%x).\n",
		      is->id, gm_ntoh_u32 (*((gm_u32_n_t *) is->lanai.sram))));
	    GM_PRINT (GM_PRINT_LEVEL >= 1,
		      ("current_time = 0x%x, start_time = 0x%x\n",
		       current_time, start_time));
	    _GM_PRINT (GM_PRINT_LEVEL >= 1, ("smp = 0x%08x\n", smp));

	    prt_str_ofst = gm_read_lanai_global_lp(is, interrupt.print.string);
	    if (prt_str_ofst)
	      {
		s_print = (char *) is->lanai.sram + prt_str_ofst;
                GM_INFO(("Lanai print string: %s\n", s_print));
	      }
	    prt_str_ofst = gm_read_lanai_global_lp(
                                          is, interrupt.failed_assertion.text);
	    if (prt_str_ofst)
	      {
		s_assert = (char *) is->lanai.sram + prt_str_ofst;
                GM_INFO(("Lanai assertion string: %s\n", s_assert));
	      }
	    
	    goto abort_running;
	  }
      }
    while (!smp);

    /* Sometimes if we read the SMP while it is being written, we
       don't get the right value, so read it again */
    {
      gm_u32_t re_smp;

      if (gm_get_SMP (is, &re_smp) != GM_SUCCESS)
	goto abort_running;
      if (re_smp != smp)
	{
	  GM_INFO (("Fixed mis-read SMP value. This is not a problem.\n"));
	}
      smp = re_smp;
    }

    /* Record pointer to LANai globals, which the MCP has placed in SMP */
    is->lanai.globals =
      (gm_lanai_globals_t *) ((char *) is->lanai.sram + smp);
    GM_PRINT (GM_PRINT_LEVEL >= 4,
	      ("LANai globals at LANai address %x.\n", smp));
  }

#define GM_DEBUG_GLOBAL_SIZE 1

  GM_PRINT (GM_DEBUG_GLOBAL_SIZE, ("Globals at 0x%p (SRAM at 0x%p)\n",
				  is->lanai.globals, is->lanai.sram));
  GM_PRINT (GM_DEBUG_GLOBAL_SIZE, ("global magic number is 0x%x.\n",
				  gm_read_lanai_global_u32 (is, magic)));
  _GM_PRINT (GM_DEBUG_GLOBAL_SIZE, ("global length is %d.\n",
				  gm_read_lanai_global_u32 (is, length)));
  _GM_PRINT (GM_DEBUG_GLOBAL_SIZE, ("sizeof(gm_lanai_globals_t) is %d.\n",
				  sizeof (gm_lanai_globals_t)));
  _GM_PRINT (GM_DEBUG_GLOBAL_SIZE, ("global end magic number is %x.\n",
				  gm_read_lanai_global_u32 (is, end_magic)));
  if ((gm_read_lanai_global_u32 (is, magic) != 0xcafebabe)
      || (gm_read_lanai_global_u32 (is, length)
	  != sizeof (gm_lanai_globals_t))
      || (gm_read_lanai_global_u32 (is, end_magic) != 0xcafebabe))
    {
      GM_NOTE (("Myrinet interface is incompatible with installed driver\n"));
      if (gm_read_lanai_global_u32 (is, magic) != 0xcafebabe)
	{
	  _GM_NOTE (("(bad magic number at start)\n"));
	}
      if (gm_read_lanai_global_u32 (is, length)
	  != sizeof (gm_lanai_globals_t))
	{
	  _GM_NOTE (("(bad globals length  lanai is %d  host is %d)\n",
		     gm_read_lanai_global_u32 (is, length),
		     sizeof (gm_lanai_globals_t)));
	}
      if (gm_read_lanai_global_u32 (is, end_magic) != 0xcafebabe)
	{
	  _GM_NOTE (("(bad magic number at end)\n"));
	}
      status = GM_INTERNAL_ERROR;
      goto abort_running;
    }

  /* Now that we know where the LANai globals are, set those
     that must be set by the host. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Setting some LANai variables.\n"));

  /* Tell the LANai the size of its SRAM */

  gm_write_lanai_global_s32
    (is, sram_length, is->lanai.eeprom.lanai_sram_size);

  /* wait for the LANai to compute its max_node_id */

  {
    gm_u32_t smp;
    gm_s32_t start_time, current_time;
    gm_u32_t max_node_id;
      
    if (gm_get_RTC (is, &start_time) != GM_SUCCESS)
      goto abort_running;

    do
      {
	max_node_id = gm_read_lanai_global_u16 (is, max_node_id);
	if (gm_get_RTC (is, &current_time) != GM_SUCCESS)
	  goto abort_running;
	/* Test for timeout. */
	if (current_time - start_time > 20 * 1000 * 1000)
	  {
	    GM_NOTE (("lanai did not set max_node_id in 10 seconds"
		      " (0x%x).\n",
		      gm_ntoh_u32 (*((gm_u32_n_t *) is->lanai.sram))));
	    goto abort_running;
	  }
      }
    while (!max_node_id);

    if (max_node_id < 16)
      {
	GM_WARN (("Board does not have enough memory (max_node_id = %d).\n",
		  max_node_id));
	_GM_WARN (("The board has %u bytes of memory.\n",
		   is->lanai.eeprom.lanai_sram_size));
	_GM_WARN (("(Perhaps the MCP was compiled with debugging turned on.)"
		   "\n"));
	goto abort_running;
      }

#if 0
#error Warning, limiting max_node_id to 400 for testing purposes
    if (max_node_id > 400)
      max_node_id = 400;	/* Temporary HACK to avoid IRIX problem */
#endif

    /* record the max_node_id */

    is->max_node_id = (gm_u16_t) max_node_id;
    GM_INFO (("Board %u supports %u remote nodes.\n",
	      is->id, (unsigned int) max_node_id));

  }

  /* Set the DMA_STATUS or BURST special register, and also set up the
     DMA chain pointers on the L5+. */

  if (gm_setup_DMA_engine (is) != GM_SUCCESS)
    goto abort_running;

  /* Set the VERSION register if the chip has one. */

  if (gm_is_l4 (is))
    {
      if (gm_set_VERSION (is) != GM_SUCCESS)
	goto abort_running;
    }

  /* Set the WINDOW register if the chip has one. */

  if (gm_is_l5 (is))
    {
      if (gm_set_WINDOW (is) != GM_SUCCESS)
	goto abort_running;
    }

  /* Setup the pointer to the lanai DMA page bitmap. */

  {
    GM_BITMAP_DECL ((*lanai_dma_bitmap), GM_MAX_HOST_DMA_PAGES);
      
    lanai_dma_bitmap
      = &((gm_lanai_globals_t *) is->lanai.globals)->debug.dma_page_bitmap;
#if GM_DMA_PAGE_BITMAP_COPY_NEEDED
    GM_BITMAP_COPY (is->dma_page_bitmap_copy, *lanai_dma_bitmap);
#endif
    is->dma_page_bitmap = lanai_dma_bitmap;
  }

  /* Allocate the page hash table. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Allocating the page hash table.\n"));
  status = gm_alloc_page_hash_table (is);
  if (status != GM_SUCCESS)
    {
      GM_WARN (("Could not allocate page hash table"));
      goto abort_running;
    }

  gm_write_lanai_global_u32 (is, bus_rate, (gm_u32_t) (is->bus_rate));
  gm_write_lanai_global_u32 (is, bus_width, (gm_u32_t) (is->bus_width));

  /* Tell the LANai the "ethernet" address of the board by
     initializing the mapper reply messages. */
  gm_write_lanai_global_byte_array
    (is, is->lanai.eeprom.lanai_board_id,
     mapper_state.config_reply.packet.address);
  gm_write_lanai_global_byte_array
    (is, is->lanai.eeprom.lanai_board_id,
     mapper_state.scout_reply.packet.address);

  /* Tell the LANai the hostname of this machine by initializing the
     mapper reply messages. */

  {
    char tempname[GM_MAX_HOST_NAME_LEN + 8];
#if !GM_ETHERBOOT
    /* loic: change the line above to make the code dependant on the
       OS if you are too lazy to provide gm_arch_gethostname (or in
       a hurry) */
    if (gm_arch_gethostname (tempname, GM_MAX_HOST_NAME_LEN) == GM_SUCCESS)
      {
	if (is->id > 0)
	  {
	    /* try to add id to end of hostname e.g. hostname:1
	       hostname:2 etc. */
	    int i;
	    for (i = 0; i < GM_MAX_HOST_NAME_LEN; i++)
	      {
		if (tempname[i] == 0)
		  {
		    break;
		  }
	      }
	    if (i < GM_MAX_HOST_NAME_LEN - 3)
	      {
		tempname[i] = ':';
		if (is->id < 10)
		  {
		    tempname[i + 1] = ('0' + is->id);
		  }
		else
		  {
 		    tempname[i + 1] = ('a' + (is->id - 10));
		  }
		tempname[i + 2] = 0;
	      }
	  }

	gm_strncpy_to_lanai_globals (is,
				     mapper_state.scout_reply.packet.hostname,
				     tempname);
      }
    else
#endif
      {
	gm_clear_lanai_global (is, mapper_state.scout_reply.packet.hostname);
      }
  }

  /* Tell the LANai where to find the various DMA regions in host
     memory.  The lanai will automatically initialize its ports to use
     these pointers until they are opened.  */

  {
    gm_dp_t r, s;

    r = gm_arch_dma_region_dma_addr (&is->page_hash.bogus_rdma_region);
    s = gm_arch_dma_region_dma_addr (&is->page_hash.bogus_sdma_region);
    gm_notice_dma_addr (is->dma_page_bitmap, r);
    gm_notice_dma_addr (is->dma_page_bitmap, s);
    gm_write_lanai_global_dp (is, page_hash.bogus_sdma_ptr, s);
    gm_write_lanai_global_dp (is, page_hash.bogus_rdma_ptr, r);

    GM_PRINT (GM_TRACE_LANAI_DMA, ("bogus_sdma_ptr = 0x%x%08x\n",
				   (gm_u32_t) s >> 31 >> 1, (gm_u32_t) s));
    _GM_PRINT (GM_TRACE_LANAI_DMA, ("bogus_rdma_ptr = 0x%x%08x\n",
				    (gm_u32_t) r >> 31 >> 1, (gm_u32_t) r));
  }

  GM_PRINT (GM_PRINT_LEVEL >= 2,
	    ("Telling the LANai the size of its SRAM is 0x%x\n",
	     is->lanai.eeprom.lanai_sram_size));

  /* Allocate the ethernet address table. */

  if (gm_alloc_addr_translation_table (is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not allocate ethernet address translation table.\n"));
      goto abort_with_page_hash;
    }

#if !GM_ENABLE_VM
  /* If the machine does not support virtual memory, tell the lanai how
     to construct a DMA address from a host virtual memory address. */

  gm_write_lanai_global_dp (is, no_VM.set,
			    gm_arch_no_VM_dma_bits_to_set (is));
  gm_write_lanai_global_dp (is, no_VM.clear,
			    gm_arch_no_VM_dma_bits_to_clear (is));
  
  /*
  gm_kton_dp (is, &is->lanai.globals->no_VM.set,
	      gm_arch_no_VM_dma_bits_to_set (is));
  gm_kton_dp (is, &is->lanai.globals->no_VM.clear,
	      gm_arch_no_VM_dma_bits_to_clear (is));
  */
#endif /* !GM_ENABLE_VM */


  /* Tell the LANai that initialization is done.  Also, mask
     all ISR bits that might cause a spurious interrupt. */

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("Reporting completed initialization to LANai.\n"));
  GM_STBAR ();
  if (gm_set_SMP (is, 0) != GM_SUCCESS)
    {
      GM_NOTE (("Could not set SMP.\n"));
      goto abort_with_page_hash;
    }
  {
    gm_u32_t smp;
    if (gm_get_SMP (is, &smp) != GM_SUCCESS)
      {
	GM_NOTE (("Could not get SMP.\n"));
	goto abort_with_page_hash;
      }
  }
  
  /* SAB */
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("set_EIMR = 0x%x\n", is->_set_EIMR));
  gm_set_EIMR (is, 0);
  GM_STBAR ();

  /****************
   * Wait for LANai to be completely initialized
   ****************/

  {
    gm_u32_t smp;
    gm_s32_t start_time, current_time;
    gm_u32_t initialized;
      
    gm_arch_spin(is,20000);
    if (gm_get_RTC (is, &start_time) != GM_SUCCESS)
      goto abort_with_page_hash;

    do
      {
	gm_arch_spin(is,10000);
	initialized = gm_read_lanai_global_u32 (is, initialized);
	if (gm_get_RTC (is, &current_time) != GM_SUCCESS)
	  goto abort_with_page_hash;
	/* Test for timeout.  This simple comparison works because RTC
	   is initialized to zero during reset */
	if (current_time - start_time > 20 * 1000 * 1000)
	  {
	    GM_NOTE (("lanai did not initialize in 10 seconds"
		      " (0x%x).\n",
		      gm_ntoh_u32 (*((gm_u32_n_t *) is->lanai.sram))));
	    goto abort_with_page_hash;
	  }
      }
    while (!initialized);
  }

  /****************
   * Read lanai-computed values
   ****************/

  /* report number of page hash cache entries */
  
  GM_INFO (("GM: Board %u page hash cache has %u bins.\n", is->id,
	    (((unsigned int)
	     gm_read_lanai_global_u32 (is, page_hash.cache.max_index)) + 1)));

  return GM_SUCCESS;

abort_with_page_hash:
  gm_disable_lanai (is);
  gm_free_page_hash_table (is);
abort_running:
  is->lanai.running = 0;
  gm_lanai_reset_on (is);
abort_with_set_SMP:
abort_with_nothing:
  return GM_FAILURE;
}

static void
gm_instance_unload_mcp (gm_instance_state_t * is)
{
  GM_CALLED ();

  is->lanai.running = 0;
  gm_lanai_reset_on (is);
  gm_free_addr_translation_table (is);
#if !GM_ETHERBOOT
  gm_free_page_hash_table (is);
#endif

  GM_RETURN_NOTHING ();
}

#if 0
void
gm_reset_everything (gm_instance_state_t * is)
{
  gm_lanai_reset_on (is);
  gm_ebus_reset_on (is);
  gm_board_reset_on (is);
  gm_arch_spin (is, 20000);
}
#endif

/* Initialize the interface, mapping all requisite data structures
   into system memory.  Also enables interrupts. */

gm_status_t
gm_instance_init (gm_instance_state_t * is, gm_u32_t unit,
		  enum gm_bus_type type)
{
#define GM_DEBUG_CACHE_LINE_SIZE 1

  gm_status_t status;

  GM_CALLED ();

  if (gm_init () != GM_SUCCESS)
    {
      GM_NOTE (("gm_instance_init: gm_init() failed. Could not init "
		"package.\n"));
      goto abort_with_nothing;
    }


  if (gm_init_instance_support () != GM_SUCCESS)
    {
      GM_NOTE (("Could not initialize instance_state package.\n"));
      goto abort_with_nothing;
    }

  /* Put the instance in the instance hash table to allow id->instance
     translations to be performed. */


#if !GM_ETHERBOOT
  if (gm_hash_insert (gm_instance_hash, &unit, is) != GM_SUCCESS)
    {
      GM_NOTE (("Could not insert state in hash table.\n"));
      goto abort_with_init;
    }
#endif

  /* Initialize the instance state structure. */

  is->id = unit;
#if GM_DMA_PAGE_BITMAP_COPY_NEEDED
  is->dma_page_bitmap = &is->dma_page_bitmap_copy;
  gm_bzero (is->dma_page_bitmap_copy, sizeof (is->dma_page_bitmap_copy));
#else
  is->dma_page_bitmap = 0;
#endif
  /* is->flags = 0; */
  /* is->lanai.* cleared */
  /* is->pause_rqst = 0 */
  /* is->pause_ack = 0 */

  gm_arch_sync_init (&is->pause_sync, is);
  gm_arch_sync_init (&is->page_hash.sync, is);
  is->lanai.eeprom.bus_type = type;	/* HACK */

  /* If this is a pci device, read the revision id and cache line size. */
#if GM_SUPPORT_PCI
  if (GM_SUPPORT_PCI && is->lanai.eeprom.bus_type == GM_MYRINET_BUS_PCI)
    {
      if (gm_read_pci_config_revision_id (is, &is->ifc.pci.config.Revision_ID)
	  != GM_SUCCESS)
	{
	  GM_WARN (("Could not read PCI board revision ID.\n"));
	  goto abort_with_syncs;
	}
      if (is->ifc.pci.config.Revision_ID == 3)
	{
	  GM_INFO (("This is a pci REV 3 board\n"));
	}
      if (gm_read_pci_cache_line_size
	  (is, &is->ifc.pci.config.Cache_Line_Size) != GM_SUCCESS)
	{
	  GM_WARN (("Could not read PCI cache line size.\n"));
	  goto abort_with_syncs;
	}
      is->cache_line_size = 4 * is->ifc.pci.config.Cache_Line_Size;
      if (!GM_POWER_OF_TWO (is->cache_line_size))
	{
	  GM_NOTE (("Cache line size reported by BIOS in PCI "
		    "configuration register (%d) is not a power of 2.  "
		    "Interface disabled.\n", is->cache_line_size));
	  goto abort_with_syncs;
	}
      GM_PRINT (GM_DEBUG_CACHE_LINE_SIZE,
                ("Cache line size reported in Myrinet card is %d\n",
                 is->cache_line_size));

      /* gm_set_64_bit_flag() depends on is->ifc.pci.config.Revision_ID */
      if (gm_set_64_bit_flag(is) != GM_SUCCESS)
        {
          GM_WARN (("Could not determine board %d DMA address size.\n", unit));
          goto abort_with_syncs;
        }
    }
#endif /* GM_SUPPORT_PCI */
  /* is->ifc.* initialized by caller. */
  if (gm_instance_init_functions (is) != GM_SUCCESS)
    {
      GM_WARN (("Could not initialize instance functions.\n"));
      goto abort_with_syncs;
    }


  /* Now it is safe to touch the interface. */

  /* Map the LANai interface to system memory. This also copies the
     eeprom into the instance state. */

  status = gm_map_io_spaces (is);
  if (status != GM_SUCCESS)
    {
      GM_WARN (("Can't map IO memory to system memory\n"));
      goto abort_with_syncs;
    }

  /* Test and clear the SRAM */

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("Testing, clearing LANai SRAM from %p to %p.\n", is->lanai.sram,
	     (char *) is->lanai.sram + is->lanai.eeprom.lanai_sram_size));

  status = gm_test_and_clear_sram (is, is->lanai.sram,
				   is->lanai.eeprom.lanai_sram_size);
  if (status != GM_SUCCESS)
    {
      GM_WARN (("sram and register tests failed.\n"));
      goto abort_with_io_spaces;
    }

  /* Create hash table to translate port numbers to port state structure
     pointers. */

#if !GM_ETHERBOOT
  is->port_hash = gm_create_hash (gm_hash_compare_ints, gm_hash_hash_int,
				  sizeof (int), 0, 1, 0);
#endif
  if (!is->port_hash)
    {
      GM_NOTE (("could not allocate port hash for instance.\n"));
      goto abort_with_io_spaces;
    }

  /* Bob,

     Should this really be here? - SAB CSPI

     In gm_map_io_spaces you turn on the lanai chip and ebus for some work.
     Afterwards, you disable ebus for some systems while you leve it on for
     other systems.  If you consistently leave in on in gm_map_io_spaces, then
     you shouldn't need it here.

     We had to leave it on in gm_map_io_spaces so that gm_test_and_clear_sram
     could run. */


  /* Copy the myrinet control program into LANai SRAM. */

  /* CSPI: This reset avoids random hang from boot kernel to loadable
     kernel I'd suspect a soft reset doesn't reset the LANai - that
     could be a problem.  Should we really do lanai_reset_on,
     ebus_reset_on, ebus_reset_off?? */
  gm_lanai_reset_on (is);
  gm_ebus_reset_on (is);
  gm_arch_spin (is, 20000);
  gm_ebus_reset_off (is);
  gm_arch_spin (is, 20000);

  /* Load the mcp into the lanai */

  status = gm_instance_load_mcp (is);

  if (status != GM_SUCCESS)
    {
      GM_NOTE (("could not load control program.\n"));
      goto abort_with_port_hash;
    }

  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_port_hash:
#if !GM_ETHERBOOT
  gm_destroy_hash (is->port_hash);
#endif
abort_with_io_spaces:
  gm_unmap_io_spaces (is);
abort_with_syncs:
  gm_arch_sync_destroy (&is->page_hash.sync);
  gm_arch_sync_destroy (&is->pause_sync);
abort_with_init:
  gm_finalize_instance_support ();
abort_with_nothing:
  GM_RETURN_STATUS (GM_FAILURE);
}

void
gm_instance_finalize (gm_instance_state_t * is)
{
  GM_CALLED ();
  gm_instance_unload_mcp (is);
#if !GM_ETHERBOOT
  gm_destroy_hash (is->port_hash);
#endif
  gm_unmap_io_spaces (is);
  gm_arch_sync_destroy (&is->page_hash.sync);
  gm_arch_sync_destroy (&is->pause_sync);
  gm_finalize_instance_support ();
  gm_finalize ();
  GM_RETURN_NOTHING ();
}

#if GM_DEBUG_INSTANCE_STATE
void
gm_dump_instance (gm_instance_state_t * is)
{
  GM_CALLED ();
  GM_PARAMETER_MAY_BE_UNUSED (is);     /* if not GM_DEBUG */

  GM_PRINT (GM_PRINT_LEVEL >= 0, ("* (gm_instance_state_t *) %p = {\n", is));
#define FOO(name,tmpl) _GM_PRINT (GM_PRINT_LEVEL >= 0,			\
				 ("  "#name" = "#tmpl";\n", is->name))
#define FOO2(name) _GM_PRINT (GM_PRINT_LEVEL >= 0, ("  "#name" = ?;\n"))
  FOO (id, %d);
  FOO (flags, %d);
  FOO2 (lanai);
  FOO (lanai.globals, %p);
  FOO (board_base, %p);
  FOO (board_span, %d);
  FOO2 (pause_rqst);
  FOO2 (pause_ack);
  FOO2 (pause_sync);
  FOO2 (daemon_rqst_q);
  FOO2 (ifc);
  FOO (_set_EIMR, %p);
  FOO (set_ISR, %p);
  FOO (get_EIMR, %p);
  FOO (get_ISR, %p);
  FOO2 (ethernet);
  FOO2 (name_table);
  FOO (max_node_id, %d);
  FOO (port_hash, %p);
  FOO2 (arch);
  FOO (clone_minor, %d);
  FOO (privileged_clone_minor, %d);
#undef FOO
#undef FOO2
  _GM_PRINT (GM_PRINT_LEVEL >= 0, ("};\n"));

  GM_RETURN_NOTHING ();
}
#endif

#if !GM_ETHERBOOT
gm_instance_state_t *
gm_instance_for_id (unsigned int unit)
{
  return (gm_instance_state_t *) gm_hash_find (gm_instance_hash, &unit);
}
#endif

#define GM_DEBUG_UPDATE_CLOCKVAL_REV3 0

static gm_status_t
gm_update_clockval_rev3 (gm_instance_state_t * is)
{
#if GM_SUPPORT_L7 | GM_SUPPORT_L8 | GM_SUPPORT_L9
  unsigned int bus_rate = 0;
  unsigned int clockval = 0;
  unsigned int max_lanai_speed = 0;
  unsigned int multiplier = 0;

  gm_assert (is);

  GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3,
	    ("gm_update_clockval_rev3: called\n"));

  /* Disable PCI bus master DMA */

  GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3, ("Disable PCI bus master DMA.\n"));
  if (gm_disable_PCI_config_command_bit (is, GM_PCI_COMMAND_MASTER)
      != GM_SUCCESS)
    {
      return (GM_FAILURE);
    }

  gm_lanai_reset_on (is);
  gm_ebus_reset_on (is);
  gm_ebus_reset_off (is);
  gm_arch_spin (is, 25000);

  if (gm_is_l9(is)) 
    {
      gm_u32_t cpuc,rtc1,rtc2,rtc;
      rtc1 = gm_read_lanai_special_reg_s32 (is, l9.RTC);
      cpuc = gm_read_lanai_special_reg_u32 (is, l9.CPUC);
      rtc2 = gm_read_lanai_special_reg_s32 (is, l9.RTC);

      rtc = (rtc1 + rtc2) / 2;
      GM_PRINT(GM_DEBUG_UPDATE_CLOCKVAL_REV3,(" rtc = %d  cpuc = %d\n",rtc,cpuc));
      if (rtc) {
	GM_PRINT(1,("Using 1x clockval - measured LANai9 rate as %d MHz\n",
		    (cpuc*2)/rtc));
	bus_rate = (cpuc*2)/rtc;
      }
    }

  if (!bus_rate) {
    GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3,
	      ("Inflating 'lanai_rate' control program.\n"));
    {
      /* Load a compressed MCP - see zlib.h for details */

      z_stream zs;
      int rv, err;

      zs.zalloc = gm_zlib_calloc;
      zs.zfree = gm_zlib_free;
      zs.opaque = 0;

      rv = inflateInit (&zs);
      if (rv != Z_OK)
	{
	  GM_NOTE (("fatal error in inflateInit(%d)\n", rv));
	  return (GM_FAILURE);
	}
      zs.next_in = (unsigned char *) gm_control_program_lanai_rate;
      zs.avail_in = gm_control_program_lanai_rate_length;
      zs.next_out = (unsigned char *) is->lanai.sram;
      zs.avail_out = is->lanai.eeprom.lanai_sram_size;

      if (((err = inflate (&zs, Z_FINISH)) != Z_STREAM_END)
	  || (zs.avail_in != 0) || (zs.avail_out == 0))
	{
	  GM_NOTE (("fatal error in 'lanai_rate' inflate err=%d?\n", err));
	  return (GM_FAILURE);
	}

      if (inflateEnd (&zs) != Z_OK)
	{
	  GM_NOTE (("fatal error in 'lanai_rate' inflateEnd\n"));
	  return (GM_FAILURE);
	}

      GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3,
		("Inflated 'lanai_rate' mcp from %ld to %ld bytes.\n",
		 gm_control_program_lanai_rate_length, zs.total_out));
    }

    GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3,
	      ("Taking the LANai out of reset\n"));
    if (gm_lanai_reset_off (is) != GM_SUCCESS)
      {
	GM_WARN (("Could not take LANai out of reset.\n"));
	return (GM_FAILURE);
      }

    {
      int loop_count = 0;
      gm_u32_n_t *iptr = (gm_u32_n_t *) is->lanai.sram;

      /* gm_arch_timed_sleep(&is->pause_sync,1); */
      gm_arch_spin (is, 50000);
      while ((loop_count++ < 20) && gm_ntoh_u32 (iptr[0]) != 0xabababab)
	{
	  /* wait */
	  /* gm_arch_timed_sleep(&is->pause_sync,1); */
	  gm_arch_spin (is, 100000);
	}

      if (gm_ntoh_u32 (iptr[0]) != 0xabababab)
	{
	  GM_WARN (("LANai 'rate' program never completed\n"));
	  _GM_WARN (("iptrs  0x%x 0x%x 0x%x\n",
		     gm_ntoh_u32 (iptr[0]), gm_ntoh_u32 (iptr[1]),
		     gm_ntoh_u32 (iptr[2])));
	  return (GM_FAILURE);
	}

      bus_rate = gm_ntoh_u32 (iptr[1]);
      GM_PRINT (GM_PRINT_LEVEL >= 0,
		("bus timing took %d iterations\n", loop_count));
    }

  }


  if (bus_rate < 20)
    {
      GM_INFO (("bus_rate (%d) is bad - setting to 66\n", bus_rate));
      bus_rate = 66;
    }

  GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3,
	    ("Putting LANai && Ebus back in reset\n"));
  gm_lanai_reset_on (is);
  gm_ebus_reset_on (is);
  gm_arch_spin (is, 20000);

  GM_INFO (("gm[%d]: LANai speed is %d MHz\n", is->id, bus_rate));
  if (gm_is_l9(is)) 
    {
      GM_INFO (("gm[%d]: PCI slot speed is %d MHz\n", is->id, bus_rate/2));
      is->bus_rate = bus_rate/2;
    }
  else 
    {
      GM_INFO (("gm[%d]: PCI slot speed is %d MHz\n", is->id, bus_rate));
      is->bus_rate = bus_rate;
    }

  clockval = is->lanai.eeprom.lanai_clockval;
  clockval = clockval & (unsigned int) 0xFFFFFF8F;

  max_lanai_speed = is->lanai.eeprom.max_lanai_speed;
  if ((max_lanai_speed == 0) || (max_lanai_speed == (gm_u16_t) 0xFFFF))
    {
      max_lanai_speed = 67;
    }

  GM_INFO (("LANai rate is %d MHz  lanai_max is %d MHz\n",
		bus_rate, max_lanai_speed));

  /* use 2x here to allow 1.5 or 2.5 multiplier */
  multiplier = (2 * (max_lanai_speed+1)) / bus_rate;
  switch (multiplier)
    {
    case 1:
    case 2:
      /* 1x clockval */
      clockval = clockval | 0x80;
      break;

    case 3:
      /* 1.5x clockval */
      clockval = clockval | 0x90;
      break;

    case 4:
      /* 2x clockval */
      clockval = clockval | 0xA0;
      break;

    case 5:
      /* 2.5x clockval */
      clockval = clockval | 0xB0;
      break;

    case 6:
      /* 3x clockval */
      clockval = clockval | 0xC0;
      break;

    case 0:
    default:
      clockval = clockval | 0x80;
      GM_NOTE (("**** multiplier is bad (%d) - using 1x clockval ****\n",
		multiplier));
      break;
    }

  GM_INFO (("multiplier = %d/2  clockval = 0x%x\n", multiplier, clockval));

  gm_write_lanai_special_reg_u32 (is, l7.CLOCK, clockval);
  GM_STBAR ();
  gm_arch_spin (is, 100000);
  GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3, ("Ebus reset off.\n"));
  gm_ebus_reset_off (is);
  gm_arch_spin (is, 20000);

  GM_PRINT (GM_DEBUG_UPDATE_CLOCKVAL_REV3, ("Enable PCI bus master DMA.\n"));
  if (gm_enable_PCI_config_command_bit (is, GM_PCI_COMMAND_MASTER)
      != GM_SUCCESS)
    {
      GM_NOTE (("Enable PCI bus master DMA failed"
		" after 'lanai_rate' test\n"));
      return (GM_FAILURE);
    }
#endif /* SUPPORT L7 | L8 | L9 */
  return (GM_SUCCESS);
}

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
