/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

#include "gm_debug.h"
#include "gm_ether.h"
#include "gm_send_queue.h"
#include "gm_trace.h"

#define GM_DEBUG_ETHER 0

/* ethernet_send and ethernet_broadcast have a lot of similar code
   the common part is factorized here in gm_ether_cmn() */
static void
gm_ether_cmn (gm_port_t * p,
	      gm_u32_t segment_cnt,
	      gm_ethernet_segment_descriptor_t segment[],
	      int evt_type, unsigned target_node_id)
{
  volatile struct gm_ethernet_send_event *ese;
  struct gm_send_queue_slot_token *sqst;
  struct gm_send_queue_slot *sqs;
  unsigned int i;
  struct gm_ethernet_send_event batch_write;

  /* BAD: The following two blocks could be combined and pipelined. */

  /* ethernet does not use completion callbacks; it uses interrupts,
     hence the null callback. */


  ese = GM_SEND_QUEUE_SLOT (p, 0, 0, ethernet);
  sqs = ((struct gm_send_queue_slot *) (ese + 1) - 1);

  /* HACK: recycle the send token, emulating the old ethernet
     behavior.  This is OK only because the ethernet only ever puts
     sends in the send queue and the sends complete in order. */

  sqst = &p->send_queue_slot_token[sqs - &p->send_queue_start[0]];
  sqst->next = 0;
  p->last_free_send_queue_slot_token->next = sqst;
  p->last_free_send_queue_slot_token = sqst;

  /* Copy the gather list */
  /* we copy each gather (ptr,len) across the IO bus */
  /* for cnt large, we might want to batch them.     */

  gm_assert (segment_cnt);
  gm_assert (segment_cnt <= GM_MAX_ETHERNET_GATHER_CNT);
  for (i = 0; i < segment_cnt; i++)
    {
#if !GM_EMULATE_BYTE_DMAS
      gm_assert (GM_DMA_ALIGNED (gm_ntoh_dp (segment[i].ptr)));
      gm_assert (GM_DMA_ALIGNED (gm_ntoh_u32 (segment[i].len)));
#endif /* !GM_EMULATE_BYTE_DMAS */
      gm_assert (gm_ntoh_dp (segment[i].ptr));
      gm_assert (gm_ntoh_u32 (segment[i].len));


      GM_PRINT (GM_DEBUG_ETHER,
		("ether_send:  segment[%d](%p): ptr = 0x%x  len = 0x%x\n", i,
		 ese, segment[i].ptr, segment[i].len));

      if (gm_ntoh_u32 (segment[i].len) > GM_ETHERNET_MTU)
	{
	  GM_WARN (("ether_send: ***********  Bad length = %d\n",
		    segment[i].len));
	}

      GM_COPY_TO_IO_SPACE (ese->gather_segment[i], segment[i]);
    }

#if !GM_STRUCT_WRITE_COMBINING
#error  GM_STRUCT_WRITE_COMBINING must be defined
#endif

  /* now write the last word which happens to contain all the extra info including type */
  batch_write.target_node_id = gm_hton_u16 ((gm_u16_t) target_node_id);
  batch_write.gather_cnt = gm_hton_u8 ((gm_u8_t) segment_cnt);
  batch_write.type = gm_hton_u8 ((gm_u8_t) evt_type);
  GM_STBAR ();
  *(gm_u32_t *) & ese->target_node_id =
    *(gm_u32_t *) & batch_write.target_node_id;
  GM_FLUSH_SEND_EVENT (p, ese);
  GM_PRINT (GM_DEBUG_ETHER,
	    ("gm_ethernet_send target_node:%d\n", target_node_id));
}

void
gm_ethernet_send (gm_port_t * p,
		  gm_u32_t segment_cnt,
		  gm_ethernet_segment_descriptor_t segment[],
		  gm_u8_t * ethernet_addr, int prepend_type)
{
  /* Convert the ethernet address (unique ID) to a GM ID. */
  int type;
  unsigned target_node_id;

  if (gm_unique_id_to_node_id (p, (char *) ethernet_addr, &target_node_id)
      != GM_SUCCESS)
    {
      GM_PRINT (GM_PRINT_LEVEL >= GM_DEBUG_ETHER,
		("cannot find target id for ether %x:%x:%x:%x:%x:%x\n",
		 ethernet_addr[0], ethernet_addr[1], ethernet_addr[2],
		 ethernet_addr[3], ethernet_addr[4], ethernet_addr[5]));
      target_node_id = 0;
    }
  type = (prepend_type
	  ? GM_ETHERNET_MARK_AND_SEND_EVENT : GM_ETHERNET_SEND_EVENT);
  gm_ether_cmn (p, segment_cnt, segment, type, target_node_id);
}

void
gm_ethernet_broadcast (gm_port_t * p,
		       gm_u32_t segment_cnt,
		       gm_ethernet_segment_descriptor_t segment[],
		       int prepend_type)
{
  int type;

  type = (prepend_type
	  ? GM_ETHERNET_MARK_AND_BROADCAST_EVENT
	  : GM_ETHERNET_BROADCAST_EVENT);
  gm_ether_cmn (p, segment_cnt, segment, type, 0);
}

void
gm_ethernet_set_recv_intr_callback (gm_port_t * p,
				    void (*callback) (void *, unsigned int,
						      gm_u16_t csum),
				    void *context)
{
  gm_assert (p);
  gm_assert (p->kernel_port_state);
  gm_assert (p->kernel_port_state->instance);

  p->kernel_port_state->instance->ethernet_recv_intr_callback = callback;
  p->kernel_port_state->instance->ethernet_recv_intr_context = context;
}

void
gm_ethernet_set_sent_intr_callback (gm_port_t * p,
				    void (*callback) (void *), void *context)
{
  gm_assert (p);
  gm_assert (p->kernel_port_state);
  gm_assert (p->kernel_port_state->instance);

  p->kernel_port_state->instance->ethernet_sent_intr_callback = callback;
  p->kernel_port_state->instance->ethernet_sent_intr_context = context;
}

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
