#ifndef _smpi_h
#define _smpi_h

#include "gm.h"
#include "gm_stbar.h"
#include "packets.h"

/* MAXIMUM OF LOCAL PROCESSES */
#define SMPI_MAX_NUMLOCALNODES 4
/* the maximum number of nodes in the cluster. pretty scalable. */
#define SMPI_MAX_NUMNODES 4096

/* FIXME : we can use GM constants */
#ifdef __alpha__
#define SMPI_CACHE_LINE_SIZE 64
#define SMPI_ALIGN(a) ((a + SMPI_CACHE_LINE_SIZE + 7) & 0xFFFFFFFFFFFFFFF8)
/* Threshold between the eager protocol and the Rendez-vous protocol */
#define SMPI_PKT_MAX_LOCALDATA_SIZE 2048
#define SMPI_PKT_LONG_MAX_LOCALDATA_SIZE 8192
#else
#define SMPI_CACHE_LINE_SIZE 64
#define SMPI_ALIGN(a) (a + SMPI_CACHE_LINE_SIZE)
/* Threshold between the eager protocol and the Rendez-vous protocol */
#define SMPI_PKT_MAX_LOCALDATA_SIZE 1024
#define SMPI_PKT_LONG_MAX_LOCALDATA_SIZE 4096
#endif

#define SMPI_MAX_INT ((unsigned int)(-1))

/* There's two behaviour dependent on the message size :
   0 <= x <= SMPI_PKT_MAX_LOCALDATA_SIZE : the sender writes 
   the packet in the shared memory and the receiver reads it. 
   There's two memory copies, but the latency is very small.
   (it's the common behaviour for OS without directcopy).
   
   x > SMPI_PKT_MAX_LOCALDATA_SIZE : the sender send a packet 
   DO_GET with the address and the length of the data buffer 
   on the sender process memory space. When the receive is 
   posted (Rendez-vous), the receiver process call GM to DIRECTCOPY_GET 
   the data from the sender user space. There's only one memory copy 
   by the kernel.
*/

#define SMPI_LENGTH_QUEUE (1 << 18) /* 256 KB */

/* Macros for flow control and rqueues management */
#define SMPI_TOTALIN(sender,receiver) smpi_shmem->rqueues_params[sender].params[receiver].msgs_total_in
#define SMPI_TOTALOUT(sender,receiver) smpi_shmem->rqueues_flow_out[receiver][sender].msgs_total_out
#define SMPI_CURRENT(sender,receiver) smpi_shmem->rqueues_params[receiver].params[sender].current
#define SMPI_NEXT(sender,receiver) smpi_shmem->rqueues_params[sender].params[receiver].next
#define SMPI_FIRST(sender,receiver) smpi_shmem->rqueues_limits[receiver][sender].first
#define SMPI_LAST(sender,receiver) smpi_shmem->rqueues_limits[receiver][sender].last


/* packets definition */
#define SMPI_PKT_BASIC \
  unsigned int mode;             /* Contains MPID_Pkt_t */             \
  unsigned int context_id;      /* Context_id */                       \
  unsigned int lrank;           /* Local rank in sending context */    \
  unsigned int tag;             /* tag is full sizeof(int) */          \
  unsigned int len;             /* Length of DATA */
    
/* This is the minimal message packet */
typedef struct {
  SMPI_PKT_BASIC
} SMPI_PKT_HEAD_T;

/* Short messages are sent eagerly (unless Ssend) */
typedef struct { 
  SMPI_PKT_BASIC    
  char buffer[SMPI_PKT_MAX_LOCALDATA_SIZE];
} SMPI_PKT_SHORT_T;

/* Long messages are sent rendez-vous if not directcopy */
typedef struct { 
  SMPI_PKT_BASIC
  void * send_id;
  void * recv_id;
  char buffer[SMPI_PKT_LONG_MAX_LOCALDATA_SIZE];
} SMPI_PKT_CONT_GET_T;

/* GET_request packet */
typedef struct {
  SMPI_PKT_BASIC
  void * send_id;
  void * address;      /* Location of data ON SENDER */
} SMPI_PKT_GET_T;

/* RNDV_request packet */
typedef struct {
  SMPI_PKT_BASIC
  void * send_id;
  void * recv_id;
} SMPI_PKT_RNDV_T;

/* Cancel packet */
typedef struct {
  SMPI_PKT_BASIC
  void * send_id;
  int cancel;        /* set to 1 if msg was cancelled - 
			0 otherwise */
} SMPI_PKT_ANTI_SEND_T;

/* all the differents types of packets */
typedef union _SMPI_PKT_T {
  SMPI_PKT_HEAD_T          head;
  SMPI_PKT_SHORT_T         short_pkt;
  SMPI_PKT_CONT_GET_T      cont_pkt;
  SMPI_PKT_GET_T           get_pkt;
  SMPI_PKT_RNDV_T          rndv_pkt;
  SMPI_PKT_ANTI_SEND_T     cancel_pkt;
  char                     pad[SMPI_CACHE_LINE_SIZE];
} SMPI_PKT_T;

/* send_requests fifo, to stack the send in case of starvation of buffer
   in the shared area memory */
struct smpi_send_fifo_req {
  void *data;
  struct _MPIR_SHANDLE *shandle;
  struct smpi_send_fifo_req *next;
  int len;
  int grank;
  int is_data;
};

/* management informations */
struct smpi_var {
  void * mmap_ptr;
  struct smpi_send_fifo_req *send_fifo_head;
  unsigned int my_local_id;
  unsigned int num_local_nodes;
  unsigned int local_nodes[SMPI_MAX_NUMNODES];
  int available_queue_length;
  int fd;
};

/* the shared area itself */
struct shared_mem {
  volatile int pid[SMPI_MAX_NUMLOCALNODES]; /* use for initial synchro */
  volatile int board_id[SMPI_MAX_NUMLOCALNODES];
  volatile int port_id[SMPI_MAX_NUMLOCALNODES];
  char pad1[SMPI_CACHE_LINE_SIZE];

  /* receive queues descriptors */
  volatile struct {
    volatile struct {
      volatile unsigned int current;
      volatile unsigned int next;
      volatile unsigned int msgs_total_in;
    } params[SMPI_MAX_NUMLOCALNODES];
    char pad[SMPI_CACHE_LINE_SIZE];
  } rqueues_params[SMPI_MAX_NUMLOCALNODES];

  /* rqueues flow control */
  volatile struct {
    volatile unsigned int msgs_total_out;
    char pad[SMPI_CACHE_LINE_SIZE-4];
  } rqueues_flow_out[SMPI_MAX_NUMLOCALNODES][SMPI_MAX_NUMLOCALNODES];
  
  volatile struct {
    volatile unsigned int first;
    volatile unsigned int last;
  } rqueues_limits[SMPI_MAX_NUMLOCALNODES][SMPI_MAX_NUMLOCALNODES];
  int pad2[SMPI_CACHE_LINE_SIZE];
  
  /* the receives queues */
  volatile char pool;
};

extern struct smpi_var smpi;
extern struct shared_mem * smpi_shmem;
extern int errno;


MPID_Device *MPID_SMP_InitMsgPass( int * argc, char ***argv, int short_len, int long_len);
void smpi_queue_send(void *data, MPIR_SHANDLE *shandle, int len, int grank, int prepend);
void smpi_post_send_bufferinplace(void * buf, int len ,int src_lrank, int tag, int context_id, int dest, MPIR_SHANDLE *shandle);
void smpi_post_send_queued(void *data, MPIR_SHANDLE *shandle, int len, int dest);
int deregister_smp_buffer(MPIR_SHANDLE *shandle);
void smpi_do_get(int from, void * source, void * target, unsigned int length);
void smpi_post_send_done_get(int destination, void * send_id);
void smpi_post_send_rndv(void * buf, int len ,int src_lrank, int tag, int context_id, int destination, MPIR_SHANDLE  *shandle);
void smpi_post_send_ok_to_send(int destination, MPIR_RHANDLE * rhandle);
int smpi_net_lookup(MPID_Device *dev,int blocking);
int MPID_SMP_Check_incoming(MPID_Device *dev,MPID_BLOCKING_TYPE blocking);
void smpi_init(void);
void smpi_finish(void);
void smpi_complete_send(unsigned int, unsigned int, unsigned int);
void smpi_complete_recv(unsigned int, unsigned int, unsigned int);
unsigned int smpi_able_to_send(int dest, int len);

MPID_Protocol *MPID_SMP_Short_setup(void);
MPID_Protocol *MPID_SMP_Rndv_setup(void);

#if defined(gm_always_assert)
#define smpi_always_assert gm_always_assert
#define smpi_assert gm_assert
#else
#define smpi_always_assert gm_assert
#define smpi_assert gm_assert_p
#endif

#endif /* smpi_h */
