#include <string.h>


#include "gmpi.h"

/* set this to 3 to always print and do regcache asserts, 1 enables clear, 2 [de]registers, 3 everything */
#define DEBUG_REGCACHE 0

/* not important to conform to the hardware, that just the unit we use for registration 
 but it should be smaller than the hardware page size, to avoid the possibility of trying to register invalid locations
*/

#define GMPI_PAGEDOWN(p) ((p)&~(GMPI_PAGE_SIZE-1))
#define GMPI_PAGEUP(p) GMPI_PAGEDOWN((p)+GMPI_PAGE_SIZE-1)

#define GM_HAS_WEAK_REGISTER 0

#if !GM_HAS_WEAK_REGISTER
#define gm_weak_register_memory gm_register_memory
#define gm_weak_deregister_memory gm_deregister_memory
#endif


#ifndef GM_NO_MEM_REG_CACHE

/* loic: maybe I should forget about sparing a few cycle, but ... */
#if GM_CPU_alpha +0
#define GMPI_PAGE_SIZE 8192
#elif GM_CPU_x86 +0
#define GMPI_PAGE_SIZE 4096
#else
#define GMPI_PAGE_SIZE GM_PAGE_LEN
#endif

#define GMPI_ZONE_MASK (GMPI_PAGE_SIZE*GMPI_ZONE_NPAGES-1)
#define GMPI_ZONE_SIZE (GMPI_PAGE_SIZE*GMPI_ZONE_NPAGES)

#define PREZONE 20

static struct gm_hash *gmpi_zone_hash;
static int lockable_pages;

void gmpi_dump_regcache(void);

static struct gmpi_zone * alloc_zone(void)
{
  struct gmpi_zone *z;
  if (gmpi.arch.free_intervals) {
    z = gmpi.arch.free_intervals;
    gmpi.arch.free_intervals = z->next;
    gm_bzero(z,sizeof(*z));
  } else {
    z = gm_malloc(sizeof(*z));
    gm_bzero(z,sizeof(*z));
  }
  return z;
}

static void free_zone(struct gmpi_zone *z)
{
  gm_bzero(z,sizeof(*z));
  z->next = gmpi.arch.free_intervals;
  gmpi.arch.free_intervals = z;
}

void gmpi_gc_regcache(void)
{
  struct gmpi_zone *z = gmpi.arch.intervals;
  gm_status_t rv;

  gm_always_assert(0);
  while (z) {
    int i;
    for (i=0;i<GMPI_ZONE_NPAGES;) {
      int j;
      if (z->status[i].registered && !z->status[i].refcount) {
	z->status[i].registered = 0;
	for (j=i+1;j<GMPI_ZONE_NPAGES;j++) {
	  if (!z->status[j].registered || z->status[j].refcount)
	    break;
	  z->status[j].registered = 0;
	}
	GM_PRINT(0,("%d from 0x%lx length 0x%lx, registered=%d  refcount=%d\n",
		    getpid(), z->start+GMPI_PAGE_SIZE*i,(j-i)*GMPI_PAGE_SIZE,
		    z->status[i].registered,z->status[i].refcount));
	rv = gm_weak_deregister_memory(gmpi.port,
		(char*)z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE);
	if (rv != GM_SUCCESS) {
		gm_perror("gm_deregister_memory failed: ",rv);
		GM_PRINT(0,("ptr=%p  len=%d\n",
			(char*)z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE));
		gmpi_dump_regcache();
		gm_abort();
	}
	lockable_pages += j - i;
	i = j;
      }
    }
    z = z->next;
  }
}

void gmpi_dump_regcache(void)
{
  struct gmpi_zone *z = gmpi.arch.intervals;

  gm_u32_t my_pid = getpid();
 GM_PRINT(0,("%d vvvvvv  dump_reg_cache    vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv\n",my_pid));
  while (z) {
    int i;
    for (i=0;i<GMPI_ZONE_NPAGES;) {
      int j;
      for (j=i+1;j<GMPI_ZONE_NPAGES;j++) {
	if (z->status[j].registered != z->status[i].registered ||
	    z->status[j].refcount != z->status[i].refcount)
	  break;
      }
      GM_PRINT(0,("%d from 0x%lx length 0x%lx, registered=%d  refcount=%d\n",
		  my_pid, z->start+GMPI_PAGE_SIZE*i,(j-i)*GMPI_PAGE_SIZE,
		  z->status[i].registered,z->status[i].refcount));
      i = j;
    }
    z = z->next;
  }
 GM_PRINT(0,("%d ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n",my_pid));
}

/* inc == -2 is special value to mean deregister */
/* inc == 2 is special value to mean register (zone creation) */
static void add_refcount(struct gmpi_zone *z,unsigned long start,unsigned long limit,int inc)
{
  int s,e,i,rv;
  int retval;
  int my_pid;

#if DEBUG_REGCACHE >= 3
  my_pid = getpid();
  GM_PRINT(0,("%d add_refcount(%p, 0x%lx, 0x%lx, %d) z->start = 0x%lx\n",my_pid,z,start,limit,inc,z->start));
#endif
  
  gm_assert(start >= z->start && limit <= z->start + GMPI_ZONE_SIZE);

  s = (start - z->start) / GMPI_PAGE_SIZE;
  e = (limit - z->start) / GMPI_PAGE_SIZE;
  gm_assert(e <= GMPI_ZONE_NPAGES);
  for (i=s;i<e;i++) {
    int j;
    /* first case: we need to deregister the next interval */
    if (inc == -2 && z->status[i].registered) {
      for (j=i;j<e;j++) {
	if (!z->status[j].registered)
	  break;
	if (z->status[j].refcount)
	  {
	    /* we try to clear a area which is busy, this should
	       never occur (in the past that was possible when intercepting
	       the malloc API */
	    GM_PRINT(0,("Unexpected state for clear_interval:"
			"zone=0x%lx:s=0x%x,e=0x%x,i=0x%x,j=0x%x,registered[j]=%d,refcount[j]=%d\n",
			z->start,s,e,i,j,
			z->status[j].registered,
			z->status[j].refcount));
	    gmpi_dump_regcache();
	    gm_abort();
	  }
	z->status[j].registered = 0;
      }
      if (j != i){
#if DEBUG_REGCACHE >= 1
        GM_PRINT(0,("%d preparing to deregister i=%d j=%d - here's the zone dump\n",
		my_pid,i,j));
	gmpi_dump_regcache();
#endif

	rv = gm_weak_deregister_memory(gmpi.port,
		(char*)z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE);
	if (rv != GM_SUCCESS) {
		gm_perror("gm_deregister_memory failed: ",rv);
		GM_PRINT(0,("ptr=%p  len=%d\n",
			(char*)z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE));
		gmpi_dump_regcache();
		gm_abort();
	}
	lockable_pages += j - i;
      }
      i = j;
      continue;
    } else if (inc > 0 && !z->status[i].registered) {
      /* second case, we need to register */
      /* first registration */
      for (j=i;j<e;j++) {
	if (z->status[j].registered)
	  break;
	gm_assert(z->status[j].refcount == 0);
	z->status[j].refcount = 1;
	z->status[j].registered = 1;
      }
      gm_assert(inc != 2 || (i == s && j == e));
#if DEBUG_REGCACHE >= 2
      GM_PRINT(0,("%d preparing to register i=%d j=%d - here's the zone dump\n",MPID_MyWorldRank,i,j));
#endif
      if (lockable_pages <= (j- i) ||
	  (retval = gm_weak_register_memory(gmpi.port,(char*)z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE)) != GM_SUCCESS){
	gmpi_gc_regcache();
	retval = gm_weak_register_memory(gmpi.port,(char*)z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE);
      }
      if (retval != GM_SUCCESS){
	GM_PRINT(0,("gm_register_memory failed: error %d, "
		    "port %x, zone=0x%lx:i=0x%x,j=0x%x,registered[j]=%d,refcount[j]=%d ",
		    retval, gmpi.port,
		    z->start,i,j,
		    z->status[j].registered,
		    z->status[j].refcount));
	GM_PRINT(0,("gm_register_memory ("
		    "port %x, start 0x%lx, len 0x%x )\n",
		    gmpi.port,
		    z->start+i*GMPI_PAGE_SIZE,(j-i)*GMPI_PAGE_SIZE));
	gmpi_dump_regcache();
	gm_abort();
      }
      lockable_pages -= (j - i);
      i = j;
    }
    if (i >= e)
      break;

    /* Other cases, the page does need to be registered or deregistered,
       just update the refcount */

    switch (inc) {
    case -2:
      gm_assert(!z->status[i].registered);
      break;
    case -1:
      gm_assert(z->status[i].registered && z->status[i].refcount > 0);
      z->status[i].refcount += inc;
      break;
    case 1:
      gm_assert(z->status[i].registered);
      z->status[i].refcount += inc;
      break;
    case 2:
    default: 
      gm_always_assert(0); /* Should never get there */
    }
  }
}

void gmpi_regcache_init(void)
{
  static int done;
  if(!done) {
    static struct gmpi_zone zone[PREZONE];
    int i;
    done = 1;
    for (i=1;i<PREZONE;i++)
      zone[i].next=&zone[i-1];
    zone[0].next = 0;
    gmpi.arch.free_intervals = &zone[PREZONE-1];
    gm_init();
    gm_always_assert(GM_PAGE_LEN);
    gmpi_zone_hash = gm_create_hash(gm_hash_compare_longs,gm_hash_hash_long,sizeof(long),0,10,0);
    lockable_pages = 512*1024*1024/GMPI_PAGE_SIZE;
  }
}

void gmpi_do_interval(unsigned long start,long length,int inc)
{
  unsigned long limit = start + length;
  struct gmpi_zone *p,**pp = &gmpi.arch.intervals;
  struct gmpi_zone *new;

  int my_pid;

  if (!gmpi_zone_hash) {
    gm_always_assert(inc == -2);
    return;
  }
#if DEBUG_REGCACHE >= 3
  my_pid = getpid();

  GM_PRINT(0,("%d gmpi_do_interval(0x%lx, 0x%lx, %d)\n", my_pid, start, length, inc));
#endif
  start = GMPI_PAGEDOWN(start);
  limit = GMPI_PAGEUP(limit);
#if DEBUG_REGCACHE >= 3
  GM_PRINT(0,("%d adjust_interval (0x%lx, 0x%lx, %d)\n", my_pid, start, (limit-start), inc));
#endif
  gm_assert(start < limit);
  while (start < limit) {
    unsigned long zonestart = start & ~GMPI_ZONE_MASK;
    p = gm_hash_find(gmpi_zone_hash,&zonestart);
    gm_assert(!p || p->start == zonestart);
    if (!p && inc != -2) {
      /*  zone not used before */
      unsigned long end = MIN(limit,zonestart+GMPI_ZONE_SIZE);
      gm_always_assert(inc == 1); /* marking zone as used */
      new = alloc_zone();
      gm_hash_insert(gmpi_zone_hash,&zonestart,new);
      gm_always_assert(new);
      new->next = p;
      new->start = zonestart;
      add_refcount(new,start,end,2);
      *pp = new;
      gmpi.arch.nbintervals += 1;
      start = end;
    } else if (p) {
      /* zone already exists */
      unsigned long end = MIN(limit,zonestart+GMPI_ZONE_SIZE);
      add_refcount(p,start,end,inc);
    }
    /* go to next zone */
    start = zonestart + GMPI_ZONE_SIZE;
  }
#if DEBUG_REGCACHE >= 3
  gmpi_dump_regcache();
#endif
}




void gmpi_clear_all_intervals(void)
{
  struct gmpi_zone *p;
  for (p=gmpi.arch.intervals;p;p=p->next) {
    add_refcount(p, p->start,p->start + GMPI_ZONE_SIZE,-2);
  }
}

void gmpi_clear_interval(unsigned long start,long length)
{
  gmpi_do_interval(start,length,-2);
}

#if 0
/* now done through gmpi_init_hook */
/* force inclusion of brk replacement objects */
extern int gmpi_mybrk;
static int *dummy_int=&gmpi_mybrk;
#endif

void gmpi_use_interval(unsigned long start,long length)
{
  gm_status_t cc;
  gmpi_do_interval(start,length,1);
#if GM_HAS_WEAK_REGISTER
  cc = gm_weak_register_memory(gmpi.port,(void*)start,length);
  gm_always_assert(cc == GM_SUCCESS);
#endif
}

void gmpi_unuse_interval(unsigned long start,long length)
{
  gmpi_do_interval(start,length,-1);
}

#else


void gmpi_use_interval(unsigned long start,long length)
{
  int retval;

  retval = gm_register_memory(gmpi.port,(void*)start,length);
  if (retval != GM_SUCCESS){
    GM_PRINT(0,("gm_register_memory failed: error %d"
		"port %x, start=0x%lx, length=0x%lx",
		retval, gmpi.port,
		start,length));
    gm_abort();
  }
}

void gmpi_unuse_interval(unsigned long start,long length)
{
  gm_status_t cc;
  cc = gm_deregister_memory(gmpi.port,(void*)start,length);
  gm_always_assert(cc == GM_SUCCESS);
}

void gmpi_regcache_init(void)
{
}

void gmpi_clear_all_intervals(void)
{
}

#endif

