#include "gmpi.h"

#include "mpid.h"
#include "mpiddev.h"
#include "mpimem.h"
#include "reqalloc.h"
#include "flow.h"
#include "chpackflow.h"

#ifdef GM_3WAY_PROTOCOL
#define MPID_CH_Rndvn_send               MPID_CH_3way_send               
#define MPID_CH_Rndvn_isend              MPID_CH_3way_isend              
#define MPID_CH_Rndvn_irecv              MPID_CH_3way_irecv              
#define MPID_CH_Rndvn_save               MPID_CH_3way_save               
#define MPID_CH_Rndvn_unxrecv_posted     MPID_CH_3way_unxrecv_posted     
#define MPID_CH_Rndvn_unxrecv_end        MPID_CH_3way_unxrecv_end        
#define MPID_CH_Rndvn_unxrecv_test_end   MPID_CH_3way_unxrecv_test_end   
#define MPID_CH_Rndvn_ok_to_send         MPID_CH_3way_ok_to_send         
#define MPID_CH_Rndvn_ack                MPID_CH_3way_ack                
#define MPID_CH_Rndvn_send_test          MPID_CH_3way_send_test          
#define MPID_CH_Rndvn_send_wait          MPID_CH_3way_send_wait          
#define MPID_CH_Rndvn_send_test_ack      MPID_CH_3way_send_test_ack      
#define MPID_CH_Rndvn_send_wait_ack      MPID_CH_3way_send_wait_ack      
#define MPID_CH_Rndvn_delete             MPID_CH_3way_delete             
#define MPID_CH_Rndvn_gm_recv_evt        MPID_CH_3way_gm_recv_evt
#define MPID_CH_Rndv_setup MPID_CH_3way_setup
#endif

/* Prototype definitions */
int MPID_CH_Rndvn_send ANSI_ARGS(( void *, int, int, int, int, int, 
					  MPID_Msgrep_t ));
int MPID_CH_Rndvn_isend ANSI_ARGS(( void *, int, int, int, int, int, 
				    MPID_Msgrep_t, MPIR_SHANDLE * ));
int MPID_CH_Rndvn_irecv ANSI_ARGS(( MPIR_RHANDLE *, int, void * ));
int MPID_CH_Rndvn_save ANSI_ARGS(( MPIR_RHANDLE *, int, void *));

int MPID_CH_Rndvn_unxrecv_posted ANSI_ARGS(( MPIR_RHANDLE *, void * ));
int MPID_CH_Rndvn_unxrecv_end ANSI_ARGS(( MPIR_RHANDLE * ));
int MPID_CH_Rndvn_unxrecv_test_end ANSI_ARGS(( MPIR_RHANDLE * ));
void MPID_CH_Rndvn_ok_to_send  ANSI_ARGS(( MPIR_RHANDLE *, MPID_RNDV_T));
int MPID_CH_Rndvn_ack ANSI_ARGS(( void *, int ));
int MPID_CH_Rndvn_send_test ANSI_ARGS(( MPIR_SHANDLE * ));
int MPID_CH_Rndvn_send_wait ANSI_ARGS(( MPIR_SHANDLE * ));
int MPID_CH_Rndvn_send_test_ack ANSI_ARGS(( MPIR_SHANDLE * ));
int MPID_CH_Rndvn_send_wait_ack ANSI_ARGS(( MPIR_SHANDLE * ));
void MPID_CH_Rndvn_delete ANSI_ARGS(( MPID_Protocol * ));
int MPID_CH_Rndvn_gm_recv_evt( MPIR_RHANDLE *rhandle, int rlength);
int MPID_CH_3way_gm_proxyrecv_evt( MPIR_RHANDLE *rhandle, int rlength);


int  MPID_DeviceCheck(MPID_BLOCKING_TYPE blocking);

/* Globals for this protocol */
/* This should be state in the protocol/device ?? */
static int CurTag    = 1024;
static int TagsInUse = 0;


#ifdef GM_3WAY_PROTOCOL
extern int rfrom;
#else
int rfrom;
#endif

static int data_verify_cksum(char *buf, int len,unsigned cksum)
{
  if (gmpi.enable_csum && gmpi_checksum((unsigned char *)buf,len) != cksum) {
    GM_PRINT(0,("mpich-gm cksum err for rndv proto at node %d from node %d: got 0x%x expected 0x%x \n",
		MPID_MyWorldRank, rfrom,gmpi_checksum((unsigned char *)buf,len), cksum));
    if (gmpi.enable_csum == 1)
      exit(1);
    /* else */
    return MPI_ERR_OTHER;
  }
  return MPI_SUCCESS;
}

/*
 * Definitions of the actual functions
 */

/*
 * This is really the same as the blocking version, since the 
 * nonblocking operations occur only in the data transmission.
 */
int MPID_CH_Rndvn_isend( buf, len, src_lrank, tag, context_id, dest,
			 msgrep, shandle )
void          *buf;
int           len, tag, context_id, src_lrank, dest;
MPID_Msgrep_t msgrep;
MPIR_SHANDLE  *shandle;
{
#define pkt (*pkt_p)
    MPID_PKT_REQUEST_SEND_T  *pkt_p = gmpi_ctl_alloc(sizeof(*pkt_p));

    gmpi_lock();
    
    DEBUG_PRINT_MSG("S Starting Rndvn_isend");
#ifdef MPID_PACK_CONTROL
    while (!MPID_PACKET_CHECK_OK(dest)) {  /* begin while !ok loop */
      /* Wait for a protocol ACK packet */
#ifdef MPID_DEBUG_ALL
      if (MPID_DebugFlag || MPID_DebugFlow)
	FPRINTF(MPID_DEBUG_FILE,
		"[%d] S Waiting for a protocol ACK packet (in rndvb isend) from %d\n",
		MPID_MyWorldRank, dest);
#endif
      MPID_DeviceCheck( MPID_BLOCKING );
    }  /* end while !ok loop */
    
    MPID_PACKET_ADD_SENT(MPID_MyWorldRank, dest )
#endif


#ifdef GM_3WAY_PROTOCOL
    pkt.mode       = MPID_PKT_3WAY_SEND;
    DEBUG_PRINT_MSG("Using 3 way protocol");
#else
    pkt.mode	   = MPID_PKT_REQUEST_SEND;
#endif
    pkt.context_id = context_id;
    pkt.lrank	   = src_lrank;
    pkt.to         = dest;                                           
    pkt.src        = MPID_MyWorldRank;                               
    pkt.seqnum     = sizeof(MPID_PKT_REQUEST_SEND_T);   
    pkt.tag	   = tag;
    pkt.len	   = len;
    MPID_DO_HETERO(pkt.msgrep = (int)msgrep);

    if (gmpi.enable_csum)
      pkt.cksum      = gmpi_checksum(buf,len);
    /* We save the address of the send handle in the packet; the receiver      
will return this to us */

    MPID_AINT_SET(pkt.send_id,shandle);
	
    /* Store info in the request for completing the message */
    shandle->is_complete     = 0;
    shandle->start	     = buf;
    shandle->bytes_as_contig = len;
    /* Set the test/wait functions */
    shandle->wait = 0;
    shandle->test = 0;
    /* Store partners rank in request in case message is cancelled */
    shandle->partner         = dest;
    shandle->gm.frag = 0; /* for multi-fragment messages */
    shandle->gm.posted = 0;
    shandle->gm.sent_evt_expected = 0;
    shandle->gm.copy = 0; /* send_now check that there is not already a temp buffer attached */
    /* shandle->test            = MPID_CH_Rndvn_send_test_ack; */
    /* shandle->finish must NOT be set here; it must be cleared/set
       when the request is created */

    /* DEBUG_PRINT_BASIC_SEND_PKT("S Sending rndv message",&pkt)*/

    MPID_PKT_PACK( &pkt, sizeof(pkt), dest );
    MPID_DRAIN_INCOMING_FOR_TINY(1);
    MPID_n_pending++;
    MPID_SendControlBlock( pkt_p, sizeof(pkt), dest );

    gmpi_unlock();
    return MPI_SUCCESS;
#undef pkt
}

int MPID_CH_Rndvn_send( buf, len, src_lrank, tag, context_id, dest,
			 msgrep )
void          *buf;
int           len, tag, context_id, src_lrank, dest;
MPID_Msgrep_t msgrep;
{
    MPIR_SHANDLE shandle;

    gmpi_lock();

    DEBUG_INIT_STRUCT(&shandle,sizeof(shandle));
    MPIR_SET_COOKIE((&shandle),MPIR_REQUEST_COOKIE);
    MPID_SendInit( &shandle );
    shandle.finish = 0;
    MPID_CH_Rndvn_isend( buf, len, src_lrank, tag, context_id, dest,
			 msgrep, &shandle );
    while (!shandle.is_complete) {
      MPID_DeviceCheck(MPID_BLOCKING);
    }
    if (shandle.finish)
      shandle.finish(&shandle);
    gmpi_unlock();
    return MPI_SUCCESS;
}


static void modify_state_when_sending_ack(MPIR_RHANDLE *rhandle)
{
  gm_assert_p(rhandle->gm.state == ST_POSTED_GOTCTL
	      || rhandle->gm.state == ST_POSTED_ACKQUEUED
	      || rhandle->gm.state == ST_POSTED_GOTFRAG
	      || rhandle->gm.state == ST_PROXY_GOTCTL
	      || rhandle->gm.state == ST_PROXY_ACKQUEUED);
  GM_DBG(rhandle->gm.state = IS_PROXY(rhandle->gm.state) ?
	 ST_PROXY_ACKSENT : ST_POSTED_ACKSENT);
}


static void ok_to_send_for_zero_length_packet(MPIR_RHANDLE *rhandle)
{
  gm_assert_p(rhandle->gm.state == ST_POSTED_GOTCTL);
  MPID_CH_Rndvn_ok_to_send( rhandle, -1);
  rhandle->is_complete = 1;
  GM_DBG(rhandle->gm.state = ST_POSTED_COMPLETED);
  if (rhandle->finish)
    (rhandle->finish)(rhandle);
}


/*
 * This is the routine called when a packet of type MPID_PKT_REQUEST_SEND is
 * seen, the receive may have been posted or not depending on the expected arg
 */
static int do_recv( MPIR_RHANDLE *rhandle, int from, void *in_pkt, int expected )
{
  int err = 0;
  int msglen,gmlen;
  MPID_PKT_REQUEST_SEND_T   *pkt = (MPID_PKT_REQUEST_SEND_T *)in_pkt;

  MPID_PKT_UNPACK( (MPID_PKT_HEAD_T *)in_pkt + 1, 
                   sizeof(MPID_PKT_REQUEST_SEND_T) - sizeof(MPID_PKT_HEAD_T),
                   from );

#ifdef MPID_PACK_CONTROL
    if (MPID_PACKET_RCVD_GET(pkt->src)) {
      MPID_SendProtoAck(pkt->to, pkt->src);
    }
    MPID_PACKET_ADD_RCVD(pkt->to, pkt->src);
#endif

  msglen = pkt->len;
  if (expected) {
    MPID_CHK_MSGLEN(rhandle,msglen,err);
  }
  MPIR_SET_COOKIE((rhandle),MPIR_REQUEST_COOKIE); 
  
  DEBUG_PRINT_MSG("Saving info on unexpected message");
  rhandle->gm.posted_req = 0;
  rhandle->s.MPI_TAG	  = pkt->tag;
  rhandle->s.MPI_SOURCE = pkt->lrank;
  rhandle->s.MPI_ERROR  = 0;
  rhandle->s.count      = msglen;
  rhandle->is_complete  = 0;
  rhandle->from         = from;
  rhandle->partner      = pkt->to;     
  rhandle->send_id      = pkt->send_id;
  rhandle->unex_buf = 0;
  rhandle->gm.netbuf = 0;
  rhandle->gm.cksum = pkt->cksum;
  MPID_DO_HETERO(rhandle->msgrep = (MPID_Msgrep_t)pkt->msgrep );

  /* WARNING - I have no idea if this this is the right place for this - rs */
#ifdef MPID_PACK_CONTROL
  while (!MPID_PACKET_CHECK_OK(from)) {  /* begin while !ok loop */
    /* Wait for a protocol ACK packet */
#ifdef MPID_DEBUG_ALL
    if (MPID_DebugFlag || MPID_DebugFlow)
      FPRINTF(MPID_DEBUG_FILE,
         "[%d] S Waiting for a protocol ACK packet (in rndvb isend) from %d\n",
	      MPID_MyWorldRank, from);
#endif
    MPID_DeviceCheck( MPID_BLOCKING );
  }  /* end while !ok loop */

  MPID_PACKET_ADD_SENT(pkt->to, from )
#endif

#if 0
  size = DATA_MESSAGE_SIZE(msglen+GMPI_DATA_HEADER_LEN);
#endif
  gmlen = MIN(GMPI_MAX_DATA_FRAG,GMPI_ROUNDUP(msglen));
  if (expected) {
    /* rhandle->wait	 = MPID_CH_Rndvn_unxrecv_end; */
    /* rhandle->test	 = MPID_CH_Rndvn_unxrecv_test_end; */
    gm_assert_p(rhandle->gm.state == ST_POSTED_GOTCTL);
    rhandle->gm.frag = 0;
    rhandle->gm.recvcallback = MPID_CH_Rndvn_gm_recv_evt;
    /* queue only for a fragment if too big */
    if (gmlen)
      gmpi_queue_recv(gmlen, MPID_CH_Rndvn_ok_to_send, rhandle);
    else 
      ok_to_send_for_zero_length_packet(rhandle);
  } else {
    rhandle->gm.recvcallback = MPID_CH_3way_gm_proxyrecv_evt;
    rhandle->gm.unex_complete = 0;
    /* Need to set the push etc routine to complete this transfer */
    gm_assert_p(rhandle->gm.state == ST_PROXY_GOTCTL);
    rhandle->push         = MPID_CH_Rndvn_unxrecv_posted;
#ifdef GM_3WAY_PROTOCOL
    rhandle->gm.frag = 0;
    gm_assert_p(gmlen > 0);
    gmpi_queue_recv(gmlen, MPID_CH_Rndvn_ok_to_send, rhandle);
#endif
  }
  return 0;
}

int MPID_CH_Rndvn_irecv( rhandle, from, in_pkt )
     MPIR_RHANDLE *rhandle;
     int          from;
     void         *in_pkt;
{
    DEBUG_PRINT_MSG("R Starting rndvn irecv");

  /* we use rhandle->buf != 0 to mean recv posted */
  gm_assert_p(rhandle->gm.state == ST_POSTED_GOTCTL);
  if (!rhandle->buf)
    rhandle->buf = (void*)1;
  return do_recv(rhandle,from,in_pkt,1);
}


int MPID_CH_Rndvn_save( rhandle, from, in_pkt )
     MPIR_RHANDLE *rhandle;
     int          from;
     void         *in_pkt;
{
  gm_assert_p(rhandle->gm.state == ST_PROXY_GOTCTL);
  rhandle->buf = 0;

  DEBUG_PRINT_MSG("Saving info on unexpected message");

  return do_recv(rhandle,from,in_pkt,0);
}


/*
 * This is an internal routine to return an OK TO SEND packet.
 * It is the same as the Rndvb version
 */
void MPID_CH_Rndvn_ok_to_send( MPIR_RHANDLE *rhandle, MPID_RNDV_T tag)
{
#define pkt (*pkt_p)
  MPID_PKT_OK_TO_SEND_T *pkt_p = gmpi_ctl_alloc(sizeof(*pkt_p));

  modify_state_when_sending_ack(rhandle);
    /* that was very wrong rhandle->recv_handle = 0; */
    pkt.mode = MPID_PKT_OK_TO_SEND;
    pkt.lrank  = MPID_MyWorldRank;                        
    pkt.src    = MPID_MyWorldRank;                    
    pkt.seqnum = sizeof(MPID_PKT_OK_TO_SEND_T);     
    pkt.send_id = rhandle->send_id;
    pkt.recv_handle = (MPID_RNDV_T)tag;
    DEBUG_PRINT_BASIC_SEND_PKT("S Ok send", &pkt);
    MPID_PKT_PACK( &pkt, sizeof(pkt), rhandle->from );
    MPID_SendControlBlock( pkt_p, sizeof(pkt), rhandle->from );
    GM_DBG(rhandle->gm.state = IS_PROXY(rhandle->gm.state)?ST_PROXY_ACKSENT:ST_POSTED_ACKSENT);
#undef pkt
}

#ifdef GM_3WAY_PROTOCOL
int MPID_CH_3way_gm_proxyrecv_evt( MPIR_RHANDLE *rhandle, int rlength)
{
  int err = MPI_SUCCESS;

  gm_assert_p(rhandle);
  gm_assert_p(rhandle->gm.netbuf);
  gm_assert_p(rhandle->gm.state == ST_PROXY_ACKSENT);
  gm_assert_p(GMPI_ROUNDUP(rhandle->s.count) == rlength);
  if (rhandle->gm.posted_req) {
    /* Recv has been posted between ack and actual reception 
       recv_handle links to final handle */
    MPIR_RHANDLE *new = rhandle->gm.posted_req;
    new->gm.frag_length = rhandle->gm.frag_length;
    new->gm.netbuf = rhandle->gm.netbuf;
    new->gm.cksum = rhandle->gm.cksum;
    gm_assert_p(!rhandle->buf);
    gm_assert_p(new->gm.state == ST_POSTED_PROXYWAIT);
    gm_assert_p(rhandle->gm.state == ST_PROXY_ACKSENT);
    gm_assert_p(rhandle->unex_buf == 0);
    MPID_RecvFree(rhandle);
    GM_DBG(new->gm.state = ST_POSTED_ACKSENT);
    new->gm.recvcallback(new,rlength);
  } else {
    rhandle->gm.unex_complete = 1; /* to say the recv has completed */
    gm_assert(rhandle->unex_buf == 0);
    rhandle->unex_buf = gmpi_xmalloc(rhandle->s.count,"MPI/ch_gm internal:3way_proxy_evt");
    /* FIXME: useless copy in the case of BIP */
    gm_bcopy(rhandle->gm.netbuf,rhandle->unex_buf,rhandle->s.count);
    gmpi_free_databuf(rhandle->gm.netbuf, rhandle->gm.frag_length);
    rhandle->gm.netbuf = 0;
    GM_DBG(rhandle->gm.state = ST_PROXY_COMPLETED);
  }
  return err;
}
#endif


int MPID_CH_Rndvn_gm_recv_evt( MPIR_RHANDLE *rhandle, int rlength)
{
  int count; /* bytes not yet received for the message */
  int err = MPI_SUCCESS;

  gm_assert_p(rhandle->gm.state == ST_POSTED_ACKSENT);
  gm_assert_p(GMPI_ROUNDUP(rhandle->s.count - rhandle->gm.frag) == 
              rlength ||
              (rlength == GMPI_MAX_DATA_FRAG && 
               rhandle->s.count - rhandle->gm.frag >
               GMPI_MAX_DATA_FRAG));
  count = rhandle->s.count - rhandle->gm.frag;
  /* check if this is a multiple fragment packet */
  if (count > rlength) {
    /* received a full fragment */
    GM_DBG(rhandle->gm.state = ST_POSTED_GOTFRAG);
    gm_assert_p(rlength == GMPI_MAX_DATA_FRAG);
    if (rhandle->gm.netbuf) { /* copy */
      gm_bcopy(rhandle->gm.netbuf,(char*)rhandle->buf+rhandle->gm.frag,rlength);
      gmpi_free_databuf(rhandle->gm.netbuf,rlength);
      rhandle->gm.netbuf = 0;
    }
    rhandle->gm.frag += rlength;
    gmpi_queue_recv(MIN(GMPI_MAX_DATA_FRAG, rhandle->s.count-rhandle->gm.frag),MPID_CH_Rndvn_ok_to_send,rhandle);
  } else {
    /* received last part of message */
    gm_assert_p(GMPI_ROUNDUP(count) == rlength);
    GM_DBG(rhandle->gm.state = ST_POSTED_COMPLETED);
    if (rhandle->gm.netbuf) {
      gm_bcopy(rhandle->gm.netbuf,(char*)rhandle->buf+rhandle->gm.frag,count);
      gmpi_free_databuf(rhandle->gm.netbuf,rlength);
      rhandle->gm.netbuf = 0;
    }
    rfrom = rhandle->from;
    err = data_verify_cksum(rhandle->buf,rhandle->s.count, rhandle->gm.cksum);
    rhandle->is_complete = 1;
    if (rhandle->finish)
      (rhandle->finish)(rhandle);
  }
  return err;
}


/* 
 * This routine is called when it is time to receive an unexpected
 * message.  
 */
int MPID_CH_Rndvn_unxrecv_posted( rhandle, in_runex )
MPIR_RHANDLE *rhandle;
void         *in_runex;
{
    MPIR_RHANDLE *runex = (MPIR_RHANDLE *)in_runex;
    int gmlen;
    int err = MPI_SUCCESS;

    gm_assert_p(IS_PROXY(runex->gm.state));
    gm_assert_p(rhandle->gm.state == ST_POSTED_CTLWAIT);

    rhandle->gm.posted_req = 0;
    gm_assert_p(runex->gm.posted_req == 0);

    rhandle->recv_handle = -1;
    rhandle->s		 = runex->s;
    /* we use rhandle->buf != 0 to mean recv posted */
    if (!rhandle->buf) {
      gm_assert_p(rhandle->s.count == 0);
      rhandle->buf = (void*)1;
    }
    /* rhandle->wait	 = MPID_CH_Rndvn_unxrecv_end; */
    /* rhandle->test	 = MPID_CH_Rndvn_unxrecv_test_end; */
    /* Must NOT set finish, since it may have been set elsewhere */
    rhandle->push	 = 0;
    rhandle->send_id = runex->send_id;
    rhandle->from        = runex->from;
    rhandle->unex_buf = 0;
    rhandle->gm.netbuf = 0;
    rhandle->gm.frag = 0;
    rhandle->gm.cksum = runex->gm.cksum;
    rhandle->gm.recvcallback = MPID_CH_Rndvn_gm_recv_evt;
#ifdef GM_3WAY_PROTOCOL
    gm_assert_p(runex->gm.state == ST_PROXY_ACKQUEUED || runex->gm.frag_length > 0);
    gm_assert_p(runex->gm.state != ST_PROXY_GOTCTL);
    if (!runex->gm.unex_complete) { /* receive not yet completed */
      /* we do not want to free runex, beause this is the receive tag the message will point to */
      gm_assert_p(runex->gm.state != ST_PROXY_COMPLETED);
      runex->gm.posted_req = rhandle;
      GM_DBG(rhandle->gm.state = ST_POSTED_PROXYWAIT);
      /* rhandle->send_id = unex->send_id : useless, we will not need it anymore in rhandle */
    } else { /* data is in unex_buf */
      gm_assert_p(runex->gm.state == ST_PROXY_COMPLETED);
      gm_bcopy(runex->unex_buf,rhandle->buf,rhandle->s.count);
      rhandle->is_complete = 1;
      GM_DBG(rhandle->gm.state = ST_POSTED_COMPLETED);

      rfrom = rhandle->from;
      err = data_verify_cksum(rhandle->buf,rhandle->s.count, rhandle->gm.cksum);
      if (rhandle->finish)
	(rhandle->finish)(rhandle);
      free(runex->unex_buf);
      runex->unex_buf = 0;
      MPID_RecvFree( runex );
    }
#else
    gmlen = MIN(GMPI_MAX_DATA_FRAG,GMPI_ROUNDUP(rhandle->s.count));
    gm_assert_p(runex->gm.state == ST_PROXY_GOTCTL);
    GM_DBG(rhandle->gm.state = ST_POSTED_GOTCTL);
    if (gmlen)
      gmpi_queue_recv(gmlen, MPID_CH_Rndvn_ok_to_send, rhandle);
    else
      ok_to_send_for_zero_length_packet(rhandle);
    MPID_RecvFree(runex);
#endif
    return err;
}



/* 
 * This is the routine that is called when an "ok to send" packet is
 * received.  
 */
int MPID_CH_Rndvn_ack( in_pkt, from_grank )
void  *in_pkt;
int   from_grank;
{
    MPID_PKT_OK_TO_SEND_T *pkt = (MPID_PKT_OK_TO_SEND_T *)in_pkt;
    MPIR_SHANDLE *shandle=0;
    char *sbuf;
    int len, gmlen;

    DEBUG_PRINT_MSG("R Starting Rndvb_ack");
#ifdef MPID_PACK_CONTROL
    if (MPID_PACKET_RCVD_GET(pkt->src)) {
      MPID_SendProtoAck(pkt->to, pkt->src);
    }
    MPID_PACKET_ADD_RCVD(pkt->to, pkt->src);
#endif

    MPID_AINT_GET(shandle,pkt->send_id);
#ifdef MPIR_HAS_COOKIES
    if (shandle->cookie != MPIR_REQUEST_COOKIE) {
        FPRINTF( stderr, "shandle is %lx\n", (long)shandle );
        FPRINTF( stderr, "shandle cookie is %lx\n", shandle->cookie );
	MPID_Print_shandle( stderr, shandle );
	/*	MPID_Abort( (struct MPIR_COMMUNICATOR *)0, 1, "MPI internal", 
		"Bad address in Rendezvous send" );*/
    }
#endif	

#ifdef MPID_DEBUG_ALL
    if (MPID_DebugFlag) {
      FPRINTF( MPID_DEBUG_FILE, "[%d]S for ", MPID_MyWorldRank );
      MPID_Print_shandle( MPID_DEBUG_FILE, shandle );
    }
#endif

    DEBUG_PRINT_MSG("Sending data on channel with nonblocking send");
#ifdef MPID_DEBUG_ALL
    if (MPID_DebugFlag) {
	FPRINTF( MPID_DEBUG_FILE, "[%d]S for ", MPID_MyWorldRank );
	MPID_Print_shandle( MPID_DEBUG_FILE, shandle );
    }
#endif
    MPID_n_pending--;
    
    if (shandle->gm.posted > 0) {
#if 0
      fprintf(stderr,"warning: we received an answer before being notified of the send completion but that is OK\n");
#endif
      /* free temporary resources (to be able to send the next fragment) */
      shandle->gm.sent_evt_expected += 1;
      gmpi_datafrag_complete(shandle);
    }
    len = shandle->bytes_as_contig-shandle->gm.frag;
    /* saving recv_handle needed in gmpi-queue_send */
    shandle->recv_handle = pkt->recv_handle;
    if (len > GMPI_MAX_DATA_FRAG) {
      len = GMPI_MAX_DATA_FRAG;
    }
    sbuf = (char*)shandle->start+shandle->gm.frag;
    shandle->gm.frag += len;
    shandle->gm.last_frag_length = len;
    shandle->gm.posted += 1;
    gm_assert_p(shandle->is_complete == 0);
    
    if (len) 
      {
	gmpi_queue_send(sbuf, shandle, len, from_grank,1,0);
      }
    else {
      gm_assert(shandle->bytes_as_contig == 0);
      shandle->is_complete = 1;
      if (shandle->finish)
	(shandle->finish)(shandle);
    }
    /* shandle->wait	 = MPID_CH_Rndvn_send_wait; */
    /* shandle->test	 = MPID_CH_Rndvn_send_test; */
    /* If the ref count is 0, we should just forget about the request, 
       as in the shared memory case.  For this, we'll need a request
       free operation in the interface */
    return MPI_SUCCESS;
}


/* 
 * CancelSend 
 * This is fairly hard.  We need to send a "please_cancel_send", 
 * which, if the message is found in the unexpected queue, removes it.
 * However, if the message is being received at the "same" moment, the
 * ok_to_send and cancel_send messages could cross.  To handle this, the
 * receiver must ack the cancel_send message (making the success of the
 * cancel non-local).  There are even more complex protocols, but we won't
 * bother.
 * 
 * Don't forget to update MPID_n_pending as needed.
 */

void MPID_CH_Rndvn_delete( p )
MPID_Protocol *p;
{
  FREE( p );
}

/*
 * The only routing really visable outside this file; it defines the
 * Blocking Rendezvous protocol.
 */
MPID_Protocol *MPID_CH_Rndv_setup()
{
    MPID_Protocol *p;

    p = (MPID_Protocol *) MALLOC( sizeof(MPID_Protocol) );
    if (!p) return 0;
    p->send	   = MPID_CH_Rndvn_send;
    p->recv	   = 0;
    p->isend	   = MPID_CH_Rndvn_isend;
    p->wait_send   = 0;
    p->push_send   = 0;
    p->cancel_send = 0;
    p->irecv	   = MPID_CH_Rndvn_irecv;
    p->wait_recv   = 0;
    p->push_recv   = 0;
    p->cancel_recv = 0;
    p->do_ack      = MPID_CH_Rndvn_ack;
    p->unex        = MPID_CH_Rndvn_save;
    p->delete      = MPID_CH_Rndvn_delete;

    return p;
}

