/*
 *  $Id: chcancel.c,v 1.2 2000/04/19 07:10:54 patrick Exp $
 *
 *  (C) 1996 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

#include "mpid.h"
#include "mpiddev.h"
#include "mpimem.h"
#include "reqalloc.h"
#include "packets.h"
#include "../util/queue.h"

/*
 * This file contains the routines to handle canceling a message
 *
 */

int expect_cancel_ack = 0;

/* 
 * CancelSend 
 * This is fairly hard.  We need to send a "please_cancel_send", 
 * which, if the message is found in the unexpected queue, removes it.
 * However, if the message is being received at the "same" moment, the
 * ok_to_send and cancel_send messages could cross.  To handle this, the
 * receiver must ack the cancel_send message (making the success of the
 * cancel non-local).  There are even more complex protocols, but we won't
 * bother.
 * 
 * Don't forget to update MPID_n_pending as needed.
 */

/* This routine is called from MPI_Cancel.  Its purpose is to send an    
   anti-send packet to the calling process's partner.  If successful,
   the error_code will return MPI_SUCCESS, otherwise the error_code will
   return MPI_ERR_OTHER */
void MPID_SendCancelPacket( request, err_code )
MPI_Request *request;
int         *err_code;

{  /* begin MPID_SendCancelPacket */

  MPID_Aint send_id;
  int (*fcn) ANSI_ARGS(( int, int, int, int, MPID_Aint ));
  MPIR_SHANDLE *shandle = (MPIR_SHANDLE *)*request;
  MPID_Device *dev = MPID_devset->dev[shandle->partner];

  /* Choose the function based on the device used */
  fcn = dev->short_msg->cancel_send;
  DEBUG_TEST_FCN(fcn,"dev->proto->cancel_send");
  MPID_AINT_SET(send_id,shandle);
  gmpi_lock();
  *err_code = (*(fcn))( MPID_MyWorldRank, shandle->partner, 0, 0, send_id );
  expect_cancel_ack++; 
  gmpi_unlock(); 
}  /* end MPID_SendCancelPacket */


/* This routine is called when a process receives an anti-send pkt.  Its 
   purpose is to search for the request found in the pkt in the unexpected
   queue.  If found, set the pkt.cancel to 1, otherwise, set pkt.cancel to 
   0.  Send this information back in an anti-send-ok pkt. */
void MPID_SendCancelOkPacket( in_pkt, from )
void *in_pkt;
int  from;

{  /* begin MPID_SendCancelOkPacket */
 
  int (*Cancel_send) ANSI_ARGS(( int, int, int, int, MPID_Aint ));
  int (*Cancel_recv) ANSI_ARGS(( void *, int, int *, MPID_Aint * ));
  MPID_Device *dev = MPID_devset->dev[from];

  int error_code;
  int found = 0;
  int cancel;
  MPID_Aint send_id;

  MPIR_SHANDLE *shandle=0;
  MPIR_RHANDLE *rhandle;

  gmpi_lock();
  
 /* Choose the function based on the device used */
  Cancel_recv = dev->short_msg->cancel_recv;
  DEBUG_TEST_FCN(Cancel_recv,"dev->proto->cancel_recv");
  error_code = (*(Cancel_recv))( in_pkt, from, &cancel, &send_id ); 
 
  MPID_AINT_GET(shandle, send_id); 

  /* Look for request, if found, delete it */
  error_code = MPID_Search_unexpected_for_request(shandle, &rhandle, &found);

  if ( (error_code != MPI_SUCCESS) || (found == 0) ) 
    cancel = 0;
  else {
      if ( rhandle->s.count < dev->long_len) {  
	  /* begin if short/eager message */
	  FREE ( rhandle->start ); 
	  rhandle->start = 0; 
	  MPID_RecvFree( rhandle ); 
      } 
      else { /* begin if rndv message */
	  MPID_RecvFree( rhandle );
      } 

      cancel = 1; 
  }

 /* Choose the function based on the device used */
  Cancel_send = dev->short_msg->cancel_send;
  DEBUG_TEST_FCN(Cancel_send,"dev->proto->cancel_send");
  error_code = (*(Cancel_send))( MPID_MyWorldRank, from, cancel, 1, send_id ); 
  gmpi_unlock(); 

}  /* end MPID_SendCancelOkPacket */

/* This routine is called when a process receives an anti-send-ok packet.
   If pkt->cancel = 1, then set the request found in the pkt as
   cancelled and complete.  If pkt->cancel = 0, do nothing. */
void MPID_RecvCancelOkPacket( in_pkt, from )
void *in_pkt;
int  from;

{  
  /* begin MPID_RecvCancelOkPacket */
  
  int (*Cancel_recv) ANSI_ARGS(( void *, int, int *, MPID_Aint * ));
  MPID_Device *dev = MPID_devset->dev[from];

  int error_code;
  int cancel;
  MPID_Aint send_id;

  MPIR_SHANDLE *shandle=0;
  
   /* Choose the function based on the device used */
  Cancel_recv = dev->short_msg->cancel_recv;
  DEBUG_TEST_FCN(Cancel_recv,"dev->proto->cancel_recv");
  error_code = (*(Cancel_recv))( in_pkt, from, &cancel, &send_id ); 
 
  MPID_AINT_GET(shandle, send_id); 

  DEBUG_PRINT_MSG("R Receive anti-send ok message\n");  

  if (cancel) {   /* begin if pkt->cancel */
      /* Mark the request as cancelled */
      shandle->s.MPI_TAG = MPIR_MSG_CANCELLED; 
      /* Mark the request as complete */
      /* shandle->is_complete = 1; */
      shandle->is_cancelled = 1;
      MPID_n_pending--;  
      DEBUG_PRINT_MSG("Request has been successfully cancelled");
  }   /* end if pkt->cancel */
  else {
      shandle->is_cancelled = 0;
      DEBUG_PRINT_MSG("Unable to cancel request");
  }
  shandle->cancel_complete = 1;

  expect_cancel_ack--;

}  /* end MPID_RecvCancelOkPacket */


/* This routine will block while a process is still expecting an ack from
   a cancel request.  Called by MPID_CH_End */
void MPID_FinishCancelPackets( dev )
MPID_Device *dev;

{  /* begin MPID_FinishCancelPackets */

    DEBUG_PRINT_MSG("Entering MPID_FinishCancelPackets");
    
    while (expect_cancel_ack > 0) {
	MPID_DeviceCheck( MPID_BLOCKING ); } 

    DEBUG_PRINT_MSG("Leaving MPID_FinishCancelPackets");

}  /* end MPID_FinishCancelPackets */
