//==========================================================================
//
//  THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
//  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
//  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
//  PURPOSE.
//
//  Copyright (c) 1999 - 2001  On2 Technologies Inc. All Rights Reserved.
//
//--------------------------------------------------------------------------


///////////////////////////////////////////////////////////////////////////
//
// dxlqt_helper.cpp
//
// Purpose: A list of helper functions to the quick time codec code
//
/////////////////////////////////////////////////////////////////////////

#include <Endian.h>
#include <ImageCodec.h>
#include "duck_dxl.h"
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "duck_dxl.h"
#include <stdlib.h>

struct DisplaySetting {
	long dotOne;
	long dotTwo;
	long dotThree;
	long dotFour;
	long dotFive;
};

// Display patterns indicate output pixel formats (*=white dot, _=black dot).
static DisplaySetting RGB24 ={0x00000000,0x00000000,0xffffffff,0x00000000,0xffffffff}; // __*_*
static DisplaySetting RGB32 ={0x00000000,0x00000000,0x00000000,0x00000000,0xffffffff}; // ____*
static DisplaySetting RGB555={0xffffffff,0x00000000,0xffffffff,0x00000000,0xffffffff}; // *_*_*
static DisplaySetting RGB565={0xffffffff,0x00000000,0x00000000,0x00000000,0xffffffff}; // *___*
static DisplaySetting UYVY  ={0xff80ff80,0x00800080,0xff80ff80,0x00800080,0x00800080}; // *_*_
static DisplaySetting YUY2  ={0x80ff80ff,0x80008000,0x80008000,0x80008000,0x80008000}; // *____
static DisplaySetting YVU9  ={0x80008000,0x80008000,0xff80ff80,0xff80ff80,0xff80ff80}; // __***
static DisplaySetting RGB8  ={0x00000000,0xffffffff,0x00000000,0xffffffff,0x00000000}; // _*_*_

// ************************************************************
// Function name	: DrawInformativeDots
// Description	    : Draw Dots on the screen 
// Return type		: void 
// Argument         : unsigned char* startingPositionOfBuffer
// Argument         : enum BITDEPTH displayType
// ************************************************************
void DrawInformativeDots( unsigned char* startingPositionOfBuffer,enum BITDEPTH displayType)
{
	const DisplaySetting *resolutionType = 0;
	switch( displayType) {
		case DXRGB24:		resolutionType = &RGB24;  break;
		case DXRGB32:		resolutionType = &RGB32;  break;
		case DXRGB16_555:	resolutionType = &RGB555;  break;
		case DXRGB16_565:	resolutionType = &RGB565;  break;
		case DXYUY2:		resolutionType = &YUY2;  break;
		case DXYVU9:		resolutionType = &YVU9;  break;
		case DXHALFTONE8:	resolutionType = &RGB8;  break;
		default: break;
	}
	*(unsigned long*)(startingPositionOfBuffer) = resolutionType->dotOne;
	*(unsigned long*)(startingPositionOfBuffer+32) = resolutionType->dotTwo;
	*(unsigned long*)(startingPositionOfBuffer+64) = resolutionType->dotThree;
	*(unsigned long*)(startingPositionOfBuffer+96) = resolutionType->dotFour;
	*(unsigned long*)(startingPositionOfBuffer+128) = resolutionType->dotFive;
}

void DrawRandom(char *addr,long w,long h,long p)
{
	int i,j,xp,yp;

	xp = w>>4;
	yp = h>>4;
	char x=rand()*256/RAND_MAX;
	for (i=0;i<h;i++)
		for(j=0;j<w;j++)
			addr[i*p+j]=x;
}


// ************************************************************
// Function name	: DrawCheckers
// Description	    : draw a checkerboard using 0-255 on the screen 
// Return type		: void 
// Argument         : char *addr
// Argument         : long w
// Argument         : long h
// ************************************************************
void DrawCheckers(char *addr,long w,long h)
{
	int i,j,xp,yp;

	xp = w>>4;
	yp = h>>4;

	for (i=0;i<h;i++)
		for(j=0;j<w;j++)
			addr[i*w+j]=(j/xp)+(i/yp)*xp;
}

 

// ************************************************************
// Function name	: Shift
// Description	    : shift all of the characters of the screen up or down
// Return type		: void 
// Argument         : char *addr
// Argument         : long w
// Argument         : long h
// ************************************************************
void Shift(char *addr,long w,long h)
{
	int i,j,s;

	s=0;
	for (i=0;i<h;i++)
		for(j=0;j<w;j++)
			addr[i*w+j]+=s;
}
 

// ************************************************************
// Function name	: bitdepth2string
// Description	    : Convert the bitdepth to a string (Helper Function)
// Return type		: void 
// Argument         : enum BITDEPTH bitdepth
// Argument         : char *str
// ************************************************************
void bitdepth2string(enum BITDEPTH bitdepth,char *str)
{
	#define f1s(X) case X: sprintf(str,"%s" #X,str);break;
	switch(bitdepth)
	{
		f1s(DXRGBNULL)
		f1s(DXRGB8)
		f1s(DXRGB16_555)
		f1s(DXRGB24)
		f1s(DXRGB_UNUSED)
		f1s(DXRGB16VESA)
		f1s(DXRGB8VESA)
		f1s(DXRGB16_565)
		f1s(DXYUY2)
		f1s(DXYVU9)
		f1s(DXYV12)
		f1s(DXUYVY)
		f1s(DXRGB32)
		f1s(DXRGB16VESA_565)
		f1s(DXHALFTONE8)
		f1s(DXMAX)
	}

}


// ************************************************************
// Function name	: condflags2string
// Description	    : Convert the condition flags to a string (Helper Function)
// Return type		: void 
// Argument         : long flags
// Argument         : char * str
// ************************************************************
void condflags2string(long flags,char * str)
{
	sprintf(str,"condflags:");
	#define f2s(X) if(flags&codecCondition##X) sprintf(str,"%s," #X,str);
	f2s(FirstBand)	 
	f2s(LastBand)	 
	f2s(FirstFrame)
	f2s(NewDepth)	 
	f2s(NewTransform)	 
	f2s(NewSrcRect)
	f2s(NewMask)
	f2s(NewMatte)
	f2s(NewTransferMode)
	f2s(NewClut)
	f2s(NewAccuracy)
	f2s(NewDestination) 
	f2s(FirstScreen)
	f2s(DoCursor)	 
	f2s(CatchUpDiff)
	f2s(MaskMayBeChanged)
	f2s(ToBuffer)
	f2s(CodecChangedMask)
	sprintf(str,"%s\n",str);

}



// ************************************************************
// Function name	: callflags2string
// Description	    : Convert the caller flags to a string (Helper Function)
// Return type		: void 
// Argument         : long flags
// Argument         : char * str
// ************************************************************
void callflags2string(long flags,char * str)
{
	sprintf(str,"callflags:");
	#define f3s(X) if(flags&codecFlag##X) sprintf(str,"%s," #X,str);

	f3s(UseImageBuffer)		
	f3s(UseScreenBuffer)	
	f3s(UpdatePrevious)		
	f3s(NoScreenUpdate)		
	f3s(WasCompressed)		
	f3s(DontOffscreen)		
	f3s(UpdatePreviousComp)	
	f3s(ForceKeyFrame)		
	f3s(OnlyScreenUpdate)
	f3s(LiveGrab)			
	f3s(DontUseNewImageBuffer)
	f3s(InterlaceUpdate)	
	f3s(CatchUpDiff)		
	f3s(SupportDisable)		
	f3s(Reenable)
	sprintf(str,"%s\n",str);
}

// ************************************************************
// Function name	: capflags2string
// Description	    : Convert the caller flags to a string (Helper Function)
// Return type		: void 
// Argument         : long flags
// Argument         : char * str
// ************************************************************
void capflags2string(long flags,char * str)
{
	sprintf(str,"callflags:");
	#define f4s(X) if(flags&codec##X) sprintf(str,"%s," #X,str);
	f4s(CanScale)
	f4s(CanMask)
	f4s(CanMatte)
	f4s(CanTransform)
	f4s(CanTransferMode)
	f4s(CanCopyPrev)
	f4s(CanSpool)
	f4s(CanClipVertical)
	f4s(CanClipRectangular)
	f4s(CanRemapColor)
	f4s(CanFastDither)
	f4s(CanSrcExtract)
	f4s(CanCopyPrevComp)
	f4s(CanAsync)
	f4s(CanMakeMask)
	f4s(CanShift)
	f4s(CanAsyncWhen)
	f4s(CanShieldCursor)
	f4s(CanManagePrevBuffer)
	f4s(HasVolatileBuffer)
	f4s(WantsRegionMask)
	f4s(ImageBufferIsOnScreen)
	f4s(WantsDestinationPixels)
	f4s(WantsSpecialScaling)
	f4s(HandlesInputs)
	f4s(CanDoIndirectSurface)
	f4s(IsSequenceSensitive)
	f4s(RequiresOffscreen)
	f4s(RequiresMaskBits)
	f4s(CanRemapResolution)
	f4s(IsDirectToScreenOnly)
	f4s(CanLockSurface)
}



/*
 * **-lutbl.c
 *
 * This file contains the looks up tables used for converting
 * from RGB to YUV.
 *
 * Each RGB value can range from 0-255 so each multiplier table
 * consists of 256 entries of 4 bytes each.  Format for most of the
 * look up table is pretty simple.  Line 0 is the value multiplied by
 * 0 and line one is the value multiplied by one.  So on and so forth.
 *
 * The format of the YR, UR and VG look up table is a little different.  The table
 * contains the value multiplied by its position number plus the add
 * for the equation and add of the round adjust.
 *
 * One table UBVRMult is used for both the UB and VR multiplication.
 * The coeficcent that they are being multipied by is the same.
 *
 * ScaleFactor = 08000h
 *    YRMult      (0.257 * ScaleFactor * X) + (16 * ScaleFactor) + (ScaleFactor / 2)
 *    YGMult       0.504 * ScaleFactor * X 
 *    YBMult       0.098 * ScaleFactor * X
 *            
 *    URMult     (-0.148 * ScaleFactor * X) + (128 * ScaleFactor) + (ScaleFactor / 2)
 *    UGMult     -0.291 * ScaleFactor * X
 *    UBVRMult    0.439 * ScaleFactor * X
 *            
 *    VGMult     (-0.368 * ScaleFactor * X) + (128 * ScaleFactor) + (ScaleFactor / 2)
 *    VBMult     -0.071 * ScaleFactor * X
 *
 *
 */


int YRMult[] =
{
   540672,           /* 0   */ 
   549093,           /* 1   */
   557514,           /* 2   */
   565935,           /* 3   */
   574356,           /* 4   */
   582777,           /* 5   */
   591198,           /* 6   */
   599619,           /* 7   */
   608040,           /* 8   */
   616461,           /* 9   */
   624882,           /* 10  */
   633303,           /* 11  */
   641724,           /* 12  */
   650145,           /* 13  */
   658566,           /* 14  */
   666987,           /* 15  */
   675408,           /* 16  */
   683829,           /* 17  */
   692250,           /* 18  */
   700671,           /* 19  */
   709092,           /* 20  */
   717513,           /* 21  */
   725934,           /* 22  */
   734355,           /* 23  */
   742776,           /* 24  */
   751197,           /* 25  */
   759618,           /* 26  */
   768039,           /* 27  */
   776460,           /* 28  */
   784881,           /* 29  */
   793302,           /* 30  */
   801723,           /* 31  */
   810144,           /* 32  */
   818565,           /* 33  */
   826986,           /* 34  */
   835407,           /* 35  */
   843828,           /* 36  */
   852249,           /* 37  */
   860670,           /* 38  */
   869091,           /* 39  */
   877512,           /* 40  */
   885933,           /* 41  */
   894354,           /* 42  */
   902775,           /* 43  */
   911196,           /* 44  */
   919617,           /* 45  */
   928038,           /* 46  */
   936459,           /* 47  */
   944880,           /* 48  */
   953301,           /* 49  */
   961722,           /* 50  */
   970143,           /* 51  */
   978564,           /* 52  */
   986985,           /* 53  */
   995406,           /* 54  */
   1003827,          /* 55  */
   1012248,          /* 56  */
   1020669,          /* 57  */
   1029090,          /* 58  */
   1037511,          /* 59  */
   1045932,          /* 60  */
   1054353,          /* 61  */
   1062774,          /* 62  */
   1071195,          /* 63  */
   1079616,          /* 64  */
   1088037,          /* 65  */
   1096458,          /* 66  */
   1104879,          /* 67  */
   1113300,          /* 68  */
   1121721,          /* 69  */
   1130142,          /* 70  */
   1138563,          /* 71  */
   1146984,          /* 72  */
   1155405,          /* 73  */
   1163826,          /* 74  */
   1172247,          /* 75  */
   1180668,          /* 76  */
   1189089,          /* 77  */
   1197510,          /* 78  */
   1205931,          /* 79  */
   1214352,          /* 80  */
   1222773,          /* 81  */
   1231194,          /* 82  */
   1239615,          /* 83  */
   1248036,          /* 84  */
   1256457,          /* 85  */
   1264878,          /* 86  */
   1273299,          /* 87  */
   1281720,          /* 88  */
   1290141,          /* 89  */
   1298562,          /* 90  */
   1306983,          /* 91  */
   1315404,          /* 92  */
   1323825,          /* 93  */
   1332246,          /* 94  */
   1340667,          /* 95  */
   1349088,          /* 96  */
   1357509,          /* 97  */
   1365930,          /* 98  */
   1374351,          /* 99  */
   1382772,          /* 100 */
   1391193,          /* 101 */
   1399614,          /* 102 */
   1408035,          /* 103 */
   1416456,          /* 104 */
   1424877,          /* 105 */
   1433298,          /* 106 */
   1441719,          /* 107 */
   1450140,          /* 108 */
   1458561,          /* 109 */
   1466982,          /* 110 */
   1475403,          /* 111 */
   1483824,          /* 112 */
   1492245,          /* 113 */
   1500666,          /* 114 */
   1509087,          /* 115 */
   1517508,          /* 116 */
   1525929,          /* 117 */
   1534350,          /* 118 */
   1542771,          /* 119 */
   1551192,          /* 120 */
   1559613,          /* 121 */
   1568034,          /* 122 */
   1576455,          /* 123 */
   1584876,          /* 124 */
   1593297,          /* 125 */
   1601718,          /* 126 */
   1610139,          /* 127 */
   1618560,          /* 128 */
   1626981,          /* 129 */
   1635402,          /* 130 */
   1643823,          /* 131 */
   1652244,          /* 132 */
   1660665,          /* 133 */
   1669086,          /* 134 */
   1677507,          /* 135 */
   1685928,          /* 136 */
   1694349,          /* 137 */
   1702770,          /* 138 */
   1711191,          /* 139 */
   1719612,          /* 140 */
   1728033,          /* 141 */
   1736454,          /* 142 */
   1744875,          /* 143 */
   1753296,          /* 144 */
   1761717,          /* 145 */
   1770138,          /* 146 */
   1778559,          /* 147 */
   1786980,          /* 148 */
   1795401,          /* 149 */
   1803822,          /* 150 */
   1812243,          /* 151 */
   1820664,          /* 152 */
   1829085,          /* 153 */
   1837506,          /* 154 */
   1845927,          /* 155 */
   1854348,          /* 156 */
   1862769,          /* 157 */
   1871190,          /* 158 */
   1879611,          /* 159 */
   1888032,          /* 160 */
   1896453,          /* 161 */
   1904874,          /* 162 */
   1913295,          /* 163 */
   1921716,          /* 164 */
   1930137,          /* 165 */
   1938558,          /* 166 */
   1946979,          /* 167 */
   1955400,          /* 168 */
   1963821,          /* 169 */
   1972242,          /* 170 */
   1980663,          /* 171 */
   1989084,          /* 172 */
   1997505,          /* 173 */
   2005926,          /* 174 */
   2014347,          /* 175 */
   2022768,          /* 176 */
   2031189,          /* 177 */
   2039610,          /* 178 */
   2048031,          /* 179 */
   2056452,          /* 180 */
   2064873,          /* 181 */
   2073294,          /* 182 */
   2081715,          /* 183 */
   2090136,          /* 184 */
   2098557,          /* 185 */
   2106978,          /* 186 */
   2115399,          /* 187 */
   2123820,          /* 188 */
   2132241,          /* 189 */
   2140662,          /* 190 */
   2149083,          /* 191 */
   2157504,          /* 192 */
   2165925,          /* 193 */
   2174346,          /* 194 */
   2182767,          /* 195 */
   2191188,          /* 196 */
   2199609,          /* 197 */
   2208030,          /* 198 */
   2216451,          /* 199 */
   2224872,          /* 200 */
   2233293,          /* 201 */
   2241714,          /* 202 */
   2250135,          /* 203 */
   2258556,          /* 204 */
   2266977,          /* 205 */
   2275398,          /* 206 */
   2283819,          /* 207 */
   2292240,          /* 208 */
   2300661,          /* 209 */
   2309082,          /* 210 */
   2317503,          /* 211 */
   2325924,          /* 212 */
   2334345,          /* 213 */
   2342766,          /* 214 */
   2351187,          /* 215 */
   2359608,          /* 216 */
   2368029,          /* 217 */
   2376450,          /* 218 */
   2384871,          /* 219 */
   2393292,          /* 220 */
   2401713,          /* 221 */
   2410134,          /* 222 */
   2418555,          /* 223 */
   2426976,          /* 224 */
   2435397,          /* 225 */
   2443818,          /* 226 */
   2452239,          /* 227 */
   2460660,          /* 228 */
   2469081,          /* 229 */
   2477502,          /* 230 */
   2485923,          /* 231 */
   2494344,          /* 232 */
   2502765,          /* 233 */
   2511186,          /* 234 */
   2519607,          /* 235 */
   2528028,          /* 236 */
   2536449,          /* 237 */
   2544870,          /* 238 */
   2553291,          /* 239 */
   2561712,          /* 240 */
   2570133,          /* 241 */
   2578554,          /* 242 */
   2586975,          /* 243 */
   2595396,          /* 244 */
   2603817,          /* 245 */
   2612238,          /* 246 */
   2620659,          /* 247 */
   2629080,          /* 248 */
   2637501,          /* 249 */
   2645922,          /* 250 */
   2654343,          /* 251 */
   2662764,          /* 252 */
   2671185,          /* 253 */
   2679606,          /* 254 */
   2688027           /* 255 */
};
            
            
int   YGMult[] =
{
   0,                /* 0   */
   16515,            /* 1   */
   33030,            /* 2   */
   49545,            /* 3   */
   66060,            /* 4   */
   82575,            /* 5   */
   99090,            /* 6   */
   115605,           /* 7   */
   132120,           /* 8   */
   148635,           /* 9   */
   165150,           /* 10  */
   181665,           /* 11  */
   198180,           /* 12  */
   214695,           /* 13  */
   231210,           /* 14  */
   247725,           /* 15  */
   264240,           /* 16  */
   280755,           /* 17  */
   297270,           /* 18  */
   313785,           /* 19  */
   330300,           /* 20  */
   346815,           /* 21  */
   363330,           /* 22  */
   379845,           /* 23  */
   396360,           /* 24  */
   412875,           /* 25  */
   429390,           /* 26  */
   445905,           /* 27  */
   462420,           /* 28  */
   478935,           /* 29  */
   495450,           /* 30  */
   511965,           /* 31  */
   528480,           /* 32  */
   544995,           /* 33  */
   561510,           /* 34  */
   578025,           /* 35  */
   594540,           /* 36  */
   611055,           /* 37  */
   627570,           /* 38  */
   644085,           /* 39  */
   660600,           /* 40  */
   677115,           /* 41  */
   693630,           /* 42  */
   710145,           /* 43  */
   726660,           /* 44  */
   743175,           /* 45  */
   759690,           /* 46  */
   776205,           /* 47  */
   792720,           /* 48  */
   809235,           /* 49  */
   825750,           /* 50  */
   842265,           /* 51  */
   858780,           /* 52  */
   875295,           /* 53  */
   891810,           /* 54  */
   908325,           /* 55  */
   924840,           /* 56  */
   941355,           /* 57  */
   957870,           /* 58  */
   974385,           /* 59  */
   990900,           /* 60  */
   1007415,          /* 61  */
   1023930,          /* 62  */
   1040445,          /* 63  */
   1056960,          /* 64  */
   1073475,          /* 65  */
   1089990,          /* 66  */
   1106505,          /* 67  */
   1123020,          /* 68  */
   1139535,          /* 69  */
   1156050,          /* 70  */
   1172565,          /* 71  */
   1189080,          /* 72  */
   1205595,          /* 73  */
   1222110,          /* 74  */
   1238625,          /* 75  */
   1255140,          /* 76  */
   1271655,          /* 77  */
   1288170,          /* 78  */
   1304685,          /* 79  */
   1321200,          /* 80  */
   1337715,          /* 81  */
   1354230,          /* 82  */
   1370745,          /* 83  */
   1387260,          /* 84  */
   1403775,          /* 85  */
   1420290,          /* 86  */
   1436805,          /* 87  */
   1453320,          /* 88  */
   1469835,          /* 89  */
   1486350,          /* 90  */
   1502865,          /* 91  */
   1519380,          /* 92  */
   1535895,          /* 93  */
   1552410,          /* 94  */
   1568925,          /* 95  */
   1585440,          /* 96  */
   1601955,          /* 97  */
   1618470,          /* 98  */
   1634985,          /* 99  */
   1651500,          /* 100 */
   1668015,          /* 101 */
   1684530,          /* 102 */
   1701045,          /* 103 */
   1717560,          /* 104 */
   1734075,          /* 105 */
   1750590,          /* 106 */
   1767105,          /* 107 */
   1783620,          /* 108 */
   1800135,          /* 109 */
   1816650,          /* 110 */
   1833165,          /* 111 */
   1849680,          /* 112 */
   1866195,          /* 113 */
   1882710,          /* 114 */
   1899225,          /* 115 */
   1915740,          /* 116 */
   1932255,          /* 117 */
   1948770,          /* 118 */
   1965285,          /* 119 */
   1981800,          /* 120 */
   1998315,          /* 121 */
   2014830,          /* 122 */
   2031345,          /* 123 */
   2047860,          /* 124 */
   2064375,          /* 125 */
   2080890,          /* 126 */
   2097405,          /* 127 */
   2113920,          /* 128 */
   2130435,          /* 129 */
   2146950,          /* 130 */
   2163465,          /* 131 */
   2179980,          /* 132 */
   2196495,          /* 133 */
   2213010,          /* 134 */
   2229525,          /* 135 */
   2246040,          /* 136 */
   2262555,          /* 137 */
   2279070,          /* 138 */
   2295585,          /* 139 */
   2312100,          /* 140 */
   2328615,          /* 141 */
   2345130,          /* 142 */
   2361645,          /* 143 */
   2378160,          /* 144 */
   2394675,          /* 145 */
   2411190,          /* 146 */
   2427705,          /* 147 */
   2444220,          /* 148 */
   2460735,          /* 149 */
   2477250,          /* 150 */
   2493765,          /* 151 */
   2510280,          /* 152 */
   2526795,          /* 153 */
   2543310,          /* 154 */
   2559825,          /* 155 */
   2576340,          /* 156 */
   2592855,          /* 157 */
   2609370,          /* 158 */
   2625885,          /* 159 */
   2642400,          /* 160 */
   2658915,          /* 161 */
   2675430,          /* 162 */
   2691945,          /* 163 */
   2708460,          /* 164 */
   2724975,          /* 165 */
   2741490,          /* 166 */
   2758005,          /* 167 */
   2774520,          /* 168 */
   2791035,          /* 169 */
   2807550,          /* 170 */
   2824065,          /* 171 */
   2840580,          /* 172 */
   2857095,          /* 173 */
   2873610,          /* 174 */
   2890125,          /* 175 */
   2906640,          /* 176 */
   2923155,          /* 177 */
   2939670,          /* 178 */
   2956185,          /* 179 */
   2972700,          /* 180 */
   2989215,          /* 181 */
   3005730,          /* 182 */
   3022245,          /* 183 */
   3038760,          /* 184 */
   3055275,          /* 185 */
   3071790,          /* 186 */
   3088305,          /* 187 */
   3104820,          /* 188 */
   3121335,          /* 189 */
   3137850,          /* 190 */
   3154365,          /* 191 */
   3170880,          /* 192 */
   3187395,          /* 193 */
   3203910,          /* 194 */
   3220425,          /* 195 */
   3236940,          /* 196 */
   3253455,          /* 197 */
   3269970,          /* 198 */
   3286485,          /* 199 */
   3303000,          /* 200 */
   3319515,          /* 201 */
   3336030,          /* 202 */
   3352545,          /* 203 */
   3369060,          /* 204 */
   3385575,          /* 205 */
   3402090,          /* 206 */
   3418605,          /* 207 */
   3435120,          /* 208 */
   3451635,          /* 209 */
   3468150,          /* 210 */
   3484665,          /* 211 */
   3501180,          /* 212 */
   3517695,          /* 213 */
   3534210,          /* 214 */
   3550725,          /* 215 */
   3567240,          /* 216 */
   3583755,          /* 217 */
   3600270,          /* 218 */
   3616785,          /* 219 */
   3633300,          /* 220 */
   3649815,          /* 221 */
   3666330,          /* 222 */
   3682845,          /* 223 */
   3699360,          /* 224 */
   3715875,          /* 225 */
   3732390,          /* 226 */
   3748905,          /* 227 */
   3765420,          /* 228 */
   3781935,          /* 229 */
   3798450,          /* 230 */
   3814965,          /* 231 */
   3831480,          /* 232 */
   3847995,          /* 233 */
   3864510,          /* 234 */
   3881025,          /* 235 */
   3897540,          /* 236 */
   3914055,          /* 237 */
   3930570,          /* 238 */
   3947085,          /* 239 */
   3963600,          /* 240 */
   3980115,          /* 241 */
   3996630,          /* 242 */
   4013145,          /* 243 */
   4029660,          /* 244 */
   4046175,          /* 245 */
   4062690,          /* 246 */
   4079205,          /* 247 */
   4095720,          /* 248 */
   4112235,          /* 249 */
   4128750,          /* 250 */
   4145265,          /* 251 */
   4161780,          /* 252 */
   4178295,          /* 253 */
   4194810,          /* 254 */
   4211325           /* 255 */
};
            
            
int   YBMult[] =
{
   0,                /* 0   */
   3211,             /* 1   */
   6422,             /* 2   */
   9633,             /* 3   */
   12844,            /* 4   */
   16055,            /* 5   */
   19266,            /* 6   */
   22477,            /* 7   */
   25688,            /* 8   */
   28899,            /* 9   */
   32110,            /* 10  */
   35321,            /* 11  */
   38532,            /* 12  */
   41743,            /* 13  */
   44954,            /* 14  */
   48165,            /* 15  */
   51376,            /* 16  */
   54587,            /* 17  */
   57798,            /* 18  */
   61009,            /* 19  */
   64220,            /* 20  */
   67431,            /* 21  */
   70642,            /* 22  */
   73853,            /* 23  */
   77064,            /* 24  */
   80275,            /* 25  */
   83486,            /* 26  */
   86697,            /* 27  */
   89908,            /* 28  */
   93119,            /* 29  */
   96330,            /* 30  */
   99541,            /* 31  */
   102752,           /* 32  */
   105963,           /* 33  */
   109174,           /* 34  */
   112385,           /* 35  */
   115596,           /* 36  */
   118807,           /* 37  */
   122018,           /* 38  */
   125229,           /* 39  */
   128440,           /* 40  */
   131651,           /* 41  */
   134862,           /* 42  */
   138073,           /* 43  */
   141284,           /* 44  */
   144495,           /* 45  */
   147706,           /* 46  */
   150917,           /* 47  */
   154128,           /* 48  */
   157339,           /* 49  */
   160550,           /* 50  */
   163761,           /* 51  */
   166972,           /* 52  */
   170183,           /* 53  */
   173394,           /* 54  */
   176605,           /* 55  */
   179816,           /* 56  */
   183027,           /* 57  */
   186238,           /* 58  */
   189449,           /* 59  */
   192660,           /* 60  */
   195871,           /* 61  */
   199082,           /* 62  */
   202293,           /* 63  */
   205504,           /* 64  */
   208715,           /* 65  */
   211926,           /* 66  */
   215137,           /* 67  */
   218348,           /* 68  */
   221559,           /* 69  */
   224770,           /* 70  */
   227981,           /* 71  */
   231192,           /* 72  */
   234403,           /* 73  */
   237614,           /* 74  */
   240825,           /* 75  */
   244036,           /* 76  */
   247247,           /* 77  */
   250458,           /* 78  */
   253669,           /* 79  */
   256880,           /* 80  */
   260091,           /* 81  */
   263302,           /* 82  */
   266513,           /* 83  */
   269724,           /* 84  */
   272935,           /* 85  */
   276146,           /* 86  */
   279357,           /* 87  */
   282568,           /* 88  */
   285779,           /* 89  */
   288990,           /* 90  */
   292201,           /* 91  */
   295412,           /* 92  */
   298623,           /* 93  */
   301834,           /* 94  */
   305045,           /* 95  */
   308256,           /* 96  */
   311467,           /* 97  */
   314678,           /* 98  */
   317889,           /* 99  */
   321100,           /* 100 */
   324311,           /* 101 */
   327522,           /* 102 */
   330733,           /* 103 */
   333944,           /* 104 */
   337155,           /* 105 */
   340366,           /* 106 */
   343577,           /* 107 */
   346788,           /* 108 */
   349999,           /* 109 */
   353210,           /* 110 */
   356421,           /* 111 */
   359632,           /* 112 */
   362843,           /* 113 */
   366054,           /* 114 */
   369265,           /* 115 */
   372476,           /* 116 */
   375687,           /* 117 */
   378898,           /* 118 */
   382109,           /* 119 */
   385320,           /* 120 */
   388531,           /* 121 */
   391742,           /* 122 */
   394953,           /* 123 */
   398164,           /* 124 */
   401375,           /* 125 */
   404586,           /* 126 */
   407797,           /* 127 */
   411008,           /* 128 */
   414219,           /* 129 */
   417430,           /* 130 */
   420641,           /* 131 */
   423852,           /* 132 */
   427063,           /* 133 */
   430274,           /* 134 */
   433485,           /* 135 */
   436696,           /* 136 */
   439907,           /* 137 */
   443118,           /* 138 */
   446329,           /* 139 */
   449540,           /* 140 */
   452751,           /* 141 */
   455962,           /* 142 */
   459173,           /* 143 */
   462384,           /* 144 */
   465595,           /* 145 */
   468806,           /* 146 */
   472017,           /* 147 */
   475228,           /* 148 */
   478439,           /* 149 */
   481650,           /* 150 */
   484861,           /* 151 */
   488072,           /* 152 */
   491283,           /* 153 */
   494494,           /* 154 */
   497705,           /* 155 */
   500916,           /* 156 */
   504127,           /* 157 */
   507338,           /* 158 */
   510549,           /* 159 */
   513760,           /* 160 */
   516971,           /* 161 */
   520182,           /* 162 */
   523393,           /* 163 */
   526604,           /* 164 */
   529815,           /* 165 */
   533026,           /* 166 */
   536237,           /* 167 */
   539448,           /* 168 */
   542659,           /* 169 */
   545870,           /* 170 */
   549081,           /* 171 */
   552292,           /* 172 */
   555503,           /* 173 */
   558714,           /* 174 */
   561925,           /* 175 */
   565136,           /* 176 */
   568347,           /* 177 */
   571558,           /* 178 */
   574769,           /* 179 */
   577980,           /* 180 */
   581191,           /* 181 */
   584402,           /* 182 */
   587613,           /* 183 */
   590824,           /* 184 */
   594035,           /* 185 */
   597246,           /* 186 */
   600457,           /* 187 */
   603668,           /* 188 */
   606879,           /* 189 */
   610090,           /* 190 */
   613301,           /* 191 */
   616512,           /* 192 */
   619723,           /* 193 */
   622934,           /* 194 */
   626145,           /* 195 */
   629356,           /* 196 */
   632567,           /* 197 */
   635778,           /* 198 */
   638989,           /* 199 */
   642200,           /* 200 */
   645411,           /* 201 */
   648622,           /* 202 */
   651833,           /* 203 */
   655044,           /* 204 */
   658255,           /* 205 */
   661466,           /* 206 */
   664677,           /* 207 */
   667888,           /* 208 */
   671099,           /* 209 */
   674310,           /* 210 */
   677521,           /* 211 */
   680732,           /* 212 */
   683943,           /* 213 */
   687154,           /* 214 */
   690365,           /* 215 */
   693576,           /* 216 */
   696787,           /* 217 */
   699998,           /* 218 */
   703209,           /* 219 */
   706420,           /* 220 */
   709631,           /* 221 */
   712842,           /* 222 */
   716053,           /* 223 */
   719264,           /* 224 */
   722475,           /* 225 */
   725686,           /* 226 */
   728897,           /* 227 */
   732108,           /* 228 */
   735319,           /* 229 */
   738530,           /* 230 */
   741741,           /* 231 */
   744952,           /* 232 */
   748163,           /* 233 */
   751374,           /* 234 */
   754585,           /* 235 */
   757796,           /* 236 */
   761007,           /* 237 */
   764218,           /* 238 */
   767429,           /* 239 */
   770640,           /* 240 */
   773851,           /* 241 */
   777062,           /* 242 */
   780273,           /* 243 */
   783484,           /* 244 */
   786695,           /* 245 */
   789906,           /* 246 */
   793117,           /* 247 */
   796328,           /* 248 */
   799539,           /* 249 */
   802750,           /* 250 */
   805961,           /* 251 */
   809172,           /* 252 */
   812383,           /* 253 */
   815594,           /* 254 */
   818805            /* 255 */
};
            
int   URMult[] =
{
   4210688,          /* 0   */
   4205839,          /* 1   */
   4200990,          /* 2   */
   4196141,          /* 3   */
   4191292,          /* 4   */
   4186443,          /* 5   */
   4181594,          /* 6   */
   4176745,          /* 7   */
   4171896,          /* 8   */
   4167047,          /* 9   */
   4162198,          /* 10  */
   4157349,          /* 11  */
   4152500,          /* 12  */
   4147651,          /* 13  */
   4142802,          /* 14  */
   4137953,          /* 15  */
   4133104,          /* 16  */
   4128255,          /* 17  */
   4123406,          /* 18  */
   4118557,          /* 19  */
   4113708,          /* 20  */
   4108859,          /* 21  */
   4104010,          /* 22  */
   4099161,          /* 23  */
   4094312,          /* 24  */
   4089463,          /* 25  */
   4084614,          /* 26  */
   4079765,          /* 27  */
   4074916,          /* 28  */
   4070067,          /* 29  */
   4065218,          /* 30  */
   4060369,          /* 31  */
   4055520,          /* 32  */
   4050671,          /* 33  */
   4045822,          /* 34  */
   4040973,          /* 35  */
   4036124,          /* 36  */
   4031275,          /* 37  */
   4026426,          /* 38  */
   4021577,          /* 39  */
   4016728,          /* 40  */
   4011879,          /* 41  */
   4007030,          /* 42  */
   4002181,          /* 43  */
   3997332,          /* 44  */
   3992483,          /* 45  */
   3987634,          /* 46  */
   3982785,          /* 47  */
   3977936,          /* 48  */
   3973087,          /* 49  */
   3968238,          /* 50  */
   3963389,          /* 51  */
   3958540,          /* 52  */
   3953691,          /* 53  */
   3948842,          /* 54  */
   3943993,          /* 55  */
   3939144,          /* 56  */
   3934295,          /* 57  */
   3929446,          /* 58  */
   3924597,          /* 59  */
   3919748,          /* 60  */
   3914899,          /* 61  */
   3910050,          /* 62  */
   3905201,          /* 63  */
   3900352,          /* 64  */
   3895503,          /* 65  */
   3890654,          /* 66  */
   3885805,          /* 67  */
   3880956,          /* 68  */
   3876107,          /* 69  */
   3871258,          /* 70  */
   3866409,          /* 71  */
   3861560,          /* 72  */
   3856711,          /* 73  */
   3851862,          /* 74  */
   3847013,          /* 75  */
   3842164,          /* 76  */
   3837315,          /* 77  */
   3832466,          /* 78  */
   3827617,          /* 79  */
   3822768,          /* 80  */
   3817919,          /* 81  */
   3813070,          /* 82  */
   3808221,          /* 83  */
   3803372,          /* 84  */
   3798523,          /* 85  */
   3793674,          /* 86  */
   3788825,          /* 87  */
   3783976,          /* 88  */
   3779127,          /* 89  */
   3774278,          /* 90  */
   3769429,          /* 91  */
   3764580,          /* 92  */
   3759731,          /* 93  */
   3754882,          /* 94  */
   3750033,          /* 95  */
   3745184,          /* 96  */
   3740335,          /* 97  */
   3735486,          /* 98  */
   3730637,          /* 99  */
   3725788,          /* 100 */
   3720939,          /* 101 */
   3716090,          /* 102 */
   3711241,          /* 103 */
   3706392,          /* 104 */
   3701543,          /* 105 */
   3696694,          /* 106 */
   3691845,          /* 107 */
   3686996,          /* 108 */
   3682147,          /* 109 */
   3677298,          /* 110 */
   3672449,          /* 111 */
   3667600,          /* 112 */
   3662751,          /* 113 */
   3657902,          /* 114 */
   3653053,          /* 115 */
   3648204,          /* 116 */
   3643355,          /* 117 */
   3638506,          /* 118 */
   3633657,          /* 119 */
   3628808,          /* 120 */
   3623959,          /* 121 */
   3619110,          /* 122 */
   3614261,          /* 123 */
   3609412,          /* 124 */
   3604563,          /* 125 */
   3599714,          /* 126 */
   3594865,          /* 127 */
   3590016,          /* 128 */
   3585167,          /* 129 */
   3580318,          /* 130 */
   3575469,          /* 131 */
   3570620,          /* 132 */
   3565771,          /* 133 */
   3560922,          /* 134 */
   3556073,          /* 135 */
   3551224,          /* 136 */
   3546375,          /* 137 */
   3541526,          /* 138 */
   3536677,          /* 139 */
   3531828,          /* 140 */
   3526979,          /* 141 */
   3522130,          /* 142 */
   3517281,          /* 143 */
   3512432,          /* 144 */
   3507583,          /* 145 */
   3502734,          /* 146 */
   3497885,          /* 147 */
   3493036,          /* 148 */
   3488187,          /* 149 */
   3483338,          /* 150 */
   3478489,          /* 151 */
   3473640,          /* 152 */
   3468791,          /* 153 */
   3463942,          /* 154 */
   3459093,          /* 155 */
   3454244,          /* 156 */
   3449395,          /* 157 */
   3444546,          /* 158 */
   3439697,          /* 159 */
   3434848,          /* 160 */
   3429999,          /* 161 */
   3425150,          /* 162 */
   3420301,          /* 163 */
   3415452,          /* 164 */
   3410603,          /* 165 */
   3405754,          /* 166 */
   3400905,          /* 167 */
   3396056,          /* 168 */
   3391207,          /* 169 */
   3386358,          /* 170 */
   3381509,          /* 171 */
   3376660,          /* 172 */
   3371811,          /* 173 */
   3366962,          /* 174 */
   3362113,          /* 175 */
   3357264,          /* 176 */
   3352415,          /* 177 */
   3347566,          /* 178 */
   3342717,          /* 179 */
   3337868,          /* 180 */
   3333019,          /* 181 */
   3328170,          /* 182 */
   3323321,          /* 183 */
   3318472,          /* 184 */
   3313623,          /* 185 */
   3308774,          /* 186 */
   3303925,          /* 187 */
   3299076,          /* 188 */
   3294227,          /* 189 */
   3289378,          /* 190 */
   3284529,          /* 191 */
   3279680,          /* 192 */
   3274831,          /* 193 */
   3269982,          /* 194 */
   3265133,          /* 195 */
   3260284,          /* 196 */
   3255435,          /* 197 */
   3250586,          /* 198 */
   3245737,          /* 199 */
   3240888,          /* 200 */
   3236039,          /* 201 */
   3231190,          /* 202 */
   3226341,          /* 203 */
   3221492,          /* 204 */
   3216643,          /* 205 */
   3211794,          /* 206 */
   3206945,          /* 207 */
   3202096,          /* 208 */
   3197247,          /* 209 */
   3192398,          /* 210 */
   3187549,          /* 211 */
   3182700,          /* 212 */
   3177851,          /* 213 */
   3173002,          /* 214 */
   3168153,          /* 215 */
   3163304,          /* 216 */
   3158455,          /* 217 */
   3153606,          /* 218 */
   3148757,          /* 219 */
   3143908,          /* 220 */
   3139059,          /* 221 */
   3134210,          /* 222 */
   3129361,          /* 223 */
   3124512,          /* 224 */
   3119663,          /* 225 */
   3114814,          /* 226 */
   3109965,          /* 227 */
   3105116,          /* 228 */
   3100267,          /* 229 */
   3095418,          /* 230 */
   3090569,          /* 231 */
   3085720,          /* 232 */
   3080871,          /* 233 */
   3076022,          /* 234 */
   3071173,          /* 235 */
   3066324,          /* 236 */
   3061475,          /* 237 */
   3056626,          /* 238 */
   3051777,          /* 239 */
   3046928,          /* 240 */
   3042079,          /* 241 */
   3037230,          /* 242 */
   3032381,          /* 243 */
   3027532,          /* 244 */
   3022683,          /* 245 */
   3017834,          /* 246 */
   3012985,          /* 247 */
   3008136,          /* 248 */
   3003287,          /* 249 */
   2998438,          /* 250 */
   2993589,          /* 251 */
   2988740,          /* 252 */
   2983891,          /* 253 */
   2979042,          /* 254 */
   2974193           /* 255 */
};
            
int   UGMult[] =
{
   0,              /* 0   */
   -9535,          /* 1   */
   -19070,         /* 2   */
   -28605,         /* 3   */
   -38140,         /* 4   */
   -47675,         /* 5   */
   -57210,         /* 6   */
   -66745,         /* 7   */
   -76280,         /* 8   */
   -85815,         /* 9   */
   -95350,         /* 10  */
   -104885,        /* 11  */
   -114420,        /* 12  */
   -123955,        /* 13  */
   -133490,        /* 14  */
   -143025,        /* 15  */
   -152560,        /* 16  */
   -162095,        /* 17  */
   -171630,        /* 18  */
   -181165,        /* 19  */
   -190700,        /* 20  */
   -200235,        /* 21  */
   -209770,        /* 22  */
   -219305,        /* 23  */
   -228840,        /* 24  */
   -238375,        /* 25  */
   -247910,        /* 26  */
   -257445,        /* 27  */
   -266980,        /* 28  */
   -276515,        /* 29  */
   -286050,        /* 30  */
   -295585,        /* 31  */
   -305120,        /* 32  */
   -314655,        /* 33  */
   -324190,        /* 34  */
   -333725,        /* 35  */
   -343260,        /* 36  */
   -352795,        /* 37  */
   -362330,        /* 38  */
   -371865,        /* 39  */
   -381400,        /* 40  */
   -390935,        /* 41  */
   -400470,        /* 42  */
   -410005,        /* 43  */
   -419540,        /* 44  */
   -429075,        /* 45  */
   -438610,        /* 46  */
   -448145,        /* 47  */
   -457680,        /* 48  */
   -467215,        /* 49  */
   -476750,        /* 50  */
   -486285,        /* 51  */
   -495820,        /* 52  */
   -505355,        /* 53  */
   -514890,        /* 54  */
   -524425,        /* 55  */
   -533960,        /* 56  */
   -543495,        /* 57  */
   -553030,        /* 58  */
   -562565,        /* 59  */
   -572100,        /* 60  */
   -581635,        /* 61  */
   -591170,        /* 62  */
   -600705,        /* 63  */
   -610240,        /* 64  */
   -619775,        /* 65  */
   -629310,        /* 66  */
   -638845,        /* 67  */
   -648380,        /* 68  */
   -657915,        /* 69  */
   -667450,        /* 70  */
   -676985,        /* 71  */
   -686520,        /* 72  */
   -696055,        /* 73  */
   -705590,        /* 74  */
   -715125,        /* 75  */
   -724660,        /* 76  */
   -734195,        /* 77  */
   -743730,        /* 78  */
   -753265,        /* 79  */
   -762800,        /* 80  */
   -772335,        /* 81  */
   -781870,        /* 82  */
   -791405,        /* 83  */
   -800940,        /* 84  */
   -810475,        /* 85  */
   -820010,        /* 86  */
   -829545,        /* 87  */
   -839080,        /* 88  */
   -848615,        /* 89  */
   -858150,        /* 90  */
   -867685,        /* 91  */
   -877220,        /* 92  */
   -886755,        /* 93  */
   -896290,        /* 94  */
   -905825,        /* 95  */
   -915360,        /* 96  */
   -924895,        /* 97  */
   -934430,        /* 98  */
   -943965,        /* 99  */
   -953500,        /* 100 */
   -963035,        /* 101 */
   -972570,        /* 102 */
   -982105,        /* 103 */
   -991640,        /* 104 */
   -1001175,       /* 105 */
   -1010710,       /* 106 */
   -1020245,       /* 107 */
   -1029780,       /* 108 */
   -1039315,       /* 109 */
   -1048850,       /* 110 */
   -1058385,       /* 111 */
   -1067920,       /* 112 */
   -1077455,       /* 113 */
   -1086990,       /* 114 */
   -1096525,       /* 115 */
   -1106060,       /* 116 */
   -1115595,       /* 117 */
   -1125130,       /* 118 */
   -1134665,       /* 119 */
   -1144200,       /* 120 */
   -1153735,       /* 121 */
   -1163270,       /* 122 */
   -1172805,       /* 123 */
   -1182340,       /* 124 */
   -1191875,       /* 125 */
   -1201410,       /* 126 */
   -1210945,       /* 127 */
   -1220480,       /* 128 */
   -1230015,       /* 129 */
   -1239550,       /* 130 */
   -1249085,       /* 131 */
   -1258620,       /* 132 */
   -1268155,       /* 133 */
   -1277690,       /* 134 */
   -1287225,       /* 135 */
   -1296760,       /* 136 */
   -1306295,       /* 137 */
   -1315830,       /* 138 */
   -1325365,       /* 139 */
   -1334900,       /* 140 */
   -1344435,       /* 141 */
   -1353970,       /* 142 */
   -1363505,       /* 143 */
   -1373040,       /* 144 */
   -1382575,       /* 145 */
   -1392110,       /* 146 */
   -1401645,       /* 147 */
   -1411180,       /* 148 */
   -1420715,       /* 149 */
   -1430250,       /* 150 */
   -1439785,       /* 151 */
   -1449320,       /* 152 */
   -1458855,       /* 153 */
   -1468390,       /* 154 */
   -1477925,       /* 155 */
   -1487460,       /* 156 */
   -1496995,       /* 157 */
   -1506530,       /* 158 */
   -1516065,       /* 159 */
   -1525600,       /* 160 */
   -1535135,       /* 161 */
   -1544670,       /* 162 */
   -1554205,       /* 163 */
   -1563740,       /* 164 */
   -1573275,       /* 165 */
   -1582810,       /* 166 */
   -1592345,       /* 167 */
   -1601880,       /* 168 */
   -1611415,       /* 169 */
   -1620950,       /* 170 */
   -1630485,       /* 171 */
   -1640020,       /* 172 */
   -1649555,       /* 173 */
   -1659090,       /* 174 */
   -1668625,       /* 175 */
   -1678160,       /* 176 */
   -1687695,       /* 177 */
   -1697230,       /* 178 */
   -1706765,       /* 179 */
   -1716300,       /* 180 */
   -1725835,       /* 181 */
   -1735370,       /* 182 */
   -1744905,       /* 183 */
   -1754440,       /* 184 */
   -1763975,       /* 185 */
   -1773510,       /* 186 */
   -1783045,       /* 187 */
   -1792580,       /* 188 */
   -1802115,       /* 189 */
   -1811650,       /* 190 */
   -1821185,       /* 191 */
   -1830720,       /* 192 */
   -1840255,       /* 193 */
   -1849790,       /* 194 */
   -1859325,       /* 195 */
   -1868860,       /* 196 */
   -1878395,       /* 197 */
   -1887930,       /* 198 */
   -1897465,       /* 199 */
   -1907000,       /* 200 */
   -1916535,       /* 201 */
   -1926070,       /* 202 */
   -1935605,       /* 203 */
   -1945140,       /* 204 */
   -1954675,       /* 205 */
   -1964210,       /* 206 */
   -1973745,       /* 207 */
   -1983280,       /* 208 */
   -1992815,       /* 209 */
   -2002350,       /* 210 */
   -2011885,       /* 211 */
   -2021420,       /* 212 */
   -2030955,       /* 213 */
   -2040490,       /* 214 */
   -2050025,       /* 215 */
   -2059560,       /* 216 */
   -2069095,       /* 217 */
   -2078630,       /* 218 */
   -2088165,       /* 219 */
   -2097700,       /* 220 */
   -2107235,       /* 221 */
   -2116770,       /* 222 */
   -2126305,       /* 223 */
   -2135840,       /* 224 */
   -2145375,       /* 225 */
   -2154910,       /* 226 */
   -2164445,       /* 227 */
   -2173980,       /* 228 */
   -2183515,       /* 229 */
   -2193050,       /* 230 */
   -2202585,       /* 231 */
   -2212120,       /* 232 */
   -2221655,       /* 233 */
   -2231190,       /* 234 */
   -2240725,       /* 235 */
   -2250260,       /* 236 */
   -2259795,       /* 237 */
   -2269330,       /* 238 */
   -2278865,       /* 239 */
   -2288400,       /* 240 */
   -2297935,       /* 241 */
   -2307470,       /* 242 */
   -2317005,       /* 243 */
   -2326540,       /* 244 */
   -2336075,       /* 245 */
   -2345610,       /* 246 */
   -2355145,       /* 247 */
   -2364680,       /* 248 */
   -2374215,       /* 249 */
   -2383750,       /* 250 */
   -2393285,       /* 251 */
   -2402820,       /* 252 */
   -2412355,       /* 253 */
   -2421890,       /* 254 */
   -2431425        /* 255 */
};
            
            
int   UBVRMult[] =
{
   0,                /* 0   */
   14385,            /* 1   */
   28770,            /* 2   */
   43155,            /* 3   */
   57540,            /* 4   */
   71925,            /* 5   */
   86310,            /* 6   */
   100695,           /* 7   */
   115080,           /* 8   */
   129465,           /* 9   */
   143850,           /* 10  */
   158235,           /* 11  */
   172620,           /* 12  */
   187005,           /* 13  */
   201390,           /* 14  */
   215775,           /* 15  */
   230160,           /* 16  */
   244545,           /* 17  */
   258930,           /* 18  */
   273315,           /* 19  */
   287700,           /* 20  */
   302085,           /* 21  */
   316470,           /* 22  */
   330855,           /* 23  */
   345240,           /* 24  */
   359625,           /* 25  */
   374010,           /* 26  */
   388395,           /* 27  */
   402780,           /* 28  */
   417165,           /* 29  */
   431550,           /* 30  */
   445935,           /* 31  */
   460320,           /* 32  */
   474705,           /* 33  */
   489090,           /* 34  */
   503475,           /* 35  */
   517860,           /* 36  */
   532245,           /* 37  */
   546630,           /* 38  */
   561015,           /* 39  */
   575400,           /* 40  */
   589785,           /* 41  */
   604170,           /* 42  */
   618555,           /* 43  */
   632940,           /* 44  */
   647325,           /* 45  */
   661710,           /* 46  */
   676095,           /* 47  */
   690480,           /* 48  */
   704865,           /* 49  */
   719250,           /* 50  */
   733635,           /* 51  */
   748020,           /* 52  */
   762405,           /* 53  */
   776790,           /* 54  */
   791175,           /* 55  */
   805560,           /* 56  */
   819945,           /* 57  */
   834330,           /* 58  */
   848715,           /* 59  */
   863100,           /* 60  */
   877485,           /* 61  */
   891870,           /* 62  */
   906255,           /* 63  */
   920640,           /* 64  */
   935025,           /* 65  */
   949410,           /* 66  */
   963795,           /* 67  */
   978180,           /* 68  */
   992565,           /* 69  */
   1006950,          /* 70  */
   1021335,          /* 71  */
   1035720,          /* 72  */
   1050105,          /* 73  */
   1064490,          /* 74  */
   1078875,          /* 75  */
   1093260,          /* 76  */
   1107645,          /* 77  */
   1122030,          /* 78  */
   1136415,          /* 79  */
   1150800,          /* 80  */
   1165185,          /* 81  */
   1179570,          /* 82  */
   1193955,          /* 83  */
   1208340,          /* 84  */
   1222725,          /* 85  */
   1237110,          /* 86  */
   1251495,          /* 87  */
   1265880,          /* 88  */
   1280265,          /* 89  */
   1294650,          /* 90  */
   1309035,          /* 91  */
   1323420,          /* 92  */
   1337805,          /* 93  */
   1352190,          /* 94  */
   1366575,          /* 95  */
   1380960,          /* 96  */
   1395345,          /* 97  */
   1409730,          /* 98  */
   1424115,          /* 99  */
   1438500,          /* 100 */
   1452885,          /* 101 */
   1467270,          /* 102 */
   1481655,          /* 103 */
   1496040,          /* 104 */
   1510425,          /* 105 */
   1524810,          /* 106 */
   1539195,          /* 107 */
   1553580,          /* 108 */
   1567965,          /* 109 */
   1582350,          /* 110 */
   1596735,          /* 111 */
   1611120,          /* 112 */
   1625505,          /* 113 */
   1639890,          /* 114 */
   1654275,          /* 115 */
   1668660,          /* 116 */
   1683045,          /* 117 */
   1697430,          /* 118 */
   1711815,          /* 119 */
   1726200,          /* 120 */
   1740585,          /* 121 */
   1754970,          /* 122 */
   1769355,          /* 123 */
   1783740,          /* 124 */
   1798125,          /* 125 */
   1812510,          /* 126 */
   1826895,          /* 127 */
   1841280,          /* 128 */
   1855665,          /* 129 */
   1870050,          /* 130 */
   1884435,          /* 131 */
   1898820,          /* 132 */
   1913205,          /* 133 */
   1927590,          /* 134 */
   1941975,          /* 135 */
   1956360,          /* 136 */
   1970745,          /* 137 */
   1985130,          /* 138 */
   1999515,          /* 139 */
   2013900,          /* 140 */
   2028285,          /* 141 */
   2042670,          /* 142 */
   2057055,          /* 143 */
   2071440,          /* 144 */
   2085825,          /* 145 */
   2100210,          /* 146 */
   2114595,          /* 147 */
   2128980,          /* 148 */
   2143365,          /* 149 */
   2157750,          /* 150 */
   2172135,          /* 151 */
   2186520,          /* 152 */
   2200905,          /* 153 */
   2215290,          /* 154 */
   2229675,          /* 155 */
   2244060,          /* 156 */
   2258445,          /* 157 */
   2272830,          /* 158 */
   2287215,          /* 159 */
   2301600,          /* 160 */
   2315985,          /* 161 */
   2330370,          /* 162 */
   2344755,          /* 163 */
   2359140,          /* 164 */
   2373525,          /* 165 */
   2387910,          /* 166 */
   2402295,          /* 167 */
   2416680,          /* 168 */
   2431065,          /* 169 */
   2445450,          /* 170 */
   2459835,          /* 171 */
   2474220,          /* 172 */
   2488605,          /* 173 */
   2502990,          /* 174 */
   2517375,          /* 175 */
   2531760,          /* 176 */
   2546145,          /* 177 */
   2560530,          /* 178 */
   2574915,          /* 179 */
   2589300,          /* 180 */
   2603685,          /* 181 */
   2618070,          /* 182 */
   2632455,          /* 183 */
   2646840,          /* 184 */
   2661225,          /* 185 */
   2675610,          /* 186 */
   2689995,          /* 187 */
   2704380,          /* 188 */
   2718765,          /* 189 */
   2733150,          /* 190 */
   2747535,          /* 191 */
   2761920,          /* 192 */
   2776305,          /* 193 */
   2790690,          /* 194 */
   2805075,          /* 195 */
   2819460,          /* 196 */
   2833845,          /* 197 */
   2848230,          /* 198 */
   2862615,          /* 199 */
   2877000,          /* 200 */
   2891385,          /* 201 */
   2905770,          /* 202 */
   2920155,          /* 203 */
   2934540,          /* 204 */
   2948925,          /* 205 */
   2963310,          /* 206 */
   2977695,          /* 207 */
   2992080,          /* 208 */
   3006465,          /* 209 */
   3020850,          /* 210 */
   3035235,          /* 211 */
   3049620,          /* 212 */
   3064005,          /* 213 */
   3078390,          /* 214 */
   3092775,          /* 215 */
   3107160,          /* 216 */
   3121545,          /* 217 */
   3135930,          /* 218 */
   3150315,          /* 219 */
   3164700,          /* 220 */
   3179085,          /* 221 */
   3193470,          /* 222 */
   3207855,          /* 223 */
   3222240,          /* 224 */
   3236625,          /* 225 */
   3251010,          /* 226 */
   3265395,          /* 227 */
   3279780,          /* 228 */
   3294165,          /* 229 */
   3308550,          /* 230 */
   3322935,          /* 231 */
   3337320,          /* 232 */
   3351705,          /* 233 */
   3366090,          /* 234 */
   3380475,          /* 235 */
   3394860,          /* 236 */
   3409245,          /* 237 */
   3423630,          /* 238 */
   3438015,          /* 239 */
   3452400,          /* 240 */
   3466785,          /* 241 */
   3481170,          /* 242 */
   3495555,          /* 243 */
   3509940,          /* 244 */
   3524325,          /* 245 */
   3538710,          /* 246 */
   3553095,          /* 247 */
   3567480,          /* 248 */
   3581865,          /* 249 */
   3596250,          /* 250 */
   3610635,          /* 251 */
   3625020,          /* 252 */
   3639405,          /* 253 */
   3653790,          /* 254 */
   3668175           /* 255 */
};
            
            
int   VGMult[] = 
{
   4210688,          /* 0   */
   4198630,          /* 1   */
   4186572,          /* 2   */
   4174514,          /* 3   */
   4162456,          /* 4   */
   4150398,          /* 5   */
   4138340,          /* 6   */
   4126282,          /* 7   */
   4114224,          /* 8   */
   4102166,          /* 9   */
   4090108,          /* 10  */
   4078050,          /* 11  */
   4065992,          /* 12  */
   4053934,          /* 13  */
   4041876,          /* 14  */
   4029818,          /* 15  */
   4017760,          /* 16  */
   4005702,          /* 17  */
   3993644,          /* 18  */
   3981586,          /* 19  */
   3969528,          /* 20  */
   3957470,          /* 21  */
   3945412,          /* 22  */
   3933354,          /* 23  */
   3921296,          /* 24  */
   3909238,          /* 25  */
   3897180,          /* 26  */
   3885122,          /* 27  */
   3873064,          /* 28  */
   3861006,          /* 29  */
   3848948,          /* 30  */
   3836890,          /* 31  */
   3824832,          /* 32  */
   3812774,          /* 33  */
   3800716,          /* 34  */
   3788658,          /* 35  */
   3776600,          /* 36  */
   3764542,          /* 37  */
   3752484,          /* 38  */
   3740426,          /* 39  */
   3728368,          /* 40  */
   3716310,          /* 41  */
   3704252,          /* 42  */
   3692194,          /* 43  */
   3680136,          /* 44  */
   3668078,          /* 45  */
   3656020,          /* 46  */
   3643962,          /* 47  */
   3631904,          /* 48  */
   3619846,          /* 49  */
   3607788,          /* 50  */
   3595730,          /* 51  */
   3583672,          /* 52  */
   3571614,          /* 53  */
   3559556,          /* 54  */
   3547498,          /* 55  */
   3535440,          /* 56  */
   3523382,          /* 57  */
   3511324,          /* 58  */
   3499266,          /* 59  */
   3487208,          /* 60  */
   3475150,          /* 61  */
   3463092,          /* 62  */
   3451034,          /* 63  */
   3438976,          /* 64  */
   3426918,          /* 65  */
   3414860,          /* 66  */
   3402802,          /* 67  */
   3390744,          /* 68  */
   3378686,          /* 69  */
   3366628,          /* 70  */
   3354570,          /* 71  */
   3342512,          /* 72  */
   3330454,          /* 73  */
   3318396,          /* 74  */
   3306338,          /* 75  */
   3294280,          /* 76  */
   3282222,          /* 77  */
   3270164,          /* 78  */
   3258106,          /* 79  */
   3246048,          /* 80  */
   3233990,          /* 81  */
   3221932,          /* 82  */
   3209874,          /* 83  */
   3197816,          /* 84  */
   3185758,          /* 85  */
   3173700,          /* 86  */
   3161642,          /* 87  */
   3149584,          /* 88  */
   3137526,          /* 89  */
   3125468,          /* 90  */
   3113410,          /* 91  */
   3101352,          /* 92  */
   3089294,          /* 93  */
   3077236,          /* 94  */
   3065178,          /* 95  */
   3053120,          /* 96  */
   3041062,          /* 97  */
   3029004,          /* 98  */
   3016946,          /* 99  */
   3004888,          /* 100 */
   2992830,          /* 101 */
   2980772,          /* 102 */
   2968714,          /* 103 */
   2956656,          /* 104 */
   2944598,          /* 105 */
   2932540,          /* 106 */
   2920482,          /* 107 */
   2908424,          /* 108 */
   2896366,          /* 109 */
   2884308,          /* 110 */
   2872250,          /* 111 */
   2860192,          /* 112 */
   2848134,          /* 113 */
   2836076,          /* 114 */
   2824018,          /* 115 */
   2811960,          /* 116 */
   2799902,          /* 117 */
   2787844,          /* 118 */
   2775786,          /* 119 */
   2763728,          /* 120 */
   2751670,          /* 121 */
   2739612,          /* 122 */
   2727554,          /* 123 */
   2715496,          /* 124 */
   2703438,          /* 125 */
   2691380,          /* 126 */
   2679322,          /* 127 */
   2667264,          /* 128 */
   2655206,          /* 129 */
   2643148,          /* 130 */
   2631090,          /* 131 */
   2619032,          /* 132 */
   2606974,          /* 133 */
   2594916,          /* 134 */
   2582858,          /* 135 */
   2570800,          /* 136 */
   2558742,          /* 137 */
   2546684,          /* 138 */
   2534626,          /* 139 */
   2522568,          /* 140 */
   2510510,          /* 141 */
   2498452,          /* 142 */
   2486394,          /* 143 */
   2474336,          /* 144 */
   2462278,          /* 145 */
   2450220,          /* 146 */
   2438162,          /* 147 */
   2426104,          /* 148 */
   2414046,          /* 149 */
   2401988,          /* 150 */
   2389930,          /* 151 */
   2377872,          /* 152 */
   2365814,          /* 153 */
   2353756,          /* 154 */
   2341698,          /* 155 */
   2329640,          /* 156 */
   2317582,          /* 157 */
   2305524,          /* 158 */
   2293466,          /* 159 */
   2281408,          /* 160 */
   2269350,          /* 161 */
   2257292,          /* 162 */
   2245234,          /* 163 */
   2233176,          /* 164 */
   2221118,          /* 165 */
   2209060,          /* 166 */
   2197002,          /* 167 */
   2184944,          /* 168 */
   2172886,          /* 169 */
   2160828,          /* 170 */
   2148770,          /* 171 */
   2136712,          /* 172 */
   2124654,          /* 173 */
   2112596,          /* 174 */
   2100538,          /* 175 */
   2088480,          /* 176 */
   2076422,          /* 177 */
   2064364,          /* 178 */
   2052306,          /* 179 */
   2040248,          /* 180 */
   2028190,          /* 181 */
   2016132,          /* 182 */
   2004074,          /* 183 */
   1992016,          /* 184 */
   1979958,          /* 185 */
   1967900,          /* 186 */
   1955842,          /* 187 */
   1943784,          /* 188 */
   1931726,          /* 189 */
   1919668,          /* 190 */
   1907610,          /* 191 */
   1895552,          /* 192 */
   1883494,          /* 193 */
   1871436,          /* 194 */
   1859378,          /* 195 */
   1847320,          /* 196 */
   1835262,          /* 197 */
   1823204,          /* 198 */
   1811146,          /* 199 */
   1799088,          /* 200 */
   1787030,          /* 201 */
   1774972,          /* 202 */
   1762914,          /* 203 */
   1750856,          /* 204 */
   1738798,          /* 205 */
   1726740,          /* 206 */
   1714682,          /* 207 */
   1702624,          /* 208 */
   1690566,          /* 209 */
   1678508,          /* 210 */
   1666450,          /* 211 */
   1654392,          /* 212 */
   1642334,          /* 213 */
   1630276,          /* 214 */
   1618218,          /* 215 */
   1606160,          /* 216 */
   1594102,          /* 217 */
   1582044,          /* 218 */
   1569986,          /* 219 */
   1557928,          /* 220 */
   1545870,          /* 221 */
   1533812,          /* 222 */
   1521754,          /* 223 */
   1509696,          /* 224 */
   1497638,          /* 225 */
   1485580,          /* 226 */
   1473522,          /* 227 */
   1461464,          /* 228 */
   1449406,          /* 229 */
   1437348,          /* 230 */
   1425290,          /* 231 */
   1413232,          /* 232 */
   1401174,          /* 233 */
   1389116,          /* 234 */
   1377058,          /* 235 */
   1365000,          /* 236 */
   1352942,          /* 237 */
   1340884,          /* 238 */
   1328826,          /* 239 */
   1316768,          /* 240 */
   1304710,          /* 241 */
   1292652,          /* 242 */
   1280594,          /* 243 */
   1268536,          /* 244 */
   1256478,          /* 245 */
   1244420,          /* 246 */
   1232362,          /* 247 */
   1220304,          /* 248 */
   1208246,          /* 249 */
   1196188,          /* 250 */
   1184130,          /* 251 */
   1172072,          /* 252 */
   1160014,          /* 253 */
   1147956,          /* 254 */
   1135898           /* 255 */
};
            
            
int  VBMult[] =
{
   0,                /* 0   */
   -2326,            /* 1   */
   -4652,            /* 2   */
   -6978,            /* 3   */
   -9304,            /* 4   */
   -11630,           /* 5   */
   -13956,           /* 6   */
   -16282,           /* 7   */
   -18608,           /* 8   */
   -20934,           /* 9   */
   -23260,           /* 10  */
   -25586,           /* 11  */
   -27912,           /* 12  */
   -30238,           /* 13  */
   -32564,           /* 14  */
   -34890,           /* 15  */
   -37216,           /* 16  */
   -39542,           /* 17  */
   -41868,           /* 18  */
   -44194,           /* 19  */
   -46520,           /* 20  */
   -48846,           /* 21  */
   -51172,           /* 22  */
   -53498,           /* 23  */
   -55824,           /* 24  */
   -58150,           /* 25  */
   -60476,           /* 26  */
   -62802,           /* 27  */
   -65128,           /* 28  */
   -67454,           /* 29  */
   -69780,           /* 30  */
   -72106,           /* 31  */
   -74432,           /* 32  */
   -76758,           /* 33  */
   -79084,           /* 34  */
   -81410,           /* 35  */
   -83736,           /* 36  */
   -86062,           /* 37  */
   -88388,           /* 38  */
   -90714,           /* 39  */
   -93040,           /* 40  */
   -95366,           /* 41  */
   -97692,           /* 42  */
   -100018,          /* 43  */
   -102344,          /* 44  */
   -104670,          /* 45  */
   -106996,          /* 46  */
   -109322,          /* 47  */
   -111648,          /* 48  */
   -113974,          /* 49  */
   -116300,          /* 50  */
   -118626,          /* 51  */
   -120952,          /* 52  */
   -123278,          /* 53  */
   -125604,          /* 54  */
   -127930,          /* 55  */
   -130256,          /* 56  */
   -132582,          /* 57  */
   -134908,          /* 58  */
   -137234,          /* 59  */
   -139560,          /* 60  */
   -141886,          /* 61  */
   -144212,          /* 62  */
   -146538,          /* 63  */
   -148864,          /* 64  */
   -151190,          /* 65  */
   -153516,          /* 66  */
   -155842,          /* 67  */
   -158168,          /* 68  */
   -160494,          /* 69  */
   -162820,          /* 70  */
   -165146,          /* 71  */
   -167472,          /* 72  */
   -169798,          /* 73  */
   -172124,          /* 74  */
   -174450,          /* 75  */
   -176776,          /* 76  */
   -179102,          /* 77  */
   -181428,          /* 78  */
   -183754,          /* 79  */
   -186080,          /* 80  */
   -188406,          /* 81  */
   -190732,          /* 82  */
   -193058,          /* 83  */
   -195384,          /* 84  */
   -197710,          /* 85  */
   -200036,          /* 86  */
   -202362,          /* 87  */
   -204688,          /* 88  */
   -207014,          /* 89  */
   -209340,          /* 90  */
   -211666,          /* 91  */
   -213992,          /* 92  */
   -216318,          /* 93  */
   -218644,          /* 94  */
   -220970,          /* 95  */
   -223296,          /* 96  */
   -225622,          /* 97  */
   -227948,          /* 98  */
   -230274,          /* 99  */
   -232600,          /* 100 */
   -234926,          /* 101 */
   -237252,          /* 102 */
   -239578,          /* 103 */
   -241904,          /* 104 */
   -244230,          /* 105 */
   -246556,          /* 106 */
   -248882,          /* 107 */
   -251208,          /* 108 */
   -253534,          /* 109 */
   -255860,          /* 110 */
   -258186,          /* 111 */
   -260512,          /* 112 */
   -262838,          /* 113 */
   -265164,          /* 114 */
   -267490,          /* 115 */
   -269816,          /* 116 */
   -272142,          /* 117 */
   -274468,          /* 118 */
   -276794,          /* 119 */
   -279120,          /* 120 */
   -281446,          /* 121 */
   -283772,          /* 122 */
   -286098,          /* 123 */
   -288424,          /* 124 */
   -290750,          /* 125 */
   -293076,          /* 126 */
   -295402,          /* 127 */
   -297728,          /* 128 */
   -300054,          /* 129 */
   -302380,          /* 130 */
   -304706,          /* 131 */
   -307032,          /* 132 */
   -309358,          /* 133 */
   -311684,          /* 134 */
   -314010,          /* 135 */
   -316336,          /* 136 */
   -318662,          /* 137 */
   -320988,          /* 138 */
   -323314,          /* 139 */
   -325640,          /* 140 */
   -327966,          /* 141 */
   -330292,          /* 142 */
   -332618,          /* 143 */
   -334944,          /* 144 */
   -337270,          /* 145 */
   -339596,          /* 146 */
   -341922,          /* 147 */
   -344248,          /* 148 */
   -346574,          /* 149 */
   -348900,          /* 150 */
   -351226,          /* 151 */
   -353552,          /* 152 */
   -355878,          /* 153 */
   -358204,          /* 154 */
   -360530,          /* 155 */
   -362856,          /* 156 */
   -365182,          /* 157 */
   -367508,          /* 158 */
   -369834,          /* 159 */
   -372160,          /* 160 */
   -374486,          /* 161 */
   -376812,          /* 162 */
   -379138,          /* 163 */
   -381464,          /* 164 */
   -383790,          /* 165 */
   -386116,          /* 166 */
   -388442,          /* 167 */
   -390768,          /* 168 */
   -393094,          /* 169 */
   -395420,          /* 170 */
   -397746,          /* 171 */
   -400072,          /* 172 */
   -402398,          /* 173 */
   -404724,          /* 174 */
   -407050,          /* 175 */
   -409376,          /* 176 */
   -411702,          /* 177 */
   -414028,          /* 178 */
   -416354,          /* 179 */
   -418680,          /* 180 */
   -421006,          /* 181 */
   -423332,          /* 182 */
   -425658,          /* 183 */
   -427984,          /* 184 */
   -430310,          /* 185 */
   -432636,          /* 186 */
   -434962,          /* 187 */
   -437288,          /* 188 */
   -439614,          /* 189 */
   -441940,          /* 190 */
   -444266,          /* 191 */
   -446592,          /* 192 */
   -448918,          /* 193 */
   -451244,          /* 194 */
   -453570,          /* 195 */
   -455896,          /* 196 */
   -458222,          /* 197 */
   -460548,          /* 198 */
   -462874,          /* 199 */
   -465200,          /* 200 */
   -467526,          /* 201 */
   -469852,          /* 202 */
   -472178,          /* 203 */
   -474504,          /* 204 */
   -476830,          /* 205 */
   -479156,          /* 206 */
   -481482,          /* 207 */
   -483808,          /* 208 */
   -486134,          /* 209 */
   -488460,          /* 210 */
   -490786,          /* 211 */
   -493112,          /* 212 */
   -495438,          /* 213 */
   -497764,          /* 214 */
   -500090,          /* 215 */
   -502416,          /* 216 */
   -504742,          /* 217 */
   -507068,          /* 218 */
   -509394,          /* 219 */
   -511720,          /* 220 */
   -514046,          /* 221 */
   -516372,          /* 222 */
   -518698,          /* 223 */
   -521024,          /* 224 */
   -523350,          /* 225 */
   -525676,          /* 226 */
   -528002,          /* 227 */
   -530328,          /* 228 */
   -532654,          /* 229 */
   -534980,          /* 230 */
   -537306,          /* 231 */
   -539632,          /* 232 */
   -541958,          /* 233 */
   -544284,          /* 234 */
   -546610,          /* 235 */
   -548936,          /* 236 */
   -551262,          /* 237 */
   -553588,          /* 238 */
   -555914,          /* 239 */
   -558240,          /* 240 */
   -560566,          /* 241 */
   -562892,          /* 242 */
   -565218,          /* 243 */
   -567544,          /* 244 */
   -569870,          /* 245 */
   -572196,          /* 246 */
   -574522,          /* 247 */
   -576848,          /* 248 */
   -579174,          /* 249 */
   -581500,          /* 250 */
   -583826,          /* 251 */
   -586152,          /* 252 */
   -588478,          /* 253 */
   -590804,          /* 254 */
   -593130           /* 255 */
};
#define ShiftFactor      15
extern "C" void ConvertRGBtoYUV(
	unsigned char *r_src,unsigned char *g_src,unsigned char *b_src, 
	int width, int height, int rgb_step, int rgb_pitch,
	unsigned char *y_src, unsigned char *u_src, unsigned char *v_src,  
	int uv_width_shift, int uv_height_shift,
	int y_step, int y_pitch,int uv_step,int uv_pitch
	)
{

	int i,j,k,l,m,n,o,p;
	int y_scaled,u_scaled,v_scaled;

	// remember r,g,b
	unsigned char *r_ptr = r_src;
	unsigned char *g_ptr = g_src;
	unsigned char *b_ptr = b_src;
	unsigned char *r_ptr2 = r_src;
	unsigned char *g_ptr2 = g_src;
	unsigned char *b_ptr2 = b_src;

	int *YRMultPtr = YRMult;  
	int *YGMultPtr = YGMult;  
	int *YBMultPtr = YBMult;  
	int *URMultPtr = URMult;  
	int *UGMultPtr = UGMult;  
	int *UBVRMultPtr = UBVRMult;
	int *VGMultPtr = VGMult;
	int *VBMultPtr = VBMult;

	int uv_width_mask = (0xFFFFFFFF>>(32-uv_width_shift));
	int uv_height_mask = (0xFFFFFFFF>>(32-uv_height_shift));

	int uv_this_row,uv_this_col;
	int uv_width_scale = 1 << uv_width_shift, uv_height_scale = 1 << uv_height_shift;
	int uv_sum_shift = uv_width_shift + uv_height_shift;
	

	int avg_round = 1 << ( uv_sum_shift - 1 );
	int r_sum,g_sum,b_sum;

	j=height+1;

	
	// do y rows
	while(--j)
	{
		// we have a uv on this row if all of the bits lower than shift are set
		uv_this_row = (uv_height_mask & j)==uv_height_mask;

		// do y columns
		i = width+1;
		k = 0;
		l = 0;
		m = 0;
		while(--i)
		{

			// calculate 3 multiply parts of rgb -> to y 
			y_scaled = YRMultPtr[r_ptr[k]];	// includes round value for shift factor
			y_scaled += YGMultPtr[g_ptr[k]];
			y_scaled += YBMultPtr[b_ptr[k]];
	        y_src[l] = (unsigned char)(y_scaled >> ShiftFactor); 

			// we have a uv on this col if all of the bits lower than shift are set
			uv_this_col = (uv_width_mask & i)==uv_width_mask;

			// we now have a row and column on which we should calculate u and v values
			if(uv_this_row && uv_this_col)
			{
				r_sum = g_sum = b_sum = avg_round;

				r_ptr2 = r_ptr + k;
				g_ptr2 = g_ptr + k;
				b_ptr2 = b_ptr + k;

				// calculate r_sum, g_sum and b_sum for given u,v
				n = uv_height_scale + 1;
				while(--n)
				{
					o = uv_width_scale+1;
					p = 0;
					while(--o)
					{
						r_sum += r_ptr2[p];
						g_sum += g_ptr2[p];
						b_sum += b_ptr2[p];

						// step back one column
						p -= rgb_step;
					}

					// step back one line
					r_ptr2 -= rgb_pitch;
					g_ptr2 -= rgb_pitch;
					b_ptr2 -= rgb_pitch;

				}

				// calculate avg instead of just sum
				r_sum >>= uv_sum_shift;
				g_sum >>= uv_sum_shift;
				b_sum >>= uv_sum_shift;

				// convert rgb -> u,v
				u_scaled = URMultPtr[r_sum];
				v_scaled = UBVRMultPtr[r_sum];

				u_scaled += UGMultPtr[g_sum];
				v_scaled += VGMultPtr[g_sum];

				u_scaled += UBVRMultPtr[b_sum];
				v_scaled += VBMultPtr[b_sum];

				// convert from fixed point integer
				u_scaled >>= ShiftFactor;
				v_scaled >>= ShiftFactor;
				
				// save in result
				u_src[m] = (unsigned char) u_scaled;
				v_src[m] = (unsigned char) v_scaled;

				m+=uv_step;

			}

			k+=rgb_step;	// next rgb pixel
			l+=y_step;		// next y pixel

		}

		// we had uv's move to the next one
		if(uv_this_row)
		{
			u_src+=uv_pitch;
			v_src+=uv_pitch;
		}
		// next row
		r_ptr+=rgb_pitch;
		g_ptr+=rgb_pitch;
		b_ptr+=rgb_pitch;
		y_src+=y_pitch;
	}



}
