/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Keyboard reading routines.
 */
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <sgtty.h>
#include "graph_serv.h"


/*
 * The following is the keyboard device associated with the screen.
 */
#define	KEYBOARD	"/dev/console"


/*
 * Defining debug disables setting of the terminal modes, making it
 * easier to see output.
 */
#define	DEBUG	0


static	int		fd;		/* file descriptor for keyboard */
static	struct sgttyb	oldsgttyb;	/* original terminal modes */


/*
 * Open the keyboard.
 * This is real simple, we just use a special file handle
 * that allows non-blocking I/O, and put the terminal into
 * character mode.
 */
int
GsOpenKeyboard()
{
	struct sgttyb	newsgttyb;	/* new terminal modes */

	fd = open(KEYBOARD, O_NONBLOCK);
	if (fd < 0)
		return -1;

	if (ioctl(fd, TIOCGETP, &oldsgttyb) < 0) {
		close(fd);
		fd = 0;
		return -1;
	}

#if !DEBUG
	newsgttyb = oldsgttyb;
	newsgttyb.sg_flags |= CBREAK;
	newsgttyb.sg_flags &= ~(EVENP | ODDP | ECHO | XTABS | CRMOD);

	if (ioctl(fd, TIOCSETP, &newsgttyb) < 0) {
		close(fd);
		fd = 0;
		return -1;
	}
#endif
	return 0;
}



/*
 * Return the possible modifiers for the keyboard.
 */
void
GsGetModifierInfo(modifiers)
	GR_MODIFIER	*modifiers;	/* returned mask of modifiers */
{
	*modifiers = 0;			/* no modifiers available */
}


/*
 * Close the keyboard.
 * This resets the terminal modes.
 */
void
GsCloseKeyboard()
{
	ioctl(fd, TIOCSETP, &oldsgttyb);
	close(fd);
	fd = 0;
}


/*
 * This reads one keystroke from the keyboard, and the current state of
 * the mode keys (ALT, SHIFT, CTRL).  Returns -1 on error, 0 if no data
 * is ready, and 1 if data was read.  This is a non-blocking call.
 * Without MINIX kernel changes, no shift keys information is available.
 */
int
GsReadKeyboard(buf, modifiers)
	GR_CHAR		*buf;		/* data character */
	GR_MODIFIER	*modifiers;	/* modifier keys (SHIFT, CTRL, etc) */
{
	int	cc;			/* characters read */

#if	DEBUG
	return	0;
#endif

	*modifiers = 0;			/* no modifiers yet */
	cc = read(fd, buf, 1);
	if (cc > 0)
		return 1;
	if ((cc < 0) && (errno != EINTR) && (errno != EAGAIN))
		return -1;
	return 0;
}

/* END CODE */
