/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Client routines to do graphics with windows and graphics contexts.
 * NOTE: These are stub routines at the moment because the server
 * routines are loaded with the client into one process.  The problem
 * with this is that you can run only one client program on the screen.
 * Eventually, the server will run in its own process, and these routines
 * then must ship their arguments across some network interface to that
 * process.  At that point, the server can then serve multiple clients
 * simultaneously.
 */
#include <stdio.h>
#include "graph_serv.h"


/*
 * The following is used to allocate ids which the server will
 * recognize as our windows and such.  Since we and the server
 * keep in step in the allocation, it is not necessary to wait
 * for the server to return resource ids.
 */
static	GR_ID	allocid;


/*
 * The following is the user defined function for handling errors.
 * If this is not set, then the default action is to close the connection
 * to the server, describe the error, and then exit.  This error function
 * will only be called when the client asks for events.
 */
static GR_ERROR_FUNC	GrErrorFunc;



/*
 * Open a connection to the graphics server.
 * Returns zero if successful, -1 on failure.
 */
int
GrOpen()
{
	GrErrorFunc = 0;
	allocid = GsOpen();
	if (allocid == 0)
		return -1;
	return 0;
}


/*
 * Close the graphics device, first flushing any waiting messages.
 */
void
GrClose()
{
	GsClose();
}


/*
 * Set an error handling routine, which will be called on any errors from
 * the server (when events are asked for by the client).  If zero is given,
 * then a default routine will be used.  Returns the previous error handler.
 */
GR_ERROR_FUNC
GrSetErrorHandler(func)
	GR_ERROR_FUNC	func;		/* function to handle error */
{
	GR_ERROR_FUNC	oldfunc;	/* previous function */

	oldfunc = GrErrorFunc;
	GrErrorFunc = func;
	return oldfunc;
}


/*
 * Return useful information about the screen.
 */
void
GrGetScreenInfo(sip)
	GR_SCREEN_INFO	*sip;		/* address of screen info */
{
	GsGetScreenInfo(sip);
}


/*
 * Return useful information about the specified font.
 */
void
GrGetFontInfo(font, fip)
	GR_FONT		font;		/* font number */
	GR_FONT_INFO	*fip;		/* address of font info */
{
	GsGetFontInfo(font, fip);
}


/*
 * Return information about the specified graphics context.
 */
void
GrGetGCInfo(gc, gcip)
	GR_GC_ID	gc;		/* graphics context */
	GR_GC_INFO	*gcip;		/* address of graphics context info */
{
	GsGetGCInfo(gc, gcip);
}


/*
 * Return the size of a text string for the font in a graphics context.
 * This is the width of the string, the height of the string,
 * and the height above the bottom of the font of the baseline for the font.
 */
void
GrGetGCTextSize(gc, cp, len, retwidth, retheight, retbase)
	GR_GC_ID	gc;		/* graphics context containing font */
	GR_CHAR		*cp;		/* address of text string */
	GR_SIZE		len;		/* length of text string */
	GR_SIZE		*retwidth;	/* returned width of string */
	GR_SIZE		*retheight;	/* returned height of string */
	GR_SIZE		*retbase;	/* returned height of baseline */
{
	GsGetGCTextSize(gc, cp, len, retwidth, retheight, retbase);
}


/*
 * Return the next event from the event queue.
 * This waits for a new one if one is not ready.
 * If it is an error event, then a user-specified routine is
 * called if it was defined, otherwise we clean up and exit.
 */
void
GrGetNextEvent(ep)
	GR_EVENT	*ep;		/* address where event is returned */
{
	GsFlush();
	while (1) {
		GsGetNextEvent(ep);
		if (ep->type != GR_EVENT_TYPE_ERROR)
			return;

		if (GrErrorFunc) {
			(*GrErrorFunc)(ep->error.code, ep->error.name,
				ep->error.id);
			continue;
		}

		GsClose();
		fprintf(stderr, "Graphics error %d, function %s, resource id %ld\n",
			ep->error.code, ep->error.name, ep->error.id);
		exit(1);
	}
}


/*
 * Return the next event from the event queue if one is ready.
 * If one is not ready, then the type GR_EVENT_TYPE_NONE is returned.
 * If it is an error event, then a user-specified routine is called
 * if it was defined, otherwise we clean up and exit.
 */
void
GrCheckNextEvent(ep)
	GR_EVENT	*ep;		/* address where event is returned */
{
	GrPeekEvent(ep);
	if (ep->type != GR_EVENT_TYPE_NONE)
		GrGetNextEvent(ep);
}


/*
 * Return the next event from the event queue if available.
 * If there is no event, then a null event type is returned.
 */
void
GrPeekEvent(ep)
	GR_EVENT	*ep;		/* address where event is returned */
{
	GsFlush();
	GsPeekEvent(ep);
}


/*
 * Select events for a window for this client.
 * The events are a bitmask for the events desired.
 */
void
GrSelectEvents(wid, eventmask)
	GR_WINDOW_ID	wid;		/* window id */
	GR_EVENT_MASK	eventmask;	/* mask of events wanted */
{
	GsSelectEvents(wid, eventmask);
}


/*
 * Allocate a new window which is a child of the specified window.
 * The window inherits the cursor of the parent window.
 */
GR_WINDOW_ID
GrNewWindow(parent, x, y, width, height, bordersize, background, bordercolor)
	GR_WINDOW_ID	parent;		/* parent id */
	GR_COORD	x;		/* x position relative to parent */
	GR_COORD	y;		/* y position relative to parent */
	GR_SIZE		width;		/* width */
	GR_SIZE		height;		/* height */
	GR_SIZE		bordersize;	/* size of border */
	GR_COLOR	background;	/* background color */
	GR_COLOR	bordercolor;	/* border color */
{
	GsNewWindow(parent, x, y, width, height, bordersize,
		background, bordercolor);

	return ++allocid;
}


/*
 * Allocate a new input-only window which is a child of the specified window.
 * The window inherits the cursor of the parent window.
 */
GR_WINDOW_ID
GrNewInputWindow(parent, x, y, width, height)
	GR_WINDOW_ID	parent;		/* parent id */
	GR_COORD	x;		/* x position relative to parent */
	GR_COORD	y;		/* y position relative to parent */
	GR_SIZE		width;		/* width */
	GR_SIZE		height;		/* height */
{
	GsNewInputWindow(parent, x, y, width, height);

	return ++allocid;
}


/*
 * Destroy an existing window.
 */
void
GrDestroyWindow(wid)
	GR_WINDOW_ID	wid;		/* window to destroy */
{
	GsDestroyWindow(wid);
}


/*
 * Return information about a window id.
 */
void
GrGetWindowInfo(wid, infoptr)
	GR_WINDOW_ID		wid;		/* window to find out about */
	GR_WINDOW_INFO		*infoptr;	/* pointer to returned data */
{
	GsGetWindowInfo(wid, infoptr);
}


/*
 * Allocate a new GC with default parameters.
 */
GR_GC_ID
GrNewGC()
{
	GsNewGC();

	return ++allocid;
}


/*
 * Allocate a new GC which is a copy of another one.
 */
GR_GC_ID
GrCopyGC(gc)
	GR_GC_ID	gc;		/* graphics context to copy */
{
	GsCopyGC(gc);

	return ++allocid;
}


/*
 * Destroy an existing graphics context.
 */
void
GrDestroyGC(gc)
	GR_GC_ID	gc;		/* graphics context to destroy */
{
	GsDestroyGC(gc);
}


/*
 * Map the window to make it (and possibly its children) visible on the screen.
 * This paints the border and background of the window, and creates an
 * exposure event to tell the client to draw into it.
 */
void
GrMapWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be mapped */
{
	GsMapWindow(wid);
}


/*
 * Unmap the window to make it and its children invisible on the screen.
 */
void
GrUnmapWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be unmapped */
{
	GsUnmapWindow(wid);
}


/*
 * Raise the window to the highest level among its siblings.
 */
void
GrRaiseWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be raised */
{
	GsRaiseWindow(wid);
}


/*
 * Lower the window to the lowest level among its siblings.
 */
void
GrLowerWindow(wid)
	GR_WINDOW_ID	wid;		/* window to be lowered */
{
	GsLowerWindow(wid);
}


/*
 * Move the window to the specified position relative to its parent.
 */
void
GrMoveWindow(wid, x, y)
	GR_WINDOW_ID	wid;		/* window to be lowered */
	GR_COORD	x;		/* new relative x position */
	GR_COORD	y;		/* new relative y position */
{
	GsMoveWindow(wid, x, y);
}


/*
 * Resize the window to be the specified size.
 */
void
GrResizeWindow(wid, width, height)
	GR_WINDOW_ID	wid;		/* window to be lowered */
	GR_SIZE		width;		/* new width of window */
	GR_SIZE		height;		/* new height of window */
{
	GsResizeWindow(wid, width, height);
}


/*
 * Clear the specified window by setting it to its background color.
 * If the exposeflag is nonzero, then this also creates an exposure
 * event for the window.
 */
void
GrClearWindow(wid, exposeflag)
	GR_WINDOW_ID	wid;		/* window id */
	GR_BOOL		exposeflag;	/* nonzero to cause an exposure */
{
	GsClearWindow(wid, exposeflag);
}


/*
 * Set the focus to a particular window.
 * This makes keyboard events only visible to that window or children of it,
 * depending on the pointer location.
 */
void
GrSetFocus(wid)
	GR_WINDOW_ID	wid;		/* window id */
{
	GsSetFocus(wid);
}


/*
 * Set the border of a window to the specified color.
 */
void
GrSetBorderColor(wid, color)
	GR_WINDOW_ID	wid;		/* window id */
	GR_COLOR	color;		/* color for border */
{
	GsSetBorderColor(wid, color);
}


/*
 * Specify a cursor for a window.
 * This cursor will only be used within that window, and by default
 * for its new children.  If the cursor is currently within this
 * window, it will be changed to the new one immediately.
 */
void
GrSetCursor(wid, width, height, hotx, hoty, foreground, background,
	fgbitmap, bgbitmap)

	GR_WINDOW_ID	wid;		/* window id to set cursor for */
	GR_SIZE		width;		/* width of cursor */
	GR_SIZE		height;		/* height of cursor */
	GR_COORD	hotx;		/* relative x position of hot spot */
	GR_COORD	hoty;		/* relative y position of hot spot */
	GR_COLOR	foreground;	/* foreground color of cursor */
	GR_COLOR	background;	/* background color of cursor */
	GR_BITMAP	*fgbitmap;	/* foreground bitmap */
	GR_BITMAP	*bgbitmap;	/* background bitmap */
{
	GsSetCursor(wid, width, height, hotx, hoty, foreground, background,
		fgbitmap, bgbitmap);
}


/*
 * Move the cursor to the specified absolute screen coordinates.
 * The coordinates are that of the defined hot spot of the cursor.
 * The cursor's appearance is changed to that defined for the window
 * in which the cursor is moved to.
 */
void
GrMoveCursor(x, y)
	GR_COORD	x;		/* new x position of cursor */
	GR_COORD	y;		/* new y position of cursor */
{
	GsMoveCursor(x, y);
}


/*
 * Flush the message buffer of any messages it may contain.
 */
void
GrFlush()
{
	GsFlush();
}


/*
 * Set the foreground color in a graphics context.
 */
void
GrSetGCForeground(gc, foreground)
	GR_GC_ID	gc;		/* graphics context id */
	GR_COLOR	foreground;	/* foreground color */
{
	GsSetGCForeground(gc, foreground);
}


/*
 * Set the background color in a graphics context.
 */
void
GrSetGCBackground(gc, background)
	GR_GC_ID	gc;		/* graphics context id */
	GR_COLOR	background;	/* background color */
{
	GsSetGCBackground(gc, background);
}


/*
 * Set the drawing mode in a graphics context.
 */
void
GrSetGCMode(gc, mode)
	GR_GC_ID	gc;		/* graphics context id */
	GR_MODE		mode;		/* drawing mode */
{
	GsSetGCMode(gc, mode);
}


/*
 * Set whether or not the background color is drawn in bitmaps and text.
 */
void
GrSetGCUseBackground(gc, flag)
	GR_GC_ID	gc;		/* graphics context id */
	GR_BOOL		flag;		/* TRUE if background is drawn */
{
	GsSetGCUseBackground(gc, flag);
}


/*
 * Set the font used for text drawing in a graphics context.
 * The font is a number identifying one of several fonts.
 * Font number 0 is always available, and is the default font.
 */
void
GrSetGCFont(gc, font)
	GR_GC_ID	gc;		/* graphics context id */
	GR_FONT		font;		/* text font */
{
	GsSetGCFont(gc, font);
}


/*
 * Draw a line in the specified drawable using the specified graphics context.
 */
void
GrLine(id, gc, x1, y1, x2, y2)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x1;
	GR_COORD	y1;
	GR_COORD	x2;
	GR_COORD	y2;
{
	GsLine(id, gc, x1, y1, x2, y2);
}


/*
 * Draw the boundary of a rectangle in the specified drawable using the
 * specified graphics context.
 */
void
GrRect(id, gc, x, y, width, height)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
{
	GsRect(id, gc, x, y, width, height);
}


/*
 * Fill a rectangle in the specified drawable using the specified
 * graphics context.
 */
void
GrFillRect(id, gc, x, y, width, height)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
{
	GsFillRect(id, gc, x, y, width, height);
}


/*
 * Draw the boundary of an ellipse in the specified drawable with
 * the specified graphics context.
 */
void
GrEllipse(id, gc, x, y, rx, ry)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		rx;
	GR_SIZE		ry;
{
	GsEllipse(id, gc, x, y, rx, ry);
}


/*
 * Fill an ellipse in the specified drawable using the specified
 * graphics context.
 */
void
GrFillEllipse(id, gc, x, y, rx, ry)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		rx;
	GR_SIZE		ry;
{
	GsFillEllipse(id, gc, x, y, rx, ry);
}


/*
 * Draw a rectangular area in the specified drawable using the specified
 * graphics, as determined by the specified bit map.  This differs from
 * rectangle drawing in that the rectangle is drawn using the foreground
 * color and possibly the background color as determined by the bit map.
 * Each row of bits is aligned to the next bitmap word boundary (so there
 * is padding at the end of the row).  The background bit values are
 * only written if the usebackground flag is set in the GC.
 */
void
GrBitmap(id, gc, x, y, width, height, bitmaptable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_BITMAP	*bitmaptable;
{
	GsBitmap(id, gc, x, y, width, height, bitmaptable);
}


/*
 * Draw a rectangular area in the specified drawable using the specified
 * graphics context.  This differs from rectangle drawing in that the
 * color values for each pixel in the rectangle are specified.  The color
 * values are restricted to 8 bit values.  The color table is indexed
 * row by row.  Values whose color matches the background color are only
 * written if the usebackground flag is set in the GC.
 */
void
GrArea8(id, gc, x, y, width, height, colortable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COLOR8	*colortable;
{
	GsArea8(id, gc, x, y, width, height, colortable);
}


/*
 * Read the color values from the specified rectangular area of the
 * specified drawable into a supplied buffer.  If the drawable is a
 * window which is obscured by other windows, then the returned values
 * will include the values from the covering windows.  Regions outside
 * of the screen boundaries, or unmapped windows will return black.
 */
void
GrReadArea8(id, x, y, width, height, colortable)
	GR_DRAW_ID	id;
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COLOR8	*colortable;
{
	GsReadArea8(id, x, y, width, height, colortable);
}


/*
 * Draw a point in the specified drawable using the specified
 * graphics context.
 */
void
GrPoint(id, gc, x, y)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
{
	GsPoint(id, gc, x, y);
}


/*
 * Draw a polygon in the specified drawable using the specified
 * graphics context.  The polygon is only complete if the first
 * point is repeated at the end.
 */
void
GrPoly(id, gc, count, pointtable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COUNT	count;
	GR_POINT	*pointtable;
{
	GsPoly(id, gc, count, pointtable);
}


/*
 * Draw a filled polygon in the specified drawable using the specified
 * graphics context.  The last point may be a duplicate of the first
 * point, but this is not required.
 */
void
GrFillPoly(id, gc, count, pointtable)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COUNT	count;
	GR_POINT	*pointtable;
{
	GsFillPoly(id, gc, count, pointtable);
}


/*
 * Draw a text string in the specified drawable using the specified
 * graphics context.  The background of the characters are only drawn
 * if the usebackground flag in the GC is set.
 */
void
GrText(id, gc, x, y, str, count)
	GR_DRAW_ID	id;
	GR_GC_ID	gc;
	GR_COORD	x;
	GR_COORD	y;
	GR_CHAR		*str;
	GR_COUNT	count;
{
	GsText(id, gc, x, y, str, count);
}

/* END CODE */
