/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * This file contains the drivers for the following special file:
 *     /dev/graphics	- graphics screen
 *
 * The driver supports the following operations (using message format m2):
 *
 *    m_type      DEVICE    PROC_NR     COUNT    POSITION  ADRRESS
 * ----------------------------------------------------------------
 * |  DISK_READ | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DISK_WRITE | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 *
 * The file contains one entry point:
 *
 *   graphics_task:	main entry when system is brought up
 *
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/graph_msg.h>
#include "graph_dev.h"


PRIVATE GR_BOOL inited;		/* TRUE if graphics mode is inited */
PRIVATE GR_FUNC readfunc;	/* function for reading */
PRIVATE char buf[GR_MAX_SIZEOF_MSG];	/* drawing commands */
PRIVATE phys_bytes buf_phys;	/* physical address of buffer */
PRIVATE message mess;		/* message buffer */


/* The following union holds all read request messages.
 * All of these messages must be of a fixed size.
 */
PRIVATE union {
  GR_MSG_HEADER head;		/* header */
  GR_MSG_READAREA8 readarea8;	/* read area (8 bit colors) */
  GR_MSG_GETSCREENINFO getscreeninfo;	/* get screen config info */
  GR_MSG_GETFONTINFO getfontinfo;	/* get font info */
} readdata;



FORWARD void do_init();
FORWARD void do_term();
FORWARD void do_points();
FORWARD void do_lines();
FORWARD void do_rects();
FORWARD void do_ellips();
FORWARD void do_setforeground();
FORWARD void do_setbackground();
FORWARD void do_setusebackground();
FORWARD void do_setmode();
FORWARD void do_area8();
FORWARD void do_bitmap();
FORWARD void do_setcliprects();
FORWARD void do_setcursor();
FORWARD void do_movecursor();
FORWARD void do_copyarea();
FORWARD void do_text();
FORWARD void do_poly();

FORWARD int do_readarea8();
FORWARD int do_getscreeninfo();
FORWARD int do_getfontinfo();

FORWARD int do_read();
FORWARD int do_write();


/*===========================================================================*
 *				graphics_task				     *
 *===========================================================================*/
PUBLIC void graphics_task()
{
/* Main program of the graphics task. */

  int r, caller, proc_nr;

  /* Initialize this task. */
  buf_phys = umap(proc_ptr, D, buf, sizeof(buf));
  inited = GR_FALSE;

  /* Here is the main loop of the graphics task.  It waits for a
   * message, carries it out, and sends a reply.
   */
  while (TRUE) {
	/* First wait for a request to read or write. */
	receive(ANY, &mess);

	if (mess.m_source < 0)
		panic("Graphics task got message from ", mess.m_source);
	caller = mess.m_source;
	proc_nr = mess.PROC_NR;

	/* Now carry out the work.  It depends on the opcode. */
	switch (mess.m_type) {
	    case DISK_READ:	r = do_read(&mess);	break;
	    case DISK_WRITE:	r = do_write(&mess);	break;
	    default:	r = EINVAL;	break;
	}

	/* Finally, prepare and send the reply message. */
	mess.m_type = TASK_REPLY;
	mess.REP_PROC_NR = proc_nr;
	mess.REP_STATUS = r;
	send(caller, &mess);
  }
}


/*===========================================================================*
 *				do_write				     *
 *===========================================================================*/
PRIVATE int do_write(m_ptr)
register message *m_ptr;	/* pointer to the newly arrived message */
{
  int cc;
  int r;
  PRIVATE phys_bytes user_phys;	/* physical address of user buf */

  /* Make sure write count is between defined minimum and maximum sizes. */
  cc = m_ptr->COUNT;
  if ((cc < sizeof(GR_MSG_HEADER)) || (cc > GR_MAX_SIZEOF_MSG))
	return EINVAL;

  /* Validate and return physical address of user buffer */
  user_phys = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, (vir_bytes) cc);
  if (user_phys == 0) return E_BAD_ADDR;

  /* Read in the complete message */
  phys_copy(user_phys, buf_phys, (phys_bytes) cc);

  /* Execute the commands in it */
  r = domessage(buf, cc);
  if (r) return r;
  return cc;
}


/*===========================================================================*
 *				do_read					     *
 *===========================================================================*/
PRIVATE int do_read(m_ptr)
register message *m_ptr;	/* pointer to the newly arrived message */
{
  int cc;
  PRIVATE phys_bytes user_phys;	/* physical address of user buf */

  if (!inited) return EIO;

  /* Make sure write count is between defined minimum and maximum sizes. */
  cc = m_ptr->COUNT;
  if ((cc < 0) || (cc > GR_MAX_SIZEOF_MSG)) return EINVAL;

  /* Validate and return physical address of user buffer */
  user_phys = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, (vir_bytes) cc);
  if (user_phys == 0) return E_BAD_ADDR;

  switch (readdata.head.func) {
      case GR_FUNC_READAREA8:
	cc = do_readarea8(&readdata.readarea8, buf, cc);
	break;

      case GR_FUNC_GETSCREENINFO:
	cc = do_getscreeninfo(&readdata.getscreeninfo, buf, cc);
	break;

      case GR_FUNC_GETFONTINFO:
	cc = do_getfontinfo(&readdata.getfontinfo, buf, cc);
	break;

      default:	cc = ENXIO;	break;
  }

  readdata.head.func = GR_FUNC_NOP;
  if (cc <= 0) return cc;

  /* Return the complete message to the user */
  phys_copy(buf_phys, user_phys, (phys_bytes) cc);

  return cc;
}


/*===========================================================================*
 *				do_readarea8				     *
 *===========================================================================*/
PRIVATE int do_readarea8(mp, buf, cc)
GR_MSG_READAREA8 *mp;		/* read request message */
char *buf;			/* buffer address for result */
int cc;				/* buffer size for result */
{
/* Read a rectangular array of pixel values from the screen. */
  GR_COLOR8 *color;
  GR_COORD row;
  GR_COORD col;

  if ((mp->width <= 0) || (mp->width > cc) || (mp->height <= 0) ||
      (mp->height > cc))
	return EINVAL;

  if (((long) mp->width * (long) mp->height * sizeof(GR_COLOR8)) > cc)
	return EINVAL;

  color = (GR_COLOR8 *) buf;
  for (row = 0; row < mp->height; row++)
	for (col = 0; col < mp->width; col++)
		*color++ = (*gr_dev.readpoint) (mp->x + col, mp->y + row);

  return mp->width * mp->height * sizeof(GR_COLOR8);
}


/*===========================================================================*
 *				do_getscreeninfo			     *
 *===========================================================================*/
PRIVATE int do_getscreeninfo(mp, buf, cc)
GR_MSG_GETSCREENINFO *mp;	/* read request message */
char *buf;			/* buffer address for result */
int cc;				/* buffer size for result */
{
/* Get screen configuration information. */
  if (cc < sizeof(GR_SCREEN_INFO)) return EINVAL;
  (*gr_dev.getscreeninfo) ((GR_SCREEN_INFO *) buf);
  return sizeof(GR_SCREEN_INFO);
}


/*===========================================================================*
 *				do_getfontinfo				     *
 *===========================================================================*/
PRIVATE int do_getfontinfo(mp, buf, cc)
GR_MSG_GETFONTINFO *mp;		/* read request message */
char *buf;			/* buffer address for result */
int cc;				/* buffer size for result */
{
/* Get font information. */
  if (cc < sizeof(GR_FONT_INFO)) return EINVAL;
  (*gr_dev.getfontinfo) (mp->font, (GR_FONT_INFO *) buf);
  return sizeof(GR_FONT_INFO);
}


/*===========================================================================*
 *				domessage				     *
 *===========================================================================*/
PRIVATE int domessage(msg, cc)
char *msg;			/* message bufer */
int cc;				/* number of characters available */
{
/* Execute all graphics messages encoded in a bufferful of data.
 * Returns error if an unknown or invalid message was encountered.
 * In this case, remaining messages in the buffer were ignored.
 */
  GR_COUNT count;
  long size;
  GR_MSG_HEADER *mp;

  while (cc > 0) {
	if (cc < sizeof(GR_MSG_HEADER)) return EIO;
	mp = (GR_MSG_HEADER *) msg;
	count = mp->count;
	if ((count < 0) || (count > GR_MAX_COUNT)) return EIO;
	if (!inited && (mp->func != GR_FUNC_INIT)) return EIO;

	/* Verify the current function, and calculate the size of the
	 * message structure for that function.
	 */
	switch (mp->func) {
	    case GR_FUNC_INIT:
		size = sizeof(GR_MSG_INIT);
		break;
	    case GR_FUNC_TERM:
		size = sizeof(GR_MSG_TERM);
		break;
	    case GR_FUNC_NOP:
		size = sizeof(GR_MSG_HEADER);
		break;
	    case GR_FUNC_DRAWPOINTS:
		size = GR_SIZEOF_MSG_POINTS(count);
		break;
	    case GR_FUNC_DRAWLINES:
		size = GR_SIZEOF_MSG_LINES(count);
		break;
	    case GR_FUNC_DRAWRECTS:
		size = GR_SIZEOF_MSG_RECTS(count);
		break;
	    case GR_FUNC_DRAWELLIPS:
		size = GR_SIZEOF_MSG_ELLIPS(count);
		break;
	    case GR_FUNC_SETMODE:
		size = sizeof(GR_MSG_SETMODE);
		break;
	    case GR_FUNC_SETFOREGROUND:
		size = sizeof(GR_MSG_SETFOREGROUND);
		break;
	    case GR_FUNC_SETBACKGROUND:
		size = sizeof(GR_MSG_SETBACKGROUND);
		break;
	    case GR_FUNC_SETUSEBACKGROUND:
		size = sizeof(GR_MSG_SETUSEBACKGROUND);
		break;
	    case GR_FUNC_DRAWAREA8:
		size = GR_SIZEOF_MSG_AREA8(count);
		break;
	    case GR_FUNC_DRAWBITMAP:
		size = GR_SIZEOF_MSG_BITMAP(count);
		break;
	    case GR_FUNC_SETCLIPRECTS:
		size = GR_SIZEOF_MSG_SETCLIPRECTS(count);
		break;
	    case GR_FUNC_DRAWPOLY:
		size = GR_SIZEOF_MSG_POLY(count);
		break;
	    case GR_FUNC_DRAWTEXT:
		size = GR_SIZEOF_MSG_TEXT(count);
		break;
	    case GR_FUNC_SETCURSOR:
		size = GR_SIZEOF_MSG_SETCURSOR(count);
		break;
	    case GR_FUNC_MOVECURSOR:
		size = sizeof(GR_MSG_MOVECURSOR);
		break;
	    case GR_FUNC_COPYAREA:
		size = sizeof(GR_MSG_COPYAREA);
		break;
	    case GR_FUNC_READAREA8:
		size = sizeof(GR_MSG_READAREA8);
		break;
	    case GR_FUNC_GETSCREENINFO:
		size = sizeof(GR_MSG_GETSCREENINFO);
		break;
	    case GR_FUNC_GETFONTINFO:
		size = sizeof(GR_MSG_GETFONTINFO);
		break;
	    default:	return ENXIO;
	}

	/* Make sure that the complete message structure for the
	 * current function fits in the remainder of the buffer.
	 */
	if ((size < sizeof(GR_MSG_HEADER)) || (size > cc)) return EIO;

	/* OK, now call the routine to handle the function. */
	switch (mp->func) {
	    case GR_FUNC_INIT:
		do_init((GR_MSG_INIT *) mp);
		if (!inited) return EIO;
		break;
	    case GR_FUNC_TERM:
		do_term((GR_MSG_TERM *) mp);
		break;
	    case GR_FUNC_DRAWPOINTS:
		do_points((GR_MSG_POINTS *) mp);
		break;
	    case GR_FUNC_DRAWLINES:
		do_lines((GR_MSG_LINES *) mp);
		break;
	    case GR_FUNC_DRAWRECTS:
		do_rects((GR_MSG_RECTS *) mp);
		break;
	    case GR_FUNC_DRAWELLIPS:
		do_ellips((GR_MSG_ELLIPS *) mp);
		break;
	    case GR_FUNC_SETMODE:
		do_setmode((GR_MSG_SETMODE *) mp);
		break;
	    case GR_FUNC_SETFOREGROUND:
		do_setforeground((GR_MSG_SETFOREGROUND *) mp);
		break;
	    case GR_FUNC_SETBACKGROUND:
		do_setbackground((GR_MSG_SETBACKGROUND *) mp);
		break;
	    case GR_FUNC_SETUSEBACKGROUND:
		do_setusebackground((GR_MSG_SETUSEBACKGROUND *) mp);
		break;
	    case GR_FUNC_DRAWAREA8:
		do_area8((GR_MSG_AREA8 *) mp);
		break;
	    case GR_FUNC_DRAWBITMAP:
		do_bitmap((GR_MSG_BITMAP *) mp);
		break;
	    case GR_FUNC_SETCLIPRECTS:
		do_setcliprects((GR_MSG_SETCLIPRECTS *) mp);
		break;
	    case GR_FUNC_DRAWPOLY:
		do_poly((GR_MSG_POLY *) mp);
		break;
	    case GR_FUNC_DRAWTEXT:
		do_text((GR_MSG_TEXT *) mp);
		break;
	    case GR_FUNC_SETCURSOR:
		do_setcursor((GR_MSG_SETCURSOR *) mp);
		break;
	    case GR_FUNC_MOVECURSOR:
		do_movecursor((GR_MSG_MOVECURSOR *) mp);
		break;
	    case GR_FUNC_COPYAREA:
		do_copyarea((GR_MSG_COPYAREA *) mp);
		break;
	    case GR_FUNC_READAREA8:
		if (cc != size) return EIO;
		readdata.readarea8 = *((GR_MSG_READAREA8 *) mp);
		break;
	    case GR_FUNC_GETSCREENINFO:
		if (cc != size) return EIO;
		readdata.getscreeninfo =
			*((GR_MSG_GETSCREENINFO *) mp);
		break;
	    case GR_FUNC_GETFONTINFO:
		if (cc != size) return EIO;
		readdata.getfontinfo =
			*((GR_MSG_GETFONTINFO *) mp);
		break;
	}

	/* Restore the cursor if it was removed because of the
	 * graphics operation.
	 */
	(*gr_dev.fixcursor) ();

	/* Advance to the next free location in the buffer. */
	msg += size;
	cc -= size;
  }
  return 0;
}


/*===========================================================================*
 *				do_init					     *
 *===========================================================================*/
PRIVATE void do_init(ip)
GR_MSG_INIT *ip;
{
/* Initialize the graphics device */
  if (ip->head.flags || ip->head.count || (ip->magic != GR_INIT_MAGIC))
	return;
  if ((*gr_dev.init) (ip->rows, ip->cols, ip->colors)) return;
  clipinit(0, 0);
  gen_initcursor();
  gr_foreground = gr_dev.white;
  gr_background = gr_dev.black;
  gr_usebg = GR_TRUE;
  readdata.head.func = GR_FUNC_NOP;
  inited = GR_TRUE;
}


/*===========================================================================*
 *				do_term					     *
 *===========================================================================*/
PRIVATE void do_term(tp)
GR_MSG_TERM *tp;
{
/* Terminate graphics mode. */
  (*gr_dev.term) ();
  inited = GR_FALSE;
}


/*===========================================================================*
 *				do_points				     *
 *===========================================================================*/
PRIVATE void do_points(mp)
GR_MSG_POINTS *mp;
{
/* Process a list of points to be drawn. */
  GR_COUNT count;
  GR_POINT *pp;

  pp = mp->points;
  count = mp->head.count;
  while (count-- > 0) {
	drawpoint(pp->x, pp->y);
	pp++;
  }
}


/*===========================================================================*
 *				do_lines				     *
 *===========================================================================*/
PRIVATE void do_lines(mp)
GR_MSG_LINES *mp;
{
/* Process a list of lines to be drawn. */
  GR_COUNT count;
  GR_LINE *lp;

  lp = mp->lines;
  count = mp->head.count;
  while (count-- > 0) {
	drawline(lp->x1, lp->y1, lp->x2, lp->y2);
	(*gr_dev.fixcursor) ();
	lp++;
  }
}


/*===========================================================================*
 *				do_rects				     *
 *===========================================================================*/
PRIVATE void do_rects(mp)
GR_MSG_RECTS *mp;
{
/* Process a list of rectangles to be drawn.
 * The rectangles can be filled or not depending on the flags.
 */
  GR_COUNT count;
  GR_RECT *rp;

  count = mp->head.count;
  for (rp = mp->rects; count-- > 0; rp++) {
	if ((rp->width <= 0) || (rp->height <= 0)) continue;
	if ((mp->head.flags & GR_FLAG_FILLAREA) == 0)
		drawrect(rp->x, rp->y, rp->width, rp->height);
	else
		fillrect(rp->x, rp->y, rp->x + rp->width - 1,
			 rp->y + rp->height - 1);
	(*gr_dev.fixcursor) ();
  }
}


/*===========================================================================*
 *				do_ellips				     *
 *===========================================================================*/
PRIVATE void do_ellips(mp)
GR_MSG_ELLIPS *mp;
{
/* Process a list of ellipses (or circles) to be drawn.
 * The ellipses can be filled or not depending on the flags.
 */
  GR_COUNT count;
  GR_ELLIPSE *ep;

  ep = mp->ellips;
  count = mp->head.count;
  while (count-- > 0) {
	if (mp->head.flags & GR_FLAG_FILLAREA)
		fillellipse(ep->x, ep->y, ep->rx, ep->ry);
	else
		drawellipse(ep->x, ep->y, ep->rx, ep->ry);
	(*gr_dev.fixcursor) ();
	ep++;
  }
}


/*===========================================================================*
 *				do_setmode				     *
 *===========================================================================*/
PRIVATE void do_setmode(mp)
GR_MSG_SETMODE *mp;
{
/* Process a set drawing mode message.
 * This is one of SET, OR, AND, or XOR.
 */
  if (mp->mode > GR_MAX_MODE) {
	gr_error = GR_TRUE;
	return;
  }
  if (gr_mode != mp->mode) {
	(*gr_dev.setmode) (mp->mode);
	gr_mode = mp->mode;
  }
}


/*===========================================================================*
 *				do_setforeground			     *
 *===========================================================================*/
PRIVATE void do_setforeground(mp)
GR_MSG_SETFOREGROUND *mp;
{
/* Process a set foreground message. */
  gr_foreground = mp->foreground;
}


/*===========================================================================*
 *				do_setbackground			     *
 *===========================================================================*/
PRIVATE void do_setbackground(mp)
GR_MSG_SETBACKGROUND *mp;
{
/* Process a set background message. */
  gr_background = mp->background;
}


/*===========================================================================*
 *				do_setusebackground			     *
 *===========================================================================*/
PRIVATE void do_setusebackground(mp)
GR_MSG_SETUSEBACKGROUND *mp;
{
/* Process a set use background message. */
  gr_usebg = mp->flag;
}


/*===========================================================================*
 *				do_area8				     *
 *===========================================================================*/
PRIVATE void do_area8(mp)
GR_MSG_AREA8 *mp;
{
/* Process a draw 8 bit color values message. */
  drawarea8(mp->x, mp->y, mp->width, mp->height, mp->colors);
}


/*===========================================================================*
 *				do_bitmap				     *
 *===========================================================================*/
PRIVATE void do_bitmap(mp)
GR_MSG_BITMAP *mp;
{
/* Process a draw bitmap message. */
  drawbitmap(mp->x, mp->y, mp->width, mp->height, mp->bitmaps);
}


/*===========================================================================*
 *				do_setcliprects				     *
 *===========================================================================*/
PRIVATE void do_setcliprects(mp)
GR_MSG_SETCLIPRECTS *mp;
{
/* Process a set clip rectangles message. */
  if ((mp->head.count < 0) || (mp->head.count > GR_MAX_CLIPRECTS)) {
	gr_error = GR_TRUE;
	return;
  }
  clipinit(mp->head.count, mp->cliprects);
}


/*===========================================================================*
 *				do_poly					     *
 *===========================================================================*/
PRIVATE void do_poly(pp)
GR_MSG_POLY *pp;
{
/* Process a draw polygon message. */
  if (pp->head.flags & GR_FLAG_FILLAREA)
	fillpoly(pp->points, pp->head.count);
  else
	drawpoly(pp->points, pp->head.count);
}


/*===========================================================================*
 *				do_text					     *
 *===========================================================================*/
PRIVATE void do_text(cp)
GR_MSG_TEXT *cp;
{
/* Process a draw text string message. */
  drawtext(cp->x, cp->y, cp->chars, cp->head.count);
}


/*===========================================================================*
 *				do_setcursor				     *
 *===========================================================================*/
PRIVATE void do_setcursor(cp)
GR_MSG_SETCURSOR *cp;
{
/* Process a set cursor message. */
  (*gr_dev.setcursor) (cp->width, cp->height, cp->foreground,
		     cp->background, cp->bitmaps,
		     &cp->bitmaps[cp->head.count / 2]);
}


/*===========================================================================*
 *				do_movecursor				     *
 *===========================================================================*/
PRIVATE void do_movecursor(mp)
GR_MSG_MOVECURSOR *mp;
{
/* Process a move cursor message. */
  (*gr_dev.movecursor) (mp->x, mp->y);
}


/*===========================================================================*
 *				do_copyarea				     *
 *===========================================================================*/
PRIVATE void do_copyarea(mp)
GR_MSG_COPYAREA *mp;
{
/* Process a copy area message. */
  copyarea(mp->sx, mp->sy, mp->width, mp->height, mp->dx, mp->dy);
}

/* END CODE */
