/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Top level drawing primitives.
 * These routines do the necessary range checking, clipping, and cursor
 * overwriting checks, and then call the lower level device dependent
 * routines to actually do the drawing.  The lower level routines are
 * only called when it is known that all the pixels to be drawn are
 * within the device area and are visible.
 */

#include "kernel.h"
#include "graph_dev.h"


/* Global symbols used throughout the graphics code. */
GR_COLOR gr_foreground;		/* current foreground color */
GR_COLOR gr_background;		/* current background color */
GR_BOOL gr_usebg;		/* TRUE if background drawn in pixmaps */
GR_BOOL gr_error;		/* true if error occurred */
GR_MODE gr_mode;		/* drawing mode */


#define	min(a,b) ((a) < (b) ? (a) : (b))

FORWARD void drawrow();
FORWARD void drawcol();
FORWARD void draw4points();
FORWARD void extendrow();


/*===========================================================================*
 *				fillellipse				     *
 *===========================================================================*/
PUBLIC void fillellipse(x, y, rx, ry)
GR_COORD x, y;			/* coordinates of center of ellipse */
GR_SIZE rx;			/* radius along x axis */
GR_SIZE ry;			/* radius along y axis */
{
/* Fill an ellipse in the foreground color, applying clipping if necessary. */
  int xp, yp;			/* current point (based on center) */
  long Asquared;		/* square of x semi axis */
  long TwoAsquared;
  long Bsquared;		/* square of y semi axis */
  long TwoBsquared;
  long d;
  long dx, dy;

  if ((rx < 0) || (ry < 0)) return;

  /* See if the ellipse is either totally visible or totally invisible.
   * If so, then the ellipse drawing is easy.
   */
  switch (cliparea(x - rx, y - ry, x + rx, y + ry)) {
      case GR_CLIP_VISIBLE:
	(*gr_dev.fillellipse) (x, y, rx, ry, gr_foreground);
	/* Fall into return */
      case GR_CLIP_INVISIBLE:	return;
  }

  xp = 0;
  yp = ry;
  Asquared = rx * rx;
  TwoAsquared = 2 * Asquared;
  Bsquared = ry * ry;
  TwoBsquared = 2 * Bsquared;
  d = Bsquared - Asquared * ry + (Asquared >> 2);
  dx = 0;
  dy = TwoAsquared * ry;

  while (dx < dy) {
	drawrow(x - xp, x + xp, y - yp);
	drawrow(x - xp, x + xp, y + yp);
	if (d > 0) {
		yp--;
		dy -= TwoAsquared;
		d -= dy;
	}
	xp++;
	dx += TwoBsquared;
	d += (Bsquared + dx);
  }
  d += ((3L * (Asquared - Bsquared) / 2L - (dx + dy)) >> 1);
  while (yp >= 0) {
	drawrow(x - xp, x + xp, y - yp);
	drawrow(x - xp, x + xp, y + yp);
	if (d < 0) {
		xp++;
		dx += TwoBsquared;
		d += dx;
	}
	yp--;
	dy -= TwoAsquared;
	d += (Asquared - dy);
  }
}


/*===========================================================================*
 *				drawpoint				     *
 *===========================================================================*/
PUBLIC void drawpoint(x, y)
GR_COORD x;
GR_COORD y;
{
/* Draw an individual point in the foreground color, applying
 * clipping if necessary.
 */
  if (clippoint(x, y)) (*gr_dev.drawpoint) (x, y, gr_foreground);
}


/*===========================================================================*
 *				drawline				     *
 *===========================================================================*/
PUBLIC void drawline(x1, y1, x2, y2)
GR_COORD x1;
GR_COORD y1;
GR_COORD x2;
GR_COORD y2;
{
/* Draw an arbitrary line in the foreground color, applying
 * clipping if necessary.
 */
  int xdelta;			/* width of rectangle around line */
  int ydelta;			/* height of rectangle around line */
  int xinc;			/* increment for moving x coordinate */
  int yinc;			/* increment for moving y coordinate */
  int rem;			/* current remainder */

  /* See if the line is horizontal or vertical. If so, then call
   * special routines.
   */
  if (y1 == y2) {
	drawrow(x1, x2, y1);
	return;
  }
  if (x1 == x2) {
	drawcol(x1, y1, y2);
	return;
  }

  /* See if the line is either totally visible or totally invisible. If
   * so, then the line drawing is easy.
   */
  switch (cliparea(x1, y1, x2, y2)) {
      case GR_CLIP_VISIBLE:
	(*gr_dev.drawline) (x1, y1, x2, y2, gr_foreground);
	/* Fall into return */
      case GR_CLIP_INVISIBLE:	return;
  }

  /* The line may be partially obscured. Do the draw line algorithm
   * checking each point against the clipping regions.
   */
  xdelta = x2 - x1;
  ydelta = y2 - y1;
  if (xdelta < 0) xdelta = -xdelta;
  if (ydelta < 0) ydelta = -ydelta;
  xinc = (x2 > x1) ? 1 : -1;
  yinc = (y2 > y1) ? 1 : -1;
  if (clippoint(x1, y1)) (*gr_dev.drawpoint) (x1, y1, gr_foreground);
  if (xdelta >= ydelta) {
	rem = xdelta / 2;
	do {
		x1 += xinc;
		rem += ydelta;
		if (rem >= xdelta) {
			rem -= xdelta;
			y1 += yinc;
		}
		if (clippoint(x1, y1))
			(*gr_dev.drawpoint) (x1, y1, gr_foreground);
	} while (x1 != x2);
  } else {
	rem = ydelta / 2;
	do {
		y1 += yinc;
		rem += xdelta;
		if (rem >= ydelta) {
			rem -= ydelta;
			x1 += xinc;
		}
		if (clippoint(x1, y1))
			(*gr_dev.drawpoint) (x1, y1, gr_foreground);
	} while (y1 != y2);
  }
}


/*===========================================================================*
 *				drawrow					     *
 *===========================================================================*/
PRIVATE void drawrow(x1, x2, y)
GR_COORD x1;
GR_COORD x2;
GR_COORD y;
{
/* Draw a horizontal line in the foreground color, applying
 * clipping if necessary.
 */
  GR_COORD temp;

  if (x1 > x2) {
	temp = x1;
	x1 = x2;
	x2 = temp;
  }
  if (x1 < 0) x1 = 0;
  if (x2 >= gr_dev.cols) x2 = gr_dev.cols - 1;

  (*gr_dev.checkcursor) (x1, y, x2, y);

  while (x1 <= x2) {
	if (clippoint(x1, y)) {
		temp = min(clipmaxx, x2);
		(*gr_dev.drawrow) (x1, temp, y, gr_foreground);
	} else
		temp = min(clipmaxx, x2);
	x1 = temp + 1;
  }
}


/*===========================================================================*
 *				drawcol					     *
 *===========================================================================*/
PRIVATE void drawcol(x, y1, y2)
GR_COORD x;
GR_COORD y1;
GR_COORD y2;
{
/* Draw a vertical line in the foreground color, applying
 * clipping if necessary.
 */
  GR_COORD temp;

  if (y1 > y2) {
	temp = y1;
	y1 = y2;
	y2 = temp;
  }
  if (y1 < 0) y1 = 0;
  if (y2 >= gr_dev.rows) y2 = gr_dev.rows - 1;

  (*gr_dev.checkcursor) (x, y1, x, y2);

  while (y1 <= y2) {
	if (clippoint(x, y1)) {
		temp = min(clipmaxy, y2);
		(*gr_dev.drawcol) (x, y1, temp, gr_foreground);
	} else
		temp = min(clipmaxy, y2);
	y1 = temp + 1;
  }
}


/*===========================================================================*
 *				drawrect				     *
 *===========================================================================*/
PUBLIC void drawrect(x, y, width, height)
GR_COORD x;
GR_COORD y;
GR_COORD width;
GR_COORD height;
{
/* Draw a rectangle in the foreground color, applying clipping if necessary.
 * This is careful to not draw points multiple times in case the rectangle
 * is being drawn using XOR.
 */
  GR_COORD maxx;
  GR_COORD maxy;

  if ((width <= 0) || (height <= 0)) return;
  maxx = x + width - 1;
  maxy = y + height - 1;
  drawline(x, y, maxx, y);
  if (height > 1) drawline(x, maxy, maxx, maxy);
  if (height < 3) return;
  y++;
  maxy--;
  drawline(x, y, x, maxy);
  if (width > 1) drawline(maxx, y, maxx, maxy);
}


/*===========================================================================*
 *				fillrect				     *
 *===========================================================================*/
PUBLIC void fillrect(x1, y1, x2, y2)
GR_COORD x1;
GR_COORD y1;
GR_COORD x2;
GR_COORD y2;
{
/* Draw a filled in rectangle in the foreground color, applying
 * clipping if necessary.
 */
  GR_COORD temp;

  if (y1 == y2) {
	drawrow(x1, x2, y1);
	return;
  }
  if (x1 == x2) {
	drawcol(x1, y1, y2);
	return;
  }
  if (x1 > x2) {
	temp = x1;
	x1 = x2;
	x2 = temp;
  }
  if (y1 > y2) {
	temp = y1;
	y1 = y2;
	y2 = temp;
  }

  /* See if the rectangle is either totally visible or totally
   * invisible. If so, then the rectangle drawing is easy.
   */
  switch (cliparea(x1, y1, x2, y2)) {
      case GR_CLIP_VISIBLE:
	(*gr_dev.fillrect) (x1, y1, x2, y2, gr_foreground);
	/* Fall into return */
      case GR_CLIP_INVISIBLE:	return;
  }

  /* The rectangle may be partially obstructed. So do it line by line. */
  while (y1 <= y2) drawrow(x1, x2, y1++);
}


/*===========================================================================*
 *				drawpoly				     *
 *===========================================================================*/
PUBLIC void drawpoly(points, count)
GR_POINT *points;
GR_COUNT count;
{
/* Draw a polygon in the foreground color, applying clipping if necessary.
 * The polygon is only closed if the first point is repeated at the end.
 * Some care is taken to plot the endpoints correctly if the current
 * drawing mode is XOR.  However, internal crossings are not handled
 * correctly.
 */
  GR_COORD firstx;
  GR_COORD firsty;
  GR_BOOL didline;

  if (count < 2) return;
  firstx = points->x;
  firsty = points->y;
  didline = GR_FALSE;

  while (count-- > 1) {
	if (didline && (gr_mode == GR_MODE_XOR))
		drawpoint(points->x, points->y);
	drawline(points[0].x, points[0].y, points[1].x, points[1].y);
	points++;
	didline = GR_TRUE;
  }
  if (gr_mode != GR_MODE_XOR) return;
  points--;
  if ((points->x == firstx) && (points->y == firsty))
	drawpoint(points->x, points->y);
}


/*===========================================================================*
 *				fillpoly				     *
 *===========================================================================*/
PUBLIC void fillpoly(points, count)
GR_POINT *points;
GR_COUNT count;
{
/* Fill a polygon in the foreground color, applying clipping if necessary.
 * Note: this routine currently only correctly fills convex polygons.
 */
  GR_POINT *pp;			/* current point */
  GR_COORD miny;		/* minimum row */
  GR_COORD maxy;		/* maximum row */
  GR_COORD minx;		/* minimum column */
  GR_COORD maxx;		/* maximum column */
  GR_COUNT i;			/* counter */

  if (count <= 0) return;

  /* First determine the minimum and maximum rows for the polygon. */
  pp = points;
  miny = pp->y;
  maxy = pp->y;
  for (i = count; i-- > 0; pp++) {
	if (miny > pp->y) miny = pp->y;
	if (maxy < pp->y) maxy = pp->y;
  }
  if (miny < 0) miny = 0;
  if (maxy >= gr_dev.rows) maxy = gr_dev.rows - 1;
  if (miny > maxy) return;

  /* Now for each row, scan the list of points and determine the
   * minimum and maximum x coordinate for each line, and plot the row.
   * The last point connects with the first point automatically.
   */
  for (; miny <= maxy; miny++) {
	minx = GR_COORD_MAX;
	maxx = GR_COORD_MIN;
	pp = points;
	for (i = count; --i > 0; pp++) {
		extendrow(miny, pp[0].x, pp[0].y, pp[1].x,
			  pp[1].y, &minx, &maxx);
	}
	extendrow(miny, pp[0].x, pp[0].y, points[0].x,
		  points[0].y, &minx, &maxx);

	if (minx <= maxx) drawrow(minx, maxx, miny);
  }
}


/*===========================================================================*
 *				extendrow				     *
 *===========================================================================*/
PRIVATE void extendrow(y, x1, y1, x2, y2, minxptr, maxxptr)
GR_COORD y;			/* row to check for intersection */
GR_COORD x1;			/* x coordinate of first endpoint */
GR_COORD y1;			/* y coordinate of first endpoint */
GR_COORD x2;			/* x coordinate of second endpoint */
GR_COORD y2;			/* y coordinate of second endpoint */
GR_COORD *minxptr;		/* address of current minimum x */
GR_COORD *maxxptr;		/* address of current maximum x */
{
/* Utility routine for filling polygons.  Find the intersection point (if
 * any) of a horizontal line with an arbitrary line, and extend the current
 * minimum and maximum x values as needed to include the intersection point.
 */
  GR_COORD x;			/* x coordinate of intersection */
  long num;			/* numerator of fraction */

  /* First make sure the specified line segment includes the specified
   * row number.  If not, then there is no intersection.
   */
  if (((y < y1) || (y > y2)) && ((y < y2) || (y > y1))) return;

  /* If a horizontal line, then check the two endpoints. */
  if (y1 == y2) {
	if (*minxptr > x1) *minxptr = x1;
	if (*minxptr > x2) *minxptr = x2;
	if (*maxxptr < x1) *maxxptr = x1;
	if (*maxxptr < x2) *maxxptr = x2;
	return;
  }

  /* If a vertical line, then check the x coordinate. */
  if (x1 == x2) {
	if (*minxptr > x1) *minxptr = x1;
	if (*maxxptr < x1) *maxxptr = x1;
	return;
  }

  /* An arbitrary line.  Calculate the intersection point using the
   * formula x = x1 + (y - y1) * (x2 - x1) / (y2 - y1).
   */
  num = ((long) (y - y1)) * ((long) (x2 - x1));
  x = x1 + num / (y2 - y1);
  if (*minxptr > x) *minxptr = x;
  if (*maxxptr < x) *maxxptr = x;
}


/*===========================================================================*
 *				drawellipse				     *
 *===========================================================================*/
PUBLIC void drawellipse(x, y, rx, ry)
GR_COORD x, y;			/* coordinates of center of ellipse */
GR_SIZE rx;			/* radius along x axis */
GR_SIZE ry;			/* radius along y axis */
{
/* Draw the boundary of an ellipse in the foreground color, applying
 * clipping if necessary.
 */
  int xp, yp;			/* current point (based on center) */
  long Asquared;		/* square of x semi axis */
  long TwoAsquared;
  long Bsquared;		/* square of y semi axis */
  long TwoBsquared;
  long d;
  long dx, dy;

  if ((rx < 0) || (ry < 0)) return;
  /* See if the ellipse is either totally visible or totally invisible.
   * If so, then the ellipse drawing is easy.
   */
  switch (cliparea(x - rx, y - ry, x + rx, y + ry)) {
      case GR_CLIP_VISIBLE:
	(*gr_dev.drawellipse) (x, y, rx, ry, gr_foreground);
	/* Fall into return */
      case GR_CLIP_INVISIBLE:	return;
  }

  xp = 0;
  yp = ry;
  Asquared = rx * rx;
  TwoAsquared = 2 * Asquared;
  Bsquared = ry * ry;
  TwoBsquared = 2 * Bsquared;
  d = Bsquared - Asquared * ry + (Asquared >> 2);
  dx = 0;
  dy = TwoAsquared * ry;

  while (dx < dy) {
	draw4points(x, y, xp, yp);
	if (d > 0) {
		yp--;
		dy -= TwoAsquared;
		d -= dy;
	}
	xp++;
	dx += TwoBsquared;
	d += (Bsquared + dx);
  }
  d += ((3L * (Asquared - Bsquared) / 2L - (dx + dy)) >> 1);
  while (yp >= 0) {
	draw4points(x, y, xp, yp);
	if (d < 0) {
		xp++;
		dx += TwoBsquared;
		d += dx;
	}
	yp--;
	dy -= TwoAsquared;
	d += (Asquared - dy);
  }
}


/*===========================================================================*
 *				draw4points				     *
 *===========================================================================*/
PRIVATE void draw4points(x, y, px, py)
GR_COORD x, y;			/* center of the points */
GR_SIZE px, py;			/* point to plot (based on center) */
{
/* Set four points symmetrically situated around a point. */
  drawpoint(x + px, y + py);
  drawpoint(x - px, y + py);
  drawpoint(x + px, y - py);
  drawpoint(x - px, y - py);
}


/*===========================================================================*
 *				copyarea				     *
 *===========================================================================*/
PUBLIC void copyarea(srcx, srcy, width, height, destx, desty)
{
/* Copy a rectangular area from one screen area to another.
 * This bypasses clipping.
 */
  GR_COORD srcx;		/* leftmost column of area to copy */
  GR_COORD srcy;		/* topmost row of area to copy */
  GR_SIZE width;		/* width of area to copy */
  GR_SIZE height;		/* height of area to copy */
  GR_COORD destx;		/* leftmost column of destination */
  GR_COORD desty;		/* topmost row of destination */

  if ((width <= 0) || (height <= 0)) return;
  if ((srcx == destx) && (srcy == desty)) return;
  (*gr_dev.checkcursor) (srcx, srcy, srcx + width - 1, srcy + height - 1);
  (*gr_dev.checkcursor) (destx, desty, destx + width - 1, desty + height - 1);
  (*gr_dev.copyarea) (srcx, srcy, width, height, destx, desty);
}


/*===========================================================================*
 *				drawarea8				     *
 *===========================================================================*/
PUBLIC void drawarea8(x, y, width, height, table)
GR_COORD x;			/* leftmost column of area */
GR_COORD y;			/* topmost row of area */
GR_SIZE width;			/* width of area */
GR_SIZE height;			/* height of area */
GR_COLOR8 *table;		/* table of 8 bit color values */
{
/* Draw a rectangle of color values, clipping if necessary.
 * The rectangle is composed of 8 bit color values so that each color
 * only uses one character.  If a color matches the background color,
 * that that pixel is only drawn if the gr_usebg flag is set.
 */
  long cellstodo;		/* remaining number of cells */
  long count;			/* number of cells of same color */
  long cc;			/* current cell count */
  long rows;			/* number of complete rows */
  GR_COORD minx;		/* minimum x value */
  GR_COORD maxx;		/* maximum x value */
  GR_COLOR savecolor;		/* saved foreground color */
  GR_BOOL dodraw;		/* TRUE if draw these points */

  minx = x;
  maxx = x + width - 1;

  /* See if the area is either totally visible or totally invisible. If
   * so, then the area drawing is easy.
   */
  switch (cliparea(minx, y, maxx, y + height - 1)) {
      case GR_CLIP_VISIBLE:
	(*gr_dev.drawarea8) (x, y, width, height, table);
	/* Fall into return */
      case GR_CLIP_INVISIBLE:	return;
  }

  savecolor = gr_foreground;
  cellstodo = width * height;
  while (cellstodo > 0) {
	/* See how many of the adjacent remaining points have the
	 * same color as the next point.
	 */
	gr_foreground = *table++;
	dodraw = (gr_usebg || (gr_foreground != gr_background));
	count = 1;
	cellstodo--;
	while ((cellstodo > 0) && (gr_foreground == *table)) {
		table++;
		count++;
		cellstodo--;
	}

	/* If there is only one point with this color, then draw it
	 * by itself.
	 */
	if (count == 1) {
		if (dodraw) drawpoint(x, y);
		if (++x > maxx) {
			x = minx;
			y++;
		}
		continue;
	}

	/* There are multiple points with the same color. If we are
	 * not at the start of a row of the rectangle, then draw this
	 * first row specially.
	 */
	if (x != minx) {
		cc = count;
		if (x + cc - 1 > maxx) cc = maxx - x + 1;
		if (dodraw) drawrow(x, x + cc - 1, y);
		count -= cc;
		x += cc;
		if (x > maxx) {
			x = minx;
			y++;
		}
	}

	/* Now the x value is at the beginning of a row if there are
	 * any points left to be drawn.  Draw all the complete rows
	 * with one call.
	 */
	rows = count / width;
	if (rows > 0) {
		if (dodraw) fillrect(x, y, maxx, y + rows - 1);
		count %= width;
		y += rows;
	}

	/* If there is a final partial row of pixels left to be
	 * drawn, then do that.
	 */
	if (count > 0) {
		if (dodraw) drawrow(x, x + count - 1, y);
		x += count;
	}
  }
  gr_foreground = savecolor;
}


/*===========================================================================*
 *				drawbitmap				     *
 *===========================================================================*/
PUBLIC void drawbitmap(x, y, width, height, table)
GR_COORD x;			/* leftmost column of area */
GR_COORD y;			/* topmost row of area */
GR_SIZE width;			/* width of area */
GR_SIZE height;			/* height of area */
GR_BITMAP *table;		/* table of bitmaps */
{
/* Draw a rectangle of foreground (and possibly background) colors as
 * determined by the specified bitmap, clipping if necessary.  The
 * background is only drawn if the gr_usebg flag is set.
 */
  GR_COORD minx;
  GR_COORD maxx;
  GR_COLOR savecolor;		/* saved foreground color */
  GR_BITMAP bitvalue;		/* bitmap word value */
  int bitcount;			/* number of bits left in bitmap word */

  switch (cliparea(x, y, x + width - 1, y + height - 1)) {
      case GR_CLIP_VISIBLE:
	if (gr_usebg) {
		(*gr_dev.fillrect) (x, y, x + width - 1,
				    y + height - 1, gr_background);
	}
	(*gr_dev.drawbitmap) (x, y, width, height, table, gr_foreground);
	/* Fall into return */

      case GR_CLIP_INVISIBLE:	return;
  }

  /* The rectangle is partially visible, so must do clipping. First
   * fill a rectangle in the background color if necessary.
   */
  if (gr_usebg) {
	savecolor = gr_foreground;
	gr_foreground = gr_background;
	fillrect(x, y, x + width - 1, y + height - 1);
	gr_foreground = savecolor;
  }
  minx = x;
  maxx = x + width - 1;
  bitcount = 0;
  while (height > 0) {
	if (bitcount <= 0) {
		bitcount = GR_BITMAPBITS;
		bitvalue = *table++;
	}
	if (GR_TESTBIT(bitvalue) && clippoint(x, y))
		(*gr_dev.drawpoint) (x, y, gr_foreground);
	bitvalue = GR_SHIFTBIT(bitvalue);
	bitcount--;
	if (x++ == maxx) {
		x = minx;
		y++;
		height--;
		bitcount = 0;
	}
  }
}


/*===========================================================================*
 *				drawtext				     *
 *===========================================================================*/
PUBLIC void drawtext(x, y, str, cc)
GR_COORD x, y;			/* location to draw string at */
GR_CHAR *str;			/* string to draw */
GR_SIZE cc;			/* number of characters */
{
/* Draw a text string at a specifed coordinates in the foreground color
 * (and possibly the background color), applying clipping if necessary.
 * The background color is only drawn if the gr_usebg flag is set.
 */
  GR_SIZE width;		/* width of text area */
  GR_SIZE height;		/* height of text area */
  GR_BITMAP bitmap[14];		/* bitmaps for characters */

  if ((cc <= 0) || (y < 0) || (x >= gr_dev.cols)) return;

  (*gr_dev.sizetext) (str, cc, &width, &height);

  switch (cliparea(x, y - height + 1, x + width - 1, y)) {
      case GR_CLIP_VISIBLE:
	if (gr_usebg) {
		(*gr_dev.fillrect) (x, y - height + 1,
				    x + width - 1, y, gr_background);
	}
	(*gr_dev.drawtext) (x, y, str, cc, gr_foreground);
	/* Fall into return */

      case GR_CLIP_INVISIBLE:	return;
  }

  /* Get the bitmap for each character individually, and then display
   * them using clipping for each one.
   */
  y -= (height - 1);
  while ((cc-- > 0) && (x < gr_dev.cols)) {
	(*gr_dev.getcharbits) (*str++, bitmap, &width, &height);
	drawbitmap(x, y, width, height, bitmap);
	x += width;
  }
}

/* END CODE */
