/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Generic device dependent routines.
 * All of these routines use other device routines to perform their work,
 * eventually getting to real device routines.  In the worst case, the
 * only real device routine implemented is drawpoint.  In the best case,
 * none of these generic routines are needed since the device can do
 * them all efficiently.  Since these are "device" level routines, no
 * clip rectangle checks are necessary since they should have already been
 * done by the calling routines, but clip checks against the device
 * area is reasonable.
 */

#include "kernel.h"
#include "graph_dev.h"

FORWARD void draw4points();


/*===========================================================================*
 *				gen_drawline				     *
 *===========================================================================*/
PUBLIC void gen_drawline(x1, y1, x2, y2, color)
GR_COORD x1;
GR_COORD y1;
GR_COORD x2;
GR_COORD y2;
GR_COLOR color;
{
/* Generic draw line routine. */
  GR_COORD temp;
  int xdelta;
  int ydelta;
  int xinc;
  int yinc;
  int rem;

  if (y1 == y2) {
	(*gr_dev.drawrow) (x1, x2, y1, color);
	return;
  }
  if (x1 == x2) {
	(*gr_dev.drawcol) (x1, y1, y2, color);
	return;
  }
  xdelta = x2 - x1;
  ydelta = y2 - y1;
  if (xdelta < 0) xdelta = -xdelta;
  if (ydelta < 0) ydelta = -ydelta;
  xinc = (x2 > x1) ? 1 : -1;
  yinc = (y2 > y1) ? 1 : -1;
  (*gr_dev.drawpoint) (x1, y1, color);
  if (xdelta >= ydelta) {
	rem = xdelta / 2;
	do {
		x1 += xinc;
		rem += ydelta;
		if (rem >= xdelta) {
			rem -= xdelta;
			y1 += yinc;
		}
		(*gr_dev.drawpoint) (x1, y1, color);
	} while (x1 != x2);
  } else {
	rem = ydelta / 2;
	do {
		y1 += yinc;
		rem += xdelta;
		if (rem >= ydelta) {
			rem -= ydelta;
			x1 += xinc;
		}
		(*gr_dev.drawpoint) (x1, y1, color);
	} while (y1 != y2);
  }
}


/*===========================================================================*
 *				gen_drawrow				     *
 *===========================================================================*/
PUBLIC void gen_drawrow(x1, x2, y, color)
GR_COORD x1;
GR_COORD x2;
GR_COORD y;
GR_COLOR color;
{
/* Generic draw row routine. */
  while (x1 <= x2) (*gr_dev.drawpoint) (x1++, y, color);
}


/*===========================================================================*
 *				gen_drawcol				     *
 *===========================================================================*/
PUBLIC void gen_drawcol(x, y1, y2, color)
GR_COORD x;
GR_COORD y1;
GR_COORD y2;
GR_COLOR color;
{
/* Generic draw column routine. */
  while (y1 <= y2) (*gr_dev.drawpoint) (x, y1++, color);
}


/*===========================================================================*
 *				gen_fillrect				     *
 *===========================================================================*/
PUBLIC void gen_fillrect(x1, y1, x2, y2, color)
GR_COORD x1;
GR_COORD y1;
GR_COORD x2;
GR_COORD y2;
GR_COLOR color;
{
/* Generic fill rectangle routine. */
  while (y1 <= y2) (*gr_dev.drawrow) (x1, x2, y1++, color);
}


/*===========================================================================*
 *				gen_drawellipse				     *
 *===========================================================================*/
PUBLIC void gen_drawellipse(x, y, rx, ry, color)
GR_COORD x, y;			/* coordinates of center of ellipse */
GR_SIZE rx;			/* radius along x axis */
GR_SIZE ry;			/* radius along y axis */
{
/* Generic ellipse drawing routine. */
  int xp, yp;			/* current point (based on center) */
  long Asquared;		/* square of x semi axis */
  long TwoAsquared;
  long Bsquared;		/* square of y semi axis */
  long TwoBsquared;
  long d;
  long dx, dy;

  if ((rx < 0) || (ry < 0)) return;

  xp = 0;
  yp = ry;
  Asquared = rx * rx;
  TwoAsquared = 2 * Asquared;
  Bsquared = ry * ry;
  TwoBsquared = 2 * Bsquared;
  d = Bsquared - Asquared * ry + (Asquared >> 2);
  dx = 0;
  dy = TwoAsquared * ry;

  while (dx < dy) {
	draw4points(x, y, xp, yp, color);
	if (d > 0) {
		yp--;
		dy -= TwoAsquared;
		d -= dy;
	}
	xp++;
	dx += TwoBsquared;
	d += (Bsquared + dx);
  }
  d += ((3L * (Asquared - Bsquared) / 2L - (dx + dy)) >> 1);
  while (yp >= 0) {
	draw4points(x, y, xp, yp, color);
	if (d < 0) {
		xp++;
		dx += TwoBsquared;
		d += dx;
	}
	yp--;
	dy -= TwoAsquared;
	d += (Asquared - dy);
  }
}


/*===========================================================================*
 *				draw4points				     *
 *===========================================================================*/
PRIVATE void draw4points(x, y, px, py, color)
GR_COORD x, y;			/* center of the points */
GR_SIZE px, py;			/* point to plot (based on center) */
GR_COLOR color;
{
/* Set four points symmetrically situated around a point. */
  (*gr_dev.drawpoint) (x + px, y + py, color);
  (*gr_dev.drawpoint) (x - px, y + py, color);
  (*gr_dev.drawpoint) (x + px, y - py, color);
  (*gr_dev.drawpoint) (x - px, y - py, color);
}


/*===========================================================================*
 *				gen_fillellipse				     *
 *===========================================================================*/
PUBLIC void gen_fillellipse(x, y, rx, ry, color)
GR_COORD x, y;			/* coordinates of center of ellipse */
GR_SIZE rx;			/* radius along x axis */
GR_SIZE ry;			/* radius along y axis */
GR_COLOR color;
{
/* Generic fill ellipse routine. */
  int xp, yp;			/* current point (based on center) */
  long Asquared;		/* square of x semi axis */
  long TwoAsquared;
  long Bsquared;		/* square of y semi axis */
  long TwoBsquared;
  long d;
  long dx, dy;

  if ((rx < 0) || (ry < 0)) return;

  xp = 0;
  yp = ry;
  Asquared = rx * rx;
  TwoAsquared = 2 * Asquared;
  Bsquared = ry * ry;
  TwoBsquared = 2 * Bsquared;
  d = Bsquared - Asquared * ry + (Asquared >> 2);
  dx = 0;
  dy = TwoAsquared * ry;

  while (dx < dy) {
	(*gr_dev.drawrow) (x - xp, x + xp, y - yp, color);
	(*gr_dev.drawrow) (x - xp, x + xp, y + yp, color);
	if (d > 0) {
		yp--;
		dy -= TwoAsquared;
		d -= dy;
	}
	xp++;
	dx += TwoBsquared;
	d += (Bsquared + dx);
  }
  d += ((3L * (Asquared - Bsquared) / 2L - (dx + dy)) >> 1);
  while (yp >= 0) {
	(*gr_dev.drawrow) (x - xp, x + xp, y - yp, color);
	(*gr_dev.drawrow) (x - xp, x + xp, y + yp, color);
	if (d < 0) {
		xp++;
		dx += TwoBsquared;
		d += dx;
	}
	yp--;
	dy -= TwoAsquared;
	d += (Asquared - dy);
  }
}


/*===========================================================================*
 *				gen_drawarea8				     *
 *===========================================================================*/
PUBLIC void gen_drawarea8(x, y, width, height, table)
GR_COORD x;			/* leftmost column of area */
GR_COORD y;			/* topmost row of area */
GR_SIZE width;			/* width of area */
GR_SIZE height;			/* height of area */
GR_COLOR8 *table;		/* table of 8 bit color values */
{
/* Generic drawing of rectangular area with 8 bit color values.
 * The rectangle is composed of 8 bit color values so that each color
 * only uses one character.  If a color matches the background color,
 * that that pixel is only drawn if the gr_usebg flag is set.
 */
  long cellstodo;		/* remaining number of cells */
  long count;			/* number of cells of same color */
  long cc;			/* current cell count */
  long rows;			/* number of complete rows */
  GR_COORD minx;		/* minimum x value */
  GR_COORD maxx;		/* maximum x value */
  GR_COLOR color;		/* color of these pixels */
  GR_BOOL dodraw;		/* TRUE if draw these points */

  minx = x;
  maxx = x + width - 1;
  cellstodo = width * height;
  while (cellstodo > 0) {
	/* See how many of the adjacent remaining points have the
	 * same color as the next point.
	 */
	color = *table++;
	dodraw = (gr_usebg || (color != gr_background));
	count = 1;
	cellstodo--;
	while ((cellstodo > 0) && (color == *table)) {
		table++;
		count++;
		cellstodo--;
	}

	/* If there is only one point with this color, then draw it
	 * by itself.
	 */
	if (count == 1) {
		if (dodraw) (*gr_dev.drawpoint) (x, y, color);
		if (++x > maxx) {
			x = minx;
			y++;
		}
		continue;
	}

	/* There are multiple points with the same color. If we are
	 * not at the start of a row of the rectangle, then draw this
	 * first row specially.
	 */
	if (x != minx) {
		cc = count;
		if (x + cc - 1 > maxx) cc = maxx - x + 1;
		if (dodraw) (*gr_dev.drawrow) (x, x + cc - 1, y, color);
		count -= cc;
		x += cc;
		if (x > maxx) {
			x = minx;
			y++;
		}
	}

	/* Now the x value is at the beginning of a row if there are
	 * any points left to be drawn.  Draw all the complete rows
	 * with one call.
	 */
	rows = count / width;
	if (rows > 0) {
		if (dodraw)
			(*gr_dev.fillrect) (x, y, maxx, y + rows - 1, color);
		count %= width;
		y += rows;
	}

	/* If there is a final partial row of pixels left to be
	 * drawn, then do that.
	 */
	if (count > 0) {
		if (dodraw) (*gr_dev.drawrow) (x, x + count - 1, y, color);
		x += count;
	}
  }
}


/*===========================================================================*
 *				gen_drawbitmap				     *
 *===========================================================================*/
PUBLIC void gen_drawbitmap(x, y, width, height, table, fgcolor)
GR_COORD x;			/* leftmost column of area */
GR_COORD y;			/* topmost row of area */
GR_SIZE width;			/* width of area */
GR_SIZE height;			/* height of area */
GR_BITMAP *table;		/* table of bitmaps */
GR_COLOR fgcolor;		/* foreground color */
{
/* Draw a rectanglular array of pixels in the foreground color as
 * determined by the specified bitmap.  This does not affect the
 * background pixels.
 */
  GR_COORD minx;
  GR_COORD maxx;
  GR_BITMAP bitvalue;		/* bitmap word value */
  int bitcount;			/* number of bits left in bitmap word */

  minx = x;
  maxx = x + width - 1;
  bitcount = 0;
  while (height > 0) {
	if (bitcount <= 0) {
		bitcount = GR_BITMAPBITS;
		bitvalue = *table++;
	}
	if (GR_TESTBIT(bitvalue)) (*gr_dev.drawpoint) (x, y, fgcolor);
	bitvalue = GR_SHIFTBIT(bitvalue);
	bitcount--;
	if (x++ == maxx) {
		x = minx;
		y++;
		height--;
		bitcount = 0;
	}
  }
}


/*===========================================================================*
 *				gen_copyarea				     *
 *===========================================================================*/
PUBLIC void gen_copyarea(srcx, srcy, width, height, destx, desty)
GR_COORD srcx;			/* leftmost column of area to copy */
GR_COORD srcy;			/* topmost row of area to copy */
GR_SIZE width;			/* width of area to copy */
GR_SIZE height;			/* height of area to copy */
GR_COORD destx;			/* leftmost column of destination */
GR_COORD desty;			/* topmost row of destination */
{
/* Generic copy of an area from one rectangle to another.
 * This bypasses clipping.
 */
  GR_COORD row;
  GR_COORD col;
  GR_COLOR color;

  for (row = 0; row < height; row++) {
	for (col = 0; col < width; col++) {
		color = (*gr_dev.readpoint) (srcx + col, srcy + row);
		(*gr_dev.drawpoint) (destx + col, desty + row, color);
	}
  }
}

/* END CODE */
