/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * The algorithms for some of these routines are taken from the book:
 * Programmer's Guide to PC and PS/2 Video Systems by Richard Wilton.
 *
 * Routines to draw pixels and lines for EGA/VGA resolutions.
 * The drawing mode in the data/rotate register is not changed in this
 * module, and must be changed as necessary by the callers.
 */

#include "protect.h"


/* Define one and only one of the following to be nonzero.
 * This is also defined in graph_ega.c.
 */
#define	VGA_ET4000	0	/* TSENG LABS ET4000 chip */
#define	VGA_STANDARD	1	/* standard VGA */
#define	EGA_STANDARD	0	/* standard EGA */


#if VGA_ET4000
#define	PIXROWS		600		/* number of pixels down */
#define	PIXCOLS		800		/* number of pixels across */
#define	PIXBYTES	100		/* number of bytes across */
#endif

#if VGA_STANDARD
#define	PIXROWS		480		/* number of pixels down */
#define	PIXCOLS		640		/* number of pixels across */
#define	PIXBYTES	80		/* number of bytes across */
#endif

#if EGA_STANDARD
#define	PIXROWS		350		/* number of pixels down */
#define	PIXCOLS		640		/* number of pixels across */
#define	PIXBYTES	80		/* number of bytes across */
#endif


#define	CHARROWS	14		/* number of pixels in char heights */
#define	CHARCOLS	8		/* number of pixels in char widths */
#define	BITMAPSIZE	2		/* number of bytes in GR_BITMAP */

#define	BUFFERADDR	0xa000		/* EGA buffer segment address */
#define	MEMADDR		0xa0000		/* physical memory address of buffer */
#define	PORTADDR	0x03ce		/* graphics controller port address */
#define	BITMASKREG	0x08		/* bit mask register */
#define	INITBITMASK	0xff08		/* set full bit mask output value */
#define	WRITEMODE2	0x0205		/* select write mode 2 output value */
#define	INITWRITEMODE	0x0005		/* restore default write mode 0 */
#define	PIXSHIFT	3		/* convert pixels to byte offset */


	.globl	_ega_getcharbits	/* read bitmap for a character */
	.globl	_ega_drawline		/* draw a line */
	.globl	_ega_drawpoint		/* set an individual pixel */
	.globl	_ega_drawtext		/* draw text string */
	.globl	_ega_readpoint		/* read an individual pixel */

	.bss
	.extern	_rom_char_addr		/* ROM char bitmap physical address */
	.text


/*
 * Routine to return the bitmap for a character from the ROM.
 * Called from C:
 *	ega_getcharbits(char, retbitmap, retwidth, retheight);
 *		GR_CHAR		char;
 *		GR_BITMAP	*retbitmap;
 *		GR_SIZE		*retwidth;
 *		GR_SIZE		*retheight;
 * The retbitmap is a pointer to an array of GR_BITMAP values (shorts),
 * of length CHARROWS.  Retwidth and retheight are the returned
 * width and height of the character.
 */

#define	char		8		/* character to get bitmap of */
#define	retaddr		12		/* returned address of bitmap */
#define	retwidth	16		/* returned width of character */
#define	retheight	20		/* returned height of character */


_ega_getcharbits:
	push	ebp		/* setup stack frame and preserve registers */
	mov	ebp, esp
	push	es

	xor	eax, eax	/* get character code */
	movb	al, char(ebp)
	mov	ecx, #CHARROWS	/* get number of bytes of data for chars */
	mul	ecx		/* make offset into ROM bitmap table */
	add	eax, _rom_char_addr	/* physical addr of bitmap for char */
	mov	bx, #FLAT_DS_SELECTOR	/* flat address space selector */
	mov	es, bx
	mov	ebx, retaddr(ebp)	/* address for returned bits */
	xor	edx, edx	/* clear low order part of bitmap values */

getcharloop:
	seg	es		/* get next byte of bitmap */
	movb	dh, (eax)
	mov	(ebx), dx	/* store in caller's buffer */
	inc	eax		/* advance ROM address */
	add	ebx, #BITMAPSIZE	/* and buffer address */
	loop	getcharloop	/* loop until did all rows of char */

	mov	ax, #CHARCOLS	/* return width of character */
	mov	ebx, retwidth(ebp)
	mov	(ebx), ax
	mov	ax, #CHARROWS	/* return height of character */
	mov	ebx, retheight(ebp)
	mov	(ebx), ax

	pop	es		/* restore registers and return */
	pop	ebp
	ret


/*
 * Routine to draw an arbitrary line.
 * Called from C:
 *	_ega_drawline(x1, y1, x2, y2, color);
 */

/* argument offsets, starting with 8 bytes (eip + ebp) */
#define	x1	8		/* first X coordinate */
#define	y1	12		/* first Y coordinate */
#define	x2	16		/* second X coordinate */
#define	y2	20		/* second Y coordinate */
#define	color	24		/* pixel value */

/* local variable offsets from ebp */
#define	rowincr	-4		/* byte increment between rows */
#define	incr1	-8		/* first increment */
#define	incr2	-12		/* second increment */
#define	routine	-16		/* routine to jump to */


_ega_drawline:
	push	ebp		/* setup stack frame and preserve registers */
	mov	ebp, esp

	/*
	 * Make sure that the line is totally within the screen area.
	 * If not, we are allowed to completely ignore the line.
	 */
	mov	eax, x1(ebp)	/* EAX := x1 */
	cmp	eax, #PIXCOLS	/* if ((x1<0) || (x1>=PIXCOLS)) return */
	jae	badline
	mov	eax, x2(ebp)	/* EAX := x2 */
	cmp	eax, #PIXCOLS	/* if ((x2<0) || (x2>=PIXCOLS)) return */
	jae	badline
	mov	eax, y1(ebp)	/* EAX := y1 */
	cmp	eax, #PIXROWS	/* if ((y1<0) || (y1>=PIXROWS)) return */
	jae	badline
	mov	eax, y2(ebp)	/* EAX := y2 */
	cmp	eax, #PIXROWS	/* if ((y2<0) || (y2>=PIXROWS)) return */
	jae	badline
	j	lineok

badline:
	pop	ebp
	ret

	/*
	 * Here when we know the line is totally drawable.
	 */
lineok:
	sub	esp, #16
	push	esi
	push	edi
	push	es

	/* configure the graphics controller */

	mov	dx, #PORTADDR	/* DX := Graphics Controller port address */
	movb	ah, color(ebp)	/* pixel value */
	xorb	al, al		/* Set/Reset register number (0) */
	out	dx, ax

	mov	ax, #0x0f01	/* AH := bit plane mask for Enable Set/Reset */
	out	dx, ax		/* AL := Enable Set/Reset register number */

	/* check for vertical line */

	mov	esi, #PIXBYTES	/* increment for video buffer */
	mov	ecx, x2(ebp)
	sub	ecx, x1(ebp)	/* CX := x2 - x1 */
	jnz	L117		/* jump if vertical line */
	jmp	VertLine
L117:
	/* force x1 < x2 */

	jns	L01		/* jump if x2 > x1 */

	neg	ecx		/* CX := x1 - x2 */

	mov	ebx, x2(ebp)	/* exchange x1 and x2 */
	xchg	ebx, x1(ebp)
	mov	x2(ebp), ebx

	mov	ebx, y2(ebp)	/* exchange y1 and y2 */
	xchg	ebx, y1(ebp)
	mov	y2(ebp), ebx

	/* calculate dy = ABS(y2 - y1) */

L01:	mov	ebx, y2(ebp)
	sub	ebx, y1(ebp)	/* BX := y2 - y1 */
	jnz	L115		/* jump if horizontal line */
	jmp	HorizLine
L115:
	jns	L03		/* jump if slope is positive */

	neg	ebx		/* BX := y1 - y2 */
	neg	esi		/* negate increment for buffer interleave */

	/* select appropriate routine for slope of line */

L03:	mov	rowincr(ebp), esi	/* save vertical increment */
	mov	routine(ebp), #LowSlope
	cmp	ebx, ecx
	jle	L04		/* jump if dy <= dx (slope <= 1) */
	mov	routine(ebp), #HighSlope
	xchg	ebx, ecx	/* exchange dy and dx */

	/* calculate initial decision variable and increments */

L04:	shl	ebx, #1		/* BX := 2 * dy */
	mov	incr1(ebp), ebx	/* incr1 := 2 * dy */
	sub	ebx, ecx
	mov	esi, ebx	/* SI := d = 2 * dy - dx */
	sub	ebx, ecx
	mov	incr2(ebp), ebx	/* incr2 := 2 * (dy - dx) */

	/* calculate first pixel address */

	push	ecx		/* preserve register */
	mov	eax, y1(ebp)	/* AX := y */
	mov	ebx, x1(ebp)	/* BX := x */
	call	PixelAddr	/* AH := bit mask */
				/* ES:BX -> buffer */
				/* CL := bits to shift left */

	mov	edi, ebx	/* es:di -> buffer */
	shlb	ah, cl		/* AH := bit mask in proper position */
	movb	bl, ah		/* AH,BL := bit mask */
	movb	al, #BITMASKREG	/* AL := Bit Mask Register number */

	pop	ecx		/* restore register */
	inc	ecx		/* CX := number of pixels to draw */

	jmp	#routine(ebp)	/* jump to appropriate routine for slope */


/*
 * Routine for vertical lines
 */
VertLine:
	mov	eax, y1(ebp)	/* AX := y1 */
	mov	ebx, y2(ebp)	/* BX := y2 */
	mov	ecx, ebx
	sub	ecx, eax	/* CX := dy */
	jge	L31		/* jump if dy >= 0 */

	neg	ecx		/* force dy >= 0 */
	mov	eax, ebx	/* AX := y2 */

L31:	inc	ecx		/* CX := number of pixels to draw */
	mov	ebx, x1(ebp)	/* BX := x */
	push	ecx		/* save register */
	call	PixelAddr	/* AH := bit mask */
				/* ES:BX -> video buffer */
				/* CL := number bits to shift left */

/* set up Graphics controller */

	shlb	ah, cl		/* AH := bit mask in proper position */
	movb	al, #BITMASKREG	/* AL := Bit Mask register number */
	out	dx, ax

	pop	ecx		/* restore register */

/* draw the line */

	push	ds
	push	es
	pop	ds
L111:	orb	(ebx), al	/* set pixel */
	add	ebx, esi	/* increment to next line */
	loop	L111
	pop	ds
	jmp	Lexit


/*
 * Routine for horizontal lines (slope = 0)
 */
HorizLine:
	push	ds		/* preserve DS */

	mov	eax, y1(ebp)
	mov	ebx, x1(ebp)
	call	PixelAddr	/* AH := bit mask */
				/* ES:BX -> video buffer */
				/* CL := number bits to shift left */
	mov	edi, ebx	/* ES:DI -> buffer */
	movb	dh, ah		/* DH := unshifted bit mask for left byte */

	notb	dh
	shlb	dh, cl		/* DH := reverse bit mask for first byte */
	notb	dh		/* DH := bit mask for first byte */

	mov	ecx, x2(ebp)
	andb	cl, #0x07
	xorb	cl, #0x07	/* CL := number of bits to shift left */
	movb	dl, #0xff	/* DL := unshifted bit mask for right byte */
	shlb	dl, cl		/* DL := bit mask for last byte */

	/* determine byte offset of first and last pixel in the line */

	mov	eax, x2(ebp)	/* AX := x2 */
	mov	ebx, x1(ebp)	/* BX := x1 */

	movb	cl, #PIXSHIFT	/* bits to convert pixels to bytes */

	shr	eax, cl		/* AX := byte offset of X2 */
	shr	ebx, cl		/* BX := byte offset of X1 */
	mov	ecx, eax
	sub	ecx, ebx	/* CX := (number of bytes in line) - 1 */

	/* get Graphics Controller port address into DX */

	mov	ebx, edx	/* BH := bit mask for first byte */
				/* BL := bit mask for last byte */
	mov	dx, #PORTADDR	/* DX := Graphics Controller port */
	movb	al, #BITMASKREG	/* AL := Bit mask Register number */

	/* make video buffer addressable through DS:SI */

	push	es
	pop	ds
	mov	esi, edi	/* DS:SI -> video buffer */

	/* set pixels in leftmost byte of the line */

	orb	bh, bh
	js	L43		/* jump if byte-aligned (x1 is leftmost) */

	or	ecx, ecx
	jnz	L42		/* jump if more than one byte in the line */

	andb	bl, bh		/* BL := bit mask for the line */
	jmp	L44

L42:	movb	ah, bh		/* AH := bit mask for first byte */
	out	dx, ax		/* update graphics controller */

	movsb			/* update bit planes */
	dec	ecx

	/* use a fast 8086 machine instruction to draw the remainder of the line */

L43:	movb	ah, #0xff	/* AH := bit mask */
	out	dx, ax		/* update Bit Mask register */
	rep
	movsb			/* update all pixels in the line */

	/* set pixels in the rightmost byte of the line */

L44:	movb	ah, bl		/* AH := bit mask for last byte */
	out	dx, ax		/* update Graphics Controller */
	movsb			/* update bit planes */

	pop	ds		/* restore ds */
	jmp	Lexit


/*
 * Routine for dy >= dx (slope <= 1)
 * ES:DI -> video buffer
 * AL = Bit Mask Register number
 * BL = bit mask for first pixel
 * CX = number of pixels to draw
 * DX = Graphics Controller port address
 * SI = decision variable
 */
LowSlope:

L10:	movb	ah, bl		/* AH := bit mask for next pixel */

L11:	orb	ah, bl		/* mask current bit position */
	rorb	bl, #1		/* rotate pixel value */
	jc	L14		/* jump if bit mask rotated to leftmost position */

	/* bit mask not shifted out */

	or	esi, esi	/* test sign of d */
	jns	L12		/* jump if d >= 0 */

	add	esi, incr1(ebp)	/* d := d + incr1 */
	loop	L11

	out	dx, ax		/* update Bit Mask register */
	seg	es
	orb	(edi), al	/* set remaining pixel(s) */
	jmp	Lexit

L12:	add	esi, incr2(ebp)	/* d := d + incr2 */
	out	dx, ax		/* update Bit Mask register */

	seg	es
	orb	(edi), al	/* update bit planes */

	add	edi, rowincr(ebp)	/* increment y */
	loop	L10
	jmp	Lexit

	/* bit mask shifted out */

L14:	out	dx, ax		/* update Bit Mask register */

	seg	es
	orb	(edi), al	/* update bit planes */
	inc	edi		/* increment x */

	or	esi, esi	/* test sign of d */
	jns	L15		/* jump if non-negative */

	add	esi, incr1(ebp)	/* d := d + incr1 */
	loop	L10
	jmp	Lexit

L15:	add	esi, incr2(ebp)	/* d := d + incr2 */
	add	edi, rowincr(ebp)	/* vertical increment */
	loop	L10
	jmp	Lexit


/*
 * Routine for dy > dx (slope > 1)
 * ES:DI -> video buffer
 * AH = bit mask for first pixel
 * AL = Bit Mask register number
 * CX = number pixels to draw
 * DX = Graphics Controller port address
 * SI = decision variable
 */
HighSlope:
	mov	ebx, rowincr(ebp)	/* BX := y increment */

L21:	out	dx, ax		/* update Bit Mask register */
L21a:	seg	es
	orb	(edi), al	/* update bit planes */

	add	edi, ebx	/* increment y */

L22:	or	esi, esi	/* test sign of d */
	jns	L23		/* jump if d >= 0 */

	add	esi, incr1(ebp)	/* d := d + incr1 */
	loop	L21a
	jmp	Lexit

L23:	add	esi, incr2(ebp)	/* d := d + incr2 */
	rorb	ah, #1		/* rotate bit mask */
	adc	edi, #0		/* increment DI if when mask rotated to */
				/* leftmost pixel position */
	loop	L21
/*	jmp	Lexit */


	/* restore default Graphics Controller state and return to caller */

Lexit:	xor	ax, ax		/* AH := 0, AL := 0 */
	out	dx, ax		/* restore Set/Reset register */

	inc	ax		/* AH := 0, AL := 1 */
	out	dx, ax		/* restore Enable Set/Reset register */

	mov	ax, #INITBITMASK	/* AH := 0xff, AL := 0 */
	out	dx, ax		/* restore Bit Mask register */

	pop	es
	pop	edi
	pop	esi
	mov	esp, ebp	/* restore registers and return */
	pop	ebp
	ret

#undef	color


/*
 * Routine to set an individual pixel value.
 * Called from C like:
 *	_ega_drawpoint(x, y, color);
 */

/* argument offsets, starting with 8 bytes (eip + ebp) */
#define	x	8		/* X coordinate */
#define	y	12		/* Y coordinate */
#define	color	16		/* pixel value */


_ega_drawpoint:
	push	ebp
	mov	ebp, esp
	push	ds		/* save registers and set up stack frame */

	mov	ecx, x(ebp)	/* ECX := x */
	cmp	ecx, #PIXCOLS	/* if ((x<0) || (x>=PIXCOLS)) return */
	jae	done

	mov	eax, y(ebp)	/* EAX := y */
	cmp	eax, #PIXROWS	/* if ((y<0) || (y>=PIXROWS)) return */
	jae	done

	mov	edx, #PIXBYTES	/* AX := (y * PIXBYTES) */
	mul	edx

	mov	ebx, ecx	/* BX := (x / 8) */
	shr	ebx, #3

	add	ebx, eax	/* BX := (y * PIXBYTES) + (x / 8) */
	add	ebx,#MEMADDR	/* add in memory physical address */

	andb	cl, #0x07	/* CL := (x % 8) */
	xorb	cl, #0x07	/* CL := 7 - (x % 8) */
	movb	ch, #0x01	/* CH := 1 << (7 - (x % 8))	(mask) */
	shlb	ch, cl

	mov	ax, #FLAT_DS_SELECTOR	/* DS := EGA buffer segment address */
	mov	ds, ax

	mov	dx, #PORTADDR	/* graphics controller port address */
	mov	ax, #WRITEMODE2	/* select write mode 2 */
	out	dx, ax		/* (load value 2 into mode register 5) */
	movb	al, #BITMASKREG	/* set the bit mask register */
	movb	ah, ch		/* (load bit mask into register 8) */
	out	dx, ax
	movb	al, (ebx)	/* dummy read to latch bit planes */
	mov	ax, color(ebp)	/* pixel value */
	movb	(ebx), al	/* write pixel back to bit planes */

	mov	ax, #INITWRITEMODE	/* restore default write mode 0 */
	out	dx, ax		/* (load value 0 into mode register 5) */
	mov	ax, #INITBITMASK	/* restore default bit mask */
	out	dx, ax		/* (load value ff into register 8) */

done:	pop	ds		/* restore registers and return */
	pop	ebp
	ret


/*
 * Routine to draw a text string using the ROM character bitmap.
 * If the background color is (long) -1 then it is not drawn.
 * Called from C like:
 *	ega_drawtext(x, y, cp, len, fg, bg);
 */
#undef	x
#undef	y
#define	x	8			/* X coordinate */
#define	y	12			/* Y coordinate */
#define	cp	16			/* character pointer */
#define	len	20			/* length of string */
#define	fg	24			/* foreground color */
#define	bg	28			/* background color */

_ega_drawtext:
	push	ebp
	mov	ebp, esp
	push	ds		/* save registers we use */
	push	es
	push	edi
	push	esi

	mov	eax, y(ebp)	/* EAX := y */
	cmp	eax, #PIXROWS	/* if ((y<0) || (y>=PIXROWS)) return */
	jae	donetext
	add	eax, #CHARROWS-1	/* convert y to top of chars */
	jl	donetext	/* if not enough room, return */

	mov	ebx, x(ebp)	/* EBX := x */
	cmp	ebx, #PIXCOLS	/* if ((x<0) || (x>=PIXCOLS)) return */
	jae	donetext

	mov	ecx, len(ebp)	/* see if length is non-positive */
	or	ecx, ecx
	jle	donetext

	cmp	ecx, #PIXCOLS	/* or very large */
	jae	donetext

	shl	ecx, #3		/* convert char length to pixel column count */
	add	ecx, ebx	/* determine first column to not touch */
	cmp	ecx, #PIXCOLS	/* see if will write off end */
	jg	donetext

charloop:
	mov	ebx, cp(ebp)	/* get address of next character */
	xor	eax, eax
	movb	al, (ebx)	/* get next character */
	inc	ebx		/* increment character pointer */
	mov	cp(ebp), ebx	/* save away */
	mov	edx,#CHARROWS	/* bytes between character bitmaps */
	mul	edx		/* convert char code to offset in ROM table */
	add	eax, _rom_char_addr	/* make physical address of char bits */
	mov	esi, eax	/* save in source register */

	mov	eax, x(ebp)
	mov	edx, #PIXBYTES	/* AX := (y * PIXBYTES) */
	mul	edx

	movb	cl, bl		/* save low order column bits */
	shr	ebx, #3		/* BX := (x / 8) */

	add	ebx, eax	/* BX := (y * PIXBYTES) + (x / 8) */
	add	ebx,#MEMADDR	/* add in memory physical address */

	andb	cl, #0x07	/* CL := (x % 8) */
	xorb	cl, #0x07	/* CL := 7 - (x % 8) */

	mov	dx, #FLAT_DS_SELECTOR	/* ES := EGA buffer segment address */
	mov	ds, dx

	movb	ch, #0x01	/* CH := 1 << (7 - (col % 8))  (mask) */
	shlb	ch, cl		/* CH := bit mask in proper position */

donetext:
	pop	esi
	pop	edi
	pop	es
	pop	ds
	ret


/*
 * Routine to read the value of an individual pixel.
 * Called from C like:
 * 	color = readpoint(x, y);
 */

#undef	x
#undef	y
#define	x	8			/* X coordinate */
#define	y	12			/* Y coordinate */


_ega_readpoint:
	push	ebp
	mov	ebp, esp

	/*
	 * Make sure that the line is totally within the screen area.
	 * If not, then return a pixel value of 0.
	 */
	mov	eax, y(ebp)	/* EAX := y */
	mov	ebx, x(ebp)	/* EBX := x */
	cmp	eax, #PIXROWS	/* if ((y<0) || (y>=PIXROWS)) return */
	jae	retzero
	cmp	ebx, #PIXCOLS	/* if ((x<0) || (x>=PIXCOLS)) return */
	jae	retzero

	push	si
	push	ds

	mov	edx, #PIXBYTES	/* AX := (y * PIXBYTES) */
	mul	edx

	movb	cl, bl		/* save low order column bits */
	shr	ebx, #3		/* BX := (x / 8) */

	add	ebx, eax	/* BX := (y * PIXBYTES) + (x / 8) */
	add	ebx,#MEMADDR	/* add in memory physical address */

	andb	cl, #0x07	/* CL := (x % 8) */
	xorb	cl, #0x07	/* CL := 7 - (x % 8) */

	mov	dx, #FLAT_DS_SELECTOR	/* ES := EGA buffer segment address */
	mov	ds, dx

	movb	ch, #0x01	/* CH := 1 << (7 - (col % 8))  (mask) */
	shlb	ch, cl		/* CH := bit mask in proper position */

	mov	esi, ebx	/* ES:SI -> region buffer byte */
	xorb	bl, bl		/* BL is used to accumulate the pixel value */

	mov	dx, #PORTADDR	/* DX := Graphics Controller port */
	mov	ax, #0x0304	/* AH := initial bit plane number */
				/* AL := Read Map Select register number */

L112:	out	dx, ax		/* select bit plane */
	movb	bh, (esi)	/* BH := byte from current bit plane */
	andb	bh, ch		/* mask one bit */
	negb	bh		/* bit 7 of BH := 1 if masked bit = 1 */
				/* bit 7 of BH := 0 if masked bit = 0 */
	rol	bx, #1		/* bit 0 of BL := next bit from pixel value */
	decb	ah		/* AH := next bit plane number */
	jge	L112

	xor	eax, eax	/* AL := pixel value */
	movb	al, bl

	pop	ds
	pop	si
	pop	ebp	
	ret

retzero:
	pop	ebp
	xor	eax, eax
	ret


/*
 * Local routine to convert row and column numbers to pixel addresses and masks.
 * Input:	EAX = row
 *		EBX = column
 * Output:
 *		AH := bit mask
 *		ES:EBX -> video buffer
 *		CL := number bits to shift left
 */
PixelAddr:
	push	edx

	mov	edx, #PIXBYTES	/* AX := (row * PIXBYTES) */
	mul	edx

	movb	cl, bl		/* save low order column bits */
	shr	ebx, #3		/* BX := (col / 8) */

	add	ebx, eax	/* BX := (row * PIXBYTES) + (col / 8) */
	add	ebx,#MEMADDR	/* add in memory physical address */

	andb	cl, #0x07	/* CL := (col % 8) */
	xorb	cl, #0x07	/* CL := 7 - (col % 8) */
	movb	ah, #0x01	/* AH := 1 << (7 - (col % 8))	(mask) */

	mov	dx, #FLAT_DS_SELECTOR	/* ES := EGA buffer segment address */
	mov	es, dx

	pop	edx

	ret
