/*
 *	I2O block device driver. 
 *
 *	(C) Copyright 1999   Red Hat Software
 *	
 *	Written by Alan Cox, Building Number Three Ltd
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 * 	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *	This is an initial test release. Most of the good code was taken
 *	from the nbd driver by Pavel Machek, who in turn took some of it
 *	from loop.c. Isn't free software great for reusability 8)
 */

#include <linux/major.h>

#include <linux/module.h>

#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/ioctl.h>
#include <linux/i2o.h>
#include <linux/blkdev.h>
#include <linux/malloc.h>
#include <linux/hdreg.h>

#include <asm/uaccess.h>
#include <asm/io.h>

#define MAJOR_NR I2O_MAJOR

#include <linux/blk.h>

#define MAX_I2OB	16


/*
 *	Some of these can be made smaller later
 */

static int i2ob_blksizes[MAX_I2OB<<4];
static int i2ob_hardsizes[MAX_I2OB<<4];
static int i2ob_sizes[MAX_I2OB<<4];
static int i2ob_media_change_flag[MAX_I2OB];

static int i2ob_context;

static int query_done;

struct i2ob_device
{
	struct i2o_controller *controller;
	int tid;
	spinlock_t queue_lock;
	int flags;
	int refcnt;
	struct request *head, *tail;
};

/*
 *	Each I2O disk is one of these.
 */

static struct i2ob_device i2ob_dev[MAX_I2OB<<4];
static int i2ob_devices = 0;
static struct hd_struct i2ob[MAX_I2OB<<4];
static struct gendisk i2ob_gendisk;	/* Declared later */

#define DEBUG( s )
/* #define DEBUG( s ) printk( s ) 
 */

static int i2ob_install_device(struct i2o_controller *, int, int);

/*
 *	Turn a Linux block request into an I2O block read/write.
 */

static void i2ob_send(struct i2ob_device *dev, struct request *req, u32 base)
{
	struct i2o_controller *c;
	int tid;
	u32 *msg;
	u32 m;
	unsigned long t;
	u64 offset;
	
	c=dev->controller;
	tid=dev->tid;
	
	/*
	 *	Message please
	 */

	t = jiffies;
	 
	do
	{
		mb();
		m = *c->post_port;
	}
	while(m==0xFFFFFFFF && (jiffies-t)<HZ);

	if(m==0xFFFFFFFF)	
	{
		printk(KERN_ERR "i2o_block: no messages ??\n");
		req->errors++;
		return;
	}
	
	/*
	 *	Build a message
	 */
	
	msg = bus_to_virt(c->mem_offset + m);
	
	msg[0] = TEN_WORD_MSG_SIZE | SGL_OFFSET_8;
	msg[2] = i2ob_context;
	msg[3] = (u32)req;	/* 64bit issue again here */
	msg[5] = req->current_nr_sectors << 9;
	
	/* This can be optimised later - just want to be sure its right for
	   starters */
	offset = ((u64)(req->sector+base)) << 9;
	msg[6] = offset & 0xFFFFFFFF;
	msg[7] = (offset>>32);
	msg[9] = virt_to_bus(req->buffer);
	
	if(req->cmd == READ)
	{
		msg[1] = I2O_CMD_BLOCK_READ<<24|HOST_TID<<12|tid;
		/* We don't yet do cache/readahead and other magic */
		msg[4] = 1<<16;	
		msg[8] = 0xD0000000|(req->current_nr_sectors << 9);
	}
	else if(req->cmd == WRITE)
	{
		msg[1] = I2O_CMD_BLOCK_WRITE<<24|HOST_TID<<12|tid;
		msg[4] = 1<<16;
		msg[8] = 0xD4000000|(req->current_nr_sectors << 9);
	}

//	printk("Send for %p\n", req);

	i2o_post_message(c,m);
	return;
}

/*
 *	Remove a request from the _locked_ request list. We update both the
 *	list chain and if this is the last item the tail pointer.
 */
 
static void i2ob_unhook_request(struct i2ob_device *dev, struct request *req)
{
	struct request **p = &dev->head;
	struct request *nt = NULL;
	static int crap = 0;
	
	while(*p!=NULL)
	{
		if(*p==req)
		{
			if(dev->tail==req)
				dev->tail = nt;
			*p=req->next;
			return;
		}
		nt=*p;
		p=&(nt->next);
	}
	if(!crap++)
		printk("i2o_block: request queue corrupt!\n");
}

/*
 *	Request completion handler
 */
 
static void i2ob_end_request(struct request *req)
{
	if (end_that_request_first( req, !req->errors, "i2o block" ))
		return;
	end_that_request_last( req );
}


/*
 *	OSM reply handler. This gets all the message replies
 */

static void i2o_block_reply(struct i2o_handler *h, struct i2o_controller *c, struct i2o_message *msg)
{
	struct request *req;
	u8 st;
	u32 *m = (u32 *)msg;
	u8 unit = (m[2]>>8)&0xF0;	/* low 4 bits are partition */
	
	if(m[0] & (1<<13))
	{
		printk("IOP fail.\n");
		printk("From %d To %d Cmd %d.\n",
			(m[1]>>12)&0xFFF,
			m[1]&0xFFF,
			m[1]>>24);
		printk("Failure Code %d.\n", m[4]>>24);
		if(m[4]&(1<<16))
			printk("Format error.\n");
		if(m[4]&(1<<17))
			printk("Path error.\n");
		if(m[4]&(1<<18))
			printk("Path State.\n");
		if(m[4]&(1<<18))
			printk("Congestion.\n");
		
		m=(u32 *)bus_to_virt(m[7]);
		printk("Failing message is %p.\n", m);
		
		/* We need to up the request failure count here and maybe
		   abort it */
		req=(struct request *)m[3];
		/* Now flush the message by making it a NOP */
		m[0]&=0x00FFFFFF;
		m[0]|=(I2O_CMD_UTIL_NOP)<<24;
		i2o_post_message(c,virt_to_bus(m));
		
	}
	else
	{
		if(m[2]&0x80000000)
		{
			query_done=1;
			return;
		}
		/*
		 *	Lets see what is cooking. We stuffed the
		 *	request in the context.
		 */
		 
		req=(struct request *)m[3];
		st=m[4]>>24;
	
		if(st!=0)
		{
			printk(KERN_ERR "i2ob: error %08X\n", m[4]);
			/*
			 *	Now error out the request block
			 */
			req->errors++;	
		}
	}
	/*
	 *	Dequeue the request.
	 */
	spin_lock(&dev->queue_lock);
//	printk("Ack for %p\n", req);
	i2ob_unhook_request(&i2ob_dev[unit], req);
	spin_unlock(&dev->queue_lock);
	i2ob_end_request(req);
}

static struct i2o_handler i2o_block_handler =
{
	i2o_block_reply,
	"I2O Block OSM",
	0
};


/*
 *	Flush all pending requests as errors. Must call with the queue
 *	locked.
 */
 
static void i2ob_clear_queue(struct i2ob_device *dev)
{
	struct request *req;

	while (1) {
		req = dev->tail;
		if (!req)
			return;
		req->errors++;
		i2ob_end_request(req);

		if (dev->tail == dev->head)
			dev->head = NULL;
		dev->tail = dev->tail->next;
	}
}

/*
 *	Write handling needs some work here, as we should issue multiple
 *	parallel writes.
 */

static void do_i2ob_request(void)
{
	struct request *req;
	int unit;
	struct i2ob_device *dev;
	unsigned long flags;

	while (CURRENT) {
		req = CURRENT;
		unit = MINOR(req->rq_dev);
		dev = &i2ob_dev[(unit&0xF0)];
		req->errors = 0;
		CURRENT = CURRENT->next;
		req->next = NULL;

		spin_unlock_irq(&io_request_lock);
		spin_lock_irqsave(&dev->queue_lock, flags);
		if (dev->head == NULL) {
			dev->head = req;
			dev->tail = req;
		} else {
			dev->tail->next = req;
			dev->tail = req;
		}
		i2ob_send(dev, req, i2ob[unit].start_sect);
		spin_unlock_irqrestore(&dev->queue_lock, flags);
		spin_lock_irq(&io_request_lock);
	}
	return;
}

/*
 *	SCSI-CAM for ioctl geometry mapping
 *	Duplicated with SCSI - this should be moved into somewhere common
 *	perhaps genhd ?
 */
 
static void i2o_block_biosparam(
	unsigned long capacity,
	unsigned short *cyls,
	unsigned char *hds,
	unsigned char *secs) 
{ 
	unsigned long heads, sectors, cylinders, temp; 

	cylinders = 1024L;			/* Set number of cylinders to max */ 
	sectors = 62L;      			/* Maximize sectors per track */ 

	temp = cylinders * sectors;		/* Compute divisor for heads */ 
	heads = capacity / temp;		/* Compute value for number of heads */
	if (capacity % temp) {			/* If no remainder, done! */ 
    		heads++;                	/* Else, increment number of heads */ 
    		temp = cylinders * heads;	/* Compute divisor for sectors */ 
    		sectors = capacity / temp;	/* Compute value for sectors per
						       track */ 
	    	if (capacity % temp) {		/* If no remainder, done! */ 
			sectors++;                  /* Else, increment number of sectors */ 
	      		temp = heads * sectors;	/* Compute divisor for cylinders */
	      		cylinders = capacity / temp;/* Compute number of cylinders */ 
		} 
	} 
	/* if something went wrong, then apparently we have to return
	   a geometry with more than 1024 cylinders */
	if (cylinders == 0 || heads > 255 || sectors > 63 || cylinders >1023) 
	{
		unsigned long temp_cyl;
		
		heads = 64;
		sectors = 32;
		temp_cyl = capacity / (heads * sectors);
		if (temp_cyl > 1024) 
		{
			heads = 255;
			sectors = 63;
		}
		cylinders = capacity / (heads * sectors);
	}
	*cyls = (unsigned int) cylinders;	/* Stuff return values */ 
	*secs = (unsigned int) sectors; 
	*hds  = (unsigned int) heads; 
} 

/*
 *	Rescan the partition tables
 */
 
static int i2ob_revalidate(kdev_t dev, int maxusage)
{
	int minor=MINOR(dev);
	int i;
	
	minor&=0xF0;
	
	i2ob_dev[minor].refcnt++;
	if(i2ob_dev[minor].refcnt>2)
	{
		i2ob_dev[minor].refcnt--;
		return -EBUSY;
	}
	
	for( i = 15; i>=0 ; i--)
	{
		int m = minor+i;
		kdev_t d = MKDEV(MAJOR_NR, m);
		struct super_block *sb = get_super(d);
		
		sync_dev(d);
		if(sb)
			invalidate_inodes(sb);
		invalidate_buffers(d);
		i2ob_gendisk.part[m].start_sect = 0;
		i2ob_gendisk.part[m].nr_sects = 0;
	}

	/*
	 *	Do a physical check and then reconfigure
	 */
	 
	i2ob_install_device(i2ob_dev[minor].controller, i2ob_dev[minor].tid,
		minor);
	i2ob_dev[minor].refcnt--;
	return 0;
}

/*
 *	Issue device specific ioctl calls.
 */

static int i2ob_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg)
{
	struct i2ob_device *dev;
	int minor;

	/* Anyone capable of this syscall can do *real bad* things */

	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;
	if (!inode)
		return -EINVAL;
	minor = MINOR(inode->i_rdev);
	if (minor >= MAX_I2OB)
		return -ENODEV;

	dev = &i2ob_dev[minor];
	switch (cmd) {
		case BLKRASET:
			if(!capable(CAP_SYS_ADMIN))  return -EACCES;
			if(arg > 0xff) return -EINVAL;
			read_ahead[MAJOR(inode->i_rdev)] = arg;
			return 0;

		case BLKRAGET:
			if (!arg)  return -EINVAL;
			return put_user(read_ahead[MAJOR(inode->i_rdev)],
					(long *) arg); 
		case BLKGETSIZE:
			return put_user(i2ob[minor].nr_sects, (long *) arg);

		case BLKFLSBUF:
			if(!capable(CAP_SYS_ADMIN))  return -EACCES;
			fsync_dev(inode->i_rdev);
			invalidate_buffers(inode->i_rdev);
			return 0;
			
		case HDIO_GETGEO:
		{
			struct hd_geometry g;
			int u=minor&0xF0;
			i2o_block_biosparam(i2ob_sizes[u]<<1, 
				&g.cylinders, &g.heads, &g.sectors);
			g.start = i2ob[minor].start_sect;
			return copy_to_user((void *)arg,&g, sizeof(g))?-EFAULT:0;
		}
	
		RO_IOCTLS(inode->i_rdev,arg);
	
		case BLKRRPART:
			if(!capable(CAP_SYS_ADMIN))
				return -EACCES;
			return i2ob_revalidate(inode->i_rdev, 1);
			
		default:
			return -EINVAL;
	}
}

/*
 *	Close the block device down
 */
 
static int i2ob_release(struct inode *inode, struct file *file)
{
	struct i2ob_device *dev;
	int minor;

	minor = MINOR(inode->i_rdev);
	if (minor >= MAX_I2OB)
		return -ENODEV;
	sync_dev(inode->i_rdev);
	dev = &i2ob_dev[(minor&0xF0)];
	if (dev->refcnt <= 0)
		printk(KERN_ALERT "i2ob_release: refcount(%d) <= 0\n", dev->refcnt);
	dev->refcnt--;
	MOD_DEC_USE_COUNT;
	return 0;
}

/*
 *	Open the block device.
 */
 
static int i2ob_open(struct inode *inode, struct file *file)
{
	int minor;

	if (!inode)
		return -EINVAL;
	minor = MINOR(inode->i_rdev);
	if (minor >= MAX_I2OB)
		return -ENODEV;

	i2ob_dev[(minor&0xF0)].refcnt++;
	MOD_INC_USE_COUNT;
	return 0;
}

static int i2ob_query_device(struct i2o_controller *c, int tid, 
	int table, int field, void *buf, int buflen)
{
	static u16 opdata[]={1,0,1,0,1,4};
	u32 *bl;
	u32 m;
	u32 *msg;
       	unsigned long t=jiffies;

	bl=kmalloc(buflen+32, GFP_KERNEL);
	if(bl==NULL)
	{
		printk(KERN_ERR "i2o_block: no memory for query buffer.\n");
		return -ENOMEM;
	}
	
	opdata[3]=table;
	opdata[5]=field;
	memcpy(bl, opdata, 12);
	       	
	do
	{
		mb();
		m = *c->post_port;
	}
	while(m==0xFFFFFFFF && (jiffies-t)<HZ);
	
	
	if(m==0xFFFFFFFF)
	{
		printk(KERN_ERR "i2o_block: controller not responding.\n");
		kfree(bl);
		return -1;
	}
	msg = bus_to_virt(c->mem_offset + m);
	
	msg[0]=NINE_WORD_MSG_SIZE|SGL_OFFSET_5;
	msg[1]=I2O_CMD_UTIL_PARAMS_GET<<24|HOST_TID<<12|tid;
	msg[2]=i2ob_context|0x80000000;	/* So we can pick it out */
	msg[3]=0;
	msg[4]=0;
	msg[5]=0x54000000|12;
	msg[6]=virt_to_bus(bl);
	msg[7]=0xD0000000|128;
	msg[8]=virt_to_bus(bl+4);
	
	bl[4]=0xFCFCFCFC;
	bl[5]=0xFAFAFCFC;

	/*
	 *	Post the message and await a reply
	 */
	 	
	query_done=0;
	mb();
		
	i2o_post_message(c,m);
	wmb();
	
	while(!query_done && (jiffies-t)<2*HZ)
	{
		schedule();
		mb();
	}
			
	if(bl[5]&0x80000000)	/* Error ? */
	{
//		printk(KERN_ERR "i2o_block: query failed. 0x%08X\n", bl[5]);
		kfree(bl);
		return -1;
	}
	if((bl[4]&0xFFFF)!=1)
	{
		printk(KERN_ERR "i2o_block: query: umm ??? 0x%08X\n", bl[4]);
	}
	
	memcpy(buf, bl+6, buflen);
	kfree(bl);
	return 0;
}

static int i2ob_install_device(struct i2o_controller *c, int tid, int unit)
{
	u64 size;
	u32 blocksize;
	u8 type;
	u32 flags, status;

	i2ob_dev[unit].controller = c;
	i2ob_dev[unit].tid = tid;
	
	/*
	 *	Ask for the current media data. If that isn't supported
	 *	then we ask for the device capacity data
	 */
	 
	if(i2ob_query_device(c, tid, 0x0004, 1, &blocksize, 4) != 0
	  || i2ob_query_device(c, tid, 0x0004, 0, &size, 8) !=0 )
	{
		i2ob_query_device(c, tid, 0x0000, 3, &blocksize, 4);
		i2ob_query_device(c, tid, 0x0000, 4, &size, 8);
	}
	
	i2ob_query_device(c, tid, 0x0000, 5, &flags, 4);
	i2ob_query_device(c, tid, 0x0000, 5, &status, 5);
	i2ob_sizes[unit] = (int)(size>>10);
	i2ob_hardsizes[unit] = blocksize;
	i2ob_gendisk.part[unit].nr_sects = i2ob_sizes[unit];
	
	i2ob[unit].nr_sects = (int)(size>>9);

	i2ob_query_device(c, tid, 0x0000, 0, &type, 1);
	
	printk("i2ohd%c: ", (unit>>4)+'a');
	if(status&(1<<10))
		printk("RAID ");
	switch(type)
	{
		case 0: printk("Disk Storage");break;
		case 4: printk("WORM");break;
		case 5: printk("CD-ROM");break;
		case 7:	printk("Optical device");break;
		default:
			printk("Type %d", type);
	}
	if(((flags & (1<<3)) && !(status & (1<<3))) ||
	   ((flags & (1<<4)) && !(status & (1<<4))))
	{
		printk(" Not loaded.\n");
		return 0;
	}
	printk(" %dMb, %d byte sectors",
		(int)(size>>20), blocksize);
	if(status&(1<<0))
	{
		u32 cachesize;
		i2ob_query_device(c, tid, 0x0003, 0, &cachesize, 4);
		cachesize>>=10;
		if(cachesize>4095)
			printk(", %dMb cache", cachesize>>10);
		else
			printk(", %dKb cache", cachesize);
	}
	printk(".\n");
	resetup_one_dev(&i2ob_gendisk, unit>>4);
	return 0;
}

static void i2ob_probe(void)
{
	int i;
	int unit = 0;
	int warned = 0;
		
	for(i=0; i< MAX_I2O_CONTROLLERS; i++)
	{
		struct i2o_controller *c=i2o_find_controller(i);
		struct i2o_device *d;
		
		if(c==NULL)
			continue;
		for(d=c->devices;d!=NULL;d=d->next)
		{
			int class = d->type & 0xFFFF;
			
			if(class!=0x1010)
				continue;
			if(unit<MAX_I2OB<<4)
			{
				i2ob_install_device(c,d->id,unit);
				unit+=16;
			}
			else
			{
				if(!warned++)
					printk("i2o_block: too many controllers, registering only %d.\n", unit);
			}
		}
	}
	i2ob_devices = unit;
}

/*
 *	Have we seen a media change ?
 */
 
static int i2ob_media_change(kdev_t dev)
{
	int i=MINOR(dev);
	i>>=4;
	if(i2ob_media_change_flag[i])
	{
		i2ob_media_change_flag[i]=0;
		return 1;
	}
	return 0;
}
 
static struct file_operations i2ob_fops =
{
	NULL,			/* lseek - default */
	block_read,		/* read - general block-dev read */
	block_write,		/* write - general block-dev write */
	NULL,			/* readdir - bad */
	NULL,			/* select */
	i2ob_ioctl,		/* ioctl */
	NULL,			/* mmap */
	i2ob_open,		/* open */
	NULL,			/* flush */
	i2ob_release,		/* release */
	NULL,			/* fsync */
	NULL,			/* fasync */
	i2ob_media_change,	/* Media Change */
	i2ob_revalidate		/* Revalidate */
};

/*
 *	Partitioning
 */
 
static void i2ob_geninit(struct gendisk *gd)
{
}
	
static struct gendisk i2ob_gendisk = 
{
	MAJOR_NR,
	"i2ohd",
	4,
	1<<4,
	MAX_I2OB,
	i2ob_geninit,
	i2ob,
	i2ob_sizes,
	0,
	NULL,
	NULL
};

/*
 * And here should be modules and kernel interface 
 *  (Just smiley confuses emacs :-)
 */

#ifdef MODULE
#define i2ob_init init_module
#endif

int i2ob_init(void)
{
	int i;

	printk("I2O block device OSM v0.06. (C) 1999 Red Hat Software.\n");
	
	/*
	 *	Register the block device interfaces
	 */

	if (register_blkdev(MAJOR_NR, "i2o_block", &i2ob_fops)) {
		printk("Unable to get major number %d for i2o_block\n",
		       MAJOR_NR);
		return -EIO;
	}
#ifdef MODULE
	printk("i2o_block: registered device at major %d\n", MAJOR_NR);
#endif

	/*
	 *	Now fill in the boiler plate
	 */
	 
	blksize_size[MAJOR_NR] = i2ob_blksizes;
	hardsect_size[MAJOR_NR] = i2ob_hardsizes;
	blk_size[MAJOR_NR] = i2ob_sizes;
	blk_dev[MAJOR_NR].request_fn = do_i2ob_request;
	for (i = 0; i < MAX_I2OB << 4; i++) {
		i2ob_dev[i].refcnt = 0;
		i2ob_dev[i].flags = 0;
		i2ob_dev[i].queue_lock = SPIN_LOCK_UNLOCKED;
		i2ob_dev[i].controller = NULL;
		i2ob_dev[i].tid = 0;
		i2ob_dev[i].head = NULL;
		i2ob_dev[i].tail = NULL;
		i2ob_blksizes[i] = 1024;
	}
	
	/*
	 *	Register the OSM handler as we will need this to probe for
	 *	drives, geometry and other goodies.
	 */

	if(i2o_install_handler(&i2o_block_handler)<0)
	{
		unregister_blkdev(MAJOR_NR, "i2o_block");
		printk(KERN_ERR "i2o_block: unable to register OSM.\n");
		return -EINVAL;
	}
	i2ob_context = i2o_block_handler.context;	 

	/*
	 *	Finally see what is actually plugged in to our controllers
	 */

	i2ob_probe();

	return 0;
}

#ifdef MODULE

EXPORT_NO_SYMBOLS;
MODULE_AUTHOR("Red Hat Software");
MODULE_DESCRIPTION("I2O Block Device OSM");

void cleanup_module(void)
{
	struct gendisk **gdp;
	
	/*
	 *	Flush the OSM
	 */

	i2o_remove_handler(&i2o_block_handler);
		 
	/*
	 *	Return the block device
	 */
	if (unregister_blkdev(MAJOR_NR, "i2o_block") != 0)
		printk("i2o_block: cleanup_module failed\n");
	else
		printk("i2o_block: module cleaned up.\n");
	
	/*
	 *	Why isnt register/unregister gendisk in the kernel ???
	 */

	for (gdp = &gendisk_head; *gdp; gdp = &((*gdp)->next))
		if (*gdp == &i2ob_gendisk)
			break;


}
#endif
