/*
 *	Tunable parameters first
 */

/* How many different OSM's are we allowing */ 
#define MAX_I2O_MODULES		64
/* How many controllers are we allowing */
#define MAX_I2O_CONTROLLERS	32


/*
 *	I2O Interface Objects
 */

#include <linux/notifier.h>
#include <asm/atomic.h>

/*
 * message structures
 */

#define    TID_SZ                                  12
#define    FUNCTION_SZ                             8


struct i2o_message
{
	u32	version_size;
	u32	function_addr;
	u32	initiator_context;
	/* List follows */
};


/*
 *	Each I2O device entity has one or more of these. There is one
 *	per device. *FIXME* how to handle multiple types on one unit.
 */
 
struct i2o_device
{
	int type;		/* Block, Net, SCSI etc (from spec) */
	int id;			/* I2O ID assigned by the controller */
	int parent;		/* Parent device */
	int flags;		/* Control flags */
	struct i2o_driver *owner;		/* Owning device */
	struct i2o_controller *controller;	/* Controlling IOP */
	struct i2o_device *next;	/* Chain */
};

/*
 *	Resource data for each PCI I2O controller
 */	 	

struct i2o_pci
{
	int irq;
};

/*
 *	Each I2O controller has one of these objects
 */
 
struct i2o_controller
{
	char name[8];
	int unit;
	int status;				/* I2O status */
	int type;
#define I2O_TYPE_PCI		0x01		/* PCI I2O controller */	
	struct notifier_block *event_notifer;	/* Events */
	atomic_t users;
	struct i2o_device *devices;		/* I2O device chain */
	struct i2o_controller *next;		/* Controller chain */
	volatile u32 *post_port;		/* Messaging ports */
	volatile u32 *read_port;
	volatile u32 *irq_mask;			/* Interrupt port */
	u32 mem_offset;				/* MFA offset */
	u32 mem_phys;				/* MFA physical */
	union
	{					/* Bus information */
		struct i2o_pci pci;
	} bus;
	void (*destructor)(struct i2o_controller *);			/* Bus specific destructor */
	int (*bind)(struct i2o_controller *, struct i2o_device *);	/* Bus specific attach/detach */
	int (*unbind)(struct i2o_controller *, struct i2o_device *);
	void *page_frame;		/* Message buffers */
	int inbound_size;		/* Inbound queue size */
};

struct i2o_handler
{
	void (*reply)(struct i2o_handler *, struct i2o_controller *, struct i2o_message *);
	char *name;
	int context;		/* Low 16 bits of the transaction info */
	/* User data follows */
};

/*
 *	Messenger inlines
 */
 
extern inline void i2o_post_message(struct i2o_controller *c, u32 m)
{
	/* The second line isnt spurious - thats forcing PCI posting */
	*c->post_port=m;
	*c->irq_mask;
}

extern inline void i2o_flush_reply(struct i2o_controller *c, u32 m)
{
	/* The second line isnt spurious - thats forcing PCI posting */
	*c->read_port=m;
	*c->irq_mask;
}

struct i2o_controller *i2o_controller_chain;

extern int i2o_install_controller(struct i2o_controller *);
extern int i2o_delete_controller(struct i2o_controller *);
extern int i2o_activate_controller(struct i2o_controller *);
extern void i2o_unlock_controller(struct i2o_controller *);
extern struct i2o_controller *i2o_find_controller(int);


extern int i2o_install_handler(struct i2o_handler *);
extern int i2o_remove_handler(struct i2o_handler *);

extern int i2o_install_device(struct i2o_controller *, struct i2o_device *);
extern int i2o_delete_device(struct i2o_device *);
extern int i2o_claim_device(struct i2o_device *, struct i2o_driver *);
extern int i2o_release_device(struct i2o_device *);

/* RedCreek's OSM default LAN receive Initiator */
#define DEFAULT_RECV_INIT_CONTEXT  0xA17  

/* Transaction Reply Lists (TRL) Control Word structure */

#define    TRL_SINGLE_FIXED_LENGTH		0x00
#define    TRL_SINGLE_VARIABLE_LENGTH		0x40
#define    TRL_MULTIPLE_FIXED_LENGTH		0x80

/* LAN Class specific functions */

#define    LAN_PACKET_SEND			0x3B
#define    LAN_SDU_SEND				0x3D
#define    LAN_RECEIVE_POST			0x3E
#define    LAN_RESET				0x35
#define    LAN_SHUTDOWN				0x37

/*
 *	Messaging API values
 */
 
#define	I2O_CMD_ADAPTER_ASSIGN		0xB3
#define	I2O_CMD_ADAPTER_READ		0xB2
#define	I2O_CMD_ADAPTER_RELEASE		0xB5
#define	I2O_CMD_BIOS_INFO_SET		0xA5
#define	I2O_CMD_BOOT_DEVICE_SET		0xA7
#define	I2O_CMD_CONFIG_VALIDATE		0xBB
#define	I2O_CMD_CONN_SETUP		0xCA
#define	I2O_CMD_DEVICE_ASSIGN		0xB7
#define	I2O_CMD_DEVICE_RELEASE		0xB9
#define	I2O_CMD_HRT_GET			0xA8
#define	I2O_CMD_ADAPTER_CLEAR		0xBE
#define	I2O_CMD_ADAPTER_CONNECT		0xC9
#define	I2O_CMD_ADAPTER_RESET		0xBD
#define	I2O_CMD_LCT_NOTIFY		0xA2
#define	I2O_CMD_OUTBOUND_INIT		0xA1
#define	I2O_CMD_PATH_ENABLE		0xD3
#define	I2O_CMD_PATH_QUIESCE		0xC5
#define	I2O_CMD_PATH_RESET		0xD7
#define	I2O_CMD_STATIC_MF_CREATE	0xDD
#define	I2O_CMD_STATIC_MF_RELEASE	0xDF
#define	I2O_CMD_STATUS_GET		0xA0
#define	I2O_CMD_SW_DOWNLOAD		0xA9
#define	I2O_CMD_SW_UPLOAD		0xAB
#define	I2O_CMD_SW_REMOVE		0xAD
#define	I2O_CMD_SYS_ENABLE		0xD1
#define	I2O_CMD_SYS_MODIFY		0xC1
#define	I2O_CMD_SYS_QUIESCE		0xC3
#define	I2O_CMD_SYS_TAB_SET		0xA3

#define I2O_CMD_UTIL_NOP		0x00
#define I2O_CMD_UTIL_PARAMS_GET		0x06
#define I2O_CMD_UTIL_CONFIG_DIALOG	0x10

#define I2O_CMD_SCSI_EXEC		0x81
#define I2O_CMD_SCSI_ABORT		0x83
#define I2O_CMD_SCSI_BUSRESET		0x27

#define I2O_CMD_BLOCK_READ		0x30
#define I2O_CMD_BLOCK_WRITE		0x31

/*
 *	Init Outbound Q status 
 */
 
#define I2O_CMD_OUTBOUND_INIT_IN_PROGRESS	0x01
#define I2O_CMD_OUTBOUND_INIT_REJECTED		0x02
#define I2O_CMD_OUTBOUND_INIT_FAILED		0x03
#define I2O_CMD_OUTBOUND_INIT_COMPLETE		0x04

/*
 *	I2O Get Status State values 
 */

#define	ADAPTER_STATE_INITIALIZING	0x01
#define	ADAPTER_STATE_RESET		0x02
#define	ADAPTER_STATE_HOLD		0x04
#define ADAPTER_STATE_READY		0x05
#define	ADAPTER_STATE_OPERATIONAL	0x08
#define	ADAPTER_STATE_FAILED		0x10
#define	ADAPTER_STATE_FAULTED		0x11
	
/* I2O API function return values */

#define I2O_RTN_NO_ERROR             0
#define I2O_RTN_NOT_INIT             1
#define I2O_RTN_FREE_Q_EMPTY         2
#define I2O_RTN_TCB_ERROR            3
#define I2O_RTN_TRANSACTION_ERROR    4
#define I2O_RTN_ADAPTER_ALREADY_INIT 5
#define I2O_RTN_MALLOC_ERROR         6
#define I2O_RTN_ADPTR_NOT_REGISTERED 7
#define I2O_RTN_MSG_REPLY_TIMEOUT    8
#define I2O_RTN_NO_STATUS            9
#define I2O_RTN_NO_FIRM_VER         10
#define	I2O_RTN_NO_LINK_SPEED       11

/* Reply message status defines */

#define	I2O_REPLY_STATUS_ABORT_PARTIAL_TRANSFER		0x03
#define	I2O_REPLY_STATUS_ERROR_DIRTY			0x04
#define	I2O_REPLY_STATUS_ERROR_NO_DATA_TRANSFER		0x05
#define	I2O_REPLY_STATUS_ERROR_PARTIAL_TRANSFER		0x06
#define	I2O_REPLY_STATUS_PROCESS_ABORT_DIRTY		0x07
#define	I2O_REPLY_STATUS_PROCESS_ABORT_NO_DATA_TRANSFER	0x08
#define	I2O_REPLY_STATUS_PROCESS_ABORT_PARTIAL_TRANSFER	0x09
#define	I2O_REPLY_STATUS_TRANSACTION_ERROR		0x0A
#define	I2O_REPLY_STATUS_PROGRESS_REPORT		0x80

/* DetailedStatusCode defines for ALL messages: Table 3-2 Detailed Status Codes.*/

#define	I2O_DS_SUCCESS				0x0000
#define	I2O_DS_BAD_KEY				0x0001
#define	I2O_DS_CHAIN_BUFFER_TOO_LARGE		0x0002
#define	I2O_DS_DEVICE_BUSY			0x0003
#define	I2O_DS_DEVICE_LOCKED			0x0004
#define	I2O_DS_DEVICE_NOT_AVAILABLE		0x0005
#define	I2O_DS_DEVICE_RESET			0x0006
#define	I2O_DS_INAPPROPRIATE_FUNCTION		0x0007
#define	I2O_DS_INSUFFICIENT_RESOURCE_HARD	0x0008
#define	I2O_DS_INSUFFICIENT_RESOURCE_SOFT	0x0009
#define	I2O_DS_INVALID_INITIATOR_ADDRESS	0x000A
#define	I2O_DS_INVALID_MESSAGE_FLAGS		0x000B
#define	I2O_DS_INVALID_OFFSET			0x000C
#define	I2O_DS_INVALID_PARAMETER		0x000D
#define	I2O_DS_INVALID_REQUEST			0x000E
#define	I2O_DS_INVALID_TARGET_ADDRESS		0x000F
#define	I2O_DS_MESSAGE_TOO_LARGE		0x0010
#define	I2O_DS_MESSAGE_TOO_SMALL		0x0011
#define	I2O_DS_MISSING_PARAMETER		0x0012
#define	I2O_DS_NO_SUCH_PAGE			0x0013
#define	I2O_DS_REPLY_BUFFER_FULL		0x0014
#define	I2O_DS_TCL_ERROR			0x0015
#define	I2O_DS_TIMEOUT				0x0016
#define	I2O_DS_UNKNOWN_ERROR			0x0017
#define	I2O_DS_UNKNOWN_FUNCTION			0x0018
#define	I2O_DS_UNSUPPORTED_FUNCTION		0x0019
#define	I2O_DS_UNSUPPORTED_VERSION		0x001A

/* DetailedStatusCode defines */

#define	I2O_DSC_SUCCESS				0x0000
#define	I2O_DSC_DEVICE_FAILURE			0x0001
#define	I2O_DSC_DESTINATION_NOT_FOUND		0x0002
#define	I2O_DSC_TRANSMIT_ERROR			0x0003
#define	I2O_DSC_TRANSMIT_ABORTED		0x0004
#define	I2O_DSC_RECEIVE_ERROR			0x0005
#define	I2O_DSC_RECEIVE_ABORTED			0x0006
#define	I2O_DSC_DMA_ERROR			0x0007
#define	I2O_DSC_BAD_PACKET_DETECTED		0x0008
#define	I2O_DSC_OUT_OF_MEMORY			0x0009
#define	I2O_DSC_BUCKET_OVERRUN			0x000A
#define	I2O_DSC_IOP_INTERNAL_ERROR		0x000B
#define	I2O_DSC_CANCELED			0x000C
#define	I2O_DSC_INVALID_TRANSACTION_CONTEXT	0x000D
#define	I2O_DSC_DESTINATION_ADDRESS_DETECTED	0x000E
#define	I2O_DSC_DESTINATION_ADDRESS_OMITTED	0x000F
#define	I2O_DSC_PARTIAL_PACKET_RETURNED		0x0010

/* Message header defines for VersionOffset */
#define RCMSGVER_1	0x0001
#define SGL_OFFSET_0    RCMSGVER_1
#define SGL_OFFSET_4    (0x0040 | RCMSGVER_1)
#define SGL_OFFSET_5    (0x0050 | RCMSGVER_1)
#define SGL_OFFSET_6    (0x0060 | RCMSGVER_1)
#define SGL_OFFSET_8    (0x0080 | RCMSGVER_1)
#define SGL_OFFSET_10   (0x00A0 | RCMSGVER_1)

#define TRL_OFFSET_5    (0x0050 | RCMSGVER_1)
#define TRL_OFFSET_6    (0x0060 | RCMSGVER_1)

 /* msg header defines for MsgFlags */
#define MSG_STATIC	0x0100
#define MSG_64BIT_CNTXT	0x0200
#define MSG_MULTI_TRANS	0x1000
#define MSG_FAIL	0x2000
#define MSG_LAST	0x4000
#define MSG_REPLY	0x8000

  /* normal LAN request message MsgFlags and VersionOffset (0x1041) */
#define LAN_MSG_REQST	(MSG_MULTI_TRANS | SGL_OFFSET_4)

 /* minimum size msg */
#define THREE_WORD_MSG_SIZE	0x00030000
#define FOUR_WORD_MSG_SIZE	0x00040000
#define FIVE_WORD_MSG_SIZE	0x00050000
#define SIX_WORD_MSG_SIZE	0x00060000
#define SEVEN_WORD_MSG_SIZE	0x00070000
#define EIGHT_WORD_MSG_SIZE	0x00080000
#define NINE_WORD_MSG_SIZE	0x00090000
#define TEN_WORD_MSG_SIZE	0x000A0000
#define I2O_MESSAGE_SIZE(x)	((x)<<16)


/* Special TID Assignments */

#define ADAPTER_TID		0
#define HOST_TID		1

#define MSG_FRAME_SIZE	512
#define NMBR_MSG_FRAMES	30

#define MSG_POOL_SIZE	16384
