/* 
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  Complications for I2O scsi
 *
 *	o	Each (bus,lun) is a logical device in I2O. We keep a map
 *		table. We spoof failed selection for unmapped units
 *	o	Request sense buffers can come back for free. 
 *	o	Scatter gather is a bit dynamic. We have to investigate at
 *		setup time.
 *	o	Some of our resources are dynamically shared. The i2o core
 *		needs a message reservation protocol to avoid swap v net
 *		deadlocking. We need to back off queue requests.
 *	
 *	In general the firmware wants to help. Where its help isn't performance
 *	useful we just ignore the aid. Its not worth the code in truth.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <asm/dma.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/blk.h>
#include <linux/version.h>
#include <linux/i2o.h>
#include "../scsi/scsi.h"
#include "../scsi/hosts.h"
#include "../scsi/sd.h"
#include "i2o_scsi.h"

#define VERSION_STRING        "Version 0.0.1"

#define dprintk(x)

#define MAXHOSTS 32

struct proc_dir_entry proc_scsi_i2o_scsi = {
	PROC_SCSI_I2O, 8, "i2o_scsi", S_IFDIR | S_IRUGO | S_IXUGO, 2
};

struct i2o_scsi_host
{
	struct i2o_controller *controller;
	s16 task[16];		/* Allow 16 devices for now */
	unsigned long tagclock[16];	/* Tag clock for queueing */
	s16 bus_task;		/* The adapter TID */
};

static int scsi_context;
static int lun_done;
static int i2o_scsi_hosts;

static u32 *retry[32];
static struct i2o_controller *retry_ctrl[32];
static struct timer_list retry_timer;
static int retry_ct = 0;


/*
 *	Retry congested frames. This actually needs pushing down into
 *	i2o core. We should only bother the OSM with this when we can't
 *	queue and retry the frame. Or perhaps we should call the OSM
 *	and its default handler should be this in the core, and this
 *	call a 2nd "I give up" handler in the OSM ?
 */
 
static void i2o_retry_run(unsigned long f)
{
	int i;
	unsigned long flags;
	
	save_flags(flags);
	cli();

	for(i=0;i<retry_ct;i++)
		i2o_post_message(retry_ctrl[i], virt_to_bus(retry[i]));
	retry_ct=0;
	
	restore_flags(flags);
}

static void flush_pending(void)
{
	int i;
	unsigned long flags;
	
	save_flags(flags);
	cli();

	for(i=0;i<retry_ct;i++)
	{
		retry[i][0]&=~0xFFFFFF;
		retry[i][0]|=I2O_CMD_UTIL_NOP<<24;
		i2o_post_message(retry_ctrl[i],virt_to_bus(retry[i]));
	}
	retry_ct=0;
	
	restore_flags(flags);
}

static void i2o_scsi_reply(struct i2o_handler *h, struct i2o_controller *c, struct i2o_message *msg)
{
	Scsi_Cmnd *current_command;
	u32 *m = (u32 *)msg;
	u8 as,ds,st;
	
	if(m[0] & (1<<13))
	{
		printk("IOP fail.\n");
		printk("From %d To %d Cmd %d.\n",
			(m[1]>>12)&0xFFF,
			m[1]&0xFFF,
			m[1]>>24);
		printk("Failure Code %d.\n", m[4]>>24);
		if(m[4]&(1<<16))
			printk("Format error.\n");
		if(m[4]&(1<<17))
			printk("Path error.\n");
		if(m[4]&(1<<18))
			printk("Path State.\n");
		if(m[4]&(1<<18))
			printk("Congestion.\n");
		
		m=(u32 *)bus_to_virt(m[7]);
		printk("Failing message is %p.\n", m);
		
		if((m[4]&(1<<18)) && retry_ct < 32)
		{
			retry_ctrl[retry_ct]=c;
			retry[retry_ct]=m;
			if(!retry_ct++)
			{
				retry_timer.expires=jiffies+1;
				add_timer(&retry_timer);
			}
		}
		else
		{
			/* Create a scsi error for this */
			current_command = (Scsi_Cmnd *)m[3];
			printk("Aborted %ld\n", current_command->serial_number);
			
			current_command->result = DID_ERROR << 16;
			current_command->scsi_done(current_command);
			
			/* Now flush the message by making it a NOP */
			m[0]&=0x00FFFFFF;
			m[0]|=(I2O_CMD_UTIL_NOP)<<24;
			i2o_post_message(c,virt_to_bus(m));
		}
		return;
	}
			
	
	/* Low byte is the adapter status, next is the device */
	as=(u8)m[4]; 
	ds=(u8)(m[4]>>8);
	st=(u8)(m[4]>>24);
	
	dprintk(("i2o got a scsi reply %08X: ", m[0]));
	dprintk(("m[2]=%08X: ", m[2]));
	dprintk(("m[4]=%08X\n", m[4]));

	if(m[2]&0x80000000)
	{
		if(m[2]&0x40000000)
		{
			dprintk(("Lun event.\n"));
			lun_done=1;
			return;
		}
		printk(KERN_ERR "i2o_scsi: bus reset reply.\n");
		return;
	}
	
	current_command = (Scsi_Cmnd *)m[3];
	
	/*
	 *	Is this a control request coming back - eg an abort ?
	 */
	 
	if(current_command==NULL)
	{
		if(st)
			printk("SCSI abort: %08X", m[4]);
		printk("SCSI abort completed.\n");
		return;
	}
	
	dprintk(("Completed %ld\n", current_command->serial_number));
	if(st && st!=0x06)
	{
		/* An error has occured */

		printk(KERN_DEBUG "SCSI error %08X", m[4]);
			
		if (ds == 0x0E) 
		{	
			/* SCSI Reset */
			current_command->result = DID_RESET << 16;
			current_command->scsi_done(current_command);
			return;
		}
		if (ds == 0x0F)
		{
			current_command->result = DID_PARITY << 16;
			current_command->scsi_done(current_command);
			return;
		}
		
		current_command->result = DID_ERROR << 16;
		current_command->scsi_done(current_command);
		return;
	}
	/*
	 *	It worked maybe ?
	 */		
	current_command->result = DID_OK << 16 | ds;
	current_command->scsi_done(current_command);
}

struct i2o_handler i2o_scsi_handler=
{
	i2o_scsi_reply,
	"I2O SCSI OSM",
	0
};

static int i2o_find_lun(struct i2o_controller *c, int tid, int *lun)
{
	static u16 opdata[]={1,0,1,0,1,4};
	u32 *bl;
	u8  *buf;
	u32 m;
	u32 *msg;
       	unsigned long t=jiffies;
       	unsigned long flags;

	bl=kmalloc(48, GFP_KERNEL);
	if(bl==NULL)
	{
		printk(KERN_ERR "i2o_scsi: no memory for LUN query buffer.\n");
		return -ENOMEM;
	}
	memcpy(bl+32, opdata, 12);
	buf=(u8 *)bl;
	       	
       	save_flags(flags);
       	sti();
	
//	printk("Get msg\n");
	do
	{
		mb();
		m = *c->post_port;
	}
	while(m==0xFFFFFFFF && (jiffies-t)<HZ);
	
//	printk("Got msg %X\n", m);
	
	if(m==0xFFFFFFFF)
	{
		restore_flags(flags);
		printk(KERN_ERR "i2o_scsi: controller not responding.\n");
		kfree(bl);
		return -1;
	}
	msg = bus_to_virt(c->mem_offset + m);
	
	msg[0]=NINE_WORD_MSG_SIZE|SGL_OFFSET_5;
	msg[1]=I2O_CMD_UTIL_PARAMS_GET<<24|HOST_TID<<12|tid;
	msg[2]=scsi_context|0xC0000000;	/* So we can pick it out */
	msg[3]=c->unit<<16|tid;
	msg[4]=0;
	msg[5]=0x54000000|12;
	msg[6]=virt_to_bus(bl+32);
	msg[7]=0xD0000000|128;
	msg[8]=virt_to_bus(bl);
	
	bl[0]=0xFCFCFCFC;
	bl[1]=0xFAFAFCFC;

	/*
	 *	Post the message and await a reply
	 */
	 	
//	printk("Post\n");
	lun_done=0;
	mb();
		
	i2o_post_message(c,m);
	wmb();
	
//	printk("Sleep\n");
	while(!lun_done && (jiffies-t)<2*HZ)
	{
		schedule();
		mb();
	}
	restore_flags(flags);
			
//	printk("Check\n");
	if(bl[1]&0x80000000)	/* Error ? */
	{
		printk(KERN_ERR "i2o_scsi: lun read failed. 0x%08X\n", bl[1]);
		kfree(bl);
		return -1;
	}
	if((bl[0]&0xFFFF)!=1)
	{
		printk(KERN_ERR "i2o_scsi: lun: umm ??? 0x%08X\n", bl[0]);
	}
	
	*lun=buf[9];
//	printk("Done\n");
	kfree(bl);
	return 0;
}

static void i2o_scsi_init(struct i2o_controller *c, struct i2o_device *d, struct i2o_scsi_host *h)
{
	struct i2o_device *unit;
	int lun;
	
	h->controller=c;
	h->bus_task=d->id;
	
	for(lun=0;lun<16;lun++)
		h->task[lun] = -1;
			
	for(unit=c->devices;unit!=NULL;unit=unit->next)
	{
		int class = unit->type&0xFFFF;
		
//		printk("Class %04X, parent %d, want %d.\n",
//			class, unit->parent, d->id);
			
		/* Only look at scsi and fc devices */
		if(class != 0x1041 && class != 0x1051)
			continue;
		/* On our bus ? */
//		printk("Found a disk.\n");
		if(unit->parent==d->id || 1 || unit->parent == d->parent)
		{
//			printk("Its ours.\n");
			if(i2o_find_lun(c, unit->id, &lun)==-1)
			{
				printk(KERN_ERR "i2o_scsi: Unable to get lun for tid %d.\n", d->id);
				continue;
			}
			h->task[lun]=unit->id;
			h->tagclock[lun]=jiffies;
		}
	}		
}

int i2o_scsi_detect(Scsi_Host_Template * tpnt)
{
	unsigned long flags;
	struct Scsi_Host *shpnt = NULL;
	int i;
	int count;

	printk("i2o_scsi.c: %s\n", VERSION_STRING);

	if(i2o_install_handler(&i2o_scsi_handler)<0)
	{
		printk(KERN_ERR "i2o_scsi: Unable to install OSM handler.\n");
		return 0;
	}
	
	scsi_context = i2o_scsi_handler.context;
	
	init_timer(&retry_timer);
	retry_timer.data = 0UL;
	retry_timer.function = i2o_retry_run;
	
	printk("SCSI OSM at %d.\n", scsi_context);

	for (count = 0, i = 0; i < MAX_I2O_CONTROLLERS; i++)
	{
		struct i2o_controller *c=i2o_find_controller(i);
		struct i2o_device *d;
		/*
		 *	This controller doesn't exist.
		 */
		
		if(c==NULL)
			continue;
			
		/*
		 *	Fixme - we need some altered device locking. This
		 *	is racing with device addition in theory. Easy to fix.
		 */
		
		for(d=c->devices;d!=NULL;d=d->next)
		{
			int class = d->type & 0xFFFF;
			/*
			 *	Scsi or fibrechannel busses only
			 */
			if(class!=0x1040 && class!=0x1050)
				continue;
		
//			printk("Found a controller.\n");	
			shpnt = scsi_register(tpnt, sizeof(struct i2o_scsi_host));
			save_flags(flags);
			cli();
			shpnt->unique_id = (u32)d;
			shpnt->io_port = 0;
			shpnt->n_io_port = 0;
			shpnt->irq = 0;
			shpnt->this_id = /* Good question */15;
			restore_flags(flags);
//			printk("Scanning I2O port %d.\n", d->id);
			i2o_scsi_init(c, d, (struct i2o_scsi_host *) shpnt->hostdata);
			count++;
		}
	}
	if(count==0)
	{
		flush_pending();
		del_timer(&retry_timer);
		i2o_remove_handler(&i2o_scsi_handler);
	}
	i2o_scsi_hosts = count;
	return count;
}

int i2o_scsi_release(struct Scsi_Host *host)
{
	if(--i2o_scsi_hosts==0)
	{
		flush_pending();
		del_timer(&retry_timer);
		i2o_remove_handler(&i2o_scsi_handler);
	}
	return 0;
}


const char *i2o_scsi_info(struct Scsi_Host *SChost)
{
//	sprintf(info, "I2O SCSI driver");
//	return info;
	return("foo");
}


/*
 * From the wd93 driver:
 * Returns true if there will be a DATA_OUT phase with this command, 
 * false otherwise.
 * (Thanks to Joerg Dorchain for the research and suggestion.)
 *
 */
static int is_dir_out(Scsi_Cmnd *cmd)
{
   switch (cmd->cmnd[0]) {
      case WRITE_6:           case WRITE_10:          case WRITE_12:
      case WRITE_LONG:        case WRITE_SAME:        case WRITE_BUFFER:
      case WRITE_VERIFY:      case WRITE_VERIFY_12:      
      case COMPARE:           case COPY:              case COPY_VERIFY:
      case SEARCH_EQUAL:      case SEARCH_HIGH:       case SEARCH_LOW:
      case SEARCH_EQUAL_12:   case SEARCH_HIGH_12:    case SEARCH_LOW_12:      
      case FORMAT_UNIT:       case REASSIGN_BLOCKS:   case RESERVE:
      case MODE_SELECT:       case MODE_SELECT_10:    case LOG_SELECT:
      case SEND_DIAGNOSTIC:   case CHANGE_DEFINITION: case UPDATE_BLOCK:
      case SET_WINDOW:        case MEDIUM_SCAN:       case SEND_VOLUME_TAG:
      case 0xea:
         return 1;
      default:
         return 0;
      }
}

int i2o_scsi_queuecommand(Scsi_Cmnd * SCpnt, void (*done) (Scsi_Cmnd *))
{
	int i;
	int tid;
	struct i2o_controller *c;
	Scsi_Cmnd *current_command;
	struct Scsi_Host *host;
	struct i2o_scsi_host *hostdata;
	u32 *msg, *mptr;
	u32 m;
	u32 *lenptr;
	int direction;
	int scsidir;
	u32 len;
	
	/*
	 *	The scsi layer should be handling this stuff
	 */

	if(is_dir_out(SCpnt))
	{
		direction=0x04000000;
		scsidir=0x80000000;
	}
	else
	{
		scsidir=0x40000000;
		direction=0x00000000;
	}
	
	/*
	 *	Do the incoming paperwork
	 */
	 
	host = SCpnt->host;
	hostdata = (struct i2o_scsi_host *)host->hostdata;
	SCpnt->scsi_done = done;
	
	if(SCpnt->target > 15)
	{
		printk(KERN_ERR "i2o_scsi: Wild target %d.\n", SCpnt->target);
		return -1;
	}
	
	tid = hostdata->task[SCpnt->target];
	
	dprintk(("qcmd: Tid = %d\n", tid));
	
	current_command = SCpnt;		/* set current command                */
	current_command->scsi_done = done;	/* set ptr to done function           */

	/* We don't have such a device. Pretend we did the command 
	   and that selection timed out */
	
	if(tid == -1)
	{
//		printk("Unknown item - spoof completion\n");
		SCpnt->result = DID_NO_CONNECT << 16;
		done(SCpnt);
		return 0;
	}
	
	dprintk(("Real scsi messages.\n"));

	c = hostdata->controller;
	
	/*
	 *	Obtain an I2O message. Right now we _have_ to obtain one
	 *	until the scsi layer stuff is cleaned up.
	 */	
	 
	do
	{
		mb();
		m = *c->post_port;
	}
	while(m==0xFFFFFFFF);
	msg = bus_to_virt(c->mem_offset + m);
	
	/*
	 *	Put together a scsi execscb message
	 */
	
	msg[1] = I2O_CMD_SCSI_EXEC<<24|HOST_TID<<12|tid;
	msg[2] = scsi_context;		/* So the I2O layer passes to us */
	/* Sorry 64bit folks. FIXME */
	msg[3] = (u32)SCpnt;		/* We want the SCSI control block back */
	/* Direction, disconnect ok, no tagging (yet) */
	msg[4] = scsidir|(1<<29)|SCpnt->cmd_len;

	/*
	 *	Attach tags to the devices
	 */	
	if(SCpnt->device->tagged_supported)
	{
		switch(SCpnt->tag)
		{
			case SIMPLE_QUEUE_TAG:
				msg[4]|=(1<<23);
				break;
			case HEAD_OF_QUEUE_TAG:
				msg[4]|=(1<<24);
				break;
			case ORDERED_QUEUE_TAG:
				msg[4]|=(1<<23)|(1<<24);
				break;
			default:
				if((jiffies - hostdata->tagclock[SCpnt->target]) > (5*HZ))
				{
					msg[4]|=(1<<23)|(1<<24);
					hostdata->tagclock[SCpnt->target]=jiffies;
				}
				else
					msg[4]|=(1<<23);
		}
	}

	mptr=msg+5;

	/* 
	 *	Write SCSI command into the message - always 16 byte block 
	 */
	 
	memcpy(mptr, SCpnt->cmnd, 16);
	mptr+=4;
	lenptr=mptr++;		/* Remember me - fill in when we know */
	
			
	/*
	 *	Now fill in the SGList and command 
	 *
	 *	FIXME: we need to set the sglist limits according to the 
	 *	message size of the I2O controller. We might only have room
	 *	for 6 or so worst case
	 */
	
	if(SCpnt->use_sg)
	{
		struct scatterlist *sg = (struct scatterlist *)SCpnt->request_buffer;
	
		len = 0;
			
		for(i = 0 ; i < SCpnt->use_sg; i++)
		{
			*mptr++=direction|0x10000000|sg->length;
			len+=sg->length;
			*mptr++=virt_to_bus(sg->address);
		}
		mptr[-2]|=0xC0000000;	/* End of List and block */
		*lenptr=len;
	}
	else
	{
		dprintk(("non sg for %p, %d\n", SCpnt->request_buffer,
				SCpnt->request_bufflen));
		*mptr++=0xD0000000|direction|SCpnt->request_bufflen;
		*mptr++=virt_to_bus(SCpnt->request_buffer);
		*lenptr = len = SCpnt->request_bufflen;
		/* No transfer ? - fix up the request */
		if(len == 0)
			msg[4]&=~0xC0000000;
	}
	
	/*
	 *	Stick the headers on 
	 */

	msg[0] = (mptr-msg)<<16 | SGL_OFFSET_10;
	
	/* Queue the message */
	i2o_post_message(c,m);
	
	mb();
	dprintk(("Issued %ld\n", current_command->serial_number));
	
	return 0;
}

static void internal_done(Scsi_Cmnd * SCpnt)
{
	SCpnt->SCp.Status++;
}

int i2o_scsi_command(Scsi_Cmnd * SCpnt)
{
	i2o_scsi_queuecommand(SCpnt, internal_done);
	SCpnt->SCp.Status = 0;
	while (!SCpnt->SCp.Status)
		barrier();
	return SCpnt->result;
}

int i2o_scsi_abort(Scsi_Cmnd * SCpnt)
{
	struct i2o_controller *c;
	struct Scsi_Host *host;
	struct i2o_scsi_host *hostdata;
	u32 *msg;
	u32 m;
	int tid;
	
	printk("i2o_scsi_abort\n");
	
	host = SCpnt->host;
	hostdata = (struct i2o_scsi_host *)host->hostdata;
	tid = hostdata->task[SCpnt->target];
	if(tid==-1)
	{
		printk(KERN_ERR "impossible command to abort.\n");
		return SCSI_ABORT_NOT_RUNNING;
	}
	c = hostdata->controller;
	
	/*
	 *	Obtain an I2O message. Right now we _have_ to obtain one
	 *	until the scsi layer stuff is cleaned up.
	 */	
	 
	do
	{
		mb();
		m = *c->post_port;
	}
	while(m==0xFFFFFFFF);
	msg = bus_to_virt(c->mem_offset + m);
	
	msg[0] = FIVE_WORD_MSG_SIZE;
	msg[1] = I2O_CMD_SCSI_ABORT<<24|HOST_TID<<12|tid;
	msg[2] = scsi_context;
	msg[3] = 0;	/* Not needed for an abort */
	msg[4] = (u32)SCpnt;	
	wmb();
	i2o_post_message(c,m);
	wmb();
//	SCpnt->result = DID_RESET << 16;
//	SCpnt->scsi_done(SCpnt);
	return SCSI_ABORT_PENDING;
}

int i2o_scsi_reset(Scsi_Cmnd * SCpnt, unsigned int reset_flags)
{
	int tid;
	struct i2o_controller *c;
	struct Scsi_Host *host;
	struct i2o_scsi_host *hostdata;
	u32 m;
	u32 *msg;

	printk("i2o_scsi_reset\n");
	
	/*
	 *	Find the TID for the bus
	 */

	host = SCpnt->host;
	hostdata = (struct i2o_scsi_host *)host->hostdata;
	tid = hostdata->bus_task;
	c = hostdata->controller;

	/*
	 *	Now send a SCSI reset request. Any remaining commands
	 *	will be aborted by the IOP. We need to catch the reply
	 *	possibly ?
	 */
	 
	m = *c->post_port;
	
	/*
	 *	No free messages, try again next time - no big deal
	 */
	 
	if(m == 0xFFFFFFFF)
		return SCSI_RESET_PUNT;
	
	msg = bus_to_virt(c->mem_offset + m);
	msg[0] = FOUR_WORD_MSG_SIZE|SGL_OFFSET_0;
	msg[1] = I2O_CMD_SCSI_BUSRESET<<24|HOST_TID<<12|tid;
	msg[2] = scsi_context|0x80000000;	
	/* We use the top bit to split controller and unit transactions */
	/* Now store unit,tid so we can tie the completion back to a specific device */
	msg[3] = c->unit << 16 | tid;
	i2o_post_message(c,m);
	return SCSI_RESET_PENDING;
}

/*
 *	This is anyones guess quite frankly.
 */
 
int i2o_scsi_bios_param(Disk * disk, kdev_t dev, int *ip)
{
	int size;

	size = disk->capacity;
	ip[0] = 64;		/* heads                        */
	ip[1] = 32;		/* sectors                      */
	if ((ip[2] = size >> 11) > 1024) {	/* cylinders, test for big disk */
		ip[0] = 255;	/* heads                        */
		ip[1] = 63;	/* sectors                      */
		ip[2] = size / (255 * 63);	/* cylinders                    */
	}
	return 0;
}

/* Loadable module support */
#ifdef MODULE

MODULE_AUTHOR("Red Hat Software");

Scsi_Host_Template driver_template = I2OSCSI;

#include "../scsi/scsi_module.c"
#endif
