/*
 *	Core I2O structure managment
 *
 *	A lot of the I2O message side code from this is taken from the
 *	Red Creek RCPCI45 adapter driver by Red Creek Communications
 */
 
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/i2o.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/malloc.h>

#include <asm/io.h>
#include <asm/spinlock.h>


/*
 *	Size of the I2O module table
 */
 

static struct i2o_handler *i2o_handlers[MAX_I2O_MODULES];
static struct i2o_controller *i2o_controllers[MAX_I2O_CONTROLLERS];


extern int i2o_online_controller(struct i2o_controller *c);

/*
 *	I2O configuration spinlock. This isnt a big deal for contention
 *	so we have one only
 */
 
static spinlock_t i2o_configuration_lock = SPIN_LOCK_UNLOCKED;

/*
 *	Install an I2O handler - these handle the asynchronous messaging
 *	from the card once it has initialised.
 */
 
int i2o_install_handler(struct i2o_handler *h)
{
	int i;
	spin_lock(&i2o_configuration_lock);
	for(i=0;i<MAX_I2O_MODULES;i++)
	{
		if(i2o_handlers[i]==NULL)
		{
			h->context = i;
			i2o_handlers[i]=h;
			spin_unlock(&i2o_configuration_lock);
			return 0;
		}
	}
	spin_unlock(&i2o_configuration_lock);
	return -ENOSPC;
}

int i2o_remove_handler(struct i2o_handler *h)
{
	i2o_handlers[h->context]=NULL;
	return 0;
}
	

/*
 *	Each I2O controller has a chain of devices on it - these match
 *	the useful parts of the LCT of the board.
 */
 
int i2o_install_device(struct i2o_controller *c, struct i2o_device *d)
{
	spin_lock(&i2o_configuration_lock);
	d->controller=c;
	d->owner=NULL;
	d->next=c->devices;
	c->devices=d;
	spin_unlock(&i2o_configuration_lock);
	return 0;
}


int i2o_delete_device(struct i2o_device *d)
{
	struct i2o_device **p;

	spin_lock(&i2o_configuration_lock);
	
	p=&(d->controller->devices);

	/*
	 *	Hey we have a driver!
	 */
	 
	if(d->owner)
	{
		spin_unlock(&i2o_configuration_lock);
		return -EBUSY;
	}

	/*
	 *	Seek, locate
	 */
	 			
	while(*p!=NULL)
	{
		if(*p==d)
		{
			/*
			 *	Destroy
			 */
			*p=d->next;
			kfree(d);
			spin_unlock(&i2o_configuration_lock);		
			return 0;
		}
		p=&((*p)->next);
	}
	spin_unlock(&i2o_configuration_lock);
	printk(KERN_ERR "i2o_destroy_device: passed invalid device.\n");
	return -EINVAL;
}


/*
 *	Add and remove controllers from the I2O controller list
 */
 
int i2o_install_controller(struct i2o_controller *c)
{
	int i;
	spin_lock(&i2o_configuration_lock);
	for(i=0;i<MAX_I2O_CONTROLLERS;i++)
	{
		if(i2o_controllers[i]==NULL)
		{
			i2o_controllers[i]=c;
			c->next=i2o_controller_chain;
			i2o_controller_chain=c;
			c->unit = i;
			sprintf(c->name, "i2o%d", i);
			spin_unlock(&i2o_configuration_lock);
			return 0;
		}
	}
	printk(KERN_ERR "No free i2o controller slots.\n");
	return -EBUSY;
}

int i2o_delete_controller(struct i2o_controller *c)
{
	struct i2o_controller **p;
	
	spin_lock(&i2o_configuration_lock);
	if(atomic_read(&c->users))
	{
		spin_unlock(&i2o_configuration_lock);
		return -EBUSY;
	}
	while(c->devices)
	{
		if(i2o_delete_device(c->devices)<0)
		{
			/* Shouldnt happen */
			spin_unlock(&i2o_configuration_lock);
			return -EBUSY;
		}
	}
	c->destructor(c);
	
	p=&i2o_controller_chain;
	
	while(*p)
	{
		if(*p==c)
		{
			*p=c->next;
			spin_unlock(&i2o_configuration_lock);
			if(c->page_frame);
				kfree(c->page_frame);
			i2o_controllers[c->unit]=NULL;
			kfree(c);
			return 0;
		}
		p=&((*p)->next);
	}
	spin_unlock(&i2o_configuration_lock);
	printk(KERN_ERR "i2o_delete_controller: bad pointer!\n");
	return -ENOENT;
}

void i2o_unlock_controller(struct i2o_controller *c)
{
	atomic_dec(&c->users);
}

struct i2o_controller *i2o_find_controller(int n)
{
	struct i2o_controller *c;
	
	if(n<0 || n>=MAX_I2O_CONTROLLERS)
		return NULL;
	
	spin_lock(&i2o_configuration_lock);
	c=i2o_controllers[n];
	if(c!=NULL)
		atomic_inc(&c->users);
	spin_unlock(&i2o_configuration_lock);
	return c;
}
	

/*
 *	Track if a device is being used by a driver
 */
 
int i2o_claim_device(struct i2o_device *d, struct i2o_driver *r)
{
	spin_lock(&i2o_configuration_lock);
	if(d->owner)
	{
		spin_unlock(&i2o_configuration_lock);
		return -EBUSY;
	}
	atomic_inc(&d->controller->users);
	d->owner=r;
	spin_unlock(&i2o_configuration_lock);
	return 0;
}

int i2o_release_device(struct i2o_device *d)
{
	spin_lock(&i2o_configuration_lock);
	if(d->owner==NULL)
	{
		spin_unlock(&i2o_configuration_lock);
		return -EINVAL;
	}
	atomic_dec(&d->controller->users);
	d->owner=NULL;
	spin_unlock(&i2o_configuration_lock);
	return 0;
}

/*
 *	This is called by the bus specific driver layer when an interrupt
 *	or poll of this card interface is desired.
 */
 
void i2o_run_queue(struct i2o_controller *c)
{
	struct i2o_message *m;
	u32 mv;
	
	while((mv=*c->read_port)!=0xFFFFFFFF)
	{
		struct i2o_handler *i;
		m=(struct i2o_message *)bus_to_virt(mv);
		/*
		 *	Temporary Debugging
		 */
		if(((m->function_addr>>24)&0xFF)==0x15)
			printk("UTFR!\n");
//		printk("dispatching.\n");
		i=i2o_handlers[m->initiator_context&(MAX_I2O_MODULES-1)];
		if(i)
			i->reply(i,c,m);
		else
			printk("Spurious reply\n");
	 	i2o_flush_reply(c,mv);
		mb();
	}
}


/*
 *	Wait up to 5 seconds for a message slot to be available.
 */
 
static u32 i2o_wait_message(struct i2o_controller *c, char *why)
{
	long time=jiffies;
	u32 m;
	while((m=*c->post_port)==0xFFFFFFFF)
	{
		if((jiffies-time)>=5*HZ)
		{
			printk(KERN_ERR "%s: Timeout waiting for message to send %s.\n", 
				c->name, why);
			return 0xFFFFFFFF;
		}
		schedule();
		barrier();
	}
	return m;
}


/*
 *	Wait up to 5 seconds for a reply to be available.
 */
 
static u32 i2o_wait_reply(struct i2o_controller *c, char *why)
{
	u32 m;
	long time=jiffies;
	
	while((m=*c->read_port)==0xFFFFFFFF)
	{
		if(jiffies-time >= 5*HZ )
		{
			printk(KERN_ERR "%s: table waiiting for %s reply.\n",
				c->name, why);
			return -ETIMEDOUT;
		}
		schedule();
	}
	return m;
}
	


/*
 *	i2o table walking. We just provide a single element retrieve. You can
 *	all sorts of fancy lookups in I2O but we have no performance critical
 *	lookups so why write all the code for it.
 */
 

int i2o_query_table(struct i2o_controller *c, int tid, void *buf, int buflen, 
	int group, int field, u32 *key, int keylen)
{
	u32 m;
	u32 *msg;
	u16 op[64];
	u32 *p;
	int i;
	u32 *rbuf;

	op[0]=1;			/* One Operation */
	op[1]=0;			/* PAD */
	op[2]=2;			/* LIST_GET */
	op[3]=group;			/* group number */
	op[4]=1;			/* 1 field */
	op[5]=field;			/* Field number */
	op[6]=1;			/* Key count */
	memcpy(op+7, key, keylen);	/* Key */
	
	m=i2o_wait_message(c, "I2O query table.");
	if(m==0xFFFFFFFF)
	{	
		return -ETIMEDOUT;
	}
	
	msg=(u32 *)(c->mem_offset+m);
	
	rbuf=kmalloc(buflen+32, GFP_KERNEL);
	if(rbuf==NULL)
	{
		printk(KERN_ERR "No free memory for table read.\n");
		return -ENOMEM;
	}
	msg[0]=FIVE_WORD_MSG_SIZE|SGL_OFFSET_5;
	msg[1]=I2O_CMD_UTIL_PARAMS_GET<<24|HOST_TID<<12|tid;
	msg[2]=0;			/* Context */
	msg[3]=0;
	msg[4]=0;
	msg[5]=0x54000000|12;
	msg[6]=virt_to_bus(op);
	msg[7]=0xD0000000|(32+buflen);
	msg[8]=virt_to_bus(rbuf);

	i2o_post_message(c,m);
	barrier();
	
	/*
	 *	Now wait for a reply
	 */
	 
	
	m=i2o_wait_reply(c, "Table read timeout");
	
	if(m==0xFFFFFFFF)
	{
		kfree(rbuf);
		return -ETIMEDOUT;
	}
	
	msg = (u32 *)bus_to_virt(m);

	if(msg[4]>>24)
	{
		printk(KERN_WARNING "ERROR 0x%08X\n", msg[4]&0xFF00FFFF);
	}
	
	p=rbuf;

	/* Ok 'p' is the reply block - lets see what happened */
	/* p0->p2 are the header */
	
	/* FIXME: endians - turn p3 to little endian */
	
	i=(p[0]&0xFFFF)<<2;		/* Message size */
	if(i<buflen)
		buflen=i;
	
	/* Do we have an error block ? */
	if(p[0]&0xFF000000)
	{
		printk(KERN_ERR "%s: error in field read.\n",
			c->name);
		kfree(rbuf);
		return -EBADR;
	}
		
	/* p[1] holds the more flag and row count - we dont care */
	
	/* Ok it worked p[2]-> hold the data */
	memcpy(buf,  p+2, buflen);
	
	kfree(rbuf);
	
	/* Finally return the message */
	*c->read_port=m;
	return buflen;
}


int i2o_query_scalar(struct i2o_controller *c, int tid, void *buf, int buflen, 
	int group, int field)
{
	u32 m;
	u32 *msg;
	u16 op[8];
	u32 *p;
	int i;
	u32 *rbuf;

	op[0]=1;			/* One Operation */
	op[1]=0;			/* PAD */
	op[2]=1;			/* FIELD_GET */
	op[3]=group;			/* group number */
	op[4]=1;			/* 1 field */
	op[5]=field;			/* Field number */

	m=i2o_wait_message(c, "I2O query scalar.");
	if(m==0xFFFFFFFF)
	{	
		return -ETIMEDOUT;
	}
	
	msg=(u32 *)(c->mem_offset+m);
	
	rbuf=kmalloc(buflen+32, GFP_KERNEL);
	if(rbuf==NULL)
	{
		printk(KERN_ERR "No free memory for scalar read.\n");
		return -ENOMEM;
	}

	msg[0]=NINE_WORD_MSG_SIZE|SGL_OFFSET_5;
	msg[1]=I2O_CMD_UTIL_PARAMS_GET<<24|HOST_TID<<12|tid;
	msg[2]=0;			/* Context */
	msg[3]=0;
	msg[4]=0;
	msg[5]=0x54000000|12;
	msg[6]=virt_to_bus(op);
	msg[7]=0xD0000000|(32+buflen);
	msg[8]=virt_to_bus(rbuf);

	i2o_post_message(c,m);
	barrier();
	
	/*
	 *	Now wait for a reply
	 */
	 
	 
	m=i2o_wait_reply(c, "Scalar read timeout");
	
	if(m==0xFFFFFFFF)
	{
		kfree(rbuf);
		return -ETIMEDOUT;
	}
	 
	msg = (u32 *)bus_to_virt(m);
	if(msg[4]>>24)
	{
		printk(KERN_WARNING "ERROR 0x%08X\n", msg[4]&0xFF00FFFF);
	}
	
	p=rbuf;

	/* Ok 'p' is the reply block - lets see what happened */
	/* p0->p2 are the header */
	
	/* FIXME: endians - turn p3 to little endian */
	
	if((p[0]&0xFFFF)!=1)	
		printk(KERN_WARNING "Suspicious field read return 0x%08X\n", p[0]);
		
	i=(p[1]&0xFFFF)<<2;		/* Message size */
	if(i<buflen)
		buflen=i;
	
	/* Do we have an error block ? */
	if(p[1]&0xFF000000)
	{
		printk(KERN_ERR "%s: error in field read.\n",
			c->name);
		kfree(rbuf);
		return -EBADR;
	}
		
	/* p[1] holds the more flag and row count - we dont care */
	
	/* Ok it worked p[2]-> hold the data */
	memcpy(buf,  p+2, buflen);
	
	kfree(rbuf);
	
	/* Finally return the message */
	*c->read_port=m;
	return buflen;
}

/*
 *	 Dump the information block associated with a given unit (TID)
 */
 
void i2o_report_controller_unit(struct i2o_controller *c, int unit)
{
	char buf[64];
	
	if(i2o_query_scalar(c, unit, buf, 16, 0xF100, 3)>=0)
	{
		buf[16]=0;
		printk("Vendor:    %s", buf);
	}
	if(i2o_query_scalar(c, unit, buf, 16, 0xF100, 4)>=0)
	{
		buf[16]=0;
		printk("Device: %s", buf);
	}
	if(i2o_query_scalar(c, unit, buf, 8, 0xF100, 6)>=0)
	{
		buf[8]=0;
		printk("Rev: %s", buf);
	}
#if 0
	if(i2o_query_scalar(c, unit, buf, 16, 0xF100, 5)>=0)
	{
		buf[16]=0;
		printk(KERN_INFO "Description: %s\n", buf);
	}
#endif	
	printk("\n");
}


/*
 *	Parse the hardware resource table. Right now we print it out
 *	and don't do a lot with it. We should collate these and then
 *	interact with the Linux resource allocation block.
 *
 *	Lets prove we can read it first eh ?
 *
 *	This is full of endianisms!
 */
 
static int i2o_parse_hrt(struct i2o_controller *c, u8 *p)
{
	u32 *rows=(u32 *)p;
	u8 *d;
	int count;
	int length;
	int i;
	int state;
	
	if(p[3]!=0)
	{
		printk(KERN_ERR "i2o: HRT table for controller is too new a version.\n");
		return -1;
	}
		
	count=p[0]|(p[1]<<8);
	length = p[2];
	
	printk(KERN_INFO "HRT has %d entries of %d bytes each.\n",
		count, length<<2);
	
	rows+=2;
	
	for(i=0;i<count;i++)
	{
		printk(KERN_INFO "Adapter %08X: ", rows[0]);
		p=(u8 *)(rows+1);
		d=(u8 *)(rows+2);
		state=p[1]<<8|p[0];
		
		printk("TID %04X:[", state&0xFFF);
		state>>=12;
		if(state&(1<<0))
			printk("H");
		if(state&(1<<2))
		{
			printk("P");
			if(state&(1<<1))
				printk("C");
		}
		if(state>9)
			printk("*");
		
		printk("]:");
		
		switch(p[3]&0xFFFF)
		{
			case 0:
				/* Adapter private bus - easy */
				printk("Local bus %d: I/O at 0x%04X Mem 0x%08X", 
					p[2], d[1]<<8|d[0], *(u32 *)(d+4));
				break;
			case 1:
				/* ISA bus */
				printk("ISA %d: CSN %d I/O at 0x%04X Mem 0x%08X",
					p[2], d[2], d[1]<<8|d[0], *(u32 *)(d+4));
				break;
					
			case 2: /* EISA bus */
				printk("EISA %d: Slot %d I/O at 0x%04X Mem 0x%08X",
					p[2], d[3], d[1]<<8|d[0], *(u32 *)(d+4));
				break;

			case 3: /* MCA bus */
				printk("MCA %d: Slot %d I/O at 0x%04X Mem 0x%08X",
					p[2], d[3], d[1]<<8|d[0], *(u32 *)(d+4));
				break;

			case 4: /* PCI bus */
				printk("PCI %d: Bus %d Device %d Function %d",
					p[2], d[2], d[1], d[0]);
				break;

			case 0x80: /* Other */
			default:
				printk("Unsupported bus type.");
				break;
		}
		printk("\n");
		rows+=length;
	}
	return 0;
}
	
/*
 *	The logical configuration table tells us what we can talk to
 *	on the board. Most of the stuff isn't interesting to us. 
 */

static int i2o_parse_lct(struct i2o_controller *c, u32 *lct)
{
	
	int i;
	int max;
	int tid;
	u32 *p;
	struct i2o_device *d;
	
	max=lct[0]&0xFFFF;
	
	max-=3;
	max/=9;
	
	printk(KERN_INFO "LCT has %d entries.\n", max);
	if(lct[1]&(1<<0))
		printk(KERN_WARNING "Configuration dialog desired.\n");
		
	p=lct+3;
	
	for(i=0;i<max;i++)
	{
		d = (struct i2o_device *)kmalloc(sizeof(struct i2o_device), GFP_KERNEL);
		if(d==NULL)
		{
			printk("i2o_core: Out of memory for LCT data.\n");
			return -ENOMEM;
		}
		
		d->controller = c;
		d->next = NULL;
		
		d->id = tid = (p[0]>>16)&0xFFF;
		d->type = p[3];
		d->parent =  (p[5]>>12)&0xFFF;
		d->flags = 0;
		
		printk(KERN_INFO "TID %d.\n", tid);
		printk(KERN_INFO "  ");

		i2o_report_controller_unit(c, tid);
		
		i2o_install_device(c, d);
		
		printk(KERN_INFO "  Category: ");
		
		switch(p[3])
		{
			case 0x1000: 
				printk("Executive            ");
				break;
			case 0x1001: 
				printk("Device Driver Manager");
				break;
			case 0x1010: 
				printk("Block Device         ");
				break;
			case 0x1011:
				printk("Tape Device          ");
				break;
			case 0x1020:
				printk("LAN Inteface         ");
				break;
			case 0x1030:
				printk("WAN Interface        ");
				break;
			case 0x1040:
				printk("Fibre Channel Port   ");
				break;
			case 0x1041:
				printk("FC Connection        ");
				break;
			case 0x1051:
				printk("SCSI Device          ");
				break;
			case 0x1060:
				printk("ATE Port             ");
				break;
			case 0x1061:
				printk("ATE Device           ");
				break;
			case 0x1080:
				printk("Secondary Bus Port   ");
				break;
			default:
				printk("0x%08X             ", p[3]);
				break;
		}
		
		printk("  Subclass: 0x%08X   Flags: ",
			p[4]);
			
		if(p[2]&(1<<0))
			printk("C");
		if(p[2]&(1<<1))
			printk("M");
		if(!(p[2]&(1<<4)))
			printk("P");
		if(!(p[2]&(1<<5)))
			printk("m");
		printk("\n");
		p+=9;
	}
	return 0;
}

/*
 *	Bring an I2O controller into HOLD state. See the 1.5
 *	spec. Basically we go
 *
 *	Wait for the message queue to initialise. 
 *	If it didnt -> controller is dead
 *	
 *	Send a get status using the message queue
 *	Poll for a reply block 88 bytes long
 *
 *	Send an initialise outbound queue
 *	Poll for a reply
 *
 *	Post our blank messages to the queue FIFO
 *
 *	Send GetHRT, Parse it
 */

int i2o_activate_controller(struct i2o_controller *c)
{
	long time;
	u32 m;
	u8 *workspace;
	u32 *msg;
	int i;
	
	printk(KERN_INFO "Configuring I2O controller at 0x%08X.\n", (u32)c->mem_phys);
	
	m=i2o_wait_message(c, "initialise");
	if(m==0xFFFFFFFF)
		return -ETIMEDOUT;

	msg=(u32 *)(c->mem_offset+m);
	
	workspace = (void *)kmalloc(88, GFP_KERNEL);
	if(workspace==NULL)
	{
		printk(KERN_ERR "IOP initialisation failed - no free memory.\n");
		return -ENOMEM;
	}
	
	memset(workspace, 0, 88);
	
	msg[0]=NINE_WORD_MSG_SIZE|SGL_OFFSET_0;
	msg[1]=I2O_CMD_STATUS_GET<<24|HOST_TID<<12|ADAPTER_TID;
	msg[2]=0;
	msg[3]=0;
	msg[4]=0;
	msg[5]=0;
	msg[6]=virt_to_phys(workspace);
	msg[7]=0;	/* 64bit host FIXME */
	msg[8]=88;

	i2o_post_message(c,m);

	/*
	 *	Wait for a reply
	 */

	time=jiffies;
		 
	while(workspace[87]!=0xFF)
	{
		if((jiffies-time)>=5*HZ)
		{
			printk(KERN_ERR "IOP get status timeout.\n");
			kfree(workspace);
			return -ETIMEDOUT;
		}
		schedule();
		barrier();
	}
	
	/*
	 *	Ok the reply has arrived. Fill in the important stuff
	 */
	 
	c->status = workspace[10];
	c->inbound_size = workspace[12]|(workspace[13]<<8);

	
	/*
	 *	If the board is running, reset it - we have no idea
	 *	what kind of a mess the previous owner left it in.
	 */
	 
//	if(c->status == ADAPTER_STATE_OPERATIONAL)
//		i2o_reset_device(c);

	
	m=i2o_wait_message(c, "initqueue");
	if(m==0xFFFFFFFF)
	{	
		kfree(workspace);
		return -ETIMEDOUT;
	}
	
	msg=(u32 *)(c->mem_offset+m);

	msg[0]= EIGHT_WORD_MSG_SIZE| TRL_OFFSET_6;
	msg[1]= I2O_CMD_OUTBOUND_INIT<<24 | HOST_TID<<12 | ADAPTER_TID;
	msg[2]= DEFAULT_RECV_INIT_CONTEXT;
	msg[3]= 0x0106;			/* Transaction context */
	msg[4]= 4096;			/* Host page frame size */
	msg[5]= MSG_FRAME_SIZE<<16|0x80;	/* Outbound msg frame size and Initcode */
	msg[6]= 0xD0000004;		/* Simple SG LE, EOB */
	msg[7]= virt_to_phys(workspace);
	*((u32 *)workspace)=0;

	/*
	 *	Post it
	 */

	i2o_post_message(c,m);
	
	barrier();
	
	time=jiffies;
	
	while(workspace[0]!=I2O_CMD_OUTBOUND_INIT_COMPLETE)
	{
		if((jiffies-time)>=5*HZ)
		{
			printk(KERN_ERR "IOP outbound initialise failed.\n");
			kfree(workspace);
			return -ETIMEDOUT;
		}
		schedule();
		barrier();
	}
	
	kfree(workspace);

	c->page_frame = kmalloc(MSG_POOL_SIZE, GFP_KERNEL);
	if(c->page_frame==NULL)
	{
		printk(KERN_ERR "IOP init failed: no memory for message page.\n");
		return -ENOMEM;
	}
	
	m=virt_to_phys(c->page_frame);
	
	for(i=0; i< NMBR_MSG_FRAMES; i++)
	{
		*c->read_port=m;
		mb();
		m+=MSG_FRAME_SIZE;
	}
	
	/*
	 *	The outbound queue is initialised and loaded,
	 *
	 *	Now we need the Hardware Resource Table. We must ask for
	 *	this next we can't issue random messages yet.
	 */


	workspace=kmalloc(2048, GFP_KERNEL);
	if(workspace==NULL)
	{
		printk(KERN_ERR "IOP init failed; no memory.\n");
		return -ENOMEM;
	}
	
	m=i2o_wait_message(c, "I2O HRT timeout.");
	if(m==0xFFFFFFFF)
	{	
		kfree(workspace);
		return -ETIMEDOUT;
	}
	
	msg=(u32 *)(c->mem_offset+m);

	msg[0]= SIX_WORD_MSG_SIZE| SGL_OFFSET_4;
	msg[1]= I2O_CMD_HRT_GET<<24 | HOST_TID<<12 | ADAPTER_TID;
	msg[2]= DEFAULT_RECV_INIT_CONTEXT;
	msg[3]= 0x0;				/* Transaction context */
	msg[4]= (0xD0000000 | 2048);		/* Simple transaction , 2K */
	msg[5]= virt_to_phys(workspace);	/* Dump it here */
	*((u32 *)workspace)=0xFFFFFFFF;
	
	i2o_post_message(c,m);
	
	barrier();
	
	/*
	 *	Now wait for a reply
	 */
	 
	m=i2o_wait_reply(c, "HRT table");
	 
	if(m==0xFFFFFFFF)
	{
		kfree(workspace);
		return -ETIMEDOUT;
	}
	
	msg=(u32 *)bus_to_virt(m);
	
	if(msg[4]>>24)
	{
		printk(KERN_WARNING "ERROR 0x%08X\n", msg[4]&0xFF00FFFF);
	}
	*c->read_port = m;
	
	i2o_parse_hrt(c, workspace);
	
	kfree(workspace);
	
	i2o_online_controller(c);
//	i2o_report_controller_unit(c, ADAPTER_TID);
	return 0;
}


/*
 *	Bring a controller online. Needs completing for multiple controllers
 */
 
int i2o_online_controller(struct i2o_controller *c)
{
	u32 m;
	u32 *msg;
	u32 systab[32];
	u32 bustab[2];
	u32 *workspace;
	
	systab[0]=1;
	systab[1]=0;
	systab[2]=0;
	systab[3]=0;
	systab[4]=0;	/* Organisation ID */
	systab[5]=2;	/* Ident 2 for now */
	systab[6]=1;	/* Memory mapped, v0, segment 1 */
	systab[7]=MSG_FRAME_SIZE>>2;	/* Message size */
	systab[8]=0;
	systab[9]=0;	/* Should be cpabilities */
	systab[10]=virt_to_phys(c->post_port);
	systab[11]=0;
	
	bustab[0]=0xD0000000;	/* hardcoded for Alans box FIXME */
	bustab[1]=0x00010000;
	
	m=i2o_wait_message(c, "SetSysTab");
	if(m==0xFFFFFFFF)
		return -ETIMEDOUT;
	
	/* Now we build the systab */
	msg=(u32 *)(c->mem_offset+m);
	
	msg[0] = NINE_WORD_MSG_SIZE|SGL_OFFSET_6;
	msg[1] = I2O_CMD_SYS_TAB_SET<<24 | HOST_TID<<12 | ADAPTER_TID;
	msg[2] = 0;	/* Context not needed */
	msg[3] = 0;
	msg[4] = (1<<16)|(2<<12);	/* Host 1 I2O 2 */
	msg[5] = 1;			/* Segment 1 */
	
	/*
	 *	Scatter Gather List
	 */

	msg[6] = 0x54000000|48;	/* One table for now */
	msg[7] = virt_to_phys(systab);
	msg[8] = 0xD4000000|48;	/* One table for now */
	msg[9] = virt_to_phys(bustab);
	
	i2o_post_message(c,m);

	barrier();
	
	/*
	 *	Now wait for a reply
	 */
	 
	 
	m=i2o_wait_reply(c, "Systab read");
		
	if(m==0xFFFFFFFF)
		return -ETIMEDOUT;
	
	msg=(u32 *)bus_to_virt(m);
	
	if(msg[4]>>24)
	{
		printk(KERN_ERR "ERROR 0x%08X\n", msg[4]&0xFF00FFFF);
	}
	*c->read_port = m;
	
	/*
	 *	Finally we go online
	 */
	 
	m=i2o_wait_message(c, "No message for SysEnable");
	
	if(m==0xFFFFFFFF)
		return -ETIMEDOUT;
	
	msg=(u32 *)(c->mem_offset+m);
	
	msg[0] = FOUR_WORD_MSG_SIZE|SGL_OFFSET_0;
	msg[1] = I2O_CMD_SYS_ENABLE<<24 | HOST_TID<<12 | ADAPTER_TID;
	msg[2] = 0;		/* Context not needed */
	msg[3] = 0;

	i2o_post_message(c,m);
	
	barrier();
	
	/*
	 *	Now wait for a reply
	 */
	 
	
	m=i2o_wait_reply(c, "Enable");

	if(m==0xFFFFFFFF)
		return -ETIMEDOUT;
	
	msg=(u32 *)bus_to_virt(m);
	
	if(msg[4]>>24)
	{
		printk(KERN_ERR "ERROR 0x%08X\n", msg[4]&0xFF00FFFF);
	}
	*c->read_port = m;
	
	/*
	 *	Grab the LCT, see what is attached
	 */
	 
	m=i2o_wait_message(c, "No message for LCT");
	
	if(m==0xFFFFFFFF)
		return -ETIMEDOUT;
	
	msg=(u32 *)(c->mem_offset+m);
	
	
	workspace = kmalloc(2048, GFP_KERNEL);
	if(workspace==NULL)
	{
		msg[0]=FOUR_WORD_MSG_SIZE|SGL_OFFSET_0;
		msg[1]= HOST_TID<<12|ADAPTER_TID;	/* NOP */
		i2o_post_message(c,m);
		printk(KERN_ERR "No free memory for i2o controller buffer.\n");
		return -ENOMEM;
	}
	
	memset(workspace, 0, 2048);
	
	msg[0] = FOUR_WORD_MSG_SIZE|SGL_OFFSET_6;
	msg[1] = I2O_CMD_LCT_NOTIFY<<24 | HOST_TID<<12 | ADAPTER_TID;
	msg[2] = 0;		/* Context not needed */
	msg[3] = 0;
	msg[4] = 0xFFFFFFFF;	/* All devices */
	msg[5] = 0x00000000;	/* Report now */
	msg[6] = 0xD0000000|2048;
	msg[7] = virt_to_bus(workspace);
	
	i2o_post_message(c,m);
	
	barrier();
	
	/*
	 *	Now wait for a reply
	 */
	 
	m=i2o_wait_reply(c, "LCT");
	
	if(m==0xFFFFFFFF)
	{
		kfree(workspace);
		return -ETIMEDOUT;
	}
	
	msg=(u32 *)bus_to_virt(m);
	
	if(msg[4]>>24)
	{
		printk(KERN_ERR "ERROR 0x%08X\n", msg[4]&0xFF00FFFF);
	}
	
	i2o_parse_lct(c, workspace);
	kfree(workspace);
	
	*c->read_port = m;
	
	return 0;
}
