/*
 *	I2O block device driver. 
 *
 *	This is an initial test release. Most of the good code was taken
 *	from the nbd driver by Pavel Machek, who in turn took some of it
 *	from loop.c. Isn't free software great for reusability 8)
 */

#include <linux/major.h>

#include <linux/module.h>

#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/ioctl.h>
#include <linux/i2o.h>
#include <linux/blkdev.h>

#include <asm/uaccess.h>
#include <asm/io.h>

#define MAJOR_NR I2O_MAJOR

#include <linux/blk.h>

#define MAX_I2OB	16


static int i2ob_blksizes[MAX_I2OB];
static int i2ob_blksize_bits[MAX_I2OB];
static int i2ob_sizes[MAX_I2OB];
static u64 i2ob_bytesizes[MAX_I2OB];

static int i2ob_context;

struct i2ob_device
{
	struct i2o_controller *controller;
	int tid;
	spinlock_t queue_lock;
	int flags;
	int refcnt;
	struct request *head, *tail;
};

static struct i2ob_device i2ob_dev[MAX_I2OB];

#define DEBUG( s )
/* #define DEBUG( s ) printk( s ) 
 */


static void i2ob_send(struct i2ob_device *dev, struct request *req)
{
	struct i2o_controller *c;
	int tid;
	u32 *msg;
	u32 m;
	unsigned long t;
	u64 offset;
	
	c=dev->controller;
	tid=dev->tid;
	
	/*
	 *	Message please
	 */

	t = jiffies;
	 
	do
	{
		mb();
		m = *c->post_port;
	}
	while(m==0xFFFFFFFF && (jiffies-t)<HZ);

	if(m==0xFFFFFFFF)	
	{
		printk(KERN_ERR "i2o_block: no messages ??\n");
		req->errors++;
		return;
	}
	
	/*
	 *	Build a message
	 */
	
	msg = bus_to_virt(c->mem_offset + m);
	
	msg[0] = TEN_WORD_MSG_SIZE | SGL_OFFSET_8;
	msg[2] = i2ob_context;
	msg[3] = (u32)req;	/* 64bit issue again here */
	msg[5] = req->current_nr_sectors << 9;
	
	/* This can be optimised later - just want to be sure its right for
	   starters */
	offset = req->sector << 9;
	msg[6] = offset & 0xFFFFFFFF;
	msg[7] = (offset>>32);
	msg[9] = virt_to_bus(req->buffer);
	
	if(req->cmd == READ)
	{
		msg[1] = I2O_CMD_BLOCK_READ<<24|HOST_TID<<12|tid;
		/* We don't yet do cache/readahead and other magic */
		msg[4] = 1<<16;	
		msg[8] = 0xD0000000|(req->current_nr_sectors << 9);
	}
	else if(req->cmd == WRITE)
	{
		msg[1] = I2O_CMD_BLOCK_WRITE<<24|HOST_TID<<12|tid;
		msg[4] = 1<<16;
		msg[8] = 0xD4000000|(req->current_nr_sectors << 9);
	}

//	printk("Send for %p\n", req);

	i2o_post_message(c,m);
	return;
}

/*
 *	Remove a request from the _locked_ request list. We update both the
 *	list chain and if this is the last item the tail pointer.
 */
 
static void i2ob_unhook_request(struct i2ob_device *dev, struct request *req)
{
	struct request **p = &dev->head;
	struct request *nt = NULL;
	static int crap = 0;
	
	while(*p!=NULL)
	{
		if(*p==req)
		{
			if(dev->tail==req)
				dev->tail = nt;
			*p=req->next;
			return;
		}
		nt=*p;
		p=&(nt->next);
	}
	if(!crap++)
		printk("i2o_block: request queue corrupt!\n");
}

static void i2ob_end_request(struct request *req)
{
	if (end_that_request_first( req, !req->errors, "i2o block" ))
		return;
	end_that_request_last( req );
}


/*
 *	OSM reply handler
 */

static void i2o_block_reply(struct i2o_handler *h, struct i2o_controller *c, struct i2o_message *msg)
{
	struct request *req;
	u8 st;
	u32 *m = (u32 *)msg;
	u8 unit = (u8)m[2]>>8;
	
	if(m[0] & (1<<13))
	{
		printk("IOP fail.\n");
		printk("From %d To %d Cmd %d.\n",
			(m[1]>>12)&0xFFF,
			m[1]&0xFFF,
			m[1]>>24);
		printk("Failure Code %d.\n", m[4]>>24);
		if(m[4]&(1<<16))
			printk("Format error.\n");
		if(m[4]&(1<<17))
			printk("Path error.\n");
		if(m[4]&(1<<18))
			printk("Path State.\n");
		if(m[4]&(1<<18))
			printk("Congestion.\n");
		
		m=(u32 *)bus_to_virt(m[7]);
		printk("Failing message is %p.\n", m);
		
		/* We need to up the request failure count here and maybe
		   abort it */
		req=(struct request *)m[3];
		/* Now flush the message by making it a NOP */
		m[0]&=0x00FFFFFF;
		m[0]|=(I2O_CMD_UTIL_NOP)<<24;
		i2o_post_message(c,virt_to_bus(m));
		
	}
	else
	{
		/*
		 *	Lets see what is cooking.
		 */
		 
		req=(struct request *)m[3];
		st=m[4]>>24;
	
		if(st!=0)
		{
			printk(KERN_ERR "i2ob: error %08X\n", m[4]);
			/*
			 *	Now error out the request block
			 */
			req->errors++;	
		}
	}
	/*
	 *	Dequeue the request.
	 */
	spin_lock(&dev->queue_lock);
//	printk("Ack for %p\n", req);
	i2ob_unhook_request(&i2ob_dev[unit], req);
	spin_unlock(&dev->queue_lock);
	i2ob_end_request(req);
}

static struct i2o_handler i2o_block_handler =
{
	i2o_block_reply,
	"I2O Block OSM",
	0
};


/*
 *	Flush all pending requests as errors. Must call with the queue
 *	locked.
 */
 
static void i2ob_clear_queue(struct i2ob_device *dev)
{
	struct request *req;

	while (1) {
		req = dev->tail;
		if (!req)
			return;
		req->errors++;
		i2ob_end_request(req);

		if (dev->tail == dev->head)
			dev->head = NULL;
		dev->tail = dev->tail->next;
	}
}

/*
 *	Write handling needs some work here, as we should issue multiple
 *	parallel writes.
 */

static void do_i2ob_request(void)
{
	struct request *req;
	int unit;
	struct i2ob_device *dev;
	unsigned long flags;

	while (CURRENT) {
		req = CURRENT;
		unit = MINOR(req->rq_dev);
		dev = &i2ob_dev[unit];
		req->errors = 0;
		CURRENT = CURRENT->next;
		req->next = NULL;

		spin_unlock_irq(&io_request_lock);
		spin_lock_irqsave(&dev->queue_lock, flags);
		if (dev->head == NULL) {
			dev->head = req;
			dev->tail = req;
		} else {
			dev->tail->next = req;
			dev->tail = req;
		}
		i2ob_send(dev, req);
		spin_unlock_irqrestore(&dev->queue_lock, flags);
		spin_lock_irq(&io_request_lock);
	}
	return;
}

/*
 *	Issue device specific ioctl calls.
 */

static int i2ob_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg)
{
	struct i2ob_device *dev;
	int minor;

	/* Anyone capable of this syscall can do *real bad* things */

	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;
	if (!inode)
		return -EINVAL;
	minor = MINOR(inode->i_rdev);
	if (minor >= MAX_I2OB)
		return -ENODEV;

	dev = &i2ob_dev[minor];
	switch (cmd) {
	case BLKGETSIZE:
		return put_user(i2ob_bytesizes[minor] >> 9, (long *) arg);
	}
	return -EINVAL;
}

/*
 *	Close the block device down
 */
 
static int i2ob_release(struct inode *inode, struct file *file)
{
	struct i2ob_device *dev;
	int minor;

	if (!inode)
		return -ENODEV;
	minor = MINOR(inode->i_rdev);
	if (minor >= MAX_I2OB)
		return -ENODEV;
	fsync_dev(inode->i_rdev);
	invalidate_buffers(inode->i_rdev);
	dev = &i2ob_dev[minor];
	if (dev->refcnt <= 0)
		printk(KERN_ALERT "i2ob_release: refcount(%d) <= 0\n", dev->refcnt);
	dev->refcnt--;
	MOD_DEC_USE_COUNT;
	return 0;
}

/*
 *	Open the block device.
 */
 
static int i2ob_open(struct inode *inode, struct file *file)
{
	int minor;

	if (!inode)
		return -EINVAL;
	minor = MINOR(inode->i_rdev);
	if (minor >= MAX_I2OB)
		return -ENODEV;

	i2ob_dev[minor].refcnt++;
	MOD_INC_USE_COUNT;
	return 0;
}

static void i2ob_probe(void)
{
	int i;
	int unit = 0;
	int warned = 0;
		
	for(i=0; i< MAX_I2O_CONTROLLERS; i++)
	{
		struct i2o_controller *c=i2o_find_controller(i);
		struct i2o_device *d;
		
		if(c==NULL)
			continue;
		for(d=c->devices;d!=NULL;d=d->next)
		{
			int class = d->type & 0xFFFF;
			
			if(class!=0x1010)
				continue;
			if(unit<MAX_I2OB)
			{
				i2ob_dev[unit].controller = c;
				i2ob_dev[unit].tid = d->id;
				unit++;
			}
			else
			{
				if(!warned++)
					printk("i2o_block: too many controllers, registering only %d.\n", unit);
			}
		}
	}	
}

static struct file_operations i2ob_fops =
{
	NULL,			/* lseek - default */
	block_read,		/* read - general block-dev read */
	block_write,		/* write - general block-dev write */
	NULL,			/* readdir - bad */
	NULL,			/* select */
	i2ob_ioctl,		/* ioctl */
	NULL,			/* mmap */
	i2ob_open,		/* open */
	NULL,			/* flush */
	i2ob_release		/* release */
};


/*
 * And here should be modules and kernel interface 
 *  (Just smiley confuses emacs :-)
 */

#ifdef MODULE
#define i2ob_init init_module
#endif

int i2ob_init(void)
{
	int i;

	/*
	 *	Register the block device interfaces
	 */

	if (register_blkdev(MAJOR_NR, "i2o_block", &i2ob_fops)) {
		printk("Unable to get major number %d for i2o_block\n",
		       MAJOR_NR);
		return -EIO;
	}
#ifdef MODULE
	printk("i2o_block: registered device at major %d\n", MAJOR_NR);
#endif

	/*
	 *	Now fill in the boiler plate
	 */
	 
	blksize_size[MAJOR_NR] = i2ob_blksizes;
	blk_size[MAJOR_NR] = i2ob_sizes;
	blk_dev[MAJOR_NR].request_fn = do_i2ob_request;
	for (i = 0; i < MAX_I2OB; i++) {
		i2ob_dev[i].refcnt = 0;
		i2ob_dev[i].flags = 0;
		i2ob_dev[i].queue_lock = SPIN_LOCK_UNLOCKED;
		i2ob_dev[i].controller = NULL;
		i2ob_dev[i].tid = 0;
		i2ob_dev[i].head = NULL;
		i2ob_dev[i].tail = NULL;
		i2ob_blksizes[i] = 1024;
		i2ob_blksize_bits[i] = 10;
		i2ob_bytesizes[i] = 0x7ffffc00; /* 2GB */
		i2ob_sizes[i] = i2ob_bytesizes[i] >> i2ob_blksize_bits[i];
	}
	
	/*
	 *	Register the OSM handler as we will need this to probe for
	 *	drives, geometry and other goodies.
	 */

	if(i2o_install_handler(&i2o_block_handler)<0)
	{
		unregister_blkdev(MAJOR_NR, "i2o_block");
		printk(KERN_ERR "i2o_block: unable to register OSM.\n");
		return -EINVAL;
	}
	i2ob_context = i2o_block_handler.context;	 
	/*
	 *	Finally see what is actually plugged in to our controllers
	 */

	i2ob_probe();
		 
	return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
	/*
	 *	Flush the OSM
	 */

	i2o_remove_handler(&i2o_block_handler);
		 
	/*
	 *	Return the block device
	 */
	if (unregister_blkdev(MAJOR_NR, "i2o_block") != 0)
		printk("i2o_block: cleanup_module failed\n");
	else
		printk("i2o_block: module cleaned up.\n");
}
#endif
