/*
 *	Example I2O driver. This is primarily meant to be a proof of
 *	concept and demo of functionality. It doesn't do all the right internal
 *	locking for its own needs. It is good enough to use netscape to play
 *	with the card configuration however.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/i2o.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/malloc.h>
#include <linux/miscdevice.h>
#include <linux/kernel.h>

#include <asm/uaccess.h>
#include <asm/io.h>

static int i2o_cfg_context = -1;
static void *page_buf;
static void *i2o_buffer;
static int i2o_ready;
static int i2o_pagelen;
static int i2o_error;
static int cfg_inuse;
static int i2o_eof;
struct wait_queue *i2o_wait_queue;

/*
 *	This is the callback for any message we have posted. The message itself
 *	will be returned to the message pool when we return from the IRQ
 *
 *	This runs in irq context so be short and sweet.
 */
 
static void i2o_cfg_reply(struct i2o_handler *h, struct i2o_controller *c, struct i2o_message *m)
{
//	u32 *mp=(u32 *)m;
	/* Error check .. */

	printk("i2o_cfg:reply received.\n");	
	/* Wake up */
	i2o_ready=1;
	i2o_eof=0;
	i2o_pagelen=strnlen(page_buf, 4096);
	printk("%d byte page.\n", i2o_pagelen);
	wake_up_interruptible(&i2o_wait_queue);
	printk("freeing buffer.\n");
	if(i2o_buffer)
	{
		kfree(i2o_buffer);
		i2o_buffer=NULL;
	}
	printk("done\n");
}

/*
 *	Each of these describes an i2o message handler. They are
 *	multiplexed by the i2o_core code
 */
 
struct i2o_handler cfg_handler=
{
	i2o_cfg_reply,
	"Configuration",
	0
};

static long long cfg_llseek(struct file *file, long long offset, int origin)
{
	return -ESPIPE;
}

/* i2ocontroller/i2odevice/page/?data */

static ssize_t cfg_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	char *ptr;
	char *tail;
	char *work;
	
	u32   *msg;
	u32   controller, tid, page;
	u32   m;
	
	struct i2o_controller *c;

	if(count > 4096)
		return -EMSGSIZE;
	if((ptr=kmalloc(count, GFP_KERNEL))==NULL)
		return -ENOBUFS;
	if(copy_from_user(ptr, buf, count))
	{
		kfree(ptr);
		return -EFAULT;
	}
	/*
	 *	Ok we have it in kernel space now we can play with it
	 */
	tail=strchr(buf, '?');
	if(tail)
		*tail++=0;
		
	controller=simple_strtoul(ptr, &work, 10);
	if(!work || *work!='/')
		goto bad;
	tid=simple_strtoul(work+1, &work, 10);
	if(!work || *work!='/')
		goto bad;
	page=simple_strtoul(work+1, &work, 10);
	if(!work || *work!='/')
		goto bad;
	work++;
	
	printk("Parsed.\n");
	
	/*
	 *	You'd normally do this when binding a driver not each run.
	 *	The find function also locks the controller into memory. Nobody
	 *	can delete it during use then.
	 */
	 
	c=i2o_find_controller(controller);
	if(c==NULL)
	{
		kfree(ptr);
		return -ENXIO;
	}
	
	printk("Controller\n");
	
	/*
	 *	Message ?
	 */

	do
	{	
		m= *c->post_port;
		if(m!=0xFFFFFFFF)
			break;
		schedule_timeout(HZ/10);
	}
	while(1);
	
	/*
	 *	Maybe I should make a macro of this ?
	 */
	 
	msg = (u32 *)(c->mem_offset+m);

	printk("Make MSG\n");
	
	/*
	 *	Fill in the buffer
	 */
	 
	msg[1] = (I2O_CMD_UTIL_CONFIG_DIALOG << 24)|HOST_TID<<12|tid;
	msg[2] = i2o_cfg_context;
	msg[3] = 0;
	msg[4] = page;
	msg[5] = 0xD0000000|4096;
	msg[6] = virt_to_bus(page_buf);
	if(tail==NULL)	/* Check for post data */
		msg[0] = SEVEN_WORD_MSG_SIZE|SGL_OFFSET_5;
	else
	{
		msg[0] = NINE_WORD_MSG_SIZE|SGL_OFFSET_5;
		msg[5] = 0x50000000|4096;
		msg[7] = 0xD4000000|strlen(tail);
		msg[8] = virt_to_bus(tail);
	}

	printk("Post\n");
	
	/*
	 *	Send it
	 */
	
	i2o_buffer = ptr;
	*c->post_port = m;
	i2o_unlock_controller(c);
	printk("Done\n");
	return count;
bad:
	kfree(ptr);
	return -EINVAL;
}

static ssize_t cfg_read(struct file *file, char *buf, size_t count, loff_t *ptr)
{
	unsigned long flags;
	int len;
	
	if(i2o_eof == i2o_pagelen)
	{
		i2o_eof++;
		return 0;
	}
	
	if(i2o_eof > i2o_pagelen)
		i2o_ready = 0;

	save_flags(flags);
	cli();
	while(!i2o_ready)
	{
		interruptible_sleep_on(&i2o_wait_queue);
		if(signal_pending(current))
		{
			restore_flags(flags);
			return -EINTR;
		}
	}
	if(i2o_error)
	{
		int err=i2o_error;
		i2o_error=0;
		i2o_eof = i2o_pagelen;
		i2o_ready = 0;
		restore_flags(flags);
		return err;
	}
	
	restore_flags(flags);
	
	len=i2o_pagelen-i2o_eof;
	if(len>count)
		len=count;
	if(copy_to_user(buf, page_buf+i2o_eof, len))
		return -EFAULT;
	i2o_eof+=len;
	return len;
}

static int cfg_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	unsigned long arg)
{
	return -EINVAL;
}

static int cfg_open(struct inode *inode, struct file *file)
{
	if(cfg_inuse)
		return -EBUSY;
	MOD_INC_USE_COUNT;
	cfg_inuse=1;
	return 0;
}

static int cfg_release(struct inode *inode, struct file *file)
{
	cfg_inuse = 0;
	MOD_DEC_USE_COUNT;
	return 0;
}


static struct file_operations config_fops =
{
	cfg_llseek,
	cfg_read,
	cfg_write,
	NULL,
	NULL /*cfg_poll*/,
	cfg_ioctl,
	NULL,		/* No mmap */
	cfg_open,
	NULL,		/* No flush */
	cfg_release
};

static struct miscdevice i2o_miscdev = {
	I2O_MINOR,
	"i2octl",
	&config_fops
};	

int init_module(void)
{
	printk(KERN_INFO "i2o configuration manager v 0.02\n");
	
	if((page_buf = kmalloc(4096, GFP_KERNEL))==NULL)
	{
		printk(KERN_ERR "i2o_config: no memory for page buffer.\n");
		return -ENOBUFS;
	}
	if(misc_register(&i2o_miscdev)==-1)
	{
		printk(KERN_ERR "i2o_config: can't register device.\n");
		kfree(page_buf);
		return -EBUSY;
	}
	/*
	 *	Install our handler
	 */
	if(i2o_install_handler(&cfg_handler)<0)
	{
		kfree(page_buf);
		printk(KERN_ERR "i2o_config: handler register failed.\n");
		misc_deregister(&i2o_miscdev);
		return -EBUSY;
	}
	/*
	 *	The low 16bits of the transaction context must match this
	 *	for everything we post. Otherwise someone else gets our mail
	 */
	i2o_cfg_context = cfg_handler.context;
	return 0;
}

void cleanup_module(void)
{
	misc_deregister(&i2o_miscdev);
	
	if(page_buf)
		kfree(page_buf);
	if(i2o_cfg_context != -1)
		i2o_remove_handler(&cfg_handler);
	if(i2o_buffer)
		kfree(i2o_buffer);
}
