#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <signal.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "libcsv.h"

#include "parcel.h"

static struct csv_table zones[1000];
static struct csv_table usa_surcharges;
static int zone_base[1000];

int debug=0;

int is_us_ups_zone(struct csv_table *t)
{
	/* Sanity checks */
	if(strcmp(CSV_ITEM(t,0,0), "ZONE CHART")!=0)
		return -1;
	if(strcmp(CSV_ITEM(t,6,0), "ZONES")!=0)
		return -1;
	return 0;
}

int us_ups_zone_range(struct csv_table *t, int *start, int *end)
{
	int r=sscanf(CSV_ITEM(t,3,0), "For shipments originating in ZIP Codes %d-01 to %d-99.  To determine zone", start,end);
	if(r!=2)
		return -1;
	return 0;
}

int us_ups_parcel_types(struct csv_table *t)
{
	int i;
	for(i=1;i<CSV_WIDTH(t,7);i++)
		sock_printf("%d:UPS:%s\n", i-1, CSV_ITEM(t,7,i));
	return 0;
}

/*
 *	Parse number of number-number ZIP ranges and give back a low/high
 *	pair for them (low==high for a single number)
 */
 
static int us_ups_zone_get(char *p, int *l, int *h)
{
	int n=sscanf(p, "%d-%d", l, h);
	
	if(n==0)
		n=sscanf(p,"%d", l);
	if(n==0)
		return -1;	/* Not valid !! */
	if(n==1)
		*h=*l;
	return 0;
}

/*
 *	Given source/dest zones find the table row to use. We don't want
 *	to find the item as we may need to sweep the entire row for 
 *	service types.
 */

int us_ups_zone_find(char *src_zip, char *dst_zip, struct csv_table **t)
{
	char buf[4];
	int sz,dz;
	int i;
	int l,h;
	
	memcpy(buf,src_zip,3);
	buf[3]=0;
	if(sscanf(buf,"%d", &sz)==-1)
		return -1;
	memcpy(buf, dst_zip,3);
	if(sscanf(buf,"%d", &dz)==-1)
		return -1;
	if(sz<0||sz>999||dz<0||dz>999)
		return -1;
		
	if(zones[sz].height==0)
		return -2;	/* No service.. */
	*t = &zones[sz];
	
	for(i=zone_base[sz];i<zones[sz].height;i++)
	{
		if(us_ups_zone_get(CSV_ITEM(&zones[sz],i,0), &l,&h)==-1)
			continue;
		if(dz>=l && dz<=h)
			return i;
	}
	return -2;	/* No service */
}		
		
	

void install_us_ups_zone(int zone, struct csv_table *t)
{
	int i=0;
	
	if(zones[zone].height)
		fprintf(stderr,"Two tables cover ZIP %d.\n", zone);
	else
		zones[zone]=*t;
		
	for(i=0;i<t->height;i++)
	{
		if(isdigit(*(CSV_ITEM(t,i,0))))
		{
			zone_base[zone]=i;
			return;
		}
	}
	fprintf(stderr, "Zone %d is corrupt.\n", zone);
}

static char *us_ups_zone_hawaii(struct csv_table *t, char *src, char *dst, int j)
{
	int r;
	int tz;
	int tr;
	int c;
	char buf[6];
	
	if(j!=6 && j!=3)
	{
		sock_printf("W: Invalid Hawaii Delivery Type %d\n", j);
		exit(1);
	}
		
	memcpy(buf, dst, 5);
	buf[6]=0;
	
	sscanf(buf, "%d", &tz);
	
	for(r=0;r<t->height-1;r++)
	{
		if(strncmp(CSV_ITEM(t, r, 0), "[2] For Hawa", 12)==0)
			break;
	}
	if(r==t->height-1)
	{
		sock_printf("W No Hawaii data for zone %d from %s\n", tz, src);
		return NULL;	/* No exception table ?? */
	}
	r++;
	for(c=0;c< CSV_WIDTH(t,r);c++)
	{
		if(!sscanf(CSV_ITEM(t,r,c), "%d", &tr))
			continue;
		if(tr==tz)
			return (j==6?"124":"224");
	}
	r+=2;	/* 126/226 row */
	for(c=0;c< CSV_WIDTH(t,r);c++)
	{
		if(!sscanf(CSV_ITEM(t,r,c), "%d", &tr))
			continue;
		if(tr==tz)
			return (j==6?"126":"226");
	}
	return NULL;	/* Cant do it */
}

static char *us_ups_zone_alaska(struct csv_table *t, char *src, char *dst, int j)
{
	int r;
	int tz;
	int tr;
	int c;
	char buf[6];
	
	if(j!=6 && j!=3)
	{
		sock_printf("W: Invalid Alaska Delivery Type %d\n", j);
		exit(1);
	}
		
	memcpy(buf, dst, 5);
	buf[6]=0;
	
	sscanf(buf, "%d", &tz);
	
	for(r=0;r<t->height-1;r++)
	{
		if(strncmp(CSV_ITEM(t, r, 0), "[3] For Alas", 12)==0)
			break;
	}
	if(r==t->height-1)
	{
		sock_printf("W No Hawaii data for zone %d from %s\n", tz, src);
		return NULL;	/* No exception table ?? */
	}
	r++;
	for(c=0;c< CSV_WIDTH(t,r);c++)
	{
		if(!sscanf(CSV_ITEM(t,r,c), "%d", &tr))
			continue;
		if(tr==tz)
			return (j==6?"124":"224");
	}
	r+=2;	/* 126/226 row */
	for(c=0;c< CSV_WIDTH(t,r);c++)
	{
		if(!sscanf(CSV_ITEM(t,r,c), "%d", &tr))
			continue;
		if(tr==tz)
			return (j==6?"126":"226");
	}
	return NULL;	/* Cant do it */
}

static void ups_set_usa_surcharge(struct csv_table *t)
{
	usa_surcharges = *t;
}

static int ups_check_usa_surcharge(char *code)
{
	int i,j;
	for(i=0; i<usa_surcharges.height;i++)
	{
		char *p = CSV_ITEM(&usa_surcharges, i, 0);
		if(p == NULL  || *p <'0' || *p >'9')
			continue;
		for(j=0;j<CSV_WIDTH(&usa_surcharges,i); j++)
		{
			p=CSV_ITEM(&usa_surcharges, i, j);
			if(memcmp(p, code, 5)==0)
				return 1;
		}
	}
	return 0;
}

void ups_init(void)
{
	DIR *d;
	struct dirent *dp;	
	struct csv_table t;
	int j;
	
	int l,h;
	
	d=opendir("USZone");
	if(d==NULL)
		perror("USZone/*");
	
	while((dp=readdir(d))!=NULL)
	{
		FILE *f;
		char path[4096];
		
		/*
		 *	Only loads the CSV files.
		 */
		 
		if(strcmp(dp->d_name+strlen(dp->d_name)-4, ".csv"))
			continue;

		/*
		 *	Where is the file actually located
		 */
		snprintf(path, sizeof(path), "USZone/%s", dp->d_name);		
		
		f=fopen(path,"r");
		
		if(f==NULL)
		{
			perror(path);
			exit(1);
		}
		
		/*
		 *	Load it
		 */
		if(csv_load_table(f, &t)==-1)
		{
			fprintf(stderr,"%s: failed at line %d: %s.\n",
				path, csv_line, csv_error);
			exit(1);
		}
		fclose(f);
		
		if(strcmp(dp->d_name, "xarea.csv")==0)
		{
			/* UPS now punish peoplee with certain zipcodes by
			   adding on a dollar */
			ups_set_usa_surcharge(&t);
		}
		else
		{
		
			/*
			 *	Sanity test
			 */
			 
			if(is_us_ups_zone(&t)==-1)
			{
				fprintf(stderr,"%s: not a valid UPS US zone file.\n",
					path);
				exit(1);
			}
	
			/*
			 *	Find the range it covers
			 */		
			 
			if(us_ups_zone_range(&t, &l, &h)==-1)
			{
	 			fprintf(stderr, "%s: unable to parse ZIP range.\n",
					path);
			}
	
			/*
			 *	Install the table
			 */
		 
			for(j=l;j<=h;j++)
				install_us_ups_zone(j,&t);
		}
	}
	ups_load_rates();		/* US Rates */
	ups_world_load();		/* World Rates */
	ups_country_load();		/* Country Codes */
	ups_load_canada();		/* Canada */
	ups_world_eas_load();		/* Oww.. */
}

/*
 *	USPS - rather saner than UPS
 */

/*
 *	In case we decide to make USPS rates loadable
 */ 
static void usps_init(void)
{
}

static char *us_usps_price(char * weight)
{
	float f;
	
	sscanf(weight,"%f", &f);
	
	if(f<=2)
		return "3.00";
	if(f<=3)
		return "4.00";
	if(f<=4)
		return "5.00";
	if(f<=5)
		return "6.00";
	return NULL;
}

static void us_usps_parcel_types(void)
{
	printf("8:USPS:USPS Priority Mail.\n");
}


static char *clean_fgets(char *ptr, int len, char *what)
{
	char *x;
	int n;
	
	if(debug)
	{
		printf("%s> ",what);
		fflush(stdout);
		x=fgets(ptr,len,stdin);
		if(x==NULL)
			return x;
	}
	else
	{
		if(sock_gets(ptr,len)==-1)
			return NULL;
	}		
	n=strlen(ptr);
	if(ptr[n-1]=='\n')
	{
		ptr[n-1]=0;
		n--;
	}
	if(ptr[n-1]=='\r')
		ptr[n-1]=0;
//	printf("%s='%s'\n", what, ptr);
	return ptr;
}


void process_task(void)
{
	char dcountry[256];
	char dcity[256];
	char dstate[256];
	char scounty[256];
	char sstate[256];
	char src[256],dst[256];
	char weight[256];
	char types[32];
	unsigned int typemask;
	int i,j;
	struct csv_table *tptr;
	char *x;
	
	/*
	 *	Read problem data
	 */
	 
	if(clean_fgets(scounty,256, "source county")==NULL)
		exit(1);
	if(clean_fgets(sstate,256, "source state")==NULL)
		exit(1);
	if(clean_fgets(src,256, "source zip")==NULL)
		exit(1);
	if(clean_fgets(dcity,256, "dest city")==NULL)
		exit(1);
	if(clean_fgets(dstate,256,"dest state")==NULL)
		exit(1);
	if(clean_fgets(dcountry,256, "dest country")==NULL)
		exit(1);
	if(clean_fgets(dst,256, "dest zip")==NULL)
		exit(1);
	if(clean_fgets(weight,256, "weight")==NULL)
		exit(1);
	if(clean_fgets(types,32,"types")==NULL)
		exit(1);
	sscanf(types, "%X", &typemask);
	
	
	if(strcasecmp(dcountry,"USA")==0)
		typemask&=0x1FF;	/* USPS, UPS USA */
	else if(strcasecmp(dcountry, "Canada")==0)
		typemask&=0xFF0000;	/* Canada has its own UPS rules, no USPS */
	else
		typemask&=0xF000;	/* UPS World, US/Canada rates don't apply */
		
	/*			
	 *	Find the item in UPS USA if UPS USA selected
	 */
	 
	if(typemask&0xFF)
	{
		i=us_ups_zone_find(src,dst, &tptr);
		if(i==-1)
		{
			sock_printf("F: Format error.\n");
			return;
		}
		if(i!=-2)
		{
			/*
			 *	Scan wanted UPS entries 
			 */
		
			for(j=1;j<CSV_WIDTH(tptr,i);j++)
			{
				char *x=CSV_ITEM(tptr,i,j);
				if(strcmp(x,"-")==0)
					continue;
				if(!(typemask&(1<<(j-1))))
					continue;
				
				if(*x=='[')
				{
					switch(x[1])
					{
						case '1':
							sock_printf("F: Unsupported international delivery.\n");
							break;
						case '2':
					 		x=us_ups_zone_hawaii(tptr, src, dst, j);
					 		break;
					 	case '3':	/* Alaska specials 1 */
					 		x=us_ups_zone_alaska(tptr, src, dst, j);
					 		break;
					 	default:
					 		sock_printf("W: Unknown special %s.\n", x);
					 		x=NULL;
					 		break;
					}
				}
				if(x!=NULL)			
				{
					char *price=us_ups_zone_to_price(x, j,  weight, ups_check_usa_surcharge(dst));
					if(price)
					{
						/* UPS are inconsistent with their use of $ */
						if(*price=='$')
							price++;
							
						/* Print out the good entry */
						sock_printf("G: S%d Z%s P%s\n",
							j-1, x, price);
					}
				}
			}
		}
	}			
	/*
	 *	Now try USPS 
	 */
		 
	if(typemask & (1<<8))
	{
		x=us_usps_price(weight);
		if(x!=NULL)
			sock_printf("G: S8 Z0 P%s\n", x);
	}
	
	/*
	 *	Check for UPS World
	 */
	 
	if(typemask & 0xF000)
	{
		struct csv_table *ct;
		int sc;
		int t;
		int line;
		
		
		if((line=ups_country_zone(&ct, &sc, scounty, sstate, dcountry))==-1)
			return;
			
		for(t=0;t<2;t++)
		{
			if(typemask&(1<<(12+t)))
			{
				char *p=ups_world_pricing(ct, sc, line, t, weight, dstate, dcountry, dst);
				if(p==NULL)
					continue;
				if(*p=='$') 
					p++;
				sock_printf("G: S%d Z- P%s\n",
					12+t, p);
			}
		}
	}
	
	/*
	 *	Canada
	 */
	if(typemask&0xFF0000)
	{
		float w;
		sscanf(weight,"%f", &w);
		
		for(i=0;i<3;i++)
		{
			char *p, *z;
			int ca_eas;
			
			z=get_zone_ca(sstate, dst, i, &ca_eas);
			if(z==NULL)
				continue;
			p=canada_price_for(i, w, z);
			if(p==NULL)
				continue;
			if(ca_eas)
				p=canada_eas(w, p);
			if(*p=='$')
				p++;
			sock_printf("G: S%d Z%s P%s\n",i+16, z, p);
		}
	}
}	


static int forking_daemon(void)
{
	struct sockaddr_in sin;
	int sock_fd,accept_fd;
	int one=1;
	
	sin.sin_family=AF_INET;
	sin.sin_addr.s_addr=INADDR_ANY;
	sin.sin_port=htons(PORT);
	
	sock_fd=socket(PF_INET, SOCK_STREAM, 0);
	if(sock_fd==-1)
	{
		perror("socket");
		exit(1);
	}
	
	setsockopt(sock_fd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));
	
	if(bind(sock_fd, (struct sockaddr *)&sin, sizeof(sin))==-1)
	{
		perror("bind");
		exit(1);
	}
	
	signal(SIGCLD, SIG_IGN);
	
	listen(sock_fd,5);
		
	while(1)
	{
		accept_fd=accept(sock_fd, NULL, 0);
		if(accept_fd==-1)
			continue;
			
		switch(fork())
		{
			case -1:
				perror("fork");
				close(accept_fd);
				continue;
			case 0:
				set_socket_fd(accept_fd);
				process_task();
				fprintf(stderr, "completed.\n");
				close(accept_fd);
				_exit(0);
			default:
				close(accept_fd);
		}
	}				
}

int main(int argc, char *argv[])
{
	ups_init();
	usps_init();
	
	if(argv[1] && strcmp(argv[1],"-s")==0)
	{
		us_ups_parcel_types(&zones[100]);
		us_usps_parcel_types();
		world_ups_parcel_types();
		canada_ups_parcel_types();
		exit(0);
	}
	
	if(argv[1] && strcmp(argv[1], "-d")==0)
		debug=1;
	
	if(debug)
	{
		while(1)
			process_task();
	}
	else
	{
		forking_daemon();
	}		
	return 0;
}
