#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <signal.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <ftw.h>
#include <math.h>

#include "libcsv.h"

#include "parcel.h"

static struct csv_table states[52];
static struct csv_table canada_rates[3];
static char *state_name[52];
static struct csv_table canada_xpd;
static int next_state=0;

/*
 *	Everything you never wanted to know about Canada
 */
 
int compare_postcodes(const char *cp1, const char *cp2)
{
	char cx1[3];
	char cx2[3];
	int i;
	
	memcpy(cx1,cp1,3);
	memcpy(cx2,cp2,3);
	
	for(i=0;i<3;i++)
	{
		cx1[i]=toupper(cx1[i]);
		cx2[i]=toupper(cx2[i]);
	};
	
	return memcmp(cx1, cx2, 3);
}

/*
 *	Find the state 
 */
 
static int which_state_to_ca(const char *name)
{
	int i;
	
	for(i=0;i<next_state;i++)
	{
		if(strcasecmp(name, state_name[i])==0)
			return i;
	}
	
	return -1;
}

/*
 *	Do the row lookup for Canadian zones.
 */
 
char *get_zone_ca(const char *state_in, const char *pcode, int type, int *eas)
{
	int i;
	int t;
	struct csv_table *tr;

	/*
	 *	Pick the right chart
	 */
	 
	if(type==2)
	{
		t=which_state_to_ca(state_in);
		if(t==-1)
		{
			sock_printf("F: unknown state '%s'.\n", state_in);
			return NULL;
		}		
		tr=&states[t];
		i=6;
	}
	else
	{
		tr=&canada_xpd;
		i=9;
	}
		
	
	for(;i<tr->height; i++)
	{
		char *lz=CSV_ITEM(tr, i, 0);
		char *hz=CSV_ITEM(tr, i, 1);
		
		if(compare_postcodes(pcode, lz)<0)
			continue;
		if(compare_postcodes(pcode, hz)>0)
			continue;
		
		/*
		 *	Surcharge ?
		 */
		 
		*eas=0;
		if(type<2 && *CSV_ITEM(tr, i, 4)=='Y')
			*eas=1;
			
		/*
		 *	Zone info
		 */
		 
		switch(type)
		{
			case 2:
				return CSV_ITEM(tr,i ,2);
			case 0:
				return CSV_ITEM(tr, i, 3);
			case 1:	
				return CSV_ITEM(tr, i, 2);
			default:
				sock_printf("F: bad canada type.\n");
				return NULL;
		}
	}
	return NULL;
}


/*
 *	Load canadian zones
 */
 
static int get_canada_zone(const char *name, const struct stat *sb, int flag)
{
	FILE *f;
	
	if(flag!=FTW_F)
		return 0;
	f=fopen(name,"r");
	if(f==NULL)
	{
		perror(name);
		return 0;
	}


	if(strcmp(name, "CAZone/canww.csv")==0)
	{
		if(csv_load_table(f,&canada_xpd)==-1)
		{
			/*
			 *	No XPD = > boom
			 */
			fprintf(stderr,"csv format error in '%s'.\n", name);
			exit(1);
		}
	}
	else
	{	
		char *sn;
		/*
		 *	States
		 */
		if(csv_load_table(f, &states[next_state])==-1)
		{
			next_state;
			fprintf(stderr, "csv format error in '%s'.\n", name);
		}
		
		sn=CSV_ITEM(&states[next_state], 0, 0);
		if(strncmp(sn, "Canada Standard Zone Charts from ",
			strlen("Canada Standard Zone Charts from "))!=0)
		{
			fprintf(stderr, "'%s' doesn't follow the format I know.\n", name);
			exit(1);
		}
		sn+=strlen("Canada Standard Zone Charts from ");
		/*
		 *	Columbia...
		 */
		if(strncmp(sn, "District of", 11)==0)
			sn+=11;
		state_name[next_state]=strdup(sn);
		next_state++;
	}
	fclose(f);
	return 0;
}

/*
 *	Get object pricing
 */


char *canada_price_for(int service, float weight, char *zone)
{
	struct csv_table *t=&canada_rates[service];
	char *entry;
	
	int i=0;
	int j;
	
	for(i=0; i<t->height; i++)
	{
		if(strcasecmp(CSV_ITEM(t, i, 0), "Weight")==0)
			break;
	}
	
	if(i == t->height)
	{
		sock_printf("F: Canadian rate table %d has no 'weight' entry.\n", service);
		return NULL;
	}
	
	for(j=1;j< CSV_WIDTH(t,i);j++)
	{
		if(strcmp(CSV_ITEM(t,i,j)+5, zone)==0)
			break;
	}
	if(j==CSV_WIDTH(t,i))
	{
		sock_printf("F: Zone '%s' is not in service table %d for Canada.\n", 
			zone, service);
		return NULL;
	}
	
	/*
	 *	Now hunt the weight
	 */
	
	for(i++;i<t->height;i++)
	{
		int w;
		char *x=CSV_ITEM(t,i,0);
		
		/* KG boxen not supported */
		if(x[2]=='K' && x[3]=='G')
			continue;
			
		if(sscanf(x,"%d", &w) <1)
		{
			if(strcasecmp(x,"Letter")==0)
				continue;
			break;
		}
		
		if(weight <= w)
			return CSV_ITEM(t,i,j);
	}
	/*
	 *	150+Kg. Ok the lovely people at UPS are out to annoy us
	 *	the format varies -by service-. Its bad enough as it is,
	 *	UPS pricing was obviously done by an out of work tax form
	 *	designer.
	 */

	/* Standard uses the "rate per pound with a minimum system and
	   ranges as does xpd, but xpr uses a different setup */
	   
//	printf("big weight = %f i=%d\n", weight, i);
	   
	if(service==1)
	{
		float ratio;
		float minimum;
		char *x;
		static char buf[32];
		
		i++;	/* Now on multipliers */
		x=CSV_ITEM(t,i,j);
		if(*x=='$')
			x++;
		if(sscanf(x, "%f", &ratio)!=1)
		{
			sock_printf("F: Canada ratio read error.\n");
			return NULL;
		}
		i++;	/* Now on min */
		x=CSV_ITEM(t,i,j);
		if(*x=='$')
			x++;
		if(sscanf(x, "%f", &minimum)!=1)
		{
			sock_printf("F: Canada minimum read error.\n");
			return NULL;
		}
		
		printf("R %f W %f M %f\n", ratio, weight, minimum);
		ratio *= ceil((double)weight);	/* Yes they screw you on fractions */
		if(ratio < minimum)
			ratio = minimum; /* and screw you again if you havent
					    been screwed hard enough yet */
		
		sprintf(buf, "$%d.%d",
			(int)ratio,
			((int)(ratio*100))%100);
		return buf;
	}
	/*
	 *	Ok we are into multi step traces. Groan
	 */

	for(++i;i<t->height;i+=2)
	{
		float wl, wh;
		int n;
		char *p=CSV_ITEM(t,i,0);
				 
		/*
		 *	Get the weight range
		 */
		n=sscanf(p,"%f-%f", &wl, &wh);
		
		if(n==0)
			break;
			
		if(n==1)
			wh=99999999999.99;
		/*
		 *	Found a weight category
		 */
		 
		if((int)weight+1>=wl && (int)weight <= wh)
		{
			static char buf[32];
			/* Price per lb */
			p=CSV_ITEM(t,i,j);
			if(sscanf(p, "$%f", &wl)<=0)
			{
				sock_printf("F: Format error in ranged data.\n");
				return NULL;
			}
			wl*=ceil((double)weight);
			/* Check versus minimum price */
			p=CSV_ITEM(t,i+1,j);
			/* They used commas in some of these which rather
			   upsets sscanf .. sigh */
			if(sscanf(p,"$%f", &wh)<=0)
			{
				sock_printf("F: Format error in ranged data minimum.\n");
				return NULL;
			}
			if(wh<wl)
				wl=wh;
			sprintf(buf,"$%d.%02d",
				(int)wl, ((int)(wl*100))%100);
			return buf;
		}
	}
	return NULL;
}
	
	
char *canada_eas(float weight, char *price)
{
	float p;
	static char buf[64];
	
	if(*price=='$')
		price++;
		
	sscanf(uncomma(price),"%f", &p);

	weight*=0.20;
	if(weight<20.00)
		weight=20.00;
	p+=weight;
	sprintf(buf, "%d.%02d",
		(int)p, ((int)(p*100))%100);
	return buf;
}

/*
 *	Load US->Canada Zone Tables
 */
 
void load_canada_zones(void)
{
	printf("Canada.\n");
	ftw("CAZone", get_canada_zone, 1);
}

/*
 *	Load US->Canada Rates
 */
 
void load_canada_rates(void)
{
	int i;
	FILE *f;
	
	/*
	 *	Lets hope UPS dont change these
	 */
	 
	static char *rate_names[]={
		"CARates/canxpd.csv",
		"CARates/can-xpr.csv",
		"CARates/canstnd.csv"
	};
	
	for(i=0;i<3;i++)
	{
		f=fopen(rate_names[i],"r");
		if(f==NULL)
		{
			perror(rate_names[i]);
			exit(1);
		}
		if(csv_load_table(f, &canada_rates[i])==-1)
		{
			fprintf(stderr, "%s is not a valid csv file.\n",
				rate_names[i]);
		}
		fclose(f);
	}
}	

void ups_load_canada(void)
{
	load_canada_zones();
	load_canada_rates();
}


void canada_ups_parcel_types(void)
{
	printf("16:UPS:Canada Expedited\n");
	printf("17:UPS:Canada Express\n");
	printf("18:UPS:Canada Standard\n");
}
	