#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/pci.h>

#include <asm/io.h>


#ifndef PCI_VENDOR_ESS
#define PCI_VENDOR_ESS			0x125D
#define PCI_DEVICE_ID_ESS_ESS19XX	0x1969
#define PCI_DEVICE_ID_ESS_ESS1968	0x1968
#endif

/*
 *	Detect an ESS19xx card
 */

static int ess19xx_install(struct pci_dev *pcidev)
{
	int iobase;
	int sbbase;
	int mpubase;
	int gamebase;
	u8 x;
	u16 w;
	u32 v;
	int i;
	
	iobase = pcidev->base_address[0] & PCI_BASE_ADDRESS_IO_MASK;
	sbbase = pcidev->base_address[1] & PCI_BASE_ADDRESS_IO_MASK;
	mpubase = pcidev->base_address[2] & PCI_BASE_ADDRESS_IO_MASK;
	gamebase = pcidev->base_address[3] & PCI_BASE_ADDRESS_IO_MASK;
	
	/*
	 *	Reset the board
	 */
	 
	x=inb(sbbase+6)&~1;
	outb(x|1, sbbase+6);
	inb(sbbase+6);
	outb(x,sbbase);
	
	/*
	 *	Wait for it to come up
	 */
	
	for(i=0; i<5000;i++)
	{
		udelay(1);
		if(!(inb(sbbase+0xE)&0x80))
			continue;
		if(inb(sbbase+0xA)!=0xAA)
			continue;
		break;
	}
	
	/*
	 *	Ok card ready. Begin setup proper
	 */
	
	/*
	 *	Use TDMA for now
	 */ 
	 
	pci_read_config_dword(pcidev, 0x50, &v);
	/* ISA IRQ emulation off */
	v&=~(1<<14|1<<13);

	/* Clear DMA bits */
	v&=~(1<<10|1<<9|1<<8);

	/* TDMA on */
	v|=(1<<8);

	/* Legacy IRQ enable */
	v|=(1<<16);

	/* Legacy IRQ setup */
	v&=~(1<<13|1<<14);
	if(pcidev->irq == 5 || pcidev->irq == 7 ||
		pcidev->irq == 9 || pcidev->irq == 10)
		v|=(1<<13);
	else
		v|=(1<<14);

	/*
	 *	MPU at 330
	 */
	
	v&=~((1<<4)|(1<<3));
	
	/*
	 *	SB at 0x220
	 */

	v&=~(1<<2);
	
	/*
	 *	Reserved write as 0
	 */
	 
	v&=~(1<<1);
	pci_write_config_dword(pcidev, 0x50, v);

	/*
	 *	Legacy mode
	 */

	pci_read_config_word(pcidev, 0x40, &w);
	w&=~(1<<15);	/* legacy decode on */
	w|=(1<<14);	/* Reserved write as 1 in this case */
	w&=~(1<<13|1<<12|1<<1);
	w|=(1<<11);		/* MPU IRQ 7 */
	w&=~(1<<10|1<<9|1<<8);	/* SB IRQ 5 */
	w|=(1<<3)|(1<<1)|(1<<0);	/* SB on , FM on, MPU on */
	/* Default to DMA 1 */
	pci_write_config_word(pcidev, 0x40, w);
	
	printk(KERN_INFO "Programmed ES188x at 0x%X to legacy mode.\n",
		iobase);
	return 1;
}


int init_ess19xx(void)
{
	struct pci_dev *pcidev=NULL;
	int count=0;
		
	if(!pci_present())
		return -ENODEV;
	
		
	while((pcidev = pci_find_device(PCI_VENDOR_ESS, PCI_DEVICE_ID_ESS_ESS19XX, pcidev))!=NULL)
	{
		count+=ess19xx_install(pcidev);
		if(count)
			return 0;
	}
	
	pcidev = NULL;
	while((pcidev = pci_find_device(PCI_VENDOR_ESS, PCI_DEVICE_ID_ESS_ESS1968, pcidev))!=NULL)
	{
		count+=ess19xx_install(pcidev);
		if(count)
			return 0;
	}
	if(count==0)
		return -ENODEV;
	return 0;
}


int init_module(void)
{
	if(init_ess19xx()<0)
	{
		printk(KERN_ERR "No ES19xx cards found.\n");
		return -ENODEV;
	}
	return 0;
}

void cleanup_module(void)
{
}

