/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * From: @(#)ring.c	5.2 (Berkeley) 3/1/91
 */
char ring_rcsid[] =
  "$Id: ring.c,v 1.30 2002/08/11 21:10:10 dholland Exp $";

/*
 * This defines a structure for a ring buffer. 
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <assert.h>
#include "ring.h"

////////////////////////////////////////////////////////////
//
// ringbuf

struct ringbuf {
    const struct datasink_c *binding;
    const struct datasource_c *srcbinding;

    char *buf;
    int size;   /* total size of buffer */
    int head;   /* next input character goes here */
    int tail;   /* next output character comes from here */
    int count;  /* chars presently stored in buffer */
    // The buffer is empty when head==tail.

    int marked;   /* this character is marked */
};

////////////////////////////////////////////////////////////

/* null data destination: throw away data passed in */
static int null_write(const char *buf, int n) {
   (void) buf;
   return n;
}

static const struct datasink_c nullsink_obj = {
   null_write,	/* write */
   null_write,	/* writeurg */
};

const struct datasink_c *const nullsink = &nullsink_obj;

////////////////////////////////////////////////////////////

#if 0
    int gets(char *buf, int max);
    int getch(int *ch);
    void ungetch(int ch);
    int full_count();

    // automatic consume
    int flush();

    /////// supply end

    // manual supply
    void putch(char c) { write(&c, 1); }
    void write(const char *buffer, int ct);
    void printf(const char *format, ...);
    int empty_count() { return size - count; }

    // automatic supply
    int read_source();

    /////// others
    void clear_mark() { marked = -1; }
    void set_mark() { marked = head; }

    int init(int size, const datasink_c *sink, const datasource_c *src);

    const datasink_c *setsink(const datasink_c *nu) {
	const datasink_c *old = binding;
	binding = nu;
	return old;
    }

#endif



/////////////////////////////////////////////////// consume //////////////

int ringbuf_gets(struct ringbuf *b, char *rbuf, int max) {
    int i=0, ch;
    assert(max>0);
    while (ringbuf_getch(b, &ch)>0 && i<max-1) {
	rbuf[i++] = ch;
    }
    rbuf[i]=0;
    return i;
}

int ringbuf_getch(struct ringbuf *b, int *ch) {
    int rv = 0;
    if (b->count > 0) {
	if (b->tail==b->marked) {
	    rv = 2;
	    b->marked = -1;
	}
	else rv = 1;
	*ch = (unsigned char) b->buf[b->tail++];
	if (b->tail>=b->size) b->tail -= b->size;
	b->count--;
    }
    return rv;   /* 0 = no more chars available */
}

void ringbuf_ungetch(struct ringbuf *b, int ch) {
    int x, och;

    x = b->tail;
    x--;
    if (x<0) x += b->size;
    och = b->buf[x];   /* avoid sign-extension and other such problems */
    if ((och&0xff) == (ch&0xff)) {
	b->tail = x;
	b->count++;
    }
    else {
	//assert(!"Bad ungetch");
	b->tail = x;
	b->count++;
    }
}

int ringbuf_full_count(struct ringbuf *rb) {
   return rb->count;
}

/*
 * Return value:
 *   -2: Significant error occurred.
 *   -1: No useful work done, data waiting to go out.
 *    0: No data was waiting, so nothing was done.
 *    1: All waiting data was written out.
 *    n: Some data written, n-1 bytes left.
 */
int ringbuf_flush(struct ringbuf *b) {
    static int busy=0;

    assert(b->binding);
    assert(b->count>=0);
    if (b->count==0) return 0;
    
    if (busy) {
	return -1;
    }
    busy=1;

    /*
     *  should always be true
     * assert(((b->size+b->head-b->tail)%b->size)==b->count);
     *
     *  Nope! The above fails if the buffer is full; then:
     * head == tail (so LHS is 0), but count == size.
     * The following formula should be better. --okir
     */
    assert(((b->head - b->tail - b->count) % b->size) == 0);

    while (b->count > 0) {
        int n, bot, top;

	bot = b->tail;
	top = b->head;
	if (top < bot) top = b->size;
	if (b->marked > bot) top = b->marked;
	assert(top-bot > 0 && top-bot <= b->count);

	if (b->marked==bot) {
	    n = b->binding->writeurg(b->buf+bot, top-bot);
	}
	else {
	    n = b->binding->write(b->buf+bot, top-bot);
	}
	if (n < 0) {
	    busy=0;
	    return -2;
	}
	else if (n==0) {
	    busy=0;
	    return -1;
	}
	
	if (b->marked==bot) b->marked = -1;
	b->tail += n;
	if (b->tail >= b->size) b->tail -= b->size;
	b->count -= n;
	assert(((b->size+b->head-b->tail)%b->size)==b->count);
	
	if (n > 0 && n < top-bot) { busy=0; return n+1; }
	/* otherwise (if we wrote all data) loop */
    }
    assert(((b->size+b->head-b->tail)%b->size)==b->count);
    busy=0;
    return 1;
}


/////////////////////////////////////////////////// supply //////////////

void ringbuf_printf(struct ringbuf *b, const char *format, ...) {
    char xbuf[256];
    int l;
    va_list ap;

    va_start(ap, format);
    l = vsnprintf(xbuf, sizeof(xbuf), format, ap);
    va_end(ap);

    /* note that because xbuf may include nulls we CANNOT use strlen(xbuf) */
    ringbuf_write(b, xbuf, l);
}

void ringbuf_write(struct ringbuf *b, const char *buffer, int ct) {
    int i;

    if (ct > b->size - b->count) {
	// Oops. We're about to overflow our buffer.
	// In practice this shouldn't ever actually happen.
	// We could return a short count, but then we'd have to check
	// and retry every call, which ranges somewhere between painful
	// and impossible.
	// Instead, we drop the data on the floor. This should only happen 
	// if (1) the tty hangs, (2) the network hangs while we're trying 
	// to send large volumes of data, or (3) massive internal logic errors.
	fprintf(stderr, "\n\ntelnet: buffer overflow, losing data, sorry\n");
	ct = b->size - b->count;
    }
    for (i=0; i<ct; i++) {
	b->buf[b->head++] = buffer[i];
	if (b->head>=b->size) b->head -= b->size;
	b->count++;
    }
}

void ringbuf_putch(struct ringbuf *b, char c) {
    ringbuf_write(b, &c, 1);
}

int ringbuf_empty_count(struct ringbuf *b) {
    return b->size - b->count;
}

int ringbuf_read_source(struct ringbuf *b) {
    int bot, top, l;

    bot = b->head;
    top = b->tail-1; /* leave room for an ungetc */
    if (top<0) top += b->size;
    if (top < bot) top = b->size;

    if (top==bot) return 0;

    l = b->srcbinding->read(b->buf+bot, top-bot);
    if (l>=0) {
	b->head += l;
	if (b->head>=b->size) b->head -= b->size;
	b->count += l;
    }
    if (l==0) l = -1;
    return l;
}

/////////////////////////////////////////////////// others //////////////

void ringbuf_clear_mark(struct ringbuf *b) {
    b->marked = -1;
}

void ringbuf_set_mark(struct ringbuf *b) {
    b->marked = b->head;
}

struct ringbuf *ringbuf_create(int sz, const struct datasink_c *sink, 
			       const struct datasource_c *src) {
    struct ringbuf *b;

#ifdef __cplusplus
    b = new ringbuf;
    b->buf = new char[sz];
#else
    b = malloc(sizeof(struct ringbuf));
    if (b==NULL) {
       return NULL;
    }
    b->buf = malloc(sz);
    if (b->buf==NULL) {
       free(b);
       return NULL;
    }
#endif

    b->size = sz;
    b->head = b->tail = 0;
    b->count = 0;
    b->marked = -1;

    b->binding = sink;
    b->srcbinding = src;

    return b;
}

const struct datasink_c *ringbuf_setsink(struct ringbuf *b,
					 const struct datasink_c *nu) {
    const struct datasink_c *old = b->binding;
    b->binding = nu;
    return old;
}
