/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "netlink.h"
#include "proto.h"
#include "ring.h"

char netlink_rcsid[] = 
  "$Id: netlink.c,v 1.24 2002/08/11 22:52:55 dholland Exp $";

/* In Linux, this is an enum */
#if defined(__linux__) || defined(IPPROTO_IP)
#define HAS_IPPROTO_IP
#endif

////////////////////////////////////////////////////////////

static int netsource_read(char *buf, int maxlen) {
   int net = nlink_getfd();
   int l = recv(net, buf, maxlen, 0);
   if (l<0 && errno == EWOULDBLOCK) l = 0;
   return l;
}

static int netsink_write(const char *buf, int len) {
   int r = nlink_send(buf, len, 0);
   if (r==-1 && (errno==ENOBUFS || errno==EWOULDBLOCK)) return 0;
   return r;
}

static int netsink_writeurg(const char *buf, int len) {
   int r, rr;
   /*
    * In 4.2 (and 4.3) systems, there is some question about
    * what byte in a sendOOB operation is the "OOB" data.
    * To make ourselves compatible, we only send ONE byte
    * out of band, the one WE THINK should be OOB (though
    * we really have more the TCP philosophy of urgent data
    * rather than the Unix philosophy of OOB data).
    */
   if (len==0) return 0;

   r = nlink_send(buf, 1, MSG_OOB);
   if (r==-1 && (errno==ENOBUFS || errno==EWOULDBLOCK)) r = 0;
   if (r<=0) return r;

   rr = nlink_send(buf+1, len-r, 0);
   if (rr==-1 && (errno==ENOBUFS || errno==EWOULDBLOCK)) rr = 0;
   if (rr<=0) return r;   /* less than ideal */

   return r+rr;
}

static const struct datasource_c chan = {
   netsource_read,	/* read */
};
const struct datasource_c *const netsrc = &chan;

static const struct datasink_c chan2 = {
   netsink_write,	/* write */
   netsink_writeurg,	/* writeurg */
};
const struct datasink_c *const netsink = &chan2;

////////////////////////////////////////////////////////////

static int netfd = -1;

int nlink_setdebug(int debug) {
    if (netfd > 0 &&
	(setsockopt(netfd, SOL_SOCKET, SO_DEBUG, &debug, sizeof(debug))) < 0) {
	perror("setsockopt (SO_DEBUG)");
    }
    return 1;
}

void nlink_close(int doshutdown) {
    if (doshutdown) {
	shutdown(netfd, 2);
    }
    close(netfd);
    netfd = -1;
}

int nlink_connect(int debug, struct hostent *host, 
		    struct sockaddr_in *sn, 
		    char *srcroute, int srlen, int tos) 
{
    int on=1;

    netfd = socket(AF_INET, SOCK_STREAM, 0);
    if (netfd < 0) {
	perror("telnet: socket");
	return 0;
    }

#if defined(IP_OPTIONS) && defined(HAS_IPPROTO_IP)
    if (srcroute) {
	if (setsockopt(netfd, IPPROTO_IP, IP_OPTIONS, srcroute, srlen) < 0)
	    perror("setsockopt (IP_OPTIONS)");
    }
#endif

#if defined(HAS_IPPROTO_IP) && defined(IP_TOS)
#if defined(HAS_GETTOS)
    struct tosent *tp;
    if (tos < 0 && (tp = gettosbyname("telnet", "tcp")))
	tos = tp->t_tos;
#endif
    if (tos < 0) tos = 020;	/* Low Delay bit */
    if (tos && (setsockopt(netfd, IPPROTO_IP, IP_TOS, &tos, sizeof(int)) < 0)
	&& (errno != ENOPROTOOPT))
	perror("telnet: setsockopt (IP_TOS) (ignored)");
#endif	/* defined(IPPROTO_IP) && defined(IP_TOS) */

    if (debug && setsockopt(netfd, SOL_SOCKET, SO_DEBUG, &on, sizeof(on)) < 0){
	perror("setsockopt (SO_DEBUG)");
    }
    
    if (connect(netfd, (struct sockaddr *)sn, sizeof(*sn)) < 0) {
#if defined(h_addr)		/* In 4.3, this is a #define */
	if (host && host->h_addr_list[1]) {
	    int oerrno = errno;
	    
	    fprintf(stderr, "telnet: connect to address %s: ",
		    inet_ntoa(sn->sin_addr));
	    errno = oerrno;
	    perror(NULL);
	    host->h_addr_list++;
	    if (host->h_length > (int)sizeof(sn->sin_addr)) {
		host->h_length = sizeof(sn->sin_addr);
	    }
	    memcpy(&sn->sin_addr, host->h_addr_list[0], host->h_length);
	    close(netfd);
	    netfd = -1;
	    return 1;
	}
#endif	/* defined(h_addr) */

	perror("telnet: Unable to connect to remote host");
	return 0;
    }
    return 2;
}


void nlink_oobinline(void) {
    int on=1;

    /* Systems without SO_OOBINLINE probably won't work */
    if (setsockopt(netfd, SOL_SOCKET, SO_OOBINLINE, &on, sizeof(on)) == -1) {
	perror("setsockopt");
    }
}


/*
 * Check to see if any out-of-band data exists on a socket (for
 * Telnet "synch" processing).
 */

int nlink_stilloob(void) {
    static struct timeval timeout = { 0, 0 };
    fd_set excepts;
    int value;

    do {
	FD_ZERO(&excepts);
	FD_SET(netfd, &excepts);
	value = select(netfd+1, NULL, NULL, &excepts, &timeout);
    } while ((value == -1) && (errno == EINTR));

    if (value < 0) {
	perror("select");
	quit();
	/* NOTREACHED */
    }
    if (FD_ISSET(netfd, &excepts)) {
	return 1;
    } else {
	return 0;
    }
}

int nlink_send(const char *s, int n, int f) {
    return send(netfd, s, n, f);
}

void nlink_nonblock(int onoff) {
    ioctl(netfd, FIONBIO, &onoff);
}

int nlink_getfd(void) {
    return netfd;
}
