/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <arpa/telnet.h>
#include "ring.h"
#include "defines.h"
#include "externs.h"
#include "environ.h"

#ifdef __cplusplus
#include "ptrarray.h"
#else
#include "carray.h"
#endif

char environ_rcsid[] = 
  "$Id: environ.c,v 1.16 2002/08/11 23:23:32 dholland Exp $";

static char *dostrdup(const char *s) {
   char *x;
#ifdef __cplusplus
   x = new char[strlen(s)+1];
   strcpy(x, s);
#else
   x = strdup(s);
   if (x==NULL) {
      /* XXX / FIXME - this is weak */
      perror("malloc");
      exit(1);
   }
#endif   
   return x;
}

static void dofree(char *s) {
#ifdef __cplusplus
      delete []s;
#else
      free(s);
#endif
}

////////////////////////////////////////////////////////////

struct enviro {
    char *var;	        /* pointer to variable name */
    char *value;        /* pointer to variable's value */
    int doexport;       /* 1 -> export with default list of variables */
};

static struct enviro *enviro_create(const char *var, const char *val, int exp){
    struct enviro *e;
#ifdef __cplusplus
    e = new enviro;
#else
    e = malloc(sizeof(struct enviro));
#endif
    e->var = dostrdup(var);
    e->value = dostrdup(val);
    e->doexport = exp;
    return e;
}

static void enviro_destroy(struct enviro *e) {
   if (e->var) {
      dofree(e->var);
   }
   e->var = NULL;

   if (e->value) {
      dofree(e->value);
   }
   e->value = NULL;

#ifdef __cplusplus
   delete e;
#else
   free(e);
#endif
}

static void enviro_change(struct enviro *e, const char *val, int exp) {
   if (e->value) {
      dofree(e->value);
   }
   e->value = dostrdup(val);
   e->doexport = exp;
}

////////////////////////////////////////////////////////////

#ifdef __cplusplus
static ptrarray<enviro> vars;
static inline int array_getnum(ptrarray<enviro> &ar) { return ar.num(); }
static inline enviro *array_getguy(ptrarray<enviro> &ar, int i) {return ar[i];}
static inline void array_setguy(ptrarray<enviro> &ar, int i, enviro *e) {
   ar[i] = e;
}
static inline void array_add(ptrarray<enviro> &ar, enviro *ep) {ar.add(ep);}

#else
static struct array *vars;
#endif

////////////////////////////////////////////////////////////

static struct enviro *env_find(const char *var) {
    int i;
    for (i=0; i<array_getnum(vars); i++) {
        struct enviro *e = array_getguy(vars, i);
        if (e && !strcmp(e->var, var)) {
	    return e;
	}
    }
    return NULL;
}

static struct enviro *env_find_remove(const char *var) {
    int i;
    for (i=0; i<array_getnum(vars); i++) {
        struct enviro *e = array_getguy(vars, i);
        if (e && !strcmp(e->var, var)) {
	    array_setguy(vars, i, NULL);
	    return e;
	}
    }
    return NULL;
}

static void env_put(const char *var, const char *val, int exp) {
    struct enviro *ep;
    int i;

    ep = env_find(var);
    if (ep) {
        enviro_change(ep, val, exp);
        return;
    }
    ep = enviro_create(var, val, exp);

    for (i=0; i<array_getnum(vars); i++) {
       if (array_getguy(vars, i)==NULL) {
	  array_setguy(vars, i, ep);
	  return;
       }
    }
    array_add(vars, ep);
}

extern char **environ;

static void env_copy(void) {

    char *s;
    int i;
    
    for (i=0; environ[i]; i++) {
	s = strchr(environ[i], '=');
	if (s) {
	    *s=0;
	    env_put(environ[i], s+1, 0);
	    *s='=';
	}
    }
}

/*
 * Special case for DISPLAY variable.  If it is ":0.0" or
 * "unix:0.0", we have to get rid of "unix" and insert our
 * hostname.
 */
static void env_fix_display(void) {
    struct enviro *ep;
    char hbuf[256];
    size_t maxlen;
    const char *cp2;
    int warn=0;

    ep = env_find("DISPLAY");
    if (!ep) return;
    ep->doexport = 1;

    if (strncmp(ep->value, ":", 1) && strncmp(ep->value, "unix:", 5)) {
	return;
    }
    cp2 = strrchr(ep->value, ':');
    if (sizeof(hbuf) < strlen(cp2)+1) {
       /* too long - give up */
       return;
    }
    maxlen = sizeof(hbuf)-(strlen(cp2)+1);
    gethostname(hbuf, maxlen);
    hbuf[maxlen] = 0;  /* ensure null termination */

    /*
     * dholland 7/30/96 if not a FQDN ask DNS
     */
    if (!strchr(hbuf, '.')) {
	struct hostent *h = gethostbyname(hbuf);
	if (h) {
	    strncpy(hbuf, h->h_name, maxlen);
	    hbuf[maxlen] = 0; /* ensure null termination */
	}
	/*
	 * dholland 8/11/02 this is somewhat dangerous, so print a warning.
	 * It would be nice to suppress this if the name came from /etc/hosts,
	 * which is pretty much trustworthy, but there's no way to tell that
	 * without reading /etc/hosts ourselves... ugh.
	 */
	warn = 1;
    }

    strcat(hbuf, cp2);

    if (warn) {
       printf("telnet: passing $DISPLAY as %s\n", hbuf);
    }

    enviro_change(ep, hbuf, 1);
}

/*********************************************** interface ***********/

void env_init(void) {
    struct enviro *ep;

#ifndef __cplusplus
    vars = array_create();
#endif

    /*
     * Get the environment strings from our process environment.
     */
    env_copy();

    /*
     * Always export DISPLAY. Also twiddle it if it aims at localhost.
     */
    env_fix_display();

    /*
     * If USER is not defined, but LOGNAME is, then add
     * USER with the value from LOGNAME.  By default, we
     * don't export the USER variable.
     */
    if (!env_find("USER")) {
	ep = env_find("LOGNAME");
	if (ep) env_put("USER", ep->value, 0);
    }

    /*
     * Automatically export PRINTER.
     */
    ep = env_find("PRINTER");
    if (ep) ep->doexport = 1;
}

void env_define(const char *var, const char *value) {
    env_put(var, value, 1);
}

void env_undefine(const char *var) {
    struct enviro *ep = env_find_remove(var);
    if (ep) {
        enviro_destroy(ep);
    }
}

void env_export(const char *var) {
    struct enviro *ep = env_find(var);
    if (ep) ep->doexport = 1;
}

void env_unexport(const char *var) {
    struct enviro *ep = env_find(var);
    if (ep) ep->doexport = 0;
}

void env_send(const char *var) {
    struct enviro *ep;

    if (my_state_is_wont(TELOPT_ENVIRON)) {
	fprintf(stderr, "Cannot send '%s': Telnet ENVIRON option disabled\n",
		var);
	return;
    }

    ep = env_find(var);
    if (!ep) {
	fprintf(stderr, "Cannot send '%s': variable not defined\n", var);
	return;
    }
    env_opt_start_info();
    env_opt_add(ep->var);
    env_opt_end(0);
}

void env_list(void) {
    int i;
    for (i=0; i<array_getnum(vars); i++) {
        struct enviro *ep = array_getguy(vars, i);
        if (ep) {
	   printf("%c %-20s %s\n",
		  ep->doexport ? '*' : ' ',
		  ep->var, ep->value);
	}
    }
}

void env_iterate(int *iter, int exported_only) {
    (void)exported_only;

    *iter = 0;
}

const char *env_next(int *iter, int exported_only) {
    while (*iter>=0 && *iter<array_getnum(vars)) {
	int k = (*iter)++;
	struct enviro *ep = array_getguy(vars, k);

	if (ep==NULL) continue; // deleted variable

	if (ep->doexport || !exported_only) {
	    return ep->var;
	}
    }
    return NULL;
}

const char *env_getvalue(const char *var, int exported_only) {
    struct enviro *ep = env_find(var);
    if (ep && (!exported_only || ep->doexport))
    	return ep->value;
    return NULL;
}
