/*
 * Copyright (c) 2000, 2001
 *	The President and Fellows of Harvard College.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * C array of void pointers. See carray.h.
 */
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include "carray.h"

struct array {
	int num;
	int max;
	void **v;
};

struct array *
array_create(void)
{
	struct array *a = malloc(sizeof(struct array));
	if (a==NULL) {
		return NULL;
	}
	a->v = NULL;
	a->num = 0;
	a->max = 0;
	return a;
}

int
array_getnum(struct array *a)
{
	return a->num;
}

void *
array_getguy(struct array *a, int index)
{
	assert(a->num <= a->max);
	assert(index >=0 && index < a->num);
	return a->v[index];
}

int
array_preallocate(struct array *a, int nguys)
{
	void **newv;
	int i;
	int newmax = a->max;

	assert(a->num >=0 && a->num <= a->max);
		
	while (nguys > newmax) {
		newmax = (newmax+1)*2;
	}
	newv = malloc(newmax * sizeof(void *));
	if (newv==NULL) {
		return ENOMEM;
	}
	a->max = newmax;
	for (i=0; i<a->num; i++) newv[i] = a->v[i];
	if (a->v!=NULL) {
		free(a->v);
	}
	a->v = newv;
	return 0;
}

int
array_setsize(struct array *a, int nguys)
{
	int result;

	assert(a->num >=0 && a->num <= a->max);

	if (nguys > a->max) {
		result = array_preallocate(a, nguys);
		if (result) {
			return result;
		}
	}
	else if (nguys==0 && a->max > 16) {
		assert(a->v!=NULL);
		free(a->v);
		a->v = NULL;
		a->max = 0;
	}
	a->num = nguys;

	return 0;
}

void
array_setguy(struct array *a, int index, void *ptr)
{
	assert(a->num <= a->max);
	assert(index >=0 && index < a->num);
	a->v[index] = ptr;
}

int
array_add(struct array *a, void *guy)
{
	int ix, result;

	ix = a->num;

	result = array_setsize(a, ix+1);
	if (result) {
		return result;
	}

	a->v[ix] = guy;

	return 0;
}

void
array_remove(struct array *a, int index)
{
	int nmove;

	assert(a->num <= a->max);
	assert(index >=0 && index < a->num);

	nmove = a->num - (index + 1);
	memmove(a->v+index, a->v+index+1, nmove*sizeof(void *));
	a->num--;
}

void
array_destroy(struct array *a)
{
	if (a->v) free(a->v);
	free(a);
}
