/*
 * Copyright (c) 2001 David A. Holland.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Author nor the names of any contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdarg.h>
#include "defs.h"
#include "ext.h"
#include "buffer.h"

extern int not42;

#define MAXMARKS  16

char buffer_rcsid[] = 
  "$Id: buffer.c,v 1.5 2001/11/06 01:28:07 dholland Exp $";

struct buffer {
	char buf[BUFSIZ+NETSLOP];
	size_t tail;
	size_t head;

	size_t urgpos;		/* netobuf only */
	int hasurg;		/* netobuf only */

#ifdef ENCRYPT
	size_t encpos;
	int docrypt;
#endif

	size_t marks[MAXMARKS];	/* netobuf only */
	int nmarks;		/* netobuf only */
};

static
void
buf_init(struct buffer *b)
{
	b->tail = 0;
	b->head = 0;

	b->urgpos = 0;
	b->hasurg = 0;
	
#ifdef ENCRYPT
	b->encpos = 0;
#endif
	
	b->nmarks = 0;
}

static
void
buf_check(const struct buffer *b)
{
	int bad = 0, i;

	if (b->head < b->tail || b->head > sizeof(b->buf)) {
		bad = 1;
	}
	if (b->hasurg && (b->urgpos < b->tail || b->urgpos > b->head)) {
		bad = 1;
	}
#ifdef ENCRYPT
	if (b->docrypt && (b->encpos < b->tail || b->encpos > b->head)) {
		bad = 1;
	}
#endif
	if (b->nmarks > MAXMARKS) {
		bad = 1;
	}
	for (i=0; i<b->nmarks; i++) {
		if (b->marks[i] > b->head) {
			bad = 1;
		}
	}
	if (bad) {
		syslog(LOG_ERR, "Corrupted buffer structure "
		       "- telnetd panic, dying\n");
		cleanup(0);
	}
}

static
size_t
buf_nchars(const struct buffer *b)
{
	buf_check(b);
	return b->head - b->tail;
}

static
size_t
buf_space(const struct buffer *b)
{
	buf_check(b);
	return (sizeof(b->buf) - NETSLOP) - b->head;
}

static
const char *
buf_get(struct buffer *b, size_t nchars)
{
	size_t avail;
   
	buf_check(b);
	avail = b->head - b->tail;
	if (nchars > avail) {
		return NULL;
	}
	return &b->buf[b->head - nchars];
}

static
void
buf_checkempty(struct buffer *b)
{
	buf_check(b);

	if (b->tail == b->head) {
		b->tail = b->head = 0;
		b->hasurg = 0;
#ifdef ENCRYPT
		b->encpos = 0;
#endif
		b->nmarks = 0;
	}
}

static
int
buf_read_oob(struct buffer *b, int fd)
{
	size_t len;
	int r;
	
	buf_check(b);
	len = sizeof(b->buf) - b->head;
	if (len==0) {
		/* pretend we read something */
		return 1;
	}

	r = recv(fd, b->buf+b->head, len, MSG_OOB);
	if (r >= 0) {
		b->head += r;
	}
	
	return r;
}

static
int
buf_read(struct buffer *b, int fd)
{
	size_t len;
	int r;
	
	buf_check(b);
	len = sizeof(b->buf) - b->head;
	if (len==0) {
		/* pretend we read something */
		return 1;
	}
	
	r = read(fd, b->buf+b->head, len);
	if (r >= 0) {
		b->head += r;
	}
	
	return r;
}

static
int
buf_write(struct buffer *b, int fd)
{
	int r, n;

	buf_check(b);

#ifdef ENCRYPT
	if (b->docrypt && encrypt_output) {
		size_t len = b->head - b->encpos;
		(*encrypt_output)((unsigned char *)b->buf + b->encpos, len);
		b->encpos += len;
	}
#endif

	/*
	 * If the other side is an old 4.2 client that will die on TCP
	 * urgent data, don't send it.
	 */
	if (not42==0) {
		b->hasurg = 0;
	}
	
	/*
	 * In 4.2 (and 4.3) systems, there is some question about
	 * what byte in a sendOOB operation is the "OOB" data.
	 * To make ourselves compatible, we only send ONE byte
	 * out of band, the one WE THINK should be OOB (though
	 * we really have more the TCP philosophy of urgent data
	 * rather than the Unix philosophy of OOB data).
	 *
	 * (The above comment was in the original flush code.)
	 *
	 * I have simplified the bizarre logic used in the old code, 
	 * so if it stops working please investigate.
	 */

	if (b->hasurg && b->urgpos <= b->tail) {
		r = send(fd, b->buf + b->tail, 1, MSG_OOB);
		if (r >= 0) {
			b->hasurg = 0;
		}
	}
	else if (b->hasurg) {
		n = b->urgpos - b->tail;
		r = write(fd, b->buf + b->tail, n);
	}
	else {
		n = b->head - b->tail;
		r = write(fd, b->buf + b->tail, n);
	}
	
	if (r >= 0) {
		b->tail += r;
		buf_checkempty(b);
	}

	return r;
}

static
int
buf_getc(struct buffer *b)
{
	int r;

	buf_check(b);
	if (b->tail==b->head) {
		return -1;
	}
	r = b->buf[b->tail++] & 0xff;
	buf_checkempty(b);
	return r;
}

static
int
buf_putc(struct buffer *b, int c)
{
	buf_check(b);
	if (b->head==sizeof(b->buf)) {
		return -1;
	}
	b->buf[b->head++] = c;
	return 0;
}

static
size_t
buf_append(struct buffer *b, const char *ptr, size_t len)
{
	size_t i;
	buf_check(b);
	for (i=0; i<len; i++) {
		if (buf_putc(b, ptr[i])) {
			return i;
		}
	}
	return len;
}

static
void
buf_mark(struct buffer *b)
{
	int i;
	buf_check(b);
	
	if (b->nmarks==MAXMARKS) {
		for (i=0; i<MAXMARKS-1; i++) {
			b->marks[i] = b->marks[i+1];
		}
		b->nmarks--;
	}
	
	b->marks[b->nmarks++] = b->head;
}

static
size_t
buf_charstomark(struct buffer *b)
{
	int i;
	buf_check(b);
	
	for (i=0; i<b->nmarks; i++) {
		if (b->marks[i] >= b->tail) {
			return b->marks[i] - b->tail;
		}
	}
	
	return b->head - b->tail;
}

/****************************************************/

static struct buffer ptyobuf, netobuf, netibuf;
/* XXX do we move ptyibuf in here too? */


void
buffer_init(void)
{
	buf_init(&ptyobuf);
	buf_init(&netobuf);
	buf_init(&netibuf);
}

int
buffer_netobuf_hasstuff(void)
{
	return buf_nchars(&netobuf) > 0;
}

int
buffer_ptyobuf_hasstuff(void)
{
	return buf_nchars(&ptyobuf) > 0;
}

int
buffer_netobuf_nearly_full(size_t k)
{
	return buf_space(&netobuf) < k;
}

int
buffer_ptyobuf_nearly_full(size_t k)
{
	return buf_space(&ptyobuf) < k;
}

size_t
buffer_netobuf_nchars(void)
{
	return buf_nchars(&netobuf);
}

size_t
buffer_ptyobuf_nchars(void)
{
	return buf_nchars(&ptyobuf);
}

/********/

size_t
buffer_netobuf_mustsend(void)
{
	/*
	 * The number of bytes we must send to complete a telnet command
	 * that has already started to go out.
	 *
	 * The buffer positions where the output would be complete are
	 * stored in the marks[] array. These are set by calls to 
	 * buffer_netobuf_mark().
	 */
	
	return buf_charstomark(&netobuf);
}

void
buffer_netobuf_mark(void)
{
	buf_mark(&netobuf);
}

/********/

const char *
buffer_get_netibuf(int nchars)
{
	return buf_get(&netibuf, nchars);
}

const char *
buffer_get_ptyobuf(int nchars)
{
	return buf_get(&ptyobuf, nchars);
}

const char *
buffer_get_netobuf(int nchars)
{
	return buf_get(&netobuf, nchars);
}

/********/

void
buffer_netobuf_seturg(void)
{
	netobuf.urgpos = netobuf.head;
	netobuf.hasurg = 1;
}

void
buffer_netobuf_clearurg(void)
{
	netobuf.hasurg = 0;
}

/********/

int
buffer_read_netibuf_oob(int fd)
{
	return buf_read_oob(&netibuf, fd);
}

int
buffer_read_netibuf(int fd)
{
	return buf_read(&netibuf, fd);
}

int
buffer_netibuf_getc(void)
{
	return buf_getc(&netibuf);
}

/********/

int
buffer_write_ptyobuf(int fd)
{
	return buf_write(&ptyobuf, fd);
}

int
buffer_write_netobuf(int fd)
{
	return buf_write(&netobuf, fd);
}

void
buffer_netobuf_putc(int c)
{
	while (buf_putc(&netobuf, c)) {
		netflush();
	}
}

void
buffer_ptyobuf_putc(int c)
{
	while (buf_putc(&ptyobuf, c)) {
		ptyflush();
	}
}

/********/

void
buffer_netobuf_append(const char *ptr, size_t len)
{
	size_t pos = 0;
	while (pos < len) {
		pos += buf_append(&netobuf, ptr, len);
		if (pos < len) {
			netflush();
		}
	}
}

void
netoprintf(const char *fmt, ...)
{
	char buf[4096];
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);
	
	if (len < 0 || len>=(int)sizeof(buf)) {
		/* This really should not happen. */
		syslog(LOG_ERR, "Output too large in netoprintf "
		       "- telnetd panic, dying");
		cleanup(0);
	}
	
	buffer_netobuf_append(buf, strlen(buf));
}

/********/

void
buffer_ptyobuf_purge(void)
{
	buf_init(&ptyobuf);
}

void
buffer_netobuf_rewind(size_t nchars)
{
	size_t bot, max;
	
	bot = netobuf.tail;
#ifdef ENCRYPT
	if (netobuf.docrypt && netobuf.encpos > bot) {
		bot = netobuf.encpos;
	}
#endif
	max = netobuf.head - bot;
	if (nchars > max) {
		nchars = max;
	}
	netobuf.head -= nchars;
	netobuf.marks[0] = netobuf.head;
	netobuf.nmarks = 1;
	
	buf_check(&netobuf);
}
