/*
 * Copyright (c) 1995 Akihiro Tominaga and the WIDE Project.
 * All rights reserved.
 *
 * Permission to use, copy and distribute this software in source and
 * binary forms is hereby granted provided that the above copyright
 * notice and this permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions thereof.
 * Furthermore, any supporting documentation, advertising materials,
 * and all other materials related to such a distribution and use must
 * acknowledge that the software was developed by Akihiro Tominaga and
 * the WIDE Project. The names of the author and the WIDE Project may
 * not be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/if_ether.h>
#ifndef BYTE_ORDER
#include <sys/param.h>
#endif
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>

#include "dhcp.h"
#include "dhcpc.h"
#include "common.h"
#include "config.h"
#include "dhcpc_subr.h"

#define DHCPCACHE  "/etc/dhcpcache_"
#define REQLEASE   3600
#define	DHCP_SLEEP()  usleep(random() % 9000000 + 1000000)

struct dhcp_param *paramp;
struct dhcp_if *intface;
int has_valid_param;
char cachename[sizeof(DHCPCACHE) + IFNAMSIZ];

void
release()
{
  int fd;
  int zero;
  struct dhcp_reqspec reqspec;

  bzero(&reqspec, sizeof(reqspec));
  if (has_valid_param == 1) {
    reqspec.ipaddr.s_addr = paramp->yiaddr.s_addr;
    if (paramp->server_id != NULL)
      reqspec.srvaddr.s_addr = paramp->server_id->s_addr;
    dhcp_release(intface, &reqspec);
    if ((fd = open(cachename, O_WRONLY, 0444)) >= 0) {
      zero = 0;  
      write(fd, (void *) &zero, sizeof(zero));
      close(fd); 
    }
  }

  reset_if(intface);
  exit(0);
}


void
finish()
{
  reset_if(intface);
  exit(0);
}


void
usage()
{
  fprintf(stderr, "Usage: dhcpc [-d] if_name\n");
  exit(1);
}


void
use_parameter(intf, paramp, optmap, cachename)
  struct dhcp_if *intf;
  struct dhcp_param *paramp;
  struct dhcp_optmap *optmap;
  char *cachename;
{
  static int first = 0;
  static struct in_addr addr, mask, brdaddr;
  int fd;
  int n;
  int length;
  char *bufp;

  if (paramp == NULL || optmap == NULL || cachename == NULL)
    return;

  if (first == 0) {
    first = 1;
    bzero(&addr, sizeof(struct in_addr));
    bzero(&mask, sizeof(struct in_addr));
    bzero(&brdaddr, sizeof(struct in_addr));
  }

  if (paramp->yiaddr.s_addr == addr.s_addr &&
      paramp->subnet_mask != NULL &&
      paramp->subnet_mask->s_addr == mask.s_addr &&
      paramp->brdcast_addr != NULL &&
      paramp->brdcast_addr->s_addr == brdaddr.s_addr)
    return;

  addr.s_addr = paramp->yiaddr.s_addr;
  if (dhcp_optmap_isset(SUBNET_MASK, optmap)) {
    mask.s_addr = paramp->subnet_mask->s_addr;
  } else {
    mask.s_addr = 0;
  }
  if (dhcp_optmap_isset(BRDCAST_ADDR, optmap)) {
    brdaddr.s_addr = paramp->brdcast_addr->s_addr;
  } else {
    brdaddr.s_addr = 0;
  }

#ifndef DEBUG
  config_if(intf, &addr, ((mask.s_addr != 0) ? &mask : NULL),
	    ((brdaddr.s_addr != 0) ? &brdaddr : NULL));
  set_route(paramp, optmap);
#endif

  if ((fd = open(cachename, O_WRONLY | O_CREAT | O_TRUNC, 0444)) >= 0) {
    length = ntohs(rcvmsg.udp->uh_ulen) + rcvmsg.ip->ip_hl * WORD + ETHERHL;
    write(fd, (void *) &paramp->lease_origin, sizeof(paramp->lease_origin));
    write(fd, (void *) &length, sizeof(length));

    bufp = (char *) rcvmsg.ether;
    while (length > 0) {
      n = write(fd, (void *) bufp, length);
      length -= n;
      bufp += n;
    }

    close(fd);
  }

  return;
}


int
read_cache(cachename, paramp, optmap)
  char *cachename;
  struct dhcp_param *paramp;
  struct dhcp_optmap *optmap;
{
  int fd;
  int length;
  int n;
  char *buf;
  char *tmp;
  time_t lease_origin;

  if ((fd = open(cachename, O_RDONLY, 0)) < 0) {
    return(-1);
  }

  /* read lease origin epoch */
  if (read(fd, (void *) &lease_origin, sizeof(lease_origin)) !=
      sizeof(lease_origin)) {
    Syslog(LOG_WARNING, "read() error in read_cache(): %m");
    close(fd);
    return(-1);
  }

  /* read length of DHCP message size */
  if (read(fd, (void *) &length, sizeof(length)) != sizeof(length)) {
    Syslog(LOG_WARNING, "read() error in read_cache(): %m");
    close(fd);
    return(-1);
  }

  if ((buf = calloc(1, length + QWOFF)) == NULL) {
    Syslog(LOG_WARNING, "calloc error in read_cache(): %m");
    close(fd);
    return(-1);
  }

  tmp = &buf[QWOFF];
  while ((n = (read(fd, (void *) tmp, length))) > 0) {
    length -= n;
    tmp += n;
  }

  if (length != 0) {
    free(buf);
    free(paramp);
    close(fd);
    return(-1);
  }

  rcvmsg.ether = (struct ether_header *) &buf[QWOFF];
  rcvmsg.ip = (struct ip *) &buf[ETHERHL + QWOFF];
  if (ntohs(rcvmsg.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL + QWOFF) {
    rcvmsg.udp = (struct udphdr *)
      &buf[ETHERHL + rcvmsg.ip->ip_hl * WORD + QWOFF];
    rcvmsg.dhcp = (struct dhcp *)
      &buf[ETHERHL + rcvmsg.ip->ip_hl * WORD + UDPHL + QWOFF];
  } else {
    rcvmsg.udp = NULL;
    rcvmsg.dhcp = NULL;
  }

  if (dhcp_msgtoparam(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
		      paramp, optmap) != OK) {
    clean_param(paramp);
    free(paramp);
    free(buf);
    close(fd);
    return(-1);
  }

  paramp->lease_origin = lease_origin;
  free(buf);
  close(fd);
  return(0);
}
  

int
main(argc, argv)
  int argc;
  char **argv;
{
  char intfname[IFNAMSIZ];
  int debug = 0;
  int n = 0;
  int first;
  time_t curr_epoch;
  struct dhcp_reqspec reqspec;
  struct dhcp_optlist optlist;
  struct dhcp_optmap optmap;
  struct dhcp_param *tmpparam;


  bzero(&reqspec, sizeof(reqspec));

  if (argc < 2) usage();
  --argc, ++argv;
  if (argv[0][0] == '-' && argv[0][1] == 'd') {
    debug = 1;
    --argc, ++argv;
  }

  if (argc < 1) usage();
  strcpy(intfname, argv[0]);

  /*
   * go into background and disassociate from controlling terminal
   */
  if (debug == 0) {
    if(fork() != 0) exit(0);
    for (n = 0; n < 10; n++)
      close(n);
    open("/", O_RDONLY);
    dup2(0, 1);
    dup2(0, 2);
    if ((n = open("/dev/tty", O_RDWR)) >= 0) {
      ioctl(n, TIOCNOTTY, (char *) 0);
      close(n);
    }
  }


  /*
   * Main
   */
  if ((intface = dhcp_init(intfname, NULL)) == NULL) {
    exit(1);
  }

  strcpy(cachename, DHCPCACHE);
  strcpy(&cachename[strlen(DHCPCACHE)], intface->name);

  if ((int) signal(SIGTERM, (void *) finish) < 0) {
    Syslog(LOG_ERR, "cannot set signal handler(SIGTERM): %m");
    exit(1);
  }
  if ((int) signal(SIGINT, (void *) finish) < 0) {
    Syslog(LOG_ERR, "cannot set signal handler(SIGINT): %m");
    exit(1);
  }
  if ((int) signal(SIGUSR2, (void *) release) < 0) {
    Syslog(LOG_ERR, "cannot set signal handler(SIGUSR2): %m");
    exit(1);
  }

  DHCP_SLEEP();
  first = 1;
  if ((tmpparam =
       (struct dhcp_param *)calloc(1, sizeof(struct dhcp_param))) == NULL) {
    Syslog(LOG_WARNING, "calloc error in main(): %m");
    first = 0;
  }
  while (1) {
    has_valid_param = 0;
    if (first == 1) {
      first = 0;
      if (read_cache(cachename, tmpparam, &optmap) == 0) {
	reqspec.ipaddr.s_addr = tmpparam->yiaddr.s_addr;
	reqspec.select_offer = DHCPOFFER_SELECT_FIRST;
	reqspec.ck_ipdup = DHCP_CHKADDR_ON;
	reqspec.wait_usec = 0;
	reqspec.lease = REQLEASE;
	dhcp_optlist_clr(&optlist);
	dhcp_optlist_add(SUBNET_MASK, &optlist);
	dhcp_optlist_add(ROUTER, &optlist);
	dhcp_optlist_add(BRDCAST_ADDR, &optlist);

	if ((paramp =
	     dhcp_reuseparam(intface, &optlist, &reqspec, &optmap)) != NULL) {
	  use_parameter(intface, paramp, &optmap, cachename);
	  has_valid_param = 1;
	  goto extend_loop;
	}

	if (time(&curr_epoch) == -1) {
	  Syslog(LOG_WARNING, "time() error in main: %m");
	  continue;
	}

	if (dhcp_errno == ENOREPLY &&
	    tmpparam->lease_origin + tmpparam->lease_duration > curr_epoch) {
	  use_parameter(intface, tmpparam, &optmap, cachename);
	  paramp = tmpparam;
	  has_valid_param = 1;
	  goto extend_loop;
	}
      }
    }

    reqspec.select_offer = DHCPOFFER_SELECT_FIRST;
    reqspec.ck_ipdup = DHCP_CHKADDR_ON;
    reqspec.wait_usec = 0;
    reqspec.lease = REQLEASE;
    dhcp_optlist_clr(&optlist);
    dhcp_optlist_add(SUBNET_MASK, &optlist);
    dhcp_optlist_add(ROUTER, &optlist);
    dhcp_optlist_add(BRDCAST_ADDR, &optlist);

    if ((paramp =
	 dhcp_getparam(intface, &optlist, &reqspec, &optmap)) == NULL) {
      if (dhcp_errno != EIPDUP) {
	has_valid_param = 0;
	reset_if(intface);
	Syslog(LOG_ERR, "There is no appropriate reply");
	exit(1);
      } else {
	has_valid_param = 0;
	Syslog(LOG_ERR, "Restart from beginning");
	DHCP_SLEEP();
	continue;
      }
    }

    use_parameter(intface, paramp, &optmap, cachename);

  extend_loop:

    while (paramp->lease_duration == ~0) {
      sigpause(0);
    }

    while (1) {
      if (time(&curr_epoch) == -1) {
	Syslog(LOG_WARNING, "time() error in main: %m");
	continue;
      }

      if (paramp->lease_origin + paramp->lease_duration <= curr_epoch) {
	reset_if(intface);
	Syslog(LOG_WARNING, "The lease has already expired");
	break;
      }
      else if (paramp->lease_origin + paramp->dhcp_t2 <= curr_epoch &&
	  curr_epoch < paramp->lease_origin + paramp->lease_duration) {
	reqspec.extend_phase = REBINDING;
	reqspec.wait_usec = (paramp->lease_origin + paramp->lease_duration -
			     curr_epoch) * 1000000 / 2;
	if (reqspec.wait_usec < 60 * 1000000) {
	  reqspec.wait_usec *= 2;
	}
      }
      else if (paramp->lease_origin + paramp->dhcp_t1 <= curr_epoch &&
	       curr_epoch < paramp->lease_origin + paramp->dhcp_t2) {
	reqspec.extend_phase = RENEWING;
	if (paramp->server_id != NULL)
	  reqspec.srvaddr = *paramp->server_id;
	reqspec.wait_usec =
	  (paramp->lease_origin + paramp->dhcp_t2 - curr_epoch) * 1000000 / 2;
	if (reqspec.wait_usec < 60 * 1000000) {
	  reqspec.wait_usec *= 2;
	}
      } else {
	sleep(paramp->lease_origin + paramp->dhcp_t1 - curr_epoch);
	continue;
      }

      if ((tmpparam =
	   dhcp_extend(intface, &optlist, &reqspec, &optmap)) != NULL) {
	paramp = tmpparam;
	use_parameter(intface, paramp, &optmap, cachename);
	has_valid_param = 1;
      }
      else if (dhcp_errno != SUCCESS && dhcp_errno != ENOREPLY) {
	has_valid_param = 0;
	reset_if(intface);
	break;
      }
    }

    has_valid_param = 0;
    reset_if(intface);
  }

  return(0);
}
