/*
 * Copyright (c) 1995 Akihiro Tominaga and the WIDE Project.
 * All rights reserved.
 *
 * Permission to use, copy and distribute this software in source and
 * binary forms is hereby granted provided that the above copyright
 * notice and this permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions thereof.
 * Furthermore, any supporting documentation, advertising materials,
 * and all other materials related to such a distribution and use must
 * acknowledge that the software was developed by Akihiro Tominaga and
 * the WIDE Project. The names of the author and the WIDE Project may
 * not be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/if_ether.h>
#include <sys/param.h>
#include <arpa/inet.h>

#include "dhcp.h"
#include "dhcpc.h"
#include "common.h"

long generate_xid();
int config_if();
void set_route();
int flushroutes();


/*
 * halt network, and reset the interface
 */
void
reset_if(intf)
  struct dhcp_if *intf;
{
  struct in_addr addr, mask, brdaddr;

  addr.s_addr = htonl(generate_xid(NULL) & 0xfff);
  mask.s_addr = htonl(0xff000000);
  brdaddr.s_addr = inet_addr("255.255.255.255");

#ifndef DEBUG
  config_if(intf, &addr, &mask, &brdaddr);
  flushroutes();
#endif

  return;
}


/*
 * configure network interface
 *    address, netmask, and broadcast address
 */
#ifdef __bsdi__

int
config_if(intf, addr, mask, brdcst)
  struct dhcp_if *intf;
  struct in_addr *addr;
  struct in_addr *mask;
  struct in_addr *brdcst;
{
  int sockfd = 0;
  struct ifreq ridreq;
  struct ifaliasreq addreq;
#define SIN(x) ((struct sockaddr_in *) &(x))

  flushroutes();
  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    return(-1);
  }

  bzero(&ridreq, sizeof(struct ifreq));
  strncpy(ridreq.ifr_name, intf->name, sizeof(ridreq.ifr_name));
  ioctl(sockfd, SIOCDIFADDR, (caddr_t) &ridreq);

  bzero(&addreq, sizeof(struct ifaliasreq));
  strncpy(addreq.ifra_name, intf->name, sizeof(ridreq.ifr_name));
  SIN(addreq.ifra_addr)->sin_len = sizeof(struct sockaddr_in);
  SIN(addreq.ifra_addr)->sin_family = AF_INET;
  if (addr) {
    SIN(addreq.ifra_addr)->sin_addr.s_addr = addr->s_addr;
  }
  if (mask) {
    SIN(addreq.ifra_mask)->sin_len = sizeof(struct sockaddr_in);
    SIN(addreq.ifra_mask)->sin_addr.s_addr = mask->s_addr;
  }
  if (brdcst) {
    SIN(addreq.ifra_broadaddr)->sin_len = sizeof(struct sockaddr_in);
    SIN(addreq.ifra_broadaddr)->sin_family = AF_INET;
    SIN(addreq.ifra_broadaddr)->sin_addr.s_addr = brdcst->s_addr;
  }
  else if (addr && mask) {
    SIN(addreq.ifra_broadaddr)->sin_len = sizeof(struct sockaddr_in);
    SIN(addreq.ifra_broadaddr)->sin_family = AF_INET;
    SIN(addreq.ifra_broadaddr)->sin_addr.s_addr = addr->s_addr | ~mask->s_addr;
  }

  if (ioctl(sockfd, SIOCAIFADDR, (caddr_t) &addreq) < 0) {
    perror("SIOCAIFADDR");
    close(sockfd);
    return(-1);
  }

  close(sockfd);
  return(0);
}

#else /* not __bsdi__ */

int
config_if(intf, addr, mask, brdcst)
  struct dhcp_if *intf;
  struct in_addr *addr;
  struct in_addr *mask;
  struct in_addr *brdcst;
{
  int sockfd = 0;
  struct ifreq ifr;

  flushroutes();
  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    return(-1);
  }

  if (mask != NULL) {
    bzero(&ifr, sizeof(struct ifreq));
    strcpy(ifr.ifr_name, intf->name);
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = mask->s_addr;
    if (ioctl(sockfd, SIOCSIFNETMASK, &ifr) < 0) {
      close(sockfd);
      return(-1);
    }
  }
  if (addr != NULL) {
    bzero(&ifr, sizeof(struct ifreq));
    strcpy(ifr.ifr_name, intf->name);
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = addr->s_addr;
    if (ioctl(sockfd, SIOCSIFADDR, &ifr) < 0) {
      close(sockfd);
      return(-1);
    }
  }
  if (brdcst != NULL) {
    bzero(&ifr, sizeof(struct ifreq));
    strcpy(ifr.ifr_name, intf->name);
    ((struct sockaddr_in *) &ifr.ifr_broadaddr)->sin_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_broadaddr)->sin_addr.s_addr = brdcst->s_addr;
    if (ioctl(sockfd, SIOCSIFBRDADDR, &ifr) < 0) {
      close(sockfd);
      return(-1);
    }
  }
  close(sockfd);

  return(0);
}

#endif /* __bsdi__ */


/*
 * set routing table
 */
void
set_route(param, optmap)
  struct dhcp_param *param;
  struct dhcp_optmap *optmap;
{
  int sockfd = 0;
#if !defined(__bsdi__) && !defined(__FreeBSD__)
#define  ortentry  rtentry
#endif
  struct ortentry rt;
  struct sockaddr dst, gateway;

  if (param == NULL)
    return;

  sockfd = socket(AF_INET, SOCK_RAW, 0);
  if (sockfd < 0) {
    Syslog(LOG_WARNING, "socket() error in set_route(): %m");
    return;
  }

  /*
   * set default route
   */
  if (dhcp_optmap_isset(ROUTER, optmap) &&
      param->router != NULL && param->router->addr != NULL) {
    bzero(&rt, sizeof(struct ortentry));
    bzero(&dst, sizeof(struct sockaddr));
    bzero(&gateway, sizeof(struct sockaddr));
    rt.rt_flags = RTF_UP | RTF_GATEWAY;
#ifdef __bsdi__
    ((struct sockaddr_in *) &dst)->sin_len = sizeof(struct sockaddr_in);
#endif
    ((struct sockaddr_in *) &dst)->sin_family = AF_INET;
    ((struct sockaddr_in *) &dst)->sin_addr.s_addr = INADDR_ANY;
#ifdef __bsdi__
    ((struct sockaddr_in *) &gateway)->sin_len = sizeof(struct sockaddr_in);
#endif
    ((struct sockaddr_in *) &gateway)->sin_family = AF_INET;
    ((struct sockaddr_in *) &gateway)->sin_addr.s_addr =
      param->router->addr->s_addr;
    rt.rt_dst = dst;
    rt.rt_gateway = gateway;
    if (ioctl(sockfd, SIOCADDRT, &rt) < 0)
      Syslog(LOG_WARNING, "SIOCADDRT (default route): %m");
  }

  close(sockfd);
  return;
}
